﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace nw4f.meshlib
{
    /// <summary>
    /// ヒープリスト
    /// 内部クラス
    /// </summary>
    internal class HeapList<T> where T : QemHeapData
    {
        private SourceType mKey = SourceType.Position;
        private List<T> mList = new List<T>();

        /// <summary>
        /// 左の子を取得
        /// </summary>
        private int left(int id) { return safeindex(2 * id + 1); }

        /// <summary>
        /// 右の子を取得
        /// </summary>
        private int right(int id) { return safeindex(2 * (id + 1)); }

        /// <summary>
        /// 親を取得
        /// </summary>
        private int parent(int i) { return i <= 0 ? -1 : safeindex((i - 1) / 2); }

        /// <summary>
        /// 安全に使えるインデックスを取得する
        /// </summary>
        private int safeindex(int i) { return (i < mList.Count && i >= 0) ? i : -1; }

        /// <summary>
        /// リストを取得する
        /// </summary>
        internal SourceType Key
        {
            get { return mKey; }
            set { mKey = value; }
        }

        /// <summary>
        /// リストを取得する
        /// </summary>
        internal List<T> List
        {
            get { return mList; }
        }

        /// <summary>
        /// 最小（根）を取得する
        /// </summary>
        internal T Top
        {
            get
            {
                if (mList.Count > 0)
                {
                    return mList[0];
                }
                return default(T);
            }
        }

        /// <summary>
        /// ヒープに追加
        /// </summary>
        /// <param name="val">値</param>
        internal int Push(T val)
        {
            int n = mList.Count;
            mList.Add(val);
            val.HeapPos = n;
            return HeapUp(n);
        }

        /// <summary>
        /// ヒープから削除
        /// </summary>
        /// <param name="val"></param>
        internal void Pop()
        {
            T min = mList[0];
            int n = mList.Count - 1;

            SwapCells(0, n);
            mList.RemoveAt(n);

            HeapDown(0);
        }

        /// <summary>
        /// 指定のオブジェクトを削除
        /// </summary>
        /// <param name="obj"></param>
        internal void Pop(T obj)
        {
            int n = mList.Count - 1;
            int op = obj.HeapPos;
            SwapCells(op, n);
            mList.RemoveAt(n);
            if (op < mList.Count)
            {
                Update(op);
            }
        }

        private int HeapUp(int i)
        {
            int id = i;
            int p = parent(i);

            while (p >= 0 && id > 0 && mList[p].CompareTo(mList[id], mKey) > 0)
            {
                SwapCells(id, p);
                id = p;
                p = parent(id);
            }
            return id;
        }

        private void HeapDown(int i)
        {
            int id = i;
            int lc = left(id);
            int rc = right(id);
            int c = 0;
            while (lc > 0)
            {
                if (rc > 0 && mList[lc].CompareTo(mList[rc], mKey) > 0)
                {
                    c = rc;
                }
                else
                {
                    c = lc;
                }

                if (mList[c].CompareTo(mList[id], mKey) < 0)
                {
                    SwapCells(id, c);
                    id = c;
                    lc = left(id);
                    rc = right(id);
                }
                else
                {
                    break;
                }
            }
        }

        private void SwapCells(int i, int j)
        {
            T temp = mList[i];
            mList[i] = mList[j];
            mList[i].HeapPos = i;
            mList[j] = temp;
            mList[j].HeapPos = j;
        }

        /// <summary>
        /// ヒープに追加
        /// </summary>
        /// <param name="val">値</param>
        internal void Update(int id)
        {
            int n = mList.Count;
            int p = parent(id);

            if (p >= 0 && id > 0 && mList[id].CompareTo(mList[p], mKey) < 0)
            {
                HeapUp(id);
            }
            else
            {
                HeapDown(id);
            }
        }

        public void PrintAll()
        {
            int n = mList.Count;
            for (int i = 0; i < n; i++)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine($"{mList[i]}");
            }
        }
    }
}
