﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;

using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using nw.g3d.toollib.Ext;

namespace nw.g3d.optcvtr
{
    // マテリアルアニメオプティマイザ
    internal class MaterialAnimOptimizer : Optimizer
    {
        // コンストラクタ
        public MaterialAnimOptimizer(g3doptcvtrParams fileOption)
            : base(fileOption)
        {
            DisableCompressOption();
            // マージ
            if (fileOption.Merge)
            {
                if (fileOption.MergeOptions == null)
                {
                    Strings.Throw("Error_OptionNotSpecified", fileOption.GetOptionName(() => fileOption.MergeOptions));
                }

                var mergeOptions = fileOption.MergeOptions;

                // マージファイルの設定
                if (!File.Exists(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_OptionFileNotFound", mergeOptions.MergeFile);
                }
                if (!G3dPath.IsMaterialAnimPath(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_Merge_InvalidFileType", mergeOptions.MergeFile);
                }

                this.MergeFilePath = mergeOptions.MergeFile;
            }

            // 同一頂点の削除
            CheckUnsupportedOption(fileOption, () => fileOption.DeleteNearVertex);

            // サブメッシュ分割
            CheckUnsupportedOption(fileOption, () => fileOption.UniteSubmesh);
            CheckUnsupportedOption(fileOption, () => fileOption.DivideSubmesh);

            // PolygonReduction
            CheckUnsupportedOption(fileOption, () => fileOption.PolygonReduction);
            CheckUnsupportedOption(fileOption, () => fileOption.PolygonReductionOptions);

            // プリミティブ最適化
            CheckUnsupportedOption(fileOption, () => fileOption.OptimizePrimitive);

            // LOD モデル結合・削除
            CheckUnsupportedOption(fileOption, () => fileOption.UniteLodModel);
            CheckUnsupportedOption(fileOption, () => fileOption.TrimLodModel);

            // マテリアルアニメ統合
            // fmaファイルの場合はオプションファイルが指定されていない場合はエラーとする
            if (fileOption.Combine && fileOption.CombineOptions != null)
            {
                string errorPath;
                if (!IfMaterialAnimCombineUtility.CheckCombinePaths(fileOption.CombineOptions.Paths, out errorPath))
                {
                    Strings.Throw("Error_Combine_InvalidFilePath", errorPath);
                }
                CombineFilePaths = fileOption.CombineOptions.Paths;
                CombineIgnoreFrameLoopUnmatch = fileOption.CombineOptions.IgnoreFrameLoopUnmatch;
                CombineIgnoreAnimationConflict = fileOption.CombineOptions.IgnoreAnimationConflict;
                this.CombineProcessLogArgument = fileOption.CombineOptions.CreateProcessLogArgument();
            }

            // 量子化分析
            if (fileOption.QuantizationAnalysis)
            {
                this.Optimizers.Add(new IfMaterialAnimQuantizationAnalysisOptimizer());
            }
        }

        //=====================================================================
        // 最適化
        internal override void Optimize()
        {
            // 読み込み
            Read();
            var materialAnim = this.nw4f_3dif.Item as material_animType;

            // データの正当性チェック
            Check(materialAnim);

            // 修正
            Repair(materialAnim);

            // マージ
            Merge(materialAnim);

            // マテリアルアニメ統合
            Combine(materialAnim);

            // シェーダパラメータアニメ最適化
            foreach (var optimizer in this.Optimizers)
            {
                optimizer.Optimize(materialAnim, this.Streams);

                // 最適化ログの追加
                if (optimizer.EnableProcessLog)
                {
                    IfProcessLogUtility.Add(this.nw4f_3dif,
                        optimizer.Process, optimizer.Argument, optimizer.GetResult());
                }
            }

            IfProcessLogUtility.ToDistinctLatest(this.nw4f_3dif);

            // 書き出し
            Write();
        }

        //---------------------------------------------------------------------
        // 確認
        private void Check(material_animType materialAnim)
        {
            if (!CheckFile) { return; }

            // ストリームインデックスの正当性を確認する
            string checkLog = string.Empty;
            bool checkResult =
                StreamChecker.CheckStreamIndex(out checkLog, materialAnim, this.Streams);
            // チェックが失敗したらログを例外として投げて表示させる
            if (!checkResult)
            {
                throw new Exception(checkLog);
            }
            else
            {
                Console.WriteLine("ok.");
            }
        }

        //---------------------------------------------------------------------
        // 修正
        private void Repair(material_animType materialAnim)
        {
            if (!RepairFile) { return; }

            StreamUtility.SortStream(materialAnim, this.Streams);
        }

        //---------------------------------------------------------------------
        // マージ
        private void Merge(material_animType materialAnim)
        {
            if (this.MergeFilePath == null) { return; }

            this.MergeStopwatch.Start();
            // マージファイルの読み込み
            var oldStreams = new List<G3dStream>();
            var oldIf = IfReadUtility.Read(
                oldStreams, this.MergeFilePath, this.XsdBasePath);
            var oldShaderParamAnim = oldIf.Item as material_animType;
            IfMaterialAnimMerger.Merge(
                materialAnim,
                this.Streams,
                oldShaderParamAnim,
                oldStreams);

            // 最適化ログの追加
            IfProcessLogUtility.Add(this.nw4f_3dif, "merge",
                Path.GetFullPath(this.MergeFilePath).Replace('\\', '/'), string.Empty);

            this.MergeStopwatch.Stop();
        }

        //---------------------------------------------------------------------
        // マテリアルアニメ統合
        private void Combine(material_animType materialAnim)
        {
            if (this.CombineFilePaths == null) { return; }
            this.CombineStopwatch.Start();

            var combineSettings = new CombineSettings()
            {
                XsdBasePath = this.XsdBasePath,
                IgnoreFrameLoopUnmatch = this.CombineIgnoreFrameLoopUnmatch,
                IgnoreAnimationConflict = this.CombineIgnoreAnimationConflict
            };
            if (!IfMaterialAnimCombineUtility.Combine(materialAnim, this.Streams, this.CombineFilePaths, combineSettings))
            {
                throw new Exception(combineSettings.MessageLog);
            }

            // 統合ログの追加
            IfProcessLogUtility.Add(this.nw4f_3dif, "combine", this.CombineProcessLogArgument, string.Empty);
            this.CombineStopwatch.Stop();
        }

        //---------------------------------------------------------------------
        // ログの取得
        internal override string GetLog()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendLine(base.GetLog());
            builder.AppendLine(GetReadLog());

            // マージ
            if (this.MergeFilePath != null)
            {
                builder.AppendLine(Strings.Get("MergeLog",
                    this.MergeStopwatch.ElapsedMilliseconds,
                    this.MergeFilePath));
            }

            // 統合
            if (this.CombineFilePaths != null)
            {
                builder.AppendLine(Strings.Get("CombineLog",
                    this.CombineStopwatch.ElapsedMilliseconds,
                    string.Join(",", this.CombineFilePaths.Select(s => Path.GetFullPath(s).Replace('\\', '/')))));
            }

            // シェーダパラメータアニメ最適化
            foreach (var optimizer in this.Optimizers)
            {
                builder.AppendLine("    " + optimizer.ToString());
            }

            builder.Append(GetWriteLog());
            return builder.ToString();
        }

        //=====================================================================
        private readonly string MergeFilePath;
        private readonly bool RepairFile = false;
        private readonly bool CheckFile = false;
        private readonly string[] CombineFilePaths;
        private readonly bool CombineIgnoreAnimationConflict = true;
        private readonly bool CombineIgnoreFrameLoopUnmatch = false;
        private readonly string CombineProcessLogArgument = string.Empty;
        protected readonly Stopwatch CombineStopwatch = new Stopwatch();
        protected readonly Stopwatch MergeStopwatch = new Stopwatch();
        private List<IfMaterialAnimOptimizer> Optimizers =
            new List<IfMaterialAnimOptimizer>();
    }
}
