﻿using Nintendo.ToolFoundation.CommandLine;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace _3dIntermediateFileMaterialGenerator
{
    public class Context
    {
        public enum SubcommandType
        {
            Create,
            Add,
            Remove,
            Rename,
        }

        public string XsdBasePath { get; set; } = null;

        public string InputFmdPath { get; private set; } = string.Empty;

        public string OutputFmdPath { get; private set; } = string.Empty;

        public List<string> MaterialNames { get; } = new List<string>();

        public string NewMaterialName { get; private set; } = string.Empty;

        public string OldMaterialName { get; private set; } = string.Empty;

        public SubcommandType Subcommand { get; private set; }

        public Context(string[] args)
        {
            // 引数パース
            var rootCommand = ActionCommand.CreateRootCommand(true);
            rootCommand.GetBuilder().SetDescription("マテリアルを生成、編集するコマンドラインツールです。");
            rootCommand.AddFlagOption('s', "silent", () => this.isSilent = true)
                .GetBuilder()
                .SetDescription("ログを無効にします。");

            {
                var subcommand = rootCommand.AddCommand("create", commandName => Subcommand = SubcommandType.Create);
                subcommand.GetBuilder()
                    .SetDescription("モデル中間ファイルを作成します。");

                subcommand.AddValueOption("material-names", names =>
                {
                    string[] splitedNames = names.Split(',');
                    foreach (string name in splitedNames)
                    {
                        MaterialNames.Add(name.Trim());
                    }
                })
                    .GetBuilder()
                    .Require()
                    .SetDescription("生成するマテリアル名を指定します。\"matX,matY\" のようにカンマで区切ることで複数のマテリアルを指定できます。");

                subcommand.AddValueOption('o', "output", path => this.OutputFmdPath = path)
                    .GetBuilder()
                    .Require()
                    .SetDescription("出力ファイルパスを指定します。")
                    .SetValueName("path");
            }

            {
                var subcommand = rootCommand.AddCommand("add", commandName => Subcommand = SubcommandType.Add);
                subcommand.GetBuilder()
                    .SetDescription("既存のモデル中間ファイルにマテリアルを追加します。");

                subcommand.AddValue(1, input => this.InputFmdPath = input)
                    .GetBuilder().Require()
                    .SetDescription("マテリアルを追加するファイルを指定します。");

                subcommand.AddValueOption("material-names", names =>
                {
                    string[] splitedNames = names.Split(',');
                    foreach (string name in splitedNames)
                    {
                        MaterialNames.Add(name.Trim());
                    }
                })
                    .GetBuilder()
                    .Require()
                    .SetDescription("追加するマテリアル名を指定します。\"matX,matY\" のようにカンマで区切ることで複数のマテリアルを指定できます。");

                subcommand.AddValueOption('o', "output", path => this.OutputFmdPath = path)
                    .GetBuilder()
                    .SetDescription("出力ファイルパスを指定します。")
                    .SetValueName("path");
            }

            {
                var subcommand = rootCommand.AddCommand("remove", commandName => Subcommand = SubcommandType.Remove);
                subcommand.GetBuilder()
                    .SetDescription("既存のモデル中間ファイルからマテリアルを削除します。");

                subcommand.AddValue(1, input => this.InputFmdPath = input)
                    .GetBuilder().Require()
                    .SetDescription("マテリアルを削除するファイルを指定します。");

                subcommand.AddValueOption("material-names", names =>
                {
                    string[] splitedNames = names.Split(',');
                    foreach (string name in splitedNames)
                    {
                        MaterialNames.Add(name.Trim());
                    }
                })
                    .GetBuilder()
                    .Require()
                    .SetDescription("削除するマテリアル名を指定します。\"matX,matY\" のようにカンマで区切ることで複数のマテリアルを指定できます。");

                subcommand.AddValueOption('o', "output", path => this.OutputFmdPath = path)
                    .GetBuilder()
                    .SetDescription("出力ファイルパスを指定します。")
                    .SetValueName("path");
            }

            {
                var subcommand = rootCommand.AddCommand("rename", commandName => Subcommand = SubcommandType.Rename);
                subcommand.GetBuilder()
                    .SetDescription("既存のモデル中間ファイルのマテリアル名を変更します。");

                subcommand.AddValue(1, input => this.InputFmdPath = input)
                    .GetBuilder()
                    .Require()
                    .SetDescription("マテリアル名を変更するファイルを指定します。");

                subcommand.AddValueOption("old-material-name", name =>
                {
                    OldMaterialName = name;
                })
                    .GetBuilder().Require()
                    .SetDescription("名前を変更するマテリアル名を指定します。");

                subcommand.AddValueOption("new-material-name", name =>
                {
                    NewMaterialName = name;
                })
                    .GetBuilder().Require()
                    .SetDescription("変更後の名前を指定します。");

                subcommand.AddValueOption('o', "output", path => this.OutputFmdPath = path)
                    .GetBuilder()
                    .SetDescription("出力ファイルパスを指定します。")
                    .SetValueName("path");
            }

            CommandLine.ParseArgs(args, rootCommand, new ParseSettings()
            {
                ErrorAction = message =>
                {
                    throw new Exception($"オプションが正しくありません。{message}\n{CommandLine.GetHelpText(args, rootCommand)}");
                },
                HelpWriter = this.WriteMessage
            });

            if (string.IsNullOrEmpty(this.InputFmdPath) && string.IsNullOrEmpty(this.OutputFmdPath))
            {
                throw new Exception("入力ファイルの指定がないときは、出力ファイルの指定が必要です。");
            }

            if (MaterialNames.Count == 0)
            {
                MaterialNames.Add("material1");
            }
        }

        public void WriteMessage(string message)
        {
            if (this.isSilent)
            {
                return;
            }

            Console.WriteLine(message);
        }

        public void WriteErrorMessage(string message)
        {
            Console.Error.WriteLine($"エラー: {message}");
        }

        [Conditional("DEBUG")]
        public void WriteWarningMessage(string message)
        {
            Console.Error.WriteLine($"警告: {message}");
        }

        private bool isSilent = false;
    }
}
