﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using Nintendo.G3dTool.Entities;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public interface ITextureConverter
    {
        // プラットフォーム固有のコンバータを初期化する。
        // basepath にはコンバータに必要なDLLのパスを渡す
        bool Initialize(string[] basePath);

        // コンバータのプラットフォーム名
        string PlatformName { get; }

        void Destroy();

        Bitmap[] ConvertTo1d2dBitmap(textureType texture, List<G3dStream> streams);

        Bitmap[][] ConvertTo2dArrayBitmap(textureType texture, List<G3dStream> streams);

        Bitmap[][] ConvertTo3dBitmap(textureType texture, List<G3dStream> streams);

        Bitmap[][] ConvertToCubeArrayBitmap(textureType texture, List<G3dStream> streams);

        Bitmap[][] ConvertToCubeBitmap(textureType texture, List<G3dStream> streams);

        Bitmap[][] ConvertTo1dArrayBitmap(textureType texture, List<G3dStream> streams);

        TextureData[] ConvertTo1d2dStream(textureType texture, List<G3dStream> streams);

        TextureData[][] ConvertTo3dStream(textureType texture, List<G3dStream> streams);

        TextureData[][] ConvertToCubeStream(textureType texture, List<G3dStream> streams);

        TextureData[][] ConvertTo1dArrayStream(textureType texture, List<G3dStream> streams);

        TextureData[][] ConvertTo2dArrayStream(textureType texture, List<G3dStream> streams);

        TextureData[][] ConvertToCubeArrayStream(textureType texture, List<G3dStream> streams);


        Bitmap[] ConvertTo1d2dBitmap(Texture texture);

        Bitmap[][] ConvertTo2dArrayBitmap(Texture texture);

        Bitmap[][] ConvertTo3dBitmap(Texture texture);

        Bitmap[][] ConvertToCubeArrayBitmap(Texture texture);

        Bitmap[][] ConvertToCubeBitmap(Texture texture);

        Bitmap[][] ConvertTo1dArrayBitmap(Texture texture);

        TextureData[] ConvertTo1d2dStream(Texture texture);

        TextureData[][] ConvertTo3dStream(Texture texture);

        TextureData[][] ConvertToCubeStream(Texture texture);

        TextureData[][] ConvertTo1dArrayStream(Texture texture);

        TextureData[][] ConvertTo2dArrayStream(Texture texture);

        TextureData[][] ConvertToCubeArrayStream(Texture texture);
    }

    public class TextureData
    {
        public texture_info_quantize_typeType QuantizeType { get; set; }
        public texture_info_dimensionType DimensionType { get; set; }
        public Bitmap OriginalImage { get; set; }
        public byte[] ByteImage { get; set; }
        public float[] FloatImage { get; set; }
        public int Width { get; set; }
        public int Height { get; set; }
        public int Depth { get; set; }
        public bool IsFloat { get; set; }
    }

    public class OriginalImageConverter : ITextureConverter
    {
        public bool Initialize(string[] basePath)
        {
            return true;
        }

        public string PlatformName
        {
            get
            {
                return "OriginalImage";
            }
        }

        public void Destroy()
        {
        }

        public Bitmap[] ConvertTo1d2dBitmap(textureType texture, List<G3dStream> streams)
        {
            return OriginalImageUtility.ConvertOriginalImageToBitmaps(texture, streams);
        }

        public Bitmap[][] ConvertTo2dArrayBitmap(textureType texture, List<G3dStream> streams)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture, streams);
        }

        public Bitmap[][] ConvertTo3dBitmap(textureType texture, List<G3dStream> streams)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture, streams);
        }

        public Bitmap[][] ConvertToCubeArrayBitmap(textureType texture, List<G3dStream> streams)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture, streams);
        }

        public Bitmap[][] ConvertToCubeBitmap(textureType texture, List<G3dStream> streams)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture, streams);
        }

        public Bitmap[][] ConvertTo1dArrayBitmap(textureType texture, List<G3dStream> streams)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture, streams);
        }

        public TextureData[] ConvertTo1d2dStream(textureType texture, List<G3dStream> streams)
        {
            var bitmaps = ConvertTo1d2dBitmap(texture, streams);
            var results = new TextureData[bitmaps.Length];
            for (var i = 0; i < bitmaps.Length; ++i)
            {
                var info = texture.texture_info;
                results[i] = new TextureData
                {
                    QuantizeType = info.quantize_type,
                    DimensionType = info.dimension,
                    Width = info.width,
                    Height = info.height,
                    Depth = info.depth,
                    IsFloat = false,
                    OriginalImage = bitmaps[i],
                    ByteImage = null,
                    FloatImage = null
                };
            }

            return results;
        }

        public TextureData[][] ConvertTo3dStream(textureType texture, List<G3dStream> streams)
        {
            return null;
        }

        public TextureData[][] ConvertToCubeStream(textureType texture, List<G3dStream> streams)
        {
            return null;
        }

        public TextureData[][] ConvertTo1dArrayStream(textureType texture, List<G3dStream> streams)
        {
            return null;
        }

        public TextureData[][] ConvertTo2dArrayStream(textureType texture, List<G3dStream> streams)
        {
            return null;
        }

        public TextureData[][] ConvertToCubeArrayStream(textureType texture, List<G3dStream> streams)
        {
            return null;
        }

        public Bitmap[] ConvertTo1d2dBitmap(Texture texture)
        {
            return OriginalImageUtility.ConvertOriginalImageToBitmaps(texture);
        }

        public Bitmap[][] ConvertTo2dArrayBitmap(Texture texture)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture);
        }

        public Bitmap[][] ConvertTo3dBitmap(Texture texture)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture);
        }

        public Bitmap[][] ConvertToCubeArrayBitmap(Texture texture)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture);
        }

        public Bitmap[][] ConvertToCubeBitmap(Texture texture)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture);
        }

        public Bitmap[][] ConvertTo1dArrayBitmap(Texture texture)
        {
            return OriginalImageUtility.ConvertOriginalImageArrayToBitmapArrays(texture);
        }

        public TextureData[] ConvertTo1d2dStream(Texture texture)
        {
            var bitmaps = ConvertTo1d2dBitmap(texture);
            var results = new TextureData[bitmaps.Length];
            for (var i = 0; i < bitmaps.Length; ++i)
            {
                var info = texture.TextureInfo;
                results[i] = new TextureData
                {
                    QuantizeType = info.QuantizeType,
                    DimensionType = info.Dimension,
                    Width = info.Width,
                    Height = info.Height,
                    Depth = info.Depth,
                    IsFloat = false,
                    OriginalImage = bitmaps[i],
                    ByteImage = null,
                    FloatImage = null
                };
            }

            return results;
        }

        public TextureData[][] ConvertTo3dStream(Texture texture)
        {
            return null;
        }

        public TextureData[][] ConvertToCubeStream(Texture texture)
        {
            return null;
        }

        public TextureData[][] ConvertTo1dArrayStream(Texture texture)
        {
            return null;
        }

        public TextureData[][] ConvertTo2dArrayStream(Texture texture)
        {
            return null;
        }

        public TextureData[][] ConvertToCubeArrayStream(Texture texture)
        {
            return null;
        }
    }

    public static class OriginalImageUtility
    {
        // TODO: textureType を引数に取るインターフェースはいつか消す

        /// <summary>
        /// テクスチャのオリジナルイメージから指定されたインデックスのビットマップを作成します。
        /// ミップマップが有る場合は生成して返します。
        /// </summary>
        /// <param name="texture"></param>
        /// <param name="streams"></param>
        /// <param name="originalImageIndex"></param>
        /// <returns>各ミップマップを含むBitmap配列</returns>
        public static Bitmap[] ConvertOriginalImageToBitmaps(textureType texture, List<G3dStream> streams, int originalImageIndex = 0)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.NotNull(texture);

            var info = texture.texture_info;
            var bitmaps = new Bitmap[info.mip_level];

            // オリジナルイメージが存在しないテクスチャの場合はサイズなど合わせた赤色のビットマップを返す
            if (texture.original_image_array?.original_image == null)
            {
                var width = info.width;
                var height = info.height;
                for (var i = 0; i < info.mip_level; i++)
                {
                    var dst = new Bitmap(width, height, PixelFormat.Format32bppArgb);
                    using (var g = Graphics.FromImage(dst))
                    {
                        g.Clear(Color.Black);
                    }
                    bitmaps[i] = dst;
                    width = Math.Max(width >> 1, 1);
                    height = Math.Max(height >> 1, 1);
                }
                return bitmaps;
            }

            Nintendo.Foundation.Contracts.Assertion.Operation.True(texture.original_image_array.original_image.Length > originalImageIndex);
            var orgImage = texture.original_image_array.original_image[originalImageIndex];
            for (var i = 0; i < info.mip_level; i++)
            {
                Bitmap dst;

                if (i == 0)
                {
                    var format = orgImage.format;
                    var stream = streams[orgImage.stream_index];
                    var width = orgImage.width;
                    var height = orgImage.height;

                    dst = new Bitmap(width, height, PixelFormat.Format32bppArgb);
                    var dstBitmapData = dst.LockBits(
                        new Rectangle(0, 0, dst.Width, dst.Height),
                        ImageLockMode.ReadWrite,
                        dst.PixelFormat);
                    unsafe
                    {
                        var dstBmpPtr = (byte*)(void*)dstBitmapData.Scan0;
                        var dstStride = dstBitmapData.Stride;
                        Nintendo.Foundation.Contracts.Assertion.Operation.True(dstBmpPtr != null);
                        switch (format)
                        {
                            case original_image_formatType.rgb8:
                                {
                                    const int Step = 3;
                                    G3dParallel.For(0, height, y =>
                                        {
                                            var dstIdx = y * dstStride;
                                            var srcIdx = y * width * Step;
                                            for (var x = 0; x < width; x++)
                                            {
                                                dstBmpPtr[dstIdx + 2] = stream.ByteData[srcIdx + 0]; // R
                                                dstBmpPtr[dstIdx + 1] = stream.ByteData[srcIdx + 1]; // G
                                                dstBmpPtr[dstIdx + 0] = stream.ByteData[srcIdx + 2]; // B
                                                dstBmpPtr[dstIdx + 3] = 255; // A
                                                dstIdx += 4;
                                                srcIdx += Step;
                                            }
                                    });
                                }
                                break;
                            case original_image_formatType.rgba8:
                                {
                                    const int Step = 4;
                                    G3dParallel.For(0, height, y =>
                                    {
                                        var dstIdx = y * dstStride;
                                        var srcIdx = y * width * Step;
                                        for (var x = 0; x < width; x++)
                                        {
                                            dstBmpPtr[dstIdx + 2] = stream.ByteData[srcIdx + 0]; // R
                                            dstBmpPtr[dstIdx + 1] = stream.ByteData[srcIdx + 1]; // G
                                            dstBmpPtr[dstIdx + 0] = stream.ByteData[srcIdx + 2]; // B
                                            dstBmpPtr[dstIdx + 3] = stream.ByteData[srcIdx + 3]; // A
                                            dstIdx += 4;
                                            srcIdx += Step;
                                        }
                                    });
                                }
                                break;
                            case original_image_formatType.rgb32f:
                                {
                                    const int Step = 3;
                                    G3dParallel.For(0, height, y =>
                                    {
                                        var dstIdx = y * dstStride;
                                        var srcIdx = y * width * Step;
                                        for (var x = 0; x < width; x++)
                                        {
                                            dstBmpPtr[dstIdx + 2] = LinearFloatToSrgbByte(stream.FloatData[srcIdx + 0]); // R
                                            dstBmpPtr[dstIdx + 1] = LinearFloatToSrgbByte(stream.FloatData[srcIdx + 1]); // G
                                            dstBmpPtr[dstIdx + 0] = LinearFloatToSrgbByte(stream.FloatData[srcIdx + 2]); // B
                                            dstBmpPtr[dstIdx + 3] = 255; // A
                                            dstIdx += 4;
                                            srcIdx += Step;
                                        }
                                    });
                                }
                                break;
                            case original_image_formatType.rgba32f:
                                {
                                    const int Step = 4;
                                    G3dParallel.For(0, height, y =>
                                    {
                                        var dstIdx = y * dstStride;
                                        var srcIdx = y * width * Step;
                                        for (var x = 0; x < width; x++)
                                        {
                                            dstBmpPtr[dstIdx + 2] = LinearFloatToSrgbByte(stream.FloatData[srcIdx + 0]); // R
                                            dstBmpPtr[dstIdx + 1] = LinearFloatToSrgbByte(stream.FloatData[srcIdx + 1]); // G
                                            dstBmpPtr[dstIdx + 0] = LinearFloatToSrgbByte(stream.FloatData[srcIdx + 2]); // B
                                            dstBmpPtr[dstIdx + 3] = (byte)(255 * stream.FloatData[srcIdx + 3]); // A
                                            dstIdx += 4;
                                            srcIdx += Step;
                                        }
                                    });
                                }
                                break;
                            default:
                                Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                                break;
                        }
                    }
                    dst.UnlockBits(dstBitmapData);
                }
                else
                {
                    // オリジナルイメージにはミップマップが含まれないので生成する。
                    // 高速化のために一つ上レベルのミップマップからサンプルする。
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(i > 0 && bitmaps[i - 1] != null);
                    var src = bitmaps[i - 1];
                    dst = new Bitmap(Math.Max(src.Width >> 1, 1), Math.Max(src.Height >> 1, 1));
                    var srcBitmapData = src.LockBits(
                        new Rectangle(0, 0, src.Width, src.Height),
                        ImageLockMode.ReadOnly,
                        src.PixelFormat);
                    var dstBitmapData = dst.LockBits(
                        new Rectangle(0, 0, dst.Width, dst.Height),
                        ImageLockMode.ReadWrite,
                        dst.PixelFormat);
                    unsafe
                    {
                        var srcBmpPtr = (uint*)(void*)srcBitmapData.Scan0;
                        var dstBmpPtr = (uint*)(void*)dstBitmapData.Scan0;
                        Nintendo.Foundation.Contracts.Assertion.Operation.True(srcBmpPtr != null && dstBmpPtr != null);
                        var srcStride = srcBitmapData.Stride >> 2;
                        var dstStride = dstBitmapData.Stride >> 2;
                        var width = dst.Width;
                        var height = dst.Height;

                        G3dParallel.For(
                            0,
                            height,
                            y =>
                                {
                                    var dstIdx = y * dstStride;
                                    var srcIdx = y * 2 * srcStride;
                                    for (var x = 0; x < width; x++)
                                    {
                                        dstBmpPtr[dstIdx] = srcBmpPtr[srcIdx];
                                        dstIdx += 1;
                                        srcIdx += 2;
                                    }
                                });
                    }
                    src.UnlockBits(srcBitmapData);
                    dst.UnlockBits(dstBitmapData);
                }
                bitmaps[i] = dst;
            }

            return bitmaps;
        }


        /// <summary>
        /// テクスチャのオリジナルイメージから指定されたインデックスのビットマップを作成します。
        /// ミップマップが有る場合は生成して返します。
        /// </summary>
        /// <param name="texture"></param>
        /// <param name="streams"></param>
        /// <param name="originalImageIndex"></param>
        /// <returns>各ミップマップを含むBitmap配列</returns>
        public static Bitmap[] ConvertOriginalImageToBitmaps(Texture texture, int originalImageIndex = 0)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.NotNull(texture);

            var info = texture.TextureInfo;
            var bitmaps = new Bitmap[info.MipLevel];

            // オリジナルイメージが存在しないテクスチャの場合はサイズなど合わせた赤色のビットマップを返す
            if (texture.OriginalImages.Count == 0)
            {
                var width = info.Width;
                var height = info.Height;
                for (var i = 0; i < info.MipLevel; i++)
                {
                    var dst = new Bitmap(width, height, PixelFormat.Format32bppArgb);
                    using (var g = Graphics.FromImage(dst))
                    {
                        g.Clear(Color.Black);
                    }
                    bitmaps[i] = dst;
                    width = Math.Max(width >> 1, 1);
                    height = Math.Max(height >> 1, 1);
                }
                return bitmaps;
            }

            Nintendo.Foundation.Contracts.Assertion.Operation.True(texture.OriginalImages.Count > originalImageIndex);
            var orgImage = texture.OriginalImages[originalImageIndex];
            for (var i = 0; i < info.MipLevel; i++)
            {
                Bitmap dst;

                if (i == 0)
                {
                    var format = orgImage.Format;
                    var width = orgImage.Width;
                    var height = orgImage.Height;

                    dst = new Bitmap(width, height, PixelFormat.Format32bppArgb);
                    var dstBitmapData = dst.LockBits(
                        new Rectangle(0, 0, dst.Width, dst.Height),
                        ImageLockMode.ReadWrite,
                        dst.PixelFormat);
                    unsafe
                    {
                        var dstBmpPtr = (byte*)(void*)dstBitmapData.Scan0;
                        var dstStride = dstBitmapData.Stride;
                        Nintendo.Foundation.Contracts.Assertion.Operation.True(dstBmpPtr != null);
                        switch (format)
                        {
                            case original_image_formatType.rgb8:
                                {
                                    var stream = orgImage.Stream as StreamByte;
                                    const int Step = 3;
                                    G3dParallel.For(0, height, y =>
                                    {
                                        var dstIdx = y * dstStride;
                                        var srcIdx = y * width * Step;
                                        for (var x = 0; x < width; x++)
                                        {
                                            dstBmpPtr[dstIdx + 2] = stream.Values[srcIdx + 0]; // R
                                            dstBmpPtr[dstIdx + 1] = stream.Values[srcIdx + 1]; // G
                                            dstBmpPtr[dstIdx + 0] = stream.Values[srcIdx + 2]; // B
                                            dstBmpPtr[dstIdx + 3] = 255; // A
                                            dstIdx += 4;
                                            srcIdx += Step;
                                        }
                                    });
                                }
                                break;
                            case original_image_formatType.rgba8:
                                {
                                    var stream = orgImage.Stream as StreamByte;
                                    const int Step = 4;
                                    G3dParallel.For(0, height, y =>
                                    {
                                        var dstIdx = y * dstStride;
                                        var srcIdx = y * width * Step;
                                        for (var x = 0; x < width; x++)
                                        {
                                            dstBmpPtr[dstIdx + 2] = stream.Values[srcIdx + 0]; // R
                                            dstBmpPtr[dstIdx + 1] = stream.Values[srcIdx + 1]; // G
                                            dstBmpPtr[dstIdx + 0] = stream.Values[srcIdx + 2]; // B
                                            dstBmpPtr[dstIdx + 3] = stream.Values[srcIdx + 3]; // A
                                            dstIdx += 4;
                                            srcIdx += Step;
                                        }
                                    });
                                }
                                break;
                            case original_image_formatType.rgb32f:
                                {
                                    var stream = orgImage.Stream as StreamFloat;
                                    const int Step = 3;
                                    G3dParallel.For(0, height, y =>
                                    {
                                        var dstIdx = y * dstStride;
                                        var srcIdx = y * width * Step;
                                        for (var x = 0; x < width; x++)
                                        {
                                            dstBmpPtr[dstIdx + 2] = LinearFloatToSrgbByte(stream.Values[srcIdx + 0]); // R
                                            dstBmpPtr[dstIdx + 1] = LinearFloatToSrgbByte(stream.Values[srcIdx + 1]); // G
                                            dstBmpPtr[dstIdx + 0] = LinearFloatToSrgbByte(stream.Values[srcIdx + 2]); // B
                                            dstBmpPtr[dstIdx + 3] = 255; // A
                                            dstIdx += 4;
                                            srcIdx += Step;
                                        }
                                    });
                                }
                                break;
                            case original_image_formatType.rgba32f:
                                {
                                    var stream = orgImage.Stream as StreamFloat;
                                    const int Step = 4;
                                    G3dParallel.For(0, height, y =>
                                    {
                                        var dstIdx = y * dstStride;
                                        var srcIdx = y * width * Step;
                                        for (var x = 0; x < width; x++)
                                        {
                                            dstBmpPtr[dstIdx + 2] = LinearFloatToSrgbByte(stream.Values[srcIdx + 0]); // R
                                            dstBmpPtr[dstIdx + 1] = LinearFloatToSrgbByte(stream.Values[srcIdx + 1]); // G
                                            dstBmpPtr[dstIdx + 0] = LinearFloatToSrgbByte(stream.Values[srcIdx + 2]); // B
                                            dstBmpPtr[dstIdx + 3] = (byte)(255 * stream.Values[srcIdx + 3]); // A
                                            dstIdx += 4;
                                            srcIdx += Step;
                                        }
                                    });
                                }
                                break;
                            default:
                                Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                                break;
                        }
                    }
                    dst.UnlockBits(dstBitmapData);
                }
                else
                {
                    // オリジナルイメージにはミップマップが含まれないので生成する。
                    // 高速化のために一つ上レベルのミップマップからサンプルする。
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(i > 0 && bitmaps[i - 1] != null);
                    var src = bitmaps[i - 1];
                    dst = new Bitmap(Math.Max(src.Width >> 1, 1), Math.Max(src.Height >> 1, 1));
                    var srcBitmapData = src.LockBits(
                        new Rectangle(0, 0, src.Width, src.Height),
                        ImageLockMode.ReadOnly,
                        src.PixelFormat);
                    var dstBitmapData = dst.LockBits(
                        new Rectangle(0, 0, dst.Width, dst.Height),
                        ImageLockMode.ReadWrite,
                        dst.PixelFormat);
                    unsafe
                    {
                        var srcBmpPtr = (uint*)(void*)srcBitmapData.Scan0;
                        var dstBmpPtr = (uint*)(void*)dstBitmapData.Scan0;
                        Nintendo.Foundation.Contracts.Assertion.Operation.True(srcBmpPtr != null && dstBmpPtr != null);
                        var srcStride = srcBitmapData.Stride >> 2;
                        var dstStride = dstBitmapData.Stride >> 2;
                        var width = dst.Width;
                        var height = dst.Height;

                        G3dParallel.For(
                            0,
                            height,
                            y =>
                            {
                                var dstIdx = y * dstStride;
                                var srcIdx = y * 2 * srcStride;
                                for (var x = 0; x < width; x++)
                                {
                                    dstBmpPtr[dstIdx] = srcBmpPtr[srcIdx];
                                    dstIdx += 1;
                                    srcIdx += 2;
                                }
                            });
                    }
                    src.UnlockBits(srcBitmapData);
                    dst.UnlockBits(dstBitmapData);
                }
                bitmaps[i] = dst;
            }

            return bitmaps;
        }

        /// <summary>
        /// リニア空間のfloatをsRGB空間のbyteに変換
        /// </summary>
        /// <param name="v"></param>
        /// <returns></returns>
        private static byte LinearFloatToSrgbByte(float v)
        {
            var r = v <= 0.0031308 ? v * 12.92 : Math.Pow(v, 1 / 2.4) * 1.055 - 0.055;
            r = Math.Min(Math.Max(0.0, r), 1.0);
            return (byte)(255 * r);
        }

        /// <summary>
        /// テクスチャの全てのオリジナルイメージからビットマップを作成します。
        /// ミップマップが有る場合は生成して返します。
        /// </summary>
        /// <param name="texture"></param>
        /// <param name="streams"></param>
        /// <returns>
        /// 各テクスチャとそのミップマップからなるジャグ配列("配列の配列")を返します。
        /// {
        ///     new Bitmap[]{tex1_mip0, tex1_mip1, tex1_mip2, tex1_mip3, ...},
        ///     new Bitmap[]{tex2_mip0, tex2_mip1, tex2_mip2, tex2_mip3, ...},
        ///     new Bitmap[]{tex3_mip0, tex3_mip1, tex3_mip2, tex3_mip3, ...},
        ///     ...
        /// }
        /// このような形の配列を返します。
        /// </returns>
        public static Bitmap[][] ConvertOriginalImageArrayToBitmapArrays(textureType texture, List<G3dStream> streams)
        {
            var info = texture.texture_info;
            var bitmaps = new Bitmap[info.depth][];
            for (var d = 0; d < info.depth; d++)
            {
                bitmaps[d] = ConvertOriginalImageToBitmaps(texture, streams, d);
            }
            return bitmaps;
        }

        public static Bitmap[][] ConvertOriginalImageArrayToBitmapArrays(Texture texture)
        {
            var info = texture.TextureInfo;
            var bitmaps = new Bitmap[info.Depth][];
            for (var d = 0; d < info.Depth; d++)
            {
                bitmaps[d] = ConvertOriginalImageToBitmaps(texture, d);
            }
            return bitmaps;
        }
    }
}
