﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public static class IfShaderOptionUtility
    {
        /// <summary>
        /// オプションの比較
        /// </summary>
        public static IfShaderAssignUtility.CompareResult Compare(IEnumerable<shader_optionType> options, shading_modelType shadingModel)
        {
            IEnumerable<option_varType> option_vars = shadingModel.StaticOptions();

            // マテリアルに含まれないコンバイナーを比較対象から外す。
            if (options.Count() < option_vars.Count())
            {
                var optionIds = new HashSet<string>(options.Select(x => x.id));
                option_vars = option_vars.Where(x => (x.ui_item?.value != ui_item_valueType.combiner) || optionIds.Contains(x.id) );
            }

            // 個数の比較
            if (options.Count() != option_vars.Count())
            {
                return IfShaderAssignUtility.CompareResult.Conflict;
            }

            // id の比較
            bool sameOrder = true;
            if (!options.Select(x => x.id).SequenceEqual(option_vars.Select(x => x.id)))
            {
                if (options.Select(x => x.id).OrderBy(x => x).SequenceEqual(option_vars.Select(x => x.id).OrderBy(x => x)))
                {
                    sameOrder = false;
                }
                else
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            var pairs = from option in options
                        from var in option_vars
                        where option.id == var.id
                        select new { option, var };

            // choice の確認 (コンバイナーは対象外)
            foreach (var pair in pairs.Where(x => x.var.ui_item?.value != ui_item_valueType.combiner))
            {
                if (!IfShaderOptionUtility.IsOptionValueContainedInChoice(pair.option.value, pair.var.choice))
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            return sameOrder ? IfShaderAssignUtility.CompareResult.Ok : IfShaderAssignUtility.CompareResult.NotSameOrder;
        }

        /// <summary>
        /// シェーダーオプションを作成する
        /// </summary>
        internal static IEnumerable<shader_optionType> CreateOptions(shading_modelType shadingModel, IEnumerable<shader_optionType> oldOptions)
        {
            foreach (var option_var in shadingModel.StaticOptions())
            {
                var option = new shader_optionType()
                {
                    id = option_var.id,
                    value = option_var.@default,
                };

                // 元の値を割り当てる
                var old = oldOptions.FirstOrDefault(
                    x => x.id == option_var.id);

                // コンバイナーは choice 範囲外も有効。
                if (old != null && ((option_var.ui_item?.value == ui_item_valueType.combiner) || IsOptionValueContainedInChoice(old.value, option_var.choice)))
                {
                    option.value = old.value;
                }

                yield return option;
            }
        }

        /// <summary>
        /// オプション割り当て変更メッセージの取得
        /// </summary>
        internal static IEnumerable<string> GetUpdateOptionMessages(IEnumerable<shader_optionType> shaderOptions, shading_modelType shadingModel, bool ignoreOrder, IfShaderAssignUtility.InconsistentMessageType type)
        {
            // シェーダー定義側
            IEnumerable<option_varType> option_vars = shadingModel.StaticOptions();

            // 過剰のチェック
            var over = shaderOptions.Select(x => x.id).Except(option_vars.Select(x => x.id)).ToArray();
            foreach (var item in over)
            {
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Over_Option, type), item);
            }

            // 不足のチェック
            var lack = option_vars.Where(x => !shaderOptions.Any(y => y.id == x.id)).ToArray();
            foreach (var item in lack)
            {
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Lack_Option, type), item.id) +
                    string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Default, type), item.@default.Trim().Replace('\t', ' ').Replace('\n', ' '));
            }

            // 範囲のチェック
            var pairs = from shader_option in shaderOptions
                        from option_var in option_vars
                        where shader_option.id == option_var.id && (option_var.ui_item?.value != ui_item_valueType.combiner) && !IfShaderOptionUtility.IsOptionValueContainedInChoice(shader_option.value, option_var.choice)
                        select new { option = shader_option, var = option_var };

            foreach (var item in pairs)
            {
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Out_Choice_Option, type), item.option.id, item.option.value, item.var.choice) +
                    string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Default, type), item.var.@default.Trim().Replace('\t', ' ').Replace('\n', ' '));
            }

            // 順番の比較
            if (!ignoreOrder
                && over.Length == 0
                && lack.Length == 0
                && shaderOptions.Zip(option_vars, (x, y) => x.id == y.id).Any(x => !x))
            {
                yield return IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Disorder_Option, type);
            }
        }

        private static IEnumerable<option_varType> StaticOptions(this shading_modelType shadingModel)
        {
            return shadingModel.option_var_array != null ?
                shadingModel.option_var_array.option_var.Where(x => x.type == option_var_typeType.@static) :
                Enumerable.Empty<option_varType>();
        }

        internal enum OptionType
        {
            Bool,
            Int,
            Enum,
        }

        internal static OptionType Type(string choice)
        {
            if (choice == "bool")
            {
                return OptionType.Bool;
            }

            int min, max;
            if (IfShaderAssignUtility.TryParseIntRange(choice, out min, out max))
            {
                return OptionType.Int;
            }
            return OptionType.Enum;
        }

        /*
        /// <summary>
        /// chioce からオプションを取得
        /// </summary>
        public static string[] Enum(string choice)
        {
            return choice.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries)
                .Select(x => x.Split(new char[] { ':' })[0].Trim()).ToArray();
        }
         */

        /// <summary>
        /// chioce からオプションを取得
        /// </summary>
        public static string[] Enum(string choice)
        {
            return choice.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries)
                .Select(x =>
                    {
                        int pos = x.IndexOf(':');
                        if (pos < 0)
                        {
                            pos = x.Length;
                        }
                        return x.Substring(0, pos).Trim();
                    }).ToArray();
        }

        /// <summary>
        /// choice を オプション->エイリアス のDictionary にする。
        /// </summary>
        public static Dictionary<string, string> Alias(string choice)
        {
            var aliases = choice.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
            var dict = new Dictionary<string, string>();
            foreach (var option in aliases)
            {
                int pos = option.IndexOf(':');
                if (pos >= 0)
                {
                    dict.Add(option.Substring(0, pos).Trim(), option.Substring(pos + 1).Trim());
                }
            }
            return dict;
        }

        /// <summary>
        /// choice が等しいかどうか
        /// </summary>
        public static bool ChoiceEquivalent(string lhs, string rhs)
        {
            if (Type(lhs) != Type(rhs))
            {
                return false;
            }

            switch (Type(lhs))
            {
                case OptionType.Bool:
                    return true;
                case OptionType.Enum:
                    return Enum(lhs).OrderBy(x => x).SequenceEqual(Enum(rhs).OrderBy(x => x));
                case OptionType.Int:
                    {
                        int min0, min1, max0, max1;
                        IfShaderAssignUtility.TryParseIntRange(lhs, out min0, out max0);
                        IfShaderAssignUtility.TryParseIntRange(rhs, out min1, out max1);
                        return min0 == min1 && max0 == max1;
                    }
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// value が　choice に含まれるかどうか
        /// </summary>
        public static bool IsOptionValueContainedInChoice(string value, string choice)
        {
            // そのまま引き継いでいいかのチェック
            switch (Type(choice))
            {
                case OptionType.Bool:
                    {
                        try
                        {
                            G3dDataParser.ParseInt2BoolArray(value);
                        }
                        catch
                        {
                            return false;
                        }
                    }
                    break;
                case OptionType.Enum:
                    {
                        if (!Enum(choice).Contains(value))
                        {
                            return false;
                        }
                    }
                    break;
                case OptionType.Int:
                    {
                        try
                        {
                            int parsedValue;
                            if (!int.TryParse(value, out parsedValue))
                            {
                                return false;
                            }
                            int min = 0;
                            int max = 0;
                            if (!IfShaderAssignUtility.TryParseIntRange(choice, out min, out max))
                            {
                                return false;
                            }
                            else if (parsedValue < min || max < parsedValue)
                            {
                                return false;
                            }
                        }
                        catch
                        {
                            return false;
                        }
                    }
                    break;
            }

            return true;
        }

        /// <summary>
        /// option_varType のチェック
        /// </summary>
        public static OptionVarErrorType ValidateOptionVar(option_varType option)
        {
            switch (Type(option.choice))
            {
                case OptionType.Bool:
                    {
                        // default のチェック
                        if (!string.IsNullOrEmpty(option.@default))
                        {
                            bool[] defaults = null;
                            try
                            {
                                defaults = G3dDataParser.ParseInt2BoolArray(option.@default);
                            }
                            catch
                            {
                                return OptionVarErrorType.DefaultCannotParse;
                            }
                        }
                    }
                    break;
                case OptionType.Enum:
                    {
                        // 列挙値を整数値として扱うかどうか。
                        bool isInteger = false;

                        // choice のチェック
                        string[] choices = null;
                        if (!string.IsNullOrEmpty(option.choice))
                        {
                            // choice には英数字とアンダースコア、または整数値が設定されている。
                            do
                            {
                                OptionVarErrorType error;

                                // 整数値チェック
                                error = ValidateChoiceList(option, x => { int tmpValue; return int.TryParse(x, out tmpValue); }, out choices);
                                if (error == OptionVarErrorType.OK)
                                {
                                    // 整数値。
                                    isInteger = true;
                                    break;
                                }

                                // 列挙値チェック
                                error = ValidateChoiceList(option, x => { var match = stringChoiceItemRegex.Match(x); return x.Length != 0 && match.Success && match.Length == x.Length; }, out choices);
                                if (error == OptionVarErrorType.OK)
                                {
                                    // 列挙値。
                                    break;
                                }

                                // パースエラー。
                                return error;
                            }
                            while (false);
                        }

                        // default のチェック
                        if (!string.IsNullOrEmpty(option.@default))
                        {
                            var defaults = G3dDataParser.Tokenize(option.@default);

                            if (choices != null)
                            {
                                // default が choice に含まれるか
                                foreach (var value in defaults)
                                {
                                    if (!choices.Contains(value))
                                    {
                                        return OptionVarErrorType.DefaultNotInChoice;
                                    }
                                }
                            }
                            else if (isInteger)
                            {
                                // default を整数型としてパース出来るか
                                try
                                {
                                    G3dDataParser.ParseIntArray(option.@default);
                                }
                                catch
                                {
                                    return OptionVarErrorType.DefaultCannotParse;
                                }
                            }
                            else
                            {
                                // default の構成文字
                                foreach (var value in defaults)
                                {
                                    var match = stringChoiceItemRegex.Match(value);
                                    if (value.Length == 0 || !match.Success || match.Length != value.Length)
                                    {
                                        return OptionVarErrorType.DefaultCannotParse;
                                    }
                                }
                            }
                        }
                    }
                    break;
                case OptionType.Int:
                    {
                        bool hasMinMax = false;
                        int min = 0;
                        int max = 0;

                        int[] choices = null;
                        // choice のチェック
                        if (!string.IsNullOrEmpty(option.choice))
                        {
                            // パース出来るか
                            if (IfShaderAssignUtility.TryParseIntRange(option.choice, out min, out max))
                            {
                                // 範囲が正しいか
                                if (min > max)
                                {
                                    return OptionVarErrorType.InvalidRange;
                                }

                                hasMinMax = true;
                            }
                            else
                            {
                                string[] stringChoices = null;
                                var error = ValidateChoiceList(option, x => { int o; return int.TryParse(x, out o); }, out stringChoices);
                                if (error != OptionVarErrorType.OK)
                                {
                                    return error;
                                }
                                choices = stringChoices.Select(x => { int o; int.TryParse(x, out o); return o; }).ToArray();
                            }
                        }

                        // default のチェック
                        if (!string.IsNullOrEmpty(option.@default))
                        {
                            int[] values = null;

                            // パース出来るか
                            try
                            {
                                values = G3dDataParser.ParseIntArray(option.@default);
                            }
                            catch
                            {
                                return OptionVarErrorType.DefaultCannotParse;
                            }

                            if (values != null)
                            {
                                // default が choice に含まれるか
                                if (hasMinMax)
                                {
                                    foreach (var value in values)
                                    {
                                        if (value < min || max < value)
                                        {
                                            return OptionVarErrorType.DefaultNotInChoice;
                                        }
                                    }
                                }
                                else if (choices != null)
                                {
                                    foreach (var value in values)
                                    {
                                        if (!choices.Contains(value))
                                        {
                                            return OptionVarErrorType.DefaultNotInChoice;
                                        }
                                    }
                                }
                            }
                        }
                    }
                    break;
                default:
                    throw new NotImplementedException("Unimplemented OptionType");
            }

            return OptionVarErrorType.OK;
        }

        /// <summary>
        /// option_varType のエラーの種類
        /// </summary>
        public enum OptionVarErrorType
        {
            /// <summary>
            /// 問題なし
            /// </summary>
            OK,

            /// <summary>
            /// 数値型の choice がパース出来ない
            /// 文字列型の choice が不正な文字を使用している
            /// </summary>
            ChoiceCannotParse,

            /// <summary>
            /// min > max
            /// </summary>
            InvalidRange,

            /// <summary>
            /// string 型で choice が重複
            /// </summary>
            DuplicateChoice,

            /// <summary>
            /// string 型で alias が重複
            /// </summary>
            DuplicateAlias,

            /// <summary>
            /// 数値型の default がパース出来ない
            /// 文字列型の default が不正な文字を使用している
            /// </summary>
            DefaultCannotParse,

            /// <summary>
            /// default が　choice に含まれない
            /// </summary>
            DefaultNotInChoice,
        }

        private static Regex stringChoiceItemRegex = new Regex("[a-zA-Z0-9_]+");

        // TODO: 空白の考慮
        // [^:,]+ は選択肢
        // (:[^,]+)? はエイリアス
        // (,[^:,]+(:[^,]+)?)* は2つ目以降の項目
        private static Regex stringChoicePunctuationRegex = new Regex("^([^:,]+(:[^,]+)?(,[^:,]+(:[^,]+)?)*)$");

        private static OptionVarErrorType ValidateChoiceList(option_varType option, Predicate<string> predicate, out string[] choices)
        {
            choices = null;

            // choice の区切り文字チェック
            {
                var match = stringChoicePunctuationRegex.Match(option.choice);
                if (!match.Success || match.Length != option.choice.Length)
                {
                    return OptionVarErrorType.ChoiceCannotParse;
                }
            }

            choices = IfShaderOptionUtility.Enum(option.choice);

            // choice の構成文字
            foreach (var choice in choices)
            {
                if (predicate != null)
                {
                    if (!predicate(choice))
                    {
                        return OptionVarErrorType.ChoiceCannotParse;
                    }
                }
            }

            // choice の重複
            if (choices.Distinct().Count() != choices.Length)
            {
                return OptionVarErrorType.DuplicateChoice;
            }

            // alias の重複
            var alias = IfShaderOptionUtility.Alias(option.choice);
            if (choices.Select(x => alias.ContainsKey(x) ? alias[x] : x).Distinct().Count() != choices.Length)
            {
                return OptionVarErrorType.DuplicateAlias;
            }

            return OptionVarErrorType.OK;
        }
    }
}
