﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // シェイプアニメーションの量子化分析
    public class IfShapeAnimQuantizationAnalysisOptimizer
        : IfShapeAnimOptimizer
    {
        // コンストラクタ
        public IfShapeAnimQuantizationAnalysisOptimizer() :
            base("IfShapeAnimQuantizationAnalysisOptimizer_Log") { }

        // プロセス
        public override string Process
        {
            get { return "quantization_analysis"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Empty;
        }

         private int FrameSize;
         private int QuantizedFrameSize;

        // 最適化
        protected override void Optimize()
        {
            shape_anim_infoType shapeAnimInfo = this.Target.shape_anim_info;

            // シェイプアニメーションの量子化解析
            if (this.Target.vertex_shape_anim_array != null &&
                this.Target.vertex_shape_anim_array.vertex_shape_anim != null)
            {
                vertex_shape_anim_arrayType shapeAnimArray = this.Target.vertex_shape_anim_array;

                foreach (vertex_shape_animType shapeAnim in shapeAnimArray.vertex_shape_anim)
                {
                    foreach (shape_anim_targetType target in shapeAnim.shape_anim_target)
                    {
                        if (target.Item != null)
                        {
                            // カーブデータを構築する
                            IfAnimCurve animCurve = new IfAnimCurve();
                            animCurve.curveType = IfAnimCurve.CurveType.Float;
                            animCurve.BuildIfAnimCurve(target.Item, false, this.Streams);

                            // カーブの量子化を評価して中間ファイルに適用する
                            IfQuantizationAnalysisResult result =
                                IfShapeAnimQuantizationAnalysis.Analyse(
                                    shapeAnimInfo,
                                    animCurve,
                                    false);

                            this.EnableProcessLog |= !result.skipped;

                            ApplyQuantize(target, result);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// アニメーションに量子化タイプを適応します。
        /// </summary>
        private void ApplyQuantize(dynamic target, IfQuantizationAnalysisResult analysisResult)
        {
            curve_frame_typeType frameType = curve_frame_typeType.none;
            curve_key_typeType keyType = curve_key_typeType.none;
            int frameCount = 0;

            if (target.Item != null)
            {
                IG3dQuantizedCurve curve = target.Item as IG3dQuantizedCurve;
                curve.frame_type = analysisResult.frameType;
                curve.key_type = analysisResult.keyType;
                curve.scale = analysisResult.scale;
                curve.offset = analysisResult.offset;

                frameType = curve.frame_type;
                keyType = curve.key_type;
                frameCount = curve.count;
            }

            this.FrameSize += frameCount * sizeof(float);
            switch (frameType)
            {
                case curve_frame_typeType.none:
                case curve_frame_typeType.frame32:
                    this.QuantizedFrameSize += frameCount * sizeof(float);
                    break;
                case curve_frame_typeType.frame16:
                    this.QuantizedFrameSize += frameCount * sizeof(ushort);
                    break;
                case curve_frame_typeType.frame8:
                    this.QuantizedFrameSize += frameCount * sizeof(byte);
                    break;
            }
        }
    }
}
