﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // シーンアニメーションの量子化分析
    public class IfSceneAnimQuantizationAnalysisOptimizer
        : IfSceneAnimOptimizer
    {
        // コンストラクタ
        public IfSceneAnimQuantizationAnalysisOptimizer() :
            base("IfSceneAnimQuantizationAnalysisOptimizer_Log") { }

        // プロセス
        public override string Process
        {
            get { return "quantization_analysis"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Empty;
        }

         private int FrameSize;
         private int QuantizedFrameSize;

        // 最適化
        protected override void Optimize()
        {
            scene_anim_infoType sceneAnimInfo = this.Target.scene_anim_info;

            // カメラアニメーションの量子化解析
            if (this.Target.camera_anim_array != null &&
                this.Target.camera_anim_array.camera_anim != null)
            {
                camera_anim_arrayType cameraAnimArray = this.Target.camera_anim_array;

                foreach (camera_animType cameraAnim in cameraAnimArray.camera_anim)
                {
                    foreach (camera_anim_targetType target in cameraAnim.camera_anim_target ?? Enumerable.Empty<camera_anim_targetType>())
                    {
                        if (target.Item != null)
                        {
                            // カーブデータを構築する
                            IfAnimCurve animCurve = new IfAnimCurve();
                            animCurve.curveType = IfAnimCurve.CurveType.Float;

                            bool toRadian =
                                target.target == camera_anim_target_targetType.rotate_x ||
                                target.target == camera_anim_target_targetType.rotate_y ||
                                target.target == camera_anim_target_targetType.rotate_z;

                            animCurve.BuildIfAnimCurve(target.Item, toRadian, this.Streams);

                            // カーブの量子化を評価して中間ファイルに適用する
                            IfQuantizationAnalysisResult result =
                                IfSceneAnimQuantizationAnalysis.Analyse(
                                    sceneAnimInfo,
                                    target.target,
                                    animCurve,
                                    false);

                            this.EnableProcessLog |= !result.skipped;

                            ApplyQuantize(target, result);
                        }
                    }
                }
            }

            // ライトアニメーションの量子化解析
            if (this.Target.light_anim_array != null &&
                this.Target.light_anim_array.light_anim != null)
            {
                light_anim_arrayType lightAnimArray = this.Target.light_anim_array;

                foreach (light_animType lightAnim in lightAnimArray.light_anim)
                {
                    foreach (light_anim_targetType target in lightAnim.light_anim_target ?? Enumerable.Empty<light_anim_targetType>())
                    {
                        if (target.Item != null)
                        {
                            // カーブデータを構築する
                            IfAnimCurve animCurve = new IfAnimCurve();
                            animCurve.curveType = IfAnimCurve.CurveType.Float;
                            animCurve.BuildIfAnimCurve(target.Item, false, this.Streams);

                            // カーブの量子化を評価して中間ファイルに適用する
                            IfQuantizationAnalysisResult result =
                                IfSceneAnimQuantizationAnalysis.Analyse(
                                    sceneAnimInfo,
                                    target.target,
                                    animCurve,
                                    false);

                            this.EnableProcessLog |= !result.skipped;

                            ApplyQuantize(target, result);
                        }
                    }
                }
            }

            // フォグアニメーションの量子化解析
            if (this.Target.fog_anim_array != null &&
                this.Target.fog_anim_array.fog_anim != null)
            {
                fog_anim_arrayType fogAnimArray = this.Target.fog_anim_array;

                foreach (fog_animType fogAnim in fogAnimArray.fog_anim)
                {
                    foreach (fog_anim_targetType target in fogAnim.fog_anim_target ?? Enumerable.Empty<fog_anim_targetType>())
                    {
                        if (target.Item != null)
                        {
                            // カーブデータを構築する
                            IfAnimCurve animCurve = new IfAnimCurve();
                            animCurve.curveType = IfAnimCurve.CurveType.Float;
                            animCurve.BuildIfAnimCurve(target.Item, false, this.Streams);

                            // カーブの量子化を評価して中間ファイルに適用する
                            IfQuantizationAnalysisResult result =
                                IfSceneAnimQuantizationAnalysis.Analyse(
                                    sceneAnimInfo,
                                    target.target,
                                    animCurve,
                                    false);

                            this.EnableProcessLog |= !result.skipped;

                            ApplyQuantize(target, result);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// アニメーションに量子化タイプを適応します。
        /// </summary>
        private void ApplyQuantize(dynamic target, IfQuantizationAnalysisResult analysisResult)
        {
            curve_frame_typeType frameType = curve_frame_typeType.none;
            curve_key_typeType keyType = curve_key_typeType.none;
            int frameCount = 0;

            if (target.Item != null)
            {
                IG3dQuantizedCurve curve = target.Item as IG3dQuantizedCurve;
                curve.frame_type = analysisResult.frameType;
                curve.key_type = analysisResult.keyType;
                curve.scale = analysisResult.scale;
                curve.offset = analysisResult.offset;

                frameType = curve.frame_type;
                keyType = curve.key_type;
                frameCount = curve.count;
            }

            this.FrameSize += frameCount * sizeof(float);
            switch (frameType)
            {
                case curve_frame_typeType.none:
                case curve_frame_typeType.frame32:
                    this.QuantizedFrameSize += frameCount * sizeof(float);
                    break;
                case curve_frame_typeType.frame16:
                    this.QuantizedFrameSize += frameCount * sizeof(ushort);
                    break;
                case curve_frame_typeType.frame8:
                    this.QuantizedFrameSize += frameCount * sizeof(byte);
                    break;
            }
        }
    }
}
