﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;

namespace nw.g3d.iflib
{
    /// <summary>
    /// 頂点シェーダーの動作を真似て、その結果を取得するためのクラスです。
    /// </summary>
    public class IfPseudoVertexShader
    {
        private const int LatestTriangleMax = 3;

        /// <summary>
        /// 頂点キャッシュクラスを渡して初期化するコンストラクタです。
        /// </summary>
        /// <param name="pseudoVertexCache">擬似頂点キャッシュです。</param>
        public IfPseudoVertexShader(IfPseudoVertexCache pseudoVertexCache)
        {
            this.PseudoVertexCache = pseudoVertexCache;
        }

        /// <summary>
        /// 頂点キャッシュクラスの取得・設定を行います。
        /// </summary>
        public IfPseudoVertexCache PseudoVertexCache { get; set; }

        /// <summary>
        /// 描画を行った頂点数を取得します。
        /// </summary>
        public int TotalVertexCount { get; private set; }

        /// <summary>
        /// 処理した三角形の数を取得します。
        /// </summary>
        public int TotalTriangleCount { get; private set; }

        /// <summary>
        /// キャッシュミスが発生した回数を取得します。
        /// </summary>
        public int CacheMissCount { get; private set; }

        /// <summary>
        /// 全変換処理に対するキャッシュミスの割合です。
        /// </summary>
        public float CacheMissRatio
        {
            get
            {
                float cacheMissRatio = 0.0f;

                if (0 < this.TotalVertexCount)
                {
                    cacheMissRatio = (float)this.CacheMissCount / (float)this.TotalVertexCount;
                }

                return cacheMissRatio;
            }
        }

        /// <summary>
        /// 三角形を描くのに必要な頂点数の平均です。
        /// </summary>
        public float AverageCacheMissRatio
        {
            get
            {
                float averageCacheMissRatio = 0.0f;

                if (0 < this.TotalTriangleCount)
                {
                    averageCacheMissRatio = (float)this.CacheMissCount / (float)this.TotalTriangleCount;
                }

                return averageCacheMissRatio;
            }
        }

#if false
        /// <summary>
        /// 直前のトライアングルに１頂点分キャッシュヒットした回数
        /// </summary>
        public int LatestTriangle1HitCount { get; private set; }

        /// <summary>
        /// 直前のトライアングルに２頂点分キャッシュヒットした回数
        /// </summary>
        public int LatestTriangle2HitCount { get; private set; }

        /// <summary>
        /// 最近のトライアングルにキャッシュヒットできた回数
        /// </summary>
        public int LatestTriangleHitCount { get; private set; }
#endif

        /// <summary>
        /// トライアングルリストにおける擬似シェーディング動作を行い、その動作情報を更新します。
        /// </summary>
        /// <param name="triangles">動作を行うためのトライアングルリストです。</param>
        public void TransformVertices(IList<IfTriangle> triangles)
        {
#if true
            foreach (IfTriangle tri in triangles)
            {
                foreach (int posId in tri.PositionIds)
                {
                    this.TransformVertex(posId);
                    ++TotalVertexCount;
                }

                ++TotalTriangleCount;
            }
#else
            Triangle latestTriangle = null;
            Queue<Triangle> latestTriangles = new Queue<Triangle>();
            foreach (Triangle t in triangles)
            {
                // 一つ前のトライアングルとの非共有点のリストを取得する
                // 非共有点のみ変換でのキャッシュヒット判定を行う。
                List<int> unsharedVertices = t.GetUnsharedVertex(latestTriangle);
                foreach (int pid in unsharedVertices)
                {
                    this.TransformVertex(pid);
                }

                // トライアングルのキャッシュヒットの種類を判定
                int sharedVertexCount = Triangle.VertexCount - unsharedVertices.Count;
                if (2 <= sharedVertexCount)
                {
                    ++this.LatestTriangle2HitCount;
                }
                else if (sharedVertexCount == 1)
                {
                    ++this.LatestTriangle1HitCount;
                }
                else
                {
                    foreach (Triangle lt in latestTriangles)
                    {
                        if (t.ContainsSharedVertex(lt))
                        {
                            ++this.LatestTriangleHitCount;
                            break;
                        }
                    }
                }

                latestTriangle = t;
                latestTriangles.Enqueue(t);
                if (LatestTriangleMax < latestTriangles.Count)
                {
                    latestTriangles.Dequeue();
                }
            }
#endif
        }

        /// <summary>
        /// 頂点の変換処理を行った場合において、頂点キャッシュの動作だけ行います。
        /// </summary>
        /// <param name="positionId">変換処理を行う座標ＩＤです。</param>
        /// <returns>キャッシュにヒットした場合は true が返ります。</returns>
        private bool TransformVertex(int positionId)
        {
            bool isHitCache = false;

            if (this.PseudoVertexCache.IsContains(positionId))
            {
                isHitCache = true;
            }
            else
            {
                ++this.CacheMissCount;
                this.PseudoVertexCache.EnqueuePosition(positionId);
            }

            return isHitCache;
        }

        /// <summary>
        /// 頂点キャッシュと計算結果の情報を初期状態に戻します。
        /// </summary>
        public void Clear()
        {
            this.PseudoVertexCache.Clear();
            this.ClearResult();
        }

        /// <summary>
        /// 計算結果の情報を初期状態に戻します。
        /// </summary>
        private void ClearResult()
        {
            this.TotalVertexCount = 0;
            this.TotalTriangleCount = 0;
            this.CacheMissCount = 0;
#if false
            this.LatestTriangle1HitCount = 0;
            this.LatestTriangle2HitCount = 0;
            this.LatestTriangleHitCount = 0;
#endif
        }
    }
}
