﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;

namespace nw.g3d.iflib
{
    /// <summary>
    /// 描画する単位でのポリゴンの集合を管理するクラスです。
    /// </summary>
    public class IfPolygonPrimitive
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public IfPolygonPrimitive()
        {
            this.Triangles = new List<IfTriangle>();
        }

        /// <summary>
        /// トライアングルのリストです。
        /// </summary>
        public IList<IfTriangle> Triangles { get; private set; }
    }

    public class IfPolygonMesh
    {
        private const int VertexNumberMax = short.MaxValue;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public IfPolygonMesh()
        {
            this.PolygonPrimitives = new List<IfPolygonPrimitive>();
            this.AnalyzedTriangles = new List<IfAnalyzedTriangle>();
            this.Occurrences = null;
            this.Adjacencys = null;
            this.MaxAdjacency = 0;
        }

        /// <summary>
        /// 頂点リストの頂点数 (頂点インデックスの最大値 + 1)
        /// </summary>
        public int VerticesNumber { get; private set; }

        /// <summary>
        /// 最適化結果を外部のデータに復元するために必要な情報です。
        /// </summary>
        public object Data { get; set; }

        /// <summary>
        /// 描画単位のポリゴンの集合です。
        /// </summary>
        public IList<IfPolygonPrimitive> PolygonPrimitives { get; private set; }

        /// <summary>
        /// 解析情報を含むトライアングルリストです。
        /// </summary>
        public IList<IfAnalyzedTriangle> AnalyzedTriangles { get; private set; }

        /// <summary>
        /// 頂点ごとの所属するトライアングルの数のリスト
        /// </summary>
        public IList<int> Occurrences { get; private set; }

        /// <summary>
        /// 頂点ごとの所属するトライアングルのリストのリスト
        /// </summary>
        public int[][] Adjacencys { get; private set; }

        /// <summary>
        /// 頂点が所属するトライアングルの数の最大数
        /// </summary>
        public int MaxAdjacency { get; private set; }

        /// <summary>
        /// 隣接するトライアングルが最大の頂点か三角形を表すリスト
        /// </summary>
        public IList<int> MaxAdjacencyList { get; private set; }

        /// <summary>
        /// 解析済みトライアングルの使用済みフラグをクリアします。
        /// </summary>
        public void ClearAnalizedTrianglesUsedFlag()
        {
            foreach (IfAnalyzedTriangle analyzedTriangle in this.AnalyzedTriangles)
            {
                analyzedTriangle.IsUsed = false;
            }
        }

        /// <summary>
        /// 頂点を解析します。
        /// </summary>
        public void AnalyzeVertices()
        {
            this.VerticesNumber = 0;
            foreach (IfAnalyzedTriangle analyzedTriangle in this.AnalyzedTriangles)
            {
                foreach (int p in analyzedTriangle.Triangle.PositionIds)
                {
                    if (this.VerticesNumber < p)
                    {
                        this.VerticesNumber = p;
                    }
                }
            }

            this.VerticesNumber += 1;

            // 頂点の利用回数をカウント
            this.Occurrences = new List<int>(this.VerticesNumber);

            for (int i = 0; i < this.VerticesNumber; ++i)
            {
                this.Occurrences.Add(0);
            }

            foreach (IfAnalyzedTriangle analyzedTriangle in this.AnalyzedTriangles)
            {
                for (int i = 0; i < IfTriangle.VertexCount; ++i)
                {
                    int v = analyzedTriangle.Triangle.PositionIds[i];

                    if (this.Occurrences[v] == VertexNumberMax)
                    {
                        throw new Exception("Unsupported mesh : vertex shared by too meny triangles.");
                    }

                    ++this.Occurrences[v];
                }
            }

            // 頂点ごとに所有されているトライアングルをリスト化
            this.MaxAdjacency = 0;
            this.Adjacencys = new int[this.VerticesNumber][];
            for (int i = 0; i < this.VerticesNumber; ++i)
            {
                this.Adjacencys[i] = new int[this.Occurrences[i]];

                if (this.MaxAdjacency < this.Occurrences[i])
                {
                    this.MaxAdjacency = this.Occurrences[i];
                }
            }

            this.MaxAdjacencyList = new List<int>();
            for (int i = 0; i < this.VerticesNumber; ++i)
            {
                if (this.MaxAdjacency == this.Occurrences[i])
                {
                    this.MaxAdjacencyList.Add(i);
                }
                this.Occurrences[i] = 0;
            }

            for (int i = 0; i < this.AnalyzedTriangles.Count; ++i)
            {
                IfTriangle triangle = this.AnalyzedTriangles[i].Triangle;
                for (int j = 0; j < IfTriangle.VertexCount; ++j)
                {
                    this.Adjacencys[triangle.PositionIds[j]][this.Occurrences[triangle.PositionIds[j]]] = i;
                    ++this.Occurrences[triangle.PositionIds[j]];
                }
            }
        }

        /// <summary>
        /// トライアングルを解析します。
        /// </summary>
        public void AnalyzeTriangles()
        {
            this.AnalyzeVertices();

            foreach (IfAnalyzedTriangle at in this.AnalyzedTriangles)
            {
                this.FindAdjacentTriangles(at, at.SharedVertex1Triangles, at.SharedVertex2Triangles);
            }
        }

        /// <summary>
        /// 隣り合っているトライアングルの数を取得します。
        /// </summary>
        /// <param name="triangle">元になるトライアングルです。</param>
        /// <returns>triangle と隣接するトライアングルの数を返します。</returns>
        public int GetAdjacentTriangleCount(IfAnalyzedTriangle triangle)
        {
            var foundTriangles = new List<int>();

            foreach (int p in triangle.Triangle.PositionIds)
            {
                foreach (int t in this.Adjacencys[p])
                {
                    if (!this.AnalyzedTriangles[t].IsUsed &&
                        triangle != this.AnalyzedTriangles[t] &&
                        !foundTriangles.Contains(t))
                    {
                        foundTriangles.Add(t);
                    }
                }
            }

            return foundTriangles.Count;
        }

        /// <summary>
        /// 隣り合うトライアングルを取得します。
        /// </summary>
        /// <param name="triangle">元になるトライアングル</param>
        /// <param name="foundTriangles1Hit">共有頂点が１つのトライアングル</param>
        /// <param name="foundTriangles2Hit">共有頂点が２つのトライアングル</param>
        public void FindAdjacentTriangles(
            IfAnalyzedTriangle triangle,
            IList<IfAnalyzedTriangle> foundTriangles1Hit,
            IList<IfAnalyzedTriangle> foundTriangles2Hit)
        {
            Dictionary<int, int> triangleRef = new Dictionary<int, int>();
            foreach (int p in triangle.Triangle.PositionIds)
            {
                foreach (int t in this.Adjacencys[p])
                {
                    if (!this.AnalyzedTriangles[t].IsUsed && this.AnalyzedTriangles[t] != triangle)
                    {
                        if (triangleRef.ContainsKey(t))
                        {
                            ++triangleRef[t];
                        }
                        else
                        {
                            triangleRef.Add(t, 1);
                        }
                    }
                }
            }

            foreach (KeyValuePair<int, int> t in triangleRef)
            {
                if (t.Value == 2)
                {
                    foundTriangles2Hit.Add(this.AnalyzedTriangles[t.Key]);
                }
                else if (t.Value == 1)
                {
                    foundTriangles1Hit.Add(this.AnalyzedTriangles[t.Key]);
                }
            }
        }
    }
}
