﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;

namespace nw.g3d.iflib
{
    public class SubmeshClusterSet
    {
        private readonly List<SubmeshCluster> submeshClusters = new List<SubmeshCluster>();
        private readonly IList<G3dStream> streams;
        private readonly modelType model;
        private readonly shapeType shape;
        private readonly int[] indexStream;
        private readonly submeshType submesh;
        private readonly Vector3 position = new Vector3(0.0f, 0.0f, 0.0f);
        private float area = 0.0f;
        private ReadOnlyCollection<int> source = null;

        public SubmeshClusterSet(
            modelType model,
            List<G3dStream> streams,
            shapeType shape,
            int[] indexStream, submeshType submesh,
            string mode)
        {
            this.model = model;
            this.streams = streams;
            this.shape = shape;
            this.indexStream = indexStream;
            this.submesh = submesh;
            this.source = new ReadOnlyCollection<int>(Array.AsReadOnly<int>(indexStream));
        }

        public ReadOnlyCollection<int> Source
        {
            get
            {
                return this.source;
            }
        }

        internal IList<SubmeshCluster> SubmeshClusters
        {
            get
            {
                return this.submeshClusters;
            }
        }

        public void MakeClustersWithCachePriority()
        {
            int triangleCount = this.indexStream.Length / 3;
            List<SubmeshCluster.Triangle> triangles = new List<SubmeshCluster.Triangle>();
            PseudoVertexCacheManager cacheManager = new PseudoVertexCacheManager();
            for (int i = 0; i < triangleCount; ++i)
            {
                int index = i * 3;
                int[] indices = new int[] { this.indexStream[index], this.indexStream[index + 1], this.indexStream[index + 2] };
                {
                    SubmeshCluster.Triangle triangle = new SubmeshCluster.Triangle(i, indices);
                    triangles.Add(triangle);
                }
                cacheManager.TransformVertices(indices.ToList());
                if (cacheManager.CacheMissRatio < 0.75f)
                {
                    SubmeshCluster cluster = new SubmeshCluster();
                    foreach (var triangle in triangles)
                    {
                        cluster.Add(triangle);
                    }
                    triangles.Clear();
                    this.submeshClusters.Add(cluster);
                    cacheManager.Clear();
                }
            }

            if (triangles.Count > 0)
            {
                SubmeshCluster cluster = new SubmeshCluster();
                foreach (var triangle in triangles)
                {
                    cluster.Add(triangle);
                }
                this.submeshClusters.Add(cluster);
            }
        }

        public void MakeClusters()
        {
            int triangleCount = this.indexStream.Length / 3;
            List<SubmeshCluster.Triangle> triangles = new List<SubmeshCluster.Triangle>();
            for (int i = 0; i < triangleCount; ++i)
            {
                int index = i * 3;
                int[] indices = new int[] { this.indexStream[index], this.indexStream[index + 1], this.indexStream[index + 2] };
                SubmeshCluster.Triangle triangle = new SubmeshCluster.Triangle(i, indices);
                triangles.Add(triangle);
            }

            SubmeshCluster cluster = new SubmeshCluster();
            foreach (var triangle in triangles)
            {
                cluster.Add(triangle);
                triangles.Remove(triangle);
                break;
            }
            if (triangles.Count == 0)
            {
                this.submeshClusters.Add(cluster);
            }
#if DEBUG
            int preCount = indexStream.Length;
            int preClusterCount = this.submeshClusters.Count;
#endif
            do
            {
                foreach (var triangle in triangles)
                {
                    if (!cluster.HasSharedEdge(triangle))
                    {
                        this.submeshClusters.Add(cluster);
                        cluster = new SubmeshCluster();
                    }
                    cluster.Add(triangle);
                    triangles.Remove(triangle);
                    if (triangles.Count == 0)
                    {
                        this.submeshClusters.Add(cluster);
                    }
                    break;
                }
            } while (triangles.Count > 0);
            this.MergeClusters();
#if DEBUG
            int postCount = this.Source.Count;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(preCount == postCount);

            int postClusterCount = this.submeshClusters.Count;
            if (preClusterCount > 0)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(postClusterCount > 0);
            }
#endif
        }

        public void Calc()
        {
            vertexType vertex = this.model.vertex_array.vertex[shape.shape_info.vertex_index];
            int attribIndex = GetVtxAttribIndex(vertex, "position");
            G3dStream vStream = streams[vertex.vtx_attrib_array.vtx_attrib[attribIndex].stream_index];

            foreach (var cluster in this.submeshClusters)
            {
                cluster.CalcAreaAndNormal(vStream);

                this.area += cluster.Area;
                this.position.X += cluster.Position.X;
                this.position.Y += cluster.Position.Y;
                this.position.Z += cluster.Position.Z;

                cluster.Setup();
            }
            this.position.X /= this.area * 3.0f;
            this.position.Y /= this.area * 3.0f;
            this.position.Z /= this.area * 3.0f;

            foreach (var cluster in this.submeshClusters)
            {
                cluster.CalcScore(this.position);
            }
            this.submeshClusters.Sort(CompareScore);

            List<int> indices = new List<int>();
#if false
            {
                List<SubmeshCluster> dstClusters1;
                List<SubmeshCluster> dstClusters2;
                float divideArea = this.area / 2.0f;
                DivideClusters(divideArea, this.submeshClusters, out dstClusters1, out dstClusters2);
                {
                    List<SubmeshCluster.Triangle> triangles = new List<SubmeshCluster.Triangle>();
                    foreach (var cluster in dstClusters1)
                    {
                        triangles.AddRange(cluster.Triangles);
                    }
                    triangles.Sort(SortTriangleLowIndex);
                    foreach (var triangle in triangles)
                    {
                        indices.AddRange(triangle.VertexIndices);
                    }
                }

                {
                    List<SubmeshCluster.Triangle> triangles = new List<SubmeshCluster.Triangle>();
                    foreach (var cluster in dstClusters2)
                    {
                        triangles.AddRange(cluster.Triangles);
                    }
                    triangles.Sort(SortTriangleLowIndex);
                    foreach (var triangle in triangles)
                    {
                        indices.AddRange(triangle.VertexIndices);
                    }
                }
            }
#else
            foreach (var cluster in this.submeshClusters)
            {
                indices.AddRange(cluster.GetVertexIndices());
            }
#endif
            float cacheMissRatio = CalcAverageCacheMissRatio(indices);
#if DEBUG
            Nintendo.Foundation.Contracts.Assertion.Operation.True(indices.Count > 0);
#endif
            this.source = indices.AsReadOnly();
        }

        private void MergeClusters()
        {
            if (this.submeshClusters.Count < 2)
            {
                return;
            }

            List<SubmeshCluster> mergedClusters = new List<SubmeshCluster>();
            do
            {
                foreach (var cluster1 in this.submeshClusters)
                {
                    List<SubmeshCluster> removeTargets = new List<SubmeshCluster>();
                    for (int j = 1; j < this.submeshClusters.Count; ++j)
                    {
                        SubmeshCluster cluster2 = this.submeshClusters[j];
                        if (cluster1.HasSharedEdge(cluster2))
                        {
                            removeTargets.Add(cluster2);
                            cluster1.Merge(cluster2);
                            break;
                        }
                    }

                    foreach (var removeTarget in removeTargets)
                    {
                        this.submeshClusters.Remove(removeTarget);
                    }

                    mergedClusters.Add(cluster1);
                    this.submeshClusters.Remove(cluster1);
                    break;
                }
            } while (this.submeshClusters.Count > 0);

            this.submeshClusters.AddRange(mergedClusters);
        }

        private static void DivideClusters(float divideArea, List<SubmeshCluster> srcClusters, out List<SubmeshCluster> dstClusters1, out List<SubmeshCluster> dstClusters2)
        {
            dstClusters1 = new List<SubmeshCluster>();
            dstClusters2 = new List<SubmeshCluster>();
            int clusterIndex = 0;
            int clusterCount = srcClusters.Count;
            float sumArea = 0.0f;
            for (; clusterIndex < clusterCount; ++clusterIndex)
            {
                var cluster = srcClusters[clusterIndex];
                sumArea += cluster.Area;
                dstClusters1.Add(cluster);
                if (sumArea >= divideArea)
                {
                    break;
                }
            }

            ++clusterIndex;
            for (; clusterIndex < clusterCount; ++clusterIndex)
            {
                var cluster = srcClusters[clusterIndex];
                dstClusters2.Add(cluster);
            }
        }

        private static int GetVtxAttribIndex(vertexType vertex, string attribName)
        {
            int vtxAttribIndex = 0;
            foreach (vtx_attribType vtxAttrib in vertex.vtx_attrib_array.Enumerate())
            {
                // hint を用いて attribName の Stream を検索する。
                if (vtxAttrib.hint.StartsWith(attribName))
                {
                    return vtxAttribIndex;
                }
                ++vtxAttribIndex;
            }
            return -1;
        }

        private static int CompareScore(SubmeshCluster x, SubmeshCluster y)
        {
            float result = x.Score - y.Score;

            if (result < 0f)
            {
                return -1;
            }
            else if (result > 0f)
            {
                return 1;
            }
            return 0;
        }

        private static int SortTriangleLowIndex(SubmeshCluster.Triangle x, SubmeshCluster.Triangle y)
        {
            return x.Index - y.Index;
        }

        private static float CalcAverageCacheMissRatio(IList<int> indexStream)
        {
            if (indexStream == null || indexStream.Count == 0)
            {
                return 0.0f;
            }

            Queue<int> vertexCache = new Queue<int>();
            int cacheMissCount = 0;
            foreach (int index in indexStream)
            {
                if (vertexCache.Contains(index)) { continue; }
                vertexCache.Enqueue(index);
                if (vertexCache.Count > G3dConstant.VertexCacheSize)
                {
                    vertexCache.Dequeue();
                    ++cacheMissCount;
                }
            }
            return (float)(cacheMissCount) / (float)(indexStream.Count);
        }
    }
}
