﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;

namespace nw.g3d.iflib
{
    internal class IfShapePrimitiveOptimizer
    {
        // モード
        private enum PrimitiveOptimizeMode
        {
            @default,
            force,
            brute_force,
            forsyth
        }

        // コンストラクタ
        internal IfShapePrimitiveOptimizer(
            modelType model, List<G3dStream> streams, shapeType shape,
            string mode, bool useSortSubmeshIndex)
        {
            this.model = model;
            this.Streams = streams;
            this.shape = shape;
            this.Mode = mode;

            var lod_offset = this.model.vertex_array.vertex[this.shape.shape_info.vertex_index].lod_offset;
            if (lod_offset != null && lod_offset.count != this.shape.mesh_array.mesh.Length - 1)
            {
                IfStrings.Throw("IfShapePrimitiveOptimizer_Error_InvalidLodOffset", this.shape.name);
            }

            foreach (var mesh in this.shape.mesh_array.Enumerate())
            {
                Optimizers.Add(new IfMeshPrimitiveOptimizer(
                    this.model, this.Streams, this.shape, mesh, this.Mode, useSortSubmeshIndex));
            }
        }

        //---------------------------------------------------------------------
        // 最適化
        internal void Optimize()
        {
            // スキップする場合も初期化は必要
            foreach (var optimizer in this.Optimizers)
            {
                optimizer.Initialize();
            }

            // プリミティブ最適化のオプション引数が不正な文字の場合に、default になるように文字列を修正する。
            PrimitiveOptimizeMode newMode;
            if (!Enum.TryParse(this.Mode, false, out newMode))
            {
                newMode = PrimitiveOptimizeMode.@default;
            }

            // 既に同等以上の最適化が施されていれば、スキップする
            if (!string.IsNullOrEmpty(shape.shape_info.optimize_primitive_mode))
            {
                PrimitiveOptimizeMode alreadyMode;
                if (!Enum.TryParse(shape.shape_info.optimize_primitive_mode, false, out alreadyMode))
                {
                    throw new System.IO.InvalidDataException();
                }
                // brute_forceまではより上位のアルゴリズムを選択しない限り、処理を行わない。
                if (alreadyMode >= newMode && alreadyMode <= PrimitiveOptimizeMode.brute_force && newMode <= PrimitiveOptimizeMode.brute_force)
                {
                    return;
                }
                if (alreadyMode == newMode)
                {
                    return;
                }
            }

            // DCC から出力した状態のハッシュを求める
            string hash = shape.shape_info.original_shape_hash;
            if (hash == string.Empty)
            {
                hash = IfOptimizePrimitiveHashCalculator.Calculate(
                    this.model, this.Streams, this.shape);
            }

            foreach (var optimizer in this.Optimizers)
            {
                optimizer.Optimize();
            }

            // インデックスの出現順にあわせた頂点列の並べ替え
            if (this.IsSuccess()) { AlignShape(); }

            // 再最適化を防ぐため、最適化の結果に関わらず、mode と hash は更新する
            shape.shape_info.optimize_primitive_mode = newMode.ToString();
            shape.shape_info.original_shape_hash = hash;
        }

        //---------------------------------------------------------------------
        // シェイプの整列
        private void AlignShape()
        {
            if (this.model.vertex_array == null ||
                this.model.vertex_array.vertex == null) { return; }

            // 初期化
            vertexType vertex =
                this.model.vertex_array.vertex[this.shape.shape_info.vertex_index];
            vtx_attribType posAttr = Array.Find(vertex.vtx_attrib_array.vtx_attrib,
                delegate (vtx_attribType vtx_attrib)
                {
                    return (vtx_attrib.name == "_p0");
                });
            var lod_offset = new List<int>() { 0 };
            if (vertex.lod_offset != null)
            {
                lod_offset.AddRange(G3dDataParser.ParseIntArray(vertex.lod_offset.Value));
            }
            lod_offset.Add(posAttr.count);

            for (int iMesh = 0; iMesh < this.shape.mesh_array.mesh.Length; ++iMesh)
            {
                int found = lod_offset.FindIndex(0, iMesh, p => p == lod_offset[iMesh]);
                if (found >= 0)
                {
                    // 頂点共有
                    this.Optimizers[iMesh].AlignSharedMesh(vertex, this.Optimizers[found].UsedVertices);
                }
                else
                {
                    this.Optimizers[iMesh].AlignMesh(vertex, lod_offset.Min(p => p > lod_offset[iMesh] ? p : posAttr.count) - lod_offset[iMesh]);
                }
            }

            // 整列結果の適用
            for (int iMesh = 0; iMesh < this.shape.mesh_array.mesh.Length; ++iMesh)
            {
                int found = lod_offset.FindIndex(0, iMesh, p => p == lod_offset[iMesh]);
                if (found < 0)
                {
                    ApplyAlignedVertexIndices(vertex, iMesh, lod_offset[iMesh]);
                }
                ApplyAlignedIndexIndices(iMesh);
            }
        }

        // 整列済み頂点インデックス配列の適用
        private void ApplyAlignedVertexIndices(vertexType vertex, int meshIndex, int lod_offset)
        {
            // アトリビュート単位で並列化する
            vtx_attribType[] vtx_attribs = vertex.vtx_attrib_array.vtx_attrib;
            G3dParallel.ForEach(vtx_attribs, delegate (vtx_attribType vtx_attrib)
            {
                G3dStream stream = this.Streams[vtx_attrib.stream_index];
                vtx_attrib_typeType type = vtx_attrib.type;
                int component = (int)type % 4 + 1;
                if (type >= vtx_attrib_typeType.@float)
                {
                    ApplyAlignedVertexIndices<float>(stream.FloatData, component,
                        this.Optimizers[meshIndex].AlignedVertexIndices, lod_offset);
                }
                else
                {
                    ApplyAlignedVertexIndices<int>(stream.IntData, component,
                        this.Optimizers[meshIndex].AlignedVertexIndices, lod_offset);
                }
            });
        }

        // 整列済み頂点インデックス配列の適用
        private void ApplyAlignedVertexIndices<TType>(
            List<TType> data, int component, int[] alignedVertexIndices, int lod_offset)
        {
            TType[] tempData = new TType[alignedVertexIndices.Length * component];
            int offset = lod_offset * component;
            int index = 0;
            for (int i = 0; i < alignedVertexIndices.Length; i++)
            {
                for (int j = 0; j < component; j++)
                {
                    tempData[index] = data[alignedVertexIndices[i] * component + j + offset];
                    index++;
                }
            }

            for (int i = 0; i < tempData.Length; i++) { data[i + offset] = tempData[i]; }
        }

        // 整列済みインデックスインデックス配列の適用
        private void ApplyAlignedIndexIndices(int meshIndex)
        {
            List<int> destination = this.Streams[shape.mesh_array.mesh[meshIndex].stream_index].IntData;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(destination.Count == this.Optimizers[meshIndex].AlignedIndexIndices.Length);

            for (int i = 0; i < destination.Count; i++)
            {
                destination[i] = this.Optimizers[meshIndex].AlignedIndexIndices[i];
            }
        }

        //---------------------------------------------------------------------
        // トライアングル数の取得
        internal int GetTriangleCount()
        {
            return Optimizers.Sum(
                delegate (IfMeshPrimitiveOptimizer meshOptimizer)
                { return meshOptimizer.GetTriangleCount(); });
        }

        // 元の処理頂点数の取得
        internal int GetSourceProcessingVertexCount()
        {
            return Optimizers.Sum(
                delegate (IfMeshPrimitiveOptimizer meshOptimizer)
                { return meshOptimizer.GetSourceProcessingVertexCount(); });
        }

        // 処理頂点数の取得
        internal int GetProcessingVertexCount()
        {
            return Optimizers.Sum(
                delegate (IfMeshPrimitiveOptimizer meshOptimizer)
                { return meshOptimizer.GetProcessingVertexCount(); });
        }

        // 成功したか
        internal bool IsSuccess()
        {
            foreach (IfMeshPrimitiveOptimizer meshOptimizer in this.Optimizers)
            {
                if (meshOptimizer.IsSuccess()) { return true; }
            }
            return false;
        }

        //---------------------------------------------------------------------
        private readonly modelType model;
        private readonly List<G3dStream> Streams;
        private readonly shapeType shape;
        private readonly string Mode;

        private readonly List<IfMeshPrimitiveOptimizer> Optimizers =
            new List<IfMeshPrimitiveOptimizer>();
    }
}
