﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using System.IO;
using System.Linq;

namespace nw.g3d.iflib
{
    // LOD モデル結合
    public class IfModelUniteLodModelOptimizer : IfModelOptimizer
    {
        private string xsdBasePath = null;
        public const int MaxLodLevel = 9;

        public class LodShapeInfo
        {
            public HashSet<int> BaseBoneIndex
            {
                get
                {
                    return baseBoneIndex;
                }
            }
            public List<string> BaseBoneName
            {
                get
                {
                    return baseBoneName;
                }
            }

            public HashSet<int> LodBoneIndex
            {
                get
                {
                    return lodBoneIndex;
                }
            }
            public List<string> LodBoneName
            {
                get
                {
                    return lodBoneName;
                }
            }

            private HashSet<int> baseBoneIndex = new HashSet<int>();
            private List<string> baseBoneName = new List<string>();
            private HashSet<int> lodBoneIndex = new HashSet<int>();
            private List<string> lodBoneName = new List<string>();
        }

        // matrix_index から bone_index への逆引きテーブル
        public class LodIndexTable
        {
            public Dictionary<int, int> baseSmoothPaletteToIndex { get; set; } = new Dictionary<int, int>();
            public Dictionary<int, int> baseRigidPaletteToIndex { get; set; } = new Dictionary<int, int>();
            public Dictionary<int, int> lodSmoothPaletteToIndex { get; set; } = new Dictionary<int, int>();
            public Dictionary<int, int> lodRigidPaletteToIndex { get; set; } = new Dictionary<int, int>();
        }

        // 結合させる LOD モデルの配列
        private IDictionary<int, string> LodModelFiles;

        // コンストラクタ
        public IfModelUniteLodModelOptimizer(IDictionary<int, string> lodModelFiles, string argument, string xsdBasePath) :
            base("IfModelUniteLodModelOptimizer_Log", argument)
        {
            this.LodModelFiles = lodModelFiles;
            this.xsdBasePath = xsdBasePath;
        }

        public override string ToString()
        {
            return $"{base.ToString()} {this.GetResult()}";
        }

        // プロセス
        public override string Process
        {
            get { return "unite_lod_model"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Empty;
        }

        private enum SkinningKind
        {
            RigidBody = 0,
            RigidSkinning = 1,
            SmoothSkinning = 2
        }

        public class ErrorBox
        {
            public int index { get; set; }
            public string baseName { get; set; }
            public string lodName { get; set; }
        }

        public class ErrorBoxs
        {
            public List<ErrorBox> errorBoxs { get; set; } = new List<ErrorBox>();
            public string IndexString
            {
                get;
                private set;
            }
            public string BaseNameString
            {
                get;
                private set;
            }
            public string LodNameString
            {
                get;
                private set;
            }

            public int Count
            {
                get { return errorBoxs.Count; }
            }

            public void Add(ErrorBox errorBox)
            {
                errorBoxs.Add(errorBox);
            }

            public void MakeUp()
            {
                if (errorBoxs.Count != 0)
                {
                    StringBuilder indices = new StringBuilder();
                    StringBuilder baseNames = new StringBuilder();
                    StringBuilder lodNames = new StringBuilder();

                    int errorIndex = 0;
                    foreach (ErrorBox errorBox in errorBoxs)
                    {
                        if (errorIndex != 0)
                        {
                            indices.Append(", ");
                            baseNames.Append(", ");
                            lodNames.Append(", ");
                        }
                        indices.Append(errorBox.index);
                        baseNames.Append(errorBox.baseName);
                        lodNames.Append(errorBox.lodName);
                        ++errorIndex;
                    }

                    IndexString = indices.ToString();
                    BaseNameString = baseNames.ToString();
                    LodNameString = lodNames.ToString();
                }
            }
        }

        private enum BoneError
        {
            RigidToSmooth,
            BaseInvModelMatrix,
            LodInvModelMatrix,
            NumBoneError
        }

        private struct LodModelSet
        {
            public int level { get; set; }
            public List<G3dStream> streams { get; set; }
            public modelType file { get; set; }
        }

        // 最適化
        protected override void Optimize()
        {
            // ベースモデル
            modelType baseFile = this.Target;
            List<G3dStream> baseStreams = Streams;

            CheckStreamLength(baseFile, Streams);

            // LOD モデルをレベル順に読み込む
            int lastSkipLevel = 1;
            int beforeLevel = 0;
            List<LodModelSet> lodModels = new List<LodModelSet>();
            string uniteLodMode;
            {
                StringBuilder sbUniteLodMode = new StringBuilder(64);
                sbUniteLodMode.Append(";ulm");
                foreach (var lodModelPath in this.LodModelFiles.OrderBy(p => p.Key))
                {
                    if (!File.Exists(lodModelPath.Value))
                    {
                        IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_OpenLodFile", lodModelPath);
                    }

                    LodModelSet lodModelSet = new LodModelSet();
                    lodModelSet.level = lodModelPath.Key;
                    lodModelSet.streams = new List<G3dStream>();
                    lodModelSet.file = (modelType)IfReadUtility.Read(lodModelSet.streams, lodModelPath.Value, this.xsdBasePath).Item;

                    lodModels.Add(lodModelSet);

                    if (beforeLevel + 1 != lodModelPath.Key)
                    {
                        lastSkipLevel = lodModelPath.Key;
                    }
                    beforeLevel = lodModelPath.Key;

                    sbUniteLodMode.Append(" lm").Append(lodModelPath.Key);
                }
                uniteLodMode = sbUniteLodMode.ToString();
            }

            // レベル抜けがないかチェック
            foreach (var shape in baseFile.shape_array.shape)
            {
                if (shape.mesh_array.mesh.Length < lastSkipLevel)
                {
                    IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_SkippedLodLevel", lastSkipLevel);
                }
            }

            // まず AdjustLodModel 相当の処理をすべての LOD モデルに対して行う
            foreach (var lodModelSet in lodModels)
            {
                // TODO: エラー発生時にファイル名を表示する。
                LodIndexTable indextable = new LodIndexTable();

                List<G3dStream> lodStreams = lodModelSet.streams;
                modelType lodFile = lodModelSet.file;

                //<bone> の数と name と parent_name が同じかどうかをチェック
                if (baseFile.skeleton.bone_array.length != lodFile.skeleton.bone_array.length)
                {
                    IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_SkeletonCount", baseFile.skeleton.bone_array.length, lodFile.skeleton.bone_array.length);
                }

                ErrorBoxs[] boneErrors = new ErrorBoxs[(int)BoneError.NumBoneError];
                for (int i = 0; i < (int)BoneError.NumBoneError; ++i)
                {
                    boneErrors[i] = new ErrorBoxs();
                }
                // マトリクスインデクスから boneIndex へ変換するテーブルを作成する
                for (int baseBoneIndex = 0; baseBoneIndex < baseFile.skeleton.bone_array.length; ++baseBoneIndex)
                {
                    boneType baseBone = baseFile.skeleton.bone_array.bone[baseBoneIndex];
                    boneType lodBone = lodFile.skeleton.bone_array.bone[baseBoneIndex];

                    if (baseBone.name != lodBone.name ||
                        baseBone.parent_name != lodBone.parent_name)
                    {
                        IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_Bone", baseBone.name, lodBone.name);
                    }

                    if (!baseBone.scale.SequenceEqual(lodBone.scale) ||
                        !baseBone.rotate.SequenceEqual(lodBone.rotate) ||
                        !baseBone.translate.SequenceEqual(lodBone.translate))
                    {
                        IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_BoneSrt", lodBone.name);
                    }

                    // baseBone の smooth skinning が無効な場合に、lodBone は smooth skinning で使うことができない。
                    int baseSmoothIndex = baseBone.matrix_index[0];
                    if (baseSmoothIndex != -1)
                    {
                        indextable.baseSmoothPaletteToIndex.Add(baseSmoothIndex, baseBoneIndex);
                    }
                    int baseRigidIndex = baseBone.matrix_index[1];
                    if (baseRigidIndex != -1)
                    {
                        indextable.baseRigidPaletteToIndex.Add(baseRigidIndex, baseBoneIndex);
                    }

                    int lodSmoothIndex = lodBone.matrix_index[0];
                    if (lodSmoothIndex != -1)
                    {
                        indextable.lodSmoothPaletteToIndex.Add(lodSmoothIndex, baseBoneIndex);
                    }
                    int lodRigidIndex = lodBone.matrix_index[1];
                    if (lodRigidIndex != -1)
                    {
                        indextable.lodRigidPaletteToIndex.Add(lodRigidIndex, baseBoneIndex);
                    }

                    if (baseSmoothIndex == -1 &&
                        lodSmoothIndex != -1)
                    {
                        ErrorBox box = new ErrorBox();
                        box.index = baseBoneIndex;
                        box.baseName = baseBone.name;
                        box.lodName = lodBone.name;
                        boneErrors[(int)BoneError.RigidToSmooth].Add(box);
                    }

                    if (baseRigidIndex != -1 &&
                        string.IsNullOrEmpty(baseBone.inv_model_matrix))
                    {
                        ErrorBox box = new ErrorBox();
                        box.index = baseBoneIndex;
                        box.baseName = baseBone.name;
                        box.lodName = lodBone.name;
                        boneErrors[(int)BoneError.BaseInvModelMatrix].Add(box);
                    }

                    if (lodRigidIndex != -1 &&
                        string.IsNullOrEmpty(lodBone.inv_model_matrix))
                    {
                        ErrorBox box = new ErrorBox();
                        box.index = baseBoneIndex;
                        box.baseName = baseBone.name;
                        box.lodName = lodBone.name;
                        boneErrors[(int)BoneError.LodInvModelMatrix].Add(box);
                    }
                }

                {
                    ErrorBoxs errorBoxs = boneErrors[(int)BoneError.RigidToSmooth];

                    if (errorBoxs.Count != 0)
                    {
                        errorBoxs.MakeUp();
                        IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_RigidToSmooth", errorBoxs.BaseNameString, errorBoxs.LodNameString);
                    }
                }

                {
                    ErrorBoxs errorBoxs = boneErrors[(int)BoneError.BaseInvModelMatrix];

                    if (errorBoxs.Count != 0)
                    {
                        errorBoxs.MakeUp();
                        IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_BaseInvModelMatrixNotFound", errorBoxs.IndexString, errorBoxs.BaseNameString, errorBoxs.LodNameString);
                    }
                }

                {
                    ErrorBoxs errorBoxs = boneErrors[(int)BoneError.LodInvModelMatrix];

                    if (errorBoxs.Count != 0)
                    {
                        errorBoxs.MakeUp();
                        IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_LodInvModelMatrixNotFound", errorBoxs.IndexString, errorBoxs.BaseNameString, errorBoxs.LodNameString);
                    }
                }

                //<material> の数と name が同じかどうかをチェック
                if (baseFile.material_array.length != lodFile.material_array.length)
                {
                    IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_MaterialCount", baseFile.material_array.length, lodFile.material_array.length);
                }

                for (int i = 0; i < baseFile.material_array.length; ++i)
                {
                    materialType baseMat = baseFile.material_array.material[i];
                    materialType lodMat = baseFile.material_array.material[i];

                    if (baseMat.name != lodMat.name)
                    {
                        IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_Material", baseMat.name, lodMat.name);
                    }
                }

                //<shape> の数と name bone_name vertex_index mat_name
                if (baseFile.shape_array.length != lodFile.shape_array.length)
                {
                    IEnumerable<string> baseShapeNames = baseFile.shape_array.Items.Select(x => x.name);
                    IEnumerable<string> lodShapeNames = lodFile.shape_array.Items.Select(x => x.name);
                    IEnumerable<string> lodLackShapeNames = baseShapeNames.Except(lodShapeNames);
                    IEnumerable<string> baseLackShapeNames = lodShapeNames.Except(baseShapeNames);
                    IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_ShapeCount",
                        baseFile.shape_array.length, CreateCommaSeparatedListString(baseShapeNames),
                        lodFile.shape_array.length, CreateCommaSeparatedListString(lodShapeNames),
                        CreateCommaSeparatedListString(lodLackShapeNames.Union(baseLackShapeNames)));
                }

                bool[] smooth_to_rigid = new bool[lodFile.shape_array.length];
                for (int i = 0; i < baseFile.shape_array.length; ++i)
                {
                    smooth_to_rigid[i] = false;
                }

                List<LodShapeInfo> shapeInfoArray = new List<LodShapeInfo>(baseFile.shape_array.length);
                for (int i = 0; i < baseFile.shape_array.length; ++i)
                {
                    LodShapeInfo shapeInfo = new LodShapeInfo();

                    shapeType baseShape = baseFile.shape_array.shape[i];
                    shape_infoType baseShapeInfo = baseShape.shape_info;
                    shapeType lodShape = lodFile.shape_array.shape[i];
                    shape_infoType lodShapeInfo = lodShape.shape_info;

                    if (baseShape.name != lodShape.name ||
                        baseShapeInfo.bone_name != lodShapeInfo.bone_name ||
                        baseShapeInfo.mat_name != lodShapeInfo.mat_name ||
                        baseShapeInfo.vertex_index != lodShapeInfo.vertex_index)
                    {
                        IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_Shape", baseShape.name, lodShape.name);
                    }

                    if (baseShapeInfo.vertex_skinning_count <= (int)SkinningKind.RigidSkinning)
                    {
                        // rigid body, rigid skinning の場合は完全に一致している必要がある
                        if (baseShapeInfo.vertex_skinning_count != lodShapeInfo.vertex_skinning_count)
                        {
                            IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_ShapeSkinning", baseShape.name, lodShape.name);
                        }
                    }
                    else
                    {
                        // smooth skinning の場合は rigid body にならないようにする。
                        if (lodShapeInfo.vertex_skinning_count == (int)SkinningKind.RigidBody)
                        {
                            IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_ShapeSkinning", baseShape.name, lodShape.name);
                        }
                        else if (lodShapeInfo.vertex_skinning_count == (int)SkinningKind.RigidSkinning)
                        {
                            // smooth skinning から rigid skinning へ変わった shape
                            smooth_to_rigid[i] = true;
                        }
                    }

                    // shape が参照しているボーン一覧を作成する
                    vertexType baseVertex = baseFile.vertex_array.vertex[baseShape.shape_info.vertex_index];
                    vertexType lodVertex = lodFile.vertex_array.vertex[lodShape.shape_info.vertex_index];
                    foreach (vtx_attribType vtx_attrib in baseVertex.vtx_attrib_array.vtx_attrib)
                    {
                        if (vtx_attrib.hint.StartsWith("blendindex"))
                        {
                            List<int> indexStream = baseStreams[vtx_attrib.stream_index].IntData;
                            foreach (int index in indexStream)
                            {
                                int boneIndex = -1;
                                if (baseShapeInfo.vertex_skinning_count == (int)SkinningKind.RigidSkinning)
                                {
                                    indextable.baseRigidPaletteToIndex.TryGetValue(index, out boneIndex);
                                }
                                else if (baseShapeInfo.vertex_skinning_count >= (int)SkinningKind.SmoothSkinning)
                                {
                                    indextable.baseSmoothPaletteToIndex.TryGetValue(index, out boneIndex);
                                }

                                // 既に追加されていた場合は追加されない
                                if (boneIndex != -1 &&
                                    shapeInfo.BaseBoneIndex.Add(boneIndex))
                                {
                                    shapeInfo.BaseBoneName.Add(baseFile.skeleton.bone_array.bone[boneIndex].name);
                                }
                            }
                        }
                    }

                    foreach (vtx_attribType vtx_attrib in lodVertex.vtx_attrib_array.vtx_attrib)
                    {
                        if (vtx_attrib.hint.StartsWith("blendindex"))
                        {
                            List<int> indexStream = lodStreams[vtx_attrib.stream_index].IntData;
                            foreach (int index in indexStream)
                            {
                                int boneIndex = -1;
                                if (lodShapeInfo.vertex_skinning_count == (int)SkinningKind.RigidSkinning)
                                {
                                    indextable.lodRigidPaletteToIndex.TryGetValue(index, out boneIndex);
                                }
                                else if (lodShapeInfo.vertex_skinning_count >= (int)SkinningKind.SmoothSkinning)
                                {
                                    indextable.lodSmoothPaletteToIndex.TryGetValue(index, out boneIndex);
                                }

                                // 既に追加されていた場合は追加されない
                                if (boneIndex != -1 &&
                                    shapeInfo.LodBoneIndex.Add(boneIndex))
                                {
                                    shapeInfo.LodBoneName.Add(lodFile.skeleton.bone_array.bone[boneIndex].name);
                                }
                            }
                        }
                    }

                    foreach (string boneName in shapeInfo.LodBoneName)
                    {
                        string match = shapeInfo.BaseBoneName.Find(delegate (string name) { return name == boneName; });
                        if (match == null)
                        {
                            int index = shapeInfo.LodBoneName.IndexOf(boneName);
                            int boneIndex = shapeInfo.LodBoneIndex.ElementAt(index);
                            boneType bone = lodFile.skeleton.bone_array.bone[boneIndex];

                            int matrixIndex = -1;
                            if (lodShapeInfo.vertex_skinning_count == (int)SkinningKind.RigidSkinning)
                            {
                                matrixIndex = bone.matrix_index[1];
                            }
                            else if (lodShapeInfo.vertex_skinning_count >= (int)SkinningKind.SmoothSkinning)
                            {
                                matrixIndex = bone.matrix_index[0];
                            }

                            IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_ShapeBoneNotFound",
                                baseShape.name, lodShape.name, boneName, boneIndex, matrixIndex);
                        }
                    }

                    shapeInfoArray.Add(shapeInfo);
                }

                // lod のスケルトンを元データのスケルトンで置き換える。
                lodFile.skeleton.bone_array.bone = baseFile.skeleton.bone_array.bone;
                // lod のマテリアルを元データのマテリアルで置き換える。
                lodFile.material_array.material = baseFile.material_array.material;

                if (baseFile.original_material_array != null)
                {
                    lodFile.original_material_array.original_material = baseFile.original_material_array.original_material;
                }

                //<vtx_attrib> の blendindex の値を matrix_index 対応テーブルに
                //沿って書き換える
                //・テーブルに無かったらエラー
                for (int i = 0; i < lodFile.shape_array.length; ++i)
                {
                    shapeType lodShape = lodFile.shape_array.shape[i];
                    vertexType lodVertex = lodFile.vertex_array.vertex[lodShape.shape_info.vertex_index];
                    shapeType baseShape = baseFile.shape_array.shape[i];
                    vertexType baseVertex = baseFile.vertex_array.vertex[baseShape.shape_info.vertex_index];

                    if (smooth_to_rigid[i])
                    {
                        // smooth to rigid skinning

                        // LOD によって smooth skinning が rigid skinning になっていた場合に
                        // マトリクスパレットを変えないために smooth skinning に戻す。
                        lodShape.shape_info.vertex_skinning_count = (int)SkinningKind.SmoothSkinning;

                        // baseVertex での Name と Hint を覚えて lodVertex に反映する。
                        string blendWeightName = null;
                        string blendWeightHint = null;
                        for (int j = 0; j < baseVertex.vtx_attrib_array.length; ++j)
                        {
                            vtx_attribType vtx_attrib = baseVertex.vtx_attrib_array.vtx_attrib[j];
                            if (vtx_attrib.hint.StartsWith("blendweight"))
                            {
                                blendWeightName = vtx_attrib.name;
                                blendWeightHint = vtx_attrib.hint;
                            }
                        }

                        List<int> lodIndexStream = null;
                        for (int j = 0; j < lodVertex.vtx_attrib_array.length; ++j)
                        {
                            vtx_attribType vtx_attrib = lodVertex.vtx_attrib_array.vtx_attrib[j];
                            // lodshape の blendindex のストリームを取得しておく
                            if (vtx_attrib.hint.StartsWith("blendindex"))
                            {
                                G3dStream stream = lodStreams[vtx_attrib.stream_index];
                                lodIndexStream = stream.IntData;
                            }
                        }
                        if (lodIndexStream == null)
                        {
                            throw new InvalidDataException();
                        }

                        // vtx_buffer_array を baseVertex のもので置き換える。
                        // attrib の weight が増えることへ対応する。
                        // FetchShader を共有するためには vtx_buffer は base と lod で変えることはできない。
                        lodVertex.vtx_buffer_array = baseVertex.vtx_buffer_array;

                        // 新しい vtx_attrib_array を作成する。
                        vtx_attribType[] vtx_attrib_array = new vtx_attribType[lodVertex.vtx_attrib_array.length + 1];
                        int vtx_count = 0;
                        int idx_count = 0;
                        for (int j = 0; j < lodVertex.vtx_attrib_array.length; ++j)
                        {
                            vtx_attribType vtx_attrib = lodVertex.vtx_attrib_array.vtx_attrib[j];
                            G3dStream stream = lodStreams[vtx_attrib.stream_index];
                            if (vtx_attrib.hint.StartsWith("position") ||
                                vtx_attrib.hint.StartsWith("normal") ||
                                vtx_attrib.hint.StartsWith("tangent") ||
                                vtx_attrib.hint.StartsWith("binormal"))
                            {
                                for (int k = 0; k < vtx_attrib.count; ++k)
                                {
                                    int elementSize = stream.FloatData.Count / vtx_attrib.count;
                                    // ボーン取得
                                    int boneIndex = -1;
                                    indextable.lodRigidPaletteToIndex.TryGetValue(lodIndexStream[k], out boneIndex);
                                    if (boneIndex == -1)
                                    {
                                        throw new InvalidDataException();
                                    }
                                    boneType invBone = baseFile.skeleton.bone_array.bone[boneIndex];
                                    string[] cInvModelMatrix = invBone.inv_model_matrix.Split(null);
                                    List<float> fInvModelMatrix = new List<float>();
                                    foreach (string c in cInvModelMatrix)
                                    {
                                        if (!string.IsNullOrEmpty(c))
                                        {
                                            fInvModelMatrix.Add(Convert.ToSingle(c));
                                        }
                                    }

                                    Matrix34 invMtx = new Matrix34();
                                    for (int row = 0; row < 3; ++row)
                                    {
                                        for (int colunm = 0; colunm < 4; ++colunm)
                                        {
                                            int matIdx = row * 4 + colunm;
                                            invMtx[row, colunm] = fInvModelMatrix[matIdx];
                                        }
                                    }

                                    if (vtx_attrib.hint.StartsWith("position"))
                                    {
                                        invMtx.Invert();

                                        // position の座標系をボーンローカル座標からモデル座標に変換する。
                                        List<float> posStream = stream.FloatData;
                                        Vector3 vec = new Vector3(posStream[k * elementSize], posStream[k * elementSize + 1], posStream[k * elementSize + 2]);
                                        Vector3 temp = invMtx * vec;
                                        for (int vecIdx = 0; vecIdx < 3; ++vecIdx)
                                        {
                                            posStream[k * elementSize + vecIdx] = temp[vecIdx];
                                        }
                                    }
                                    else if (vtx_attrib.hint.StartsWith("tangent") ||
                                             vtx_attrib.hint.StartsWith("binormal"))
                                    {
                                        invMtx.Invert();

                                        // tangent/binormal の座標系をボーンローカル座標からモデル座標に変換する。
                                        List<float> tanStream = stream.FloatData;
                                        Vector3 vec = new Vector3(tanStream[k * elementSize], tanStream[k * elementSize + 1], tanStream[k * elementSize + 2]);
                                        Vector3 temp = invMtx * vec;
                                        temp.Normalize();
                                        for (int vecIdx = 0; vecIdx < 3; ++vecIdx)
                                        {
                                            tanStream[k * elementSize + vecIdx] = temp[vecIdx];
                                        }
                                    }
                                    else
                                    {
                                        invMtx.Transpose();

                                        // normal の座標系を position と同様に変換する。
                                        // 逆転置行列を用いて正確な法線方向を計算する。
                                        List<float> normStream = stream.FloatData;
                                        Vector3 vec = new Vector3(normStream[k * elementSize], normStream[k * elementSize + 1], normStream[k * elementSize + 2]);
                                        Vector3 temp = invMtx * vec;
                                        temp.Normalize();
                                        for (int vecIdx = 0; vecIdx < 3; ++vecIdx)
                                        {
                                            normStream[k * elementSize + vecIdx] = temp[vecIdx];
                                        }
                                    }
                                }
                            }
                            else if (vtx_attrib.hint.StartsWith("blendindex"))
                            {
                                vtx_attrib.type = vtx_attrib_typeType.uint2;
                                vtx_attrib.quantize_type = vtx_attrib_quantize_typeType.none;
                                stream.column = 2;
                                List<int> indexStream = stream.IntData;
                                int originalCount = indexStream.Count;
                                for (int k = 0; k < originalCount; ++k)
                                {
                                    int boneIndex = -1;
                                    indextable.lodRigidPaletteToIndex.TryGetValue(indexStream[k], out boneIndex);
                                    if (boneIndex == -1)
                                    {
                                        throw new InvalidDataException();
                                    }
                                    boneType baseBone = baseFile.skeleton.bone_array.bone[boneIndex];
                                    // ベースモデルのスムーススキニングのインデクスを取得する。
                                    int baseIndex = baseBone.matrix_index[0];
                                    if (baseIndex == -1)
                                    {
                                        throw new InvalidDataException();
                                    }
                                    indexStream[k] = baseIndex;
                                }

                                for (int k = 0; k < originalCount; ++k)
                                {
                                    // 同一インデクスをコピーする。
                                    indexStream.Insert(k * 2 + 1, indexStream[k * 2]);
                                }
                                vtx_count = vtx_attrib.count;
                                idx_count = indexStream.Count;
                            }
                            else if (vtx_attrib.hint.StartsWith("blendweight"))
                            {
                                throw new InvalidDataException();
                            }

                            vtx_attrib_array[j] = vtx_attrib;
                        }

                        // rigid skinning は blendweight が存在しないので挿入する。
                        {
                            vtx_attribType vtx_attrib = new vtx_attribType();
                            vtx_attrib.name = blendWeightName;
                            vtx_attrib.hint = blendWeightHint;
                            vtx_attrib.type = vtx_attrib_typeType.float2;
                            vtx_attrib.quantize_type = vtx_attrib_quantize_typeType.none;
                            vtx_attrib.count = vtx_count;
                            vtx_attrib.stream_index = lodStreams.Count;
                            vtx_attrib_array[lodVertex.vtx_attrib_array.length] = vtx_attrib;

                            G3dStream stream = new G3dStream();
                            stream.type = stream_typeType.@float;
                            stream.column = 2;
                            List<float> weightStream = stream.FloatData;
                            for (int k = 0; k < idx_count; ++k)
                            {
                                float weight = 0.0f;
                                if (k % 2 == 0)
                                {
                                    weight = 1.0f;
                                }

                                weightStream.Add(weight);
                            }

                            lodStreams.Add(stream);
                        }

                        lodVertex.vtx_attrib_array.vtx_attrib = vtx_attrib_array;
                        lodVertex.vtx_attrib_array.length = lodVertex.vtx_attrib_array.vtx_attrib.Length;
                    }
                    else
                    {
                        for (int j = 0; j < lodVertex.vtx_attrib_array.length; ++j)
                        {
                            vtx_attribType vtx_attrib = lodVertex.vtx_attrib_array.vtx_attrib[j];
                            if (vtx_attrib.hint.StartsWith("blendindex"))
                            {
                                List<int> indexStream = lodStreams[vtx_attrib.stream_index].IntData;
                                if (vtx_attrib.type == vtx_attrib_typeType.@uint)
                                {
                                    // rigid skinning
                                    for (int k = 0; k < indexStream.Count; ++k)
                                    {
                                        int boneIndex = -1;
                                        indextable.lodRigidPaletteToIndex.TryGetValue(indexStream[k], out boneIndex);
                                        if (boneIndex == -1)
                                        {
                                            throw new InvalidDataException();
                                        }
                                        boneType baseBone = baseFile.skeleton.bone_array.bone[boneIndex];
                                        // ベースモデルのリジッドスキニングのインデクスを取得する。
                                        int baseIndex = baseBone.matrix_index[1];
                                        if (baseIndex == -1)
                                        {
                                            throw new InvalidDataException();
                                        }

                                        indexStream[k] = baseIndex;
                                    }
                                }
                                else
                                {
                                    // smooth skinning
                                    for (int k = 0; k < indexStream.Count; ++k)
                                    {
                                        int boneIndex = -1;
                                        indextable.lodSmoothPaletteToIndex.TryGetValue(indexStream[k], out boneIndex);
                                        if (boneIndex == -1)
                                        {
                                            throw new InvalidDataException();
                                        }
                                        boneType baseBone = baseFile.skeleton.bone_array.bone[boneIndex];
                                        // ベースモデルのスムーススキニングのインデクスを取得する。
                                        int baseIndex = baseBone.matrix_index[0];
                                        if (baseIndex == -1)
                                        {
                                            throw new InvalidDataException();
                                        }

                                        indexStream[k] = baseIndex;
                                    }
                                }
                            }
                        }
                    }

                    if (lodVertex.vtx_attrib_array.length != baseVertex.vtx_attrib_array.length)
                    {
                        IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_VertexAttribCount",
                            baseShape.name,
                            baseVertex.vtx_attrib_array.length, CreateAttributeListString(baseVertex.vtx_attrib_array.Items),
                            lodVertex.vtx_attrib_array.length, CreateAttributeListString(lodVertex.vtx_attrib_array.Items));
                    }

                    // blendweight と blendindex の次元数を揃える
                    for (int attributeIndex = 0; attributeIndex < lodVertex.vtx_attrib_array.length; ++attributeIndex)
                    {
                        vtx_attribType baseVtx_attrib = baseVertex.vtx_attrib_array.vtx_attrib[attributeIndex];
                        vtx_attribType lodVtx_attrib = lodVertex.vtx_attrib_array.vtx_attrib[attributeIndex];

                        if (baseVtx_attrib.name != lodVtx_attrib.name)
                        {
                            IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_VertexAttrib",
                                $"name of {GetIntermediateFileElementName(typeof(vtx_attribType))}",
                                baseShape.name,
                                baseVtx_attrib.name, CreateAttributeListString(baseVertex.vtx_attrib_array.Items),
                                lodVtx_attrib.name, CreateAttributeListString(lodVertex.vtx_attrib_array.Items));
                        }

                        if (baseVtx_attrib.hint != lodVtx_attrib.hint)
                        {
                            IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_VertexAttrib",
                                $"hint of {GetIntermediateFileElementName(typeof(vtx_attribType))}",
                                baseShape.name,
                                baseVtx_attrib.name, CreateAttributeListString(baseVertex.vtx_attrib_array.Items),
                                lodVtx_attrib.name, CreateAttributeListString(lodVertex.vtx_attrib_array.Items));
                        }

                        if (baseVtx_attrib.type != lodVtx_attrib.type)
                        {
                            // スキニングインデックス、スキニングウェイトの型を自動補正
                            G3dStream baseStream = baseStreams[baseVtx_attrib.stream_index];
                            G3dStream lodStream = lodStreams[lodVtx_attrib.stream_index];
                            if (baseStream.type != lodStream.type)
                            {
                                // ストリーム型が一致しない
                                IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_VertexAttrib",
                                    $"type of {GetIntermediateFileElementName(typeof(streamType))}",
                                    baseShape.name,
                                    baseVtx_attrib.name, CreateAttributeListString(baseVertex.vtx_attrib_array.Items),
                                    lodVtx_attrib.name, CreateAttributeListString(lodVertex.vtx_attrib_array.Items));
                            }
                            int baseDim = baseStream.column;
                            int lodDim = lodStream.column;
                            if (baseDim < lodDim)
                            {
                                // LOD モデルの方がインフルエンス数が多い場合は補正できない
                                IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_VertexAttrib",
                                    "column of <stream>",
                                    baseShape.name,
                                    baseVtx_attrib.name, CreateAttributeListString(baseVertex.vtx_attrib_array.Items),
                                    lodVtx_attrib.name, CreateAttributeListString(lodVertex.vtx_attrib_array.Items));
                            }

                            if (baseVtx_attrib.hint.StartsWith("blendweight"))
                            {
                                // float 以外は想定しない
                                if (baseStream.type != stream_typeType.@float)
                                {
                                    throw new InvalidDataException();
                                }

                                List<float> source = new List<float>(lodStream.FloatData);
                                lodStream.FloatData.Clear();
                                for (int k = 0; k < source.Count;)
                                {
                                    for (int l = 0; l < lodDim; ++l, ++k)
                                    {
                                        lodStream.FloatData.Add(source[k]);
                                    }
                                    for (int l = lodDim; l < baseDim; ++l)
                                    {
                                        // weight は 0 で埋める
                                        lodStream.FloatData.Add(0.0f);
                                    }
                                }
                                switch (baseDim)
                                {
                                    case 1:
                                        lodVtx_attrib.type = vtx_attrib_typeType.@float;
                                        break;
                                    case 2:
                                        lodVtx_attrib.type = vtx_attrib_typeType.float2;
                                        break;
                                    case 3:
                                        lodVtx_attrib.type = vtx_attrib_typeType.float3;
                                        break;
                                    case 4:
                                        lodVtx_attrib.type = vtx_attrib_typeType.float4;
                                        break;
                                    default:
                                        throw new InvalidDataException();
                                }
                                lodStream.column = baseDim;
                            }
                            else if (baseVtx_attrib.hint.StartsWith("blendindex"))
                            {
                                // int 以外は想定しない
                                if (baseStream.type != stream_typeType.@int)
                                {
                                    throw new InvalidDataException();
                                }

                                List<int> source = new List<int>(lodStream.IntData);
                                lodStream.IntData.Clear();
                                for (int k = 0; k < source.Count;)
                                {
                                    int padding = source[k];
                                    for (int l = 0; l < lodDim; ++l, ++k)
                                    {
                                        lodStream.IntData.Add(source[k]);
                                    }
                                    for (int l = lodDim; l < baseDim; ++l)
                                    {
                                        // index は先頭要素で埋める
                                        lodStream.IntData.Add(padding);
                                    }
                                }
                                switch (baseDim)
                                {
                                    case 1:
                                        lodVtx_attrib.type = vtx_attrib_typeType.@uint;
                                        break;
                                    case 2:
                                        lodVtx_attrib.type = vtx_attrib_typeType.uint2;
                                        break;
                                    case 3:
                                        lodVtx_attrib.type = vtx_attrib_typeType.uint3;
                                        break;
                                    case 4:
                                        lodVtx_attrib.type = vtx_attrib_typeType.uint4;
                                        break;
                                    default:
                                        throw new InvalidDataException();
                                }
                                lodStream.column = baseDim;
                            }
                            else
                            {
                                // blendweight と blendindex 以外は補正できない
                                IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_VertexAttrib",
                                    $"type of {GetIntermediateFileElementName(typeof(vtx_attribType))}",
                                    baseShape.name,
                                    baseVtx_attrib.name, CreateAttributeListString(baseVertex.vtx_attrib_array.Items),
                                    lodVtx_attrib.name, CreateAttributeListString(lodVertex.vtx_attrib_array.Items));
                            }
                        }
                    }
                }
                CheckStreamLength(lodFile, lodStreams);
            }

            // Stream と Mesh の結合を行う
            {
                IList<bool> streamMergerdFlag = new List<bool>();
                for (int i = 0; i < Streams.Count; ++i)
                {
                    // false で埋める。(不要？)
                    streamMergerdFlag.Add(false);
                }

                for (int baseModelShapeIndex = 0; baseModelShapeIndex < baseFile.shape_array.length; ++baseModelShapeIndex)
                {
                    shapeType baseShape = baseFile.shape_array.shape[baseModelShapeIndex];
                    vertexType baseVertex = baseFile.vertex_array.vertex[baseShape.shape_info.vertex_index];

                    int vertexCount = baseVertex.vtx_attrib_array.vtx_attrib[0].count;

                    List<int> lod_offsets = new List<int> { 0 };
                    if (baseVertex.lod_offset != null)
                    {
                        lod_offsets.AddRange(G3dDataParser.ParseIntArray(baseVertex.lod_offset.Value));
                    }
                    int numLevel = Math.Max(lod_offsets.Count, lodModels.Last().level + 1);
                    lod_offsets.Add(vertexCount);

                    List<int> new_lod_offsets = new List<int>();

                    // Stream のマージ
                    for (int baseModelVertexAttrIndex = 0; baseModelVertexAttrIndex < baseVertex.vtx_attrib_array.length; ++baseModelVertexAttrIndex)
                    {
                        vtx_attribType baseVtx_attrib = baseVertex.vtx_attrib_array.vtx_attrib[baseModelVertexAttrIndex];
                        G3dStream baseStream = Streams[baseVtx_attrib.stream_index];

                        G3dStream newStream = new G3dStream();
                        newStream.type = baseStream.type;
                        newStream.column = baseStream.column;

                        if (baseVtx_attrib.count != vertexCount)
                        {
                            // 念のためチェック
                            throw new InvalidDataException();
                        }

                        if (streamMergerdFlag[baseVtx_attrib.stream_index])
                        {
                            // １つのストリームが複数の頂点属性から参照されているのでマージできない。
                            throw new InvalidDataException();
                        }
                        streamMergerdFlag[baseVtx_attrib.stream_index] = true;

                        // 量子化設定を無くす
                        baseVtx_attrib.quantize_type = vtx_attrib_quantize_typeType.none;

                        int sum = 0;
                        int idxLodModels = 0;
                        for (int level = 0; level < numLevel; ++level)
                        {
                            G3dStream mergeStream = baseStream;
                            int start = 0;
                            int count = 0;
                            int shareLevel = -1;
                            if (idxLodModels < lodModels.Count && lodModels[idxLodModels].level == level)
                            {
                                LodModelSet lodModelSet = lodModels[idxLodModels++];
                                shapeType lodShape = lodModelSet.file.shape_array.shape[baseModelShapeIndex];
                                vertexType lodVertex = lodModelSet.file.vertex_array.vertex[lodShape.shape_info.vertex_index];
                                vtx_attribType lodVtx_attrib = lodVertex.vtx_attrib_array.vtx_attrib[baseModelVertexAttrIndex];
                                if (baseVtx_attrib.name != lodVtx_attrib.name)
                                {
                                    IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_VertexAttrib",
                                        $"name of {GetIntermediateFileElementName(typeof(vtx_attribType))}",
                                        baseShape.name,
                                        baseVtx_attrib.name, CreateAttributeListString(baseVertex.vtx_attrib_array.Items),
                                        lodVtx_attrib.name, CreateAttributeListString(lodVertex.vtx_attrib_array.Items));
                                }

                                if (baseVtx_attrib.hint != lodVtx_attrib.hint)
                                {
                                    IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_VertexAttrib",
                                        $"hint of {GetIntermediateFileElementName(typeof(vtx_attribType))}",
                                        baseShape.name,
                                        baseVtx_attrib.name, CreateAttributeListString(baseVertex.vtx_attrib_array.Items),
                                        lodVtx_attrib.name, CreateAttributeListString(lodVertex.vtx_attrib_array.Items));
                                }

                                if (baseVtx_attrib.type != lodVtx_attrib.type)
                                {
                                    IfStrings.Throw("IfModelUniteLodModelOptimizer_Error_VertexAttrib",
                                        $"type of {GetIntermediateFileElementName(typeof(vtx_attribType))}",
                                        baseShape.name,
                                        baseVtx_attrib.name, CreateAttributeListString(baseVertex.vtx_attrib_array.Items),
                                        lodVtx_attrib.name, CreateAttributeListString(lodVertex.vtx_attrib_array.Items));
                                }

                                mergeStream = lodModelSet.streams[lodVtx_attrib.stream_index];
                                count = lodVtx_attrib.count;
                            }
                            else
                            {
                                start = lod_offsets[level];
                                for (int share = 0; share < level; ++share)
                                {
                                    if (lod_offsets[share] == lod_offsets[level] && lodModels.FindIndex(p => p.level == share) < 0)
                                    {
                                        shareLevel = share;
                                        break;
                                    }
                                }
                                if (shareLevel < 0)
                                {
                                    count = lod_offsets.Min(p => p > start ? p : vertexCount) - start;
                                }
                            }

                            // マージ
                            switch (baseStream.type)
                            {
                                case stream_typeType.@float:
                                    newStream.FloatData.AddRange(mergeStream.FloatData.Skip(
                                        start * baseStream.column).Take(count * baseStream.column));
                                    break;
                                case stream_typeType.@int:
                                    newStream.IntData.AddRange(mergeStream.IntData.Skip(
                                        start * baseStream.column).Take(count * baseStream.column));
                                    break;
                                case stream_typeType.@byte:
                                    newStream.ByteData.AddRange(mergeStream.ByteData.Skip(
                                        start * baseStream.column).Take(count * baseStream.column));
                                    break;
                                default:
                                    throw new InvalidDataException();
                            }
                            if (baseModelVertexAttrIndex == 0)
                            {
                                new_lod_offsets.Add(shareLevel >= 0 ? new_lod_offsets[shareLevel] : sum);
                            }
                            sum += count;
                        }
                        baseVtx_attrib.count = sum;
                        Streams[baseVtx_attrib.stream_index] = newStream;
                    }

                    // lod_offset の設定
                    baseVertex.lod_offset = new lod_offsetType();
                    baseVertex.lod_offset.count = new_lod_offsets.Count - 1;
                    baseVertex.lod_offset.Value = new_lod_offsets.Skip(1).Select(p => p.ToString() + " ").Aggregate((a, b) => a + b).TrimEnd();

                    // mesh のマージ
                    meshType[] newMesh = new meshType[numLevel];
                    int idxLod = 0;
                    for (int level = 0; level < numLevel; ++level)
                    {
                        if (idxLod < lodModels.Count && lodModels[idxLod].level == level)
                        {
                            LodModelSet lodModelSet = lodModels[idxLod++];
                            shapeType lodShape = lodModelSet.file.shape_array.shape[baseModelShapeIndex];
                            meshType lodMesh = lodShape.mesh_array.mesh[0];

                            Streams.Add(lodModelSet.streams[lodMesh.stream_index]);
                            lodMesh.stream_index = Streams.Count - 1;

                            newMesh[level] = lodMesh;
                        }
                        else
                        {
                            newMesh[level] = baseShape.mesh_array.mesh[level];
                        }
                        newMesh[level].quantize_type = mesh_quantize_typeType.none;
                    }
                    baseShape.mesh_array.mesh = newMesh;

                    // polygon_reduction_mode の変更
                    if (!string.IsNullOrEmpty(baseShape.shape_info.polygon_reduction_mode))
                    {
                        baseShape.shape_info.polygon_reduction_mode += uniteLodMode;
                    }
                    baseShape.shape_info.optimize_primitive_mode = string.Empty;
                    foreach (var vtxAttrib in baseVertex.vtx_attrib_array.vtx_attrib)
                    {
                        vtxAttrib.quantize_type = vtx_attrib_quantize_typeType.none;
                    }
                }
            }
            StreamUtility.SortStream(this.Target, this.Streams);
            EnableProcessLog = true;
        }

        private string GetIntermediateFileElementName(Type type)
        {
            return $"<{type.Name.Substring(0, type.Name.Length - "Type".Length)}>";
        }

        private string CreateAttributeListString(IEnumerable<vtx_attribType> attrs)
        {
            StringBuilder baseAttrList = new StringBuilder();
            foreach (vtx_attribType attr in attrs)
            {
                baseAttrList.Append($"{attr.name}, ");
            }
            return baseAttrList.ToString(0, baseAttrList.Length - 2);
        }

        private string CreateCommaSeparatedListString(IEnumerable<string> items)
        {
            StringBuilder result = new StringBuilder();
            foreach (var item in items)
            {
                result.Append($"{item}, ");
            }
            if (items.Count() > 0)
            {
                result.Remove(result.Length - 2, 2);
            }
            return result.ToString();
        }

        // targetModel の vtx_attrib の情報と、それが参照するストリームの要素数が一致しているかをチェックする
        private void CheckStreamLength(modelType targetModel, List<G3dStream> streams)
        {
            for (int i = 0; i < targetModel.vertex_array.length; ++i)
            {
                vertexType vertex = targetModel.vertex_array.vertex[i];
                for (int j = 0; j < vertex.vtx_attrib_array.length; ++j)
                {
                    vtx_attribType vtx_attrib = vertex.vtx_attrib_array.vtx_attrib[j];
                    int vertexElementCount;
                    switch (vtx_attrib.type)
                    {
                        case vtx_attrib_typeType.@int:
                        case vtx_attrib_typeType.@uint:
                        case vtx_attrib_typeType.@float:
                            vertexElementCount = vtx_attrib.count * 1;
                            break;
                        case vtx_attrib_typeType.int2:
                        case vtx_attrib_typeType.uint2:
                        case vtx_attrib_typeType.float2:
                            vertexElementCount = vtx_attrib.count * 2;
                            break;
                        case vtx_attrib_typeType.int3:
                        case vtx_attrib_typeType.uint3:
                        case vtx_attrib_typeType.float3:
                            vertexElementCount = vtx_attrib.count * 3;
                            break;
                        case vtx_attrib_typeType.int4:
                        case vtx_attrib_typeType.uint4:
                        case vtx_attrib_typeType.float4:
                            vertexElementCount = vtx_attrib.count * 4;
                            break;
                        default:
                            throw new InvalidDataException();
                    }

                    G3dStream stream = streams[vtx_attrib.stream_index];
                    int streamLength;
                    switch (stream.type)
                    {
                        case stream_typeType.@float:
                            streamLength = stream.FloatData.Count;
                            break;
                        case stream_typeType.@int:
                            streamLength = stream.IntData.Count;
                            break;
                        case stream_typeType.@byte:
                            streamLength = stream.ByteData.Count;
                            break;
                        default:
                            throw new InvalidDataException();
                    }

                    if (streamLength != vertexElementCount)
                    {
                        // 頂点属性から求めた要素数とストリームの実体が一致しない
                        throw new InvalidDataException();
                    }
                }
            }
        }
    }
}
