﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
#define EXCUSION_TRIANGLE
#define DIVIDE_CHILDREN
//#define POLYGON_TEST

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using nw.g3d.nw4f_3dif;
using System.Collections;
using System.Xml;
using System.Diagnostics;
using System.Text.RegularExpressions;

using ArgumentProcessor = nw.g3d.iflib.IfDivideUtility.ArgumentProcessor;
using Triangle = nw.g3d.iflib.IfDivideUtility.DivideTriangle;
using ITriangle = nw.g3d.iflib.IfDivideUtility.ITriangle;
using Plane = nw.g3d.iflib.IfDivideUtility.Plane;
using AABB = nw.g3d.iflib.IfDivideUtility.DivideAABB;
using DivideMesh = nw.g3d.iflib.IfDivideUtility.DivideMesh;
using MeshNode = nw.g3d.iflib.IfDivideUtility.MeshNode;

namespace nw.g3d.iflib
{
    // モデルのサブメッシュ分割
    public class IfModelDivideSubmeshOptimizer : IfModelOptimizer
    {
        // コンストラクタ
        public IfModelDivideSubmeshOptimizer(string argument) :
            base("IfModelDivideSubmeshOptimizer_Log", argument)
        { }

        // プロセス
        public override string Process
        {
            get { return "divide_submesh"; }
        }

        public override string ToString()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendLine(string.Format("{0}\n{1}", base.ToString(), GetResult()));

            foreach (shapeType shape in this.Target.shape_array.shape)
            {
                builder.Append(string.Format("{0}:", shape.name));
                int index = 0;
                foreach (IfDivideUtility.ResultCode code in dicResults[shape.index].Keys)
                {
                    if (index++ % 3 == 0)
                    {
                        builder.AppendLine();
                        builder.Append("    ");
                    }
                    builder.Append(string.Format("{0,12}:{1,4}  ", code, dicResults[shape.index][code]));
                }
                builder.AppendLine();
            }

            return builder.ToString();
        }

        public override string GetResult()
        {
            StringBuilder builder = new StringBuilder();
            for (int shapeIndex = 0; shapeIndex < ShapeCount; ++shapeIndex)
            {
                builder.AppendLine(string.Format("{2,3} => {1,-3} ({0})", ShapeName[shapeIndex], AfterSubmeshCount[shapeIndex], BeforeSubmeshCount[shapeIndex]));
            }
            return builder.ToString();
        }

        private int ShapeCount { get; set; }
        private string[] ShapeName { get { return shapeName; } }
        private int[] BeforeSubmeshCount { get { return beforeSubmeshCount; } }
        private int[] AfterSubmeshCount { get { return afterSubmeshCount; } }
        private string[] shapeName = null;
        private int[] beforeSubmeshCount = null;
        private int[] afterSubmeshCount = null;
        private Dictionary<IfDivideUtility.ResultCode, uint>[] dicResults = null;

        // 最適化
        protected override void Optimize()
        {
            if (this.Target.vertex_array == null ||
                this.Target.vertex_array.vertex == null) { return; }
            if (this.Target.shape_array == null ||
                this.Target.shape_array.shape == null) { return; }

            // 頂点列のハッシュが求められていなかったら、求めておく

            // 引数の分割ロジックに従ってサブメッシュを分割する
            // 引数が同じなら分割しない
            // 引数と入力が同じなら、同じ結果を返す

            const int DepthMaxEx = 1;
            float widthMax = 100.0f;
            int depthMax = 5;
            int minTriangleCount = 1;

            // 百分率で指定
            const float ReductionTargetMin = 0.0f;
            const float ReductionTargetMax = 95.0f;

            bool isLodTrimed = false;

            ArgumentProcessor processor = new ArgumentProcessor(this.Argument);

            if (string.IsNullOrEmpty(processor.Function))
            {
                // 引数が少なすぎる場合はエラー
                IfStrings.Throw("IfModelDivideSubmeshOptimizer_Error_NotEnoughArguments", this.Argument);
            }

            if (processor.Function == "kd")
            {
                string max_width = null;
                string max_depth = null;
                foreach (var argValue in processor.Arguments)
                {
                    if (argValue.arg == "max_width")
                    {
                        max_width = argValue.value;
                    }
                    if (argValue.arg == "max_depth")
                    {
                        max_depth = argValue.value;
                    }
                    if (argValue.arg == "min_triangle_count")
                    {
                        minTriangleCount = Convert.ToInt32(argValue.value);
                    }
                }
                if (!string.IsNullOrEmpty(max_width))
                {
                    widthMax = Convert.ToSingle(max_width);
                }

                if (!string.IsNullOrEmpty(max_depth))
                {
                    depthMax = Convert.ToInt32(max_depth);
                }
            }
            else
            {
                // mode が指定されていない場合はエラー
                IfStrings.Throw("IfModelDivideSubmeshOptimizer_Error_UnknownArguments", processor.Function, "mode");
            }

            ShapeCount = this.Target.shape_array.Count;
            shapeName = new string[ShapeCount];
            beforeSubmeshCount = new int[ShapeCount];
            afterSubmeshCount = new int[ShapeCount];

            dicResults = new Dictionary<IfDivideUtility.ResultCode, uint>[this.Target.shape_array.length];
            for (int iDic = 0; iDic < this.Target.shape_array.length; ++iDic)
            {
                dicResults[iDic] = new Dictionary<IfDivideUtility.ResultCode, uint>();
                foreach (IfDivideUtility.ResultCode code in Enum.GetValues(typeof(IfDivideUtility.ResultCode)))
                {
                    dicResults[iDic].Add(code, 0);
                }
            }

            G3dParallel.ForEach(this.Target.shape_array.shape, delegate (shapeType shape)
            {
                // DCC から出力した状態のハッシュを求める
                string hash = shape.shape_info.original_shape_hash;
                if (hash == string.Empty)
                {
                    hash = IfOptimizePrimitiveHashCalculator.Calculate(
                        this.Target, this.Streams, shape);
                    shape.shape_info.original_shape_hash = hash;
                }
                shapeName[shape.index] = shape.name;
                beforeSubmeshCount[shape.index] = shape.mesh_array.mesh[0].submesh_array.Count;
                // とりあえず、同じ数を入れておく
                afterSubmeshCount[shape.index] = shape.mesh_array.mesh[0].submesh_array.Count;

                // 過去の分割と同じ引数が指定された時は何もしない
                if (shape.shape_info.divide_submesh_mode == null ||
                    shape.shape_info.divide_submesh_mode.CompareTo(this.Argument) != 0)
                {
                    EnableProcessLog = true;

                    boneType bone = this.Target.skeleton.bone_array.Enumerate().Single(p => p.name == shape.shape_info.bone_name);
                    string shape_name = shape.name;
                    string mat_name = shape.shape_info.mat_name;

                    Debug.WriteLine("mat_name: " + mat_name);

                    // リジッドボディ以外はサブメッシュ分割できない。
                    if (shape.shape_info.vertex_skinning_count != 0)
                    {
                        Console.Error.WriteLine(
                            IfStrings.Get("IfModelDivideSubmeshOptimizer_Warning_SkinnedShape", shape.name));
                        return;
                    }

                    // triangle 以外はサブメッシュ分割できない。
                    if (shape.mesh_array.mesh[0].mode != mesh_modeType.triangles)
                    {
                        Console.Error.WriteLine(
                            IfStrings.Get("IfModelDivideSubmeshOptimizer_Warning_UnsupportedMeshMode", shape.name));
                        return;
                    }

                    // LOD がある場合は消す
                    if (shape.mesh_array.mesh.Length > 1)
                    {
                        IfModelTrimLodModelOptimizer trimLodOptimizer = new IfModelTrimLodModelOptimizer();
                        trimLodOptimizer.Target = this.Target;
                        trimLodOptimizer.Streams = this.Streams;
                        trimLodOptimizer.TrimLod(shape);
                        shape.shape_info.polygon_reduction_mode = string.Empty;
                        isLodTrimed = true;
                    }

                    // rootMesh の構築
                    DivideMesh rootMesh = new DivideMesh(null, new AABB());
                    G3dStream iStream = Streams[shape.mesh_array.mesh[0].stream_index];
                    {
                        vertexType vertex = this.Target.vertex_array.vertex[shape.shape_info.vertex_index];
                        int vtxAttribIndex = 0;
                        int positionIndex = -1;
                        foreach (vtx_attribType vtxAttrib in vertex.vtx_attrib_array.Enumerate())
                        {
                            // hint を用いて position の Stream を検索する。
                            if (vtxAttrib.hint.StartsWith("position"))
                            {
                                positionIndex = vtxAttribIndex;
                                break;
                            }
                            ++vtxAttribIndex;
                        }

                        G3dStream vStream = Streams[vertex.vtx_attrib_array.vtx_attrib[positionIndex].stream_index];

                        int triangleCount = shape.mesh_array.mesh[0].count / 3;
                        Vector3 shapeMin = new Vector3(float.MaxValue, float.MaxValue, float.MaxValue);
                        Vector3 shapeMax = new Vector3(float.MinValue, float.MinValue, float.MinValue);
                        for (int i = 0; i < triangleCount; ++i)
                        {
                            int iIndex = i * 3;
                            Triangle triangle = new Triangle();
                            iStream.IntData.CopyTo(iIndex, triangle.Indices, 0, 3);

                            Vector3 min = new Vector3(float.MaxValue, float.MaxValue, float.MaxValue);
                            Vector3 max = new Vector3(float.MinValue, float.MinValue, float.MinValue);

                            for (int j = 0; j < 3; ++j)
                            {
                                int vIndex = triangle.Indices[j] * 3;
                                float[] data = { vStream.FloatData[vIndex], vStream.FloatData[vIndex + 1], vStream.FloatData[vIndex + 2] };
                                triangle.Vertices[j] = new Vector3(data);

                                // ScaleTransform
                                triangle.Vertices[j].X *= bone.scale[0];
                                triangle.Vertices[j].Y *= bone.scale[1];
                                triangle.Vertices[j].Z *= bone.scale[2];

                                IfDivideUtility.MinVector3(min, triangle.Vertices[j], min);
                                IfDivideUtility.MaxVector3(max, triangle.Vertices[j], max);
                            }

                            triangle.Center.Set((max + min) / 2.0f);

                            // 除外ポリゴンに印をつける。
                            if ((max.X - min.X) > widthMax ||
                                (max.Y - min.Y) > widthMax ||
                                (max.Z - min.Z) > widthMax)
                            {
                                triangle.IsExclusion = true;
                            }
                            rootMesh.AddTriangle(triangle);
                        }
                        rootMesh.Calc();
                    }

                    // rootMesh の分割
                    {
                        // 除外ポリゴンを別の AABB に分離するアルゴリズム
                        {
                            IfDivideUtility.DivideExculusion algorithm = new IfDivideUtility.DivideExculusion();
                            algorithm.DepthMax = DepthMaxEx;
                            rootMesh.Divide(algorithm);
                        }

                        // シェイプ圧縮前のシェイプが何個のシェイプで構成されていたかによって処理を分ける。
                        // もし２つ以上のシェイプで構築されていた場合は縮減率を用いた方法で分割を行う。
                        if (shape.shape_info.original_shape_count == 1)
                        {
                            IfDivideUtility.DivideEigen algorithm = new IfDivideUtility.DivideEigen();
                            algorithm.DepthMax = depthMax;
                            algorithm.WidthMax = widthMax;
                            algorithm.MinTriangleCount = minTriangleCount;
                            rootMesh.Divide(algorithm);
                        }
                        else
                        {
                            IfDivideUtility.DivideSquashing algorithm = new IfDivideUtility.DivideSquashing();
                            algorithm.DepthMax = depthMax;
                            algorithm.WidthMax = widthMax;
                            algorithm.MinTriangleCount = minTriangleCount;
                            algorithm.ReductionTargetMax = ReductionTargetMax;
                            algorithm.ReductionTargetMin = ReductionTargetMin;
                            rootMesh.Divide(algorithm);
                            // リーフノードをさらに細かく分割する。
                            IfDivideUtility.DivideEigen subAlgorithm = new IfDivideUtility.DivideEigen();
                            subAlgorithm.DepthMax = depthMax;
                            subAlgorithm.WidthMax = widthMax;
                            subAlgorithm.MinTriangleCount = minTriangleCount;
                            rootMesh.Divide(subAlgorithm);
                        }
                    }

                    // メッシュのマージ
                    {
                        List<DivideMesh> leafNodeArray = DivideMesh.BakeLeaf(rootMesh);
                        // マージ対象が存在しなくなるまで内包する AABB のマージを行う。
                        bool isMerge = true;
                        while (isMerge)
                        {
                            isMerge = false;
                            for (int leafIndex = 0; leafIndex < leafNodeArray.Count; ++leafIndex)
                            {
                                DivideMesh leafMesh = leafNodeArray[leafIndex];
                                for (int targetIndex = 0; targetIndex < leafNodeArray.Count; ++targetIndex)
                                {
                                    if (leafIndex == targetIndex)
                                    {
                                        // 自分はスキップ
                                        continue;
                                    }

                                    DivideMesh targetMesh = leafNodeArray[targetIndex];
                                    // 例外ポリゴンを含むメッシュはマージ対象にしない。
                                    if (targetMesh.HasExclusionPolygon)
                                    {
                                        continue;
                                    }

                                    if (targetMesh.Test(leafMesh))
                                    {
                                        DivideMesh newRootNode = targetMesh.Merge(leafMesh);
                                        // targetCube は leafCube を内包している。
                                        if (newRootNode != null)
                                        {
                                            // マージした結果 rootCube のみになった場合は
                                            // rootCube を入れ替える。
                                            rootMesh = newRootNode;
                                        }

                                        // マージ用リストから削除する。
                                        leafNodeArray.Remove(leafMesh);
                                        --leafIndex;

                                        isMerge = true;

                                        break;
                                    }
                                }
                            }
                        }
                    }

                    // 中間ファイルへの書き出し
                    {
                        {
                            DivideMesh.BreadthSearch(rootMesh);
                            DivideMesh.DepthSearch(rootMesh);

                            List<MeshNode> boundingNodeArray = new List<MeshNode>();
                            int submeshCount = 0;
                            List<DivideMesh> bakedMeshArray = DivideMesh.Bake(rootMesh);
                            foreach (DivideMesh mesh in bakedMeshArray)
                            {
                                MeshNode boundingNode = new MeshNode(mesh);
                                boundingNodeArray.Add(boundingNode);
                                if (mesh.Triangles.Count > 0)
                                {
                                    ++submeshCount;
                                }
                            }

                            // 中間ファイルへの書き出し
                            //
                            // IndexStream の構築
                            iStream.IntData.Clear();
                            // トライアングルの順序が入れ替わるのでプリミティブ最適化は無効化する。
                            shape.shape_info.optimize_primitive_mode = string.Empty;

                            // DepthIndex を用いて深さ優先でソートを行います。
                            boundingNodeArray.Sort(
                                delegate (MeshNode lhs, MeshNode rhs)
                                {
                                    return lhs.DepthIndex - rhs.DepthIndex;
                                });

                            // submesh_array 書き出し
                            submeshType[] submeshArray = new submeshType[submeshCount];
                            int submeshIndex = 0;
                            foreach (MeshNode node in boundingNodeArray)
                            {
                                if (node.AddToIndexStream(iStream.IntData, submeshIndex))
                                {
                                    submeshType submesh = new submeshType();
                                    submesh.count = node.IndexCount;
                                    submesh.offset = node.IndexOffset;

                                    submeshArray[submeshIndex] = submesh;
                                    ++submeshIndex;
                                }
                            }

                            shape.mesh_array.mesh[0].submesh_array = new submesh_arrayType();
                            shape.mesh_array.mesh[0].submesh_array.submesh = submeshArray;

                            // BreadthIndex を用いて幅優先でソートを行います。
                            boundingNodeArray.Sort(
                                delegate (MeshNode lhs, MeshNode rhs)
                                {
                                    return lhs.BreadthIndex - rhs.BreadthIndex;
                                });
                        }
                        afterSubmeshCount[shape.index] = shape.mesh_array.mesh[0].submesh_array.Count;

                        shape.shape_info.divide_submesh_mode = this.Argument;
                    }

                    // ログ出力
                    {
                        if (shape.mesh_array.mesh[0].submesh_array != null)
                        {
                            int submeshIndex = 0;
                            foreach (submeshType submesh in shape.mesh_array.mesh[0].submesh_array.Enumerate())
                            {
                                Debug.WriteLine(string.Format("{0}: shape[{1}].submesh[{2}].count: {3}", shape.shape_info.mat_name, shape.index, submeshIndex, submesh.count));
                                ++submeshIndex;
                            }
                        }

                        GatherResultCodes(rootMesh, dicResults[shape.index]);
                    }
                }
            });

            if (isLodTrimed)
            {
                StreamUtility.SortStream(this.Target, this.Streams);
            }

#if POLYGON_TEST
            foreach (shapeType shape in this.Target.shape_array.Enumerate())
            {
                vertexType vertex = this.Target.vertex_array.vertex[shape.shape_info.vertex_index];
                int[] stream = Streams[shape.mesh_array.mesh[0].stream_index].IntData.ToArray();
                for (int i = 0; i < shape.mesh_array.mesh[0].count / 3; ++i)
                {
                    // 存在しない頂点を参照しているポリゴンがある
                    if (stream[i * 3] >= vertex.vtx_attrib_array.vtx_attrib[0].count ||
                        stream[i * 3 + 1] >= vertex.vtx_attrib_array.vtx_attrib[0].count ||
                        stream[i * 3 + 2] >= vertex.vtx_attrib_array.vtx_attrib[0].count)
                    {
                        Debug.WriteLine("error vertex: unknown vertex");
                    }
                    // 同じ頂点を参照するポリゴンがある
                    if (stream[i * 3] == stream[i * 3 + 1] ||
                        stream[i * 3] == stream[i * 3 + 2] ||
                        stream[i * 3 + 1] == stream[i * 3 + 2])
                    {
                        Debug.WriteLine("error vertex: same ref vertex");
                    }
                    // 同一頂点が存在している
                    for (int j = i + 1; j < shape.mesh_array.mesh[0].count / 3; ++j)
                    {
                        if (stream[i * 3] == stream[j * 3] &&
                            stream[i * 3 + 1] == stream[j * 3 + 1] &&
                            stream[i * 3 + 2] == stream[j * 3 + 2])
                        {
                            Debug.WriteLine("error vertex: same vertex");
                        }
                    }
                }
            }
#endif
        }

        private void GatherResultCodes(IfDivideUtility.DivideMesh mesh, Dictionary<IfDivideUtility.ResultCode, uint> dic)
        {
            if (mesh.Children[0] != null)
            {
                GatherResultCodes(mesh.Children[0], dic);
            }
            if (mesh.Children[1] != null)
            {
                GatherResultCodes(mesh.Children[1], dic);
            }
            ++dic[mesh.DivideResult];
        }
    }
}
