﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデルマージコンテキスト
    public class IfModelMergeContext
    {
        // コンストラクタ
        public IfModelMergeContext(
            modelType newModel,
            List<G3dStream> newStreams,
            modelType oldModel,
            List<G3dStream> oldStreams)
        {
            this.NewModel = newModel;
            this.NewStreams = newStreams;
            this.OldModel = oldModel;
            this.OldStreams = oldStreams;
        }

        // セットアップ
        // strict が true の場合は、名前を持たない要素のノード構成が違った場合に
        // テーブルを構築しない
        // 返り値が true なら名前を持たない要素のノード構成に変更有り
        public bool Setup(bool strict)
        {
            if (this.NewModel.material_array == null ||
                this.NewModel.material_array.material == null) { return false; }
            if (this.OldModel.material_array == null ||
                this.OldModel.material_array.material == null) { return false; }

            // <material> 名前一致でマージ
            IfMergeUtility.SetupTableByName<materialType>(this.MaterialTable,
                this.NewModel.material_array.material,
                this.OldModel.material_array.material);

            // <bone> 名前一致でマージ
            IfMergeUtility.SetupTableByName<boneType>(this.BoneTable,
                this.NewModel.skeleton.bone_array.bone,
                this.OldModel.skeleton.bone_array.bone);

            // 同じボーンとマテリアルを参照するシェイプのグループを作成する
            if (this.NewModel.shape_array != null &&
                this.NewModel.shape_array.shape != null &&
                this.OldModel.shape_array != null &&
                this.OldModel.shape_array.shape != null)
            {
                foreach (shapeType newShape in this.NewModel.shape_array.shape)
                {
                    string newBoneName = newShape.shape_info.bone_name;
                    string newMateName = newShape.shape_info.mat_name;
                    IfMergeShapeGroup shapeGroup =
                        GetShapeGroup(newBoneName, newMateName);
                    shapeGroup.NewShapes.Add(newShape);
                }
                foreach (shapeType oldShape in this.OldModel.shape_array.shape)
                {
                    string oldBoneName = oldShape.shape_info.bone_name;
                    string oldMateName = oldShape.shape_info.mat_name;
                    IfMergeShapeGroup shapeGroup =
                        GetShapeGroup(oldBoneName, oldMateName);
                    shapeGroup.OldShapes.Add(oldShape);
                }
            }

            // シェイプグループを設定する
            foreach (Dictionary<string, IfMergeShapeGroup> shapeDict in this.ShapeTable.Values)
            {
                foreach (IfMergeShapeGroup shapeGroup in shapeDict.Values)
                {
                    shapeGroup.Setup();
                }
            }
            return false;
        }

        private IfMergeShapeGroup GetShapeGroup(string boneName, string mateName)
        {
            IfMergeShapeGroup shapeGroup = null;
            Dictionary<string, IfMergeShapeGroup> shapeDict;
            if (!this.ShapeTable.TryGetValue(boneName, out shapeDict))
            {
                // Dictionary<string, IfMergeShapeGroup> の新規作成
                shapeDict = new Dictionary<string, IfMergeShapeGroup>();
                this.ShapeTable.Add(boneName, shapeDict);

                shapeGroup = new IfMergeShapeGroup();
                shapeDict.Add(mateName, shapeGroup);
            }
            else
            {
                if (!shapeDict.TryGetValue(mateName, out shapeGroup))
                {
                    // IfMergeShapeGroup の新規作成
                    shapeGroup = new IfMergeShapeGroup();
                    shapeDict.Add(mateName, shapeGroup);
                }
            }
            return shapeGroup;
        }

        //=====================================================================
        // 再出力した新しいモデル、マージ先
        public readonly modelType NewModel;
        // 再出力した新しいストリーム、マージ先
        public readonly List<G3dStream> NewStreams;
        // 編集済みの古いモデル、マージ元
        public readonly modelType OldModel;
        // 編集済みの古いストリーム、マージ元
        public readonly List<G3dStream> OldStreams;

        // マージテーブル
        public readonly Dictionary<materialType, materialType> MaterialTable =
            new Dictionary<materialType, materialType>();
        public readonly Dictionary<boneType, boneType> BoneTable =
            new Dictionary<boneType, boneType>();
        // 参照するボーンの名前とマテリアルの名前をキーにし、
        // それらのボーンとマテリアルを参照するシェープのリストを値としたテーブルです。
        public readonly Dictionary<string, Dictionary<string, IfMergeShapeGroup>> ShapeTable =
            new Dictionary<string, Dictionary<string, IfMergeShapeGroup>>();
    }

    // マージするシェイプのグループ
    public class IfMergeShapeGroup
    {
        public readonly List<shapeType> NewShapes =
            new List<shapeType>();
        public readonly List<shapeType> OldShapes =
            new List<shapeType>();
        public readonly Dictionary<shapeType, shapeType> ShapeTable =
            new Dictionary<shapeType, shapeType>();

        public void Setup()
        {
            // シェイプのテーブルを作成する
            IfMergeUtility.SetupTableByName<shapeType>(this.ShapeTable,
                this.NewShapes.ToArray(), this.OldShapes.ToArray());
        }

        public bool IsStructChanged()
        {
            if (this.NewShapes.Count != this.OldShapes.Count)
            {
                return true;
            }
            return false;
        }
    }
}
