﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using System.IO;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // テキスト中間ファイル読み込みユーティリティ
    public static class IfTextReadUtility
    {
        // テキスト中間ファイルの読み込み
        public static nw4f_3difType Read(string filePath, string xsdBasePath)
        {
            bool updated;
            return Read(filePath, xsdBasePath, out updated);
        }

        // テキスト中間ファイルの読み込み
        public static nw4f_3difType Read(
            string filePath, string xsdBasePath, out bool updated)
        {
            /// TODO: 拡張子に対する内容チェック
            Nintendo.Foundation.Contracts.Assertion.Operation.True(G3dPath.IsTextPath(filePath));
            // File.ReadAllBytes method locks the file (maybe for cache optimization)
            return Read(CustomFile.ReadAllBytes(filePath), xsdBasePath, out updated);
        }

        private static class CustomFile
        {
            public static byte[] ReadAllBytes(string filename)
            {
                byte[] buffer;

                // only FileShare changes from original File.ReadAllBytes method (Read is used in the original method)
                using (var stream = new FileStream(filename, FileMode.Open, FileAccess.Read, FileShare.ReadWrite))
                {
                    int offset = 0;
                    var length = stream.Length;

                    if (length > 0x7FFFFFFFL)
                    {
                        throw new IOException("File too long");
                    }

                    var count = (int)length;
                    buffer = new byte[count];

                    while (count > 0)
                    {
                        var num4 = stream.Read(buffer, offset, count);
                        if (num4 == 0)
                        {
                            throw new IOException("Unexpected End Of File");
                        }

                        offset += num4;
                        count -= num4;
                    }
                }

                return buffer;
            }
        }

        //---------------------------------------------------------------------
        // テキスト中間ファイルの読み込み
        public static nw4f_3difType Read(byte[] fileImage, string xsdBasePath)
        {
            bool updated;
            return Read(fileImage, xsdBasePath, out updated);
        }

        // テキスト中間ファイルの読み込み
        public static nw4f_3difType Read(
            byte[] inputFileImage, string xsdBasePath, out bool updated)
        {
            byte[] fileImage = inputFileImage;
            updated = false;
            if (xsdBasePath != null)
            {
                // バージョンアップデート
                byte[] updatedFileImage = IfVersionUpdater.Update(inputFileImage, xsdBasePath);
                if (updatedFileImage != null)
                {
                    fileImage = updatedFileImage;
                    updated = true;
                }
            }

            nw4f_3difType nw4f_3dif = null;
            XmlReaderSettings readerSettings = new XmlReaderSettings();
            readerSettings.CloseInput = true;

            // スキーマパスの指定があれば読み込む
            if (xsdBasePath != null)
            {
                string version = IfReadUtility.GetVersion(fileImage);
                XmlSchema schema = IfTextReadUtility.GetXsd(xsdBasePath, version);
                readerSettings.ValidationType = ValidationType.Schema;
                readerSettings.Schemas.Add(schema);
            }

            // 読み込み
            using (XmlReader ifReader = XmlReader.Create(
                new MemoryStream(fileImage), readerSettings))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(nw4f_3difType));
                nw4f_3dif = (nw4f_3difType)serializer.Deserialize(ifReader);
            }

            updated = updated || IfToolData.Update(nw4f_3dif);
            return nw4f_3dif;
        }

        //---------------------------------------------------------------------
        // スキーマの読み込み
        public static XmlSchema GetXsd(string xsdBasePath, string version)
        {
            // バージョン番号をパスに反映する
            string xsdPath = IfReadUtility.GetXsdPath(xsdBasePath, version);

            // スレッドセーフで無さそうなので、キャッシュしない
            using (XmlReader xsdReader = XmlReader.Create(xsdPath))
            {
                return XmlSchema.Read(xsdReader, null);
            }
        }

        internal static XmlSchema GetXsd(string xsdBasePath, string xsdFileName, string version)
        {
            // バージョン番号をパスに反映する
            string xsdPath = IfReadUtility.GetXsdPath(xsdBasePath, xsdFileName, version);

            // スレッドセーフで無さそうなので、キャッシュしない
            using (XmlReader xsdReader = XmlReader.Create(xsdPath))
            {
                return XmlSchema.Read(xsdReader, null);
            }
        }
    }
}
