﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // ボーンビジビリティアニメーションのマージボーン圧縮
    public class IfBoneVisibilityAnimBoneMergeCompressor
        : IfBoneVisibilityAnimCompressor
    {
        // コンストラクタ
        public IfBoneVisibilityAnimBoneMergeCompressor() :
            base("IfBoneVisibilityAnimBoneMergeCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_bone_merge"; }
        }

        // 圧縮
        protected override void Compress()
        {
            CompressBoneMerge(this.Target, this.Streams);

            // 不要になったストリームを削除する
            BoneVisibilityAnimCompressUtility.RemoveUselessStream(this.Target, this.Streams);
        }

        internal static void CompressBoneMerge(bone_visibility_animType target, List<G3dStream> streams)
        {
            bone_vis_bone_animType[] bone_anims =
                target.bone_vis_bone_anim_array.bone_vis_bone_anim;
            int boneAnimCount = bone_anims.Length;
            BoneVisAnimInfo[] boneAnimInfos = new BoneVisAnimInfo[boneAnimCount];
            BoneVisAnimInfo[] parent;

            // ボーンアニメーション情報の初期化
            for (int i = 0; i < boneAnimCount; i++)
            { boneAnimInfos[i] = new BoneVisAnimInfo(bone_anims[i]); }
            for (int i = 0; i < boneAnimCount; i++)
            { boneAnimInfos[i].Setup(boneAnimInfos); }

            // 削除可能なボーンの洗い出し
            int numFound;
            do
            {
                numFound = 0;
                for (int ianim = 1; ianim < boneAnimCount; ++ianim)
                {
                    BoneVisAnimInfo animInfo = boneAnimInfos[ianim];
                    if (animInfo.RemoveFlag)
                    {
                        continue;
                    }
                    if (CheckRemovable(animInfo))
                    {
                        ++numFound;
                        animInfo.RemoveFlag = true;
                    }
                }
            } while (numFound > 0);

            // 以下の条件に当てはまる場合はルートボーンを削除する。
            // - 削除可能
            // - ルートボーンの削除されない子供の数が 1 つだけ
            // ※ Merge ボーン圧縮ではボーン名のチェックを行わない。
            //    これは NW4R 版の実装に合わせるためである。
            BoneVisAnimInfo rootAnimInfo = boneAnimInfos[0];
            BoneVisibilityAnimCompressUtility.GetThinOutBoneRef(boneAnimInfos, out parent);
            BoneVisAnimInfo[] rootChild = Array.FindAll(
                parent, delegate(BoneVisAnimInfo bone) { return (bone == rootAnimInfo); });
            if (CheckRemovable(rootAnimInfo) && (rootChild.Length <= 1))
            {
                rootAnimInfo.RemoveFlag = true;
            }

            // ボーン圧縮後のルートボーンを見つける。
            if (rootAnimInfo.RemoveFlag)
            {
                int idx = Array.FindIndex(
                    parent, delegate(BoneVisAnimInfo bone) { return (bone == rootAnimInfo); });
                // 全てのノードが削除される場合は idx が -1 になる。
                // フォーマッタがボーンが存在していない場合に対応していないので、ルートノードのみ残す。
                if (idx >= 0)
                {
                    rootAnimInfo = boneAnimInfos[idx];
                }
                else
                {
                    rootAnimInfo.RemoveFlag = false;
                }
            }

            // アニメーションが付いているボーンが削除されるか確認する
            BoneVisibilityAnimCompressUtility.CheckAnimationRemoved(boneAnimInfos);
            // 新しいアニメーションのインデックスを設定する
            BoneVisibilityAnimCompressUtility.SetNewIndex(rootAnimInfo);
            // アニメーションの削除
            BoneVisibilityAnimCompressUtility.RemoveBoneAnims(target, boneAnimInfos);
        }

        private static bool CheckRemovable(BoneVisAnimInfo animInfo)
        {
            bone_vis_bone_animType anim = animInfo.BoneAnim;
            if (!anim.compress_enable)
            {
                // 圧縮禁止
                return false;
            }
            // レンダリングに使用されているボーンは圧縮しない
            if (animInfo.RenderMatrix)
            {
                return false;
            }
            return true;
        }
    }
}
