﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Runtime;
using System.Linq;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using nw.g3d.toollib;
using System.Resources;

namespace nw.g3d.filter
{
    // 中間ファイルフィルタ
    public class g3dfilter : SimpleFilterTool<g3dfilterParams>
    {
        // エントリポイント
        public static void Main(string[] args)
        {
            string path = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            ProfileOptimization.SetProfileRoot(path);
            ProfileOptimization.StartProfile(Path.GetFileNameWithoutExtension(
                Assembly.GetExecutingAssembly().Location) + ".JIT.Profile");

            var helpResourceManager = new ResourceManager("nw.g3d.filter.Resources.CommandLineHelp", typeof(g3dfilter).Assembly);
            g3dfilterParams.CustomDescriptionConverterFunc = (x, y) =>
            {
                return helpResourceManager.GetString(x);
            };

            g3dfilter g3dfilter = new g3dfilter();
            G3dToolUtility.Run(delegate { g3dfilter.Run(); });
        }

        // コンストラクタ
        public g3dfilter() : base("nw.g3d.filter.Resources.StringResource") { }

        protected override void ProcessProgramArgument(g3dfilterParams param)
        {
            base.ProcessProgramArgument(param);

            this.DisableFileInfo = param.DisableFileInfo;

            // 要素は一つ
            this.FilterScripts.Add(new IfFilterScript(param.Script, param.Argument));
        }

        // ファイルパスのチェック
        protected override bool CheckFilePath(string filePath)
        {
            // 中間ファイルかどうかをチェック
            if (!G3dPath.IsPath(filePath)) { return false; }

            return base.CheckFilePath(filePath);
        }

        //---------------------------------------------------------------------
        // 処理
        protected override void Process(ProcessFileDelegate processFile)
        {
            // フィルタターゲットの設定
            this.FilterTargets.AddRange(this.FilePaths.Select(x => new IfFilterTarget(x, this.XsdBasePath, string.Empty, this.DisableFileInfo)));

            // フィルタ
            for (int i = 0; i < this.FilterScripts.Count; i++)
            {
                this.TempScript = this.FilterScripts[i];
                this.WriteProgress("Filter", i + 1, this.FilterScripts.Count,
                    this.TempScript.ScriptPath);

                // スクリプト初期化
                Stopwatch swInitScript = new Stopwatch();
                swInitScript.Start();
                {
                    this.TempScript.Initialize();
                }
                swInitScript.Stop();
                this.WriteProgress("InitializeScript", swInitScript.ElapsedMilliseconds);

                // プレフィルタ
                if (this.TempScript.HasPreFilter)
                {
                    Stopwatch swPreFilter = new Stopwatch();
                    swPreFilter.Start();
                    {
                        if (!this.TempScript.PreFilter(this.FilterTargets))
                        {
                            Strings.Throw("Error_Filter_ScriptFailed",
                                this.TempScript.ScriptPath);
                        }
                    }
                    swPreFilter.Stop();
                    this.WriteProgress("PreFilter", swPreFilter.ElapsedMilliseconds);
                }

                // フィルタ
                base.Process(processFile);

                // ポストフィルタ
                if (this.TempScript.HasPostFilter)
                {
                    Stopwatch swPostFilter = new Stopwatch();
                    swPostFilter.Start();
                    {
                        if (!this.TempScript.PostFilter(this.FilterTargets))
                        {
                            Strings.Throw("Error_Filter_ScriptFailed",
                                this.TempScript.ScriptPath);
                        }
                    }
                    swPostFilter.Stop();
                    this.WriteProgress("PostFilter", swPostFilter.ElapsedMilliseconds);
                }
            }
        }

        // ファイル処理
        protected override void ProcessFile(int index, string filePath)
        {
            if (!this.FilterTargets[index].Filter(this.TempScript))
            {
                Strings.Throw("Error_Filter_ScriptFailed",
                    this.TempScript.ScriptPath, filePath);
            }
        }

        //---------------------------------------------------------------------
        // フィルタエンジン
        private readonly List<IfFilterScript> FilterScripts = new List<IfFilterScript>();
        // フィルタターゲット
        private readonly List<IfFilterTarget> FilterTargets = new List<IfFilterTarget>();

        private bool DisableFileInfo;
        private IfFilterScript TempScript;
    }
}
