﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime;
using System.Text;
using nw.g3d.nw4f_3dif;
using nw.g3d.toollib;
using Nintendo.Foundation.IO;

namespace nw.g3d.ifcvtr
{
    // 中間ファイルコンバータ
    public class g3difcvtr
    {
        /// <summary>
        /// テストから実行する際に使う引数です。
        /// </summary>
        public string Arguments { get; set; } = null;

        /// <summary>
        /// テストから実行する際に使う XSD パスです。
        /// </summary>
        public string XsdBasePath { get; set; } = null;

        /// <summary>
        /// テストから実行する際に使う 3dTools パスです。
        /// </summary>
        public string G3dToolRoot { get; set; } = null;

        // エントリポイント
        public static void Main()
        {
            string path = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            ProfileOptimization.SetProfileRoot(path);
            ProfileOptimization.StartProfile(Path.GetFileNameWithoutExtension(
                Assembly.GetExecutingAssembly().Location) + ".JIT.Profile");

            // エラー時にポーズしない
            G3dToolUtility.EnableErrorPause = false;
            g3difcvtr g3difcvtr = new g3difcvtr();
            G3dToolUtility.Run(g3difcvtr.Run);
        }

        // 実行
        public void Run()
        {
            // 全体計測
            Stopwatch swRun = new Stopwatch();
            swRun.Start();
            {
                // 正常終了できなければ、異常終了
                Environment.ExitCode = 1;

                // ローカライズ処理
                Strings.Initialize("nw.g3d.ifcvtr.Resources.StringResource");
                G3dLocalization.SetCulture();

                // 初期化
                Stopwatch swInitialize = new Stopwatch();
                swInitialize.Start();
                {
                    if (!Initialize())
                    {
                        Environment.ExitCode = 0;
                        return;
                    }
                }
                swInitialize.Stop();
                this.WriteProgress("Initialize", swInitialize.ElapsedMilliseconds);

                // 処理
                Process();
                Environment.ExitCode = 0;
            }
            swRun.Stop();
            this.WriteProgress("Finish", swRun.ElapsedMilliseconds);
        }

        //---------------------------------------------------------------------
        // 初期化
        private bool Initialize()
        {
            CommandLineOptions param;
            var setting = new CommandLineParserSettings()
            {
                ApplicationDescription = CommandLineOptions.DescriptionConverter("ApplicationDescription", null),
                // 例外処理に任せる
                ErrorWriter = x => { },
            };

            try
            {
                IEnumerable<string> arguments = string.IsNullOrEmpty(this.Arguments) ? Environment.GetCommandLineArgs().Skip(1) : this.Arguments.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                if (!(new CommandLineParser(setting).ParseArgs(arguments, out param)))
                {
                    return false;
                }
            }
            catch
            {
                throw;
            }

            ProcessArgument(param);

            return true;
        }

        private void ProcessArgument(CommandLineOptions param)
        {
            // 進捗表示切り替え
            this.Silent = param.Silent;

            // 並列処理数
            G3dParallel.Job = param.Job;

            if (!string.IsNullOrEmpty(param.Output) && param.Paths.Length > 1)
            {
                Strings.Throw("Error_OutputForMultipleInput");
            }

            string g3dToolRoot = !string.IsNullOrEmpty(this.G3dToolRoot) ? this.G3dToolRoot : G3dToolUtility.GetG3dToolRootPath();
            string xsdBasePath = !string.IsNullOrEmpty(this.XsdBasePath) ? this.XsdBasePath : G3dToolUtility.GetXsdBasePath();
            foreach (var path in param.Paths)
            {
                this.Converters.Add(Converter.Create(path, param, xsdBasePath, g3dToolRoot));
            }
        }

        //---------------------------------------------------------------------
        // 処理
        private void Process()
        {
            this.Process(this.Converters);
        }

        private void Process(List<Converter> converters)
        {
            G3dParallel.ForEach(converters, delegate(Converter converter)
            {
                Stopwatch swProcessFile = new Stopwatch();
                swProcessFile.Start();
                {
                    converter.Convert();
                }
                swProcessFile.Stop();
                this.WriteProgress("ProcessFile",
                    swProcessFile.ElapsedMilliseconds, converter.GetLog());
            });

            // サブコンバータの呼び出し
            List<Converter> subConverters = new List<Converter>();
            foreach (Converter converter in converters)
            {
                foreach (Converter subConverter in converter.SubConverters)
                {
                    // ソースパスが同一でないサブコンバータを収集する
                    /// TODO: ソースパスとアウトプットパスの両方を見るべきか悩む
                    int index = subConverters.FindIndex(delegate(Converter target)
                    {
                        string lhs = Path.GetFullPath(target.SourceFilePath);
                        string rhs = Path.GetFullPath(subConverter.SourceFilePath);
                        return (lhs == rhs);
                    });
                    if (index == -1) { subConverters.Add(subConverter); }
                }
            }
            if (subConverters.Count == 0) { return; }
            Process(subConverters);
        }

        //---------------------------------------------------------------------
        // 進捗の書き出し
        internal void WriteProgress(string id, params object[] args)
        {
            if (this.Silent) { return; }
            string progress = Strings.Get(id, args);
            // Console、Debug はスレッドセーフ
            Console.WriteLine(progress);
            Debug.WriteLine(progress);
        }

        private bool Silent;
        private readonly List<Converter> Converters = new List<Converter>();
    }
}
