﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.toollib;
using System.IO;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.ifcvtr
{
    // コンバータ
    internal abstract class Converter
    {
        internal static Converter Create(
            string sourcePath, CommandLineOptions param, string xsdBasePath, string g3dToolRoot)
        {
            // 入力ファイルが存在するか
            if (!File.Exists(sourcePath))
            {
                Strings.Throw("Error_InputFileNotFound", sourcePath);
            }

            string extension = Path.GetExtension(sourcePath);
            if (extension == ".obj")
            {
                // obj 形式
                return new ObjConverter(sourcePath, param, g3dToolRoot);
            }
            else if (
                (extension == ".bmp") ||
                (extension == ".jpg") ||
                (extension == ".jpeg") ||
                (extension == ".png"))
            {
                return new ImageConverter(sourcePath, param, g3dToolRoot);
            }
            else if (G3dPath.IsModelPath(sourcePath))
            {
                // モデル中間ファイル
                return new IfModelConverter(sourcePath, param, xsdBasePath);
            }
            else if (Common3dConverter.ExaminesWhetherIsSupportedExtension(extension))
            {
                // Assimp で読めるファイル形式
                return new Common3dConverter(sourcePath, param, xsdBasePath, g3dToolRoot);
            }
            else if (G3dPath.IsSkeletalAnimPath(sourcePath))
            {
                // スケルタルアニメーション中間ファイル
                return new IfSkeletalAnimationConverter(sourcePath, param, xsdBasePath);
            }
            else
            {
                // 対応していないファイル
                Strings.Throw("Error_InputFileIsNotSupported", sourcePath);
            }
            return null;
        }

        //---------------------------------------------------------------------
        // コンストラクタ
        internal Converter(string sourceFilePath)
        {
            this.SourceFilePath = sourceFilePath;
        }

        // コンストラクタ
        internal Converter(string sourceFilePath, string outputFilePath)
        {
            this.SourceFilePath = sourceFilePath;
            this.OutputFilePath = outputFilePath;
        }

        //---------------------------------------------------------------------
        // 変換
        internal abstract void Convert();

        // ログの取得
        internal virtual string GetLog()
        {
            return this.SourceFilePath;
        }

        //---------------------------------------------------------------------
        internal readonly List<Converter> SubConverters = new List<Converter>();
        internal readonly string SourceFilePath;
        internal string OutputFilePath { get; set; }
    }
}
