﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime;
using System.Text;

using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using nw.g3d.toollib;
using Nintendo.Foundation.IO;

namespace nw.g3d.ifassign
{
    // アサインユーティリティ
    public class g3difassign
    {
        /// <summary>
        /// テストから実行する際に使う引数です。
        /// </summary>
        public string Arguments { get; set; } = null;

        /// <summary>
        /// テストから実行する際に使う XSD パスです。
        /// </summary>
        public static string XsdBasePath = null;

        // エントリポイント
        public static void Main(string[] args)
        {
            string path = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            ProfileOptimization.SetProfileRoot(path);
            ProfileOptimization.StartProfile(Path.GetFileNameWithoutExtension(
                Assembly.GetExecutingAssembly().Location) + ".JIT.Profile");

            // エラー時にポーズしない
            G3dToolUtility.EnableErrorPause = false;
            g3difassign ifassign = new g3difassign();
            G3dToolUtility.Run(delegate { ifassign.Run(); });
        }

        // 実行
        public void Run()
        {
            {
                // 正常終了できなければ、異常終了
                Environment.ExitCode = 1;

                // ローカライズ処理
                Strings.Initialize("nw.g3d.ifassign.Resources.StringResource");
                G3dLocalization.SetCulture();

                // 初期化
                if (!Initialize())
                {
                    Environment.ExitCode = 0;
                }

                // 実行
                if (Jobs.Count > 0)
                {
                    ProcessJobs();
                }

                Environment.ExitCode = 0;
            }
        }

        //---------------------------------------------------------------------
        // 初期化
        private bool Initialize()
        {
            g3difassignGlobalParams globalParam;
            List<g3difassignParams> paramList;
            var setting = new CommandLineParserSettings()
            {
                ApplicationDescription = g3difassignParams.DescriptionConverter("ApplicationDescription", null),
                // 例外処理に任せる
                ErrorWriter = x => { },
            };

            try
            {
                IEnumerable<string> arguments = string.IsNullOrEmpty(this.Arguments) ?
                    arguments = Environment.GetCommandLineArgs().Skip(1) :
                    SplitArgumentText(this.Arguments);
                if (!(new CommandLineParser(setting).ParseArgsWithJobList(arguments, out globalParam, out paramList)))
                {
                    return false;
                }
            }
            catch
            {
                throw;
            }

            ProcessArgument(globalParam, paramList);

            return true;
        }

        private enum Mode
        {
            assign = 1,
            update = 2,
            search_path = 4,
        }

        /// <summary>
        /// 自動テスト用の引数をパース単位に分割する処理です。テスト時にしか利用しないので制限付きでしか解析しません。
        /// </summary>
        /// <param name="argumentText">解析する引数テキスト</param>
        /// <returns>分割された引数</returns>
        private static IEnumerable<string> SplitArgumentText(string argumentText)
        {
            List<string> tmpArguments = new List<string>();
            var spaceSplitted = argumentText.Split(' ');
            StringBuilder tmpArg = new StringBuilder();
            foreach (string arg in spaceSplitted)
            {
                if (!string.IsNullOrEmpty(tmpArg.ToString()))
                {
                    tmpArg.Append($" {arg}");
                    if (!arg.EndsWith("\\\"") && arg.EndsWith("\""))
                    {
                        tmpArguments.Add(tmpArg.ToString().Trim('\"'));
                        tmpArg.Clear();
                    }
                }
                else
                {
                    if (arg.StartsWith("\""))
                    {
                        tmpArg.Append(arg);
                    }
                    else
                    {
                        tmpArguments.Add(arg.Trim('\"'));
                    }
                }
            }
            return tmpArguments.ToArray();
        }

        private void ProcessArgument(g3difassignGlobalParams globalParams, List<g3difassignParams> paramList)
        {
            // 進捗表示切り替え
            this.Silent = globalParams.Silent;

            // 並列処理数
            if (globalParams.Job > 0)
            {
                G3dParallel.Job = globalParams.Job;
            }

            foreach (var param in paramList)
            {
                if (param.Assign != null)
                {
                    param.Assign.DisableFileInfo = param.Assign.DisableFileInfo || param.DisableFileInfo;
                    Jobs.Add(new AssignShaderUtility(param.Assign));
                }
                else if (param.UpdateModel != null)
                {
                    param.UpdateModel.DisableFileInfo = param.UpdateModel.DisableFileInfo || param.DisableFileInfo;
                    Jobs.Add(new UpdateShaderUtility(param.UpdateModel));
                }
                else if (param.UpdateAnimation != null)
                {
                    param.UpdateAnimation.DisableFileInfo = param.UpdateAnimation.DisableFileInfo || param.DisableFileInfo;
                    Jobs.Add(new UpdateShaderParamAnimationUtility(param.UpdateAnimation));
                }
                else if (param.AssignSearchPath != null)
                {
                    param.AssignSearchPath.DisableFileInfo = param.AssignSearchPath.DisableFileInfo || param.DisableFileInfo;
                    Jobs.Add(new AssignSearchPathUtility(param.AssignSearchPath));
                }
                else if (param.AssignParentMaterial != null)
                {
                    param.AssignParentMaterial.DisableFileInfo = param.AssignParentMaterial.DisableFileInfo || param.DisableFileInfo;
                    Jobs.Add(new AssignParentMaterialUtility(param.AssignParentMaterial));
                }
                else if (param.AssignMaterialReferenceBehavior != null)
                {
                    param.AssignMaterialReferenceBehavior.DisableFileInfo = param.AssignMaterialReferenceBehavior.DisableFileInfo || param.DisableFileInfo;
                    Jobs.Add(new AssignMaterialReferenceBehaviorUtility(param.AssignMaterialReferenceBehavior));
                }
                else if (param.ApplyParentMaterial != null)
                {
                    param.ApplyParentMaterial.DisableFileInfo = param.ApplyParentMaterial.DisableFileInfo || param.DisableFileInfo;
                    Jobs.Add(new ApplyParentMaterialUtility(param.ApplyParentMaterial));
                }
            }
        }

        private void ProcessJobs()
        {
            // 処理
            WriteLine(string.Format(Resources.StringResource.Start, Jobs.Count));

            var finished = new bool[Jobs.Count];
            try
            {
                G3dParallel.ForEach(Jobs.Select((job, index) => new { job, index }), x =>
                {
                    x.job.Process();
                    finished[x.index] = true;
                });
            }
            finally
            {
                foreach (var job in Jobs)
                {
                    var message = job.MessageBuilder.ToString();
                    if (message.Length > 0)
                    {
                        WriteLine(job.MessageBuilder.ToString());
                    }
                }

                WriteLine(string.Format(Resources.StringResource.Finish, finished.Count(x => x)));
            }
        }

        private List<AssignUtility> Jobs = new List<AssignUtility>();

        private bool Silent = false;

        private void WriteLine(string message)
        {
            if (!this.Silent)
            {
                Console.WriteLine(message);
                Debug.WriteLine(message);
            }
        }

        /// <summary>
        /// シェーダ定義ファイル
        /// ファイル名をキーにして検索
        /// </summary>
        private static Dictionary<string, shader_definitionType> shaderDefinitions = new Dictionary<string, shader_definitionType>();

        internal static shader_definitionType GetShaderDefinitionFromPath(string shaderPath, List<G3dStream> shaderDefinitionStreams)
        {
            shader_definitionType shaderDefinition;
            lock (shaderDefinitions)
            {
                var shaderFullPath = Path.GetFullPath(shaderPath).ToLower();
                if (shaderDefinitions.ContainsKey(shaderFullPath))
                {
                    // 既に探していれば再利用
                    shaderDefinition = shaderDefinitions[shaderFullPath];
                }
                else
                {
                    // シェーダ定義を登録
                    var nwif = IfReadUtility.Read(
                        shaderDefinitionStreams,
                        shaderFullPath,
                        XsdBasePath != null ? XsdBasePath : G3dToolUtility.GetXsdBasePath());
                    shaderDefinition = (shader_definitionType)nwif.Item;
                    shaderDefinitions.Add(shaderFullPath, shaderDefinition);
                }
            }
            return shaderDefinition;
        }
    }
}
