﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using nw.g3d.ifassign.Resources;
using nw.g3d.iflib;
using nw.g3d.iflib.nw3de;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.ifassign
{
    internal class AssignParentMaterialUtility : AssignUtility
    {
        private readonly string Input;
        private readonly string Output;
        private readonly g3difassignParams.ParentMaterialAssignOption[] ParentMaterialAssigns;
        private readonly string[] ParentMaterialPaths;


        internal AssignParentMaterialUtility(g3difassignParams.AssignParentMaterialSubCommand programOption)
            : base(programOption)
        {
            Input = programOption.Path;
            Output = programOption.Output;
            ParentMaterialAssigns = programOption.ParentMaterialAssigns;
            ParentMaterialPaths = programOption.ParentMaterialPaths?.SelectMany(x => x.Split(';')).ToArray();
        }

        internal override void Process()
        {
            var modelPath = Environment.ExpandEnvironmentVariables(Input);
            var outputPath = modelPath;

            // パスのチェック
            if (!File.Exists(modelPath))
            {
                throw new Exception(string.Format(Resources.StringResource.Error_InputFileNotFound, modelPath));
            }
            if (!G3dPath.IsModelPath(modelPath))
            {
                throw new Exception(string.Format(Resources.StringResource.Error_InputFileIsNotModel, modelPath));
            }

            // 出力パスの設定
            if (Output != null)
            {
                outputPath = Output;

                // もっと厳密にチェックもできけるけど複雑なケースもあるので簡易チェック
                string inExt = Path.GetExtension(modelPath).ToLower();
                string outExt = Path.GetExtension(outputPath).ToLower();
                if (outExt.Length == 0 ||
                    !"ab".Contains(outExt[outExt.Length - 1]) ||
                    inExt.Substring(0, inExt.Length - 1) != outExt.Substring(0, outExt.Length - 1))
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_OutputFileExt, outputPath));
                }
            }

            // ファイル読み込み
            var streams = new List<G3dStream>();
            var nwif = IfReadUtility.Read(streams, modelPath, g3difassign.XsdBasePath);
            var model = nwif.Item as modelType;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(model != null);

            var parentMaterialInfoList = new List<ParentMaterialInfo>();

            if (ParentMaterialAssigns != null)
            {
                foreach (var assign in ParentMaterialAssigns)
                {
                    var materialName = assign.Material;
                    if (string.IsNullOrEmpty(materialName))
                    {
                        var materials = model?.material_array?.material;
                        if (materials != null && materials.Any())
                        {
                            foreach (var material in materials.Where(x => string.CompareOrdinal(x.name, assign.ParentMaterial) != 0))
                            {
                                AddParentMaterial(parentMaterialInfoList, material.name, assign.ParentMaterial, assign.ParentModel);
                            }
                        }
                    }
                    else
                    {
                        AddParentMaterial(parentMaterialInfoList, materialName, assign.ParentMaterial, assign.ParentModel);
                    }
                }
            }

            if (parentMaterialInfoList.Any())
            {
                List<G3dStream> parentStreams;
                nw4f_3difType parentNwif;
                modelType parentModel;

                foreach (var parentMaterialInfo in parentMaterialInfoList)
                {
                    // 対象マテリアルのチェック
                    var material = model?.material_array?.material?.FirstOrDefault(x => x.name == parentMaterialInfo.MaterialName);
                    if (material == null)
                    {
                        throw new Exception(string.Format(StringResource.AssignBaseMaterialUtility_Process_ErrorDerivedMaterialNotExist, parentMaterialInfo.MaterialName));
                    }

                    var materialReference = IfToolData.GetMaterialReference(material);

                    // 親マテリアル指定が無いものは削除
                    parentMaterialInfo.ParentMaterials = parentMaterialInfo.ParentMaterials.Where(x => !string.IsNullOrEmpty(x.MaterialName)).ToList();
                    // 親マテリアル指定が全て空のときは親マテリアル設定を削除
                    if (parentMaterialInfo.ParentMaterials.Any())
                    {
                        foreach (var parentMaterial in parentMaterialInfo.ParentMaterials)
                        {
                            // 親モデルファイル未指定で、対象マテリアルが親マテリアルと同じでないかチェック
                            if (string.IsNullOrEmpty(parentMaterial.Path) && parentMaterialInfo.MaterialName == parentMaterial.MaterialName)
                            {
                                throw new Exception(string.Format(StringResource.AssignBaseMaterialUtility_Process_ErrorSameMaterial,
                                    parentMaterialInfo.MaterialName));
                            }


                            // 対象シェーディングモデルのチェック
                            var shaderAssign = material.shader_assign;
                            if (shaderAssign == null || string.IsNullOrEmpty(shaderAssign.shader_archive) ||
                                string.IsNullOrEmpty(shaderAssign.shading_model))
                            {
                                throw new Exception(string.Format(
                                    StringResource.AssignBaseMaterialUtility_Process_ErrorDerivedMaterialNotShaderAssigned,
                                    parentMaterialInfo.MaterialName));
                            }

                            var parentDocumentPath = modelPath;
                            parentModel = model;

                            materialType parentMaterialType = null;
                            if (!string.IsNullOrEmpty(parentMaterial.Path))
                            {
                                // 親ファイル指定時はファイルをチェック
                                parentDocumentPath =
                                    IfApplyBaseMaterialUtility.GetParentDocumentFullPath(parentMaterial.Path, modelPath, ParentMaterialPaths);

                                // パスのチェック
                                if (!File.Exists(parentDocumentPath))
                                {
                                    throw new Exception(string.Format(StringResource.AssignBaseMaterialUtility_Process_ErrorBaseMaterialFileNotFound,
                                        parentDocumentPath ?? parentMaterial.Path));
                                }
                                if (!G3dPath.IsModelPath(parentDocumentPath) && !G3dPath.IsMaterialPath(parentDocumentPath))
                                {
                                    throw new Exception(string.Format(
                                        StringResource.AssignBaseMaterialUtility_Process_ErrorBaseMaterialFileIsNotModel,
                                        parentDocumentPath ?? parentMaterial.Path));
                                }
                                parentStreams = new List<G3dStream>();
                                parentNwif = IfReadUtility.Read(parentStreams, parentDocumentPath, g3difassign.XsdBasePath);
                                switch(parentNwif.Item)
                                {
                                    case materialType materialType:
                                        // マテリアル中間ファイルではファイル名がマテリアル名の仕様。
                                        parentMaterialType =
                                            string.Equals(System.IO.Path.GetFileNameWithoutExtension(parentDocumentPath), parentMaterial.MaterialName, StringComparison.Ordinal) ?
                                            materialType :
                                            null;
                                        break;
                                    default:
                                        parentModel = (modelType)parentNwif.Item;
                                        Nintendo.Foundation.Contracts.Assertion.Operation.True(parentModel != null);
                                        parentMaterialType =
                                            parentModel?.material_array?.material?.FirstOrDefault(x => x.name == parentMaterial.MaterialName);
                                        break;
                                }
                            }
                            else
                            {
                                parentMaterialType = model.material_array?.Items?.FirstOrDefault(x => x.name == parentMaterial.MaterialName);
                            }

                            // 親マテリアルのチェック
                            if (parentMaterialType == null)
                            {
                                throw new Exception(string.Format(StringResource.AssignBaseMaterialUtility_Process_ErrorBaseMaterialNotFound,
                                    parentMaterial.MaterialName));
                            }

                            // 基底シェーディングモデルのチェック
                            var parentShaderAssign = parentMaterialType.shader_assign;
                            if (parentShaderAssign == null || string.IsNullOrEmpty(parentShaderAssign.shader_archive) ||
                                string.IsNullOrEmpty(parentShaderAssign.shading_model))
                            {
                                throw new Exception(string.Format(StringResource.AssignBaseMaterialUtility_Process_ErrorBaseMaterialNotShaderAssigned,
                                    parentMaterial.MaterialName));
                            }

                            if (shaderAssign.shader_archive != parentShaderAssign.shader_archive)
                            {
                                throw new Exception(string.Format(
                                    StringResource.AssignBaseMaterialUtility_Process_ErrorDerivedAndBaseMaterialIsNotSameShader,
                                    shaderAssign.shader_archive, parentShaderAssign.shader_archive));
                            }

                            if (shaderAssign.shading_model != parentShaderAssign.shading_model)
                            {
                                throw new Exception(string.Format(
                                    StringResource.AssignBaseMaterialUtility_Process_ErrorDerivedAndBaseMaterialIsNotSameShadingModel,
                                    shaderAssign.shading_model, parentShaderAssign.shading_model));
                            }
                        }
                    }
                    else
                    {
                        parentMaterialInfo.ParentMaterials = null;
                    }

                    materialReference.ParentMaterials = parentMaterialInfo.ParentMaterials;
                    IfToolData.SetMaterialReference(material, materialReference);
                }
            }
            else
            {
                // 既存の ParentMaterials を削除
                if (model?.material_array != null)
                {
                    foreach (var material in model?.material_array?.material)
                    {
                        var materialReference = IfToolData.GetMaterialReference(material);
                        materialReference.ParentMaterials = null;
                        IfToolData.SetMaterialReference(material, null);
                    }
                }
            }

            // サンプラーを制限
            {
                var materials = model?.material_array?.material;
                if (materials != null)
                {
                    // モデル内の複数のマテリアルを処理する可能性があるため、オリジナルのモデルをコピーしておく
                    var orgnwif = CloneNwif(nwif);
                    var orgModel = orgnwif?.Item as modelType;
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(orgModel != null);

                    IfApplyBaseMaterialUtility.Initialize();
                    foreach (var material in materials)
                    {
                        // 再帰的に親マテリアル情報を集める
                        var baseMaterialData = IfApplyBaseMaterialUtility.CollectBaseMaterialData(orgModel, material.name, modelPath, ParentMaterialPaths);
                        // サンプラー
                        samplerType[] samplers = baseMaterialData.UpdateSamplers();
                        if (samplers != null && samplers.Any())
                        {
                            if (baseMaterialData.Material.sampler_array == null)
                            {
                                baseMaterialData.Material.sampler_array = new sampler_arrayType();
                            }
                            baseMaterialData.Material.sampler_array.sampler = CloneSamplers(samplers); // UpdateSamplers で増えたサンプラを追加する必要がある

                            if (material.sampler_array == null)
                            {
                                material.sampler_array = new sampler_arrayType();
                            }
                            material.sampler_array.sampler = CloneSamplers(samplers);
                        }
                        IfApplyBaseMaterialUtility.ApplySamplerStructureRestriction(material, baseMaterialData);
                    }
                }
            }

            // 中間ファイル出力
            if (!this.DisableFileInfo) { IfFileLogUtility.SetModify(nwif); }
            IfWriteUtility.Write(nwif, streams, outputPath);
        }

        private static nw4f_3difType CloneNwif(nw4f_3difType nwif)
        {
            nw4f_3difType orgnwif;
            using (var memoryStream = new System.IO.MemoryStream())
            {
                var binaryFormatter
                    = new System.Runtime.Serialization
                        .Formatters.Binary.BinaryFormatter();
                binaryFormatter.Serialize(memoryStream, nwif);
                memoryStream.Seek(0, SeekOrigin.Begin);
                orgnwif = binaryFormatter.Deserialize(memoryStream) as nw4f_3difType;
            }
            return orgnwif;
        }

        private static samplerType[] CloneSamplers(samplerType[] samplers)
        {
            var cloneSamplers = new List<samplerType>();

            foreach (var sampler in samplers)
            {
                using (var memoryStream = new System.IO.MemoryStream())
                {
                    var binaryFormatter
                        = new System.Runtime.Serialization
                            .Formatters.Binary.BinaryFormatter();
                    binaryFormatter.Serialize(memoryStream, sampler);
                    memoryStream.Seek(0, SeekOrigin.Begin);
                    var cloneSampler = binaryFormatter.Deserialize(memoryStream) as samplerType;
                    cloneSamplers.Add(cloneSampler);
                }
            }
            return cloneSamplers.ToArray();
        }

        private static void AddParentMaterial(
            List<ParentMaterialInfo> parentMaterialInfoList,
            string materialName,
            string parentMaterial,
            string parentPath)
        {
            var info = parentMaterialInfoList.FirstOrDefault(x => x.MaterialName == materialName);
            if (info == null)
            {
                info = new ParentMaterialInfo { MaterialName = materialName };
                parentMaterialInfoList.Add(info);
            }
            if (info.ParentMaterials == null)
            {
                info.ParentMaterials = new List<ParentMaterial>();
            }

            // マテリアル名にマテリアル中間ファイルが指定されていればそれを使う。
            if (!string.IsNullOrEmpty(parentMaterial) && G3dPath.IsMaterialPath(parentMaterial))
            {
                // マテリアル中間ファイルではファイル名がマテリアル名前となる。
                parentPath = parentMaterial;
                parentMaterial = System.IO.Path.GetFileNameWithoutExtension(parentMaterial);
            }

            info.ParentMaterials.Add(new ParentMaterial() { MaterialName = parentMaterial, Path = parentPath });
        }
    }
}
