﻿using System.Collections.Generic;
using System.Linq;
using nw.g3d.iflib;
using nw.g3d.iflib.nw3de;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.ifassign
{
    using System;

    using nw.g3d.ifassign.Resources;

    internal class AssignMaterialReferenceBehaviorUtility : AssignUtility
    {
        private readonly string Input;
        private readonly string shaderDefFolderPath;
        private readonly string Output;
        private readonly g3difassignParams.MaterialReferenceBehaviorAssignOption[] referenceBehaviorAssigns;
        private readonly g3difassignParams.MaterialReferenceSamplerBehaviorAssignOption[] referenceSamplerBehaviorAssigns;

        private static readonly string[] SamplerParamIDs = {
            "tex_name",
            "wrap_u",
            "wrap_v",
            "wrap_w",
            "filter_mag",
            "filter_min",
            "filter_mip",
            "filter_max_aniso",
            "lod_min",
            "lod_max",
            "lod_bias",
        };

        public AssignMaterialReferenceBehaviorUtility(g3difassignParams.MaterialReferenceBehaviorAssignSubCommand programOption)
            : base(programOption)
        {
            Input = programOption.Path;
            shaderDefFolderPath = programOption.ShaderPath;
            Output = programOption.Output;
            referenceBehaviorAssigns = programOption.MaterialReferenceBehaviorAssigns ?? new g3difassignParams.MaterialReferenceBehaviorAssignOption[0];
            referenceSamplerBehaviorAssigns = programOption.MaterialReferenceSamplerBehaviorAssigns ??
                                              new g3difassignParams.MaterialReferenceSamplerBehaviorAssignOption[0];




            // サンプラー名が指定されたとき、Type にSampler以外が指定されたときはエラー
            foreach (var option in referenceBehaviorAssigns)
            {
                // Type が Sampler または Nullでサンプラー名が指定されているときはサンプラー設定モード
                if ((option.Type == g3difassignParams.ShaderAssignType.Sampler) ||
                    (option.Type == null && !string.IsNullOrEmpty(option.Sampler)))
                {
                    // サンプラーパラメータのidが不正な値の場合はエラー
                    if (!string.IsNullOrEmpty(option.Id))
                    {
                        if (SamplerParamIDs.All(x => x != option.Id))
                        {
                            throw new Exception(string.Format(StringResource.Error_InvalidSamplerParamId, option.Id));
                        }
                    }
                    continue;
                }

                // サンプラー以外のTypeで、サンプラー関係のオプションは不正
                if (!string.IsNullOrEmpty(option.Sampler))
                {
                    throw new Exception(StringResource.Error_InvalidTypeWithSamplerName);
                }
            }
        }

        private IEnumerable<string> FindUsedShaderArchiveNames(modelType model)
        {
            var result = new List<string>();
            if (model.material_array == null)
            {
                return result;
            }

            foreach (var material in model.material_array.Items)
            {
                if (material.shader_assign != null)
                {
                    if (!result.Contains(material.shader_assign.shader_archive))
                    {
                        result.Add(material.shader_assign.shader_archive);
                    }
                }
            }

            return result.Distinct();
        }

        internal override void Process()
        {
            string modelPath;
            string outputPath;
            ApplyParentMaterialUtility.ModelInOutPath(Input, Output, out modelPath, out outputPath);

            // ファイル読み込み
            var streams = new List<G3dStream>();
            var nwif = IfReadUtility.Read(streams, modelPath, g3difassign.XsdBasePath);
            var model = (modelType)nwif.Item;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(model != null);

            var shaderDefs = new List<ShaderDefinitionFile>();
            if (!string.IsNullOrEmpty(this.shaderDefFolderPath))
            {
                // シェーダー定義の読み込み
                var shaderArchiveNames = FindUsedShaderArchiveNames(model);
                var fsdPaths = new List<string>();
                var shaderDefFolderFullPath = System.IO.Path.GetFullPath(Environment.ExpandEnvironmentVariables(this.shaderDefFolderPath));
                fsdPaths.AddRange(System.IO.Directory.EnumerateFiles(shaderDefFolderFullPath, "*.fsdb", System.IO.SearchOption.AllDirectories));
                fsdPaths.AddRange(System.IO.Directory.EnumerateFiles(shaderDefFolderFullPath, "*.fsda", System.IO.SearchOption.AllDirectories));
                foreach (var shaderArchiveName in shaderArchiveNames)
                {
                    var fsdPath = fsdPaths.FirstOrDefault(x => System.IO.Path.GetFileNameWithoutExtension(x) == shaderArchiveName);
                    if (string.IsNullOrEmpty(fsdPath))
                    {
                        continue;
                    }

                    var shaderDefFile = IfReadUtility.Read(fsdPath, g3difassign.XsdBasePath);
                    var shaderDef = shaderDefFile.Item as shader_definitionType;
                    Nintendo.Foundation.Contracts.Assertion.Operation.NotNull(shaderDef);
                    shaderDefs.Add(new ShaderDefinitionFile() { Name = shaderArchiveName, ShaderDefinition = shaderDef });
                }
            }

            // サンプラ設定
            foreach (var param in referenceSamplerBehaviorAssigns)
            {
                var materials = new List<materialType>();
                if (param.Material == null)
                {
                    materials.AddRange(model?.material_array?.Items ?? new materialType[0]);
                }
                else
                {
                    var material = model?.material_array?.material?.FirstOrDefault(x => x.name == param.Material);
                    if (material == null)
                    {
                        throw new Exception(string.Format(StringResource.AssignMaterialReferenceBehaviorUtility_Process_MaterialNotExist, param.Material));
                    }
                    materials.Add(material);
                }
                AddOrReplaceMaterialReferenceSamplerBehavior(materials, param);
            }

            // 参照設定
            foreach (var param in referenceBehaviorAssigns)
            {
                if (string.IsNullOrEmpty(param.Material))
                {
                    // 指定がない場合は全マテリアル対象
                    if (model?.material_array?.Items?.Any() == true )
                    {
                        AddOrReplaceMaterialReferenceBehavior(model.material_array.Items, param, shaderDefs);
                    }
                }
                else
                {
                    var material = model?.material_array?.material?.FirstOrDefault(x => x.name == param.Material);
                    if (material == null)
                    {
                        throw new Exception(string.Format(StringResource.AssignMaterialReferenceBehaviorUtility_Process_MaterialNotExist, param.Material));
                    }
                    AddOrReplaceMaterialReferenceBehavior(
                        new[] { material }, param, shaderDefs);
                }
            }

            // シェーダーが指定されているときは、存在しないパラメータに対するマテリアル参照情報を整理する
            if (shaderDefs.Any() && model?.material_array?.Items?.Any() == true)
            {
                foreach (var material in model.material_array.Items)
                {
                    var shaderAssign = material.shader_assign;
                    var shaderDef = shaderDefs.FirstOrDefault(x => x.Name == shaderAssign.shader_archive);
                    var shadingModel = shaderDef?.ShaderDefinition.shading_model_array.shading_model.FirstOrDefault(x => x.name == shaderAssign.shading_model);

                    if (shadingModel != null)
                    {
                        var materialReference = IfToolData.GetMaterialReference(material);
                        IfApplyBaseMaterialUtility.UpdateMaterialReference(materialReference, shadingModel);
                        IfToolData.SetMaterialReference(material, materialReference);
                    }
                }
            }



            // 中間ファイル出力
            if (!this.DisableFileInfo) { IfFileLogUtility.SetModify(nwif); }
            IfWriteUtility.Write(nwif, streams, outputPath);
        }

        private static void AddOrReplaceMaterialReferenceSamplerBehavior(
            IEnumerable<materialType> materials,
            g3difassignParams.MaterialReferenceSamplerBehaviorAssignOption option)
        {
            foreach (var material in materials)
            {
                var materialReference = IfToolData.GetMaterialReference(material);
                var samplers = GetSamplers(material, option.Sampler);

                if (option.SamplerStructureRestriction.HasValue)
                {
                    materialReference.SamplerBehaviors.ChildSamplerStructureRestriction = option.SamplerStructureRestriction.Value;
                }

                foreach (var sampler in samplers)
                {
                    var samplerBehaviorItem = GetSamplerBehaviorItem(materialReference, sampler);
                    if (option.IsSamplerRequiredForChild.HasValue)
                    {
                        samplerBehaviorItem.IsRequiredForChild = option.IsSamplerRequiredForChild.Value;
                    }
                }
                IfToolData.SetMaterialReference(material, materialReference);
            }
        }

        private class ShaderDefinitionFile
        {
            public string Name { get; set; }
            public shader_definitionType ShaderDefinition { get; set; }
        }

        private static void AddOrReplaceMaterialReferenceBehavior(
            IEnumerable<materialType> targetMaterials,
            g3difassignParams.MaterialReferenceBehaviorAssignOption option,
            IEnumerable<ShaderDefinitionFile> shaderDefs)
        {
            foreach (var material in targetMaterials)
            {
                var materialReference = IfToolData.GetMaterialReference(material);

                var id = option.Id;
                var type = option.Type;
                var shaderAssign = material.shader_assign;
                var allId = string.IsNullOrEmpty(id);

                ShaderDefinitionFile shaderDef = shaderDefs.FirstOrDefault(x => x.Name == shaderAssign.shader_archive);
                shading_modelType shadingModel = shaderDef?.ShaderDefinition.shading_model_array.shading_model.FirstOrDefault(x => x.name == shaderAssign.shading_model);

                // サンプラーに対する設定
                if (type == g3difassignParams.ShaderAssignType.Sampler || !string.IsNullOrEmpty(option.Sampler))
                {
                    var samplerName = option.Sampler;
                    var samplers = GetSamplers(material, samplerName);
                    if (samplers.Any())
                    {
                        foreach (var sampler in samplers)
                        {
                            option.Sampler = sampler;
                            foreach (var paramId in SamplerParamIDs.Where(x => allId || x == id))
                            {
                                AddOrReplaceMaterialReferenceBehavior(materialReference, g3difassignParams.ShaderAssignType.Sampler, option, paramId);
                            }
                        }
                    }
                    option.Sampler = samplerName;
                    IfToolData.SetMaterialReference(material, materialReference);
                    continue;
                }

                // サンプラ以外の特定のタイプとIDに対する設定
                if (type != null && !string.IsNullOrEmpty(id))
                {
                    AddOrReplaceMaterialReferenceBehavior(materialReference, type.Value, option);
                    IfToolData.SetMaterialReference(material, materialReference);
                    continue;
                }

                if (type == null || type.Value == g3difassignParams.ShaderAssignType.AttribAssign)
                {
                    List<string> targetIds = new List<string>();
                    if (shadingModel != null)
                    {
                        var vars = shadingModel.attrib_var_array.Items;
                        targetIds.AddRange(vars.Where(x => allId || x.id == id).Select(x => x.id));
                    }
                    else
                    {
                        var assigns = shaderAssign?.attrib_assign_array?.attrib_assign ?? new attrib_assignType[0];
                        targetIds.AddRange(assigns.Where(x => allId || x.id == id).Select(x => x.id));
                    }

                    AddOrReplaceMaterialReferenceBehavior(materialReference, g3difassignParams.ShaderAssignType.AttribAssign, option, targetIds);
                }

                if (type == null || type.Value == g3difassignParams.ShaderAssignType.RenderInfo)
                {
                    List<string> targetIds = new List<string>();
                    if (shadingModel != null)
                    {
                        var vars = shadingModel.render_info_slot_array.Items;
                        targetIds.AddRange(vars.Where(x => allId || x.name == id).Select(x => x.name));
                    }
                    else
                    {
                        var assigns = shaderAssign?.render_info_array?.render_info ?? new render_infoType[0];
                        targetIds.AddRange(assigns.Where(x => allId || x.name == id).Select(x => x.name));
                    }

                    AddOrReplaceMaterialReferenceBehavior(materialReference, g3difassignParams.ShaderAssignType.RenderInfo, option, targetIds);
                }

                if (type == null || type.Value == g3difassignParams.ShaderAssignType.SamplerAssign)
                {
                    List<string> targetIds = new List<string>();
                    if (shadingModel != null)
                    {
                        var vars = shadingModel.sampler_var_array.Items;
                        targetIds.AddRange(vars.Where(x => allId || x.id == id).Select(x => x.id));
                    }
                    else
                    {
                        var assigns = shaderAssign?.sampler_assign_array?.sampler_assign ?? new sampler_assignType[0];
                        targetIds.AddRange(assigns.Where(x => allId || x.id == id).Select(x => x.id));
                    }

                    AddOrReplaceMaterialReferenceBehavior(materialReference, g3difassignParams.ShaderAssignType.SamplerAssign, option, targetIds);
                }

                if (type == null || type.Value == g3difassignParams.ShaderAssignType.ShaderOption)
                {
                    List<string> targetIds = new List<string>();
                    if (shadingModel != null)
                    {
                        var vars = shadingModel.option_var_array.Items;
                        targetIds.AddRange(vars.Where(x => allId || x.id == id).Select(x => x.id));
                    }
                    else
                    {
                        var assigns = shaderAssign?.shader_option_array?.shader_option ?? new shader_optionType[0];
                        targetIds.AddRange(assigns.Where(x => allId || x.id == id).Select(x => x.id));
                    }

                    AddOrReplaceMaterialReferenceBehavior(materialReference, g3difassignParams.ShaderAssignType.ShaderOption, option, targetIds);
                }

                if (type == null || type.Value == g3difassignParams.ShaderAssignType.ShaderParam)
                {
                    List<string> targetIds = new List<string>();
                    if (shadingModel != null)
                    {
                        var vars = shadingModel.block_var_array.Items;
                        targetIds.AddRange(vars.Where(x => allId || x.id == id).Select(x => x.id));
                    }
                    else
                    {
                        var assigns = shaderAssign?.shader_param_array?.shader_param ?? new shader_paramType[0];
                        targetIds.AddRange(assigns.Where(x => allId || x.id == id).Select(x => x.id));
                    }

                    AddOrReplaceMaterialReferenceBehavior(materialReference, g3difassignParams.ShaderAssignType.ShaderParam, option, targetIds);
                }
                IfToolData.SetMaterialReference(material, materialReference);
            }
        }

        private static List<string> GetSamplers(materialType material, string samplerName)
        {
            var samplers = new List<string>();
            if (!string.IsNullOrEmpty(samplerName))
            {
                samplers.Add(samplerName);
            }
            else
            {
                samplers.AddRange(material.sampler_array?.sampler?.Select(x => x.name) ?? new string[0]);
            }
            return samplers;
        }

        private static MaterialReferenceBehavior GetMaterialReferenceBehavior(nw3de_MaterialReferenceBehavior nw3DeMaterialReferenceBehavior,
            string materialName)
        {
            var info = nw3DeMaterialReferenceBehavior.MaterialReferenceBehaviors.FirstOrDefault(
                x => x.MaterialName == materialName);
            if (info == null)
            {
                info = new MaterialReferenceBehavior() {MaterialName = materialName};
                nw3DeMaterialReferenceBehavior.MaterialReferenceBehaviors.Add(info);
            }
            return info;
        }

        private static void AddOrReplaceMaterialReferenceBehavior(
            nw3de_MaterialReference materialReference, g3difassignParams.ShaderAssignType assignType, g3difassignParams.MaterialReferenceBehaviorAssignOption option,
            IEnumerable<string> targetIds)
        {
            foreach (string targetId in targetIds)
            {
                AddOrReplaceMaterialReferenceBehavior(materialReference, assignType, option, targetId);
            }
        }

        private static void AddOrReplaceMaterialReferenceBehavior(nw3de_MaterialReference materialReference, g3difassignParams.ShaderAssignType assignType, g3difassignParams.MaterialReferenceBehaviorAssignOption option, string id = null)
        {
            switch (assignType)
            {
                case g3difassignParams.ShaderAssignType.AttribAssign:
                    AddOrReplaceBehaviorItem(materialReference.AttribAssignBehaviors, option, id);
                    break;
                case g3difassignParams.ShaderAssignType.RenderInfo:
                    AddOrReplaceBehaviorItem(materialReference.RenderInfoBehaviors, option, id);
                    break;
                case g3difassignParams.ShaderAssignType.SamplerAssign:
                    AddOrReplaceBehaviorItem(materialReference.SamplerAssignBehaviors, option, id);
                    break;
                case g3difassignParams.ShaderAssignType.ShaderOption:
                    AddOrReplaceBehaviorItem(materialReference.ShaderOptionBehaviors, option, id);
                    break;
                case g3difassignParams.ShaderAssignType.ShaderParam:
                    AddOrReplaceBehaviorItem(materialReference.ShaderParamBehaviors, option, id);
                    break;
                case g3difassignParams.ShaderAssignType.Sampler:
                    var samplerBehaviorItem = GetSamplerBehaviorItem(materialReference, option.Sampler);
                    AddOrReplaceBehaviorItem(samplerBehaviorItem.SamplerParamBehaviors, option, id);
                    break;
                default:
                    throw new ArgumentOutOfRangeException(nameof(assignType), assignType, null);
            }
        }

        private static SamplerBehaviorItem GetSamplerBehaviorItem(nw3de_MaterialReference materialReference, string samplerName)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.False(string.IsNullOrEmpty(samplerName));
            if (materialReference.SamplerBehaviors == null)
            {
                materialReference.SamplerBehaviors = new MaterialReferenceSamplerBehaviors() { };
            }

            if (materialReference.SamplerBehaviors.SamplerBehaviorItems == null)
            {
                materialReference.SamplerBehaviors.SamplerBehaviorItems = new List<SamplerBehaviorItem>();
            }

            var samplerBehaviorItem = materialReference.SamplerBehaviors.SamplerBehaviorItems.FirstOrDefault(x => x.Id == samplerName);
            if (samplerBehaviorItem == null)
            {
                samplerBehaviorItem = new SamplerBehaviorItem() { Id = samplerName };
                materialReference.SamplerBehaviors.SamplerBehaviorItems.Add(samplerBehaviorItem);
            }

            if (samplerBehaviorItem.SamplerParamBehaviors == null)
            {
                samplerBehaviorItem.SamplerParamBehaviors = new List<MaterialReferenceBehaviorItem>();
            }

            return samplerBehaviorItem;
        }

        private static void AddOrReplaceBehaviorItem(List<MaterialReferenceBehaviorItem> behaviorItems, g3difassignParams.MaterialReferenceBehaviorAssignOption param, string id = null)
        {
            var newItem = new MaterialReferenceBehaviorItem()
            {
                Id = id ?? param.Id,
            };

            Nintendo.Foundation.Contracts.Assertion.Operation.NotNull(newItem.Id);

            var oldItem = behaviorItems.FirstOrDefault(x => x.Id.Equals(newItem.Id));
            if (oldItem != null)
            {
                if (param.IsUnassignedParamOnly)
                {
                    return;
                }

                behaviorItems.Remove(oldItem);

                // 指定がないものは既存の設定を維持する
                newItem.Value = param.Value ?? oldItem.Value;
                newItem.ChildRestriction = param.ChildRestriction ?? oldItem.ChildRestriction;
            }
            else
            {
                // 指定がないものはデフォルト設定
                newItem.Value = param.Value ?? ShaderItemValueState.Refer;
                newItem.ChildRestriction = param.ChildRestriction ?? ShaderItemRestrictionState.None;
            }

            behaviorItems.Add(newItem);
        }
    }
}
