﻿using G3dCombinerShaderConverter.Resources;
using Nintendo.G3dTool.Entities;
using Nintendo.ToolFoundation.CommandLine;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System;
using System.Linq;
using System.Collections.Generic;

namespace G3dCombinerShaderConverter
{
    public class CombinerShaderConverter
    {
        public CombinerShaderConverter(string[] args)
        {
            // 引数パース
            var rootCommand = ActionCommand.CreateRootCommand(true);
            rootCommand.AddFlagOption('s', "silent", () => this.IsSilent = true)
                .GetBuilder()
                .SetDescription(CommandLineHelp.silent);

            rootCommand
                .GetBuilder()
                .SetDescription("コンバイナー情報を埋め込みます。");

            rootCommand.AddValue(1, inputPath =>
            {
                string path = Environment.ExpandEnvironmentVariables(inputPath);
                if (!System.IO.File.Exists(path))
                {
                    throw new Exception($"指定されたファイルが存在しません\n\"{path}\"");
                }

                string ext = System.IO.Path.GetExtension(path);
                if (ext.Substring(1, 3) != "fsd")
                {
                    throw new Exception($"入力ファイルがシェーダー定義ファイルではありません\n\"{path}\"");
                }
                this.InputFilePath = path;
            })
                .GetBuilder()
                .Require()
                .SetDescription("入力となるシェーダー定義ファイルを指定します。")
                .SetValueName("path");

            rootCommand.AddValueOption('o', "output", inputPath =>
            {
                string path = Environment.ExpandEnvironmentVariables(inputPath);
                if (!System.IO.File.Exists(path))
                {
                    throw new Exception($"指定されたファイルが存在しません\n\"{path}\"");
                }

                string ext = System.IO.Path.GetExtension(path);
                if (ext.Substring(1, 3) != "fsd")
                {
                    throw new Exception($"出力ファイルがシェーダー定義ファイルではありません\n\"{path}\"");
                }

                this.OutputFilePath = path;
            })
                .GetBuilder()
                .SetDescription("出力先のシェーダー定義ファイルパスを指定します。指定しない場合は入力ファイルを上書きします。")
                .SetValueName("path");

            rootCommand.AddValueOption("combiner-shader-path", inputPath =>
            {
                this.CombinerShaderFilePaths.AddRange(Utility.ExpandFilePathPatterns(inputPath, "*.fcmb"));
            })
                .GetBuilder()
                .SetDescription("ソースコードに埋め込むコンバイナーシェーダーソースコードのバリエーションとなる .fcmb ファイル、もしくはファイルが格納されているフォルダーを指定します。セミコロン区切りで複数パスを入力することができます。")
                .SetValueName("paths");

            CommandLine.ParseArgs(args, rootCommand, new ParseSettings()
            {
                ErrorAction = message =>
                {
                    throw new Exception($"{message}\n指定された引数: {string.Join(" ", args)}\n{CommandLine.GetHelpText(args, rootCommand)}");
                },
                HelpWriter = this.WriteMessage
            });
        }

        /// <summary>
        /// 探索パスと相対ファイルパスを組み合わせに対して、存在するパスを列挙します。
        /// </summary>
        /// <param name="searchFolders">探索パス</param>
        /// <param name="relativeFilePaths">探索パスからの相対ファイルパス</param>
        /// <returns></returns>
        private IEnumerable<string> CreateExistingFilePathList(IEnumerable<string> searchFolders, IEnumerable<string> relativeFilePaths)
        {
            foreach (string relativePath in relativeFilePaths)
            {
                foreach (string includeFolderPath in searchFolders)
                {
                    string path = System.IO.Path.Combine(includeFolderPath, relativePath);
                    if (System.IO.File.Exists(path))
                    {
                        yield return path;
                        break;
                    }
                }
            }
        }

        public void Execute()
        {
            // TODO: コンバイナーエディターの dll 版が使えるようになったら環境変数指定を消す
            CombinerUtility.CombinerEditorPath = Environment.GetEnvironmentVariable("G3D_COMBINER_EDITOR_PATH");

            ExecuteEmbeddingToShaderDefinition(
                this.InputFilePath,
                string.IsNullOrEmpty(this.OutputFilePath) ? this.InputFilePath : this.OutputFilePath,
                this.CombinerShaderFilePaths);
        }

        public void ExecuteEmbeddingToShaderDefinition(
            string inputFsdPath,
            string outputFsdPath,
            List<string> fcmbFilePaths)
        {
            IntermediateFile file = IfReadUtility.ReadIntermediateFile(inputFsdPath, Utility.FindXsdPath());
            ShaderDefinition shaderDef = file.GetRootEntity<ShaderDefinition>();
            Nintendo.ToolFoundation.Contracts.Ensure.Operation.NotNull(shaderDef);

            CombinerUtility.EmbedCombinerOptionInfoIntoShaderDefition(
                ref shaderDef,
                fcmbFilePaths);

            if (fcmbFilePaths.Count > 0)
            {
                CombinerUtility.EmbedCombinerSourceIntoShaderDefinition(
                    ref shaderDef,
                    fcmbFilePaths,
                    WriteMessageLine, WriteWarningMessageLine, WriteErrorMessageLine);

                CombinerUtility.ConvertCombinerEnumChoiceToHashValue(ref shaderDef);
            }

            IfWriteUtility.WriteIntermediateFile(file, outputFsdPath, Utility.FindXsdPath());
        }

        private bool IsSilent { get; set; } = false;

        private string InputFilePath { get; set; } = string.Empty;

        private string OutputFilePath { get; set; } = string.Empty;

        private List<string> CombinerShaderFilePaths { get; } = new List<string>();

        private void WriteMessageLine(string message)
        {
            IoUtility.WriteMessage(message + Environment.NewLine);
        }

        private void WriteMessage(string message)
        {
            if (this.IsSilent)
            {
                return;
            }

            IoUtility.WriteMessage(message);
        }

        private void WriteErrorMessageLine(string message)
        {
            IoUtility.WriteErrorMessageLine($"エラー: {message}");
        }

        private void WriteWarningMessageLine(string message)
        {
            IoUtility.WriteErrorMessageLine($"警告: {message}");
        }
    }
}
