﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RXMLElement RMtx44
// ParseConfigFile
// RFileBuf RFileMove

// RDataStream RUserData
// RTransformNode RBone RSkeleton

//=============================================================================
// include
//=============================================================================
#include "DccCommon.h"
#include "DccOutput.h"
#include "DccImage.h"
#pragma comment(lib, "shlwapi.lib")

using namespace std;

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//=============================================================================
// constants
//=============================================================================

// 選択されたノードを中間ファイル名する場合の特別な中間ファイル名です。
const char* RExpOpt::SpecialOutFileNameForNode = "@node";

// テクスチャを出力するフォルダの名前です。
const char* RExpOpt::TexFolderName = "textures";

// Maya 上でルートボーンが複数ある場合にグループ化するルートボーンの名前です。
const char* RBone::DefaultRootName = "nw4f_root";

//-----------------------------------------------------------------------------
// vector
const RVec2 RVec2::kZero(0.0f, 0.0f);
const RVec2 RVec2::kOne(1.0f, 1.0f);

const RVec3 RVec3::kZero(0.0f, 0.0f, 0.0f);
const RVec3 RVec3::kOne(1.0f, 1.0f, 1.0f);
const RVec3 RVec3::kXAxis(1.0f, 0.0f, 0.0f);
const RVec3 RVec3::kYAxis(0.0f, 1.0f, 0.0f);
const RVec3 RVec3::kZAxis(0.0f, 0.0f, 1.0f);
const RVec3 RVec3::kXNegAxis(-1.0f,  0.0f,  0.0f);
const RVec3 RVec3::kYNegAxis( 0.0f, -1.0f,  0.0f);
const RVec3 RVec3::kZNegAxis( 0.0f,  0.0f, -1.0f);

const RVec4 RVec4::kZero(0.0f, 0.0f, 0.0f, 0.0f);
const RVec4 RVec4::kOne(1.0f, 1.0f, 1.0f, 1.0f);
const RVec4 RVec4::kBlack(0.0f, 0.0f, 0.0f, 1.0f);
const RVec4 RVec4::kWhite(1.0f, 1.0f, 1.0f, 1.0f);

//-----------------------------------------------------------------------------
// rgba color
const RRgbaColor RRgbaColor::kBlack(0, 0, 0, RRgbaColor::COL_MAX);
const RRgbaColor RRgbaColor::kWhite(RRgbaColor::COL_MAX, RRgbaColor::COL_MAX, RRgbaColor::COL_MAX, RRgbaColor::COL_MAX);

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

const char* IntermediateFileVersion          = "4.0.0"; //!< <nw4f_3dif> 要素のバージョンです。

const char* ModelElementVersion              = "4.0.0"; //!< <model> 要素のバージョンです。
const char* SkeletalAnimElementVersion       = "4.0.0"; //!< <skeletal_anim> 要素のバージョンです。
const char* BoneVisibilityAnimElementVersion = "4.0.0"; //!< <bone_visibility_anim> 要素のバージョンです。
const char* MaterialAnimElementVersion       = "4.0.0"; //!< <material_anim> 要素のバージョンです。
const char* ShaderParamAnimElementVersion    = "4.0.0"; //!< <shader_param_anim> 要素のバージョンです。
const char* TexPatternAnimElementVersion     = "4.0.0"; //!< <tex_pattern_anim> 要素のバージョンです。
const char* ShapeAnimElementVersion          = "4.0.0"; //!< <shape_anim> 要素のバージョンです。
const char* SceneAnimElementVersion          = "4.0.0"; //!< <scene_anim> 要素のバージョンです。

const char* const FileTypeExtStrs[] = //!< 中間ファイルタイプに対応する拡張子文字列配列です。
{
    "fmd",
    "fsk",
    "fvb",
    "fcl",
    "fts",
    "ftp",
    "fsh",

    "fsn",

    "ftx",
};

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 2 のべき乗の数なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPower2Number(const int number, const bool includeOne)
{
    if (includeOne && number == 1)
    {
        return true;
    }

    int cur = 1;
    do
    {
        cur *= 2;
        if (number == cur)
        {
            return true;
        }
    } while (cur < number);

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 指定した数以上で最小の 2 のべき乗の数を返します。
//-----------------------------------------------------------------------------
int RGetPower2Number(const int number, const bool includeOne)
{
    int ret = (includeOne) ? 1 : 2;
    while (ret < number)
    {
        ret <<= 1;
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS の 1 バイト目の文字なら true を返します。
//-----------------------------------------------------------------------------
bool RIsShiftJisFirstByte(const unsigned char val)
{
    return ((0x81 <= val && val <= 0x9f) ||
            (0xe0 <= val && val <= 0xef));
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS の文字列を UTF-8 の文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetUtf8FromShiftJis(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // Shift JIS -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::string();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);

    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> UTF-8
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int utf8Size = WideCharToMultiByte(CP_UTF8, 0, wcharBuf, -1, NULL, 0, NULL, NULL);
    char* utf8Buf = new char[utf8Size];
    WideCharToMultiByte(CP_UTF8, 0, wcharBuf, -1, utf8Buf, utf8Size, NULL, NULL);
    std::string dst(utf8Buf, strlen(utf8Buf));
    // 通常 strlen(utf8Buf) = utf8Size - 1

    //cerr << "[" << src.c_str() << "] [" << dst.c_str() << "] " << src.size() << ", " << wcharSize << ", " << utf8Size << ", " << dst.size() << endl;
    //for (int i = 0; i < utf8Size; ++i) cerr << hex << static_cast<int>(static_cast<uint8_t>(utf8Buf[i])) << dec << endl;

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;
    delete[] utf8Buf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief UTF-8 の文字列を Shift-JIS の文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetShiftJisFromUtf8(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // UTF-8 -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_UTF8, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::string();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_UTF8, 0, src.c_str(), -1, wcharBuf, wcharSize);

    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> Shift JIS
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int sjisSize = WideCharToMultiByte(CP_ACP, 0, wcharBuf, -1, NULL, 0, NULL, NULL);
    char* sjisBuf = new char[sjisSize];
    WideCharToMultiByte(CP_ACP, 0, wcharBuf, -1, sjisBuf, sjisSize, NULL, NULL);
    std::string dst(sjisBuf, strlen(sjisBuf));
    // 通常 strlen(sjisBuf) = sjisSize - 1

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;
    delete[] sjisBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS の文字列をユニコードの文字列に変換して返します。
//-----------------------------------------------------------------------------
std::wstring RGetUnicodeFromShiftJis(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // Shift JIS -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::wstring();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);
    std::wstring dst(wcharBuf, wcslen(wcharBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ユニコードの文字列を Shift-JIS の文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetShiftJisFromUnicode(const std::wstring& src)
{
    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> Shift JIS
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int sjisSize = WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, NULL, 0, NULL, NULL);
    if (sjisSize == 0)
    {
        return std::string();
    }

    char* sjisBuf = new char[sjisSize];
    WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, sjisBuf, sjisSize, NULL, NULL);
    std::string dst(sjisBuf, strlen(sjisBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] sjisBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ASCII 文字列なら true を返します。
//-----------------------------------------------------------------------------
bool RIsAsciiString(const std::string& str)
{
    for (size_t ic = 0; ic < str.length(); ++ic)
    {
        const char c = str[ic];
        if (c < 0x20)
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 文字列配列を結合した 1 つの文字列を返します。
//!
//! @param[in] srcs 文字列配列です。
//! @param[in] sep 区切り文字列です。
//!
//! @return 結合した 1 つの文字列を返します。
//-----------------------------------------------------------------------------
std::string RJoinStrings(const RStringArray& srcs, const std::string& sep)
{
    std::string dst;
    for (size_t iSrc = 0; iSrc < srcs.size(); ++iSrc)
    {
        if (iSrc > 0) dst += sep;
        dst += srcs[iSrc];
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 文字列を指定した文字で分割した文字列配列を取得します。
//!
//! @param[out] dsts 文字列配列を格納します。
//! @param[in] src 文字列です。
//! @param[in] sep 区切り文字列です。
//!
//! @return 文字列配列のサイズを返します。
//-----------------------------------------------------------------------------
int RSeparateString(RStringArray& dsts, const std::string& src, const std::string& sep)
{
    dsts.clear();
    std::string cur = src;
    while (!cur.empty())
    {
        const size_t iSep = cur.find(sep);
        if (iSep == std::string::npos)
        {
            dsts.push_back(cur);
            break;
        }
        dsts.push_back(cur.substr(0, iSep));
        cur = (iSep + 1 < cur.size()) ? cur.substr(iSep + 1) : std::string();
    }
    return static_cast<int>(dsts.size());
}

//-----------------------------------------------------------------------------
//! @brief ユニコード C 配列を RStringArray に変換します。
//-----------------------------------------------------------------------------
int RGetStringArray(RStringArray& dsts, const wchar_t* srcs[])
{
    dsts.clear();
    int iStr = 0;
    while (srcs[iStr] != NULL)
    {
        dsts.push_back(RGetShiftJisFromUnicode(srcs[iStr]));
        ++iStr;
    }
    return static_cast<int>(dsts.size());
}

//-----------------------------------------------------------------------------
//! @brief 文字列を小文字に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetLowerCaseString(const std::string& src)
{
    std::string dst = src;
    int count = static_cast<int>(dst.size());
    for (int ic = 0; ic < count; ++ic)
    {
        char c = dst[ic];
        if ('A' <= c && c <= 'Z')
        {
            dst[ic] = c - 'A' + 'a';
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 文字列を大文字に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetUpperCaseString(const std::string& src)
{
    std::string dst = src;
    int count = static_cast<int>(dst.size());
    for (int ic = 0; ic < count; ++ic)
    {
        char c = dst[ic];
        if ('a' <= c && c <= 'z')
        {
            dst[ic] = c - 'a' + 'A';
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 大文字／小文字を無視して 2 つの文字列を比較し、同じなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSameStringNoCase(const std::string& str1, const std::string& str2)
{
    return (RGetLowerCaseString(str1) == RGetLowerCaseString(str2));
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimLeftString(const std::string& src, const char* targets)
{
    const size_t iLeft = src.find_first_not_of(targets);
    return (iLeft != std::string::npos) ? src.substr(iLeft) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の後の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimRightString(const std::string& src, const char* targets)
{
    const size_t iRight = src.find_last_not_of(targets);
    return (iRight != std::string::npos) ? src.substr(0, iRight + 1) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前後の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimString(const std::string& src, const char* targets)
{
    const size_t iLeft = src.find_first_not_of(targets);
    if (iLeft != std::string::npos)
    {
        const size_t iRight = src.find_last_not_of(targets);
        if (iRight != std::string::npos)
        {
            return src.substr(iLeft, iRight - iLeft + 1);
        }
    }
    return std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の最後の改行コードを削除します。
//-----------------------------------------------------------------------------
std::string RCutEndlString(const std::string& src)
{
    const char* str = src.c_str();
    const int count = static_cast<int>(strlen(str));
    for (int ic = count - 1; ic >= 0; --ic)
    {
        const char c = str[ic];
        if (c != '\r' && c != '\n')
        {
            return std::string(str, ic + 1);
        }
    }
    return "";
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前後の引用符を削除します。
//-----------------------------------------------------------------------------
std::string RDequoteString(const std::string& src, const char quote)
{
    std::string dst(src);
    const size_t iquotL = dst.find(quote);
    if (iquotL != std::string::npos)
    {
        if (iquotL == dst.size() - 1) // 末尾にのみ引用符が存在する場合、その引用符を削除します。
        {
            dst = dst.substr(0, iquotL);
        }
        else
        {
            dst = dst.substr(iquotL + 1);
            const size_t iquotR = dst.rfind(quote);
            if (iquotR != std::string::npos)
            {
                dst = dst.substr(0, iquotR);
            }
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ダブルクォートをシングルクォートに置換した文字列を返します。
//-----------------------------------------------------------------------------
std::string RConvertToSingleQuote(const std::string& src)
{
    std::string dst(src);
    for (size_t ic = 0; ic < dst.size(); ++ic)
    {
        if (dst[ic] == '\"')
        {
            dst[ic] = '\'';
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 文字列をトークンに分解します。
//-----------------------------------------------------------------------------
int RTokenizeString(RStringArray& tokens, const std::string& input, const char* delims)
{
    tokens.clear();
    size_t iEnd = 0;
    for (;;)
    {
        const size_t iStart = input.find_first_not_of(delims, iEnd);
        if (iStart == std::string::npos)
        {
            break;
        }
        iEnd = input.find_first_of(delims, iStart);
        if (iEnd == std::string::npos)
        {
            tokens.push_back(input.substr(iStart));
            break;
        }
        tokens.push_back(input.substr(iStart, iEnd - iStart));
    }
    return static_cast<int>(tokens.size());
}

//-----------------------------------------------------------------------------
//! @brief 文字列を末尾の数字の前で分割します。
//-----------------------------------------------------------------------------
std::string RSplitStringEndDigit(std::string& digit, const std::string& src)
{
    const int count = static_cast<int>(src.size());
    int digitCount = 0;
    for (int ic = count - 1; ic >= 0; --ic)
    {
        const char c = src[ic];
        if ('0' <= c && c <= '9')
        {
            ++digitCount;
        }
        else
        {
            break;
        }
    }
    digit = (digitCount > 0) ? src.substr(count - digitCount) : std::string();
    return src.substr(0, count - digitCount);
}

//-----------------------------------------------------------------------------
//! @brief int 型の数値を文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetNumberString(const int num, const char* format)
{
    char buf[64];
    const char* fmt = (format != NULL) ? format : "%d";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, num);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief uint32_t 型の数値を文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetNumberString(const uint32_t num, const char* format)
{
    char buf[64];
    const char* fmt = (format != NULL) ? format : "%u";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, num);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief ポインタを文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetPointerString(const void* ptr, const char* format)
{
    char buf[64];
    const char* fmt = (format != NULL) ? format : "%p";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, ptr);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief 文字列中の {n} を引数配列の n 番目の値で置換した文字列を返します。
//-----------------------------------------------------------------------------
std::string RReplaceArgumentsInString(const std::string& src, const RStringArray& args)
{
    if (args.empty())
    {
        return src;
    }

    char srcArgBuf[8];
    std::string dst = src;
    for (size_t argIdx = 0; argIdx < args.size(); ++argIdx)
    {
        sprintf_s(srcArgBuf, "{%d}", static_cast<int>(argIdx));
        const std::string srcArg(srcArgBuf);
        const std::string& dstArg = args[argIdx];
        size_t startIdx = 0;
        for (;;)
        {
            const size_t findIdx = dst.find(srcArg, startIdx);
            if (findIdx == std::string::npos)
            {
                break;
            }
            dst = dst.substr(0, findIdx) + dstArg + dst.substr(findIdx + srcArg.size());
            startIdx += findIdx + dstArg.size();
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS 文字列を XML 用にエンコードします。
//-----------------------------------------------------------------------------
static const std::string XML_AMP_STR  = "&amp;";
static const std::string XML_LT_STR   = "&lt;";
static const std::string XML_GT_STR   = "&gt;";
static const std::string XML_QUOT_STR = "&quot;";

std::string REncodeXmlString(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (RIsShiftJisFirstByte(static_cast<unsigned char>(c)))
        {
            dst += c;
            if (++ic < src.size())
            {
                dst += src[ic];
            }
        }
        else if (c == '&')
        {
            dst += XML_AMP_STR;
        }
        else if (c == '<')
        {
            dst += XML_LT_STR;
        }
        else if (c == '>')
        {
            dst += XML_GT_STR;
        }
        else if (c == '\"')
        {
            dst += XML_QUOT_STR;
        }
        else
        {
            dst += c;
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief XML 用にエンコードされた Shift-JIS 文字列をデコードします。
//-----------------------------------------------------------------------------
std::string RDecodeXmlString(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (c == '&')
        {
            if (src.find(XML_AMP_STR, ic) == ic)
            {
                dst += '&';
                ic += XML_AMP_STR.size() - 1;
            }
            else if (src.find(XML_LT_STR, ic) == ic)
            {
                dst += '<';
                ic += XML_LT_STR.size() - 1;
            }
            else if (src.find(XML_GT_STR, ic) == ic)
            {
                dst += '>';
                ic += XML_GT_STR.size() - 1;
            }
            else if (src.find(XML_QUOT_STR, ic) == ic)
            {
                dst += '\"';
                ic += XML_QUOT_STR.size() - 1;
            }
            else
            {
                dst += c;
            }
        }
        else
        {
            dst += c;
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS 文字列の制御文字をエスケープして他のコマンドに渡せる文字列を返します。
//-----------------------------------------------------------------------------
std::string REscapeString(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (RIsShiftJisFirstByte(static_cast<unsigned char>(c)))
        {
            dst += c;
            if (++ic < src.size())
            {
                dst += src[ic];
            }
        }
        else if (c == '\t') dst += "\\t";
        else if (c == '\n') dst += "\\n";
        else if (c == '\r') dst += "\\r";
        else if (c == '\"') dst += "\\\"";
        else dst += c;
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief オプション文字列用にエンコードされたスクリプトの文字列をデコードします。
//!        シングルクォートをダブルクォート、'!' を ';'、
//!        &#x27! をシングルクォート、&#x21! を '!' に変換します。
//!
//! @param[in] src エンコードされたスクリプトの文字列です。
//!
//! @return デコードされたスクリプトの文字列を返します。
//-----------------------------------------------------------------------------
std::string RDecodeScriptString(const std::string& src)
{
    std::string dst;
    const int count = static_cast<int>(src.size());
    for (int ic = 0; ic < count; ++ic)
    {
        char c = src[ic];
        if (c == '&')
        {
            if (ic + 5 < count)
            {
                const std::string w = src.substr(ic, 6);
                if      (w == "&#x27!") { c = '\''; ic += 5; }
                else if (w == "&#x21!") { c = '!' ; ic += 5; }
            }
        }
        else if (c == '\'') c = '\"';
        else if (c == '!' ) c = ';';
        dst += c;
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief Windows 形式のファイルパスを返します。"/" を "\" に変換します。
//!        Shift-JIS の std::string に対応しています。
//!
//! @param[in] path UNIX 形式のファイルパスです。
//!
//! @return Windows 形式のファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetWindowsFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    char* buf = new char[size + 1];

    const char* src = path.c_str();
    char* dst = buf;

    while (*src)
    {
        char c = *src++;
        if (c == '/')
        {
            c = '\\';
        }
        *dst++ = c;
    }
    *dst = '\0';

    std::string ret = buf;
    delete[] buf;

    return ret;
}

//-----------------------------------------------------------------------------
//! @brief UNIX 形式のファイルパスを返します。"\" を "/" に変換します。
//!        Shift-JIS の std::string に対応しています。
//!
//! @param[in] path Windows 形式のファイルパスです。
//!
//! @return UNIX 形式のファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetUnixFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    char* buf = new char[size + 1];

    const char* src = path.c_str();
    char* dst = buf;

    while (*src)
    {
        char c = *src++;
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            if (*src)
            {
                *dst++ = c;
                c = *src++;
            }
        }
        else if (c == '\\')
        {
            c = '/';
        }
        *dst++ = c;
    }
    *dst = '\0';

    std::string ret = buf;
    delete[] buf;

    return ret;
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列からファイル名を抽出して返します。
//-----------------------------------------------------------------------------
std::string RGetFileNameFromFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
    }
    if (iSlash >= 0)
    {
        return std::string(str + iSlash + 1);
    }
    else
    {
        return path;
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列からフォルダのパス部分を抽出して返します。
//!        返り値の最後に / は付きません。
//-----------------------------------------------------------------------------
std::string RGetFolderFromFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
    }
    if (iSlash >= 0)
    {
        return std::string(str, iSlash); // without last slash
        //return std::string(str, iSlash + 1); // with last slash
    }
    else
    {
        return std::string(".");
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列から拡張子を抽出して返します。
//-----------------------------------------------------------------------------
std::string RGetExtensionFromFilePath(const std::string& path, const bool lower)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    int iDot = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
        else if (c == '.')
        {
            iDot = ic;
        }
    }

    if (iDot >= 0 && (iSlash == -1 || iDot > iSlash))
    {
        const std::string ext = std::string(str + iDot + 1);
        return (lower) ? RGetLowerCaseString(ext) : ext;
    }
    else
    {
        return std::string(); // empty
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列から拡張子を除いた部分を抽出して返します。
//-----------------------------------------------------------------------------
std::string RGetNoExtensionFilePath(const std::string& path)
{
    int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    int iDot = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
        else if (c == '.')
        {
            iDot = ic;
        }
    }
    if (iDot >= 0 && (iSlash == -1 || iDot > iSlash))
    {
        return std::string(str, iDot);
    }
    else
    {
        return path;
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイルパスがフルパスなら true を返します。
//-----------------------------------------------------------------------------
bool RIsFullFilePath(const std::string& path)
{
    if (path.size() >= 2)
    {
        if (path[1] == ':'                       ||
            (path[0] == '\\' && path[1] == '\\') ||
            (path[0] == '/'  && path[1] == '/' ))
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ファイルパスをフルパスに変換します。
//!
//! @param[in] path 元のファイルパスです。
//! @param[in] isUnix フォルダの区切りを / にするなら true を指定します。
//!
//! @return フルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetFullFilePath(const std::string& path, const bool isUnix)
{
    // 環境変数を展開します。
    std::string dst(path);
    if (dst.find('%') != std::string::npos)
    {
        char expandPath[MAX_PATH];
        if (::ExpandEnvironmentStringsA(dst.c_str(), expandPath, sizeof(expandPath)) != 0)
        {
            dst = expandPath;
        }
    }

    // フルパスに変換します。
    char fullPath[MAX_PATH];
    if (::PathSearchAndQualifyA(dst.c_str(), fullPath, sizeof(fullPath)))
    {
        dst = fullPath;
    }

    // ドライブ文字を大文字にします。
    if (dst.size() >= 2 && dst[1] == ':')
    {
        const char drive = dst[0];
        if ('a' <= drive && drive <= 'z')
        {
            dst[0] = drive - 'a' + 'A';
        }
    }

    return (isUnix) ? RGetUnixFilePath(dst) : dst;
}

//-----------------------------------------------------------------------------
//! @brief 末尾にスラッシュを追加したファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetFilePathWithEndingSlash(const std::string& path)
{
    return (!path.empty() && path[path.size() - 1] == '/') ? path : path + '/';
}

//-----------------------------------------------------------------------------
//! @brief 末尾にバックスラッシュを追加したファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetFilePathWithEndingBackslash(const std::string& path)
{
    return (!path.empty() && path[path.size() - 1] == '\\') ? path : path + '\\';
}

//-----------------------------------------------------------------------------
//! @brief 有効なファイルパス文字列なら true を返します。
//-----------------------------------------------------------------------------
bool RIsValidFilePathString(const std::string& path)
{
    int size = static_cast<int>(path.size());
    for (int ic = 0; ic < size; ++ic)
    {
        char c = path[ic];
        if (c == '*' ||
            c == '?' ||
            c == '<' ||
            c == '>' ||
            c == '|' ||
            c == '\"')
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 有効なファイル名文字列（Shift-JIS）なら true を返します。
//-----------------------------------------------------------------------------
bool RIsValidFileNameString(const std::string& fileName)
{
    int size = static_cast<int>(fileName.size());
    for (int ic = 0; ic < size; ++ic)
    {
        char c = fileName[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (
            c == ':'  ||
            c == '/'  ||
            c == '*'  ||
            c == '?'  ||
            c == '<'  ||
            c == '>'  ||
            c == '|'  ||
            c == '\"' ||
            c == '\\')
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief ファイルが存在するなら true を返します。
//-----------------------------------------------------------------------------
bool RFileExists(const std::string& path)
{
    WIN32_FIND_DATA ffd;
    #ifdef UNICODE
    HANDLE  hFindFile = FindFirstFile(RGetUnicodeFromShiftJis(path).c_str(), &ffd);
    #else
    HANDLE  hFindFile = FindFirstFile(path.c_str(), &ffd);
    #endif
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);

    if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) // check is file
    {
        return true;
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief フォルダが存在するなら true を返します。
//!        path の最後に '\' または '/' が付いていても付いていなくても構いません。
//-----------------------------------------------------------------------------
bool RFolderExists(const std::string& path)
{
    std::string unixPath = RGetUnixFilePath(path);
    int len = static_cast<int>(unixPath.length());
    if (len == 0)
    {
        return false;
    }

    while (len > 0)
    {
        if (unixPath[len - 1] != '/')
        {
            break;
        }
        --len;
    }
    string noSlashPath = string(unixPath.c_str(), len);
    //if (len != strlen(path))
    //  cerr << "no slash: " << path.c_str() << " -> " << noSlashPath.c_str() << endl;

    WIN32_FIND_DATA ffd;
    #ifdef UNICODE
    HANDLE  hFindFile = FindFirstFile(RGetUnicodeFromShiftJis(noSlashPath).c_str(), &ffd);
    #else
    HANDLE  hFindFile = FindFirstFile(noSlashPath.c_str(), &ffd);
    #endif
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);

    if (ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) // check is folder
    {
        return true;
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイル用出力ストリームのフォーマットを初期化します。
//-----------------------------------------------------------------------------
void RInitOutStreamFormat(std::ostream& os)
{
    //-----------------------------------------------------------------------------
    // set format
    os.setf(ios_base::dec, ios_base::basefield);
    os.fill(' ');
    os.setf(ios_base::left, ios_base::adjustfield);
    os.setf(ios_base::uppercase); // 浮動小数の e を大文字に
    #if (_MSC_VER < 1900)
    _set_output_format(_TWO_DIGIT_EXPONENT); // 浮動小数の指数部を 2 桁に
        // VS2015 ではデフォルトで 2 桁です。
    #endif
    os.precision(R_FLOAT_OUT_PRECISION);
    //os.setf(ios_base::showpoint);
    //os.setf(ios_base::fixed, ios_base::floatfield);
    //os.setf(ios_base::scientific, ios_base::floatfield);
    //os.setf(ios_base::fmtflags(0), ios_base::floatfield); // default
}

//-----------------------------------------------------------------------------
//! @brief UTF-8 の BOM を出力します。
//-----------------------------------------------------------------------------
void ROutUtf8Bom(std::ostream& os)
{
    static const unsigned char utf8BOM[] = { 0xef, 0xbb, 0xbf };
    os.write(reinterpret_cast<const char*>(utf8BOM), 3);
}

//-----------------------------------------------------------------------------
//! @brief UTF-16 の BOM を出力します。
//-----------------------------------------------------------------------------
void ROutUtf16Bom(std::ostream& os)
{
    static const unsigned char utf16BOM[] = { 0xff, 0xfe };
    os.write(reinterpret_cast<const char*>(utf16BOM), 2);
}

//-----------------------------------------------------------------------------
//! @brief Shift JIS 文字列を UTF-8 文字列に変換して出力します。
//-----------------------------------------------------------------------------
void ROutUtf8FromShiftJis(std::ostream& os, const std::string& sjisStr)
{
    // UTF-8 の複数バイトコードは 0x80 から 0xfd までの範囲で表現されるので、
    // os はテキストモードでも改行の問題はない
    std::string utf8Str = RGetUtf8FromShiftJis(sjisStr);
    if (utf8Str.size() > 0)
    {
        os.write(utf8Str.c_str(), static_cast<int>(utf8Str.size()));
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定した数のタブ文字を出力します。
//-----------------------------------------------------------------------------
const char* RTab(const int count)
{
    const int TABS_MAX = 16;
    static const char* tabs = "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t"; // TABS_MAX 個

    if (count == 0)
    {
        return "";
    }
    else
    {
        return tabs + (TABS_MAX - count);
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの bool 型用文字列を出力します。
//-----------------------------------------------------------------------------
const char* RBoolStr(const bool value)
{
    return (value) ? "true" : "false";
}

//-----------------------------------------------------------------------------
//! @brief パディングを出力します。
//-----------------------------------------------------------------------------
void ROutPadding(std::ostream& os, const int size, const char value)
{
    if (size > 0)
    {
        std::string padding;
        for (int ibyte = 0; ibyte < size; ++ibyte)
        {
            padding += value;
        }
        os << padding;
    }
}

//-----------------------------------------------------------------------------
//! @brief float 型配列の値が一定なら true を返します。
//!        インデックス 0 の値と他の値の差がすべて許容値未満なら一定とみなします。
//-----------------------------------------------------------------------------
bool RIsConstantArray(const RFloatArray& array, const float tolerance)
{
    const int count = static_cast<int>(array.size());
    if (count >= 2)
    {
        const float firstValue = array[0];
        for (int index = 1; index < count; ++index)
        {
            if (!RIsSame(array[index], firstValue, tolerance))
            {
                return false;
            }
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief float 型配列の値がすべて正（-tolerance 以上）なら true を返します。
//-----------------------------------------------------------------------------
bool RIsArrayAllPositive(const RFloatArray& array, const float tolerance)
{
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        if (array[index] < -tolerance)
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief float 型配列の 0 に近い値を 0 にします。
//-----------------------------------------------------------------------------
void RSnapToZero(RFloatArray& array)
{
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        array[index] = RSnapToZero(array[index]);
    }
}

//-----------------------------------------------------------------------------
//! @brief double 型配列の 0 に近い値を 0 にします。
//-----------------------------------------------------------------------------
void RSnapToZero(RDoubleArray& array)
{
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        array[index] = RSnapToZero(array[index]);
    }
}

//-----------------------------------------------------------------------------
//! @brief float 型配列のすべての値をスケールします。
//-----------------------------------------------------------------------------
void RScaleArray(RFloatArray& array, const float scale)
{
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        array[index] *= scale;
    }
}

//-----------------------------------------------------------------------------
//! @brief double 型配列のすべての値をスケールします。
//-----------------------------------------------------------------------------
void RScaleArray(RDoubleArray& array, const double scale)
{
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        array[index] *= scale;
    }
}

//-----------------------------------------------------------------------------
//! @brief XYZ 各成分の float 型配列のすべての値をスケールします。
//-----------------------------------------------------------------------------
void RScaleArray(
    RFloatArray& xArray,
    RFloatArray& yArray,
    RFloatArray& zArray,
    const float scale
)
{
    RScaleArray(xArray, scale);
    RScaleArray(yArray, scale);
    RScaleArray(zArray, scale);
}

//-----------------------------------------------------------------------------
//! @brief XYZ 各成分の double 型配列のすべての値をスケールします。
//-----------------------------------------------------------------------------
void RScaleArray(
    RDoubleArray& xArray,
    RDoubleArray& yArray,
    RDoubleArray& zArray,
    const double scale
)
{
    RScaleArray(xArray, scale);
    RScaleArray(yArray, scale);
    RScaleArray(zArray, scale);
}

//-----------------------------------------------------------------------------
//! @brief base64 の文字を設定します。
//!
//! @param[in] bb 入力データです。1 ～ 3 バイトを uint32_t 型に格納します。
//!               1 バイトの場合は byte0
//!               2 バイトの場合は (byte0 <<  8) | byte1
//!               3 バイトの場合は (byte0 << 16) | (byte1 << 8) | byte2
//! @param[in] srclen 入力データのバイト数 - 1 です。
//! @param[out] dst 出力データのポインタです。
//-----------------------------------------------------------------------------
static void SetBase64Char(uint32_t bb, int srclen, uint8_t* dst)
{
    static const uint8_t* base64Chars =
        (const uint8_t*)"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

    // 最終位置の計算
    int i;
    for (i = srclen; i < 2; ++i)
    {
        bb <<= 8;
    }

    // BASE64変換
    int base = 18;
    int x;
    for (x = 0; x < srclen + 2; ++x, base -= 6)
    {
        *dst++ = base64Chars[(bb >> base) & 0x3f];
    }

    // 端数の判断
    for (i = x; i < 4; ++i)
    {
        *dst++ = (unsigned char)'='; // 端数
    }
}

//-----------------------------------------------------------------------------
//! @brief base64 でエンコードします。
//-----------------------------------------------------------------------------
uint8_t* REncodeBase64(const uint8_t* srcBuf, const uint32_t srcSize, uint32_t& dstSize)
{
    //-----------------------------------------------------------------------------
    // alloc dst buffer
    uint32_t allocSize = srcSize + (srcSize / 2);
    allocSize = ((allocSize + 3) >> 2) << 2; // align 4
    uint8_t* dstBuf = new uint8_t[allocSize];

    //-----------------------------------------------------------------------------
    // encode
    const uint8_t* src = srcBuf;
    const uint8_t* srcEnd = srcBuf + srcSize;
    uint8_t* dst = dstBuf;

    uint32_t bb = 0;
    int i = 0;
    while (src < srcEnd)
    {
        bb <<= 8;
        bb |= static_cast<uint32_t>(*src);

        // 24bit 単位に編集
        if (i == 2)
        {
            SetBase64Char(bb, i, dst);
            bb = 0;
            i = 0;
            dst += 4;
        }
        else
        {
            ++i;
        }
        ++src;
    }

    // 24bit に満たない場合
    if (i)
    {
        SetBase64Char(bb, i - 1, dst);
        dst += 4;
    }

    //-----------------------------------------------------------------------------
    // end
    dstSize = static_cast<uint32_t>(dst - dstBuf);

    return dstBuf;
}

//-----------------------------------------------------------------------------
//! @brief base64 の入力データの 1 文字を 0 ～ 63 の整数値に変換して返します。
//!        入力データの 1 文字が不正なら -1 を返します。
//!
//! @param[in] c 入力データの 1 文字です。
//!
//! @return 整数値を返します。
//-----------------------------------------------------------------------------
static int GetBase64Value(const int c)
{
    if (c >= static_cast<int>('A') && c <= static_cast<int>('Z'))
    {
        return c - static_cast<int>('A');
    }
    else if (c >= static_cast<int>('a') && c <= static_cast<int>('z'))
    {
        return c - static_cast<int>('a') + 26;
    }
    else if (c >= static_cast<int>('0') && c <= static_cast<int>('9'))
    {
        return c - static_cast<int>('0') + 52;
    }
    else if (static_cast<int>('+') == c)
    {
        return 62;
    }
    else if (static_cast<int>('/') == c)
    {
        return 63;
    }
    else if (static_cast<int>('=') == c)
    {
        return 0;
    }
    else
    {
        return -1;
    }
}

//-----------------------------------------------------------------------------
//! @brief base64 をデコードします。
//-----------------------------------------------------------------------------
uint8_t* RDecodeBase64(const uint8_t* srcBuf, const uint32_t srcSize, uint32_t& dstSize)
{
    //-----------------------------------------------------------------------------
    // alloc dst buffer
    uint32_t allocSize = srcSize;
    uint8_t* dstBuf = new uint8_t[allocSize];

    //-----------------------------------------------------------------------------
    // decode
    const uint8_t* src = srcBuf;
    const uint8_t* srcEnd = srcBuf + srcSize;
    uint8_t* dst = dstBuf;
    while (src < srcEnd)
    {
        int ichar = 0;
        uint32_t base64 = 0;
        while (src < srcEnd)
        {
            int c = *src++;
            if (c == '=')
            {
                break;
            }
            int val = GetBase64Value(c);
            if (val != -1)
            {
                base64 = (base64 << 6) | val;
                ++ichar;
                if (ichar >= 4)
                {
                    break;
                }
            }
        }
        if (ichar == 0)
        {
            break;
        }

        int outSize = 3;
        if (ichar < 4)
        {
            base64 <<= (4 - ichar) * 6;
            outSize = (ichar == 2) ? 1 : 2;
        }

        for (int idst = 0; idst < outSize; ++idst)
        {
            *dst++ = static_cast<uint8_t>((base64 >> ((2 - idst) * 8)) & 0xff);
        }
    }

    //-----------------------------------------------------------------------------
    // end
    dstSize = static_cast<uint32_t>(dst - dstBuf);

    return dstBuf;
}

//-----------------------------------------------------------------------------
//! @brief 環境変数の値を取得します。
//-----------------------------------------------------------------------------
std::string RGetEnvVar(const char* name)
{
    std::string ret;
    char* pValue;
    size_t length;
    if (_dupenv_s(&pValue, &length, name) == 0)
    {
        // 環境変数が見つからない場合は、pValue が NULL に
        // length が 0、戻り値が 0（成功）となります。
        if (pValue != NULL)
        {
            ret = pValue;
            free(pValue);
        }
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief 環境変数が定義されていて値が 0 でなければ true を返します。
//-----------------------------------------------------------------------------
bool RIsEnvVarNotZero(const char* name)
{
    const std::string value = RGetEnvVar(name);
    return (!value.empty() && atoi(value.c_str()) != 0);
}

//-----------------------------------------------------------------------------
//! @brief 呼び出し側プロセスのプロセス ID 文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetCurrentProcessIdString()
{
    return RGetNumberString(static_cast<int>(GetCurrentProcessId()));
}

//-----------------------------------------------------------------------------
//! @brief 一時フォルダのパスを取得します。パスの最後にはスラッシュが付きます。
//!
//! @return 一時フォルダのパスを返します。
//-----------------------------------------------------------------------------
std::string RGetTempFolderPath()
{
    const int PATH_BUF_LENGTH = 1024;
    char pathBuf[PATH_BUF_LENGTH];
    if (GetTempPathA(PATH_BUF_LENGTH, pathBuf) != 0)
    {
        std::string path = RGetFilePathWithEndingSlash(RGetUnixFilePath(pathBuf));
        return path;
    }
    return "";
}

//-----------------------------------------------------------------------------
//! @brief PC のユーザー名を返します。
//-----------------------------------------------------------------------------
std::string RGetUserName()
{
    const int userNameSize = 256;
    char userName[userNameSize];
    DWORD userNameSizeRet = userNameSize;
    GetUserNameA(userName, &userNameSizeRet);
    return std::string(userName);
}

//-----------------------------------------------------------------------------
//! @brief PC のコンピュータ名を返します。
//-----------------------------------------------------------------------------
std::string RGetComputerName()
{
    const int computerNameSize = MAX_COMPUTERNAME_LENGTH + 1;
    char computerName[computerNameSize];
    DWORD computerNameSizeRet = computerNameSize;
    GetComputerNameA(computerName, &computerNameSizeRet);
    return std::string(computerName);
}

//-----------------------------------------------------------------------------
//! @brief SYSTEMTIME から time_t を取得して返します。
//-----------------------------------------------------------------------------
time_t RGetTimeT(const SYSTEMTIME& st)
{
    FILETIME ft;
    ::SystemTimeToFileTime(&st, &ft);

    LONGLONG ll;
    ll = (static_cast<LONGLONG>(ft.dwHighDateTime) << 32) + ft.dwLowDateTime;
    return static_cast<time_t>((ll - 116444736000000000) / 10000000);
}

//-----------------------------------------------------------------------------
//! @brief ISO 8601 に準拠した書式で現在の日時を返します。
//-----------------------------------------------------------------------------
std::string RGetDateTimeString()
{
    SYSTEMTIME time;
    GetLocalTime(&time);

    TIME_ZONE_INFORMATION tzi;
    GetTimeZoneInformation(&tzi);
    const int diffMin = -tzi.Bias; // UTC からの差（分単位）にするため符号反転します。

    char dtStr[64];
    sprintf_s(dtStr, "%d-%02d-%02dT%02d:%02d:%02d%c%02d:%02d",
        time.wYear, time.wMonth, time.wDay,
        time.wHour, time.wMinute, time.wSecond,
        (diffMin >= 0) ? '+' : '-', RAbs(diffMin) / 60, diffMin % 60);
    return std::string(dtStr);
}

//-----------------------------------------------------------------------------
//! @brief 最後に実行した Windows API のエラーメッセージを取得します。
//-----------------------------------------------------------------------------
std::string RGetWinErrorMessage()
{
    LPVOID lpMsgBuf;
    FormatMessage(
        FORMAT_MESSAGE_ALLOCATE_BUFFER |
        FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL,
        GetLastError(),
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // 既定の言語
        (LPTSTR)&lpMsgBuf,
        0,
        NULL
    );
    std::string msg(reinterpret_cast<const char*>(lpMsgBuf));
    LocalFree(lpMsgBuf);
    return msg;
}

//-----------------------------------------------------------------------------
//! @brief Windows のプロセスを作成します。
//!
//! @param[out] pi プロセスの情報を格納します。
//! @param[in] cmd コマンド文字列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//! @param[in] getsStdOut 標準出力ハンドルを使用するなら true です。
//! @param[in] getsStdErr 標準エラーハンドルを使用するなら true です。
//! @param[in] hWriteOut 標準出力ハンドルを指定します。
//! @param[in] hWriteErr 標準エラーハンドルを指定します。
//!
//! @return 成功なら 0 以外を返します。
//-----------------------------------------------------------------------------
static int CreateProcessExec(
    PROCESS_INFORMATION& pi,
    const char* cmd,
    const int showWindow = SW_HIDE,
    bool getsStdOut = false,
    bool getsStdErr = false,
    HANDLE hWriteOut = INVALID_HANDLE_VALUE,
    HANDLE hWriteErr = INVALID_HANDLE_VALUE
)
{
    //-----------------------------------------------------------------------------
    // init process information struct
    ZeroMemory(&pi, sizeof(PROCESS_INFORMATION));

    //-----------------------------------------------------------------------------
    // init startup info struct
    STARTUPINFOA si;
    ZeroMemory(&si, sizeof(si));
    si.cb          = sizeof(si);
    si.dwFlags     = STARTF_USESHOWWINDOW;
    si.wShowWindow = (WORD)showWindow;
    if (getsStdOut || getsStdErr)
    {
        si.dwFlags |= STARTF_USESTDHANDLES;
        if (getsStdOut)
        {
            si.hStdOutput = hWriteOut;
        }
        if (getsStdErr)
        {
            si.hStdError = hWriteErr;
        }
    }

    //-----------------------------------------------------------------------------
    // create process
    int retCode = CreateProcessA(
        NULL,   // lpApplicationName
        const_cast<char*>(cmd), // lpCommandLine
        NULL,   // lpProcessAttributes
        NULL,   // lpThreadAttributes
        (getsStdOut || getsStdErr), // bInheritHandles
        0,      // dwCreationFlags
        NULL,   // lpEnvironment
        NULL,   // lpCurrentDirectory
        &si,    // lpStartupInfo
        &pi);   // lpProcessInformation

    return retCode;
}

//-----------------------------------------------------------------------------
//! @brief Windows のプロセスを実行します。
//!
//! @param[out] pStatus エラーなら 0、成功なら 0 以外を格納します。
//!                     終了コードでエラーを判別できない場合に使用します。
//!                     NULL なら何も格納しません。
//! @param[in] cmd コマンド文字列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//!
//! @return コマンドの終了コードを返します。
//-----------------------------------------------------------------------------
int RExecProcess(int* pStatus, const char* cmd, const int showWindow)
{
    //-----------------------------------------------------------------------------
    // set status to success
    if (pStatus != NULL)
    {
        *pStatus = 1;
    }

    //-----------------------------------------------------------------------------
    // create process
    PROCESS_INFORMATION pi;
    if (CreateProcessExec(pi, cmd, showWindow) == 0)
    {
        cerr << "Error: Cannot create the process: " << cmd << endl;
        if (pStatus != NULL)
        {
            *pStatus = 0;
        }
        return 1;
    }

    //-----------------------------------------------------------------------------
    // wait until boot finish
    WaitForInputIdle(pi.hProcess, INFINITE);

    //-----------------------------------------------------------------------------
    // wait until command finish
    WaitForSingleObject(pi.hProcess, INFINITE);

    //-----------------------------------------------------------------------------
    // get return code
    DWORD exitCode;
    if (GetExitCodeProcess(pi.hProcess, &exitCode) == 0)
    {
        cerr << "Error: Cannot get the process exit code: " << cmd << endl;
        if (pStatus != NULL)
        {
            *pStatus = 0;
        }
        exitCode = 1;
    }
    //cerr << "My exec process: " << cmd << endl;
    //cerr << "exit code = " << exitCode << endl;

    //-----------------------------------------------------------------------------
    // close process handle
    CloseHandle(pi.hThread);
    CloseHandle(pi.hProcess);

    return exitCode;
}

//-----------------------------------------------------------------------------
//! @brief Windows のパイプから文字列をリードします。
//!
//! @param[out] dst 文字列を格納します。
//! @param[in] pipe パイプのハンドルです。
//! @param[in] name パイプの名前です（エラー表示用）。
//!
//! @return 成功なら true を返します。
//-----------------------------------------------------------------------------
static bool ReadFromPipe(std::string& dst, HANDLE pipe, const std::string& name)
{
    dst.clear();

    DWORD len;
    if (!PeekNamedPipe(pipe, NULL, 0, NULL, &len, NULL))
    {
        cerr << "Error: Cannot peek the named pipe (" << name << ")" << endl;
        return false;
    }

    if (len > 0)
    {
        dst.assign(len, '\0');
        DWORD rlen;
        if (!ReadFile(pipe, &dst[0], len, &rlen, NULL))
        {
            cerr << "Error: Cannot read from the pipe (" << name << ")"  << endl;
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief Windows のプロセスからメッセージを取得します。
//!
//! @param[in,out] outMsg 標準出力から取得した文字列を追加します。
//! @param[in,out] errMsg 標準エラーから取得した文字列を追加します。
//! @param[in] hReadOut 標準出力パイプのリードハンドルです。標準出力を取得しないなら nullptr を指定します。
//! @param[in] hReadErr 標準エラーパイプのリードハンドルです。標準エラーを取得しないなら nullptr を指定します。
//!
//! @return 成功なら true を返します。
//-----------------------------------------------------------------------------
static bool ReadProcessMessage(
    std::string& outMsg,
    std::string& errMsg,
    HANDLE hReadOut,
    HANDLE hReadErr
)
{
    //-----------------------------------------------------------------------------
    // read out pipe
    if (hReadOut != nullptr)
    {
        std::string str;
        if (!ReadFromPipe(str, hReadOut, "stdout"))
        {
            return false;
        }
        outMsg += str;
    }

    //-----------------------------------------------------------------------------
    // read err pipe
    if (hReadErr != nullptr)
    {
        std::string str;
        if (!ReadFromPipe(str, hReadErr, "stderr"))
        {
            return false;
        }
        errMsg += str;
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief Windows プロセスのコンストラクタです。
//-----------------------------------------------------------------------------
RWinProcess::RWinProcess(const bool getsStdOutOrErr)
:   m_hReadOut(nullptr),
    m_hWriteOut(nullptr),
    m_hReadErr(nullptr),
    m_hWriteErr(nullptr),
    m_IsValid(false),
    m_IsStarted(false),
    m_IsLaunched(false),
    m_ExitCode(0)
{
    //cerr << "RWinProcess(" << getsStdOutOrErr << ")" << endl;
    ZeroMemory(&m_Pi, sizeof(PROCESS_INFORMATION));
    m_Sa.nLength = sizeof(m_Sa);
    m_Sa.lpSecurityDescriptor = 0;
    m_Sa.bInheritHandle = TRUE;
    if (getsStdOutOrErr)
    {
        if (!CreatePipe(&m_hReadOut, &m_hWriteOut, &m_Sa, 0))
        {
            m_ErrMsg = "Cannot create the pipe (stdout)";
            return;
        }
        if (!CreatePipe(&m_hReadErr, &m_hWriteErr, &m_Sa, 0))
        {
            m_ErrMsg = "Cannot create the pipe (stderr)";
            return;
        }
    }
    m_IsValid = true;
}

//-----------------------------------------------------------------------------
//! @brief Windows プロセスのデストラクタです。
//-----------------------------------------------------------------------------
RWinProcess::~RWinProcess()
{
    //cerr << "~RWinProcess()" << endl;
    const int CheckTimeOut = 100; // ms
    WaitFinish(CheckTimeOut);

    if (m_Pi.hThread != nullptr)
    {
        CloseHandle(m_Pi.hThread);
    }
    if (m_Pi.hProcess != nullptr)
    {
        CloseHandle(m_Pi.hProcess);
    }
    if (m_hReadOut != nullptr)
    {
        CloseHandle(m_hReadOut);
    }
    if (m_hWriteOut != nullptr)
    {
        CloseHandle(m_hWriteOut);
    }
    if (m_hReadErr != nullptr)
    {
        CloseHandle(m_hReadErr);
    }
    if (m_hWriteErr != nullptr)
    {
        CloseHandle(m_hWriteErr);
    }
}

//-----------------------------------------------------------------------------
//! @brief Windows プロセスを開始します。
//-----------------------------------------------------------------------------
bool RWinProcess::Start(const char* cmd, const int showWindow)
{
    if (!m_IsValid)
    {
        m_ErrMsg = "Process is not valid";
        return false;
    }

    if (m_IsStarted)
    {
        m_ErrMsg = "Process is already started";
        return false;
    }

    m_IsLaunched = false;
    if (CreateProcessExec(m_Pi, cmd, showWindow,
        (m_hWriteOut != nullptr), (m_hWriteErr != nullptr), m_hWriteOut, m_hWriteErr) == 0)
    {
        m_ErrMsg = "Cannot create the process";
        return false;
    }

    //cerr << "Process::Start(): " << cmd << endl;
    m_IsStarted = true;
    return true;
}

//-----------------------------------------------------------------------------
//! @brief Windows プロセスが開始していれば終了まで待ちます。
//-----------------------------------------------------------------------------
void RWinProcess::WaitFinish(const int checkTimeOut)
{
    //cerr << "Process::WaitFinish(" << checkTimeOut << "): " << m_IsValid << ", " << m_IsStarted << endl;
    if (m_IsValid && m_IsStarted)
    {
        while (!CheckFinish(checkTimeOut))
        {
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief Windows プロセスが終了しているかチェックします。
//-----------------------------------------------------------------------------
bool RWinProcess::CheckFinish(const int timeOut)
{
    if (!m_IsValid || !m_IsStarted)
    {
        return true;
    }

    //-----------------------------------------------------------------------------
    // プロセスの起動が完了しているかチェックします。
    if (!m_IsLaunched)
    {
        if (WaitForInputIdle(m_Pi.hProcess, timeOut) == WAIT_TIMEOUT)
        {
            return false;
        }
        else
        {
            m_IsLaunched = true;
        }
    }

    //-----------------------------------------------------------------------------
    // プロセスが完了しているかチェックします。
    const DWORD processState = WaitForSingleObject(m_Pi.hProcess, timeOut);
    if (processState == WAIT_OBJECT_0)
    {
        DWORD exitCode;
        if (GetExitCodeProcess(m_Pi.hProcess, &exitCode) == 0)
        {
            m_ErrMsg = "Cannot get the process exit code";
            m_ExitCode = 1;
        }
        else
        {
            m_ExitCode = exitCode;
        }
        CloseHandle(m_Pi.hThread);
        m_Pi.hThread = nullptr;
        CloseHandle(m_Pi.hProcess);
        m_Pi.hProcess = nullptr;
        if (!ReadProcessMessage(m_OutMsg, m_ErrMsg, m_hReadOut, m_hReadErr))
        {
            m_ExitCode = 1;
        }
        m_IsStarted = m_IsLaunched = false;
    }
    else if (processState == WAIT_TIMEOUT)
    {
        // タイムアウトになった場合はパイプから標準出力と標準エラーを取得します。
        // パイプのバッファが一杯になりプロセスが無限ループに陥るのを防ぎます。
        if (!ReadProcessMessage(m_OutMsg, m_ErrMsg, m_hReadOut, m_hReadErr))
        {
            m_ExitCode = 1;
            m_IsStarted = m_IsLaunched = false;
        }
    }
    return !m_IsStarted;
}

//-----------------------------------------------------------------------------
//! @brief Windows のプロセスを実行し、標準出力および標準エラーを取得します。
//!
//! @param[out] pStatus エラーなら 0、成功なら 0 以外を格納します。
//!                     終了コードでエラーを判別できない場合に使用します。
//!                     NULL なら何も格納しません。
//! @param[out] pOutMsg 標準出力を格納します。nullptr なら格納しません。
//! @param[out] pErrMsg 標準エラー出力を格納します。nullptr なら格納しません。
//! @param[in] cmd コマンド文字列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//!
//! @return 終了コードを返します。
//-----------------------------------------------------------------------------
int RExecProcessWithPipe(
    int* pStatus,
    std::string* pOutMsg,
    std::string* pErrMsg,
    const char* cmd,
    const int showWindow
)
{
    //-----------------------------------------------------------------------------
    // ステータスを失敗に設定します。
    if (pStatus != nullptr)
    {
        *pStatus = 0;
    }

    //-----------------------------------------------------------------------------
    // 出力メッセージをクリアします。
    if (pOutMsg != nullptr)
    {
        pOutMsg->clear();
    }
    if (pErrMsg != nullptr)
    {
        pErrMsg->clear();
    }

    //-----------------------------------------------------------------------------
    // 標準出力と標準エラーの両方を取得しない場合は、
    // パイプを使用しないバージョンを呼び出します。
    const bool getsStdOutOrErr = (pOutMsg != nullptr || pErrMsg != nullptr);
    if (!getsStdOutOrErr)
    {
        return RExecProcess(pStatus, cmd, showWindow);
    }

    //-----------------------------------------------------------------------------
    // プロセスを実行します。
    int exitCode = 1;
    std::string errMsg;
    RWinProcess proc(getsStdOutOrErr);
    if (proc.IsValid())
    {
        if (proc.Start(cmd, showWindow))
        {
            const int CheckTimeOut = 100; // ms
            proc.WaitFinish(CheckTimeOut);
            exitCode = proc.GetExitCode();
            if (pOutMsg != nullptr)
            {
                *pOutMsg = RCutEndlString(proc.GetOutMessage());
            }
        }
    }

    //-----------------------------------------------------------------------------
    // エラーメッセージを設定します。
    if (pErrMsg != nullptr)
    {
        *pErrMsg = RCutEndlString(proc.GetErrorMessage());
        if (!pErrMsg->empty() && !proc.IsStarted())
        {
            *pErrMsg += std::string(": ") + cmd;
        }
    }

    //-----------------------------------------------------------------------------
    // 終了コードが 0 ならステータスを成功に設定します。
    if (exitCode == 0 && pStatus != nullptr)
    {
        *pStatus = 1;
    }

    return exitCode;
}

//-----------------------------------------------------------------------------
//! @brief Windows のプロセスを並列実行し、標準出力および標準エラーを取得します。
//!
//! @param[out] pStatus エラーなら 0、成功なら 0 以外を格納します。
//!                     終了コードでエラーを判別できない場合に使用します。
//!                     nullptr なら何も格納しません。
//! @param[out] pOutMsg 標準出力を格納します。nullptr なら格納しません。
//! @param[out] pErrMsg 標準エラー出力を格納します。nullptr なら格納しません。
//! @param[in] cmds コマンド文字列配列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//!
//! @return 終了コードを返します。
//-----------------------------------------------------------------------------
int RExecProcessParallel(
    int* pStatus,
    std::string* pOutMsg,
    std::string* pErrMsg,
    const RStringArray& cmds,
    const int showWindow
)
{
    //-----------------------------------------------------------------------------
    // ステータスを失敗に設定します。
    if (pStatus != nullptr)
    {
        *pStatus = 0;
    }

    //-----------------------------------------------------------------------------
    // 出力メッセージをクリアします。
    if (pOutMsg != nullptr)
    {
        pOutMsg->clear();
    }
    if (pErrMsg != nullptr)
    {
        pErrMsg->clear();
    }

    //-----------------------------------------------------------------------------
    // プロセスをコマンド数だけ作成します。
    int exitCode = 0;
    RStringArray outMsgs(cmds.size()); // コマンド順に格納します。
    RStringArray errMsgs; // エラー発生順に格納します。
    const bool getsStdOutOrErr = (pOutMsg != nullptr || pErrMsg != nullptr);
    std::vector<RWinProcess*> pProcs;
    for (size_t cmdIdx = 0; cmdIdx < cmds.size(); ++cmdIdx)
    {
        RWinProcess* pProc = new RWinProcess(getsStdOutOrErr);
        pProcs.push_back(pProc);
        if (!pProc->IsValid())
        {
            exitCode = 1;
            errMsgs.push_back(pProc->GetErrorMessage());
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // 各プロセスを開始します。
    if (exitCode == 0)
    {
        for (size_t cmdIdx = 0; cmdIdx < cmds.size(); ++cmdIdx)
        {
            RWinProcess* pProc = pProcs[cmdIdx];
            const std::string& cmd = cmds[cmdIdx];
            if (!pProc->Start(cmd.c_str(), showWindow))
            {
                exitCode = 1;
                errMsgs.push_back(pProc->GetErrorMessage() + ": " + cmd);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 各プロセスの終了を待ちます。
    const int TimeOut = 100; // ms
    if (exitCode == 0)
    {
        for (;;)
        {
            int runningCount = 0;
            for (size_t cmdIdx = 0; cmdIdx < cmds.size(); ++cmdIdx)
            {
                RWinProcess* pProc = pProcs[cmdIdx];
                if (pProc->IsStarted())
                {
                    if (pProc->CheckFinish(TimeOut))
                    {
                        const int procExitCode = pProc->GetExitCode();
                        if (procExitCode != 0 && exitCode == 0)
                        {
                            exitCode = procExitCode;
                        }
                        const std::string& outMsg = pProc->GetOutMessage();
                        outMsgs[cmdIdx] = RCutEndlString(outMsg);
                        const std::string& errMsg = pProc->GetErrorMessage();
                        if (!errMsg.empty())
                        {
                            errMsgs.push_back(RCutEndlString(errMsg));
                        }
                    }
                    else
                    {
                        ++runningCount;
                    }
                }
            }
            if (runningCount == 0)
            {
                break;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 各プロセスの出力メッセージを収集します。
    if (pOutMsg != nullptr)
    {
        for (size_t msgIdx = 0; msgIdx < outMsgs.size(); ++msgIdx)
        {
            const std::string& outMsg = outMsgs[msgIdx];
            if (!outMsg.empty())
            {
                if (!pOutMsg->empty())
                {
                    *pOutMsg += "\n";
                }
                *pOutMsg += outMsg;
            }
        }
    }
    if (pErrMsg != nullptr)
    {
        for (size_t msgIdx = 0; msgIdx < errMsgs.size(); ++msgIdx)
        {
            *pErrMsg += errMsgs[msgIdx] + ((msgIdx < errMsgs.size() - 1) ? "\n" : "");
        }
    }

    //-----------------------------------------------------------------------------
    // 各プロセスを削除します。
    for (size_t procIdx = 0; procIdx < pProcs.size(); ++procIdx)
    {
        delete pProcs[procIdx];
    }

    //-----------------------------------------------------------------------------
    // 終了コードが 0 ならステータスを成功に設定します。
    if (exitCode == 0 && pStatus != nullptr)
    {
        *pStatus = 1;
    }

    return exitCode;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief Windows の DWORD 型のレジストリ値を取得します。
//-----------------------------------------------------------------------------
DWORD RGetWinRegDword(
    HKEY hParentKey,
    const char* lpSubKey,
    const char* lpValueName,
    const DWORD defaultValue
)
{
    DWORD value = defaultValue;
    HKEY hKey;
    if (::RegOpenKeyExA(hParentKey, lpSubKey, 0, KEY_READ, &hKey) == ERROR_SUCCESS)
    {
        DWORD dwSize = sizeof(value);
        if (::RegQueryValueExA(hKey, lpValueName, NULL, NULL,
            reinterpret_cast<LPBYTE>(&value), &dwSize) == ERROR_SUCCESS)
        {
            // success
        }
        ::RegCloseKey(hKey);
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief Windows の文字列型のレジストリ値を取得します。
//-----------------------------------------------------------------------------
std::string RGetWinRegString(
    HKEY hParentKey,
    const char* lpSubKey,
    const char* lpValueName,
    const char* defaultValue
)
{
    std::string value = defaultValue;
    HKEY hKey;
    if (::RegOpenKeyExA(hParentKey, lpSubKey, 0, KEY_READ, &hKey) == ERROR_SUCCESS)
    {
        DWORD dwSize;
        if (::RegQueryValueExA(hKey, lpValueName, NULL, NULL, NULL, &dwSize) == ERROR_SUCCESS)
        {
            char* strBuf = new char[dwSize + 1];
            if (::RegQueryValueExA(hKey, lpValueName, NULL, NULL,
                reinterpret_cast<LPBYTE>(strBuf), &dwSize) == ERROR_SUCCESS)
            {
                value = strBuf;
            }
            delete[] strBuf;
        }
        ::RegCloseKey(hKey);
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイル用の名前の文字列の使用できない文字（':'）を
//!        '_' に変換した文字列を返します。
//-----------------------------------------------------------------------------
std::string RAdjustNameChar(const std::string& src)
{
    std::string dst = "";
    const int count = static_cast<int>(src.size());
    for (int ic = 0; ic < count; ++ic)
    {
        char c = src[ic];
        if (c == ':')
        {
            c = '_';
        }
        dst += c;
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの各要素の名前に使用できる文字なら true を返します。
//!        各要素とはノード、マテリアル、シェイプ、テクスチャです。
//-----------------------------------------------------------------------------
bool RIsValidElementNameChar(const char c)
{
    return (
        ('0' <= c && c <= '9') ||
        ('A' <= c && c <= 'Z') ||
        ('a' <= c && c <= 'z') ||
        c == '-' ||
        c == '.' ||
        c == '_');
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの各要素の名前に使用できる文字列なら true を返します。
//!        各要素とはノード、マテリアル、シェイプ、テクスチャです。
//-----------------------------------------------------------------------------
bool RIsValidElementNameString(const std::string& name)
{
    int count = static_cast<int>(name.size());
    for (int ic = 0; ic < count; ++ic)
    {
        if (!RIsValidElementNameChar(name[ic]))
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの各要素の名前に使用できない文字を
//!        '_' に変換した文字列を返します。
//!        各要素とはノード、マテリアル、シェイプ、テクスチャです。
//-----------------------------------------------------------------------------
std::string RAdjustElementNameString(const std::string& name)
{
    std::string dst = "";
    int count = static_cast<int>(name.size());
    for (int ic = 0; ic < count; ++ic)
    {
        char c = name[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            c = '_';
            ++ic;
        }
        else if (!RIsValidElementNameChar(c))
        {
            c = '_';
        }
        dst += c;
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのプリセット名に使用できる文字列なら true を返します。
//-----------------------------------------------------------------------------
bool RIsValidPresetNameString(const std::string& name)
{
    if (!RIsAsciiString(name))
    {
        return false;
    }
    else
    {
        for (size_t ic = 0; ic < name.length(); ++ic)
        {
            const char c = name[ic];
            if (c == ';'  ||
                c == ':'  ||
                c == '/'  ||
                c == '\\' ||
                c == '*'  ||
                c == '?'  ||
                c == '<'  ||
                c == '>'  ||
                c == '|'  ||
                c == '\"')
            {
                return false;
            }
        }
        return true;
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイル情報に出力する元ファイルのパスを取得します。
//-----------------------------------------------------------------------------
std::string RGetFileInfoSourceFilePath(
    const std::string& fullPath,
    const std::string& projectRootPath
)
{
    std::string srcPath(fullPath);
    if (!fullPath.empty() && !projectRootPath.empty())
    {
        char relativePath[MAX_PATH];
        if (::PathRelativePathToA(relativePath,
            RGetWindowsFilePath(projectRootPath).c_str(), FILE_ATTRIBUTE_DIRECTORY,
            RGetWindowsFilePath(fullPath       ).c_str(), FILE_ATTRIBUTE_NORMAL))
        {
            srcPath = RGetUnixFilePath(relativePath);
            if (srcPath.find("./") == 0)
            {
                srcPath = srcPath.substr(2);
            }
        }
    }
    return srcPath;
}

//-----------------------------------------------------------------------------
//! @brief 処理結果ステータスのコンストラクタです（日英メッセージ。引数配列）。
//-----------------------------------------------------------------------------
RStatus::RStatus(
    const RStatusCode code,
    const std::string& messageJp,
    const std::string& messageEn,
    const RStringArray& args
)
: m_Code(code)
{
    m_MessageJp = RReplaceArgumentsInString(messageJp, args);
    m_MessageEn = RReplaceArgumentsInString(messageEn, args);
}

//-----------------------------------------------------------------------------
//! @brief 処理結果ステータスのコンストラクタです（日英メッセージ。引数配列。ワイド文字列）。
//-----------------------------------------------------------------------------
RStatus::RStatus(
    const RStatusCode code,
    const std::wstring& messageJpW,
    const std::string& messageEn,
    const RStringArray& args
)
: m_Code(code)
{
    m_MessageJp = RReplaceArgumentsInString(RGetShiftJisFromUnicode(messageJpW), args);
    m_MessageEn = RReplaceArgumentsInString(messageEn, args);
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の文字列から属性の値を取得します。
//-----------------------------------------------------------------------------
std::string RGetXmlAttr(
    const std::string& xmlStr,
    const std::string& name,
    const char* alt
)
{
    size_t iStart = xmlStr.find(name + "=\"");
    if (iStart != std::string::npos)
    {
        iStart += name.size() + 2;
        const size_t iEnd = xmlStr.find("\"", iStart);
        if (iEnd != std::string::npos)
        {
            return xmlStr.substr(iStart, iEnd - iStart);
        }
    }
    return alt;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素のエラーを表示します。
//-----------------------------------------------------------------------------
void RXMLElement::DisplayError(const std::string& message) const
{
    if (m_ErrorFunc != NULL)
    {
        m_ErrorFunc(message.c_str(), m_pErrorUserData);
    }
    else
    {
        cerr << message << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の属性を解析してノードに記録します。
//-----------------------------------------------------------------------------
int RXMLElement::LoadAttributes(const char* src, const char end)
{
    int pos = 0;
    for (;;)
    {
        pos += static_cast<int>(strspn(&src[pos], " \t\r\n"));
        if (src[pos] == '>' || src[pos] == end)
        {
            break;
        }
        int namelen = static_cast<int>(strcspn(&src[pos], "="));
        std::string attrName(&src[pos], namelen);
        pos += namelen;
        std::string value;
        if (src[pos] == '=')
        {
            ++pos;
            if (src[pos] == '\"' || src[pos] == '\'')
            {
                char end2[2] = { src[pos], };
                ++pos;
                int valuelen = static_cast<int>(strcspn(&src[pos], end2));
                value = std::string(&src[pos], valuelen);
                pos += valuelen;
                pos += (src[pos] != '\0');
            }
            else
            {
                char end2[] = " \t\r\n";
                int valuelen = static_cast<int>(strcspn(&src[pos], end2));
                value = std::string(&src[pos], valuelen);
                pos += valuelen;
            }
        }
        attributes[attrName] = value;
    }
    return pos;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の代入演算子です。
//-----------------------------------------------------------------------------
RXMLElement& RXMLElement::operator=(const RXMLElement& r)
{
    name = r.name;
    text = r.text;
    attributes.clear();
    for (AttrMap::const_iterator it = r.attributes.begin(); it != r.attributes.end(); ++it)
    {
        attributes[it->first] = it->second;
    }
    nodes.clear();
    nodes.resize(r.nodes.size());
    for (size_t i = 0; i < r.nodes.size(); ++i)
    {
        nodes[i] = r.nodes[i];
    }

    m_ErrorFunc = r.m_ErrorFunc;
    m_pErrorUserData = r.m_pErrorUserData;

    return *this;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の指定のパスに該当する要素を指定個数分検索します。
//-----------------------------------------------------------------------------
void RXMLElement::FindElements(
    std::vector<const RXMLElement*>& list,
    const std::string& path,
    const bool showError,
    const int maxCount
) const
{
    // パス名を階層に区切ります。
    RStringArray entries;
    std::string subPath(path);
    for (size_t ic = 0; ic < subPath.size(); ++ic)
    {
        if (subPath[ic] == '\\')
        {
            subPath[ic] = '/';
        }
    }
    while (!subPath.empty())
    {
        int span = static_cast<int>(strcspn(subPath.c_str(), "/"));
        entries.push_back(subPath.substr(0, span));
        span += (subPath[span] == '/');
        subPath = subPath.substr(span);
    }
    const int entryCount = static_cast<int>(entries.size());

    // 階層の上から順に探索します（一意性が無いので総当たり）。
    int count = maxCount;
    std::vector<const RXMLElement*> stack(entryCount);
    std::vector<int> index(entryCount);
    int level = 0;
    index[0] = 0;
    stack[0] = this;
    const RXMLElement* node = this;
    int pos = 0;
    for (;;)
    {
        bool found = false;
        if (level >= entryCount)
        {
            // 全階層マッチしたら現在位置を記録します。
            list.push_back(node);
            if (--count <= 0)
            {
                // 指定の個数が見つかればここで終了です。
                break;
            }
        }
        else
        {
            // そうでなければ現在の階層を探索します。
            for (; pos < static_cast<int>(node->nodes.size()); ++pos)
            {
                if (node->nodes[pos].name == entries[level])
                {
                    found = true;
                    break;
                }
            }
        }
        if (found)
        {
            // マッチするノードがあればさらに下層へ。
            // 失敗時のために現在位置を退避します。
            index[level] = pos + 1;
            stack[level] = node;
            ++level;
            node = &node->nodes[pos];
            pos = 0;
        }
        else if (--level >= 0)
        {
            // 該当ノードが無ければスタックを戻ります。
            node = stack[level];
            pos = index[level];
        }
        else // 全要素の探索が終了した。
        {
            break;
        }
    }

    if (showError && list.size() == 0)
    {
        DisplayError("Element is not found: <" + path + ">");
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 文書を解析してツリーを構築します（ASCII テキスト限定）。
//-----------------------------------------------------------------------------
void RXMLElement::LoadDocument(RXMLElement* decl, const std::string& xmlDoc)
{
    RXMLElement dummy;
    decl = (decl != NULL) ? decl : &dummy;
    const char* src = xmlDoc.c_str();
    int len = static_cast<int>(xmlDoc.size());
    if (static_cast<uint8_t>(src[0]) == 0xef)
    {
        // BOM を読み捨てます。
        src += 3;
        len -= 3;
    }
    std::vector<RXMLElement*> stack;
    RXMLElement* node = this;
    stack.push_back(node);
    for (int pos = 0; pos < len; )
    {
        if (src[pos] == '\0')
        {
            break;
        }
        const int span = static_cast<int>(strcspn(&src[pos], "<"));
        node->text += std::string(&src[pos], span);
        pos += span;
        if (src[pos] == '<')
        {
            ++pos;
            if (src[pos] == '?') // 宣言要素。
            {
                ++pos;
                int namelen = static_cast<int>(strcspn(&src[pos], " \t\r\n?"));
                std::string nodeName(&src[pos], namelen);
                pos += namelen;
                pos += decl->LoadAttributes(&src[pos], '?');
                pos += (src[pos] == '?');
            }
            else // 開始・終了タグ。
            {
                bool endtag = (src[pos] == '/');
                pos += endtag;
                int namelen = static_cast<int>(strcspn(&src[pos], " \t\r\n/>"));
                std::string nodeName(&src[pos], namelen);
                pos += namelen;
                if (!endtag) // 開始タグ。
                {
                    node = node->CreateElement();
                    node->name = nodeName;
                    stack.push_back(node);
                    pos += node->LoadAttributes(&src[pos], '/');
                    endtag = (src[pos] == '/');
                    pos += endtag;
                }
                if (endtag) // 終了タグ。
                {
                    for (int i = static_cast<int>(stack.size()) - 1; i >= 0; --i)
                    {
                        if (stack[i]->name == nodeName)
                        {
                            if (i > 0 && stack[i] == node)
                            {
                                node = stack[i - 1];
                            }
                            stack.erase(stack.end() - 1);
                            break;
                        }
                    }
                }
                pos += static_cast<int>(strcspn(&src[pos], ">"));
            }
            pos += (src[pos] == '>');
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief float 型 2 次元ベクトルを 3 次元ベクトルから生成します。
//-----------------------------------------------------------------------------
RVec2::RVec2(const RVec3& v3)
{
    x = v3.x;
    y = v3.y;
}

//-----------------------------------------------------------------------------
//! @brief float 型 2 次元ベクトルを 4 次元ベクトルから生成します。
//-----------------------------------------------------------------------------
RVec2::RVec2(const RVec4& v4)
{
    x = v4.x;
    y = v4.y;
}

//-----------------------------------------------------------------------------
//! @brief float 型 3 次元ベクトルを 4 次元ベクトルから生成します。
//-----------------------------------------------------------------------------
RVec3::RVec3(const RVec4& v4)
{
    x = v4.x;
    y = v4.y;
    z = v4.z;
}

//-----------------------------------------------------------------------------
//! @brief float 型 3 次元ベクトルに 4 x 4 行列を乗算したベクトルを返します。
//!        行列の移動成分は適用しません。
//-----------------------------------------------------------------------------
RVec3 RVec3::operator*(const RMtx44& m) const
{
    float dx = x * m[0][0] + y * m[1][0] + z * m[2][0];
    float dy = x * m[0][1] + y * m[1][1] + z * m[2][1];
    float dz = x * m[0][2] + y * m[1][2] + z * m[2][2];
    return RVec3(dx, dy, dz);
}

//-----------------------------------------------------------------------------
//! @brief float 型 3 次元ベクトルに 4 x 4 行列を乗算します。
//!        行列の移動成分は適用しません。
//-----------------------------------------------------------------------------
RVec3& RVec3::operator*=(const RMtx44& m)
{
    float dx = x * m[0][0] + y * m[1][0] + z * m[2][0];
    float dy = x * m[0][1] + y * m[1][1] + z * m[2][1];
    float dz = x * m[0][2] + y * m[1][2] + z * m[2][2];
    x = dx;
    y = dy;
    z = dz;
    return *this;
}

//-----------------------------------------------------------------------------
//! @brief float 型 4 次元ベクトルに 4 x 4 行列を乗算したベクトルを返します。
//-----------------------------------------------------------------------------
RVec4 RVec4::operator*(const RMtx44& m) const
{
    float dx = x * m[0][0] + y * m[1][0] + z * m[2][0] + w * m[3][0];
    float dy = x * m[0][1] + y * m[1][1] + z * m[2][1] + w * m[3][1];
    float dz = x * m[0][2] + y * m[1][2] + z * m[2][2] + w * m[3][2];
    float dw = x * m[0][3] + y * m[1][3] + z * m[2][3] + w * m[3][3];
    return RVec4(dx, dy, dz, dw);
}

//-----------------------------------------------------------------------------
//! @brief float 型 4 次元ベクトルに 4 x 4 行列を乗算します。
//-----------------------------------------------------------------------------
RVec4& RVec4::operator*=(const RMtx44& m)
{
    float dx = x * m[0][0] + y * m[1][0] + z * m[2][0] + w * m[3][0];
    float dy = x * m[0][1] + y * m[1][1] + z * m[2][1] + w * m[3][1];
    float dz = x * m[0][2] + y * m[1][2] + z * m[2][2] + w * m[3][2];
    float dw = x * m[0][3] + y * m[1][3] + z * m[2][3] + w * m[3][3];
    x = dx;
    y = dy;
    z = dz;
    w = dw;
    return *this;
}

//-----------------------------------------------------------------------------
//! @brief int 型 2 次元ベクトルを 3 次元ベクトルから生成します。
//-----------------------------------------------------------------------------
RIVec2::RIVec2(const RIVec3& v3)
{
    x = v3.x;
    y = v3.y;
}

//-----------------------------------------------------------------------------
//! @brief int 型 2 次元ベクトルを 4 次元ベクトルから生成します。
//-----------------------------------------------------------------------------
RIVec2::RIVec2(const RIVec4& v4)
{
    x = v4.x;
    y = v4.y;
}

//-----------------------------------------------------------------------------
//! @brief int 型 3 次元ベクトルを 4 次元ベクトルから生成します。
//-----------------------------------------------------------------------------
RIVec3::RIVec3(const RIVec4& v4)
{
    x = v4.x;
    y = v4.y;
    z = v4.z;
}

//-----------------------------------------------------------------------------
//! @brief float 型 4 x 4 行列に他の行列を乗算した行列を返します。
//-----------------------------------------------------------------------------
RMtx44 RMtx44::operator*(const RMtx44& rhs) const
{
    RMtx44 r;
    for (int row = 0; row < 4; ++row)
    {
        for (int col = 0; col < 4; ++col)
        {
            r.m_Value[row][col] =
                m_Value[row][0] * rhs.m_Value[0][col] +
                m_Value[row][1] * rhs.m_Value[1][col] +
                m_Value[row][2] * rhs.m_Value[2][col] +
                m_Value[row][3] * rhs.m_Value[3][col];
        }
    }
    return r;
}

//-----------------------------------------------------------------------------
//! @brief float 型 4 x 4 行列の逆行列を返します。
//-----------------------------------------------------------------------------
RMtx44 RMtx44::Inverse() const
{
    RVec4 res0, res1, res2, res3;

    float tmp0 = (m_Value[2][2] * m_Value[0][3]) - (m_Value[0][2] * m_Value[2][3]);
    float tmp1 = (m_Value[3][2] * m_Value[1][3]) - (m_Value[1][2] * m_Value[3][3]);
    float tmp2 = (m_Value[0][1] * m_Value[2][2]) - (m_Value[2][1] * m_Value[0][2]);
    float tmp3 = (m_Value[1][1] * m_Value[3][2]) - (m_Value[3][1] * m_Value[1][2]);
    float tmp4 = (m_Value[2][1] * m_Value[0][3]) - (m_Value[0][1] * m_Value[2][3]);
    float tmp5 = (m_Value[3][1] * m_Value[1][3]) - (m_Value[1][1] * m_Value[3][3]);
    res0.x = (m_Value[2][1] * tmp1) - (m_Value[2][3] * tmp3) - (m_Value[2][2] * tmp5);
    res0.y = (m_Value[3][1] * tmp0) - (m_Value[3][3] * tmp2) - (m_Value[3][2] * tmp4);
    res0.z = (m_Value[0][3] * tmp3) + (m_Value[0][2] * tmp5) - (m_Value[0][1] * tmp1);
    res0.w = (m_Value[1][3] * tmp2) + (m_Value[1][2] * tmp4) - (m_Value[1][1] * tmp0);
    const float det =
        (m_Value[0][0] * res0.x) +
        (m_Value[1][0] * res0.y) +
        (m_Value[2][0] * res0.z) +
        (m_Value[3][0] * res0.w);
    const float detInv = (det != 0.0f) ? 1.0f / det : 1.0f;

    res1.x = m_Value[2][0] * tmp1;
    res1.y = m_Value[3][0] * tmp0;
    res1.z = m_Value[0][0] * tmp1;
    res1.w = m_Value[1][0] * tmp0;
    res3.x = m_Value[2][0] * tmp3;
    res3.y = m_Value[3][0] * tmp2;
    res3.z = m_Value[0][0] * tmp3;
    res3.w = m_Value[1][0] * tmp2;
    res2.x = m_Value[2][0] * tmp5;
    res2.y = m_Value[3][0] * tmp4;
    res2.z = m_Value[0][0] * tmp5;
    res2.w = m_Value[1][0] * tmp4;
    tmp0 = (m_Value[2][0] * m_Value[0][1]) - (m_Value[0][0] * m_Value[2][1]);
    tmp1 = (m_Value[3][0] * m_Value[1][1]) - (m_Value[1][0] * m_Value[3][1]);
    tmp2 = (m_Value[2][0] * m_Value[0][3]) - (m_Value[0][0] * m_Value[2][3]);
    tmp3 = (m_Value[3][0] * m_Value[1][3]) - (m_Value[1][0] * m_Value[3][3]);
    tmp4 = (m_Value[2][0] * m_Value[0][2]) - (m_Value[0][0] * m_Value[2][2]);
    tmp5 = (m_Value[3][0] * m_Value[1][2]) - (m_Value[1][0] * m_Value[3][2]);
    res2.x = (m_Value[2][3] * tmp1) - (m_Value[2][1] * tmp3) + res2.x;
    res2.y = (m_Value[3][3] * tmp0) - (m_Value[3][1] * tmp2) + res2.y;
    res2.z = (m_Value[0][1] * tmp3) - (m_Value[0][3] * tmp1) - res2.z;
    res2.w = (m_Value[1][1] * tmp2) - (m_Value[1][3] * tmp0) - res2.w;
    res3.x = (m_Value[2][1] * tmp5) - (m_Value[2][2] * tmp1) + res3.x;
    res3.y = (m_Value[3][1] * tmp4) - (m_Value[3][2] * tmp0) + res3.y;
    res3.z = (m_Value[0][2] * tmp1) - (m_Value[0][1] * tmp5) - res3.z;
    res3.w = (m_Value[1][2] * tmp0) - (m_Value[1][1] * tmp4) - res3.w;
    res1.x = (m_Value[2][2] * tmp3) - (m_Value[2][3] * tmp5) - res1.x;
    res1.y = (m_Value[3][2] * tmp2) - (m_Value[3][3] * tmp4) - res1.y;
    res1.z = (m_Value[0][3] * tmp5) - (m_Value[0][2] * tmp3) + res1.z;
    res1.w = (m_Value[1][3] * tmp4) - (m_Value[1][2] * tmp2) + res1.w;

    RMtx44 inv;
    inv.m_Value[0][0] = detInv * res0.x;
    inv.m_Value[0][1] = detInv * res0.y;
    inv.m_Value[0][2] = detInv * res0.z;
    inv.m_Value[0][3] = detInv * res0.w;

    inv.m_Value[1][0] = detInv * res1.x;
    inv.m_Value[1][1] = detInv * res1.y;
    inv.m_Value[1][2] = detInv * res1.z;
    inv.m_Value[1][3] = detInv * res1.w;

    inv.m_Value[2][0] = detInv * res2.x;
    inv.m_Value[2][1] = detInv * res2.y;
    inv.m_Value[2][2] = detInv * res2.z;
    inv.m_Value[2][3] = detInv * res2.w;

    inv.m_Value[3][0] = detInv * res3.x;
    inv.m_Value[3][1] = detInv * res3.y;
    inv.m_Value[3][2] = detInv * res3.z;
    inv.m_Value[3][3] = detInv * res3.w;

    return inv;
}

//-----------------------------------------------------------------------------
//! @brief 変換行列（座標を左から掛けるタイプ）を設定します。
//-----------------------------------------------------------------------------
RMtx44& RMtx44::SetTransform(
    const RVec3& scale,
    const RVec3& rotate,
    const RVec3& translate
)
{
    const float sinx = static_cast<float>(sin(rotate.x * R_M_DEG_TO_RAD));
    const float siny = static_cast<float>(sin(rotate.y * R_M_DEG_TO_RAD));
    const float sinz = static_cast<float>(sin(rotate.z * R_M_DEG_TO_RAD));
    const float cosx = static_cast<float>(cos(rotate.x * R_M_DEG_TO_RAD));
    const float cosy = static_cast<float>(cos(rotate.y * R_M_DEG_TO_RAD));
    const float cosz = static_cast<float>(cos(rotate.z * R_M_DEG_TO_RAD));

    // SRT
    m_Value[0][0] = (cosy * cosz) * scale.x;
    m_Value[0][1] = (cosy * sinz) * scale.x;
    m_Value[0][2] = (-siny      ) * scale.x;

    m_Value[1][0] = (sinx * siny * cosz - cosx * sinz) * scale.y;
    m_Value[1][1] = (sinx * siny * sinz + cosx * cosz) * scale.y;
    m_Value[1][2] = (sinx * cosy                     ) * scale.y;

    m_Value[2][0] = (cosx * siny * cosz + sinx * sinz) * scale.z;
    m_Value[2][1] = (cosx * siny * sinz - sinx * cosz) * scale.z;
    m_Value[2][2] = (cosx * cosy                     ) * scale.z;

    m_Value[3][0] = translate.x;
    m_Value[3][1] = translate.y;
    m_Value[3][2] = translate.z;

    m_Value[0][3] = m_Value[1][3] = m_Value[2][3] = 0.0f;
    m_Value[3][3] = 1.0f;

    return *this;
}

//-----------------------------------------------------------------------------
//! @brief 変換行列（座標を左から掛けるタイプ）を設定します。
//-----------------------------------------------------------------------------
RMtx44& RMtx44::SetTransformQuat(
    const RVec3& scale,
    const RVec4& rotate,
    const RVec3& translate
)
{
    const float qx = rotate.x;
    const float qy = rotate.y;
    const float qz = rotate.z;
    const float qw = rotate.w;

    m_Value[0][0] = (1.0f - 2.0f * qy * qy - 2.0f * qz * qz) * scale.x;
    m_Value[0][1] = (2.0f * qx * qy + 2.0f * qw * qz       ) * scale.x;
    m_Value[0][2] = (2.0f * qx * qz - 2.0f * qw * qy       ) * scale.x;

    m_Value[1][0] = (2.0f * qx * qy - 2.0f * qw * qz       ) * scale.y;
    m_Value[1][1] = (1.0f - 2.0f * qx * qx - 2.0f * qz * qz) * scale.y;
    m_Value[1][2] = (2.0f * qy * qz + 2.0f * qw * qx       ) * scale.y;

    m_Value[2][0] = (2.0f * qx * qz + 2.0f * qw * qy       ) * scale.z;
    m_Value[2][1] = (2.0f * qy * qz - 2.0f * qw * qx       ) * scale.z;
    m_Value[2][2] = (1.0f - 2.0f * qx * qx - 2.0f * qy * qy) * scale.z;

    m_Value[3][0] = translate.x;
    m_Value[3][1] = translate.y;
    m_Value[3][2] = translate.z;

    m_Value[0][3] = m_Value[1][3] = m_Value[2][3] = 0.0f;
    m_Value[3][3] = 1.0f;

    return *this;
}

//-----------------------------------------------------------------------------
//! @brief 回転を取得します。行列は座標を左から掛けるタイプで、正規化されている必要があります。
//-----------------------------------------------------------------------------
RVec3 RMtx44::GetRotate() const
{
    const float PI_2 = static_cast<float>(R_M_PI) / 2.0f;
    const float cosEpsilon = 0.00001f;

    RVec3 rotate;

    float syr = -m_Value[0][2];
    float cyr2 = 1.0f - syr * syr;
    if (cyr2 < 0.0f)
    {
        cyr2 = 0.0f;
    }
    float cyr = static_cast<float>(sqrt(cyr2));
    if (cyr < cosEpsilon)
    {
        rotate.x = atan2(-m_Value[2][1], m_Value[1][1]);
        rotate.y = (syr > 0.0f) ? PI_2 : -PI_2;
        rotate.z = 0.0f;
    }
    else
    {
        rotate.x = atan2(m_Value[1][2], m_Value[2][2]);
        rotate.y = atan2(syr, cyr);
        rotate.z = atan2(m_Value[0][1], m_Value[0][0]);
    }
    rotate.x *= static_cast<float>(R_M_RAD_TO_DEG);
    rotate.y *= static_cast<float>(R_M_RAD_TO_DEG);
    rotate.z *= static_cast<float>(R_M_RAD_TO_DEG);

    return rotate;
}

//-----------------------------------------------------------------------------
//! @brief 変換を取得します。行列は座標を左から掛けるタイプになっている必要があります。
//-----------------------------------------------------------------------------
void RMtx44::GetTransform(RVec3& scale, RVec3& rotate, RVec3& translate) const
{
    RMtx44 mtx = *this; // copy !

    //-----------------------------------------------------------------------------
    // get scale & normalize matrix
    for (int iaxis = 0; iaxis < 3; ++iaxis)
    {
        float f0 = mtx[iaxis][0];
        float f1 = mtx[iaxis][1];
        float f2 = mtx[iaxis][2];
        scale[iaxis] = static_cast<float>(sqrt(f0 * f0 + f1 * f1 + f2 * f2));
        if (!RIsSame(scale[iaxis], 0.0f))
        {
            if (iaxis == 2) // added (2007/03/01)
            {
                RVec3 ax(mtx[0][0], mtx[0][1], mtx[0][2]);
                RVec3 ay(mtx[1][0], mtx[1][1], mtx[1][2]);
                RVec3 cz = ax ^ ay;
                if (RVec3(f0, f1, f2) * cz < 0.0f)
                {
                    scale[iaxis] = -scale[iaxis];
                }
            }
            float d = 1.0f / scale[iaxis];
            mtx[iaxis][0] *= d;
            mtx[iaxis][1] *= d;
            mtx[iaxis][2] *= d;
        }
    }

    //-----------------------------------------------------------------------------
    // get rotate (xyz order)
    rotate = mtx.GetRotate();

    //-----------------------------------------------------------------------------
    // get translate
    translate.x = mtx[3][0];
    translate.y = mtx[3][1];
    translate.z = mtx[3][2];
}

//-----------------------------------------------------------------------------
//! @brief float 型 4 x 4 行列の 3 x 4 成分を出力します。
//-----------------------------------------------------------------------------
void RMtx44::Out34(std::ostream& os, const int tabCount) const
{
    for (int row = 0; row < 3; ++row)
    {
        os << RTab(tabCount)
           << m_Value[row][0] << ' '
           << m_Value[row][1] << ' '
           << m_Value[row][2] << ' '
           << m_Value[row][3] << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定した値に近くて回転行列が同じになるオイラー回転を取得します。
//!        回転順序が XYZ の場合に対応しています。角度の単位は degree です。
//!
//! @param[in] dst 近づけるオイラー回転です。
//!
//! @return 指定した値に近いオイラー回転を返します。
//-----------------------------------------------------------------------------
RVec3 RVec3::GetClosestRotationXyz(const RVec3& dst) const
{
    //-----------------------------------------------------------------------------
    // 角度差が一定値未満なら現在の値を返します。
    if (std::abs(dst.x - x) < 90.0f &&
        std::abs(dst.y - y) < 90.0f &&
        std::abs(dst.z - z) < 90.0f)
    {
        return *this;
    }

    //-----------------------------------------------------------------------------
    // 角度が (-180,180] の範囲に収まるようにシフトします。
    const float sx = RShiftAngle180(nullptr, x);
    const float sy = RShiftAngle180(nullptr, y);
    const float sz = RShiftAngle180(nullptr, z);
    float shiftX = 0.0f;
    float shiftY = 0.0f;
    float shiftZ = 0.0f;
    const float dx = RShiftAngle180(&shiftX, dst.x);
    const float dy = RShiftAngle180(&shiftY, dst.y);
    const float dz = RShiftAngle180(&shiftZ, dst.z);

    //-----------------------------------------------------------------------------
    // シフトした角度差が一定値以上なら回転行列が同じになるオイラー回転について角度差を比較します。
    //
    // sin( a +/- 180) = -sin(a)
    // cos( a +/- 180) = -cos(a)
    // sin(-a +/- 180) =  sin(a)
    // cos(-a +/- 180) = -cos(a)
    //
    // XYZ 回転行列の [0][2] = -sin(ry) なので、sin(ry) の符号が変わらないようにします。
    float mx = sx;
    float my = sy;
    float mz = sz;
    const float ax = std::abs(dx - mx);
    const float ay = std::abs(dy - my);
    const float az = std::abs(dz - mz);
    if (ax >= 90.0f ||
        ay >= 90.0f ||
        az >= 90.0f)
    {
        const float tests[][3] =
        {
            { sx - 360.0f,  sy         , sz          },
            { sx + 360.0f,  sy         , sz          },
            { sx         ,  sy - 360.0f, sz          },
            { sx         ,  sy + 360.0f, sz          },
            { sx         ,  sy         , sz - 360.0f },
            { sx         ,  sy         , sz + 360.0f },

            { sx - 360.0f,  sy - 360.0f, sz          },
            { sx - 360.0f,  sy + 360.0f, sz          },
            { sx + 360.0f,  sy - 360.0f, sz          },
            { sx + 360.0f,  sy + 360.0f, sz          },
            { sx         ,  sy - 360.0f, sz - 360.0f },
            { sx         ,  sy - 360.0f, sz + 360.0f },
            { sx         ,  sy + 360.0f, sz - 360.0f },
            { sx         ,  sy + 360.0f, sz + 360.0f },
            { sx - 360.0f,  sy         , sz - 360.0f },
            { sx - 360.0f,  sy         , sz + 360.0f },
            { sx + 360.0f,  sy         , sz - 360.0f },
            { sx + 360.0f,  sy         , sz + 360.0f },

            { sx - 360.0f,  sy - 360.0f, sz - 360.0f },
            { sx - 360.0f,  sy - 360.0f, sz + 360.0f },
            { sx - 360.0f,  sy + 360.0f, sz - 360.0f },
            { sx - 360.0f,  sy + 360.0f, sz + 360.0f },
            { sx + 360.0f,  sy - 360.0f, sz - 360.0f },
            { sx + 360.0f,  sy - 360.0f, sz + 360.0f },
            { sx + 360.0f,  sy + 360.0f, sz - 360.0f },
            { sx + 360.0f,  sy + 360.0f, sz + 360.0f },

            { sx - 180.0f, -180.0f - sy, sz - 180.0f },
            { sx - 180.0f, -180.0f - sy, sz + 180.0f },
            { sx + 180.0f, -180.0f - sy, sz - 180.0f },
            { sx + 180.0f, -180.0f - sy, sz + 180.0f },
            { sx - 180.0f,  180.0f - sy, sz - 180.0f },
            { sx - 180.0f,  180.0f - sy, sz + 180.0f },
            { sx + 180.0f,  180.0f - sy, sz - 180.0f },
            { sx + 180.0f,  180.0f - sy, sz + 180.0f },
        };

        float diffMin = ax + ay + az;
        const int testCount = sizeof(tests) / sizeof(tests[0]);
        for (int testIdx = 0; testIdx < testCount; ++testIdx)
        {
            const float* test = tests[testIdx];
            const float diff =
                std::abs(dx - test[0]) +
                std::abs(dy - test[1]) +
                std::abs(dz - test[2]);
            if (diff < diffMin)
            {
                mx = test[0];
                my = test[1];
                mz = test[2];
                diffMin = diff;
            }
        }
    }
    return RVec3(mx - shiftX, my - shiftY, mz - shiftZ);
}

//-----------------------------------------------------------------------------
//! @brief 指定した値に近くて回転行列が同じになるオイラー回転を取得します。
//!        回転順序が ZXY の場合に対応しています。角度の単位は degree です。
//!
//! @param[in] dst 近づけるオイラー回転です。
//!
//! @return 指定した値に近いオイラー回転を返します。
//-----------------------------------------------------------------------------
RVec3 RVec3::GetClosestRotationZxy(const RVec3& dst) const
{
    //-----------------------------------------------------------------------------
    // 角度差が一定値未満なら現在の値を返します。
    if (std::abs(dst.x - x) < 90.0f &&
        std::abs(dst.y - y) < 90.0f &&
        std::abs(dst.z - z) < 90.0f)
    {
        return *this;
    }

    //-----------------------------------------------------------------------------
    // 角度が (-180,180] の範囲に収まるようにシフトします。
    const float sx = RShiftAngle180(nullptr, x);
    const float sy = RShiftAngle180(nullptr, y);
    const float sz = RShiftAngle180(nullptr, z);
    float shiftX = 0.0f;
    float shiftY = 0.0f;
    float shiftZ = 0.0f;
    const float dx = RShiftAngle180(&shiftX, dst.x);
    const float dy = RShiftAngle180(&shiftY, dst.y);
    const float dz = RShiftAngle180(&shiftZ, dst.z);

    //-----------------------------------------------------------------------------
    // シフトした角度差が一定値以上なら回転行列が同じになるオイラー回転について角度差を比較します。
    //
    // sin( a +/- 180) = -sin(a)
    // cos( a +/- 180) = -cos(a)
    // sin(-a +/- 180) =  sin(a)
    // cos(-a +/- 180) = -cos(a)
    //
    // ZXY 回転行列の [1][2] = -sin(rx) なので、sin(rx) の符号が変わらないようにします。
    float mx = sx;
    float my = sy;
    float mz = sz;
    const float ax = std::abs(dx - mx);
    const float ay = std::abs(dy - my);
    const float az = std::abs(dz - mz);
    if (ax >= 90.0f ||
        ay >= 90.0f ||
        az >= 90.0f)
    {
        const float tests[][3] =
        {
            { sx - 360.0f,  sy         , sz          },
            { sx + 360.0f,  sy         , sz          },
            { sx         ,  sy - 360.0f, sz          },
            { sx         ,  sy + 360.0f, sz          },
            { sx         ,  sy         , sz - 360.0f },
            { sx         ,  sy         , sz + 360.0f },

            { sx - 360.0f,  sy - 360.0f, sz          },
            { sx - 360.0f,  sy + 360.0f, sz          },
            { sx + 360.0f,  sy - 360.0f, sz          },
            { sx + 360.0f,  sy + 360.0f, sz          },
            { sx         ,  sy - 360.0f, sz - 360.0f },
            { sx         ,  sy - 360.0f, sz + 360.0f },
            { sx         ,  sy + 360.0f, sz - 360.0f },
            { sx         ,  sy + 360.0f, sz + 360.0f },
            { sx - 360.0f,  sy         , sz - 360.0f },
            { sx - 360.0f,  sy         , sz + 360.0f },
            { sx + 360.0f,  sy         , sz - 360.0f },
            { sx + 360.0f,  sy         , sz + 360.0f },

            { sx - 360.0f,  sy - 360.0f, sz - 360.0f },
            { sx - 360.0f,  sy - 360.0f, sz + 360.0f },
            { sx - 360.0f,  sy + 360.0f, sz - 360.0f },
            { sx - 360.0f,  sy + 360.0f, sz + 360.0f },
            { sx + 360.0f,  sy - 360.0f, sz - 360.0f },
            { sx + 360.0f,  sy - 360.0f, sz + 360.0f },
            { sx + 360.0f,  sy + 360.0f, sz - 360.0f },
            { sx + 360.0f,  sy + 360.0f, sz + 360.0f },

            { -180.0f - sx, sy - 180.0f, sz - 180.0f },
            { -180.0f - sx, sy - 180.0f, sz + 180.0f },
            { -180.0f - sx, sy + 180.0f, sz - 180.0f },
            { -180.0f - sx, sy + 180.0f, sz + 180.0f },
            {  180.0f - sx, sy - 180.0f, sz - 180.0f },
            {  180.0f - sx, sy - 180.0f, sz + 180.0f },
            {  180.0f - sx, sy + 180.0f, sz - 180.0f },
            {  180.0f - sx, sy + 180.0f, sz + 180.0f },
        };

        float diffMin = ax + ay + az;
        const int testCount = sizeof(tests) / sizeof(tests[0]);
        for (int testIdx = 0; testIdx < testCount; ++testIdx)
        {
            const float* test = tests[testIdx];
            const float diff =
                std::abs(dx - test[0]) +
                std::abs(dy - test[1]) +
                std::abs(dz - test[2]);
            if (diff < diffMin)
            {
                mx = test[0];
                my = test[1];
                mz = test[2];
                diffMin = diff;
            }
        }
    }
    return RVec3(mx - shiftX, my - shiftY, mz - shiftZ);
}

//-----------------------------------------------------------------------------
//! @brief 座標の重心を返します。
//-----------------------------------------------------------------------------
RVec3 RGetCenterOfGravity(const RVec3Array& poss)
{
    RVec3 centerPos(0.0f, 0.0f, 0.0f);
    const int posCount = static_cast<int>(poss.size());
    if (posCount > 0)
    {
        for (int ipos = 0; ipos < posCount; ++ipos)
        {
            centerPos += poss[ipos];
        }
        centerPos *= 1.0f / static_cast<float>(posCount);
    }
    return centerPos;
}

//-----------------------------------------------------------------------------
//! @brief 頂点座標と UV から接線と従法線を計算します。
//!        法線がゼロベクトルでなければ、接線と法線および従法線と法線が
//!        それぞれ直交するように補正します（接線と従法線は直交するとは限りません）。
//-----------------------------------------------------------------------------
void RCalcTangent(
    RVec3& tangent,
    RVec3& binormal,
    const RVec3& pos0,
    const RVec3& pos1,
    const RVec3& pos2,
    const RVec2& uv0,
    const RVec2& uv1,
    const RVec2& uv2,
    const RVec3& normal
)
{
    // pos0 = uv0.x * tangent + uv0.y * binormal
    // pos1 = uv1.x * tangent + uv1.y * binormal
    // pos2 = uv2.x * tangent + uv2.y * binormal

    // pos1 - pos0 = (uv1.x - uv0.x) * tangent + (uv1.y - uv0.y) * binormal
    // pos2 - pos0 = (uv2.x - uv0.x) * tangent + (uv2.y - uv0.y) * binormal

    // p1 = u1 * tx + v1 * bx (p1 = pos1.x - pos0.x, u1 = uv1.x - uv0.x, v1 = uv1.y - uv0.y)
    // p2 = u2 * tx + v2 * bx (p2 = pos2.x - pos0.x, u2 = uv2.x - uv0.x, v2 = uv2.y - uv0.y)

    // p1 * v2 = u1 * v2 * tx + v1 * v2 * bx (A)
    // p2 * v1 = u2 * v1 * tx + v1 * v2 * bx (B)
    // (A) - (B)
    // - (p2 * v1 - p1 * v2) = (u1 * v2 - u2 * v1) * tx

    // p1 * u2 = u1 * u2 * tx + u2 * v1 * bx (C)
    // p2 * u1 = u1 * u2 * tx + u1 * v2 * bx (D)
    // (D) - (C)
    // - (p1 * u2 - p2 * u1) = (u1 * v2 - u2 * v1) * bx

    const float u1 = uv1.x - uv0.x, v1 = uv1.y - uv0.y;
    const float u2 = uv2.x - uv0.x, v2 = uv2.y - uv0.y;
    const float cp = u1 * v2 - u2 * v1;
    if (cp != 0.0f)
    {
        const RVec3 dp1 = pos1 - pos0;
        const RVec3 dp2 = pos2 - pos0;
        tangent = RVec3(
            -(dp2.x * v1 - dp1.x * v2) / cp,
            -(dp2.y * v1 - dp1.y * v2) / cp,
            -(dp2.z * v1 - dp1.z * v2) / cp);
        binormal = RVec3(
            -(dp1.x * u2 - dp2.x * u1) / cp,
            -(dp1.y * u2 - dp2.y * u1) / cp,
            -(dp1.z * u2 - dp2.z * u1) / cp);
        if (normal != RVec3::kZero)
        {
            RVec3 vb = normal ^ tangent;
            tangent = vb ^ normal;
            RVec3 vt = binormal ^ normal;
            binormal = normal ^ vt;
        }
    }
    else
    {
        if (normal == RVec3::kZero)
        {
            tangent  = RVec3::kXAxis;
            binormal = RVec3::kYAxis;
        }
        else
        {
            if (normal.IsEquivalent(RVec3::kYAxis) ||
                normal.IsEquivalent(RVec3::kYNegAxis))
            {
                tangent = RVec3::kZNegAxis ^ normal;
            }
            else
            {
                tangent = RVec3::kYAxis ^ normal;
            }
            binormal = normal ^ tangent;
            tangent = binormal ^ normal;
        }
    }
    tangent.Normalize();
    binormal.Normalize();
}

//-----------------------------------------------------------------------------
//! @brief 接線、従法線、法線が右手座標系（UV のワインド順序が時計回り、前向き）
//!        なら true を返します。
//-----------------------------------------------------------------------------
bool RIsRightHandTangent(const RVec3& tangent, const RVec3& binormal, const RVec3& normal)
{
    return (tangent ^ binormal) * normal >= 0.0f;
}

//-----------------------------------------------------------------------------
//! @brief RFloatArray から許容値付きで値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int RFindValueInArray(
    const RFloatArray& array,
    const float& value,
    const float tolerance,
    const int startIdx
)
{
    const int count = static_cast<int>(array.size());
    for (int index = startIdx; index < count; ++index)
    {
        if (RIsSame(array[index], value, tolerance))
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief RVec2Array から許容値付きで値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int RFindValueInArray(
    const RVec2Array& array,
    const RVec2& value,
    const float tolerance,
    const int startIdx
)
{
    const int count = static_cast<int>(array.size());
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index].IsEquivalent(value, tolerance))
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief RVec3Array から許容値付きで値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int RFindValueInArray(
    const RVec3Array& array,
    const RVec3& value,
    const float tolerance,
    const int startIdx
)
{
    const int count = static_cast<int>(array.size());
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index].IsEquivalent(value, tolerance))
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief RVec4Array から許容値付きで値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int RFindValueInArray(
    const RVec4Array& array,
    const RVec4& value,
    const float tolerance,
    const int startIdx
)
{
    const int count = static_cast<int>(array.size());
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index].IsEquivalent(value, tolerance))
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief RFloatArray の値の範囲（最小値と最大値）を取得します。
//-----------------------------------------------------------------------------
void RGetArrayRange(
    float& valueMin,
    float& valueMax,
    const RFloatArray& array
)
{
    const int valCount = static_cast<int>(array.size());
    if (valCount == 0)
    {
        valueMin = 0.0f;
        valueMax = 0.0f;
        return;
    }

    valueMin = static_cast<float>(LONG_MAX);
    valueMax = static_cast<float>(LONG_MIN);
    for (int ival = 0; ival < valCount; ++ival)
    {
        const float val = array[ival];
        if (val < valueMin) valueMin = val;
        if (val > valueMax) valueMax = val;
    }
}

//-----------------------------------------------------------------------------
//! @brief RVec2Array の値の範囲（最小値と最大値）を取得します。
//-----------------------------------------------------------------------------
void RGetArrayRange(
    RVec2& valueMin,
    RVec2& valueMax,
    const RVec2Array& array
)
{
    const int vecCount = static_cast<int>(array.size());
    if (vecCount == 0)
    {
        valueMin.x = valueMin.y = 0.0f;
        valueMax.x = valueMax.y = 0.0f;
        return;
    }

    valueMin.x = valueMin.y = static_cast<float>(LONG_MAX);
    valueMax.x = valueMax.y = static_cast<float>(LONG_MIN);

    for (int ivec = 0; ivec < vecCount; ++ivec)
    {
        const RVec2& vec = array[ivec];
        if (vec.x < valueMin.x) valueMin.x = vec.x;
        if (vec.y < valueMin.y) valueMin.y = vec.y;
        if (vec.x > valueMax.x) valueMax.x = vec.x;
        if (vec.y > valueMax.y) valueMax.y = vec.y;
    }
}

//-----------------------------------------------------------------------------
//! @brief RVec3Array の値の範囲（最小値と最大値）を取得します。
//-----------------------------------------------------------------------------
void RGetArrayRange(
    RVec3& valueMin,
    RVec3& valueMax,
    const RVec3Array& array
)
{
    const int vecCount = static_cast<int>(array.size());
    if (vecCount == 0)
    {
        valueMin.x = valueMin.y = valueMin.z = 0.0f;
        valueMax.x = valueMax.y = valueMax.z = 0.0f;
        return;
    }

    valueMin.x = valueMin.y = valueMin.z = static_cast<float>(LONG_MAX);
    valueMax.x = valueMax.y = valueMax.z = static_cast<float>(LONG_MIN);

    for (int ivec = 0; ivec < vecCount; ++ivec)
    {
        const RVec3& vec = array[ivec];
        if (vec.x < valueMin.x) valueMin.x = vec.x;
        if (vec.y < valueMin.y) valueMin.y = vec.y;
        if (vec.z < valueMin.z) valueMin.z = vec.z;
        if (vec.x > valueMax.x) valueMax.x = vec.x;
        if (vec.y > valueMax.y) valueMax.y = vec.y;
        if (vec.z > valueMax.z) valueMax.z = vec.z;
    }
}

//-----------------------------------------------------------------------------
//! @brief RVec4Array の値の範囲（最小値と最大値）を取得します。
//-----------------------------------------------------------------------------
void RGetArrayRange(
    RVec4& valueMin,
    RVec4& valueMax,
    const RVec4Array& array
)
{
    const int vecCount = static_cast<int>(array.size());
    if (vecCount == 0)
    {
        valueMin.x = valueMin.y = valueMin.z = valueMin.w = 0.0f;
        valueMax.x = valueMax.y = valueMax.z = valueMax.w = 0.0f;
        return;
    }

    valueMin.x = valueMin.y = valueMin.z = valueMin.w = static_cast<float>(LONG_MAX);
    valueMax.x = valueMax.y = valueMax.z = valueMax.w = static_cast<float>(LONG_MIN);

    for (int ivec = 0; ivec < vecCount; ++ivec)
    {
        const RVec4& vec = array[ivec];
        if (vec.x < valueMin.x) valueMin.x = vec.x;
        if (vec.y < valueMin.y) valueMin.y = vec.y;
        if (vec.z < valueMin.z) valueMin.z = vec.z;
        if (vec.w < valueMin.w) valueMin.w = vec.w;
        if (vec.x > valueMax.x) valueMax.x = vec.x;
        if (vec.y > valueMax.y) valueMax.y = vec.y;
        if (vec.z > valueMax.z) valueMax.z = vec.z;
        if (vec.w > valueMax.w) valueMax.w = vec.w;
    }
}

//-----------------------------------------------------------------------------
//! @brief RVec3Array の Z 成分がすべて 0（値の絶対値が許容値未満）なら
//!        true を返します。
//-----------------------------------------------------------------------------
bool RIsAllZeroZ(const RVec3Array& array, const float tolerance)
{
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        if (!RIsSame(array[index].z, 0.0f, tolerance))
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief RRgbaColorArray のアルファ成分がすべて最大値なら true を返します。
//-----------------------------------------------------------------------------
bool RIsAllAlphaMax(const RRgbaColorArray& array)
{
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        if (array[index].a != RRgbaColor::COL_MAX)
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief RFloatArray の値の絶対値の最大値を返します。
//-----------------------------------------------------------------------------
float RGetArrayMaxAbsValue(const RFloatArray& array)
{
    const int count = static_cast<int>(array.size());
    if (count == 0)
    {
        return 0.0f;
    }
    else
    {
        float maxAbsVal = static_cast<float>(LONG_MIN);
        for (int index = 0; index < count; ++index)
        {
            const float absVal = RAbs(array[index]);
            if (absVal > maxAbsVal)
            {
                maxAbsVal = absVal;
            }
        }
        return maxAbsVal;
    }
}

//-----------------------------------------------------------------------------
//! @brief 有向バウンディングボックスの jocobi 計算です。
//!        NUMERICAL RECIPES in C より
//!        http://www.library.cornell.edu/nr/bookcpdf/c11-1.pdf
//-----------------------------------------------------------------------------
static void JacobiRotate(float a[3][3], float s, float tau, int i, int j, int k, int l)
{
    float h, g;
    g = a[i][j];
    h = a[k][l];
    a[i][j] = g - s * ( h + g *tau );
    a[k][l] = h + s * ( g - h *tau );
}

// return true is solved
static bool JacobiCalc(float a[3][3], float v[3][3], float d[3])
{
    int n = 3;
    int i, j, iq, ip;
    float tresh, theta, tau, t, sm, s, h, g, c, b[3], z[3];

    for (ip = 0; ip < n; ip++)
    {
        for( iq = 0; iq < n; iq++ ) v[ip][iq] = 0.0f;
        v[ip][ip] = 1.0f;
    }
    for (ip = 0; ip < n; ip++)
    {
        b[ip] = d[ip] = a[ip][ip];
        z[ip] = 0.0f;
    }
    for (i = 0; i < 50; i++)
    {
        sm = 0.0f;
        for (ip = 0; ip < n - 1; ip++)
        {
            for (iq = ip + 1; iq < n; iq++) sm += static_cast<float>(fabs(a[ip][iq]));
        }

        if ( RIsSame( sm, 0.0f ) ) return true;
        if (i < 3) tresh = 0.2f * sm / ( n * n );
        else tresh = 0.0f;
        for (ip = 0; ip < n - 1; ip++)
        {
            for (iq = ip + 1; iq < n; iq++)
            {
                g = 100.0f * static_cast<float>(fabs( a[ip][iq] ));
                if (i > 3 && RIsSame( ( fabs( d[ip] ) + g ), fabs( d[ip]) )
                    && RIsSame( ( fabs( d[iq] ) + g ), fabs( d[iq] ) ) ) a[ip][iq] = 0.0f;
                else if (fabs( a[ip][iq] ) > tresh)
                {
                    h = d[iq] - d[ip];
                    if ( RIsSame(( fabs( h ) + g ), fabs( h ))) t = a[ip][iq] / h;
                    else
                    {
                        theta = 0.5f * h / a[ip][iq];
                        t = 1.0f / ( static_cast<float>(fabs( theta )) + static_cast<float>(sqrt( 1.0f + theta * theta )) );
                        if (theta < 0.0f) t = -t;
                    }
                    c = 1.0f / static_cast<float>(sqrt( 1 + t * t ));
                    s = t * c;
                    tau = s / ( 1.0f + c );
                    h = t * a[ip][iq];
                    z[ip] -= h;
                    z[iq] += h;
                    d[ip] -= h;
                    d[iq] += h;
                    a[ip][iq] = 0.0f;

                    for (j = 0; j < ip; j++) JacobiRotate(a, s, tau, j, ip, j, iq);
                    for (j = ip + 1; j < iq; j++) JacobiRotate(a, s, tau, ip, j, j, iq);
                    for (j = iq + 1; j < n; j++) JacobiRotate(a, s, tau, ip, j, iq, j);
                    for (j = 0; j < n; j++) JacobiRotate(v, s, tau, j, ip, j, iq);
                }
            }
        }
        for (ip = 0; ip < n; ip++)
        {
            b[ip] += z[ip];
            d[ip] = b[ip];
            z[ip] = 0.0f;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 有向バウンディングボックスを PCA で計算します。
//!        http://www.nomuraz.com/denpa/prog005.htm
//!        COBBTree.cpp
//!        からの移植です。
//-----------------------------------------------------------------------------
void ROrientedBB::CalculateByPca(const RVec3Array& poss, const bool roundToZero)
{
    // 座標の数をチェック
    const int posCount = static_cast<int>(poss.size());
    if (posCount == 0)
    {
        Init();
        return;
    }

    // 平均位置 m を計算
    RVec3 m = RVec3(0.0f, 0.0f, 0.0f);
    for (int ipos = 0; ipos < posCount; ++ipos)
    {
        m += poss[ipos];
    }
    const float invPosCount = 1.0f / static_cast<float>(posCount);
    m *= invPosCount;

    // 共分散行列を作成
    float C11 = 0.0f, C22 = 0.0f, C33 = 0.0f, C12 = 0.0f, C13 = 0.0f, C23 = 0.0f;
    for (int ipos = 0; ipos < posCount; ++ipos)
    {
        const RVec3& pos = poss[ipos];
        C11 += (pos.x - m.x) * (pos.x - m.x);
        C22 += (pos.y - m.y) * (pos.y - m.y);
        C33 += (pos.z - m.z) * (pos.z - m.z);
        C12 += (pos.x - m.x) * (pos.y - m.y);
        C13 += (pos.x - m.x) * (pos.z - m.z);
        C23 += (pos.y - m.y) * (pos.z - m.z);
    }
    C11 *= invPosCount;
    C22 *= invPosCount;
    C33 *= invPosCount;
    C12 *= invPosCount;
    C13 *= invPosCount;
    C23 *= invPosCount;

    float cvMtx[3][3] =
    {
        { C11, C12, C13 },
        { C12, C22, C23 },
        { C13, C23, C33 },
    };

    // jacobi 法で固有値 & 固有ベクトルを算出
    float EigenVectors[3][3];
    float EigenValue[3];
    bool solved = JacobiCalc(cvMtx, EigenVectors, EigenValue);
    if (solved)
    {
        // 固有値を降順でソート
        struct SORT
        {
            int ID;
            float Value;
        } Sort[3] = { { 0, EigenValue[0] }, { 1, EigenValue[1] }, { 2, EigenValue[2] } };

        for (int j = 0; j < 2; ++j)
        {
            for (int i = 2; i > j; --i)
            {
                if (Sort[i - 1].Value < Sort[i].Value)
                {
                    SORT a = Sort[i];
                    Sort[i] = Sort[i - 1];
                    Sort[i - 1] = a;
                }
            }
        }

        for (int i = 0; i < 3; ++i)
        {
            m_AxisVecs[i] = RVec3(
                EigenVectors[0][Sort[i].ID],
                EigenVectors[1][Sort[i].ID],
                EigenVectors[2][Sort[i].ID]);
        }
    }
    else
    {
        // 解がなければ AABB に
        m_AxisVecs[0] = RVec3(1.0f, 0.0f, 0.0f);
        m_AxisVecs[1] = RVec3(0.0f, 1.0f, 0.0f);
        m_AxisVecs[2] = RVec3(0.0f, 0.0f, 1.0f);
    }

    // 境界ボックスを算出
    float XfMin[3], XfMax[3];
    XfMax[0] = XfMin[0] = m_AxisVecs[0] * poss[0];
    XfMax[1] = XfMin[1] = m_AxisVecs[1] * poss[0];
    XfMax[2] = XfMin[2] = m_AxisVecs[2] * poss[0];
    for (int ipos = 1; ipos < posCount; ++ipos)
    {
        const RVec3& pos = poss[ipos];
        float XfTemp;
        XfTemp = m_AxisVecs[0] * pos;
        XfMin[0] = min(XfMin[0], XfTemp);
        XfMax[0] = max(XfMax[0], XfTemp);

        XfTemp = m_AxisVecs[1] * pos;
        XfMin[1] = min(XfMin[1], XfTemp);
        XfMax[1] = max(XfMax[1], XfTemp);

        XfTemp = m_AxisVecs[2] * pos;
        XfMin[2] = min(XfMin[2], XfTemp);
        XfMax[2] = max(XfMax[2], XfTemp);
    }

    m_CenterPos =
        m_AxisVecs[0] * ((XfMax[0] + XfMin[0]) * 0.5f) +
        m_AxisVecs[1] * ((XfMax[1] + XfMin[1]) * 0.5f) +
        m_AxisVecs[2] * ((XfMax[2] + XfMin[2]) * 0.5f);

    m_Sizes[0] = XfMax[0] - XfMin[0];
    m_Sizes[1] = XfMax[1] - XfMin[1];
    m_Sizes[2] = XfMax[2] - XfMin[2];

    if (roundToZero)
    {
        m_CenterPos.SnapToZero();
        for (int iaxis = 0; iaxis < 3; ++iaxis)
        {
            m_AxisVecs[iaxis].SnapToZero();
            m_Sizes[iaxis] = RSnapToZero(m_Sizes[iaxis]);
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 有向バウンディングボックスを出力します。
//-----------------------------------------------------------------------------
void ROrientedBB::Out(std::ostream& os, const int tabCount) const
{
    const int& tc = tabCount;
    os << RTab(tc) << "<OrientedBoundingBox>" << R_ENDL;
    ROutVecAttr(os, tc + 1, "CenterPosition", m_CenterPos);
    os << RTab(tc + 1) << "<OrientationMatrix"
       <<   " M00=\"" << m_AxisVecs[0].x
       << "\" M01=\"" << m_AxisVecs[0].y
       << "\" M02=\"" << m_AxisVecs[0].z
       << "\" M10=\"" << m_AxisVecs[1].x
       << "\" M11=\"" << m_AxisVecs[1].y
       << "\" M12=\"" << m_AxisVecs[1].z
       << "\" M20=\"" << m_AxisVecs[2].x
       << "\" M21=\"" << m_AxisVecs[2].y
       << "\" M22=\"" << m_AxisVecs[2].z << "\" />" << R_ENDL;
    RVec3 sizeVec(m_Sizes[0], m_Sizes[1], m_Sizes[2]);
    ROutVecAttr(os, tc + 1, "Size", sizeVec);
    os << RTab(tc) << "</OrientedBoundingBox>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプションのアニメーションのフレーム数を更新します。
//-----------------------------------------------------------------------------
void RExpOpt::UpdateFrameCount()
{
    m_FrameCount = m_EndFrame - m_StartFrame + 1;
    if (m_FrameCount <= 0) // 通常は起こりません。
    {
        m_FrameCount = 1;
    }
    m_SubFrameCount = m_FrameCount * m_FramePrecision - (m_FramePrecision - 1);
    // 最終フレームより後は対象外なので (m_FramePrecision - 1) を引きます。

    //m_OutFrameCount = (m_LoopAnim && m_FrameCount >= 2) ?
    //  (m_FrameCount - 1) : m_FrameCount;
    m_OutFrameCount = m_FrameCount - 1;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルがバイナリ形式なら true を返します。
//-----------------------------------------------------------------------------
bool RExpOpt::IsBinaryFormat(const RExpOpt::FileType fileType) const
{
    //if (fileType == FMD) return false;
    //if (fileType == FVB) return false;
    return (m_IsBinaryFormat || fileType == FTX);
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイル名に追加するサフィックスを返します。
//-----------------------------------------------------------------------------
std::string RExpOpt::GetSuffix(const FileType fileType) const
{
    return (!UsesSeparatedFma()) ? ""     :
           (fileType == FCL    ) ? "_fcl" :
           (fileType == FTS    ) ? "_fts" :
           (fileType == FTP    ) ? "_ftp" :
           "";
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの拡張子を返します。
//!        コンフィグファイルの設定に従って fma と fcl / fts / ftp を切り替えます。
//-----------------------------------------------------------------------------
std::string RExpOpt::GetExtension(const RExpOpt::FileType fileType) const
{
    const std::string ext = (!m_UsesFclFtsFtp && IsFclFtsFtp(fileType)) ?
        "fma" : FileTypeExtStrs[fileType];
    return ext + (IsBinaryFormat(fileType) ? "b" : "a");
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルの設定に関係なくマテリアルアニメーションに関しては
//!        fcl / fts / ftp の形式の中間ファイルの拡張子を返します。
//-----------------------------------------------------------------------------
std::string RExpOpt::GetNoFmaExtension(const RExpOpt::FileType fileType) const
{
    const std::string ext = FileTypeExtStrs[fileType];
    return ext + (IsBinaryFormat(fileType) ? "b" : "a");
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのルート要素のバージョンを取得します。
//!
//! @return ルート要素のバージョンを返します。
//-----------------------------------------------------------------------------
std::string RExpOpt::GetRootElementVersion()
{
    return IntermediateFileVersion;
}

//-----------------------------------------------------------------------------
//! @brief 各中間ファイルのタイプ要素の名前を取得します。
//-----------------------------------------------------------------------------
std::string RExpOpt::GetTypeElementName(const FileType fileType) const
{
    if (!m_UsesFclFtsFtp && IsFclFtsFtp(fileType))
    {
        return "material_anim";
    }

    switch (fileType)
    {
    case FMD: return "model";
    case FSK: return "skeletal_anim";
    case FVB: return "bone_visibility_anim";
    case FCL: return "shader_param_anim";
    case FTS: return "shader_param_anim";
    case FTP: return "tex_pattern_anim";
    case FSH: return "shape_anim";
    case FSN: return "scene_anim";

    default:  return "unknown";
    }
}

//-----------------------------------------------------------------------------
//! @brief 各中間ファイルのタイプ要素のバージョンを取得します。
//-----------------------------------------------------------------------------
std::string RExpOpt::GetTypeElementVersion(const FileType fileType) const
{
    if (!m_UsesFclFtsFtp && IsFclFtsFtp(fileType))
    {
        return MaterialAnimElementVersion;
    }

    switch (fileType)
    {
    case FMD: return ModelElementVersion;
    case FSK: return SkeletalAnimElementVersion;
    case FVB: return BoneVisibilityAnimElementVersion;
    case FCL: return ShaderParamAnimElementVersion;
    case FTS: return ShaderParamAnimElementVersion;
    case FTP: return TexPatternAnimElementVersion;
    case FSH: return ShapeAnimElementVersion;
    case FSN: return SceneAnimElementVersion;

    default:  return IntermediateFileVersion;
    }
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプションのテクスチャカスタマイズをヒント情報から検索します（const 版）。
//!        見つからなければ NULL を返します。
//-----------------------------------------------------------------------------
const RExpOpt::RTexCustom* RExpOpt::FindTexCustom(const std::string& hint) const
{
    for (size_t iTc = 0; iTc < m_TexCustoms.size(); ++iTc)
    {
        const RTexCustom& texCustom = m_TexCustoms[iTc];
        if (texCustom.m_Hint == hint)
        {
            return &texCustom;
        }
    }
    return NULL;
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプションのテクスチャカスタマイズをヒント情報から検索します。
//!        見つからなければ NULL を返します。
//-----------------------------------------------------------------------------
RExpOpt::RTexCustom* RExpOpt::FindTexCustom(const std::string& hint)
{
    for (size_t iTc = 0; iTc < m_TexCustoms.size(); ++iTc)
    {
        RTexCustom& texCustom = m_TexCustoms[iTc];
        if (texCustom.m_Hint == hint)
        {
            return &texCustom;
        }
    }
    return NULL;
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプションからテクスチャのリニア変換フラグを取得します。
//-----------------------------------------------------------------------------
int RExpOpt::GetTexLinearFlag(const std::string& hint) const
{
    const RTexCustom* pTexCustom = FindTexCustom(hint);
    if (pTexCustom != NULL)
    {
        return pTexCustom->m_LinearFlag;
    }
    else // changed (2012/11/15)
    {
        if (hint == RImage::HINT_ALBEDO     ||
            hint == RImage::HINT_EMISSION   ||
            hint == RImage::HINT_SPECULAR   ||
            hint == RImage::HINT_REFLECTION)
        {
            return RImage::LINEAR_RGB;
        }
        else
        {
            return RImage::LINEAR_NONE;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief エクスポートオプションのコンフィグファイルを解析して一部のオプションを設定します。
//-----------------------------------------------------------------------------
RStatus RExpOpt::ParseConfigFile(const std::string& configPath)
{
    static const std::string TEX_CUSTOM_TAG_TOP = "texture_";

    //-----------------------------------------------------------------------------
    // 初期化します。
    m_TexCustoms.clear();

    //-----------------------------------------------------------------------------
    // ファイルをオープンします。
    std::ifstream ifs(configPath.c_str(), ios_base::in);
    if (!ifs)
    {
        return RStatus(RStatus::FAILURE, // RShowError: Cannot open file
            L"コンフィグファイルを開けません: {0}",
            "The config file cannot be opened: {0}", configPath);
    }

    //-----------------------------------------------------------------------------
    // 1 行ずつ解析します。
    while (!ifs.eof())
    {
        //-----------------------------------------------------------------------------
        // 1 行取得してタグと値に分解します。
        std::string lineBuf;
        getline(ifs, lineBuf);
        lineBuf = RTrimString(lineBuf);
        if (lineBuf.empty() || lineBuf[0] == '#')
        {
            continue;
        }

        const size_t iEqual = lineBuf.find("=");
        if (iEqual == std::string::npos)
        {
            continue;
        }
        const std::string tag = RTrimString(lineBuf.substr(0, iEqual));
        const std::string val = RDequoteString(lineBuf.substr(iEqual + 1));
        const bool isBool = (
            val == "true" || val == "false" ||
            val == "on"   || val == "off"   ||
            val == "1"    || val == "0");
        const bool boolVal = (val == "true" || val == "on" || val == "1");
        bool isValErr = false;
        const size_t iTexCustom = tag.find(TEX_CUSTOM_TAG_TOP);

        //-----------------------------------------------------------------------------
        // 各オプションを設定します。
        if (iTexCustom == 0)
        {
            const std::string hint = tag.substr(TEX_CUSTOM_TAG_TOP.size());
            const int linearFlag = RImage::GetLinearFlagFromOptString(val);
            if (!hint.empty() && linearFlag != RImage::LINEAR_INVALID)
            {
                RTexCustom* pTexCustom = FindTexCustom(hint);
                if (pTexCustom != NULL)
                {
                    pTexCustom->m_LinearFlag = linearFlag;
                }
                else
                {
                    m_TexCustoms.push_back(RTexCustom(hint, linearFlag));
                }
            }
            else
            {
                isValErr = true;
            }
        }
        else if (tag == "display_profile")
        {
            if (isBool) m_DisplaysProfile = boolVal;
            else isValErr = true;
        }
        else if (tag == "project_root")
        {
            m_ProjectRootPath = RTrimString(val);
        }
        else if (tag == "max_vertex_skinning_count")
        {
            const int maxVertexSkinningCount = atoi(val.c_str());
            if (1 <= maxVertexSkinningCount && maxVertexSkinningCount <= VERTEX_SKINNING_COUNT_MAX)
            {
                m_MaxVertexSkinningCount = maxVertexSkinningCount;
            }
            else
            {
                isValErr = true;
            }
        }
        else if (tag == "adjust_smooth_skinning")
        {
            if (isBool) m_AdjustsSmoothSkinning = boolVal;
            else isValErr = true;
        }
        else if (tag == "compress_ignoring_vertex_skinning_count")
        {
            m_CompressesIgnoringVertexSkinningCount = boolVal;
            isValErr = !isBool;
        }
        else if (tag == "optimize_primitive")
        {
            m_OptimizePrimitive =
                (val == "none"       ) ? OPTIMIZE_PRIMITIVE_NONE        :
                (val == "default"    ) ? OPTIMIZE_PRIMITIVE_DEFAULT     :
                (val == "force"      ) ? OPTIMIZE_PRIMITIVE_FORCE       :
                (val == "brute_force") ? OPTIMIZE_PRIMITIVE_BRUTE_FORCE :
                (val == "forsyth"    ) ? OPTIMIZE_PRIMITIVE_FORSYTH     :
                OPTIMIZE_PRIMITIVE_INVALID;
            isValErr = (m_OptimizePrimitive == OPTIMIZE_PRIMITIVE_INVALID);
        }
        else if (tag == "optimize_triangulation")
        {
            if      (val == "none" ) m_OptimizeTriangulation = OPTIMIZE_TRIANGULATION_NONE;
            else if (val == "angle") m_OptimizeTriangulation = OPTIMIZE_TRIANGULATION_ANGLE;
            else isValErr = true;
        }
        else if (tag == "quantization_analysis")
        {
            if (isBool) m_QuantizationAnalysis = boolVal;
            else isValErr = true;
        }
        else if (tag == "delete_near_vertex")
        {
            m_DeleteNearVertex = RConvertToSingleQuote(RTrimString(val));
        }
        else if (tag == "use_srgb_fetch")
        {
            if (isBool) m_UsesSrgbFetch = boolVal;
            else isValErr = true;
        }
        else if (tag == "normal_texture_format")
        {
            RStringArray tokens;
            const int tokenCount = RTokenizeString(tokens, val, ",");
            const std::string formatName = (tokenCount >= 1) ? RTrimString(tokens[0]) : "";
            const std::string compSel    = (tokenCount >= 2) ? RTrimString(tokens[1]) : "";
            const RImage::Format format = RImage::GetFormatFromName(formatName);
            if (format != RImage::Format_Invalid &&
                !RImage::IsFloatFormat(format)   &&
                (compSel.empty() || RImage::IsValidCompSel(compSel)))
            {
                m_NormalTextureFormat  = format;
                m_NormalTextureCompSel = compSel;
            }
            else
            {
                isValErr = true;
            }
        }
        else if (tag == "enable_weighted_compress")
        {
            if (isBool) m_EnablesWeightedCompress = boolVal;
            else isValErr = true;
        }
        else if (tag == "filter_mip")
        {
            if      (val == "point" ) m_IsFilterMipLinear = false;
            else if (val == "linear") m_IsFilterMipLinear = true;
            else isValErr = true;
        }
        else if (tag == "sampler_merge_priority")
        {
            if (isBool) m_SamplerMergePriority = boolVal;
            else isValErr = true;
        }
        else if (tag == "use_fcl_fts_ftp")
        {
            if (isBool)
            {
                m_UsesFclFtsFtp = boolVal;
            }
            else
            {
                isValErr = true;
            }
        }
        else if (tag == "separate_material_anim")
        {
            if (isBool)
            {
                m_SeparatesFma = boolVal;
            }
            else
            {
                isValErr = true;
            }
        }
        else if (tag == "bone_visibility_merge_priority")
        {
            m_BoneVisMergePriority = boolVal;
            isValErr = !isBool;
        }
        else if (tag == "export_bone_visibility_all")
        {
            if (isBool) m_ExportsBoneVisAll = boolVal;
            else isValErr = true;
        }
        else if (tag == "error_preset_none")
        {
            m_ChecksPresetNoneError = boolVal;
            isValErr = !isBool;
        }
        else if (tag == "error_wrong_file_name")
        {
            if (isBool) m_ChecksWrongFileNameError = boolVal;
            else isValErr = true;
        }
        else if (tag == "error_no_texture_file_always")
        {
            if (isBool)
            {
                m_ChecksNoTextureFileErrorAlways = boolVal;
            }
            else
            {
                isValErr = true;
            }
        }
        else if (tag == "warning_node_name_changed")
        {
            if (isBool) m_WarnsNodeNameChanged = boolVal;
            else isValErr = true;
        }
        else if (tag == "use_temp_folder_for_script")
        {
            if (isBool) m_UsesTmpFolderForScript = boolVal;
            else isValErr = true;
        }
        else if (tag == "user_custom_ui")
        {
            m_UserCustomUiScript = RDecodeScriptString(RTrimString(val));
        }
        else if (tag == "multi_export_user_custom_ui")
        {
            m_MultiExportUserCustomUiScript = RDecodeScriptString(RTrimString(val));
        }
        else
        {
            return RStatus(RStatus::FAILURE, // RShowError: Configuration file is wrong (invalid tag)
                L"コンフィグファイルが不正です: {0}",
                "Configuration file is wrong: {0}", tag);
        }

        if (isValErr)
        {
            return RStatus(RStatus::FAILURE, // RShowError: Configuration file is wrong (invalid value)
                L"コンフィグファイルが不正です: {0}",
                "Configuration file is wrong: {0}", tag + "=\"" + val + "\"");
        }
    }
    ifs.close();

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ファイルバッファにファイルをリードします。
//-----------------------------------------------------------------------------
bool RFileBuf::Read(const std::string& path)
{
    RFreeAndClearArray(m_FileBuf);

    m_FilePath = path;
    m_FileSize = 0;

    std::ifstream ifs(path.c_str(), ios_base::in | ios_base::binary);
    if (ifs)
    {
        //-----------------------------------------------------------------------------
        // read file
        ifs.seekg(0L, ios_base::end);
        m_FileSize = static_cast<size_t>(ifs.tellg());
        ifs.seekg(0L, ios_base::beg);
        if (m_FileSize > 0)
        {
            m_FileBuf = new uint8_t[m_FileSize];
            ifs.read(reinterpret_cast<char*>(m_FileBuf), m_FileSize);
        }
        else
        {
            m_FileBuf = new uint8_t[1];
        }

        //-----------------------------------------------------------------------------
        // close file
        ifs.close();
        return true;
    }
    else
    {
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイル移動情報に従ってファイルを移動します。
//-----------------------------------------------------------------------------
RStatus RFileMove::Move() const
{
    if (RFileExists(m_Src))
    {
        if (RFileExists(m_Dst))
        {
            // MoveFile は上書きできないので、
            // 移動先のファイルが存在する場合は消去
            if (!::DeleteFileA(m_Dst.c_str()))
            {
                return RStatus::FAILURE;
            }
        }
        if (::MoveFileA(m_Src.c_str(), m_Dst.c_str()))
        {
            return RStatus::SUCCESS;
        }
    }
    return RStatus::FAILURE;
}

//-----------------------------------------------------------------------------
// @brief データ列のバイト値用のコンストラクタです。
//-----------------------------------------------------------------------------
RDataStream::RDataStream(const uint8_t* array, const int bytes, const int column)
:   m_Type(BYTE),
    m_Column(column)
{
    m_ByteValues.resize(bytes);
    for (int ibyte = 0; ibyte < bytes; ++ibyte)
    {
        m_ByteValues[ibyte] = array[ibyte];
    }
}

//-----------------------------------------------------------------------------
// @brief データ列にバイト値を追加します。
//-----------------------------------------------------------------------------
void RDataStream::Append(const uint8_t* array, const int bytes)
{
    const int prevSize = static_cast<int>(m_ByteValues.size());
    m_ByteValues.resize(prevSize + bytes);
    for (int ibyte = 0; ibyte < bytes; ++ibyte)
    {
        m_ByteValues[prevSize + ibyte] = array[ibyte];
    }
}

//-----------------------------------------------------------------------------
// @brief データ列を出力します。
//-----------------------------------------------------------------------------
void RDataStream::Out(std::ostream& os, const int tabCount, const int index) const
{
    static const char* const typeStrs[] = { "float", "int", "byte", "string", "wstring" };

    const int& tc = tabCount;
    const int count = GetCount();
    if (count > 0)
    {
        const bool isString = (m_Type == STRING || m_Type == WSTRING);
        os << RTab(tc) << "<stream stream_index=\"" << index
           << "\" type=\"" << typeStrs[m_Type]
           << "\" count=\"" << count
           << "\" column=\"" << m_Column
           << "\">";
        if (!isString) os << R_ENDL;

        if (m_Type == FLOAT)
        {
            ROutDataValues(os, tc + 1, m_FloatValues, m_Column);
        }
        else if (m_Type == INT)
        {
            ROutDataValues(os, tc + 1, m_IntValues, m_Column);
        }
        else if (m_Type == BYTE)
        {
            ROutDataValues(os, tc + 1, m_ByteValues, m_Column);
        }
        else // STRING, WSTRING
        {
            os << RGetUtf8FromShiftJis(REncodeXmlString(m_StringValue));
        }

        if (!isString) os << RTab(tc);
        os << "</stream>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
// @brief データ列配列を出力します。
//-----------------------------------------------------------------------------
void ROutDataStreams(std::ostream& os, const RDataStreamArray& array)
{
    ROutArrayElement(os, 0, array, "stream_array");
}

//-----------------------------------------------------------------------------
//! @brief データ列配をバイナリ形式で出力します。
//-----------------------------------------------------------------------------
void RDataStream::OutBinary(std::ostream& os) const
{
    static const char* const STREAM_IDENTIFIER = "stream  ";

    //-----------------------------------------------------------------------------
    // ヘッダを出力します。
    os.write(STREAM_IDENTIFIER, strlen(STREAM_IDENTIFIER));

    int binType = 0;
    switch (m_Type)
    {
    case FLOAT  : binType = 0; break;
    case INT    : binType = 1; break;
    case BYTE   : binType = 2; break;
    case STRING : binType = 3; break;
    case WSTRING: binType = 4; break;
    default: break;
    }
    os.write(reinterpret_cast<const char*>(&binType), sizeof(binType));

    const int count = GetCount();
    os.write(reinterpret_cast<const char*>(&count), sizeof(count));

    os.write(reinterpret_cast<const char*>(&m_Column), sizeof(m_Column));

    const int dataSize = GetSize();
    os.write(reinterpret_cast<const char*>(&dataSize), sizeof(dataSize));

    int padding = 0;
    os.write(reinterpret_cast<const char*>(&padding), sizeof(padding));
    os.write(reinterpret_cast<const char*>(&padding), sizeof(padding));

    //-----------------------------------------------------------------------------
    // データを出力します。
    uint8_t* dataBuf = new uint8_t[dataSize];
    uint8_t* dst = dataBuf;
    if (m_Type == FLOAT)
    {
        for (int iData = 0; iData < count; ++iData, dst += sizeof(float))
        {
            memcpy(dst, &m_FloatValues[iData], sizeof(float));
        }
    }
    else if (m_Type == INT)
    {
        for (int iData = 0; iData < count; ++iData, dst += sizeof(int))
        {
            memcpy(dst, &m_IntValues[iData], sizeof(int));
        }
    }
    else if (m_Type == BYTE)
    {
        for (int iData = 0; iData < count; ++iData)
        {
            dst[iData] = m_ByteValues[iData];
        }
    }
    else if (m_Type == STRING)
    {
        memcpy(dst, m_StringValue.c_str(), m_StringValue.size());
        dst[m_StringValue.size()] = '\0'; // 終端文字を出力します。
    }
    else if (m_Type == WSTRING)
    {
        const std::wstring wstr = RGetUnicodeFromShiftJis(m_StringValue);
        memcpy(dst, wstr.c_str(), wstr.size() * sizeof(wchar_t));
        *(reinterpret_cast<wchar_t*>(dst) + wstr.size()) = 0; // 終端文字を出力します。
    }
    os.write(reinterpret_cast<const char*>(dataBuf), GetSize());
    delete[] dataBuf;

    //-----------------------------------------------------------------------------
    // パディングを出力します。
    ROutPadding(os, GetBinarySize() - (HEADER_SIZE + GetSize()));
}

//-----------------------------------------------------------------------------
//! @brief データ列配列をバイナリ形式で出力します。
//-----------------------------------------------------------------------------
void ROutBinaryDataStreams(
    std::ostream& os,
    const RDataStreamArray& dataStreams,
    const bool outZeroPadding
)
{
    // 参考
    // 3dIntermediateFileLibrary\Sources\Io\Format\IfBinaryFormatter.cs
    // 3dIntermediateFileLibrary\Sources\Io\Write\IfBinaryWriteUtility.cs の Write

    static const char* const STREAM_ARRAY_IDENTIFIER = "g3dstrma";

    //-----------------------------------------------------------------------------
    // データ列がなければ何も出力しません。
    const int streamCount = static_cast<int>(dataStreams.size());
    if (streamCount == 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // バイナリ形式の後半部の前のゼロパディングを出力します。
    if (outZeroPadding)
    {
        const int xmlSize = static_cast<int>(os.tellp());
        const int binOfs = RAlignValue(xmlSize + 1, R_BINARY_ALIGNMENT);
        ROutPadding(os, binOfs - xmlSize);
    }

    //-----------------------------------------------------------------------------
    // ストリーム配列チャンクを出力します。
    os.write(STREAM_ARRAY_IDENTIFIER, strlen(STREAM_ARRAY_IDENTIFIER));
    os.write(reinterpret_cast<const char*>(&streamCount), sizeof(streamCount));
    const int headerSize = 8 + 4 + (4 + 4) * streamCount;
    const int streamAddrTop = RAlignValue(headerSize, R_BINARY_ALIGNMENT);

    int streamAddr = streamAddrTop;
    for (int istream = 0; istream < streamCount; ++istream)
    {
        const int streamSize = dataStreams[istream].GetBinarySize();
        os.write(reinterpret_cast<const char*>(&streamAddr), sizeof(streamAddr));
        os.write(reinterpret_cast<const char*>(&streamSize), sizeof(streamSize));
        streamAddr += streamSize;
    }
    ROutPadding(os, streamAddrTop - headerSize);

    //-----------------------------------------------------------------------------
    // 各ストリームチャンクを出力します。
    for (int istream = 0; istream < streamCount; ++istream)
    {
        dataStreams[istream].OutBinary(os);
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの先頭からバイナリデータまでのオフセットを取得します。
//-----------------------------------------------------------------------------
int RGetBinaryOffset(const void* fileBuf, const int fileSize)
{
    int binOfs = fileSize;
    const uint8_t* buf = reinterpret_cast<const uint8_t*>(fileBuf);
    for (int iBuf = 0; iBuf < fileSize; ++iBuf)
    {
        if (buf[iBuf] == 0x00)
        {
            binOfs = RAlignValue(iBuf + 1, R_BINARY_ALIGNMENT);
            break;
        }
    }
    return binOfs;
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータを出力します。
//-----------------------------------------------------------------------------
void RUserData::Out(std::ostream& os, const int tabCount, const int dataIdx) const
{
    static const char* const elementNames[] =
    {
        "user_int", "user_float", "user_string", "user_wstring"
    };

    const int& tc = tabCount;

    os << RTab(tc) << "<user_data index=\"" << dataIdx
       << "\" name=\"" << RGetUtf8FromShiftJis(m_Name)
       << "\">" << R_ENDL;

    const char* elementName = elementNames[m_Type];
    const int valueCount = GetValueCount();
    os << RTab(tc + 1) << "<" << elementName << " count=\"" << valueCount << "\"";
    if (valueCount > 0)
    {
        os << ">" << R_ENDL;
        if (m_Type == INT)
        {
            // 次のフォーマットで出力するの正式ですが、
            // 量子化分析でフォーマットが整えられるので、
            // DCC ツールからは単純なフォーマットで出力します。
            //
            // ・4 タブの間隔を空けつつ、1 行あたり 4 つの値を出力
            // ・符号が + の場合は数値の前にスペースを挿入
            //
            // Tool\G3dTool\nw4f_3dif\src\Format\G3dDataFormatter.cs
            // の FormatUserIntData、FormatUserFloatData
            //
            ROutDataValues(os, tc + 2, m_IntValues, 4);
        }
        else if (m_Type == FLOAT)
        {
            ROutDataValues(os, tc + 2, m_FloatValues, 4);
        }
        else // STRING, WSTRING
        {
            const char* subElemName = (m_Type == STRING) ? "string" : "wstring";
            for (int iVal = 0; iVal < valueCount; ++iVal)
            {
                os << RTab(tc + 2) << "<" << subElemName << ">"
                   << RGetUtf8FromShiftJis(REncodeXmlString(m_StringValues[iVal]))
                   << "</" << subElemName << ">" << R_ENDL;
            }
        }
        os << RTab(tc + 1) << "</" << elementName << ">" << R_ENDL;
    }
    else
    {
        os << " />" << R_ENDL;
    }
    os << RTab(tc) << "</user_data>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 有効な文字列型ユーザーデータの値なら true を返します。
//-----------------------------------------------------------------------------
bool RUserData::IsValidString(const std::string& str)
{
    const int count = static_cast<int>(str.size());
    for (int ic = 0; ic < count; ++ic)
    {
        const char c = str[ic];
        if (c == '\n')
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief トランスフォームノードの SRT を出力します。
//-----------------------------------------------------------------------------
void RTransformNode::OutSrt(std::ostream& os, const int tabCount) const
{
    ROutTransform(os, tabCount, m_Scale, m_Rotate, m_Translate);
}

//-----------------------------------------------------------------------------
//! @brief ボーンの行列関連の属性を出力します。
//-----------------------------------------------------------------------------
void RBone::OutMtxAttrib(std::ostream& os, const int tabCount) const
{
    const int& tc = tabCount;
    os << RTab(tc) << "rigid_body=\"" << RBoolStr(m_RigidBody) << "\"" << R_ENDL;
    os << RTab(tc) << "matrix_index=\"" << m_SmoothMtxIdx << " " << m_RigidMtxIdx << "\"" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief ボーンを出力します。
//-----------------------------------------------------------------------------
void RBone::Out(
    std::ostream& os,
    const int tabCount,
    const int boneIdx,
    const std::string& parentName
) const
{
    static const char* const billboardStrings[] =
    {
        "none",
        "world_viewvector",
        "world_viewpoint",
        "screen_viewvector",
        "screen_viewpoint",
        "yaxis_viewvector",
        "yaxis_viewpoint",
    };
    static const char* const sideStrings[] = { "none", "center", "left", "right" };

    const int& tc = tabCount;

    const bool outInvMtx = (m_SmoothMtxIdx != -1 || m_RigidMtxIdx != -1);
    const bool hasChildElem = outInvMtx || !m_UserDatas.empty();

    os << RTab(tc) << "<bone index=\"" << boneIdx
       << "\" name=\"" << RGetUtf8FromShiftJis(m_Name)
       << "\" parent_name=\"" << RGetUtf8FromShiftJis(parentName)
       << "\"" << R_ENDL;
    OutMtxAttrib(os, tc + 1);
    OutSrt(os, tc + 1);
    os << RTab(tc + 1) << "scale_compensate=\"" << RBoolStr(m_ScaleCompensate) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "billboard=\"" << billboardStrings[m_Billboard] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "visibility=\"" << RBoolStr(m_Visibility) << "\"" << R_ENDL;

    os << RTab(tc + 1) << "compress_enable=\"" << RBoolStr(m_CompressEnable) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "side=\"" << sideStrings[m_Side] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "type=\"" << m_Type << "\"" << R_ENDL;
    os << RTab(tc) << (hasChildElem ? ">" : "/>") << R_ENDL;

    //-----------------------------------------------------------------------------
    // バインドポーズの逆行列を出力します。
    if (outInvMtx)
    {
        os << RTab(tc + 1) << "<inv_model_matrix>" << R_ENDL;
        m_InvModelMtx.Out34(os, tc + 2);
        os << RTab(tc + 1) << "</inv_model_matrix>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // ボーンのユーザーデータ配列を出力します。
    ROutArrayElement(os, tc + 1, m_UserDatas, "user_data_array");

    //-----------------------------------------------------------------------------
    // end bone
    if (hasChildElem)
    {
        os << RTab(tc) << "</bone>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief スケルタルアニメーションアニメーションするパラメータの名前を返します。
//-----------------------------------------------------------------------------
const char* RBone::GetParamName(const int paramIdx)
{
    static const char* const paramNames[] =
    {
        "scale_x"    , "scale_y"    , "scale_z"    ,
        "rotate_x"   , "rotate_y"   , "rotate_z"   ,
        "translate_x", "translate_y", "translate_z",
    };
    return (0 <= paramIdx && paramIdx < PARAM_COUNT) ?
        paramNames[paramIdx] : "?";
}

//-----------------------------------------------------------------------------
//! @brief スケルトンのスケールと回転の計算方法を出力します。
//-----------------------------------------------------------------------------
void RSkeleton::OutScaleRotateMode(std::ostream& os, const int tabCount) const
{
    static const char* const scaleModeStrs[] = { "standard", "maya", "softimage" };
    static const char* const rotateModeStrs[] = { "euler_xyz", "quaternion" };

    const int& tc = tabCount;
    os << RTab(tc) << "scale_mode=\"" << scaleModeStrs[m_ScaleMode] << "\"" << R_ENDL;
    os << RTab(tc) << "rotate_mode=\"" << rotateModeStrs[m_RotateMode] << "\"" << R_ENDL; // 現在は euler で固定
}

//-----------------------------------------------------------------------------
//! @brief スケルトン情報を出力します。
//-----------------------------------------------------------------------------
void RSkeleton::OutInfo(std::ostream& os, const int tabCount) const
{
    const int& tc = tabCount;
    os << RTab(tc) << "<skeleton_info" << R_ENDL;
    os << RTab(tc + 1) << "scale_enable=\"" << RBoolStr(m_ScaleEnable) << "\"" << R_ENDL;
    OutScaleRotateMode(os, tc + 1);
    os << RTab(tc + 1) << "motion_mirroring_enable=\"" << RBoolStr(m_MotionMirroringEnable) << "\"" << R_ENDL;
    os << RTab(tc) << "/>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief カメラのアニメーションするパラメータの名前を返します。
//-----------------------------------------------------------------------------
const char* RCamera::GetParamName(const int paramIdx)
{
    static const char* const paramNames[] =
    {
        "position_x", "position_y", "position_z",
        "aim_x", "aim_y", "aim_z",
        "twist",
        "rotate_x", "rotate_y", "rotate_z",
        "aspect",
        "near",
        "far",
        "ortho_height",
        "persp_fovy",
    };
    return (0 <= paramIdx && paramIdx < PARAM_COUNT) ?
        paramNames[paramIdx] : "?";
}

//-----------------------------------------------------------------------------
//! @brief ライトのアニメーションするパラメータの名前を返します。
//-----------------------------------------------------------------------------
const char* RLight::GetParamName(const int paramIdx)
{
    static const char* const paramNames[] =
    {
        "enable",
        "position_x", "position_y", "position_z",
        "direction_x", "direction_y", "direction_z",
        "aim_x", "aim_y", "aim_z",
        "dist_attn_start", "dist_attn_end",
        "angle_attn_start", "angle_attn_end",
        "color0_r", "color0_g", "color0_b",
        "color1_r", "color1_g", "color1_b",
    };
    return (0 <= paramIdx && paramIdx < PARAM_COUNT) ?
        paramNames[paramIdx] : "?";
}

//-----------------------------------------------------------------------------
//! @brief フォグのアニメーションするパラメータの名前を返します。
//-----------------------------------------------------------------------------
const char* RFog::GetParamName(const int paramIdx)
{
    static const char* const paramNames[] =
    {
        "dist_attn_start", "dist_attn_end",
        "color_r", "color_g", "color_b",
    };
    return (0 <= paramIdx && paramIdx < PARAM_COUNT) ?
        paramNames[paramIdx] : "?";
}

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

