﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// UpdateConstantFlag UpdateUseFlag

// MakeAnimKeys MakeStepAnimKeys OptimizeAnimKeys OptimizeStepAnimKeys
// RMakeVectorAnimKeys OptimizeVectorAnimKeys
// EvaluateAnimKeys
// MakeContinuousAngleArray MakeContinuousXyzAngleArray MakeContinuousZxyAngleArray

// OutAnimCurve

//=============================================================================
// 処理選択用マクロです。
//=============================================================================
#define OPTIMIZE_KEY_2PASS_SW // 高速化のため 2 パスでキーを最適化するなら定義します。

//=============================================================================
// include
//=============================================================================
#include "DccAnimation.h"

using namespace std;

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//=============================================================================
// constants
//=============================================================================
const int UNIT_FRAME_RANGE = 1024; // 最適化する単位のフレーム範囲です。2 のべき乗にする必要があります。

//-----------------------------------------------------------------------------
//! @brief 連続的な角度を取得します。角度の単位は degree です。
//!
//! @param[in] value 対象となるフレームの角度です。
//! @param[in] lastValue 1 つ前のフレームの角度です。
//!
//! @return 連続的になるように調整した角度を返します。
//-----------------------------------------------------------------------------
float RGetContinuousAngle(const float value, const float lastValue)
{
    float diff = value - lastValue;

    // clamp (-360, 360)
    if (RAbs(diff) >= 360.0f)
    {
        const int iShift = static_cast<int>(diff / 360.0f);
        diff -= 360.0f * iShift;
    }

    // clamp [-180, 180)
    if (diff >= 180.0f)
    {
        diff -= 360.0f;
    }
    else if (diff < -180.0f)
    {
        diff += 360.0f;
    }

    return (lastValue + diff);
}

//-----------------------------------------------------------------------------
//! @brief 角度の配列が連続的な値になるように調整します。角度の単位は degree です。
//!
//! @param[in,out] values 角度の配列です。
//!
//! @return 角度の配列を変更した場合は true を返します。
//-----------------------------------------------------------------------------
bool RMakeContinuousAngleArray(RFloatArray& values)
{
    //-----------------------------------------------------------------------------
    // check frame count
    const int subFrameCount = static_cast<int>(values.size());
    if (subFrameCount < 2)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // check constant
    if (RIsConstantArray(values))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // adjust
    bool isChanged = false;
    for (int iFrame = 1; iFrame < subFrameCount; ++iFrame)
    {
        float v = values[iFrame];
        const float lv = values[iFrame - 1];
        float diff = v - lv;
        float diffAbs = RAbs(diff);

        // 差を 360 未満にします。
        while (diffAbs >= 360.0f)
        {
            if (diff > 0.0f)
            {
                v -= 360.0f;
            }
            else
            {
                v += 360.0f;
            }
            diffAbs -= 360.0f;
        }

        // ±360 の値をテスト
        diff = v - lv;
        if (diffAbs > 180.0f)
        {
            if (diff > 0.0f)
            {
                float tv = v - 360.0f;
                if (RAbs(tv - lv) < diffAbs)
                {
                    v = tv;
                }
            }
            else
            {
                float tv = v + 360.0f;
                if (RAbs(tv - lv) < diffAbs)
                {
                    v = tv;
                }
            }
        }

        if (v != values[iFrame])
        {
            isChanged = true;
            values[iFrame] = v;
        }
    }

    return isChanged;
}

//-----------------------------------------------------------------------------
//! @brief 角度の配列が連続的な値になるように XYZ 3 軸同時に調整します。
//!        回転順序が XYZ の場合に対応しています。角度の単位は degree です。
//!
//! @param[in,out] rxArray X 軸回転の角度の配列です。
//! @param[in,out] ryArray Y 軸回転の角度の配列です。
//! @param[in,out] rzArray Z 軸回転の角度の配列です。
//!
//! @return 角度の配列を変更した場合は true を返します。
//-----------------------------------------------------------------------------
bool RMakeContinuousXyzAngleArray(
    RFloatArray& rxArray,
    RFloatArray& ryArray,
    RFloatArray& rzArray
)
{
    //-----------------------------------------------------------------------------
    // check frame count
    const int frameCount = static_cast<int>(rxArray.size());
    if (frameCount < 2)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // check constant
    const bool isConstantX = RIsConstantArray(rxArray);
    const bool isConstantY = RIsConstantArray(ryArray);
    const bool isConstantZ = RIsConstantArray(rzArray);
    if (isConstantX && isConstantY && isConstantZ)
    {
        return false;
    }
    else if (isConstantY && isConstantZ)
    {
        return RMakeContinuousAngleArray(rxArray);
    }
    else if (isConstantZ && isConstantX)
    {
        return RMakeContinuousAngleArray(ryArray);
    }
    else if (isConstantX && isConstantY)
    {
        return RMakeContinuousAngleArray(rzArray);
    }

    //-----------------------------------------------------------------------------
    // adjust
    bool isChanged = false;
    RVec3 prev(rxArray[0], ryArray[0], rzArray[0]);
    for (int frameIdx = 1; frameIdx < frameCount; ++frameIdx)
    {
        const RVec3 cur(rxArray[frameIdx], ryArray[frameIdx], rzArray[frameIdx]);
        const RVec3 adjusted = cur.GetClosestRotationXyz(prev);
        if (adjusted != cur)
        {
            rxArray[frameIdx] = adjusted.x;
            ryArray[frameIdx] = adjusted.y;
            rzArray[frameIdx] = adjusted.z;
            isChanged = true;
        }
        prev = adjusted;
    }
    return isChanged;
}

//-----------------------------------------------------------------------------
//! @brief 角度の配列が連続的な値になるように XYZ 3 軸同時に調整します。
//!        回転順序が ZXY の場合に対応しています。角度の単位は degree です。
//!
//! @param[in,out] rxArray X 軸回転の角度の配列です。
//! @param[in,out] ryArray Y 軸回転の角度の配列です。
//! @param[in,out] rzArray Z 軸回転の角度の配列です。
//!
//! @return 角度の配列を変更した場合は true を返します。
//-----------------------------------------------------------------------------
bool RMakeContinuousZxyAngleArray(
    RFloatArray& rxArray,
    RFloatArray& ryArray,
    RFloatArray& rzArray
)
{
    //-----------------------------------------------------------------------------
    // check frame count
    const int frameCount = static_cast<int>(rxArray.size());
    if (frameCount < 2)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // check constant
    const bool isConstantX = RIsConstantArray(rxArray);
    const bool isConstantY = RIsConstantArray(ryArray);
    const bool isConstantZ = RIsConstantArray(rzArray);
    if (isConstantX && isConstantY && isConstantZ)
    {
        return false;
    }
    else if (isConstantY && isConstantZ)
    {
        return RMakeContinuousAngleArray(rxArray);
    }
    else if (isConstantZ && isConstantX)
    {
        return RMakeContinuousAngleArray(ryArray);
    }
    else if (isConstantX && isConstantY)
    {
        return RMakeContinuousAngleArray(rzArray);
    }

    //-----------------------------------------------------------------------------
    // adjust
    bool isChanged = false;
    RVec3 prev(rxArray[0], ryArray[0], rzArray[0]);
    for (int frameIdx = 1; frameIdx < frameCount; ++frameIdx)
    {
        const RVec3 cur(rxArray[frameIdx], ryArray[frameIdx], rzArray[frameIdx]);
        const RVec3 adjusted = cur.GetClosestRotationZxy(prev);
        if (adjusted != cur)
        {
            rxArray[frameIdx] = adjusted.x;
            ryArray[frameIdx] = adjusted.y;
            rzArray[frameIdx] = adjusted.z;
            isChanged = true;
        }
        prev = adjusted;
    }
    return isChanged;
}

//-----------------------------------------------------------------------------
//! @brief 角度の配列の絶対値が小さくなるように 0 度付近にシフトします。
//!        例えば値の範囲が [360, 400] の場合、360 を引いて [0, 40] の範囲にします。
//!
//! @param[in,out] values 角度の配列です。
//!
//! @return 角度の配列を変更した場合は true を返します。
//-----------------------------------------------------------------------------
static bool ShiftAngleFullValue(RFloatArray& values)
{
    //-----------------------------------------------------------------------------
    // get value range
    float valueMin = static_cast<float>(LONG_MAX);
    float valueMax = static_cast<float>(LONG_MIN);
    const int subFrameCount = static_cast<int>(values.size());
    for (int iFrame = 0; iFrame < subFrameCount; ++iFrame)
    {
        const float value = values[iFrame];
        if (value < valueMin)
        {
            valueMin = value;
        }
        if (value > valueMax)
        {
            valueMax = value;
        }
    }

    //-----------------------------------------------------------------------------
    // shift
    const float valueCenter = (valueMin + valueMax) / 2.0f;
    if (valueCenter <= -360.0f || valueCenter >= 360.0f)
    {
        const int iShift = RRound(valueCenter / 360.0f);
        float valueOfs = -360.0f * iShift;
        for (int iFrame = 0; iFrame < subFrameCount; ++iFrame)
        {
            values[iFrame] += valueOfs;
        }
        return true;
    }
    else
    {
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief 2 つのキーをエルミート補間した値を取得します。
//!
//! @param[in] key0 評価するフレームの直前のキーです。
//! @param[in] key1 評価するフレームの直後のキーです。
//! @param[in] frame 評価するフレームです。
//!
//! @return エルミート補間した値を返します。
//-----------------------------------------------------------------------------
static float GetHermiteInterp(
    const RAnimKey& key0,
    const RAnimKey& key1,
    const float frame
)
{
    if (key0.m_Frame == key1.m_Frame)
    {
        return key1.m_Value;
    }
    else
    {
        const float t1 = frame - key0.m_Frame;
        const float t2 = 1.0f / (key1.m_Frame - key0.m_Frame);
        const float v0 = key0.m_Value;
        const float v1 = key1.m_Value;
        const float s0 = key0.m_OtSlope;
        const float s1 = key1.m_InSlope;

        const float t1t1t2 = t1 * t1 * t2;
        const float t1t1t2t2 = t1t1t2 * t2;
        const float t1t1t1t2t2 = t1 * t1t1t2t2;
        const float t1t1t1t2t2t2 = t1t1t1t2t2 * t2;

        return v0 * (2.0f * t1t1t1t2t2t2 - 3.0f * t1t1t2t2 + 1.0f)
                + v1 * (-2.0f * t1t1t1t2t2t2 + 3.0f * t1t1t2t2)
                + s0 * (t1t1t1t2t2 - 2.0f * t1t1t2 + t1)
                + s1 * (t1t1t1t2t2 - t1t1t2);
    }
}

//-----------------------------------------------------------------------------
//! @brief 2 つのキーを線形補間した値を取得します。
//!
//! @param[in] key0 評価するフレームの直前のキーです。
//! @param[in] key1 評価するフレームの直後のキーです。
//! @param[in] frame 評価するフレームです。
//!
//! @return 線形補間した値を返します。
//-----------------------------------------------------------------------------
static float GetLinearInterp(
    const RAnimKey& key0,
    const RAnimKey& key1,
    const float frame
)
{
    if (key0.m_Frame == key1.m_Frame)
    {
        return key1.m_Value;
    }
    else
    {
        const float t = (frame - key0.m_Frame) / (key1.m_Frame - key0.m_Frame);
        return (1.0f - t) * key0.m_Value + t * key1.m_Value;
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定したフレームにおけるアニメーションの値を評価します。
//!
//! @param[in] keys アニメーションキー配列です。
//! @param[in] frame 評価するフレームです。
//!
//! @return アニメーションの値を返します。
//-----------------------------------------------------------------------------
float EvaluateAnimKeys(const RAnimKeyArray& keys, const float frame)
{
    //-----------------------------------------------------------------------------
    // check key count
    const int keyCount = static_cast<int>(keys.size());
    if (keyCount == 1)
    {
        return keys[0].m_Value;
    }

    //-----------------------------------------------------------------------------
    // check out side of keys
    int iKeyL = 0;
    int iKeyR = keyCount - 1;
    if (frame <= keys[iKeyL].m_Frame)
    {
        return keys[iKeyL].m_Value;
    }
    if (frame >= keys[iKeyR].m_Frame)
    {
        return keys[iKeyR].m_Value;
    }

    //-----------------------------------------------------------------------------
    // find 2 keys near current frame
    while (iKeyL != iKeyR - 1 && iKeyL != iKeyR)
    {
        const int iKeyCenter = (iKeyL + iKeyR) / 2;
        if (frame <= keys[iKeyCenter].m_Frame)
        {
            iKeyR = iKeyCenter;
        }
        else if (frame >= keys[iKeyCenter].m_Frame)
        {
            iKeyL = iKeyCenter;
        }
    }

    //-----------------------------------------------------------------------------
    // same frame special
    const RAnimKey& key0 = keys[iKeyL];
    const RAnimKey& key1 = keys[iKeyR];
    if (RIsSame(frame, key1.m_Frame, R_FRAME_TOLERANCE))
    {
        if (iKeyR < keyCount - 1 &&
            key1.m_Frame == keys[iKeyR + 1].m_Frame)
        {
            // 同一フレームに値の異なるキーがある場合
            return keys[iKeyR + 1].m_Value;
        }
        else
        {
            return key1.m_Value;
        }
    }

    //-----------------------------------------------------------------------------
    // interpolation
    if (key0.m_Type == RAnimKey::HERMITE)
    {
        return GetHermiteInterp(key0, key1, frame);
    }
    else if (key0.m_Type == RAnimKey::LINEAR)
    {
        return GetLinearInterp(key0, key1, frame);
    }
    else // STEP
    {
        return key0.m_Value;
    }
}

//-----------------------------------------------------------------------------
//! @brief キーが 2 つのアニメーションキー配列を最適化します。
//!
//! @param[in,out] keys アニメーションキー配列です。
//-----------------------------------------------------------------------------
static void OptimizeTwoAnimKeys(RAnimKeyArray& keys)
{
    if (keys.size() == 2)
    {
        if (RIsSame(keys[0].m_Value, keys[1].m_Value) &&
            keys[0].m_Type == RAnimKey::HERMITE       &&
            keys[1].m_Type == RAnimKey::HERMITE       &&
            RIsSame(keys[0].m_OtSlope, 0.0f)          &&
            RIsSame(keys[1].m_InSlope, 0.0f))
        {
            keys.resize(1); // 最初のキーだけ残します。
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションキー配列を最適化します。
//!
//! @param[in,out] keys アニメーションキー配列です。
//!                     全サブフレームの数だけキーが設定されている必要があります。
//! @param[in] fullValues ベイクしたデータ（全サブフレームにおける値の配列）です。
//! @param[in] frameFunc 整数サブフレームから浮動小数点数フレームを取得する関数です。
//! @param[in] pFrameParam frameFunc に渡す引数です。
//! @param[in] tolerance 誤差の許容値です。
//-----------------------------------------------------------------------------
static void OptimizeAnimKeys(
    RAnimKeyArray& keys,
    const RFloatArray& fullValues,
    const RFloatFrameFunc frameFunc,
    const void* pFrameParam,
    float tolerance
)
{
    //-----------------------------------------------------------------------------
    // キー数をチェックします。2 つ以下なら専用の最適化をします。
    const int keyCount = static_cast<int>(keys.size());
    if (keyCount <= 2)
    {
        OptimizeTwoAnimKeys(keys);
        return;
    }

    //-----------------------------------------------------------------------------
    // 元のサブフレームごとのキー使用フラグ配列を 1 で初期化します。
    const int orgKeyCount = static_cast<int>(keys.size()); // 全サブフレームの数と同じである必要があります。
    RIntArray useFlags(orgKeyCount, 1);

    //-----------------------------------------------------------------------------
    // 最初のパスでは、最適化する単位のフレーム範囲ごとに不要なキーの使用フラグを 0 にします。
    int iKeyPrev = 0; // 使用する前のキーのインデックス（= サブフレームインデックス）です。
    for (int iKey = 1; iKey < orgKeyCount - 1; ++iKey)
    {
        #ifdef OPTIMIZE_KEY_2PASS_SW
        if ((iKey & (UNIT_FRAME_RANGE - 1)) == 0)
        {
            // 全フレーム範囲を一度に処理すると、
            // 不要なキーが多い場合に評価するフレーム範囲がどんどん広がるので、
            // 最適化する単位の境界部分のキーをかならず残して
            // 評価するフレーム範囲を制限します。
            iKeyPrev = iKey;
            continue;
        }
        #endif
        const int iKeyNext = iKey + 1;
        const RAnimKey& key0 = keys[iKeyPrev];
        const RAnimKey& key1 = keys[iKeyNext];
        float errMax = 0.0f;
        for (int iFrame = iKeyPrev + 1; iFrame < iKeyNext; ++iFrame)
        {
            const float floatFrame = frameFunc(iFrame, pFrameParam);
            const float value = GetHermiteInterp(key0, key1, floatFrame);
            const float err = RAbs(fullValues[iFrame] - value);
            if (err > errMax)
            {
                errMax = err;
                if (errMax >= tolerance)
                {
                    break;
                }
            }
        }
        if (errMax == 0.0f || errMax < tolerance)
        {
            useFlags[iKey] = 0; // 誤差が 0 か許容値より小さければ使用フラグを 0 にします。
        }
        else
        {
            iKeyPrev = iKey;
        }
    }

    //-----------------------------------------------------------------------------
    // 2 番目のパスでは、最適化する単位の境界部分のキーが不要なら使用フラグを 0 にします。
    #ifdef OPTIMIZE_KEY_2PASS_SW
    for (int iKey = UNIT_FRAME_RANGE; iKey < orgKeyCount - 1; iKey += UNIT_FRAME_RANGE)
    {
        iKeyPrev = iKey - 1;
        while (iKeyPrev > 0 && !useFlags[iKeyPrev]) // 使用する前のキーを検索します。
        {
            --iKeyPrev;
        }
        int iKeyNext = iKey + 1;
        while (iKeyNext < orgKeyCount && !useFlags[iKeyNext]) // 使用する次のキーを検索します。
        {
            ++iKeyNext;
        }
        //cerr << "iKey: " << iKey << ": " << iKeyPrev << ", " << iKeyNext << endl;
        const RAnimKey& key0 = keys[iKeyPrev];
        const RAnimKey& key1 = keys[iKeyNext];
        float errMax = 0.0f;
        for (int iFrame = iKeyPrev + 1; iFrame < iKeyNext; ++iFrame)
        {
            const float floatFrame = frameFunc(iFrame, pFrameParam);
            const float value = GetHermiteInterp(key0, key1, floatFrame);
            const float err = RAbs(fullValues[iFrame] - value);
            if (err > errMax)
            {
                errMax = err;
                if (errMax >= tolerance)
                {
                    break;
                }
            }
        }
        if (errMax == 0.0f || errMax < tolerance)
        {
            useFlags[iKey] = 0; // 誤差が 0 か許容値より小さければ使用フラグを 0 にします。
        }
    }
    #endif

    //-----------------------------------------------------------------------------
    // キー使用フラグが 0 でないキーから新しいキー配列を作成します。
    RAnimKeyArray newKeys;
    for (int iKey = 0; iKey < orgKeyCount; ++iKey)
    {
        if (useFlags[iKey])
        {
            newKeys.push_back(keys[iKey]);
        }
    }
    keys = newKeys;

    //-----------------------------------------------------------------------------
    // 残ったキー数が 2 つなら専用の最適化をします。
    OptimizeTwoAnimKeys(keys);
}

//-----------------------------------------------------------------------------
//! @brief ベクトルのアニメーションキーが削除可能なら true を返します。
//!        キーを削除した場合に
//!        [削除対象キーの前のキーのフレーム, 削除対象キーの次のキーのフレーム]
//!        の範囲で誤差が許容誤差未満なら true を返します。
//!
//! @param[in] pCurves XYZ 各成分のアニメーションカーブのポインタの配列です。
//! @param[in] iKeyPrev 削除対象キーの前のキーのインデックスです。
//! @param[in] iKeyNext 削除対象キーの次のキーのインデックスです。
//! @param[in] cosTolR cos(角度の許容誤差) です。
//! @param[in] frameFunc 整数サブフレームから浮動小数点数フレームを取得する関数です。
//! @param[in] pFrameParam frameFunc に渡す引数です。
//!
//! @return ベクトルのアニメーションキーが削除可能なら true を返します。
//-----------------------------------------------------------------------------
inline bool IsVectorAnimKeyCuttable(
    RAnimCurve* pCurves[],
    const int iKeyPrev,
    const int iKeyNext,
    const float cosTolR,
    const RFloatFrameFunc frameFunc,
    const void* pFrameParam
)
{
    float dotVecMin = 1.0f;
    for (int iFrame = iKeyPrev + 1; iFrame < iKeyNext; ++iFrame)
    {
        const float floatFrame = frameFunc(iFrame, pFrameParam);
        RVec3 vec, fullVec;
        for (int iXyz = 0; iXyz < R_XYZ_COUNT; ++iXyz)
        {
            const RAnimCurve& curve = *pCurves[iXyz];
            vec[iXyz] = (curve.m_ConstantFlag) ? curve.m_FullValues[0] :
                GetHermiteInterp(curve.m_Keys[iKeyPrev], curve.m_Keys[iKeyNext], floatFrame);
            fullVec[iXyz] = curve.m_FullValues[iFrame];
        }
        vec.Normalize(); // 各成分ごとにエルミート補間したので正規化します。
        const float dotVec = vec * fullVec;
        if (dotVec < dotVecMin)
        {
            dotVecMin = dotVec;
            if (dotVecMin <= cosTolR)
            {
                break;
            }
        }
    }
    return (dotVecMin == 1.0f || dotVecMin > cosTolR);
}

//-----------------------------------------------------------------------------
//! @brief ベクトルのアニメーションキー配列を最適化します。
//!
//! @param[in,out] pCurves XYZ 各成分のアニメーションカーブのポインタの配列です。
//!                        全サブフレームの数だけキーが設定されている必要があります。
//! @param[in] frameFunc 整数サブフレームから浮動小数点数フレームを取得する関数です。
//! @param[in] pFrameParam frameFunc に渡す引数です。
//-----------------------------------------------------------------------------
static void OptimizeVectorAnimKeys(
    RAnimCurve* pCurves[],
    const RFloatFrameFunc frameFunc,
    const void* pFrameParam
)
{
    //-----------------------------------------------------------------------------
    // 一定でないカーブを検索します。
    const RAnimCurve* pNoConstCurve = NULL;
    for (int iXyz = 0; iXyz < R_XYZ_COUNT; ++iXyz)
    {
        const RAnimCurve* pCurve = pCurves[iXyz];
        if (!pCurve->m_ConstantFlag)
        {
            pNoConstCurve = pCurve;
            break;
        }
    }
    if (pNoConstCurve == NULL)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // キー数をチェックします。2 つ以下なら専用の最適化をします。
    const RAnimKeyArray& keys = pNoConstCurve->m_Keys;
    const int keyCount = static_cast<int>(keys.size());
    if (keyCount <= 2)
    {
        for (int iXyz = 0; iXyz < R_XYZ_COUNT; ++iXyz)
        {
            OptimizeTwoAnimKeys(pCurves[iXyz]->m_Keys);
        }
        return;
    }

    //-----------------------------------------------------------------------------
    // 元のサブフレームごとのキー使用フラグ配列を 1 で初期化します。
    const int orgKeyCount = static_cast<int>(keys.size()); // 全サブフレームの数と同じである必要があります。
    RIntArray useFlags(orgKeyCount, 1);
    const float toleranceR = pNoConstCurve->m_Tolerance;
    const float cosTolR = static_cast<float>(cos(toleranceR * R_M_DEG_TO_RAD));

    //-----------------------------------------------------------------------------
    // 最初のパスでは、最適化する単位のフレーム範囲ごとに不要なキーの使用フラグを 0 にします。
    int iKeyPrev = 0; // 使用する前のキーのインデックス（= サブフレームインデックス）です。
    for (int iKey = 1; iKey < orgKeyCount - 1; ++iKey)
    {
        #ifdef OPTIMIZE_KEY_2PASS_SW
        if ((iKey & (UNIT_FRAME_RANGE - 1)) == 0)
        {
            // 全フレーム範囲を一度に処理すると、
            // 不要なキーが多い場合に評価するフレーム範囲がどんどん広がるので、
            // 最適化する単位の境界部分のキーをかならず残して
            // 評価するフレーム範囲を制限します。
            iKeyPrev = iKey;
            continue;
        }
        #endif
        const int iKeyNext = iKey + 1;
        if (IsVectorAnimKeyCuttable(pCurves, iKeyPrev, iKeyNext, cosTolR, frameFunc, pFrameParam))
        {
            useFlags[iKey] = 0; // 削除可能なら使用フラグを 0 にします。
        }
        else
        {
            iKeyPrev = iKey;
        }
    }

    //-----------------------------------------------------------------------------
    // 2 番目のパスでは、最適化する単位の境界部分のキーが不要なら使用フラグを 0 にします。
    #ifdef OPTIMIZE_KEY_2PASS_SW
    for (int iKey = UNIT_FRAME_RANGE; iKey < orgKeyCount - 1; iKey += UNIT_FRAME_RANGE)
    {
        iKeyPrev = iKey - 1;
        while (iKeyPrev > 0 && !useFlags[iKeyPrev]) // 使用する前のキーを検索します。
        {
            --iKeyPrev;
        }
        int iKeyNext = iKey + 1;
        while (iKeyNext < orgKeyCount && !useFlags[iKeyNext]) // 使用する次のキーを検索します。
        {
            ++iKeyNext;
        }
        //cerr << "iKey: " << iKey << ": " << iKeyPrev << ", " << iKeyNext << endl;
        if (IsVectorAnimKeyCuttable(pCurves, iKeyPrev, iKeyNext, cosTolR, frameFunc, pFrameParam))
        {
            useFlags[iKey] = 0; // 削除可能なら使用フラグを 0 にします。
        }
    }
    #endif

    //-----------------------------------------------------------------------------
    // キー使用フラグが 0 でないキーから新しいキー配列を作成します。
    for (int iXyz = 0; iXyz < R_XYZ_COUNT; ++iXyz)
    {
        RAnimCurve& curve = *pCurves[iXyz];
        if (!curve.m_ConstantFlag)
        {
            RAnimKeyArray newKeys;
            for (int iKey = 0; iKey < orgKeyCount; ++iKey)
            {
                if (useFlags[iKey])
                {
                    newKeys.push_back(curve.m_Keys[iKey]);
                }
            }
            curve.m_Keys = newKeys;

            //-----------------------------------------------------------------------------
            // 残ったキー数が 2 つなら専用の最適化をします。
            OptimizeTwoAnimKeys(curve.m_Keys);
        }
    }

    //-----------------------------------------------------------------------------
    // 誤差を確認します（デバッグ用）。
    //const int frameCount = static_cast<int>(pNoConstCurve->m_FullValues.size());
    //for (int iFrame = 0; iFrame < frameCount; ++iFrame)
    //{
    //  const float floatFrame = frameFunc(iFrame, pFrameParam);
    //  RVec3 vec, fullVec;
    //  for (int iXyz = 0; iXyz < R_XYZ_COUNT; ++iXyz)
    //  {
    //      const RAnimCurve& curve = *pCurves[iXyz];
    //      vec[iXyz] = (curve.m_ConstantFlag) ?
    //          curve.m_FullValues[0] : curve.Evaluate(floatFrame);
    //      fullVec[iXyz] = curve.m_FullValues[iFrame];
    //  }
    //  vec.Normalize(); // 各成分ごとにエルミート補間したので正規化します。
    //  const float dotVec = vec * fullVec;
    //  cerr << "vec err: " << iFrame << ": " << acos(RMin(dotVec, 1.0f)) * R_M_RAD_TO_DEG << " (" << dotVec << ")" << endl;
    //}
}

//-----------------------------------------------------------------------------
//! @brief ステップ補間用のアニメーションキー配列を最適化します。
//!        直前のキーと同じ値のキーがあれば削除します。
//!
//! @param[in,out] keys アニメーションキー配列です。
//! @param[in] tolerance 誤差の許容値です。
//-----------------------------------------------------------------------------
static void OptimizeStepAnimKeys(RAnimKeyArray& keys, const float tolerance)
{
    const int keyCount = static_cast<int>(keys.size());
    if (keyCount == 0)
    {
        return;
    }

    RAnimKeyArray dstKeys;
    dstKeys.push_back(keys[0]);

    float lastValue = keys[0].m_Value;
    for (int iKey = 1; iKey < keyCount; ++iKey)
    {
        const RAnimKey& key = keys[iKey];
        const float value = key.m_Value;
        if (!RIsSame(value, lastValue, tolerance))
        {
            lastValue = value;
            dstKeys.push_back(key);
        }
    }

    keys = dstKeys;
}

//-----------------------------------------------------------------------------
//! @brief ベイクしたデータをすべてアニメーションキーとして設定します。
//-----------------------------------------------------------------------------
static void SetAnimKeysFromFullValues(
    RAnimKeyArray& keys,
    const RFloatArray& fullValues,
    const RFloatFrameFunc frameFunc,
    const void* pFrameParam
)
{
    //-----------------------------------------------------------------------------
    // 全サブフレームについてキーを追加します。
    keys.clear();

    const int subFrameCount = static_cast<int>(fullValues.size());
    for (int iFrame = 0; iFrame < subFrameCount; ++iFrame)
    {
        keys.push_back(RAnimKey(frameFunc(iFrame, pFrameParam), fullValues[iFrame]));
    }

    //-----------------------------------------------------------------------------
    // 前後のキーの値の差分からスロープ値に設定します。
    const int keyCount = static_cast<int>(keys.size());
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        RAnimKey& key = keys[iKey];
        if (keyCount == 1)
        {
            key.m_InSlope = 0.0f;
        }
        else if (iKey == 0)
        {
            key.m_InSlope = keys[iKey + 1].m_Value - key.m_Value;
            const float scale = keys[iKey + 1].m_Frame - key.m_Frame;
            if (scale != 0.0f)
            {
                key.m_InSlope /= scale;
            }
        }
        else if (iKey == keyCount - 1)
        {
            key.m_InSlope = key.m_Value - keys[iKey - 1].m_Value;
            const float scale = key.m_Frame - keys[iKey - 1].m_Frame;
            if (scale != 0.0f)
            {
                key.m_InSlope /= scale;
            }
        }
        else
        {
            key.m_InSlope = (keys[iKey + 1].m_Value - keys[iKey - 1].m_Value);
            const float scale = keys[iKey + 1].m_Frame - keys[iKey - 1].m_Frame;
            if (scale != 0.0f)
            {
                key.m_InSlope /= scale;
            }
        }
        key.m_OtSlope = key.m_InSlope;
        key.SnapToZero();
    }
}

//-----------------------------------------------------------------------------
//! @brief ベイクしたデータからエルミート補間用のアニメーションキー配列を作成します。
//!
//! @param[out] keys アニメーションキー配列です。
//! @param[in] fullValues ベイクしたデータ（全サブフレームにおける値の配列）です。
//! @param[in] frameFunc 整数サブフレームから浮動小数点数フレームを取得する関数です。
//! @param[in] pFrameParam frameFunc に渡す引数です。
//! @param[in] tolerance 誤差の許容値です。
//-----------------------------------------------------------------------------
static void MakeAnimKeys(
    RAnimKeyArray& keys,
    const RFloatArray& fullValues,
    const RFloatFrameFunc frameFunc,
    const void* pFrameParam,
    const float tolerance
)
{
    //-----------------------------------------------------------------------------
    // ベイクしたデータをすべてアニメーションキーとして設定します。
    SetAnimKeysFromFullValues(keys, fullValues, frameFunc, pFrameParam);

    //-----------------------------------------------------------------------------
    // 不要なキーを削除します。
    OptimizeAnimKeys(keys, fullValues, frameFunc, pFrameParam, tolerance);
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブの一定フラグを更新します。
//!        最初のフレームの値と他のフレームの値との誤差が許容値未満なら一定とみなします。
//!        呼ぶ前に m_FullValues, m_Tolerance をセットしておく必要があります。
//-----------------------------------------------------------------------------
void RAnimCurve::UpdateConstantFlag()
{
    m_ConstantFlag = true;
    const int frameCount = static_cast<int>(m_FullValues.size());
    if (frameCount > 0)
    {
        const float firstValue = m_FullValues[0];
        for (int iFrame = 1; iFrame < frameCount; ++iFrame)
        {
            const float value = m_FullValues[iFrame];
            if (value != firstValue && RAbs(value - firstValue) >= m_Tolerance) // tolerance 使用
            {
                m_ConstantFlag = false;
                break;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブの使用フラグを更新します。
//!        呼ぶ前に m_FullValues, m_ConstantFlag をセットしておく必要があります。
//-----------------------------------------------------------------------------
void RAnimCurve::UpdateUseFlag(const float baseValue)
{
    // TODO: エクスポートオプションで指定した許容誤差 x 0.1 で比較？
    if (m_FullValues.size() > 0)
    {
        const float firstValue = m_FullValues[0];
        m_UseFlag = !m_ConstantFlag ||
            (!m_AngleFlag && !RIsSame(firstValue, baseValue)) ||
            ( m_AngleFlag && !RIsSameAngle(firstValue, baseValue));
    }
    else
    {
        m_UseFlag = false;
    }
}

//-----------------------------------------------------------------------------
//! @brief ベイクしたデータからエルミート補間用のアニメーションキー配列を作成します。
//!        呼ぶ前に m_FullValues, m_AngleFlag, m_Tolerance をセットしておく必要があります。
//-----------------------------------------------------------------------------
void RAnimCurve::MakeKeys(
    const RFloatFrameFunc frameFunc,
    const void* pFrameParam,
    const bool continuousAngleFlag
)
{
    //-----------------------------------------------------------------------------
    // adjust full values
    RFloatArray fullValues = m_FullValues;
    if (m_AngleFlag)
    {
        if (continuousAngleFlag)
        {
            RMakeContinuousAngleArray(fullValues);
        }

        // 角度の値が 0 付近になるように調整します。
        ShiftAngleFullValue(fullValues);
    }

    //-----------------------------------------------------------------------------
    // make anim keys
    MakeAnimKeys(m_Keys, fullValues, frameFunc, pFrameParam, m_Tolerance);

    //-----------------------------------------------------------------------------
    // 作成されたキーが 1 つなら一定とみなします。
    if (m_Keys.size() == 1)
    {
        m_ConstantFlag = true;
    }

    //-----------------------------------------------------------------------------
    // ベイク済みフラグを ON にします。
    m_Baked = true;

    //-----------------------------------------------------------------------------
    // test
    //cerr << "interp: " << m_Name << " ------------------" << R_ENDL;
    //const int subFrameCount = m_FullValues.size();
    //for (int iFrame = 0; iFrame < subFrameCount; ++iFrame)
    //{
    //  const float floatFrame = frameFunc(iFrame, pFrameParam);
    //  const float val = EvaluateAnimKeys(m_Keys, floatFrame);
    //  cerr << floatFrame << "\t" << fullValues[iFrame] << "\t"
    //       << val << "\t" << val - fullValues[iFrame] << R_ENDL;
    //}
}

//-----------------------------------------------------------------------------
//! @brief ベイクしたデータからステップ補間用のアニメーションキー配列を作成します。
//!
//! @param[out] keys アニメーションキー配列です。
//! @param[in] fullValues ベイクしたデータ（全サブフレームにおける値の配列）です。
//! @param[in] frameFunc 整数サブフレームから浮動小数点数フレームを取得する関数です。
//! @param[in] pFrameParam frameFunc に渡す引数です。
//! @param[in] tolerance 誤差の許容値です。
//-----------------------------------------------------------------------------
static void MakeStepAnimKeys(
    RAnimKeyArray& keys,
    const RFloatArray& fullValues,
    const RFloatFrameFunc frameFunc,
    const void* pFrameParam,
    const float tolerance
)
{
    //-----------------------------------------------------------------------------
    // 前サブフレームと値が異なればキーを追加します。
    keys.clear();
    const int subFrameCount = static_cast<int>(fullValues.size());
    if (subFrameCount > 0)
    {
        float lastValue = fullValues[0];
        keys.push_back(RAnimKey(0.0f, lastValue, RAnimKey::STEP));
        for (int iFrame = 1; iFrame < subFrameCount; ++iFrame)
        {
            float value = fullValues[iFrame];
            if (!RIsSame(value, lastValue, tolerance))
            {
                lastValue = value;
                keys.push_back(RAnimKey(frameFunc(iFrame, pFrameParam), lastValue, RAnimKey::STEP));
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ベイクしたデータからステップ補間用のアニメーションキー配列を作成します。
//!        呼ぶ前に m_FullValues, m_Tolerance をセットしておく必要があります。
//-----------------------------------------------------------------------------
void RAnimCurve::MakeStepKeys(const RFloatFrameFunc frameFunc, const void* pFrameParam)
{
    //-----------------------------------------------------------------------------
    // ステップ補間用のアニメーションキー配列を作成します。
    MakeStepAnimKeys(m_Keys, m_FullValues, frameFunc, pFrameParam, m_Tolerance);

    //-----------------------------------------------------------------------------
    // 作成されたキーが 1 つなら一定とみなします。
    if (m_Keys.size() == 1)
    {
        m_ConstantFlag = true;
    }

    //-----------------------------------------------------------------------------
    // ベイク済みフラグを ON にします。
    m_Baked = true;
}

//-----------------------------------------------------------------------------
//! @brief ステップ補間用のアニメーションキー配列を最適化します。
//!        直前のキーと同じ値のキーがあれば削除します。
//!        MakeStepKeys を使用しないで独自に m_Keys を設定した場合に利用します。
//!        呼ぶ前に m_Keys, m_Tolerance をセットしておく必要があります。
//-----------------------------------------------------------------------------
void RAnimCurve::OptimizeStepKeys()
{
    OptimizeStepAnimKeys(m_Keys, m_Tolerance);
}

//-----------------------------------------------------------------------------
//! @brief 指定したフレームにおけるアニメーションの値を評価します。
//!        呼ぶ前に m_Keys をセットしておく必要があります。
//-----------------------------------------------------------------------------
float RAnimCurve::Evaluate(const float frame) const
{
    return EvaluateAnimKeys(m_Keys, frame);
}

//-----------------------------------------------------------------------------
//! @brief アニメーションキー配列の接線タイプがすべて同じなら true を返します。
//-----------------------------------------------------------------------------
static bool IsAllTangentTypeSame(const RAnimKeyArray& keys)
{
    const int keyCount = static_cast<int>(keys.size());
    if (keyCount > 0)
    {
        const RAnimKey::TangentType firstType = keys[0].m_Type;
        for (int iKey = 1; iKey < keyCount; ++iKey)
        {
            if (keys[iKey].m_Type != firstType)
            {
                return false;
            }
        }
        return true;
    }
    else
    {
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブ全体の接線タイプを取得します。
//-----------------------------------------------------------------------------
RAnimKey::TangentType RAnimCurve::GetTangentType() const
{
    RAnimKey::TangentType curveType = RAnimKey::HERMITE;
    const int keyCount = static_cast<int>(m_Keys.size());
    if (keyCount > 0 && IsAllTangentTypeSame(m_Keys))
    {
        curveType = m_Keys[0].m_Type;
    }
    return curveType;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブの初期値を返します。
//-----------------------------------------------------------------------------
float RAnimCurve::GetBaseValue() const
{
    float baseValue = m_FullValues[0];
    if (m_ConstantFlag && m_AngleFlag)
    {
        baseValue = RShiftAngle(baseValue);
    }
    return baseValue;
}

//-----------------------------------------------------------------------------
//! @brief ベクトルのアニメーションカーブの一定フラグを更新します。
//-----------------------------------------------------------------------------
void RUpdateVectorAnimConstantFlag(
    RAnimCurve& curveX,
    RAnimCurve& curveY,
    RAnimCurve& curveZ,
    const float toleranceR
)
{
    //-----------------------------------------------------------------------------
    // まず XYZ 成分ごとに許容誤差 R_SAME_TOLERANCE_F で一定か判定します。
    RAnimCurve* pCurves[R_XYZ_COUNT] = { &curveX, &curveY, &curveZ };
    bool isAllConstant = true;
    for (int iXyz = 0; iXyz < R_XYZ_COUNT; ++iXyz)
    {
        RAnimCurve& curve = *pCurves[iXyz];
        curve.m_Tolerance = R_SAME_TOLERANCE_F; // 一般の許容誤差を設定します。
        curve.UpdateConstantFlag();
        if (!curve.m_ConstantFlag)
        {
            isAllConstant = false;
        }
        curve.m_Tolerance = toleranceR; // 回転の許容誤差を設定します。
    }

    //-----------------------------------------------------------------------------
    // 一定でない成分があれば、ベクトルの内積と回転の許容誤差で一定か判定します。
    if (!isAllConstant)
    {
        const float cosTolR = static_cast<float>(cos(toleranceR * R_M_DEG_TO_RAD));
        const int frameCount = static_cast<int>(pCurves[0]->m_FullValues.size());
        if (frameCount > 0)
        {
            const RVec3 firstVec(
                pCurves[0]->m_FullValues[0],
                pCurves[1]->m_FullValues[0],
                pCurves[2]->m_FullValues[0]);
            bool isVecConstant = true;
            for (int iFrame = 1; iFrame < frameCount; ++iFrame)
            {
                const RVec3 vec(
                    pCurves[0]->m_FullValues[iFrame],
                    pCurves[1]->m_FullValues[iFrame],
                    pCurves[2]->m_FullValues[iFrame]);
                if (vec * firstVec <= cosTolR)
                {
                    isVecConstant = false;
                    //cerr << "vec not const: " << pCurves[0]->m_Name << ": " << vec * firstVec << " " << cosTolR << endl;
                    break;
                }
            }

            if (isVecConstant)
            {
                for (int iXyz = 0; iXyz < R_XYZ_COUNT; ++iXyz)
                {
                    pCurves[iXyz]->m_ConstantFlag = true;
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ベイクしたデータからベクトルのアニメーションカーブのアニメーションキー配列を作成します。
//-----------------------------------------------------------------------------
void RMakeVectorAnimKeys(
    RAnimCurve& curveX,
    RAnimCurve& curveY,
    RAnimCurve& curveZ,
    const RFloatFrameFunc frameFunc,
    const void* pFrameParam
)
{
    //-----------------------------------------------------------------------------
    // すべて一定なら何もしません。
    if (curveX.m_ConstantFlag &&
        curveY.m_ConstantFlag &&
        curveZ.m_ConstantFlag)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // ベイクしたデータをすべてアニメーションキーとして設定します。
    RAnimCurve* pCurves[R_XYZ_COUNT] = { &curveX, &curveY, &curveZ };
    for (int iXyz = 0; iXyz < R_XYZ_COUNT; ++iXyz)
    {
        RAnimCurve& curve = *pCurves[iXyz];
        if (!curve.m_ConstantFlag)
        {
            SetAnimKeysFromFullValues(curve.m_Keys, curve.m_FullValues, frameFunc, pFrameParam);
        }
    }

    //-----------------------------------------------------------------------------
    // 不要なキーを削除します。
    OptimizeVectorAnimKeys(pCurves, frameFunc, pFrameParam);

    //-----------------------------------------------------------------------------
    // 作成されたキーが 1 つなら一定とみなします。
    for (int iXyz = 0; iXyz < R_XYZ_COUNT; ++iXyz)
    {
        RAnimCurve& curve = *pCurves[iXyz];
        if (!curve.m_ConstantFlag)
        {
            if (curve.m_Keys.size() == 1)
            {
                curve.m_ConstantFlag = true;
            }

            //-----------------------------------------------------------------------------
            // ベイク済みフラグを ON にします。
            curve.m_Baked = true;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブを出力します。
//-----------------------------------------------------------------------------
void RAnimCurve::Out( // OutAnimCurve
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tabCount,
    const bool isOriginal
) const
{
    static const char* const wrapStrs[] = { "clamp", "repeat", "mirror", "relative_repeat" };
    static const char* const curveElementNames[] =
    {
        "hermite_curve", "linear_curve", "step_curve"
    };
    static const char* const originalCurveElementNames[] =
    {
        "original_hermite", "original_linear", "original_step"
    };

    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // constant
    if (m_ConstantFlag)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // get curve type
    const RAnimKey::TangentType curveType = GetTangentType();

    //-----------------------------------------------------------------------------
    // append key data stream
    RFloatArray keyValues;
    const int keyCount = static_cast<int>(m_Keys.size());
    int outKeyCount = keyCount;
    for (int iKey = 0; iKey < keyCount; ++iKey)
    {
        const RAnimKey& key = m_Keys[iKey];

        // カーブが STEP 以外で直前のキーが STEP なら
        // 同じフレームに直前のキーと同じ値のキーを追加します。
        if (curveType != RAnimKey::STEP && iKey > 0)
        {
            const RAnimKey& prevKey = m_Keys[iKey - 1];
            if (prevKey.m_Type == RAnimKey::STEP &&
                !RIsSame(key.m_Value, prevKey.m_Value))
            {
                keyValues.push_back(key.m_Frame);
                keyValues.push_back(prevKey.m_Value);
                if (curveType == RAnimKey::HERMITE)
                {
                    keyValues.push_back(0.0f);
                    keyValues.push_back(0.0f);
                }
                ++outKeyCount;
            }
        }

        // 通常のキーを追加します。
        keyValues.push_back(key.m_Frame);
        keyValues.push_back(key.m_Value);
        if (curveType == RAnimKey::HERMITE)
        {
            keyValues.push_back(key.m_InSlope);
            keyValues.push_back(key.m_OtSlope);
        }
    }
    const int streamIdx = static_cast<int>(dataStreams.size());
    const int column = (curveType == RAnimKey::HERMITE) ? 4 : 2;
    dataStreams.push_back(RDataStream(keyValues, column));

    //-----------------------------------------------------------------------------
    // curve element
    os << RTab(tc) << "<"
       << ((!isOriginal) ? curveElementNames[curveType] : originalCurveElementNames[curveType])
       << " count=\"" << outKeyCount;
    if (!isOriginal)
    {
        os << "\" frame_type=\"" << "none"
           << "\" key_type=\"" << "none"
           << "\"" << R_ENDL;
        os << RTab(tc + 1) << "scale=\"" << m_Scale
           << "\" offset=\"" << m_Offset << "\"" << R_ENDL;
    }
    else
    {
        os << "\"" << R_ENDL;
    }
    os << RTab(tc + 1) << "pre_wrap=\"" << wrapStrs[m_PreWrap]
       << "\" post_wrap=\"" << wrapStrs[m_PostWrap] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "baked=\"" << RBoolStr(m_Baked) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "stream_index=\"" << streamIdx << "\"" << R_ENDL;
    os << RTab(tc) << "/>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションターゲットとアニメーションカーブを出力します。
//-----------------------------------------------------------------------------
void RAnimCurve::Out(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tabCount,
    const char* targetElementName,
    const char* targetValue,
    const bool isOriginal
) const
{
    const int& tc = tabCount;

    os << RTab(tc) << "<" << targetElementName
       << " target=\"" << targetValue
       << "\" base_value=\"" << GetBaseValue() << "\"";
    if (m_ConstantFlag)
    {
        os << " />" << R_ENDL;
    }
    else
    {
        os << ">" << R_ENDL;

        Out(os, dataStreams, tc + 1, isOriginal);

        os << RTab(tc) << "</" << targetElementName << ">" << R_ENDL;
    }
}

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

