﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// SetOption SetJobOption SetInputPath SetExportOption ShowUsage
// ConvertSub ConvertToFmd ConvertToFtx ConvertToFsk ConvertToFvb
// ConvertToFma ConvertToFcl ConvertToFts ConvertToFtp ConvertToFsn
// ParseMaterial ParseTexMap ParseBlendOperation
// ParseSkeleton ParseBone ParseShape ParseVtxAttrib ParseMesh
// ParseAnimCurve ParseCamera ParseLight ParseFog

// OutputOneFile OutputHeader OutputTextureFiles
// OutputMaterials OutputSkeleton OutputShapes OutputShapeT
// OutputBoneAnims OutputFmaAnim OutputEnvObjs
// Do3dOptimizer Do3dFormatter

// CExpOpt CModel CBone CShape CMesh CNodeVis CMaterial CTexInfo
// CTexSrtAnim CTexPatAnim CCamera CLight CFog

//=============================================================================
// include
//=============================================================================
#include "C2NnLib.h"

using namespace std;
using namespace nn::gfx::tool::dcc;

//=============================================================================
// c2nn ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace g3dTool {
namespace c2nn {

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//-----------------------------------------------------------------------------
// コンバータ
const int ConverterVersionMajor  = 4;
const int ConverterVersionMinor  = 0;
const int ConverterVersionMicro  = 0;
const int ConverterVersionBugfix = 1;

//-----------------------------------------------------------------------------
// テクスチャ
const std::string Nw4cTexFolderName = "Textures"; //!< NW4C のテクスチャフォルダ名です。

//-----------------------------------------------------------------------------
// リニア変換
const float SrgbGammaValue = 2.2f;

const int LinearRgbaFlags[R_RGBA_COUNT] =
{
    C2NnOption::LinearFlag_R,
    C2NnOption::LinearFlag_G,
    C2NnOption::LinearFlag_B,
    C2NnOption::LinearFlag_A,
};

//-----------------------------------------------------------------------------
//! @brief std::vector オブジェクトを出力します（動作確認用）。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] v std::vector オブジェクトです。
//!
//! @return 出力ストリームの参照を返します。
//-----------------------------------------------------------------------------
template<typename T>
std::ostream& operator<<(std::ostream& os, const std::vector<T>& v)
{
    os << "[";
    for (size_t i = 0; i < v.size(); ++i)
    {
        if (i != 0) os << ", ";
        os << v[i];
    }
    os << "]";
    return os;
}

//-----------------------------------------------------------------------------
//! @brief float 型 4 x 4 行列を転置します。
//!
//! @param[in,out] pMtx float 型 4 x 4 行列へのポインタです。
//!
//! @return float 型 4 x 4 行列の参照を返します。
//-----------------------------------------------------------------------------
RMtx44& TransposeMatrix(RMtx44* pMtx)
{
    RMtx44& m = *pMtx;
    const float values[4][4] =
    {
        { m[0][0], m[1][0], m[2][0], m[3][0] },
        { m[0][1], m[1][1], m[2][1], m[3][1] },
        { m[0][2], m[1][2], m[2][2], m[3][2] },
        { m[0][3], m[1][3], m[2][3], m[3][3] },
    };
    m = RMtx44(values);
    return m;
}

//=============================================================================
//! @brief コンバータ用エクスポートオプションのクラスです。
//=============================================================================
class CExpOpt : public RExpOpt
{
public:
    std::string m_SrcPath; //!< <create> の src_path に出力するパスです。
    float m_FramesPerSecond; //!< 1 秒間のフレーム数です。

public:
    //! @brief コンストラクタです。
    CExpOpt()
    {
        m_Target = EXPORT_TARGET_ALL;
        m_ExportsLod = false;

        m_MergeFmdFlag = false;
        m_MergesOutputFmd = false;
        m_MergeFtxFlag = false;
        m_MergeAnimFlag = false;
        m_MergesOutputAnim = false;

        m_Magnify = 1.0;
        m_TexSrtMode = TEX_SRT_MODE_MAYA;
        m_RemoveNamespace = false;

        m_FrameRange = FRAME_RANGE_ALL;
        m_StartFrame = 0;
        m_EndFrame = 0;
        m_LoopAnim = false;
        m_BakeAllAnim = false;
        m_FramePrecision = 1;
        m_FramesPerSecond = 60.0f;

        m_IsBinaryFormat = true;
        ClearOutFileFlag();

        InitOptimization();

        m_TolS = 0.01f;
        m_TolR = 0.1f;
        m_TolT = 0.01f;
        m_TolC = 0.001f;
        m_TolTexS = 0.01f;
        m_TolTexR = 0.1f;
        m_TolTexT = 0.01f;

        m_QuantTolS = 0.01f;
        m_QuantTolR = 0.2f;
        m_QuantTolT = 0.001f;
        m_QuantTolTexS = 0.01f;
        m_QuantTolTexR = 0.2f;
        m_QuantTolTexT = 0.01f;

        m_DisplaysProfile = false;
        m_ProjectRootPath = "";
        m_MaxVertexSkinningCount = VERTEX_SKINNING_COUNT_MAX;
        m_AdjustsSmoothSkinning = false;
        m_CompressesIgnoringVertexSkinningCount = false;
        m_UsesSrgbFetch = true;
        m_NormalTextureFormat = RImage::Format_Snorm_Bc5;
        m_NormalTextureCompSel = "";
        m_EnablesWeightedCompress = true;
        m_IsFilterMipLinear = true;
        m_SamplerMergePriority = false;
        m_UsesFclFtsFtp = true;
        m_SeparatesFma = false;
        m_BoneVisMergePriority = false;
        m_ExportsBoneVisAll = true;
        m_ChecksPresetNoneError = false;
        m_ChecksWrongFileNameError = true;
        m_ChecksNoTextureFileErrorAlways = false;
        m_WarnsNodeNameChanged = true;
        m_UsesTmpFolderForScript = false;
    }

    //! @brief デストラクタです。
    virtual ~CExpOpt()
    {
    }
};

//=============================================================================
//! @brief コンバータ用テクスチャ情報のクラスです。
//=============================================================================
class CTexInfo
{
public:
    static const std::string HintAuto; //!< ヒント情報を自動決定する場合に指定するヒント情報です。

    std::string m_TexName; //!< 入力テクスチャ名です。
    std::string m_CtexPath; //!< ctex ファイルのパスです。
    std::string m_Hint; //!< ヒント情報です。
    std::string m_OutName; //!< 出力テクスチャ名です。
    bool m_IsBinaryFormat; //!< 出力ファイルがバイナリ形式なら true です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] texName 入力テクスチャ名です。
    //! @param[in] ctexPath ctex ファイルのパスです。
    //! @param[in] hint ヒント情報です。
    //! @param[in] outName 出力テクスチャ名です。空文字なら入力テクスチャ名と同じになります。
    //! @param[in] isBinaryFormat 出力ファイルがバイナリ形式なら true です。
    //!
    CTexInfo(
        const std::string& texName,
        const std::string& ctexPath,
        const std::string& hint,
        const std::string& outName,
        const bool isBinaryFormat
    )
    : m_TexName(texName),
      m_CtexPath(ctexPath),
      m_Hint(hint),
      m_IsBinaryFormat(isBinaryFormat)
    {
        //cerr << "CTexInfo(): " << m_TexName << ": " << m_CtexPath << endl;
        m_OutName = (!outName.empty()) ? outName : texName;
    }

    //! @brief デストラクタです。
    virtual ~CTexInfo()
    {
    }

    //! @brief 出力ファイル名を返します。
    std::string GetOutFileName() const
    {
        return m_OutName + ((m_IsBinaryFormat) ? ".ftxb" : ".ftxa");
    }
};

//! @brief コンバータ用テクスチャ情報配列の定義です。
typedef std::vector<CTexInfo> CTexInfoArray;

// コンバータ用テクスチャの static メンバ変数を初期化します。
const std::string CTexInfo::HintAuto = "<auto>";

//=============================================================================
//! @brief コンバータ用テクスチャ SRT アニメーションのクラスです。
//=============================================================================
class CTexSrtAnim
{
public:
    std::string m_Hint; //!< ヒント情報です。
    std::string m_Mode; //!< テクスチャ SRT の計算方式です。
    RAnimCurve m_Anims[ROriginalTexsrt::PARAM_COUNT]; //!< アニメーションカーブ配列です。

public:
    //! @brief コンストラクタです。
    CTexSrtAnim(const std::string& hint, const std::string& mode, const CExpOpt& expOpt)
    : m_Hint(hint),
      m_Mode(mode)
    {
        for (int paramIdx = 0; paramIdx < ROriginalTexsrt::PARAM_COUNT; ++paramIdx)
        {
            RAnimCurve& curve = m_Anims[paramIdx];
            curve.m_Baked = expOpt.m_BakeAllAnim;
            curve.m_UseFlag = false;
            curve.m_AngleFlag = (paramIdx == ROriginalTexsrt::ROTATE);
        }
    }

    //! @brief デストラクタです。
    virtual ~CTexSrtAnim()
    {
    }
};

//! @brief コンバータ用テクスチャ SRT アニメーション配列の定義です。
typedef std::vector<CTexSrtAnim> CTexSrtAnimArray;

//=============================================================================
//! @brief コンバータ用テクスチャパターンアニメーションのクラスです。
//=============================================================================
class CTexPatAnim
{
public:
    std::string m_SamplerName; //!< サンプラ名です。
    std::string m_Hint; //!< ヒント情報です。
    RAnimCurve m_Anim; //!< アニメーションカーブです。

public:
    //! @brief コンストラクタです。
    CTexPatAnim(const std::string& samplerName, const std::string& hint, const CExpOpt& expOpt)
    : m_SamplerName(samplerName),
      m_Hint(hint)
    {
        RAnimCurve& curve = m_Anim;
        curve.m_Baked = expOpt.m_BakeAllAnim;
        curve.m_UseFlag = false;
    }

    //! @brief デストラクタです。
    virtual ~CTexPatAnim()
    {
    }
};

//! @brief コンバータ用テクスチャパターンアニメーション配列の定義です。
typedef std::vector<CTexPatAnim> CTexPatAnimArray;

//=============================================================================
//! @brief コンバータ用マテリアルのクラスです。
//=============================================================================
class CMaterial : public RMaterial
{
public:
    //! @brief cmcla ファイルのカラーアニメーションするパラメータを表す列挙型です。
    enum CmclaParam
    {
        SPECULAR1_R = COLOR_PARAM_COUNT,    //!< スペキュラカラー 1 の R 成分です。
        SPECULAR1_G,            //!< スペキュラカラー 1 の G 成分です。
        SPECULAR1_B,            //!< スペキュラカラー 1 の B 成分です。
        CONSTANT0_R,            //!< コンスタントカラー 0 の R 成分です。
        CONSTANT0_G,            //!< コンスタントカラー 0 の G 成分です。
        CONSTANT0_B,            //!< コンスタントカラー 0 の B 成分です。
        CONST_OPACITY0_R,       //!< コンスタントカラー 0 の不透明度の R 成分です。
        CONST_OPACITY0_G,       //!< コンスタントカラー 0 の不透明度の G 成分です。
        CONST_OPACITY0_B,       //!< コンスタントカラー 0 の不透明度の B 成分です。
        CONSTANT1_R,            //!< コンスタントカラー 1 の R 成分です。
        CONSTANT1_G,            //!< コンスタントカラー 1 の G 成分です。
        CONSTANT1_B,            //!< コンスタントカラー 1 の B 成分です。
        CONST_OPACITY1_R,       //!< コンスタントカラー 1 の不透明度の R 成分です。
        CONST_OPACITY1_G,       //!< コンスタントカラー 1 の不透明度の G 成分です。
        CONST_OPACITY1_B,       //!< コンスタントカラー 1 の不透明度の B 成分です。
        CONSTANT2_R,            //!< コンスタントカラー 2 の R 成分です。
        CONSTANT2_G,            //!< コンスタントカラー 2 の G 成分です。
        CONSTANT2_B,            //!< コンスタントカラー 2 の B 成分です。
        CONST_OPACITY2_R,       //!< コンスタントカラー 2 の不透明度の R 成分です。
        CONST_OPACITY2_G,       //!< コンスタントカラー 2 の不透明度の G 成分です。
        CONST_OPACITY2_B,       //!< コンスタントカラー 2 の不透明度の B 成分です。
        CONSTANT3_R,            //!< コンスタントカラー 3 の R 成分です。
        CONSTANT3_G,            //!< コンスタントカラー 3 の G 成分です。
        CONSTANT3_B,            //!< コンスタントカラー 3 の B 成分です。
        CONST_OPACITY3_R,       //!< コンスタントカラー 3 の不透明度の R 成分です。
        CONST_OPACITY3_G,       //!< コンスタントカラー 3 の不透明度の G 成分です。
        CONST_OPACITY3_B,       //!< コンスタントカラー 3 の不透明度の B 成分です。
        CONSTANT4_R,            //!< コンスタントカラー 4 の R 成分です。
        CONSTANT4_G,            //!< コンスタントカラー 4 の G 成分です。
        CONSTANT4_B,            //!< コンスタントカラー 4 の B 成分です。
        CONST_OPACITY4_R,       //!< コンスタントカラー 4 の不透明度の R 成分です。
        CONST_OPACITY4_G,       //!< コンスタントカラー 4 の不透明度の G 成分です。
        CONST_OPACITY4_B,       //!< コンスタントカラー 4 の不透明度の B 成分です。
        CONSTANT5_R,            //!< コンスタントカラー 5 の R 成分です。
        CONSTANT5_G,            //!< コンスタントカラー 5 の G 成分です。
        CONSTANT5_B,            //!< コンスタントカラー 5 の B 成分です。
        CONST_OPACITY5_R,       //!< コンスタントカラー 5 の不透明度の R 成分です。
        CONST_OPACITY5_G,       //!< コンスタントカラー 5 の不透明度の G 成分です。
        CONST_OPACITY5_B,       //!< コンスタントカラー 5 の不透明度の B 成分です。
        CMCLA_PARAM_COUNT       //!< パラメータの総数です。
    };

public:
    static const int CONSTANT_COUNT = 6; //!< コンスタントカラーの数です。

public:
    RAnimCurve m_Anims[CMCLA_PARAM_COUNT]; //!< カラーアニメーションのアニメーションカーブ配列です。
    CTexSrtAnimArray m_TexSrtAnims; //!< テクスチャ SRT アニメーション配列です。
    CTexPatAnimArray m_TexPatAnims; //!< テクスチャパターンアニメーション配列です。

public:
    //! @brief コンストラクタです。
    CMaterial()
    {
    }

    //! @brief デストラクタです。
    virtual ~CMaterial()
    {
    }

    //! @brief カラーアニメーションを初期化します。
    //!
    //! @param[in] expOpt エクスポートオプションです。
    //!
    void InitColorAnim(const CExpOpt& expOpt)
    {
        for (int paramIdx = 0; paramIdx < CMCLA_PARAM_COUNT; ++paramIdx)
        {
            RAnimCurve& curve = m_Anims[paramIdx];
            curve.m_Baked = expOpt.m_BakeAllAnim;
            curve.m_UseFlag = false;
        }
    }

    //! @brief カラーアニメーションが設定されているカラー数を取得します。
    //!
    //! @return カラーアニメーションが設定されているカラー数を返します。
    //!
    int GetAnimatedColorCount() const
    {
        int animatedColorCount = 0;
        for (int paramIdx = 0; paramIdx < CMCLA_PARAM_COUNT; paramIdx += R_RGB_COUNT)
        {
            if (m_Anims[paramIdx + 0].m_UseFlag ||
                m_Anims[paramIdx + 1].m_UseFlag ||
                m_Anims[paramIdx + 2].m_UseFlag)
            {
                ++animatedColorCount;
            }
        }
        return animatedColorCount;
    }
};

//! @brief コンバータ用マテリアル配列の定義です。
typedef std::vector<CMaterial> CMaterialArray;

//=============================================================================
//! @brief コンバータ用ボーンのクラスです。
//=============================================================================
class CBone : public RBone
{
public:
    int m_ParentIdx; //!< 親ボーンのインデックスです。
    RMtx44 m_GlobalMtx; //!< グローバル変換行列です。
    RMtx44 m_InvGlobalMtx; //!< グローバル変換行列の逆行列です。
    RAnimCurve m_Anims[PARAM_COUNT]; //!< スケルタルアニメーションのアニメーションカーブ配列です。
    RAnimCurve m_VisAnim; //!< ビジビリティアニメーションのアニメーションカーブです。

public:
    //! @brief コンストラクタです。
    CBone()
    : m_ParentIdx(-1)
    {
    }

    //! @brief デストラクタです。
    virtual ~CBone()
    {
    }

    //! @brief 変換行列を設定します。
    //!
    //! @param[in] parentMtx 親ボーンのグローバル変換行列です。
    //!
    void SetMatrix(const RMtx44& parentMtx)
    {
        RMtx44 localMtx;
        localMtx.SetTransform(m_Scale, m_Rotate, m_Translate);
        m_GlobalMtx = localMtx * parentMtx;
        m_InvGlobalMtx = m_GlobalMtx.Inverse();
        // 出力する逆行列は座標を右から掛けるタイプなので転置します。
        m_InvModelMtx = m_InvGlobalMtx;
        TransposeMatrix(&m_InvModelMtx);
        m_InvModelMtx.SnapToZero();
    }

    //! @brief スケルタルアニメーションを初期化します。
    //!
    //! @param[in] expOpt エクスポートオプションです。
    //!
    void InitSkeletalAnim(const CExpOpt& expOpt)
    {
        // バイナリ出力フラグをすべて OFF にします。
        m_BinarizesScale = m_BinarizesRotate = m_BinarizesTranslate = false;

        // アニメーションカーブのベース値を設定ます。
        const float baseValues[PARAM_COUNT] =
        {
            m_Scale.x    , m_Scale.y    , m_Scale.z    ,
            m_Rotate.x   , m_Rotate.y   , m_Rotate.z   ,
            m_Translate.x, m_Translate.y, m_Translate.z,
        };
        for (int paramIdx = 0; paramIdx < PARAM_COUNT; ++paramIdx)
        {
            RAnimCurve& curve = m_Anims[paramIdx];
            curve.m_FullValues.push_back(baseValues[paramIdx]);
            curve.m_AngleFlag = IsRotate(paramIdx);
            curve.m_Baked = expOpt.m_BakeAllAnim;
        }
    }
};

//! @brief コンバータ用ボーン配列の定義です。
typedef std::vector<CBone> CBoneArray;

//=============================================================================
//! @brief コンバータ用頂点属性情報のクラスです。
//=============================================================================
class CVtxAttrInfo
{
public:
    bool m_ExistFlag; //!< 頂点属性が存在していれば true です。デフォルトは false です。
    int m_CompCount; //!< 成分数です。デフォルトは 3 です。
    RPrimVtx::ValueType m_ValueType; //!< 値の型です。デフォルトは VALUE_FLOAT です。

public:
    //! @brief コンストラクタです。
    CVtxAttrInfo()
    : m_ExistFlag(false),
      m_CompCount(3),
      m_ValueType(RPrimVtx::VALUE_FLOAT)
    {
    }

    //! @brief 同値であれば true を返します。
    bool operator==(const CVtxAttrInfo& rhs) const
    {
        return (
            m_ExistFlag == rhs.m_ExistFlag &&
            m_CompCount == rhs.m_CompCount &&
            m_ValueType == rhs.m_ValueType);
    }

    //! @brief 同値でなければ true を返します。
    bool operator!=(const CVtxAttrInfo& rhs) const
    {
        return !(*this == rhs);
    }
};

//! @brief 頂点属性情報配列の定義です。
typedef std::vector<CVtxAttrInfo> CVtxAttrInfoArray;

//=============================================================================
//! @brief コンバータ用シェイプのクラスです。
//=============================================================================
class CShape
{
public:
    static const int VTX_TEX_MAX = 3; //!< cmdl ファイルの最大テクスチャ座標セット数です。
    static const int VTX_USR_MAX = 3; //!< cmdl ファイルの最大ユーザー頂点属性セット数です。

    std::string m_Name; //!< 出力用の名前です。
    std::string m_NodeName; //!< ノード名です。
    std::string m_MatName; //!< マテリアル名です。

    RShape::SkinningMode m_SkinningMode; //!< スキニングモードです。
    RVtxMtxArray m_VtxMtxs; //!< 頂点行列配列です。
    RIntArray m_VtxMtxIdxs; //!< 頂点行列インデックス配列です。

    RIntArray m_VtxIdxs; //!< 三角形群の頂点インデックス配列です。

    CVtxAttrInfo m_VtxColInfo; //!< 頂点カラー属性情報です。
    int m_VtxTexCount; //!< UV セット数です。
    CVtxAttrInfo m_VtxUsrInfos[VTX_USR_MAX]; //!< ユーザー頂点属性情報配列です。
    CVtxAttrInfo m_VtxBoneIdxInfo; //!< 頂点ボーンインデックス属性情報です。
    CVtxAttrInfo m_VtxBoneWgtInfo; //!< 頂点ボーンウェイト属性情報です。

    RVec3Array m_VtxPoss; //!< 頂点座標配列です。
    RVec3Array m_VtxNrms; //!< 法線配列です。
    RVec4Array m_VtxTans; //!< 接線配列です。
    RVec4Array m_VtxBins; //!< 従法線配列です。
    RVec4Array m_VtxCols; //!< 頂点カラー配列です。
    RVec2Array m_VtxTexss[VTX_TEX_MAX]; //!< テクスチャ座標配列の配列です。
    RVec4Array m_VtxUsrss[VTX_USR_MAX]; //!< ユーザー頂点属性配列の配列です。
    RIVec4Array m_VtxBoneIdxs; //!< 頂点ボーンインデックス配列です。
    RVec4Array  m_VtxBoneWgts; //!< 頂点ボーンウェイト配列です。
    RBoolArray  m_VtxBoneIdxConvFlags; //!< 頂点ボーンインデックス変換済みフラグ配列です。

public:
    //! @brief コンストラクタです。
    CShape()
    : m_SkinningMode(RShape::NO_SKINNING),
      m_VtxTexCount(0)
    {
    }

    //! @brief デストラクタです。
    virtual ~CShape()
    {
    }

    //! @brief ユーザー頂点属性を持つなら true を返します。
    bool HasVtxUsr() const
    {
        for (int usrIdx = 0; usrIdx < VTX_USR_MAX; ++usrIdx)
        {
            if (m_VtxUsrInfos[usrIdx].m_ExistFlag)
            {
                return true;
            }
        }
        return false;
    }
};

//! @brief コンバータ用シェイプ配列の定義です。
typedef std::vector<CShape> CShapeArray;

//=============================================================================
//! @brief コンバータ用メッシュのクラスです。
//=============================================================================
class CMesh
{
public:
    bool m_Visibility; //!< 可視性です。
    std::string m_NodeName; //!< ノード名です。
    std::string m_MatName; //!< マテリアル名です。
    int m_ShapeIdx; //!< シェイプのインデックスです。

public:
    //! @brief コンストラクタです。
    CMesh()
    : m_Visibility(true),
      m_ShapeIdx(0)
    {
    }

    //! @brief デストラクタです。
    virtual ~CMesh()
    {
    }
};

//! @brief コンバータ用メッシュ配列の定義です。
typedef std::vector<CMesh> CMeshArray;

//=============================================================================
//! @brief コンバータ用ノードビジビリティのクラスです。
//=============================================================================
class CNodeVis
{
public:
    std::string m_Name; //!< ノード名です。
    bool m_Visibility; //!< 可視性です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] name ノード名です。
    //! @param[in] visibility 可視性です。
    //!
    CNodeVis(const std::string& name, const bool visibility)
    : m_Name(name),
      m_Visibility(visibility)
    {
    }

    //! @brief デストラクタです。
    virtual ~CNodeVis()
    {
    }
};

//! @brief コンバータ用ノードビジビリティ配列の定義です。
typedef std::vector<CNodeVis> CNodeVisArray;

//=============================================================================
//! @brief コンバータ用カメラのクラスです。
//=============================================================================
class CCamera : public RCamera
{
public:
    bool m_IsAnimGot; //!< アニメーション取得済みなら true です。
    RAnimCurve m_Anims[PARAM_COUNT]; //!< アニメーションのパラメータに対応するアニメーションカーブ配列です。
    RAnimCurve m_UpAnims[R_XYZ_COUNT]; //!< 上方向ベクトルのアニメーションカーブ配列です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] name カメラ名です。
    //! @param[in] rotateMode 回転モードです。
    //! @param[in] projMode 投影モードです。
    //!
    CCamera(
        const std::string& name,
        const RotateMode rotateMode,
        const ProjectionMode projMode
    )
    : m_IsAnimGot(false)
    {
        m_Name = name;
        m_RotateMode = rotateMode;
        m_ProjectionMode = projMode;

        m_Twist = 0.0f;
        m_Aspect = 16.0f / 9.0f;
        m_NearClip = 0.01f;
        m_FarClip = 1000.0f;
        m_OrthoHeight = 20.0f;
        m_PerspFovy = 40.0f;

        m_FrameCount = 0;
        m_LoopAnim = false;
    }

    //! @brief デストラクタです。
    virtual ~CCamera()
    {
    }

    //! @brief アニメーションを初期化します。
    void InitAnim()
    {
        // アニメーションカーブのベース値を設定ます。
        const float baseValues[PARAM_COUNT] =
        {
            m_Translate.x, m_Translate.y, m_Translate.z,
            m_Aim.x, m_Aim.y, m_Aim.z,
            m_Twist,
            m_Rotate.x, m_Rotate.y, m_Rotate.z,
            m_Aspect,
            m_NearClip, m_FarClip,
            m_OrthoHeight, m_PerspFovy
        };
        for (int paramIdx = 0; paramIdx < PARAM_COUNT; ++paramIdx)
        {
            RAnimCurve& curve = m_Anims[paramIdx];
            curve.m_FullValues.push_back(baseValues[paramIdx]);
            curve.m_UseFlag = false;
            curve.m_AngleFlag = IsRotate(paramIdx);
        }
        for (int xyzIdx = 0; xyzIdx < R_XYZ_COUNT; ++xyzIdx)
        {
            RAnimCurve& curve = m_UpAnims[xyzIdx];
            curve.m_FullValues.push_back((xyzIdx == 1) ? 1.0f : 0.0f);
            curve.m_UseFlag = false;
        }
    }

    //! @brief 位置アニメーションを持つなら true を返します。
    bool HasPositionAnim() const
    {
        return (
            m_Anims[POSITION_X].m_UseFlag ||
            m_Anims[POSITION_Y].m_UseFlag ||
            m_Anims[POSITION_Z].m_UseFlag);
    }

    //! @brief 注視点の位置アニメーションを持つなら true を返します。
    bool HasAimAnim() const
    {
        return (
            m_Anims[AIM_X].m_UseFlag ||
            m_Anims[AIM_Y].m_UseFlag ||
            m_Anims[AIM_Z].m_UseFlag);
    }

    //! @brief 上方向ベクトルアニメーションを持つなら true を返します。
    bool HasUpAnim() const
    {
        return (
            m_UpAnims[0].m_UseFlag ||
            m_UpAnims[1].m_UseFlag ||
            m_UpAnims[2].m_UseFlag);
    }

    //! @brief アニメーションを出力します。
    void OutputAnim(
        std::ostream& os,
        RDataStreamArray* pDataStreams,
        const int tabCount,
        const int index
    ) const;
};

//! @brief コンバータ用カメラ配列の定義です。
typedef std::vector<CCamera> CCameraArray;

//=============================================================================
//! @brief コンバータ用ライトのクラスです。
//=============================================================================
class CLight : public RLight
{
public:
    bool m_IsAnimGot; //!< アニメーション取得済みなら true です。
    RAnimCurve m_Anims[PARAM_COUNT]; //!< アニメーションのパラメータに対応するアニメーションカーブ配列です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] name ライト名です。
    //! @param[in] type タイプです。
    //!
    CLight(const std::string& name, const Type type)
    : m_IsAnimGot(false)
    {
        m_Name = name;
        m_Type = type;

        m_Enable = true;
        m_Direction = RVec3::kZNegAxis;
        m_Aim = (m_Type == RLight::SPOT) ?
            m_Translate + m_Direction : RVec3::kZero;

        m_UsesDistAttn = false;
        m_DistAttnStart = 0.0f;
        m_DistAttnEnd   = 10.0f;
        m_AngleAttnStart = 20.0f;
        m_AngleAttnEnd   = 20.0f;

        m_Color0 = RVec3::kOne;
        m_Color1 = RVec3::kZero;

        m_FrameCount = 0;
        m_LoopAnim = false;
    }

    //! @brief デストラクタです。
    virtual ~CLight()
    {
    }

    //! @brief アニメーションを初期化します。
    void InitAnim()
    {
        m_IsAnimGot = false;
        // アニメーションカーブのベース値を設定ます。
        const float baseValues[PARAM_COUNT] =
        {
            (m_Enable) ? 1.0f : 0.0f,
            m_Translate.x, m_Translate.y, m_Translate.z,
            m_Direction.x, m_Direction.y, m_Direction.z,
            m_Aim.x, m_Aim.y, m_Aim.z,
            m_DistAttnStart, m_DistAttnEnd,
            m_AngleAttnStart, m_AngleAttnEnd,
            m_Color0.x, m_Color0.y, m_Color0.z,
            m_Color1.x, m_Color1.y, m_Color1.z,
        };
        for (int paramIdx = 0; paramIdx < PARAM_COUNT; ++paramIdx)
        {
            RAnimCurve& curve = m_Anims[paramIdx];
            curve.m_FullValues.push_back(baseValues[paramIdx]);
            curve.m_UseFlag = false;
            curve.m_AngleFlag = IsRotate(paramIdx);
        }
    }

    //! @brief 位置アニメーションを持つなら true を返します。
    bool HasPositionAnim() const
    {
        return (
            m_Anims[POSITION_X].m_UseFlag ||
            m_Anims[POSITION_Y].m_UseFlag ||
            m_Anims[POSITION_Z].m_UseFlag);
    }

    //! @brief 方向アニメーションを持つなら true を返します。
    bool HasDirectionAnim() const
    {
        return (
            m_Anims[DIRECTION_X].m_UseFlag ||
            m_Anims[DIRECTION_Y].m_UseFlag ||
            m_Anims[DIRECTION_Z].m_UseFlag);
    }

    //! @brief アニメーションを出力します。
    void OutputAnim(
        std::ostream& os,
        RDataStreamArray* pDataStreams,
        const int tabCount,
        const int index
    ) const;
};

//! @brief コンバータ用ライト配列の定義です。
typedef std::vector<CLight> CLightArray;

//=============================================================================
//! @brief コンバータ用フォグのクラスです。
//=============================================================================
class CFog : public RFog
{
public:
    bool m_IsAnimGot; //!< アニメーション取得済みなら true です。
    RAnimCurve m_Anims[PARAM_COUNT]; //!< アニメーションのパラメータに対応するアニメーションカーブ配列です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] name フォグ名です。
    //!
    explicit CFog(const std::string& name)
    : m_IsAnimGot(false)
    {
        m_Name = name;

        m_DistAttnStart = 0.0f;
        m_DistAttnEnd = 200.0f;

        m_FrameCount = 0;
        m_LoopAnim = false;
    }

    //! @brief デストラクタです。
    virtual ~CFog()
    {
    }

    //! @brief アニメーションを初期化します。
    void InitAnim()
    {
        m_IsAnimGot = false;
        // アニメーションカーブのベース値を設定ます。
        const float baseValues[PARAM_COUNT] =
        {
            m_DistAttnStart, m_DistAttnEnd,
            m_Color.x, m_Color.y, m_Color.z,
        };
        for (int paramIdx = 0; paramIdx < PARAM_COUNT; ++paramIdx)
        {
            RAnimCurve& curve = m_Anims[paramIdx];
            curve.m_FullValues.push_back(baseValues[paramIdx]);
            curve.m_UseFlag = false;
        }
    }

    //! @brief アニメーションを出力します。
    void OutputAnim(
        std::ostream& os,
        RDataStreamArray* pDataStreams,
        const int tabCount,
        const int index
    ) const;
};

//! @brief コンバータ用フォグ配列の定義です。
typedef std::vector<CFog> CFogArray;

//=============================================================================
//! @brief コンバータ用モデルのクラスです。
//=============================================================================
class CModel
{
public:
    CTexInfoArray m_TexInfos; //!< テクスチャ情報配列です。
    CMaterialArray m_Materials; //!< マテリアル配列です。
    RSkeleton m_Skeleton; //!< スケルトンです。
    CBoneArray m_Bones; //!< ボーン配列です。
    CShapeArray m_Shapes; //!< シェイプ配列です。
    CCameraArray m_Cameras; //!< カメラ配列です。
    CLightArray m_Lights; //!< ライト配列です。
    CFogArray m_Fogs; //!< フォグ配列です。

    int m_SmoothSkinningShapeCount; //!< スムーススキニングのシェイプ数です。
    int m_RigidSkinningShapeCount; //!< リジッドスキニングのシェイプ数です。
    int m_SmoothSkinningMtxCount; //!< スムーススキニングのシェイプ数です。
    int m_RigidSkinningMtxCount; //!< リジッドスキニングのシェイプ数です。
    int m_TotalTriangleCount; //!< 出力した三角形の総数です。
    int m_TotalIndexCount; //!< 出力した頂点インデックスの総数です。
    int m_TotalVertexCount; //!< 出力した頂点の総数です。
    int m_TotalProcessVertexCount; //!< 頂点シェーダが処理する頂点の総数です。

    RUserDataArray m_UserDatas; //!< ユーザーデータ配列です。
    RUserDataArray m_AnimUserDatas; //!< アニメーション用ユーザーデータ配列です。

public:
    //! @brief コンストラクタです。
    CModel()
    : m_SmoothSkinningShapeCount(0),
      m_RigidSkinningShapeCount(0),
      m_SmoothSkinningMtxCount(0),
      m_RigidSkinningMtxCount(0),
      m_TotalTriangleCount(0),
      m_TotalIndexCount(0),
      m_TotalVertexCount(0),
      m_TotalProcessVertexCount(0)
    {
    }

    //! @brief デストラクタです。
    virtual ~CModel()
    {
    }
};

//-----------------------------------------------------------------------------
//! @brief ダブルクォートで囲まれた文字列を返します。
//-----------------------------------------------------------------------------
std::string GetDoubleQuatedStr(const std::string& src)
{
    const size_t i0 = src.find_first_of("\"");
    const size_t i1 = src.find_last_of("\"");
    return (i0 != std::string::npos &&
            i1 != std::string::npos &&
            i0 < i1) ?
            src.substr(i0 + 1, i1 - i0 - 1) : "";
}

//-----------------------------------------------------------------------------
//! @brief 角括弧で囲まれた文字列を返します。
//-----------------------------------------------------------------------------
std::string GetBracketedStr(const std::string& src)
{
    const size_t i0 = src.find_first_of("[");
    const size_t i1 = src.find_last_of("]");
    return (i0 != std::string::npos &&
            i1 != std::string::npos &&
            i0 < i1) ?
            src.substr(i0 + 1, i1 - i0 - 1) : "";
}

//-----------------------------------------------------------------------------
//! @brief オプション文字列からリニア変換フラグを取得します。
//!
//! @param[in] str リニア変換フラグを表すオプション文字列です。
//!
//! @return リニア変換フラグを返します。
//-----------------------------------------------------------------------------
int GetLinearFlagFromOptString(const std::string& str)
{
    if (str.size() == R_RGBA_COUNT)
    {
        int linearFlag = C2NnOption::LinearFlag_None;
        for (int rgbaIdx = 0; rgbaIdx < R_RGBA_COUNT; ++rgbaIdx)
        {
            const char c = str[rgbaIdx];
            if (c == '0')
            {

            }
            else if (c == '1')
            {
                linearFlag |= LinearRgbaFlags[rgbaIdx];
            }
            else
            {
                return C2NnOption::LinearFlag_Invalid;
            }
        }
        return linearFlag;
    }
    return C2NnOption::LinearFlag_Invalid;
}

//-----------------------------------------------------------------------------
//! @brief リニア変換フラグからオプション文字列を取得します。
//!
//! @param[in] linearFlag リニア変換フラグです。
//!
//! @return オプション文字列を返します。
//-----------------------------------------------------------------------------
std::string GetOptStringFromLinearFlag(const int linearFlag)
{
    if (linearFlag < 0)
    {
        return "0000";
    }
    else
    {
        char strBuf[R_RGBA_COUNT];
        for (int rgbaIdx = 0; rgbaIdx < R_RGBA_COUNT; ++rgbaIdx)
        {
            strBuf[rgbaIdx] = ((linearFlag & LinearRgbaFlags[rgbaIdx]) != 0) ? '1' : '0';
        }
        return std::string(strBuf, sizeof(strBuf));
    }
}

//-----------------------------------------------------------------------------
//! @brief サポートしている NW4C 中間ファイルの拡張子なら true を返します。
//-----------------------------------------------------------------------------
bool IsSupportedNw4cExtension(const std::string& ext)
{
    return (
        ext == "cmdl"  ||
        ext == "ctex"  ||
        ext == "cskla" ||
        ext == "cmdla" ||
        ext == "cmata" ||
        ext == "cmcla" ||
        ext == "cmtsa" ||
        ext == "cmtpa" ||
        ext == "ccam"  ||
        ext == "clgt"  ||
        ext == "cfog"  ||
        ext == "cenv"  ||
        ext == "cres"
    );
}

//-----------------------------------------------------------------------------
//! @brief NintendoSDK 中間ファイルの拡張子なら true を返します。
//!
//! @param[in] ext 拡張子です。
//!
//! @return NintendoSDK 中間ファイルの拡張子なら true を返します。
//-----------------------------------------------------------------------------
bool IsNnExtension(const std::string& ext)
{
    return (
        ext == "fmdb" || ext == "fmda" ||
        ext == "ftxb" || ext == "ftxa" ||
        ext == "fskb" || ext == "fska" ||
        ext == "fmab" || ext == "fmaa" ||
        ext == "fspb" || ext == "fspa" ||
        ext == "fclb" || ext == "fcla" ||
        ext == "ftsb" || ext == "ftsa" ||
        ext == "ftpb" || ext == "ftpa" ||
        ext == "fvbb" || ext == "fvba" ||
        ext == "fvmb" || ext == "fvma" ||
        ext == "fshb" || ext == "fsha" ||
        ext == "fsnb" || ext == "fsna" ||
        ext == "fscb" || ext == "fsca" ||
        ext == "fsdb" || ext == "fsda" ||
        ext == "fsvb" || ext == "fsva"
    );
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルフォーマッターを実行します。
//!
//! @param[in] filePath 中間ファイルのパスです。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus Do3dFormatter(
    const std::string& filePath,
    const RExpOpt::FileType fileType,
    const C2NnOption& opt
)
{
    //-----------------------------------------------------------------------------
    // 現在 fmd ファイルのみ処理します。
    if (fileType != RExpOpt::FMD || !opt.m_ArrangesFormat)
    {
        return RStatus::SUCCESS;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルフォーマッターの exe ファイルが存在するかチェックします。
    std::string cvtrPath = opt.m_3dToolsPath + "3dIntermediateFileFormatter.exe";
    if (!RFileExists(cvtrPath))
    {
        return RStatus(RStatus::FAILURE, "The intermediate file formatter cannot be found: " + cvtrPath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルフォーマッターの引数を設定します。
    std::string cmd = "\"" + RGetWindowsFilePath(cvtrPath) + "\"";
    cmd += " -s"; // --silent
    cmd += " \"" + filePath + "\"";
    //cerr << "if formatter: " << cmd << endl;

    //-----------------------------------------------------------------------------
    // 中間ファイルフォーマッターを実行します。
    std::string errMsg;
    const int exitCode = RExecProcessWithPipe(nullptr, nullptr, &errMsg, cmd.c_str(), SW_HIDE);
    if (exitCode != 0)
    {
        return RStatus(RStatus::FAILURE, "The intermediate file formatter processing failed\n" + errMsg); // RShowError
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザを実行します。
//!
//! @param[in] filePath 中間ファイルのパスです。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus Do3dOptimizer(
    const std::string& filePath,
    const RExpOpt::FileType fileType,
    const C2NnOption& opt
)
{
    //-----------------------------------------------------------------------------
    // 最適化が不要なら中間ファイルフォーマッターを実行します。
    bool optimizes = false;
    //bool optimizes = true;
    if (!optimizes)
    {
        return Do3dFormatter(filePath, fileType, opt);
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザの exe ファイルが存在するかチェックします。
    std::string cvtrPath = opt.m_3dToolsPath + "3dIntermediateFileOptimizer.exe";
    if (!RFileExists(cvtrPath))
    {
        return RStatus(RStatus::FAILURE, "The intermediate file optimizer cannot be found: " + cvtrPath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザの引数を設定します。
    std::string cmd = "\"" + RGetWindowsFilePath(cvtrPath) + "\"";
    cmd += " -s"; // --silent
    cmd += " \"" + filePath + "\"";
    cmd += " --quantization-analysis";
    if (fileType == RExpOpt::FMD)
    {
        cmd += " --optimize-primitive --optimize-primitive-options=\"--mode=Normal\"";
    }
    //cerr << "optimizer: " << cmd << endl;

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザを実行します。
    std::string errMsg;
    const int exitCode = RExecProcessWithPipe(nullptr, nullptr, &errMsg, cmd.c_str(), SW_HIDE);
    if (exitCode != 0)
    {
        return RStatus(RStatus::FAILURE, "The intermediate file optimizer processing failed\n" + errMsg); // RShowError
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータを解析します。
//!
//! @param[in,out] pDatas ユーザーデータ配列へのポインタです。
//! @param[in] userDataElem ユーザーデータ要素です。
//-----------------------------------------------------------------------------
void ParseUserData(RUserDataArray* pDatas, const RXMLElement* userDataElem)
{
    pDatas->push_back(RUserData());
    RUserData& data = pDatas->back();

    //-----------------------------------------------------------------------------
    // 名前を取得します。
    // NintendoSDK では「ハイフン（-）」と「ドット（.）」が使用できないので
    // 「アンダーバー（_）」に変換します。
    // NW4C        で使用できる文字: [0-9A-Za-z\-\._]+
    // NintendoSDK で使用できる文字: [0-9A-Za-z_]+
    const RXMLElement* keyElem = userDataElem->FindElement("Key");
    data.m_Name = RAdjustElementNameString(keyElem->text);

    //-----------------------------------------------------------------------------
    // タイプを取得します。
    // 文字列の場合、STRING にしておいて、
    // 値がすべて ASCII 文字列でなければ WSTRING に変更します。
    //const std::string kind = userDataElem->GetAttribute("DataKind");
    if      (userDataElem->name == "IntegerArrayMetaDataXml") data.m_Type = RUserData::INT;
    else if (userDataElem->name == "FloatArrayMetaDataXml"  ) data.m_Type = RUserData::FLOAT;
    else data.m_Type = RUserData::STRING; // StringArrayMetaDataXml

    //-----------------------------------------------------------------------------
    // 値を取得します。
    const RXMLElement* valuesElem = userDataElem->FindElement("Values");
    for (size_t valueIdx = 0; valueIdx < valuesElem->nodes.size(); ++valueIdx)
    {
        const RXMLElement* valueElem = &valuesElem->nodes[valueIdx];
        if (data.m_Type == RUserData::INT)
        {
            const int iv = atol(valueElem->text.c_str());
            data.m_IntValues.push_back(iv);
        }
        else if (data.m_Type == RUserData::FLOAT)
        {
            const float fv = static_cast<float>(atof(valueElem->text.c_str()));
            data.m_FloatValues.push_back(fv);
        }
        else // STRING or WSTRING
        {
            const std::string sv = RDecodeXmlString(RGetShiftJisFromUtf8(valueElem->text));
            data.m_StringValues.push_back(sv);
            if (data.m_Type == RUserData::STRING && !RIsAsciiString(sv))
            {
                data.m_Type = RUserData::WSTRING;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータ配列を解析します。
//!
//! @param[in,out] pDatas ユーザーデータ配列へのポインタです。
//! @param[in] userDatasElem ユーザーデータ配列要素です。
//-----------------------------------------------------------------------------
void ParseUserDatas(RUserDataArray* pDatas, const RXMLElement* userDatasElem)
{
    if (userDatasElem != nullptr)
    {
        for (size_t dataIdx = 0; dataIdx < userDatasElem->nodes.size(); ++dataIdx)
        {
            ParseUserData(pDatas, &userDatasElem->nodes[dataIdx]);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief XYZ 属性を取得します。
//-----------------------------------------------------------------------------
void GetXyzAttr(RVec3& v, const RXMLElement* elem)
{
    v.x = static_cast<float>(atof(elem->GetAttribute("X").c_str()));
    v.y = static_cast<float>(atof(elem->GetAttribute("Y").c_str()));
    v.z = static_cast<float>(atof(elem->GetAttribute("Z").c_str()));
}

//-----------------------------------------------------------------------------
//! @brief RGB 属性を取得します。
//-----------------------------------------------------------------------------
void GetRgbAttr(RVec3& v, const RXMLElement* elem)
{
    v.x = static_cast<float>(atof(elem->GetAttribute("R").c_str()));
    v.y = static_cast<float>(atof(elem->GetAttribute("G").c_str()));
    v.z = static_cast<float>(atof(elem->GetAttribute("B").c_str()));
}

//-----------------------------------------------------------------------------
//! @brief RGBA 属性を取得します。
//-----------------------------------------------------------------------------
void GetRgbaAttr(RVec4& v, const RXMLElement* elem)
{
    v.x = static_cast<float>(atof(elem->GetAttribute("R").c_str()));
    v.y = static_cast<float>(atof(elem->GetAttribute("G").c_str()));
    v.z = static_cast<float>(atof(elem->GetAttribute("B").c_str()));
    v.w = static_cast<float>(atof(elem->GetAttribute("A").c_str()));
}

//-----------------------------------------------------------------------------
//! @brief オブジェクト名でオブジェクト配列を検索します。
//-----------------------------------------------------------------------------
template<typename T>
const int FindObjectByName(const std::vector<T>& objs, const std::string& name)
{
    for (size_t objIdx = 0; objIdx < objs.size(); ++objIdx)
    {
        if (objs[objIdx].m_Name == name)
        {
            return static_cast<int>(objIdx);
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief ユニークなオブジェクト名を取得します。
//!
//! @param[in] objs オブジェクト配列です。
//! @param[in] orgName 元のオブジェクト名です。
//!
//! @return ユニークなオブジェクト名を返します。
//-----------------------------------------------------------------------------
template<typename T>
std::string GetUniqueObjectName(const std::vector<T>& objs, const std::string& orgName)
{
    int id = 1;
    std::string name = orgName;
    while (FindObjectByName(objs, name) != -1)
    {
        name = orgName + "_" + RGetNumberString(id++);
    }
    return name;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ名でテクスチャ情報配列を検索します。
//-----------------------------------------------------------------------------
const int FindTexInfoByTexName(const CTexInfoArray& texInfos, const std::string& texName)
{
    for (size_t texInfoIdx = 0; texInfoIdx < texInfos.size(); ++texInfoIdx)
    {
        if (texInfos[texInfoIdx].m_TexName == texName)
        {
            return static_cast<int>(texInfoIdx);
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief ヒント情報でテクスチャ SRT アニメーション配列を検索します。
//-----------------------------------------------------------------------------
const int FindTexSrtAnimByHint(const CTexSrtAnimArray& texSrtAnims, const std::string& hint)
{
    for (size_t texSrtAnimIdx = 0; texSrtAnimIdx < texSrtAnims.size(); ++texSrtAnimIdx)
    {
        if (texSrtAnims[texSrtAnimIdx].m_Hint == hint)
        {
            return static_cast<int>(texSrtAnimIdx);
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief ヒント情報でテクスチャパターンアニメーション配列を検索します。
//-----------------------------------------------------------------------------
//const int FindTexPatAnimByHint(const CTexPatAnimArray& texPatAnims, const std::string& hint)
//{
//  for (size_t texPatAnimIdx = 0; texPatAnimIdx < texPatAnims.size(); ++texPatAnimIdx)
//  {
//      if (texPatAnims[texPatAnimIdx].m_Hint == hint)
//      {
//          return static_cast<int>(texPatAnimIdx);
//      }
//  }
//  return -1;
//}

//-----------------------------------------------------------------------------
//! @brief テクスチャ参照パスからテクスチャ名と ctex ファイルのパスを取得します。
//!
//! @param[out] pCtexPath ctex ファイルのパスを格納します。
//! @param[in] refPath テクスチャ参照パスです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] srcRootFolder NW4C 中間ファイルが存在するフォルダのパスです。
//!
//! @return テクスチャ名を返します。
//-----------------------------------------------------------------------------
std::string GetTexNameFromRefPath(
    std::string* pCtexPath,
    const std::string& refPath,
    const CIntermediateFile& inputFile,
    const std::string& srcRootFolder
)
{
    //-----------------------------------------------------------------------------
    // テクスチャ名を取得します。
    const size_t dq0Idx = refPath.find_first_of("\"");
    const size_t dq1Idx = refPath.find_last_of("\"");
    std::string texName = refPath.substr(dq0Idx + 1, dq1Idx - dq0Idx - 1);

    //-----------------------------------------------------------------------------
    // ctex ファイルのパスを取得します。
    const size_t clIdx = refPath.find_first_of(":");
    if (clIdx == std::string::npos) // 入力ファイル内のテクスチャを参照している場合
    {
        *pCtexPath = inputFile.m_Path;
    }
    else
    {
        const std::string subPath = refPath.substr(clIdx + 1); // @file: 形式のみ対応
        *pCtexPath = srcRootFolder + subPath;
        if (!RFileExists(*pCtexPath)) // 指定されたパスになければルートフォルダ直下のファイルを使用します。
        {
            *pCtexPath = srcRootFolder + RGetFileNameFromFilePath(subPath);
            if (!RFileExists(*pCtexPath)) // ルートフォルダ直下にもなければルートフォルダ/Textures のファイルを使用します。
            {
                *pCtexPath = srcRootFolder + Nw4cTexFolderName + "/" + RGetFileNameFromFilePath(subPath);
            }
        }
        *pCtexPath = RGetFullFilePath(*pCtexPath, true);
    }

    //-----------------------------------------------------------------------------
    // テクスチャ名が * の場合は、ctex ファイル名をテクスチャ名とします。
    if (texName == "*")
    {
        texName = RGetNoExtensionFilePath(RGetFileNameFromFilePath(*pCtexPath));
    }
    return texName;
}

//-----------------------------------------------------------------------------
//! @brief 指定フォルダに存在する全 ctex ファイルからテクスチャ情報を追加します。
//!
//! @param[in,out] pTexInfos テクスチャ情報配列へのポインタです。
//! @param[in] texFolderPath テクスチャフォルダのパスです。
//-----------------------------------------------------------------------------
void AddTexInfosFromFolder(
    CTexInfoArray* pTexInfos,
    const std::string& texFolderPath
)
{
    const std::string findPath = texFolderPath + "*.ctex";
    WIN32_FIND_DATA ffd;
    HANDLE hFindFile = FindFirstFileA(findPath.c_str(), &ffd);
    if (hFindFile != INVALID_HANDLE_VALUE)
    {
        do
        {
            if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
            {
                const std::string texName = RGetNoExtensionFilePath(ffd.cFileName);
                if (FindTexInfoByTexName(*pTexInfos, texName) == -1)
                {
                    const std::string ctexPath = texFolderPath + ffd.cFileName;
                    pTexInfos->push_back(CTexInfo(texName, ctexPath, CTexInfo::HintAuto, "", true));
                }
            }
        } while (FindNextFile(hFindFile, &ffd));
        FindClose(hFindFile);
    }
}

//-----------------------------------------------------------------------------
//! @brief サンプラのラップを取得します。
//-----------------------------------------------------------------------------
RSampler::Wrap GetSamplerWrap(const std::string& wrap)
{
    return
        (wrap == "Repeat"        ) ? RSampler::REPEAT :
        (wrap == "MirroredRepeat") ? RSampler::MIRROR :
        RSampler::CLAMP; // ClampToEdge / ClampToBorder
}

//-----------------------------------------------------------------------------
//! @brief サンプラの拡大縮小フィルタを取得します。
//-----------------------------------------------------------------------------
RSampler::Filter GetSamplerMagMinFilter(const std::string& filter)
{
    if (filter == "Nearest" ||
        filter == "NearestMipmapNearest" ||
        filter == "NearestMipmapLinear")
    {
        return RSampler::POINT;
    }
    else if (filter == "Linear" ||
             filter == "LinearMipmapNearest" ||
             filter == "LinearMipmapLinear")
    {
        return RSampler::LINEAR;
    }
    else
    {
        return RSampler::NONE;
    }
}

//-----------------------------------------------------------------------------
//! @brief サンプラのミップマップフィルタを取得します。
//-----------------------------------------------------------------------------
RSampler::Filter GetSamplerMipFilter(const std::string& filter)
{
    if (filter == "NearestMipmapNearest" ||
        filter == "LinearMipmapNearest")
    {
        return RSampler::POINT;
    }
    else if (filter == "NearestMipmapLinear" ||
             filter == "LinearMipmapLinear")
    {
        return RSampler::LINEAR;
    }
    else
    {
        return RSampler::NONE;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ行列モードを取得します。
//-----------------------------------------------------------------------------
ROriginalTexsrt::Mode GetTexMtxMode(const std::string& mode)
{
    return
        (mode == "DccMaya"  ) ? ROriginalTexsrt::MAYA :
        (mode == "Dcc3dsMax") ? ROriginalTexsrt::MAX  :
        ROriginalTexsrt::SOFTIMAGE; // DccSoftimage
}

//-----------------------------------------------------------------------------
//! @brief テクスチャマッパを解析します。
//!
//! @param[in,out] pSamplers サンプラ配列へのポインタです。
//! @param[in,out] pTexInfos テクスチャ情報配列へのポインタです。
//! @param[in] texMapElem テクスチャマッパ要素です。
//! @param[in] coordinatorElem テクスチャコーディネイタ要素です。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] srcRootFolder NW4C 中間ファイルが存在するフォルダのパスです。
//! @param[in] convertsTex ctex ファイルを ftx ファイルに変換するなら true です。
//! @param[in] orgMatName オリジナルマテリアル名です。
//! @param[in] bumpTexIdx バンプテクスチャのテクスチャユニットインデックスです。
//!                       バンプテクスチャを使用しない場合は -1 です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseTexMap(
    RSamplerArray* pSamplers,
    CTexInfoArray* pTexInfos,
    const RXMLElement* texMapElem,
    const RXMLElement* coordinatorElem,
    const CIntermediateFile& inputFile,
    const std::string& srcRootFolder,
    const bool convertsTex,
    const std::string& orgMatName,
    const int bumpTexIdx
)
{
    //-----------------------------------------------------------------------------
    // テクスチャ名と ctex ファイルのパスを取得します。
    const std::string refPath = RDecodeXmlString(texMapElem->FindElement("TextureReference")->text);
    std::string ctexPath;
    const std::string texName = GetTexNameFromRefPath(&ctexPath, refPath, inputFile, srcRootFolder);
    //cerr << "tex map: " << texName << ": " << ctexPath << R_ENDL;
    const bool ctexExists = RFileExists(ctexPath);
    if (!ctexExists && convertsTex)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + ctexPath + // RShowError
            " (" + orgMatName + ")");
    }

    //-----------------------------------------------------------------------------
    // テクスチャコーディネイタの属性を取得します。
    ROriginalTexsrt texSrt;
    bool isReflection = false;
    if (coordinatorElem != nullptr)
    {
        const std::string mappingMethod = coordinatorElem->GetAttribute("MappingMethod");
        isReflection = (mappingMethod == "CameraCubeEnvMap" || mappingMethod == "CameraSphereEnvMap");
        texSrt.m_Mode        = GetTexMtxMode(coordinatorElem->GetAttribute("MatrixMode"));
        texSrt.m_Scale.x     = static_cast<float>(atof(coordinatorElem->GetAttribute("ScaleS").c_str()));
        texSrt.m_Scale.y     = static_cast<float>(atof(coordinatorElem->GetAttribute("ScaleT").c_str()));
        texSrt.m_Rotate      = static_cast<float>(atof(coordinatorElem->GetAttribute("Rotate").c_str()));
        texSrt.m_Translate.x = static_cast<float>(atof(coordinatorElem->GetAttribute("TranslateS").c_str()));
        texSrt.m_Translate.y = static_cast<float>(atof(coordinatorElem->GetAttribute("TranslateT").c_str()));
        texSrt.m_UvHintIdx   = atol(coordinatorElem->GetAttribute("SourceCoordinate").c_str());
    }

    //-----------------------------------------------------------------------------
    // サンプラを追加します。
    const RSampler::Hint hint = (static_cast<int>(pSamplers->size()) == bumpTexIdx) ?
        RSampler::NORMAL : ((isReflection) ? RSampler::REFLECTION : RSampler::ALBEDO);
    int hintIndex = 0;
    for (size_t samplerIdx = 0; samplerIdx < pSamplers->size(); ++samplerIdx)
    {
        if ((*pSamplers)[samplerIdx].m_Hint == hint)
        {
            hintIndex = (*pSamplers)[samplerIdx].m_HintIndex + 1;
        }
    }
    pSamplers->push_back(RSampler(hint, hintIndex));
    RSampler& sampler = pSamplers->back();

    //-----------------------------------------------------------------------------
    // テクスチャサンプラの属性を取得します。
    const RXMLElement* samplerElem = texMapElem->FindElement("StandardTextureSamplerCtr");
    sampler.m_TexName = texName;
    sampler.m_WrapU = GetSamplerWrap(samplerElem->GetAttribute("WrapS"));
    sampler.m_WrapV = GetSamplerWrap(samplerElem->GetAttribute("WrapT"));
    sampler.m_FilterMag = GetSamplerMagMinFilter(samplerElem->GetAttribute("MagFilter"));
    sampler.m_FilterMin = GetSamplerMagMinFilter(samplerElem->GetAttribute("MinFilter"));
    sampler.m_FilterMip = GetSamplerMipFilter(samplerElem->GetAttribute("MinFilter"));
    sampler.m_LodMin  = static_cast<float>(atof(samplerElem->GetAttribute("MinLod").c_str()));
    sampler.m_LodBias = static_cast<float>(atof(samplerElem->GetAttribute("LodBias").c_str()));
    sampler.m_OriginalTexsrt = texSrt;

    //-----------------------------------------------------------------------------
    // テクスチャ情報を追加します。
    const std::string imgHint = sampler.GetImageHintString();
    int texInfoIdx = FindTexInfoByTexName(*pTexInfos, texName);
    if (texInfoIdx == -1)
    {
        //texInfoIdx = static_cast<int>(*pTexInfos.size());
        pTexInfos->push_back(CTexInfo(texName, ctexPath, imgHint, "", true));
        //if (ctexExists)
        //{
        //  // ミップマップ数を取得し、ミップマップありなら MinFilter 属性がミップマップ用でなくても
        //  // m_FilterMip を POINT または LINEAR に設定？
        //}
    }
    else
    {
        if ((*pTexInfos)[texInfoIdx].m_Hint != imgHint)
        {
            // TODO: 必要なら警告（Texture is used for different usage）
        }
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 表示面を返します。
//-----------------------------------------------------------------------------
RRenderState::DisplayFace GetDisplayFace(const std::string& cullingMode)
{
    return
        (cullingMode == "Never"    ) ? RRenderState::BOTH  :
        (cullingMode == "BackFace" ) ? RRenderState::FRONT :
        (cullingMode == "FrontFace") ? RRenderState::BACK  :
        RRenderState::FACE_NONE; // Always
}

//-----------------------------------------------------------------------------
//! @brief ブレンドモードを返します。
//-----------------------------------------------------------------------------
RRenderState::BlendMode GetBlendMode(const std::string& blendMode)
{
    if (blendMode == "Blend" || blendMode == "SeparateBlend")
    {
        return RRenderState::COLOR;
    }
    else if (blendMode == "Logic")
    {
        return RRenderState::LOGIC;
    }
    else
    {
        return RRenderState::BLEND_NONE; // NotUsed
    }
}

//-----------------------------------------------------------------------------
//! @brief デプステストの関数を返します。
//-----------------------------------------------------------------------------
RDepthTest::Func GetDepthTestFunc(const std::string& func)
{
    return
        (func == "Never"   ) ? RDepthTest::NEVER   :
        (func == "Less"    ) ? RDepthTest::LESS    :
        (func == "Equal"   ) ? RDepthTest::EQUAL   :
        (func == "Lequal"  ) ? RDepthTest::LEQUAL  :
        (func == "Greater" ) ? RDepthTest::GREATER :
        (func == "Notequal") ? RDepthTest::NEQUAL  :
        (func == "Gequal"  ) ? RDepthTest::GEQUAL  :
        RDepthTest::ALWAYS; // Always
}

//-----------------------------------------------------------------------------
//! @brief アルファテストの関数を返します。
//-----------------------------------------------------------------------------
RAlphaTest::Func GetAlphaTestFunc(const std::string& func)
{
    return
        (func == "Never"   ) ? RAlphaTest::NEVER   :
        (func == "Less"    ) ? RAlphaTest::LESS    :
        (func == "Equal"   ) ? RAlphaTest::EQUAL   :
        (func == "Lequal"  ) ? RAlphaTest::LEQUAL  :
        (func == "Greater" ) ? RAlphaTest::GREATER :
        (func == "Notequal") ? RAlphaTest::NEQUAL  :
        (func == "Gequal"  ) ? RAlphaTest::GEQUAL  :
        RAlphaTest::ALWAYS; // Always
}

//-----------------------------------------------------------------------------
//! @brief カラーブレンドの重み係数を取得します。
//!
//! @param[in] factor <RgbParameter> の重み係数です。
//!
//! @return カラーブレンドの重み係数を返します。
//-----------------------------------------------------------------------------
RColorBlend::Func GetColorBlendFactor(const std::string& factor)
{
    return
        (factor == "Zero"                    ) ? RColorBlend::ZERO                  :
        (factor == "One"                     ) ? RColorBlend::ONE                   :
        (factor == "SourceColor"             ) ? RColorBlend::SRC_COLOR             :
        (factor == "OneMinusSourceColor"     ) ? RColorBlend::ONE_MINUS_SRC_COLOR   :
        (factor == "DestinationColor"        ) ? RColorBlend::DST_COLOR             :
        (factor == "OneMinusDestinationColor") ? RColorBlend::ONE_MINUS_DST_COLOR   :
        (factor == "SourceAlpha"             ) ? RColorBlend::SRC_ALPHA             :
        (factor == "OneMinusSourceAlpha"     ) ? RColorBlend::ONE_MINUS_SRC_ALPHA   :
        (factor == "DestinationAlpha"        ) ? RColorBlend::DST_ALPHA             :
        (factor == "OneMinusDestinationAlpha") ? RColorBlend::ONE_MINUS_DST_ALPHA   :
        (factor == "ConstantColor"           ) ? RColorBlend::CONST_COLOR           :
        (factor == "OneMinusConstantColor"   ) ? RColorBlend::ONE_MINUS_CONST_COLOR :
        (factor == "ConstantAlpha"           ) ? RColorBlend::CONST_ALPHA           :
        (factor == "OneMinusConstantAlpha"   ) ? RColorBlend::ONE_MINUS_CONST_ALPHA :
        RColorBlend::SRC_ALPHA_SATURATE; // SourceAlphaSaturate

    // 以下は CTR には存在しません。
    // RColorBlend::SRC1_COLOR
    // RColorBlend::ONE_MINUS_SRC1_COLOR
    // RColorBlend::SRC1_ALPHA
    // RColorBlend::ONE_MINUS_SRC1_ALPHA
}

//-----------------------------------------------------------------------------
//! @brief アルファブレンドの重み係数を取得します。
//!
//! @param[in] factor <AlphaParameter> の重み係数です。
//!
//! @return アルファブレンドの重み係数を返します。
//-----------------------------------------------------------------------------
RColorBlend::Func GetAlphaBlendFactor(const std::string& factor)
{
    return
        (factor == "Zero"                    ) ? RColorBlend::ZERO                  :
        (factor == "One"                     ) ? RColorBlend::ONE                   :
        (factor == "SourceColor"             ) ? RColorBlend::SRC_ALPHA             :
        (factor == "OneMinusSourceColor"     ) ? RColorBlend::ONE_MINUS_SRC_ALPHA   :
        (factor == "DestinationColor"        ) ? RColorBlend::DST_ALPHA             :
        (factor == "OneMinusDestinationColor") ? RColorBlend::ONE_MINUS_DST_ALPHA   :
        (factor == "SourceAlpha"             ) ? RColorBlend::SRC_ALPHA             :
        (factor == "OneMinusSourceAlpha"     ) ? RColorBlend::ONE_MINUS_SRC_ALPHA   :
        (factor == "DestinationAlpha"        ) ? RColorBlend::DST_ALPHA             :
        (factor == "OneMinusDestinationAlpha") ? RColorBlend::ONE_MINUS_DST_ALPHA   :
        (factor == "ConstantColor"           ) ? RColorBlend::CONST_ALPHA           :
        (factor == "OneMinusConstantColor"   ) ? RColorBlend::ONE_MINUS_CONST_ALPHA :
        (factor == "ConstantAlpha"           ) ? RColorBlend::CONST_ALPHA           :
        (factor == "OneMinusConstantAlpha"   ) ? RColorBlend::ONE_MINUS_CONST_ALPHA :
        RColorBlend::SRC_ALPHA_SATURATE; // SourceAlphaSaturate
}

//-----------------------------------------------------------------------------
//! @brief カラーブレンドのオペレーションを返します。
//-----------------------------------------------------------------------------
RColorBlend::Op GetColorBlendOp(const std::string& equation)
{
    return
        (equation == "FuncAdd"            ) ? RColorBlend::ADD           :
        (equation == "FuncSubtract"       ) ? RColorBlend::SRC_MINUS_DST :
        (equation == "FuncReverseSubtract") ? RColorBlend::DST_MINUS_SRC :
        (equation == "Min"                ) ? RColorBlend::MIN           :
        RColorBlend::MAX; // Max
}

//-----------------------------------------------------------------------------
//! @brief 論理ブレンドのオペレーションを返します。
//-----------------------------------------------------------------------------
RLogicalBlend::Op GetLogicalBlendOp(const std::string& op)
{
    return
        (op == "Clear"       ) ? RLogicalBlend::CLEAR    :
        (op == "Copy"        ) ? RLogicalBlend::COPY     :
        (op == "NoOp"        ) ? RLogicalBlend::NO_OP    :
        (op == "Set"         ) ? RLogicalBlend::SET      :
        (op == "CopyInverted") ? RLogicalBlend::INV_COPY :
        (op == "Invert"      ) ? RLogicalBlend::INV      :
        (op == "AndReverse"  ) ? RLogicalBlend::REV_AND  :
        (op == "OrReverse"   ) ? RLogicalBlend::REV_OR   :
        (op == "And"         ) ? RLogicalBlend::AND      :
        (op == "Or"          ) ? RLogicalBlend::OR       :
        (op == "Nand"        ) ? RLogicalBlend::NAND     :
        (op == "Nor"         ) ? RLogicalBlend::NOR      :
        (op == "Xor"         ) ? RLogicalBlend::XOR      :
        (op == "Equiv"       ) ? RLogicalBlend::EQUIV    :
        (op == "AndInverted" ) ? RLogicalBlend::INV_AND  :
        RLogicalBlend::INV_OR; // OrInverted
}

//-----------------------------------------------------------------------------
//! @brief ブレンドオペレーションを解析します。
//!
//! @param[in,out] pRenderState レンダーステートへのポインタです。
//! @param[in] blendOpElem ブレンドオペレーション要素です。
//-----------------------------------------------------------------------------
void ParseBlendOperation(
    RRenderState* pRenderState,
    const RXMLElement* blendOpElem
)
{
    //-----------------------------------------------------------------------------
    // ブレンドモードを取得します。
    const std::string blendMode = blendOpElem->GetAttribute("Mode");
    pRenderState->m_BlendMode = GetBlendMode(blendMode);

    //-----------------------------------------------------------------------------
    // カラーブレンドを取得します。
    RColorBlend& colorBlend = pRenderState->m_ColorBlend;

    if (blendMode == "NotUsed" || blendMode == "Logic")
    {
        colorBlend.m_RgbSrcFunc   = RColorBlend::SRC_ALPHA;
        colorBlend.m_RgbDstFunc   = RColorBlend::ONE_MINUS_SRC_ALPHA;
        colorBlend.m_RgbOp        = RColorBlend::ADD;
        colorBlend.m_AlphaSrcFunc = RColorBlend::ONE;
        colorBlend.m_AlphaDstFunc = RColorBlend::ZERO;
        colorBlend.m_AlphaOp      = RColorBlend::ADD;
        colorBlend.m_ConstColor   = RVec4::kZero;
    }
    else
    {
        const RXMLElement* rgbElem = blendOpElem->FindElement("RgbParameter");
        colorBlend.m_RgbSrcFunc = GetColorBlendFactor(rgbElem->GetAttribute("BlendFunctionSource"));
        colorBlend.m_RgbDstFunc = GetColorBlendFactor(rgbElem->GetAttribute("BlendFunctionDestination"));
        colorBlend.m_RgbOp = GetColorBlendOp(rgbElem->GetAttribute("BlendEquation"));

        if (blendMode == "Blend")
        {
            colorBlend.m_AlphaSrcFunc = GetAlphaBlendFactor(rgbElem->GetAttribute("BlendFunctionSource"));
            colorBlend.m_AlphaDstFunc = GetAlphaBlendFactor(rgbElem->GetAttribute("BlendFunctionDestination"));
            colorBlend.m_AlphaOp      = colorBlend.m_RgbOp;
        }
        else // SeparateBlend
        {
            const RXMLElement* alphaElem = blendOpElem->FindElement("AlphaParameter");
            colorBlend.m_AlphaSrcFunc = GetAlphaBlendFactor(alphaElem->GetAttribute("BlendFunctionSource"));
            colorBlend.m_AlphaDstFunc = GetAlphaBlendFactor(alphaElem->GetAttribute("BlendFunctionDestination"));
            colorBlend.m_AlphaOp = GetColorBlendOp(alphaElem->GetAttribute("BlendEquation"));
        }

        const RXMLElement* colorElem = blendOpElem->FindElement("BlendColor");
        GetRgbaAttr(colorBlend.m_ConstColor, colorElem);
    }

    //-----------------------------------------------------------------------------
    // 論理ブレンドを取得します。
    pRenderState->m_LogicalBlend.m_Op = GetLogicalBlendOp(blendOpElem->GetAttribute("LogicOperation"));

    //-----------------------------------------------------------------------------
    // レンダーステートのモードを設定します。
    const RDepthTest& depthTest = pRenderState->m_DepthTest;
    const RAlphaTest& alphaTest = pRenderState->m_AlphaTest;

    if (depthTest.m_Func == RDepthTest::LESS)
    {
        // 標準のレンダーステートのモードにするために
        // NW4C でのデフォルトの LESS なら
        // NintendoSDK でのデフォルトの LEQUAL に変更します。
        const_cast<RDepthTest&>(depthTest).m_Func = RDepthTest::LEQUAL;
    }

    if (!alphaTest.m_Enable)
    {
        // アルファテストが無効なら NintendoSDK でのデフォルトの GEQUAL & 0.5 に変更します。
        const_cast<RAlphaTest&>(alphaTest).m_Func = RAlphaTest::GEQUAL;
        const_cast<RAlphaTest&>(alphaTest).m_Value = 0.5f;
    }
    else if (alphaTest.m_Func == RAlphaTest::GREATER && alphaTest.m_Value != 0.0f)
    {
        // アルファテストが有効で、関数が
        // NW4C でのデフォルトの GREATER（比較値が 0 以外）なら
        // NintendoSDK でのデフォルトの GEQUAL に変更します。
        const_cast<RAlphaTest&>(alphaTest).m_Func = RAlphaTest::GEQUAL;
    }

    pRenderState->m_Mode = RRenderState::CUSTOM;
    const bool isStandardDepthTest = (depthTest.m_Enable && depthTest.m_Func == RDepthTest::LEQUAL);
    if (pRenderState->m_BlendMode == RRenderState::BLEND_NONE)
    {
        if (isStandardDepthTest && depthTest.m_Write)
        {
            if (alphaTest.m_Enable)
            {
                pRenderState->m_Mode = RRenderState::MASK;
            }
            else
            {
                pRenderState->m_Mode = RRenderState::OPA;
            }
        }
    }
    else if (pRenderState->m_BlendMode == RRenderState::COLOR)
    {
        if (isStandardDepthTest && !depthTest.m_Write)
        {
            pRenderState->m_Mode = RRenderState::XLU;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief マテリアルを解析します。
//!
//! @param[in,out] pMaterials マテリアル配列へのポインタです。
//! @param[in,out] pTexInfos テクスチャ情報配列へのポインタです。
//! @param[in] matElem マテリアル要素です。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] srcRootFolder NW4C 中間ファイルが存在するフォルダのパスです。
//! @param[in] convertsTex ctex ファイルを ftx ファイルに変換するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseMaterial(
    CMaterialArray* pMaterials,
    CTexInfoArray* pTexInfos,
    const RXMLElement* matElem,
    const CIntermediateFile& inputFile,
    const std::string& srcRootFolder,
    const bool convertsTex
)
{
    RStatus status;

    pMaterials->push_back(CMaterial());
    CMaterial& mat = pMaterials->back();

    //-----------------------------------------------------------------------------
    // 属性を取得します。
    const std::string orgMatName = matElem->GetAttribute("Name");
    mat.m_Name = RAdjustElementNameString(orgMatName);
    mat.m_CompressEnable = (matElem->GetAttribute("IsCompressible") == "true");

    //-----------------------------------------------------------------------------
    // カラーを取得します。
    const RXMLElement* colElem = matElem->FindElement("MaterialColor");
    const RXMLElement* diffuseElem = colElem->FindElement("Diffuse");
    GetRgbAttr(mat.m_Diffuse , diffuseElem);
    GetRgbAttr(mat.m_Ambient , colElem->FindElement("Ambient"));
        // ↑ NW4C では DCC ツールから常に (1, 1, 1, 0) を出力していた
        //    NintendoSDK では DCC ツール上のマテリアルの Ambient カラーを反映
    GetRgbAttr(mat.m_Emission, colElem->FindElement("Emission"));
    GetRgbAttr(mat.m_Specular, colElem->FindElement("Specular0"));
    const float alpha = static_cast<float>(atof(diffuseElem->GetAttribute("A").c_str()));
    mat.m_Opacity = RVec3(alpha, alpha, alpha);

    //-----------------------------------------------------------------------------
    // ラスタライゼーションを解析します。
    const RXMLElement* rasterElem = matElem->FindElement("Rasterization");
    RRenderState& renderState = mat.m_RenderState;
    renderState.m_DisplayFace = GetDisplayFace(rasterElem->GetAttribute("CullingMode"));

    //-----------------------------------------------------------------------------
    // フラグメントバンプを解析します。
    const RXMLElement* fragShaderElem = matElem->FindElement("FragmentShader");
    const RXMLElement* bumpElem = fragShaderElem->FindElement("FragmentBump");
    const std::string bumpMode = bumpElem->GetAttribute("BumpMode"); // NotUsed / AsBump / AsTangent
    int bumpTexIdx = -1;
    if (bumpMode == "AsBump")
    {
        const std::string bumpTex = bumpElem->GetAttribute("BumpTextureIndex");
        bumpTexIdx = bumpTex[bumpTex.size() - 1] - '0';
    }

    //-----------------------------------------------------------------------------
    // フラグメントライティングを解析します。
    const RXMLElement* fragLgtElem = fragShaderElem->FindElement("FragmentLighting");
    mat.m_IsSpecularEnable = (
        fragLgtElem->GetAttribute("IsDistribution0Enabled") == "true" ||
        fragLgtElem->GetAttribute("IsDistribution1Enabled") == "true");

    //-----------------------------------------------------------------------------
    // テクスチャマッパを解析します。
    const RXMLElement* texMapsElem = matElem->FindElement("TextureMappers", false);
    const RXMLElement* coordinatorsElem = matElem->FindElement("TextureCoordinators", false);
    if (texMapsElem != nullptr && !texMapsElem->nodes.empty())
    {
        for (size_t texMapIdx = 0; texMapIdx < texMapsElem->nodes.size(); ++texMapIdx)
        {
            const RXMLElement* coordinatorElem =
                (coordinatorsElem != nullptr && texMapIdx < coordinatorsElem->nodes.size()) ?
                &coordinatorsElem->nodes[texMapIdx] : nullptr;
            status = ParseTexMap(&mat.m_Samplers, pTexInfos,
                &texMapsElem->nodes[texMapIdx], coordinatorElem,
                inputFile, srcRootFolder, convertsTex, orgMatName, bumpTexIdx);
            RCheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // デプスオペレーションを解析します。
    const RXMLElement* fragOpElem = matElem->FindElement("FragmentOperation");
    const RXMLElement* depthOpElem = fragOpElem->FindElement("DepthOperation");
    RDepthTest& depthTest = renderState.m_DepthTest;
    depthTest.m_Enable = (depthOpElem->GetAttribute("IsTestEnabled") == "true");
    depthTest.m_Func = GetDepthTestFunc(depthOpElem->GetAttribute("TestFunction"));
    depthTest.m_Write = (depthOpElem->GetAttribute("IsMaskEnabled") == "true");

    //-----------------------------------------------------------------------------
    // アルファテストを解析します。
    const RXMLElement* alphaTestElem = fragShaderElem->FindElement("AlphaTest");
    RAlphaTest& alphaTest = renderState.m_AlphaTest;
    alphaTest.m_Enable = (alphaTestElem->GetAttribute("IsTestEnabled") == "true");
    alphaTest.m_Func = GetAlphaTestFunc(alphaTestElem->GetAttribute("TestFunction"));
    alphaTest.m_Value = static_cast<float>(atof(alphaTestElem->GetAttribute("TestReference").c_str()));

    //-----------------------------------------------------------------------------
    // ブレンドオペレーションを解析します。
    const RXMLElement* blendOpElem = fragOpElem->FindElement("BlendOperation");
    ParseBlendOperation(&renderState, blendOpElem);

    //-----------------------------------------------------------------------------
    // マテリアル用ユーザーデータ配列を取得します。
    ParseUserDatas(&mat.m_UserDatas, matElem->FindElement("UserData", false));

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ファイルマテリアル配列内のマテリアル要素を取得します。
//!
//! @param[in] fileMatsElem ファイルマテリアル配列要素です。
//! @param[in] refPath マテリアル参照パスです。
//!
//! @return マテリアル要素を返します。
//-----------------------------------------------------------------------------
const RXMLElement* GetFileMaterialElem(
    const RXMLElement* fileMatsElem,
    const std::string& refPath
)
{
    if (fileMatsElem != nullptr)
    {
        const std::string matName = GetDoubleQuatedStr(refPath);
        for (size_t matIdx = 0; matIdx < fileMatsElem->nodes.size(); ++matIdx)
        {
            const RXMLElement* matElem = &fileMatsElem->nodes[matIdx];
            if (matElem->GetAttribute("Name") == matName)
            {
                return matElem;
            }
        }
    }
    return nullptr;
}

//-----------------------------------------------------------------------------
//! @brief 実体のあるマテリアル要素を取得します。
//!
//! @param[in] fileMatsElem ファイルマテリアル配列要素です。
//! @param[in] matElem マテリアル要素です。
//!
//! @return マテリアル要素を返します。
//-----------------------------------------------------------------------------
const RXMLElement* GetRealMaterialElem(
    const RXMLElement* fileMatsElem,
    const RXMLElement* matElem
)
{
    if (matElem->name == "MaterialReference")
    {
        return GetFileMaterialElem(fileMatsElem, matElem->text);
    }
    else
    {
        return matElem;
    }
}

//-----------------------------------------------------------------------------
//! @brief マテリアル群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc <material_array> 要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputMaterials(std::ostream& os, const int tc, const CModel& cmodel)
{
    const int matCount = static_cast<int>(cmodel.m_Materials.size());
    if (matCount > 0)
    {
        os << RTab(tc) << "<material_array length=\"" << matCount << "\">" << R_ENDL;
        for (int matIdx = 0; matIdx < matCount; ++matIdx)
        {
            const CMaterial& mat = cmodel.m_Materials[matIdx];
            mat.Out(os, tc + 1, matIdx);
        }
        os << RTab(tc) << "</material_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief オリジナルマテリアル群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc <original_material_array> 要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputOriginalMaterials(std::ostream& os, const int tc, const CModel& cmodel)
{
    const int matCount = static_cast<int>(cmodel.m_Materials.size());
    if (matCount > 0)
    {
        os << RTab(tc) << "<original_material_array length=\"" << matCount << "\">" << R_ENDL;
        for (int matIdx = 0; matIdx < matCount; ++matIdx)
        {
            const CMaterial& mat = cmodel.m_Materials[matIdx];
            mat.OutOriginal(os, tc + 1, matIdx);
        }
        os << RTab(tc) << "</original_material_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief ノードビジビリティを解析します。
//-----------------------------------------------------------------------------
void ParseNodeVis(CNodeVisArray& nodeViss, const RXMLElement* nodeVisElem)
{
    const std::string name = RAdjustElementNameString(nodeVisElem->GetAttribute("Name"));
    const bool visibility = (nodeVisElem->GetAttribute("IsVisible") == "true");
    nodeViss.push_back(CNodeVis(name, visibility));
}

//-----------------------------------------------------------------------------
//! @brief ビルボード設定を取得します。
//-----------------------------------------------------------------------------
RBone::Billboard GetBillboard(const std::string& billboardMode)
{
    return
        (billboardMode == "World"          ) ? RBone::WORLD_VIEWVECTOR  :
        (billboardMode == "WorldViewpoint" ) ? RBone::WORLD_VIEWPOINT   :
        (billboardMode == "Screen"         ) ? RBone::SCREEN_VIEWVECTOR :
        (billboardMode == "ScreenViewpoint") ? RBone::SCREEN_VIEWPOINT  :
        (billboardMode == "YAxial"         ) ? RBone::YAXIS_VIEWVECTOR  :
        (billboardMode == "YAxialViewpoint") ? RBone::YAXIS_VIEWPOINT   :
        RBone::BILLBOARD_NONE;
}

//-----------------------------------------------------------------------------
//! @brief ノードの変換を解析します。
//!
//! @param[in,out] pNode トランスフォームノードへのポインタです。
//! @param[in] transformElem 変換要素です。
//! @param[in] conversionMagnify 移動値に掛ける倍率です。
//-----------------------------------------------------------------------------
void ParseTransform(
    RTransformNode* pNode,
    const RXMLElement* transformElem,
    const float conversionMagnify
)
{
    GetXyzAttr(pNode->m_Scale    , transformElem->FindElement("Scale"));
    GetXyzAttr(pNode->m_Rotate   , transformElem->FindElement("Rotate")); // radian
    GetXyzAttr(pNode->m_Translate, transformElem->FindElement("Translate"));
    pNode->m_Rotate *= static_cast<float>(R_M_RAD_TO_DEG);
    pNode->m_Translate *= conversionMagnify;
}

//-----------------------------------------------------------------------------
//! @brief ボーンを解析します。
//!
//! @param[out] pBones ボーン配列へのポインタです。
//! @param[in] boneElem ボーン要素です。
//! @param[in] nodeViss ノードビジビリティ配列です。
//! @param[in] conversionMagnify 移動値に掛ける倍率です。
//-----------------------------------------------------------------------------
void ParseBone(
    CBoneArray* pBones,
    const RXMLElement* boneElem,
    const CNodeVisArray& nodeViss,
    const float conversionMagnify
)
{
    pBones->push_back(CBone());
    CBone& bone = pBones->back();

    //-----------------------------------------------------------------------------
    // 属性を取得します。
    bone.m_Name = RAdjustElementNameString(boneElem->GetAttribute("Name"));
    const std::string parentName = RAdjustElementNameString(boneElem->GetAttribute("ParentBoneName"));
    bone.m_ParentIdx = FindObjectByName(*pBones, parentName);
    bone.m_ScaleCompensate = (boneElem->GetAttribute("IsSegmentScaleCompensate") == "true");
    bone.m_Billboard = GetBillboard(boneElem->GetAttribute("BillboardMode"));
    bone.m_CompressEnable = (boneElem->GetAttribute("IsCompressible") == "true");

    ParseTransform(&bone, boneElem->FindElement("Transform"), conversionMagnify);

    //-----------------------------------------------------------------------------
    // 変換行列を設定します。
    RMtx44 parentMtx;
    if (bone.m_ParentIdx != -1)
    {
        parentMtx = (*pBones)[bone.m_ParentIdx].m_GlobalMtx;
    }
    bone.SetMatrix(parentMtx);

    //-----------------------------------------------------------------------------
    // ノードビジビリティからボーンの可視性を設定します。
    const int nodeVisIdx = FindObjectByName(nodeViss, bone.m_Name);
    if (nodeVisIdx != -1)
    {
        bone.m_Visibility = nodeViss[nodeVisIdx].m_Visibility;
    }

    //-----------------------------------------------------------------------------
    // ボーン用ユーザーデータ配列を取得します。
    ParseUserDatas(&bone.m_UserDatas, boneElem->FindElement("UserData", false));
}

//-----------------------------------------------------------------------------
//! @brief スケルトンのスケール計算モードを取得します。
//-----------------------------------------------------------------------------
RSkeleton::ScaleMode GetSkeletonScaleMode(const std::string& scalingRule)
{
    return
        (scalingRule == "Maya"     ) ? RSkeleton::MAYA      :
        (scalingRule == "Softimage") ? RSkeleton::SOFTIMAGE :
        RSkeleton::STANDARD;
}

//-----------------------------------------------------------------------------
//! @brief スケルトンを解析します。
//!
//! @param[in,out] pCmodel モデルへのポインタです。
//! @param[in] skeletonElem スケルトン要素です。
//! @param[in] modelElem モデル要素です。
//! @param[in] nodeViss ノードビジビリティ配列です。
//! @param[in] conversionMagnify 移動値に掛ける倍率です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseSkeleton(
    CModel* pCmodel,
    const RXMLElement* skeletonElem,
    const RXMLElement* modelElem,
    const CNodeVisArray& nodeViss,
    const float conversionMagnify
)
{
    RSkeleton& skeleton = pCmodel->m_Skeleton;
    CBoneArray& bones = pCmodel->m_Bones;
    if (skeletonElem != nullptr) // スケルタルモデルの場合
    {
        //const std::string rootBoneName = skeletonElem->GetAttribute("RootBoneName");
        skeleton.m_ScaleMode = GetSkeletonScaleMode(skeletonElem->GetAttribute("ScalingRule"));

        const RXMLElement* bonesElem = skeletonElem->FindElement("Bones");
        for (size_t boneIdx = 0; boneIdx < bonesElem->nodes.size(); ++boneIdx)
        {
            ParseBone(&bones, &bonesElem->nodes[boneIdx], nodeViss, conversionMagnify);
        }
    }
    else // モデルの場合
    {
        CBone bone;
        //bone.m_Name = RBone::DefaultRootName;
        bone.m_Name = "mesh";
        if (modelElem != nullptr)
        {
            ParseTransform(&bone, modelElem->FindElement("Transform"), conversionMagnify);
        }
        bones.push_back(bone);
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief スケルトンを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc <skeleton> 要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputSkeleton(
    std::ostream& os,
    const int tc,
    const CModel& cmodel
)
{
    //-----------------------------------------------------------------------------
    // begin skeleton
    const RSkeleton& skeleton = cmodel.m_Skeleton;
    const CBoneArray& bones = cmodel.m_Bones;
    const int boneCount = static_cast<int>(bones.size());
    if (boneCount == 0)
    {
        return;
    }
    os << RTab(tc) << "<skeleton>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // skeleton info
    skeleton.OutInfo(os, tc + 1);

    //-----------------------------------------------------------------------------
    // loop for bone
    os << RTab(tc + 1) << "<bone_array length=\"" << boneCount << "\">" << R_ENDL;
    for (int boneIdx = 0; boneIdx < boneCount; ++boneIdx)
    {
        const CBone& bone = bones[boneIdx];
        const std::string parentName = (bone.m_ParentIdx != -1) ?
            bones[bone.m_ParentIdx].m_Name : "";
        bone.Out(os, tc + 2, boneIdx, parentName);
    }
    os << RTab(tc + 1) << "</bone_array>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // end skeleton
    os << RTab(tc) << "</skeleton>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief メッシュを解析します。
//-----------------------------------------------------------------------------
void ParseMesh(
    CMeshArray& meshes,
    const RXMLElement* meshElem
)
{
    meshes.push_back(CMesh());
    CMesh& mesh = meshes.back();

    mesh.m_Visibility = (meshElem->GetAttribute("IsVisible") == "true");
    mesh.m_NodeName = RAdjustElementNameString(meshElem->GetAttribute("MeshNodeName", "", false)); // 1.3.0 で追加

    const std::string shapePath = RDecodeXmlString(meshElem->FindElement("SeparateShapeReference")->text);
    const size_t openIdx  = shapePath.find_first_of("[");
    const size_t closeIdx = shapePath.find_last_of("]");
    mesh.m_ShapeIdx = atol(shapePath.substr(openIdx + 1, closeIdx - openIdx - 1).c_str());

    const std::string matPath = RDecodeXmlString(meshElem->FindElement("MaterialReference")->text);
    const size_t dq0Idx  = matPath.find_first_of("\"");
    const size_t dq1Idx = matPath.find_last_of("\"");
    mesh.m_MatName = RAdjustElementNameString(matPath.substr(dq0Idx + 1, dq1Idx - dq0Idx - 1));
}

//-----------------------------------------------------------------------------
//! @brief シェイプインデックスでメッシュ配列を検索します。
//-----------------------------------------------------------------------------
const int FindMeshByShape(const CMeshArray& meshes, const int shapeIdx)
{
    for (size_t meshIdx = 0; meshIdx < meshes.size(); ++meshIdx)
    {
        if (meshes[meshIdx].m_ShapeIdx == shapeIdx)
        {
            return static_cast<int>(meshIdx);
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief トライアングルストリップ風に三角形分割する際の頂点インデックスを取得します。
//!
//! @param[in] triIdx 三角形インデックスです。
//! @param[in] vtxIdx 三角形内の頂点インデックスです。
//! @param[in] orgCount 多角形の頂点数です。
//!
//! @brief 多角形内の頂点インデックスを返します。
//-----------------------------------------------------------------------------
inline int GetStripTriangulationIndex(
    const int triIdx,
    const int vtxIdx,
    const int orgCount
)
{
    int orgIdx = triIdx + vtxIdx;
    if (orgIdx >= 2)
    {
        if (((triIdx & 1) == 0) || vtxIdx == 0)
        {
            if (orgIdx & 1)
            {
                orgIdx = (orgIdx >> 1) + 1;
            }
            else
            {
                orgIdx = orgCount - (orgIdx >> 1);
            }
        }
        else
        {
            if (orgIdx & 1)
            {
                orgIdx = orgCount - (orgIdx >> 1);
            }
            else
            {
                orgIdx = (orgIdx >> 1) + 1;
            }
        }
    }
    return orgIdx;
}

//-----------------------------------------------------------------------------
//! @brief 三角形ファンのインデックス配列を三角形群のインデックス配列に変換します。
//!
//! @param[in,out] pVtxIdxs 三角形群のインデックス配列を追加します。
//! @param[in] fanIdxs 三角形ファンのインデックス配列です。
//-----------------------------------------------------------------------------
void ConvertTriangleFanToTriangles(RIntArray* pVtxIdxs, const RIntArray& fanIdxs)
{
    const int vtxCount = static_cast<int>(fanIdxs.size());
    for (int triIdx = 0; triIdx < vtxCount - 2; ++triIdx)
    {
        for (int vtxIdx = 0; vtxIdx < 3; ++vtxIdx)
        {
            // トライアングルストリップ風に分割します。
            const int srcIdx = GetStripTriangulationIndex(triIdx, vtxIdx, vtxCount);
            pVtxIdxs->push_back(fanIdxs[srcIdx]);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 三角形ストリップのインデックス配列を三角形群のインデックス配列に変換します。
//!
//! @param[in,out] pVtxIdxs 三角形群のインデックス配列を追加します。
//! @param[in] stripIdxs 三角形ストリップのインデックス配列です。
//-----------------------------------------------------------------------------
void ConvertTriangleStripToTriangles(RIntArray* pVtxIdxs, const RIntArray& stripIdxs)
{
    int vtxIdxs[3];
    const int vtxCount = static_cast<int>(stripIdxs.size());
    for (int srcIdx = 0; srcIdx < vtxCount; ++srcIdx)
    {
        if (srcIdx == 0)
        {
            vtxIdxs[0] = stripIdxs[srcIdx    ];
            vtxIdxs[1] = stripIdxs[srcIdx + 1];
            vtxIdxs[2] = stripIdxs[srcIdx + 2];
        }
        else
        {
            vtxIdxs[0] = vtxIdxs[1];
            vtxIdxs[1] = vtxIdxs[2];
            vtxIdxs[2] = stripIdxs[srcIdx];
        }

        if (vtxIdxs[0] != vtxIdxs[1] &&
            vtxIdxs[1] != vtxIdxs[2] &&
            vtxIdxs[2] != vtxIdxs[0])
        {
            pVtxIdxs->push_back(vtxIdxs[0]);
            if ((srcIdx & 1) == 0)
            {
                pVtxIdxs->push_back(vtxIdxs[1]);
                pVtxIdxs->push_back(vtxIdxs[2]);
            }
            else
            {
                pVtxIdxs->push_back(vtxIdxs[2]);
                pVtxIdxs->push_back(vtxIdxs[1]);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点インデックス列を解析します。
//!
//! @param[in,out] pCshape シェイプへのポインタです。
//! @param[in] streamElem 頂点インデックス列要素です。
//! @param[in] boneIdxs プリミティブセットのボーンインデックス配列です。
//-----------------------------------------------------------------------------
void ParseIndexStream(
    CShape* pCshape,
    const RXMLElement* streamElem,
    const RIntArray& boneIdxs
)
{
    //-----------------------------------------------------------------------------
    // 頂点インデックス列を解析します。
    const int dataCount = atol(streamElem->GetAttribute("Size").c_str());
    RIntArray values(dataCount);
    std::istringstream iss(streamElem->text);
    int dataIdx = 0;
    while (iss && dataIdx < dataCount)
    {
        std::string value;
        iss >> value;
        if (!value.empty())
        {
            values[dataIdx++] = atol(value.c_str());
        }
    }

    //-----------------------------------------------------------------------------
    // シェイプの頂点インデックス配列に追加します。
    const std::string primMode = streamElem->GetAttribute("PrimitiveMode");
    if (primMode == "Triangles")
    {
        RAppendArrayToArray(pCshape->m_VtxIdxs, values);
    }
    else if (primMode == "TriangleFan")
    {
        ConvertTriangleFanToTriangles(&pCshape->m_VtxIdxs, values);
    }
    else // TriangleStrip
    {
        ConvertTriangleStripToTriangles(&pCshape->m_VtxIdxs, values);
    }

    //-----------------------------------------------------------------------------
    // ローカルのボーンインデックスをグローバルのボーンインデックスに変換します。
    const CVtxAttrInfo& vtxBoneIdxInfo = pCshape->m_VtxBoneIdxInfo;
    if (vtxBoneIdxInfo.m_ExistFlag)
    {
        for (dataIdx = 0; dataIdx < dataCount; ++dataIdx)
        {
            const int vtxIdx = values[dataIdx];
            if (!pCshape->m_VtxBoneIdxConvFlags[vtxIdx])
            {
                pCshape->m_VtxBoneIdxConvFlags[vtxIdx] = true;
                RIVec4& idx4 = pCshape->m_VtxBoneIdxs[vtxIdx];
                for (int compIdx = 0; compIdx < vtxBoneIdxInfo.m_CompCount; ++compIdx)
                {
                    idx4[compIdx] = boneIdxs[idx4[compIdx]];
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブを解析します。
//!
//! @param[in,out] pCshape シェイプへのポインタです。
//! @param[in] primitiveElem プリミティブ要素です。
//! @param[in] boneIdxs プリミティブセットのボーンインデックス配列です。
//-----------------------------------------------------------------------------
void ParsePrimitive(
    CShape* pCshape,
    const RXMLElement* primitiveElem,
    const RIntArray& boneIdxs
)
{
    //-----------------------------------------------------------------------------
    // 頂点インデックス列を解析します。
    const RXMLElement* streamsElem = primitiveElem->FindElement("IndexStreams");
    for (size_t streamIdx = 0; streamIdx < streamsElem->nodes.size(); ++streamIdx)
    {
        ParseIndexStream(pCshape, &streamsElem->nodes[streamIdx], boneIdxs);
    }
}

//-----------------------------------------------------------------------------
//! @brief プリミティブセットを解析します。
//!
//! @param[in,out] pCshape シェイプへのポインタです。
//! @param[in] primSetElem プリミティブセット要素です。
//-----------------------------------------------------------------------------
void ParsePrimSet(
    CShape* pCshape,
    const RXMLElement* primSetElem
)
{
    //-----------------------------------------------------------------------------
    // parse bone index table
    RIntArray boneIdxs;
    const RXMLElement* boneIdxTableElem = primSetElem->FindElement("BoneIndexTable", false);
    if (boneIdxTableElem != nullptr)
    {
        RStringArray words;
        RTokenizeString(words, boneIdxTableElem->text);
        for (size_t idxIdx = 0; idxIdx < words.size(); ++idxIdx)
        {
            boneIdxs.push_back(atol(words[idxIdx].c_str()));
        }
    }

    //-----------------------------------------------------------------------------
    // プリミティブを解析します。
    const RXMLElement* primitivesElem = primSetElem->FindElement("Primitives");
    for (size_t primIdx = 0; primIdx < primitivesElem->nodes.size(); ++primIdx)
    {
        ParsePrimitive(pCshape, &primitivesElem->nodes[primIdx], boneIdxs);
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性配列の値を設定します。
//!
//! @param[in,out] pArray float 型 2 次元ベクトル配列へのポインタです。
//! @param[in] values float 値配列です。
//-----------------------------------------------------------------------------
void SetVtxAttribValues(RVec2Array* pArray, const RFloatArray& values)
{
    int srcIdx = 0;
    const int vtxCount = static_cast<int>(values.size()) / 2;
    for (int vtxIdx = 0; vtxIdx < vtxCount; ++vtxIdx)
    {
        pArray->push_back(RVec2(values[srcIdx + 0], values[srcIdx + 1]));
        srcIdx += 2;
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性配列の値を設定します。
//!
//! @param[in,out] pArray float 型 3 次元ベクトル配列へのポインタです。
//! @param[in] values float 値配列です。
//-----------------------------------------------------------------------------
void SetVtxAttribValues(RVec3Array* pArray, const RFloatArray& values)
{
    int srcIdx = 0;
    const int vtxCount = static_cast<int>(values.size()) / R_XYZ_COUNT;
    for (int vtxIdx = 0; vtxIdx < vtxCount; ++vtxIdx)
    {
        pArray->push_back(RVec3(values[srcIdx + 0], values[srcIdx + 1], values[srcIdx + 2]));
        srcIdx += R_XYZ_COUNT;
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性配列の値を設定します。
//!
//! @param[in,out] pArray float 型 4 次元ベクトル配列へのポインタです。
//! @param[in] values float 値配列です。
//! @param[in] compCount 1 頂点当たりの成分数です。
//! @param[in] fillY 成分数が 1 の場合に Y 成分に設定する値です。
//! @param[in] fillZ 成分数が 2 以下の場合に Z 成分に設定する値です。
//! @param[in] fillW 成分数が 3 以下の場合に W 成分に設定する値です。
//-----------------------------------------------------------------------------
void SetVtxAttribValues(
    RVec4Array* pArray,
    const RFloatArray& values,
    const int compCount,
    const float fillY = 0.0f,
    const float fillZ = 0.0f,
    const float fillW = 1.0f
)
{
    int srcIdx = 0;
    const int vtxCount = static_cast<int>(values.size()) / compCount;
    if (compCount == 1)
    {
        for (int vtxIdx = 0; vtxIdx < vtxCount; ++vtxIdx)
        {
            pArray->push_back(RVec4(values[srcIdx], fillY, fillZ, fillW));
            srcIdx += 1;
        }
    }
    else if (compCount == 2)
    {
        for (int vtxIdx = 0; vtxIdx < vtxCount; ++vtxIdx)
        {
            pArray->push_back(RVec4(values[srcIdx + 0], values[srcIdx + 1], fillZ, fillW));
            srcIdx += 2;
        }
    }
    else if (compCount == 3)
    {
        for (int vtxIdx = 0; vtxIdx < vtxCount; ++vtxIdx)
        {
            pArray->push_back(RVec4(values[srcIdx + 0], values[srcIdx + 1], values[srcIdx + 2], fillW));
            srcIdx += R_XYZ_COUNT;
        }
    }
    else if (compCount == 4)
    {
        for (int vtxIdx = 0; vtxIdx < vtxCount; ++vtxIdx)
        {
            pArray->push_back(RVec4(values[srcIdx + 0], values[srcIdx + 1], values[srcIdx + 2], values[srcIdx + 3]));
            srcIdx += R_XYZW_COUNT;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性配列の値を設定します（頂点ボーンインデックス配列用）。
//!
//! @param[in,out] pArray int 型 4 次元ベクトル配列へのポインタです。
//! @param[in] values float 値配列です。
//! @param[in] compCount 1 頂点当たりの成分数です。
//! @param[in] fillY 成分数が 1 の場合に Y 成分に設定する値です。
//! @param[in] fillZ 成分数が 2 以下の場合に Z 成分に設定する値です。
//! @param[in] fillW 成分数が 3 以下の場合に W 成分に設定する値です。
//-----------------------------------------------------------------------------
void SetVtxAttribValues(
    RIVec4Array* pArray,
    const RFloatArray& values,
    const int compCount,
    const float fillY = 0.0f,
    const float fillZ = 0.0f,
    const float fillW = 1.0f
)
{
    RVec4Array floatArray;
    SetVtxAttribValues(&floatArray, values, compCount, fillY, fillZ, fillW);

    pArray->resize(floatArray.size());
    for (size_t vtxIdx = 0; vtxIdx < floatArray.size(); ++vtxIdx)
    {
        const RVec4& fv = floatArray[vtxIdx];
        (*pArray)[vtxIdx] = RIVec4(RRound(fv.x), RRound(fv.y), RRound(fv.z), RRound(fv.w));
    }
}

//-----------------------------------------------------------------------------
//! @brief 3 次元ベクトル配列にオフセットを追加します。
//!
//! @param[in,out] pVecs 3 次元ベクトル配列へのポインタです。
//! @param[in] ofs オフセットです。
//-----------------------------------------------------------------------------
void AddOffsetToVectors(RVec3Array* pVecs, const RVec3& ofs)
{
    if (ofs != RVec3::kZero)
    {
        for (size_t vecIdx = 0; vecIdx < pVecs->size(); ++vecIdx)
        {
            (*pVecs)[vecIdx] += ofs;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 3 次元ベクトル配列をスケールします。
//!
//! @param[in,out] pVecs 3 次元ベクトル配列へのポインタです。
//! @param[in] scale スケールです。
//-----------------------------------------------------------------------------
void ScaleVectors(RVec3Array* pVecs, const float scale)
{
    if (scale != 1.0f)
    {
        for (size_t vecIdx = 0; vecIdx < pVecs->size(); ++vecIdx)
        {
            (*pVecs)[vecIdx] *= scale;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 3 次元ベクトル配列の長さを正規化します。
//!
//! @param[in,out] pVecs 3 次元ベクトル配列へのポインタです。
//! @param[in] pZeroReplace 長さが 0 の場合に置き換えるベクトルへのポインタです。nullptr なら置き換えません。
//-----------------------------------------------------------------------------
void NormalizeVectors(RVec3Array* pVecs, const RVec3* pZeroReplace = nullptr)
{
    for (size_t vecIdx = 0; vecIdx < pVecs->size(); ++vecIdx)
    {
        RVec3& vec = (*pVecs)[vecIdx];
        vec.Normalize();
        if (vec == RVec3::kZero && pZeroReplace != nullptr)
        {
            vec = *pZeroReplace;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 4 次元ベクトル配列の 3 次元部分の長さを正規化します。
//!
//! @param[in,out] pVecs 4 次元ベクトル配列へのポインタです。
//! @param[in] pZeroReplace 長さが 0 の場合に置き換えるベクトルへのポインタです。nullptr なら置き換えません。
//-----------------------------------------------------------------------------
void NormalizeVectors(RVec4Array* pVecs, const RVec3* pZeroReplace = nullptr)
{
    for (size_t vecIdx = 0; vecIdx < pVecs->size(); ++vecIdx)
    {
        RVec4& vec4 = (*pVecs)[vecIdx];
        RVec3 vec3(vec4);
        vec3.Normalize();
        if (vec3 == RVec3::kZero && pZeroReplace != nullptr)
        {
            vec3 = *pZeroReplace;
        }
        vec4 = RVec4(vec3, vec4.w);
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点カラーの値をスナップします。
//!
//! @param[in,out] pCols 頂点カラー配列へのポインタです。
//! @param[in] compCount 1 頂点当たりの成分数です。
//-----------------------------------------------------------------------------
void SnapVertexColors(RVec4Array* pCols, const int compCount)
{
    for (size_t colIdx = 0; colIdx < pCols->size(); ++colIdx)
    {
        RVec4& col = (*pCols)[colIdx];
        for (int compIdx = 0; compIdx < compCount; ++compIdx)
        {
            float& v = col[compIdx];
            if (RIsSame(v, 1.0f))
            {
                // Scale="0.00392157" の場合
                // 255 x 0.00392157 = 1.00000036
                // となり、量子化分析で float_16_16_16_16 になるので、
                // 1.0 にスナップし、unorm_8_8_8_8 になるようにします。
                v = 1.0f;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点カラーを sRGB からリニアに変換します。
//!
//! @param[in,out] pCols 頂点カラー配列へのポインタです。
//! @param[in] compCount 1 頂点当たりの成分数です。
//! @param[in] vtxColLinearFlag 頂点カラーのリニア変換フラグです。
//-----------------------------------------------------------------------------
void ConvertVtxColsToLinear(
    RVec4Array* pCols,
    const int compCount,
    const int vtxColLinearFlag
)
{
    if (!pCols->empty() && vtxColLinearFlag != C2NnOption::LinearFlag_None)
    {
        for (size_t colIdx = 0; colIdx < pCols->size(); ++colIdx)
        {
            RVec4& col = (*pCols)[colIdx];
            for (int compIdx = 0; compIdx < compCount; ++compIdx)
            {
                if ((vtxColLinearFlag & LinearRgbaFlags[compIdx]) != 0)
                {
                    col[compIdx] = powf(RMax(col[compIdx], 0.0f), SrgbGammaValue);
                }
            }
            // 変換後の値が 1/255 の倍数でなくても [0,1] の範囲なら、
            // 量子化分析は unorm_8_8_8_8 になります。
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ座標の V 成分を反転します。
//!
//! @param[in,out] pUvs テクスチャ座標配列へのポインタです。
//-----------------------------------------------------------------------------
void FlipTextureCoordinateV(RVec2Array* pUvs)
{
    for (size_t uvIdx = 0; uvIdx < pUvs->size(); ++uvIdx)
    {
        RVec2& uv = (*pUvs)[uvIdx];
        uv = RVec2(uv.x, 1.0f - uv.y);
    }
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性値のタイプを取得します。
//-----------------------------------------------------------------------------
RPrimVtx::ValueType GetVtxAttribValueType(const std::string& quantizedMode)
{
    R_UNUSED_VARIABLE(quantizedMode);
    return RPrimVtx::VALUE_FLOAT;
    //return
    //    (quantizedMode == "Short" ) ? RPrimVtx::VALUE_INT  :
    //    (quantizedMode == "Ushort") ? RPrimVtx::VALUE_UINT :
    //    (quantizedMode == "Byte"  ) ? RPrimVtx::VALUE_INT  :
    //    (quantizedMode == "Ubyte" ) ? RPrimVtx::VALUE_UINT :
    //    RPrimVtx::VALUE_FLOAT; // "Float", "Auto"
}

//-----------------------------------------------------------------------------
//! @brief 頂点属性を解析します。
//!
//! @param[in,out] pCshape シェイプへのポインタです。
//! @param[in] vtxAttribElem 頂点属性要素です。
//! @param[in] maxVtxCount 全頂点属性中の最大頂点数です。
//! @param[in] vtxColLinearFlag 頂点カラーのリニア変換フラグです。
//-----------------------------------------------------------------------------
void ParseVtxAttrib(
    CShape* pCshape,
    const RXMLElement* vtxAttribElem,
    const int maxVtxCount,
    const int vtxColLinearFlag
)
{
    //-----------------------------------------------------------------------------
    // 要素名から成分数を取得します。
    int compCount = 1;
    if (vtxAttribElem->name.find("Vector2") == 0)
    {
        compCount = 2;
    }
    else if (vtxAttribElem->name.find("Vector3") == 0)
    {
        compCount = 3;
    }
    else if (vtxAttribElem->name.find("Vector4") == 0)
    {
        compCount = 4;
    }
    const bool isConst = vtxAttribElem->name.find("VertexAttributeCtr") != std::string::npos;

    //-----------------------------------------------------------------------------
    // 属性を取得します。
    const std::string quantizedMode = (isConst) ?
        "Float" : vtxAttribElem->GetAttribute("QuantizedMode");
    const std::string usage = vtxAttribElem->GetAttribute("Usage");
    const double scale = (isConst) ? 1.0 : atof(vtxAttribElem->GetAttribute("Scale").c_str());
    const int vtxCount = (isConst) ? 1 : atol(vtxAttribElem->GetAttribute("VertexSize").c_str());
    const int dataCount = compCount * vtxCount;

    //-----------------------------------------------------------------------------
    // 値の配列を取得します。
    RFloatArray floatValues(dataCount);
    std::istringstream iss(vtxAttribElem->text);
    int dataIdx = 0;
    while (iss && dataIdx < dataCount)
    {
        std::string value;
        iss >> value;
        if (!value.empty())
        {
            floatValues[dataIdx++] = static_cast<float>(atof(value.c_str()) * scale);
        }
    }

    //-----------------------------------------------------------------------------
    // 値が一定なら最大頂点数までコピーします。
    if (vtxCount < maxVtxCount)
    {
        const int newCount = compCount * maxVtxCount;
        RFloatArray newValues(newCount);
        int srcIdx = 0;
        for (int dstIdx = 0; dstIdx < newCount; ++dstIdx)
        {
            newValues[dstIdx] = floatValues[srcIdx++];
            if (srcIdx >= compCount)
            {
                srcIdx = 0;
            }
        }
        floatValues = newValues;
    }

    //-----------------------------------------------------------------------------
    // 各頂点属性の値を設定します。
    CVtxAttrInfo* pVtxAttrInfo = nullptr;
    if (usage == "Position")
    {
        SetVtxAttribValues(&pCshape->m_VtxPoss, floatValues);
    }
    else if (usage == "Normal")
    {
        SetVtxAttribValues(&pCshape->m_VtxNrms, floatValues);
        NormalizeVectors(&pCshape->m_VtxNrms, &RVec3::kYAxis);
    }
    else if (usage == "Tangent")
    {
        // cmdl では接線は常に 3 成分
        // cmdl では UV のワインド順序は常に時計回り（前向き）なので接線の W 値はすべて 1
        SetVtxAttribValues(&pCshape->m_VtxTans, floatValues, compCount);
        NormalizeVectors(&pCshape->m_VtxTans, &RVec3::kXAxis);
    }
    else if (usage == "Color")
    {
        SetVtxAttribValues(&pCshape->m_VtxCols, floatValues, compCount);
        ConvertVtxColsToLinear(&pCshape->m_VtxCols, compCount, vtxColLinearFlag);
        SnapVertexColors(&pCshape->m_VtxCols, compCount);
        pVtxAttrInfo = &pCshape->m_VtxColInfo;
    }
    else if (usage.find("TextureCoordinate") == 0)
    {
        const int texIdx = usage[usage.size() - 1] - '0';
        if (0 <= texIdx && texIdx < CShape::VTX_TEX_MAX)
        {
            SetVtxAttribValues(&pCshape->m_VtxTexss[texIdx], floatValues);
            FlipTextureCoordinateV(&pCshape->m_VtxTexss[texIdx]);
            pCshape->m_VtxTexCount = RMax(pCshape->m_VtxTexCount, 1 + texIdx);
        }
    }
    else if (usage.find("UserAttribute") == 0)
    {
        const int usrIdx = usage[usage.size() - 1] - '0';
        if (0 <= usrIdx && usrIdx < CShape::VTX_USR_MAX)
        {
            SetVtxAttribValues(&pCshape->m_VtxUsrss[usrIdx], floatValues, compCount);
            SnapVertexColors(&pCshape->m_VtxUsrss[usrIdx], compCount);
            pVtxAttrInfo = &pCshape->m_VtxUsrInfos[usrIdx];
        }
    }
    else if (usage == "BoneIndex")
    {
        SetVtxAttribValues(&pCshape->m_VtxBoneIdxs, floatValues, compCount, 0.0f, 0.0f, 0.0f);
        pVtxAttrInfo = &pCshape->m_VtxBoneIdxInfo;
        pCshape->m_VtxBoneIdxConvFlags = RBoolArray(pCshape->m_VtxBoneIdxs.size(), false);
    }
    else if (usage == "BoneWeight")
    {
        SetVtxAttribValues(&pCshape->m_VtxBoneWgts, floatValues, compCount, 0.0f, 0.0f, 0.0f);
        pVtxAttrInfo = &pCshape->m_VtxBoneWgtInfo;
    }

    //-----------------------------------------------------------------------------
    // 頂点属性情報を設定します。
    if (pVtxAttrInfo != nullptr)
    {
        pVtxAttrInfo->m_ExistFlag = true;
        pVtxAttrInfo->m_CompCount = compCount;
        pVtxAttrInfo->m_ValueType = GetVtxAttribValueType(quantizedMode);
    }
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief 法線と接線から従法線を計算します。
//!
//! @param[in,out] pCshape シェイプへのポインタです。
//-----------------------------------------------------------------------------
void CalcBinormals(CShape* pCshape)
{
    pCshape->m_VtxBins.clear();
    if (pCshape->m_VtxNrms.size() == pCshape->m_VtxTans.size())
    {
        for (size_t vtxIdx = 0; vtxIdx < pCshape->m_VtxNrms.size(); ++vtxIdx)
        {
            const RVec3& nrm  = pCshape->m_VtxNrms[vtxIdx];
            const RVec4& tan4 = pCshape->m_VtxTans[vtxIdx];
            const RVec3 tan(tan4);
            RVec3 bin = (nrm ^ tan).Normal(); // cmdl では TBN は常に右手座標系
            bin.SnapToZero();
            pCshape->m_VtxBins.push_back(RVec4(bin, 1.0f));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief メッシュのノード名が不正な場合のシェイプのデフォルトボーンインデックスを取得します。
//!        先頭のプリミティブセットのボーンインデックステーブルのサイズが 1 なら
//!        ボーンインデックステーブルの値を返します。それ以外は 0 を返します。
//!
//! @param[in] shapeElem シェイプ要素です。
//!
//! @return デフォルトボーンインデックスを返します。
//-----------------------------------------------------------------------------
int GetShapeDefaultBoneIndex(const RXMLElement* shapeElem)
{
    const RXMLElement* primSetsElem = shapeElem->FindElement("PrimitiveSets");
    if (!primSetsElem->nodes.empty())
    {
        const RXMLElement* primSetElem = &primSetsElem->nodes[0];
        const RXMLElement* boneIdxTableElem = primSetElem->FindElement("BoneIndexTable", false);
        if (boneIdxTableElem != nullptr)
        {
            RStringArray words;
            RTokenizeString(words, boneIdxTableElem->text);
            if (words.size() == 1) // リジッドボディ
            {
                return atol(words[0].c_str());
            }
        }
    }
    return 0;
}

//-----------------------------------------------------------------------------
//! @brief シェイプを解析します。
//!
//! @param[in,out] pCmodel モデルへのポインタです。
//! @param[in] shapeElem シェイプ要素です。
//! @param[in] mesh メッシュです。
//! @param[in] nodeViss ノードビジビリティ配列です。
//! @param[in] conversionMagnify 頂点座標に掛ける倍率です。
//! @param[in] vtxColLinearFlag 頂点カラーのリニア変換フラグです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseShape(
    CModel* pCmodel,
    const RXMLElement* shapeElem,
    const CMesh& mesh,
    const CNodeVisArray& nodeViss,
    const float conversionMagnify,
    const int vtxColLinearFlag
)
{
    CShapeArray& cshapes = pCmodel->m_Shapes;
    CBoneArray& bones = pCmodel->m_Bones;

    //-----------------------------------------------------------------------------
    // シェイプを追加します。
    cshapes.push_back(CShape());
    CShape& cshape = cshapes.back();

    //-----------------------------------------------------------------------------
    // シェイプが属するノード名、シェイプに適用されたマテリアル名、シェイプ名を設定します。
    cshape.m_NodeName = mesh.m_NodeName;
    if (cshape.m_NodeName.empty() || FindObjectByName(bones, cshape.m_NodeName) == -1)
    {
        cshape.m_NodeName = bones[GetShapeDefaultBoneIndex(shapeElem)].m_Name;
    }
    cshape.m_MatName = mesh.m_MatName;
    cshape.m_Name = GetUniqueObjectName(cshapes, cshape.m_NodeName + "__" + cshape.m_MatName);
    //cerr << "shape: " << cshape.m_Name << endl;

    //-----------------------------------------------------------------------------
    // ノードビジビリティ要素がなければメッシュの可視性をボーンの可視性に設定します。
    const int boneIdx = FindObjectByName(bones, cshape.m_NodeName);
    if (boneIdx != -1 && nodeViss.empty())
    {
        bones[boneIdx].m_Visibility = mesh.m_Visibility;
    }

    //-----------------------------------------------------------------------------
    // 全頂点属性中の最大頂点数を取得します。
    const RXMLElement* vtxAttribsElem = shapeElem->FindElement("VertexAttributes");
    int maxVtxCount = 0;
    for (size_t attrIdx = 0; attrIdx < vtxAttribsElem->nodes.size(); ++attrIdx)
    {
        const RXMLElement* vtxAttribElem = &vtxAttribsElem->nodes[attrIdx];
        const bool isConst = vtxAttribElem->name.find("VertexAttributeCtr") != std::string::npos;
        const int vtxCount = (isConst) ? 1 : atol(vtxAttribElem->GetAttribute("VertexSize").c_str());
        maxVtxCount = RMax(maxVtxCount, vtxCount);
    }

    //-----------------------------------------------------------------------------
    // 頂点属性を解析します。
    for (size_t attrIdx = 0; attrIdx < vtxAttribsElem->nodes.size(); ++attrIdx)
    {
        ParseVtxAttrib(&cshape, &vtxAttribsElem->nodes[attrIdx], maxVtxCount, vtxColLinearFlag);
    }
    //cshape.m_ExportsUsrAsCol = (cshape.HasVtxUsr() && cshape.m_VtxCols.empty());

    //-----------------------------------------------------------------------------
    // 座標オフセットを加算します。
    RVec3 posOfs;
    GetXyzAttr(posOfs, shapeElem->FindElement("PositionOffset"));
    AddOffsetToVectors(&cshape.m_VtxPoss, posOfs);

    //-----------------------------------------------------------------------------
    // 頂点座標をスケールします。
    ScaleVectors(&cshape.m_VtxPoss, conversionMagnify);

    //-----------------------------------------------------------------------------
    // プリミティブセットを解析します。
    const RXMLElement* primSetsElem = shapeElem->FindElement("PrimitiveSets");
    for (size_t setIdx = 0; setIdx < primSetsElem->nodes.size(); ++setIdx)
    {
        ParsePrimSet(&cshape, &primSetsElem->nodes[setIdx]);
    }

    //-----------------------------------------------------------------------------
    // 頂点行列インデックスを設定します。
    const CVtxAttrInfo& vtxBoneIdxInfo = cshape.m_VtxBoneIdxInfo;
    if (!vtxBoneIdxInfo.m_ExistFlag)
    {
        //-----------------------------------------------------------------------------
        // リジッドボディの場合
        if (boneIdx != -1)
        {
            bones[boneIdx].m_RigidBody = true;
        }
        RVtxMtx vtxMtx(boneIdx);
        cshape.m_VtxMtxs.push_back(vtxMtx);
        for (int vtxIdx = 0; vtxIdx < maxVtxCount; ++vtxIdx)
        {
            cshape.m_VtxMtxIdxs.push_back(0);
        }
    }
    else if (vtxBoneIdxInfo.m_CompCount == 1)
    {
        //-----------------------------------------------------------------------------
        // リジッドスキニングの場合
        cshape.m_SkinningMode = RShape::RIGID;
        ++pCmodel->m_RigidSkinningShapeCount;
        for (int vtxIdx = 0; vtxIdx < maxVtxCount; ++vtxIdx)
        {
            const int infIdx = cshape.m_VtxBoneIdxs[vtxIdx][0];
            bones[infIdx].m_RigidMtxIdx = 0;
            RVtxMtx vtxMtx(infIdx);
            cshape.m_VtxMtxs.push_back(vtxMtx);
            cshape.m_VtxMtxIdxs.push_back(vtxIdx);
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // スムーススキニングの場合
        cshape.m_SkinningMode = RShape::SMOOTH;
        ++pCmodel->m_SmoothSkinningShapeCount;
        for (int vtxIdx = 0; vtxIdx < maxVtxCount; ++vtxIdx)
        {
            const RIVec4& idx4 = cshape.m_VtxBoneIdxs[vtxIdx];
            const RVec4&  wgt4 = cshape.m_VtxBoneWgts[vtxIdx];
            RVtxMtx vtxMtx;
            for (int compIdx = 0; compIdx < vtxBoneIdxInfo.m_CompCount; ++compIdx)
            {
                if (wgt4[compIdx] != 0.0f)
                {
                    const int infIdx = idx4[compIdx];
                    bones[infIdx].m_SmoothMtxIdx = 0;
                    vtxMtx.Append(infIdx, RRound(wgt4[compIdx] * 100.0f));
                }
            }
            cshape.m_VtxMtxs.push_back(vtxMtx);
            cshape.m_VtxMtxIdxs.push_back(vtxIdx);
        }
    }

    //-----------------------------------------------------------------------------
    // 法線と接線から従法線を計算します。
    if (!cshape.m_VtxNrms.empty() && !cshape.m_VtxTans.empty())
    {
        CalcBinormals(&cshape);
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief スキニングに使用するボーンの行列パレットインデックスを設定します。
//!        行列パレットはスムーススキン用配列の後にリジッドスキン用配列が並ぶようにします。
//!
//! @param[in,out] pCmodel モデルへのポインタです。
//-----------------------------------------------------------------------------
void SetBoneSkinningMtxIdx(CModel* pCmodel)
{
    CBoneArray& bones = pCmodel->m_Bones;
    int mtxIdx = 0;

    for (size_t boneIdx = 0; boneIdx < bones.size(); ++boneIdx)
    {
        CBone& bone = bones[boneIdx];
        if (bone.m_SmoothMtxIdx != -1)
        {
            bone.m_SmoothMtxIdx = mtxIdx++;
            ++pCmodel->m_SmoothSkinningMtxCount;
        }
    }

    for (size_t boneIdx = 0; boneIdx < bones.size(); ++boneIdx)
    {
        CBone& bone = bones[boneIdx];
        if (bone.m_RigidMtxIdx != -1)
        {
            bone.m_RigidMtxIdx = mtxIdx++;
            ++pCmodel->m_RigidSkinningMtxCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプにプリミティブを設定します。
//!
//! @param[in,out] pRshape dcc シェイプへのポインタです。
//! @param[in] cshape シェイプです。
//-----------------------------------------------------------------------------
void SetRPrim(RShape* pRshape, const CShape& cshape)
{
    //-----------------------------------------------------------------------------
    // set vertex attr flag
    pRshape->m_VtxAttrFlag[RPrimVtx::POS0] = true;
    pRshape->m_VtxAttrFlag[RPrimVtx::NRM0] = !cshape.m_VtxNrms.empty();
    pRshape->m_VtxAttrFlag[RPrimVtx::TAN0] = !cshape.m_VtxTans.empty();
    pRshape->m_VtxAttrFlag[RPrimVtx::BIN0] = !cshape.m_VtxBins.empty();
    pRshape->m_VtxAttrFlag[RPrimVtx::COL0] = !cshape.m_VtxCols.empty();
    for (int usrIdx = 0; usrIdx < CShape::VTX_USR_MAX; ++usrIdx)
    {
        // ユーザー頂点属性は color1 以降に出力します。
        pRshape->m_VtxAttrFlag[RPrimVtx::COL1 + usrIdx] = !cshape.m_VtxUsrss[usrIdx].empty();
    }
    for (int texIdx = 0; texIdx < cshape.m_VtxTexCount; ++texIdx)
    {
        pRshape->m_VtxAttrFlag[RPrimVtx::TEX0 + texIdx] = true;
    }
    pRshape->m_VtxAttrFlag[RPrimVtx::IDX0] = (cshape.m_SkinningMode != RShape::NO_SKINNING);
    pRshape->m_VtxAttrFlag[RPrimVtx::WGT0] = (cshape.m_SkinningMode == RShape::SMOOTH     );

    //-----------------------------------------------------------------------------
    // loop for face
    int vtxTotalIdx = 0;
    const int faceCount = static_cast<int>(cshape.m_VtxIdxs.size()) / 3;
    for (int faceIdx = 0; faceIdx < faceCount; ++faceIdx)
    {
        const int vtxCount = 3;

        //-----------------------------------------------------------------------------
        // create rprim
        RPrimitive rprim(vtxCount);

        //-----------------------------------------------------------------------------
        // copy attr
        for (int localVtxIdx = 0; localVtxIdx < vtxCount; ++localVtxIdx, ++vtxTotalIdx)
        {
            const int vtxIdx = cshape.m_VtxIdxs[vtxTotalIdx];
            RPrimVtx vtx;
            vtx.m_Mtx = cshape.m_VtxMtxIdxs[vtxIdx];
            vtx[RPrimVtx::POS0] = vtxIdx;
            vtx[RPrimVtx::NRM0] = (pRshape->m_VtxAttrFlag[RPrimVtx::NRM0]) ? vtxIdx : -1;
            vtx[RPrimVtx::TAN0] = (pRshape->m_VtxAttrFlag[RPrimVtx::TAN0]) ? vtxIdx : -1;
            vtx[RPrimVtx::BIN0] = (pRshape->m_VtxAttrFlag[RPrimVtx::BIN0]) ? vtxIdx : -1;
            for (int colIdx = 0; colIdx < 1 + CShape::VTX_USR_MAX; ++colIdx)
            {
                vtx[RPrimVtx::COL0 + colIdx] = (pRshape->m_VtxAttrFlag[RPrimVtx::COL0 + colIdx]) ? vtxIdx : -1;
            }
            for (int texIdx = 0; texIdx < cshape.m_VtxTexCount; ++texIdx)
            {
                vtx[RPrimVtx::TEX0 + texIdx] = vtxIdx;
            }
            rprim.SetPrimVtx(localVtxIdx, vtx);
        }

        //-----------------------------------------------------------------------------
        // append
        const bool forceTriangulate = true;
        const int triPattern = 0;
        pRshape->AppendPolygon(rprim, cshape.m_VtxMtxs, forceTriangulate, triPattern);
    }
}

//-----------------------------------------------------------------------------
//! @brief シェイプを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pVertices 頂点データ配列へのポインタです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in,out] pCmodel モデルへのポインタです。
//! @param[in] tc <shape> 要素のインデントに必要なタブの数です。
//! @param[in] shapeIdx シェイプのインデックスです。
//! @param[in] cshape シェイプです。
//! @param[in] boneMtxIdxs ボーンに対する行列パレットインデックスの配列です。
//-----------------------------------------------------------------------------
void OutputShape( // OutputShapeT
    std::ostream& os,
    RVertexArray* pVertices,
    RDataStreamArray* pDataStreams,
    CModel* pCmodel,
    const int tc,
    const int shapeIdx,
    const CShape& cshape,
    const RIntArray& boneMtxIdxs
)
{
    //-----------------------------------------------------------------------------
    // dcc シェイプオブジェクトを作成します。
    const int mtxPalCount = RShape::GetStandardMtxPalCount(false, 0);
    RShape rshape(cshape.m_SkinningMode, mtxPalCount, cshape.m_Name.c_str());

    SetRPrim(&rshape, cshape);

    rshape.SetAndOptimize(cshape.m_VtxMtxs, boneMtxIdxs);

    //-----------------------------------------------------------------------------
    // シェイプ出力情報を設定します。
    ROutShapeInfo outInfo;
    outInfo.m_Index = shapeIdx;
    outInfo.m_Name = cshape.m_Name;
    outInfo.m_MatName = cshape.m_MatName;
    outInfo.m_BoneName = cshape.m_NodeName;
    //outInfo.m_OrientedBB = cshape.m_OrientedBB;
    //outInfo.m_pUvAttribNames = &cshape.m_UvAttribNames;
    //outInfo.m_pUvHintIdxs    = &cshape.m_UvHintIdxs;

    outInfo.m_VtxInfos[RPrimVtx::POS0] = ROutShapeVtxInfo( // POS の array は RVec3Array のみ可
        3, RPrimVtx::VALUE_FLOAT, &cshape.m_VtxPoss);
    outInfo.m_VtxInfos[RPrimVtx::NRM0] = ROutShapeVtxInfo( // NRM の array は RVec3Array のみ可
        3, RPrimVtx::VALUE_FLOAT, &cshape.m_VtxNrms);
    if (rshape.m_VtxAttrFlag[RPrimVtx::TAN0])
    {
        outInfo.m_VtxInfos[RPrimVtx::TAN0] = ROutShapeVtxInfo( // TAN の array は RVec4Array のみ可
            4, RPrimVtx::VALUE_FLOAT, &cshape.m_VtxTans);
    }
    if (rshape.m_VtxAttrFlag[RPrimVtx::BIN0])
    {
        outInfo.m_VtxInfos[RPrimVtx::BIN0] = ROutShapeVtxInfo( // BIN の array は RVec4Array のみ可
            4, RPrimVtx::VALUE_FLOAT, &cshape.m_VtxBins);
    }
    for (int colIdx = 0; colIdx < 1 + CShape::VTX_USR_MAX; ++colIdx) // COL の array は RVec4Array のみ可
    {
        const CVtxAttrInfo& info = (colIdx == 0) ?
            cshape.m_VtxColInfo : cshape.m_VtxUsrInfos[colIdx - 1];
        if (info.m_ExistFlag)
        {
            const void* pArray = (colIdx == 0) ?
                &cshape.m_VtxCols : &cshape.m_VtxUsrss[colIdx - 1];
            outInfo.m_VtxInfos[RPrimVtx::COL0 + colIdx] = ROutShapeVtxInfo(
                info.m_CompCount, info.m_ValueType, pArray);
        }
    }
    for (int texIdx = 0; texIdx < cshape.m_VtxTexCount; ++texIdx) // TEX の array は RVec2Array のみ可
    {
        outInfo.m_VtxInfos[RPrimVtx::TEX0 + texIdx] = ROutShapeVtxInfo(
            2, RPrimVtx::VALUE_FLOAT, &cshape.m_VtxTexss[texIdx]);
    }
    outInfo.m_VtxInfos[RPrimVtx::IDX0] = ROutShapeVtxInfo( // IDX は quantize のみ影響
        4, RPrimVtx::VALUE_UINT, nullptr);
    outInfo.m_VtxInfos[RPrimVtx::WGT0] = ROutShapeVtxInfo( // WGT は quantize のみ影響
        4, RPrimVtx::VALUE_FLOAT, nullptr);

    //-----------------------------------------------------------------------------
    // RShape オブジェクトを出力します。
    ROutShapeResult result;
    RStatus rstatus = rshape.Out(os, *pVertices, *pDataStreams, result, tc, outInfo);
    if (!rstatus)
    {
        //std::string msg = rstatus.GetMessage();
    }

    pCmodel->m_TotalTriangleCount      += result.m_TriangleCount;
    pCmodel->m_TotalIndexCount         += result.m_IndexCount;
    pCmodel->m_TotalVertexCount        += result.m_VertexCount;
    pCmodel->m_TotalProcessVertexCount += result.m_ProcessVertexCount;
}

//-----------------------------------------------------------------------------
//! @brief シェイプ群と頂点データ群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in,out] pCmodel モデルへのポインタです。
//! @param[in] tc <shape_array> 要素のインデントに必要なタブの数です。
//-----------------------------------------------------------------------------
void OutputShapes(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    CModel* pCmodel,
    const int tc
)
{
    //-----------------------------------------------------------------------------
    // ボーンに対する行列パレットインデックスの配列を作成します。
    const CBoneArray bones = pCmodel->m_Bones;
    const CShapeArray& cshapes = pCmodel->m_Shapes;
    const int boneCount = static_cast<int>(bones.size());
    RIntArray smoothMtxIdxs(boneCount);
    RIntArray rigidMtxIdxs(boneCount);
    for (int boneIdx = 0; boneIdx < boneCount; ++boneIdx)
    {
        smoothMtxIdxs[boneIdx] = bones[boneIdx].m_SmoothMtxIdx;
        rigidMtxIdxs[boneIdx]  = bones[boneIdx].m_RigidMtxIdx;
    }

    //-----------------------------------------------------------------------------
    std::ostringstream shapeOs;
    RInitOutStreamFormat(shapeOs);

    RVertexArray vertices;

    //-----------------------------------------------------------------------------
    // loop for shape data
    for (size_t shapeIdx = 0; shapeIdx < cshapes.size(); ++shapeIdx)
    {
        const CShape& cshape = cshapes[shapeIdx];
        const RIntArray* pBoneMtxIdxs = (cshape.m_SkinningMode == RShape::SMOOTH) ?
            &smoothMtxIdxs : &rigidMtxIdxs;
        OutputShape(shapeOs, &vertices, pDataStreams, pCmodel,
            tc + 1, static_cast<int>(shapeIdx), cshape, *pBoneMtxIdxs);
    }

    //-----------------------------------------------------------------------------
    // out vertex array
    ROutArrayElement(os, 0, vertices, "vertex_array");

    //-----------------------------------------------------------------------------
    // out shape array
    os << RTab(tc) << "<shape_array length=\"" << cshapes.size() << "\">" << R_ENDL;
    os << shapeOs.str();
    os << RTab(tc) << "</shape_array>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief ボーンアニメーションを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <bone_anim> 要素のインデントに必要なタブの数です。
//! @param[in] animIdx ボーンアニメーションのインデックスです。
//! @param[in] bone ボーンです。
//! @param[in] parentName 親のボーン名です。親がなければ空文字を指定します。
//-----------------------------------------------------------------------------
void OutputBoneAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const int animIdx,
    const CBone& bone,
    const std::string& parentName
)
{
    //-----------------------------------------------------------------------------
    // begin bone anim
    os << RTab(tc) << "<bone_anim index=\"" << animIdx
       << "\" bone_name=\"" << RGetUtf8FromShiftJis(bone.m_Name)
       << "\" parent_name=\"" << RGetUtf8FromShiftJis(parentName) << "\"" << R_ENDL;
    bone.OutMtxAttrib(os, tc + 1);
    os << RTab(tc + 1) << "scale_compensate=\"" << RBoolStr(bone.m_ScaleCompensate) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "binarize_scale=\"" << RBoolStr(bone.m_BinarizesScale) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "binarize_rotate=\"" << RBoolStr(bone.m_BinarizesRotate) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "binarize_translate=\"" << RBoolStr(bone.m_BinarizesTranslate) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "compress_enable=\"" << RBoolStr(bone.m_CompressEnable) << "\"" << R_ENDL;
    os << RTab(tc) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // bone anim targets
    for (int paramIdx = 0; paramIdx < RBone::PARAM_COUNT; ++paramIdx)
    {
        const RAnimCurve& curve = bone.m_Anims[paramIdx];
        curve.Out(os, *pDataStreams, tc + 1, "bone_anim_target",
            RBone::GetParamName(paramIdx), false);
    }

    //-----------------------------------------------------------------------------
    // end bone anim
    os << RTab(tc) << "</bone_anim>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief ボーンアニメーション群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <bone_anim_array> 要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputBoneAnims(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const CModel& cmodel
)
{
    const CBoneArray& bones = cmodel.m_Bones;
    const int boneCount = static_cast<int>(bones.size());
    if (boneCount != 0)
    {
        os << RTab(tc) << "<bone_anim_array length=\"" << boneCount << "\">" << R_ENDL;
        for (int boneIdx = 0; boneIdx < boneCount; ++boneIdx)
        {
            const CBone& bone = bones[boneIdx];
            const std::string parentName = (bone.m_ParentIdx != -1) ?
                bones[bone.m_ParentIdx].m_Name : "";
            OutputBoneAnim(os, pDataStreams, tc + 1, boneIdx, bone, parentName);
        }
        os << RTab(tc) << "</bone_anim_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief ボーンビジビリティアニメーションを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <bone_vis_bone_anim> 要素のインデントに必要なタブの数です。
//! @param[in] animIdx ボーンアニメーションのインデックスです。
//! @param[in] bone ボーンです。
//! @param[in] parentName 親のボーン名です。親がなければ空文字を指定します。
//-----------------------------------------------------------------------------
void OutputBoneVisibilityAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const int animIdx,
    const CBone& bone,
    const std::string& parentName
)
{
    const RAnimCurve& curve = bone.m_VisAnim;
    os << RTab(tc) << "<bone_vis_bone_anim index=\"" << animIdx
       << "\" bone_name=\"" << RGetUtf8FromShiftJis(bone.m_Name)
       << "\" parent_name=\"" << RGetUtf8FromShiftJis(parentName) << "\"" << R_ENDL;
    bone.OutMtxAttrib(os, tc + 1);
    os << RTab(tc + 1) << "binarize_visibility=\"" << RBoolStr(true) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "compress_enable=\"" << RBoolStr(bone.m_CompressEnable) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "base_value=\"" << curve.GetBaseValue() << "\"" << R_ENDL;
    if (curve.m_ConstantFlag)
    {
        os << RTab(tc) << "/>" << R_ENDL;
    }
    else
    {
        os << RTab(tc) << ">" << R_ENDL;
        curve.Out(os, *pDataStreams, tc + 1, false);
        os << RTab(tc) << "</bone_vis_bone_anim>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief ボーンビジビリティアニメーション群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <bone_vis_bone_anim_array> 要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputBoneVisibilityAnims(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const CModel& cmodel
)
{
    const CBoneArray& bones = cmodel.m_Bones;
    const int boneCount = static_cast<int>(bones.size());
    if (boneCount != 0)
    {
        os << RTab(tc) << "<bone_vis_bone_anim_array length=\"" << boneCount << "\">" << R_ENDL;
        for (int boneIdx = 0; boneIdx < boneCount; ++boneIdx)
        {
            const CBone& bone = bones[boneIdx];
            OutputBoneVisibilityAnim(os, pDataStreams, tc + 1, boneIdx, bone, "");
        }
        os << RTab(tc) << "</bone_vis_bone_anim_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief カラーアニメーションが設定されたマテリアルの数を返します。
//!
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
int GetColorAnimtedMaterialCount(const CModel& cmodel)
{
    int colorAnimatedMatCount = 0;
    const CMaterialArray& materials = cmodel.m_Materials;
    for (size_t matIdx = 0; matIdx < materials.size(); ++matIdx)
    {
        if (materials[matIdx].GetAnimatedColorCount() != 0)
        {
            ++colorAnimatedMatCount;
        }
    }
    return colorAnimatedMatCount;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ SRT アニメーションが設定されたマテリアルの数を返します。
//!
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
int GetTexSrtAnimtedMaterialCount(const CModel& cmodel)
{
    int texSrtAnimatedMatCount = 0;
    const CMaterialArray& materials = cmodel.m_Materials;
    for (size_t matIdx = 0; matIdx < materials.size(); ++matIdx)
    {
        if (!materials[matIdx].m_TexSrtAnims.empty())
        {
            ++texSrtAnimatedMatCount;
        }
    }
    return texSrtAnimatedMatCount;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターンアニメーションが設定されたマテリアルの数を返します。
//!
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
int GetTexPatAnimtedMaterialCount(const CModel& cmodel)
{
    int texPatAnimatedMatCount = 0;
    const CMaterialArray& materials = cmodel.m_Materials;
    for (size_t matIdx = 0; matIdx < materials.size(); ++matIdx)
    {
        if (!materials[matIdx].m_TexPatAnims.empty())
        {
            ++texPatAnimatedMatCount;
        }
    }
    return texPatAnimatedMatCount;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのカラーアニメーション配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc カラーアニメーション配列要素のインデントに必要なタブの数です。
//! @param[in] mat マテリアルです。
//-----------------------------------------------------------------------------
void OutputColorAnims(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const CMaterial& mat
)
{
    static const char* const hintStrs[] =
    {
        "diffuse" ,
        "opacity" ,
        "ambient" ,
        "emission",
        "specular",
        "specular1"
    };
    static const char* const targetStrs[] = { "color_r", "color_g", "color_b" };

    //-----------------------------------------------------------------------------
    // begin original color anim array
    const int animatedColorCount = mat.GetAnimatedColorCount();
    os << RTab(tc) << "<original_color_anim_array length=\"" << animatedColorCount << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // loop for color attr
    int colorAnimIdx = 0;
    for (int paramIdx = 0; paramIdx < CMaterial::CMCLA_PARAM_COUNT; paramIdx += R_RGB_COUNT)
    {
        if (mat.m_Anims[paramIdx + 0].m_UseFlag ||
            mat.m_Anims[paramIdx + 1].m_UseFlag ||
            mat.m_Anims[paramIdx + 2].m_UseFlag)
        {
            std::string hint;
            if (paramIdx >= CMaterial::CONSTANT0_R)
            {
                const int paramOfs = paramIdx - CMaterial::CONSTANT0_R;
                hint = "constant" + RGetNumberString(paramOfs / (R_RGB_COUNT * 2));
                hint += ((paramOfs % (R_RGB_COUNT * 2)) >= 3) ? "_opacity" : "";
            }
            else
            {
                hint = hintStrs[paramIdx / R_RGB_COUNT];
            }
            os << RTab(tc + 1) << "<original_color_anim index=\"" << colorAnimIdx
               << "\" hint=\"" << hint << "\">" << R_ENDL;
            for (int rgbaIdx = 0; rgbaIdx < R_RGB_COUNT; ++rgbaIdx)
            {
                const RAnimCurve& curve = mat.m_Anims[paramIdx + rgbaIdx];
                if (curve.m_UseFlag)
                {
                    curve.Out(os, *pDataStreams, tc + 2, "original_color_anim_target",
                        targetStrs[rgbaIdx], true);
                }
            }
            os << RTab(tc + 1) << "</original_color_anim>" << R_ENDL;
            ++colorAnimIdx;
        }
    }

    //-----------------------------------------------------------------------------
    // end original color anim array
    os << RTab(tc) << "</original_color_anim_array>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief fcl ファイルのアニメーションを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <original_material_anim_array> 要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputFclAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const CModel& cmodel
)
{
    const CMaterialArray& materials = cmodel.m_Materials;
    if (!materials.empty())
    {
        os << RTab(tc) << "<original_material_anim_array length=\"" << materials.size()
           << "\">" << R_ENDL;
        for (size_t matIdx = 0; matIdx < materials.size(); ++matIdx)
        {
            const CMaterial& mat = materials[matIdx];
            os << RTab(tc + 1) << "<original_material_anim index=\"" << matIdx
               << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
            OutputColorAnims(os, pDataStreams, tc + 2, mat);
            os << RTab(tc + 1) << "</original_material_anim>" << R_ENDL;
        }
        os << RTab(tc) << "</original_material_anim_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ SRT アニメーション配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc テクスチャ SRT アニメーション配列要素のインデントに必要なタブの数です。
//! @param[in] mat マテリアルです。
//-----------------------------------------------------------------------------
void OutputTexSrtAnims(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const CMaterial& mat
)
{
    //-----------------------------------------------------------------------------
    // begin original texsrt anim array
    const CTexSrtAnimArray& texSrtAnims = mat.m_TexSrtAnims;
    os << RTab(tc) << "<original_texsrt_anim_array length=\"" << texSrtAnims.size() << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // loop for texsrt anim
    for (size_t texSrtAnimIdx = 0; texSrtAnimIdx < texSrtAnims.size(); ++texSrtAnimIdx)
    {
        const CTexSrtAnim& texSrtAnim = texSrtAnims[texSrtAnimIdx];
        os << RTab(tc + 1) << "<original_texsrt_anim index=\"" << texSrtAnimIdx
           << "\" hint=\"" << texSrtAnim.m_Hint << "\"" << R_ENDL;
        os << RTab(tc + 2) << "mode=\"" << texSrtAnim.m_Mode << "\"" << R_ENDL;
        os << RTab(tc + 1) << ">" << R_ENDL;
        for (int paramIdx = 0; paramIdx < ROriginalTexsrt::PARAM_COUNT; ++paramIdx)
        {
            const RAnimCurve& curve = texSrtAnim.m_Anims[paramIdx];
            if (curve.m_UseFlag)
            {
                curve.Out(os, *pDataStreams, tc + 2, "original_texsrt_anim_target",
                    ROriginalTexsrt::GetParamName(paramIdx), true);
            }
        }
        os << RTab(tc + 1) << "</original_texsrt_anim>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // end original texsrt anim array
    os << RTab(tc) << "</original_texsrt_anim_array>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief fts ファイルのアニメーションを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <original_material_anim_array> 要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputFtsAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const CModel& cmodel
)
{
    const CMaterialArray& materials = cmodel.m_Materials;
    if (!materials.empty())
    {
        os << RTab(tc) << "<original_material_anim_array length=\"" << materials.size()
           << "\">" << R_ENDL;
        for (size_t matIdx = 0; matIdx < materials.size(); ++matIdx)
        {
            const CMaterial& mat = materials[matIdx];
            os << RTab(tc + 1) << "<original_material_anim index=\"" << matIdx
               << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
            OutputTexSrtAnims(os, pDataStreams, tc + 2, mat);
            os << RTab(tc + 1) << "</original_material_anim>" << R_ENDL;
        }
        os << RTab(tc) << "</original_material_anim_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターン配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc テクスチャパターン配列要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputTexPatterns(std::ostream& os, const int tc, const CModel& cmodel)
{
    const CTexInfoArray& texInfos = cmodel.m_TexInfos;
    if (!texInfos.empty())
    {
        os << RTab(tc) << "<tex_pattern_array length=\"" << texInfos.size()
           << "\">" << R_ENDL;
        for (size_t texInfoIdx = 0; texInfoIdx < texInfos.size(); ++texInfoIdx)
        {
            const CTexInfo& texInfo = texInfos[texInfoIdx];
            os << RTab(tc + 1) << "<tex_pattern"
               << " pattern_index=\"" << texInfoIdx
               << "\" tex_name=\"" << RGetUtf8FromShiftJis(REncodeXmlString(texInfo.m_TexName))
               << "\" />" << R_ENDL;
        }
        os << RTab(tc) << "</tex_pattern_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのテクスチャパターンアニメーション配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] isFma fma ファイルなら true、ftp ファイルなら false を指定します。
//! @param[in] tc テクスチャパターンアニメーション配列要素のインデントに必要なタブの数です。
//! @param[in] mat マテリアルです。
//-----------------------------------------------------------------------------
void OutputTexPatAnims(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const bool isFma,
    const int tc,
    const CMaterial& mat
)
{
    const CTexPatAnimArray& texPatAnims = mat.m_TexPatAnims;
    if (isFma)
    {
        os << RTab(tc) << "<tex_pattern_anim_array length=\""
           << texPatAnims.size() << "\">" << R_ENDL;
    }
    const int texPatAnimTc = (isFma) ? tc + 1 : tc;
    const char* texPatAnimElemName = (isFma) ?
        "pattern_anim" : "pattern_anim_target";
    for (size_t texPatAnimIdx = 0; texPatAnimIdx < texPatAnims.size(); ++texPatAnimIdx)
    {
        const CTexPatAnim& texPatAnim = texPatAnims[texPatAnimIdx];
        const RAnimCurve& curve = texPatAnim.m_Anim;
        os << RTab(texPatAnimTc) << "<" << texPatAnimElemName
           << " sampler_name=\"" << texPatAnim.m_SamplerName
           << "\" hint=\"" << texPatAnim.m_Hint
           << "\" base_value=\"" << curve.GetBaseValue() << "\"";
        if (curve.m_ConstantFlag)
        {
            os << " />" << R_ENDL;
        }
        else
        {
            os << ">" << R_ENDL;
            curve.Out(os, *pDataStreams, texPatAnimTc + 1, false);
            os << RTab(texPatAnimTc) << "</" << texPatAnimElemName << ">" << R_ENDL;
        }
    }
    if (isFma)
    {
        os << RTab(tc) << "</tex_pattern_anim_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief ftp ファイルのアニメーションを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <tex_pattern_mat_anim_array> 要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputFtpAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const CModel& cmodel
)
{
    //-----------------------------------------------------------------------------
    // テクスチャパターン配列を出力します。
    OutputTexPatterns(os, tc, cmodel);

    //-----------------------------------------------------------------------------
    // テクスチャパターンアニメーションを持つマテリアルについてループします。
    const CMaterialArray& materials = cmodel.m_Materials;
    if (!materials.empty())
    {
        os << RTab(tc) << "<tex_pattern_mat_anim_array length=\"" << materials.size()
           << "\">" << R_ENDL;
        for (size_t matIdx = 0; matIdx < materials.size(); ++matIdx)
        {
            const CMaterial& mat = materials[matIdx];
            os << RTab(tc + 1) << "<tex_pattern_mat_anim index=\"" << matIdx
               << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
            OutputTexPatAnims(os, pDataStreams, false, tc + 2, mat);
            os << RTab(tc + 1) << "</tex_pattern_mat_anim>" << R_ENDL;
        }
        os << RTab(tc) << "</tex_pattern_mat_anim_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief fma ファイルのアニメーションを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] cmodel モデルです。
//! @param[in] outputsColor カラーアニメーションを含めるなら true です。
//! @param[in] outputsTexSrt テクスチャ SRT アニメーションを含めるなら true です。
//! @param[in] outputsTexPat テクスチャパターンアニメーションを含めるなら true です。
//-----------------------------------------------------------------------------
void OutputFmaAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const CModel& cmodel,
    const bool outputsColor,
    const bool outputsTexSrt,
    const bool outputsTexPat
)
{
    const int tc = 0;

    //-----------------------------------------------------------------------------
    // テクスチャパターン配列
    const int texPatAnimatedCount =
        (outputsTexPat) ? GetTexPatAnimtedMaterialCount(cmodel) : 0;
    if (texPatAnimatedCount != 0)
    {
        OutputTexPatterns(os, tc, cmodel);
    }

    //-----------------------------------------------------------------------------
    // マテリアル単位アニメーション配列
    const CMaterialArray& materials = cmodel.m_Materials;
    const int animatedCount = texPatAnimatedCount;
    if (animatedCount != 0)
    {
        os << RTab(tc) << "<per_material_anim_array length=\"" << animatedCount
           << "\">" << R_ENDL;
        int matAnimIdx = 0;
        for (size_t matIdx = 0; matIdx < materials.size(); ++matIdx)
        {
            const CMaterial& mat = materials[matIdx];
            const bool hasTexPatAnim = (outputsTexPat && !mat.m_TexPatAnims.empty());
            if (hasTexPatAnim)
            {
                os << RTab(tc + 1) << "<per_material_anim index=\"" << matAnimIdx
                   << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                OutputTexPatAnims(os, pDataStreams, true, tc + 2, mat);
                os << RTab(tc + 1) << "</per_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << RTab(tc) << "</per_material_anim_array>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // オリジナルのマテリアル単位アニメーション配列
    const int orgAnimatedCount =
        ((outputsColor ) ? GetColorAnimtedMaterialCount(cmodel)  : 0) +
        ((outputsTexSrt) ? GetTexSrtAnimtedMaterialCount(cmodel) : 0);
    if (orgAnimatedCount != 0)
    {
        os << RTab(tc) << "<original_per_material_anim_array length=\"" << orgAnimatedCount
           << "\">" << R_ENDL;
        int matAnimIdx = 0;
        for (size_t matIdx = 0; matIdx < materials.size(); ++matIdx)
        {
            const CMaterial& mat = materials[matIdx];
            const bool hasColorAnim  = (outputsColor  && mat.GetAnimatedColorCount() != 0);
            const bool hasTexSrtAnim = (outputsTexSrt && !mat.m_TexSrtAnims.empty());
            if (hasColorAnim || hasTexSrtAnim)
            {
                os << RTab(tc + 1) << "<original_per_material_anim index=\"" << matAnimIdx
                   << "\" mat_name=\"" << RGetUtf8FromShiftJis(mat.m_Name) << "\">" << R_ENDL;
                if (hasColorAnim)
                {
                    OutputColorAnims(os, pDataStreams, tc + 2, mat);
                }
                if (hasTexSrtAnim)
                {
                    OutputTexSrtAnims(os, pDataStreams, tc + 2, mat);
                }
                os << RTab(tc + 1) << "</original_per_material_anim>" << R_ENDL;
                ++matAnimIdx;
            }
        }
        os << RTab(tc) << "</original_per_material_anim_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief マテリアル関連のアニメーションを 1 つ出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] cmodel モデルです。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] expOpt エクスポートオプションです。
//-----------------------------------------------------------------------------
void OutputOneMatAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const CModel& cmodel,
    const RExpOpt::FileType fileType,
    const CExpOpt& expOpt
)
{
    if (expOpt.UsesSingleFma())
    {
        OutputFmaAnim(os, pDataStreams, cmodel,
            expOpt.ExportsColorAnim(),
            expOpt.ExportsTexSrtAnim(),
            expOpt.ExportsTexPatAnim());
    }
    else if (expOpt.UsesSeparatedFma())
    {
        switch (fileType)
        {
        case RExpOpt::FCL:
            OutputFmaAnim(os, pDataStreams, cmodel, true , false, false);
        case RExpOpt::FTS:
            OutputFmaAnim(os, pDataStreams, cmodel, false, true , false);
        case RExpOpt::FTP:
            OutputFmaAnim(os, pDataStreams, cmodel, false, false, true );
        default:
            break;
        }
    }
    else
    {
        const int tc = 0;
        switch (fileType)
        {
        case RExpOpt::FCL:
            OutputFclAnim(os, pDataStreams, tc, cmodel);
            break;
        case RExpOpt::FTS:
            OutputFtsAnim(os, pDataStreams, tc, cmodel);
            break;
        case RExpOpt::FTP:
            OutputFtpAnim(os, pDataStreams, tc, cmodel);
            break;
        default:
            break;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief カメラアニメーションを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <camera_anim> 要素のインデントに必要なタブの数です。
//! @param[in] index カメラアニメーションのインデックスです。
//-----------------------------------------------------------------------------
void CCamera::OutputAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tabCount,
    const int index
) const
{
    static const char* const rotateModeStrs[] = { "aim", "euler_zxy" };
    static const char* const projectionModeStrs[] = { "ortho", "persp" };

    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // begin
    os << RTab(tc) << "<camera_anim"
       << " index=\"" << index
       << "\" camera_name=\"" << RGetUtf8FromShiftJis(m_Name) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "frame_count=\"" << m_FrameCount
       << "\" loop=\"" << RBoolStr(m_LoopAnim) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "rotate_mode=\"" << rotateModeStrs[m_RotateMode] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "projection_mode=\"" << projectionModeStrs[m_ProjectionMode] << "\"" << R_ENDL;
    os << RTab(tc) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // camera anim targets
    for (int paramIdx = 0; paramIdx < PARAM_COUNT; ++paramIdx)
    {
        if (m_RotateMode == ROTATE_AIM)
        {
            if (ROTATE_X <= paramIdx && paramIdx <= ROTATE_Z) continue;
        }
        else // ROTATE_EULER_ZXY
        {
            if (AIM_X <= paramIdx && paramIdx <= AIM_Z) continue;
            if (paramIdx == TWIST) continue;
        }

        if (m_ProjectionMode == ORTHO)
        {
            if (paramIdx == PERSP_FOVY) continue;
        }
        else // PERSP
        {
            if (paramIdx == ORTHO_HEIGHT) continue;
        }

        const RAnimCurve& curve = m_Anims[paramIdx];
        curve.Out(os, *pDataStreams, tc + 1, "camera_anim_target",
            GetParamName(paramIdx), false);
    }

    //-----------------------------------------------------------------------------
    // カメラのユーザーデータ配列を出力します。
    ROutArrayElement(os, tc + 1, m_UserDatas, "user_data_array");

    //-----------------------------------------------------------------------------
    // end
    os << RTab(tc) << "</camera_anim>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief ライトアニメーションを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <light_anim> 要素のインデントに必要なタブの数です。
//! @param[in] index ライトアニメーションのインデックスです。
//-----------------------------------------------------------------------------
void CLight::OutputAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tabCount,
    const int index
) const
{
    static const char* const typeStrs[] = { "ambient", "directional", "point", "spot" };

    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // begin
    os << RTab(tc) << "<light_anim"
       << " index=\"" << index
       << "\" light_name=\"" << RGetUtf8FromShiftJis(m_Name) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "frame_count=\"" << m_FrameCount
       << "\" loop=\"" << RBoolStr(m_LoopAnim) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "type=\"" << typeStrs[m_Type] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "dist_attn_func =\"" << m_DistAttnFunc << "\"" << R_ENDL;
    os << RTab(tc + 1) << "angle_attn_func =\"" << m_AngleAttnFunc << "\"" << R_ENDL;
    os << RTab(tc) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // light anim targets
    for (int paramIdx = 0; paramIdx < PARAM_COUNT; ++paramIdx)
    {
        if (POSITION_X <= paramIdx && paramIdx <= POSITION_Z)
        {
            if (m_Type != POINT && m_Type != SPOT) continue;
        }
        else if (DIRECTION_X <= paramIdx && paramIdx <= DIRECTION_Z)
        {
            if (m_Type != DIRECTIONAL) continue;
        }
        else if (AIM_X <= paramIdx && paramIdx <= AIM_Z)
        {
            if (m_Type != SPOT) continue;
        }
        else if (paramIdx == DIST_ATTN_START || paramIdx == DIST_ATTN_END)
        {
            if (!m_UsesDistAttn) continue;
        }
        else if (paramIdx == ANGLE_ATTN_START || paramIdx == ANGLE_ATTN_END)
        {
            if (m_Type != SPOT) continue;
        }
        else if (COLOR1_R <= paramIdx && paramIdx <= COLOR1_B)
        {
            continue; // 本コンバータからは color1_* を出力しません。
        }

        const RAnimCurve& curve = m_Anims[paramIdx];
        curve.Out(os, *pDataStreams, tc + 1, "light_anim_target",
            GetParamName(paramIdx), false);
    }

    //-----------------------------------------------------------------------------
    // ライトのユーザーデータ配列を出力します。
    ROutArrayElement(os, tc + 1, m_UserDatas, "user_data_array");

    //-----------------------------------------------------------------------------
    // end
    os << RTab(tc) << "</light_anim>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief フォグアニメーションを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <fog_anim> 要素のインデントに必要なタブの数です。
//! @param[in] index フォグアニメーションのインデックスです。
//-----------------------------------------------------------------------------
void CFog::OutputAnim(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tabCount,
    const int index
) const
{
    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // begin
    os << RTab(tc) << "<fog_anim"
       << " index=\"" << index
       << "\" fog_name=\"" << RGetUtf8FromShiftJis(m_Name) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "frame_count=\"" << m_FrameCount
       << "\" loop=\"" << RBoolStr(m_LoopAnim) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "dist_attn_func =\"" << m_DistAttnFunc << "\"" << R_ENDL;
    os << RTab(tc) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // light anim targets
    for (int paramIdx = 0; paramIdx < PARAM_COUNT; ++paramIdx)
    {
        const RAnimCurve& curve = m_Anims[paramIdx];
        curve.Out(os, *pDataStreams, tc + 1, "fog_anim_target",
            GetParamName(paramIdx), false);
    }

    //-----------------------------------------------------------------------------
    // フォグのユーザーデータ配列を出力します。
    ROutArrayElement(os, tc + 1, m_UserDatas, "user_data_array");

    //-----------------------------------------------------------------------------
    // end
    os << RTab(tc) << "</fog_anim>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 環境オブジェクト群を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pDataStreams データ列配列へのポインタです。
//! @param[in] tc <****_anim_array> 要素のインデントに必要なタブの数です。
//! @param[in] cmodel モデルです。
//-----------------------------------------------------------------------------
void OutputEnvObjs(
    std::ostream& os,
    RDataStreamArray* pDataStreams,
    const int tc,
    const CModel& cmodel
)
{
    //-----------------------------------------------------------------------------
    // カメラアニメーション配列を出力します。
    const CCameraArray& cams = cmodel.m_Cameras;
    if (!cams.empty())
    {
        os << RTab(tc) << "<camera_anim_array length=\"" << cams.size()
           << "\">" << R_ENDL;
        for (size_t camIdx = 0; camIdx < cams.size(); ++camIdx)
        {
            cams[camIdx].OutputAnim(os, pDataStreams, tc + 1, static_cast<int>(camIdx));
        }
        os << RTab(tc) << "</camera_anim_array>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // ライトアニメーション配列を出力します。
    const CLightArray& lgts = cmodel.m_Lights;
    if (!lgts.empty())
    {
        os << RTab(tc) << "<light_anim_array length=\"" << lgts.size()
           << "\">" << R_ENDL;
        for (size_t lgtIdx = 0; lgtIdx < lgts.size(); ++lgtIdx)
        {
            lgts[lgtIdx].OutputAnim(os, pDataStreams, tc + 1, static_cast<int>(lgtIdx));
        }
        os << RTab(tc) << "</light_anim_array>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // フォグアニメーション配列を出力します。
    const CFogArray& fogs = cmodel.m_Fogs;
    if (!fogs.empty())
    {
        os << RTab(tc) << "<fog_anim_array length=\"" << fogs.size()
           << "\">" << R_ENDL;
        for (size_t fogIdx = 0; fogIdx < fogs.size(); ++fogIdx)
        {
            fogs[fogIdx].OutputAnim(os, pDataStreams, tc + 1, static_cast<int>(fogIdx));
        }
        os << RTab(tc) << "</fog_anim_array>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイル情報に出力する元ファイルのパスを取得します。
//!
//! @param[in] fullPath 元ファイルのフルパスです。
//! @param[in] projectRootPath プロジェクトのルートフォルダーのパスです。
//!                            空文字でなければ元ファイルのパスはルートフォルダーからの相対パスとなります。
//!
//! @return 元ファイルのパスを返します。
//-----------------------------------------------------------------------------
std::string GetSourceFilePath(
    const std::string& fullPath,
    const std::string& projectRootPath
)
{
    std::string srcPath(fullPath);
    if (!projectRootPath.empty())
    {
        char relativePath[512];
        if (::PathRelativePathToA(relativePath,
            RGetWindowsFilePath(projectRootPath).c_str(), FILE_ATTRIBUTE_DIRECTORY,
            RGetWindowsFilePath(fullPath       ).c_str(), FILE_ATTRIBUTE_NORMAL))
        {
            srcPath = RGetUnixFilePath(relativePath);
            if (srcPath.find("./") == 0)
            {
                srcPath = srcPath.substr(2);
            }
        }
    }
    return srcPath;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのヘッダを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc 要素のインデントに必要なタブの数です。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] cmodel モデルです。
//! @param[in] expOpt エクスポートオプションです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputHeader(
    std::ostream& os,
    const int tc,
    const RExpOpt::FileType fileType,
    const CModel& cmodel,
    const CExpOpt& expOpt,
    const C2NnOption& opt
)
{
    //-----------------------------------------------------------------------------
    // xml header
    os << RTab(tc) << "<?xml version=\"1.0\" encoding=\"utf-8\"?>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // begin intermediate file
    os << RTab(tc) << "<nw4f_3dif version=\"" << RExpOpt::GetRootElementVersion()
       << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // file info
    if (!opt.m_DisablesFileInfo)
    {
        os << RTab(tc) << "<file_info>" << R_ENDL;
        const std::string toolName = "3dCmdlConverter";
        os << RTab(tc + 1) << "<create tool_name=\"" << RGetUtf8FromShiftJis(REncodeXmlString(toolName))
           << "\" tool_version=\"" << ConverterVersionMajor << "." << ConverterVersionMinor << "." << ConverterVersionMicro << "\"" << R_ENDL;
        const std::string srcPath = GetSourceFilePath(expOpt.m_SrcPath, opt.m_ProjectRootPath);
        os << RTab(tc + 2) << "src_path=\"" << RGetUtf8FromShiftJis(REncodeXmlString(srcPath)) << "\"" << R_ENDL;
        os << RTab(tc + 1) << "/>" << R_ENDL;
        os << RTab(tc) << "</file_info>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // begin root element
    os << RTab(tc) << "<" << expOpt.GetTypeElementName(fileType)
       << " version=\"" << expOpt.GetTypeElementVersion(fileType) << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // model info
    if (fileType == RExpOpt::FMD)
    {
        os << RTab(tc) << "<model_info" << R_ENDL;
        os << RTab(tc + 1) << "material_count=\"" << cmodel.m_Materials.size() << "\"" << R_ENDL;
        os << RTab(tc + 1) << "bone_count=\"" << cmodel.m_Bones.size() << "\"" << R_ENDL;
        os << RTab(tc + 1) << "shape_count=\"" << cmodel.m_Shapes.size() << "\"" << R_ENDL;
        os << RTab(tc + 1) << "smooth_skinning_shape=\"" << cmodel.m_SmoothSkinningShapeCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "rigid_skinning_shape=\"" << cmodel.m_RigidSkinningShapeCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "smooth_skinning_matrix=\"" << cmodel.m_SmoothSkinningMtxCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "rigid_skinning_matrix=\"" << cmodel.m_RigidSkinningMtxCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "total_process_vertex=\"" << cmodel.m_TotalProcessVertexCount << "\"" << R_ENDL;
        os << RTab(tc + 1) << "unite_pos_quantize=\"" << RBoolStr(false) << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_preset=\"" << RGetUtf8FromShiftJis(REncodeXmlString(expOpt.m_PresetName)) << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_magnify=\"" << expOpt.m_Magnify * opt.m_ConversionMagnify << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_start_frame=\"" << expOpt.m_StartFrame << "\"" << R_ENDL;
        os << RTab(tc) << "/>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // animation info
    const bool isFma = (!expOpt.m_UsesFclFtsFtp && expOpt.IsFclFtsFtp(fileType));
    const char* animInfoElem = nullptr;
    if (isFma)
    {
        animInfoElem = "material_anim_info";
    }
    else
    {
        switch (fileType)
        {
        case RExpOpt::FSK: animInfoElem = "skeletal_anim_info"       ; break;
        case RExpOpt::FVB: animInfoElem = "bone_visibility_anim_info"; break;
        case RExpOpt::FCL: animInfoElem = "shader_param_anim_info"   ; break;
        case RExpOpt::FTS: animInfoElem = "shader_param_anim_info"   ; break;
        case RExpOpt::FTP: animInfoElem = "tex_pattern_anim_info"    ; break;
        case RExpOpt::FSH: animInfoElem = "shape_anim_info"          ; break;
        case RExpOpt::FSN: animInfoElem = "scene_anim_info"          ; break;
        default: break;
        }
    }

    if (animInfoElem != nullptr)
    {
        os << RTab(tc) << "<" << animInfoElem;
        if (fileType != RExpOpt::FSN)
        {
            os << " frame_count=\"" << expOpt.m_OutFrameCount
               << "\" loop=\"" << RBoolStr(expOpt.m_LoopAnim)
               << "\"" << R_ENDL;
        }
        else
        {
            os << R_ENDL;
        }
        os << RTab(tc + 1) << "frame_resolution=\"" << expOpt.m_FramePrecision << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_preset=\"" << RGetUtf8FromShiftJis(REncodeXmlString(expOpt.m_PresetName)) << "\"" << R_ENDL;
        if (fileType == RExpOpt::FSK ||
            fileType == RExpOpt::FSN)
        {
            os << RTab(tc + 1) << "dcc_magnify=\"" << expOpt.m_Magnify * opt.m_ConversionMagnify << "\"" << R_ENDL;
        }
        os << RTab(tc + 1) << "dcc_start_frame=\"" << expOpt.m_StartFrame << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_end_frame=\"" << expOpt.m_EndFrame << "\"" << R_ENDL;
        os << RTab(tc + 1) << "dcc_fps=\"" << expOpt.m_FramesPerSecond << "\"" << R_ENDL;
        os << RTab(tc + 1) << "bake_all=\"" << RBoolStr(expOpt.m_BakeAllAnim) << "\"" << R_ENDL;

        if (fileType == RExpOpt::FSK)
        {
            cmodel.m_Skeleton.OutScaleRotateMode(os, tc + 1);
            os << RTab(tc + 1) << "bake_tolerance_scale=\""     << expOpt.m_TolS << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_rotate=\""    << expOpt.m_TolR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_translate=\"" << expOpt.m_TolT << "\"" << R_ENDL;
        }
        else if (isFma                    ||
                 fileType == RExpOpt::FCL ||
                 fileType == RExpOpt::FTS)
        {
            os << RTab(tc + 1) << "bake_tolerance_color=\""         << expOpt.m_TolC    << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_tex_scale=\""     << expOpt.m_TolTexS << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_tex_rotate=\""    << expOpt.m_TolTexR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_tex_translate=\"" << expOpt.m_TolTexT << "\"" << R_ENDL;
        }
        else if (fileType == RExpOpt::FSN)
        {
            os << RTab(tc + 1) << "bake_tolerance_rotate=\""    << expOpt.m_TolR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_translate=\"" << expOpt.m_TolT << "\"" << R_ENDL;
            os << RTab(tc + 1) << "bake_tolerance_color=\""     << expOpt.m_TolC << "\"" << R_ENDL;
        }

        if (fileType == RExpOpt::FSK)
        {
            os << RTab(tc + 1) << "quantize_tolerance_scale=\""     << expOpt.m_QuantTolS << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_rotate=\""    << expOpt.m_QuantTolR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_translate=\"" << expOpt.m_QuantTolT << "\"" << R_ENDL;
        }
        else if (isFma                    ||
                 fileType == RExpOpt::FCL ||
                 fileType == RExpOpt::FTS)
        {
            os << RTab(tc + 1) << "quantize_tolerance_tex_scale=\""     << expOpt.m_QuantTolTexS << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_tex_rotate=\""    << expOpt.m_QuantTolTexR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_tex_translate=\"" << expOpt.m_QuantTolTexT << "\"" << R_ENDL;
        }
        else if (fileType == RExpOpt::FSN)
        {
            os << RTab(tc + 1) << "quantize_tolerance_rotate=\""    << expOpt.m_QuantTolR << "\"" << R_ENDL;
            os << RTab(tc + 1) << "quantize_tolerance_translate=\"" << expOpt.m_QuantTolT << "\"" << R_ENDL;
        }

        os << RTab(tc) << "/>" << R_ENDL;
    }

    return RStatus::SUCCESS;
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのフッタを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tc 要素のインデントに必要なタブの数です。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] expOpt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputFooter(
    std::ostream& os,
    const int tc,
    const RExpOpt::FileType fileType,
    const CExpOpt& expOpt
)
{
    //-----------------------------------------------------------------------------
    // 編集用コメント
    if (!expOpt.m_CommentText.empty())
    {
        os << RTab(tc) << "<comment label=\"" << ""
           << "\" color=\"" << ""
           << "\" text=\"" << RGetUtf8FromShiftJis(REncodeXmlString(expOpt.m_CommentText))
           << "\" />" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // end root element
    os << RTab(tc) << "</" << expOpt.GetTypeElementName(fileType) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // end intermediate file
    os << RTab(tc) << "</nw4f_3dif>" << R_ENDL;

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルを 1 つ出力します。
//!
//! @param[in,out] pCmodel モデルへのポインタです。
//! @param[in] filePath 出力する中間ファイルのパスです。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] expOpt エクスポートオプションです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputOneFile(
    CModel* pCmodel,
    const std::string& filePath,
    const RExpOpt::FileType fileType,
    const CExpOpt& expOpt,
    const C2NnOption& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 各中間ファイルの内容を文字列ストリームに出力します。
    // （内容を出力しないと確定できないヘッダの情報があるため）
    std::ostringstream contentOs;
    RInitOutStreamFormat(contentOs);
    RDataStreamArray dataStreams;
    const int tc = 0;

    if (fileType == RExpOpt::FMD)
    {
        OutputMaterials(contentOs, tc, *pCmodel);
        OutputSkeleton(contentOs, tc, *pCmodel);
        OutputShapes(contentOs, &dataStreams, pCmodel, tc);
        OutputOriginalMaterials(contentOs, tc, *pCmodel);
    }
    else if (fileType == RExpOpt::FSK)
    {
        OutputBoneAnims(contentOs, &dataStreams, tc, *pCmodel);
    }
    else if (fileType == RExpOpt::FVB)
    {
        OutputBoneVisibilityAnims(contentOs, &dataStreams, tc, *pCmodel);
    }
    else if (fileType == RExpOpt::FCL ||
             fileType == RExpOpt::FTS ||
             fileType == RExpOpt::FTP)
    {
        OutputOneMatAnim(contentOs, &dataStreams, *pCmodel, fileType, expOpt);
    }
    else if (fileType == RExpOpt::FSN)
    {
        OutputEnvObjs(contentOs, &dataStreams, tc, *pCmodel);
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルに対するユーザーデータを出力します。
    if (fileType == RExpOpt::FMD)
    {
        ROutArrayElement(contentOs, tc, pCmodel->m_UserDatas, "user_data_array");
    }
    else
    {
        ROutArrayElement(contentOs, tc, pCmodel->m_AnimUserDatas, "user_data_array");
    }

    //-----------------------------------------------------------------------------
    // 出力フォルダを作成します。
    status = CreateFolderIfNotExist(RGetFolderFromFilePath(filePath));
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // ファイルを開きます。
    std::ofstream ofs;
    ofs.open(filePath.c_str(), ios_base::binary);
    if (!ofs)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + filePath); // RShowError
    }
    std::ostream& os = ofs;
    ROutUtf8Bom(os);
    RInitOutStreamFormat(os);

    //-----------------------------------------------------------------------------
    // ヘッダを出力します。
    OutputHeader(os, 0, fileType, *pCmodel, expOpt, opt);

    //-----------------------------------------------------------------------------
    // 各中間ファイルの内容を出力します。
    os << contentOs.str();

    //-----------------------------------------------------------------------------
    // アスキー形式の場合、ここでデータ列配列を出力します。
    if (!opt.m_IsBinaryFormat)
    {
        ROutDataStreams(os, dataStreams);
    }

    //-----------------------------------------------------------------------------
    // フッタを出力します。
    OutputFooter(os, 0, fileType, expOpt);

    //-----------------------------------------------------------------------------
    // バイナリ形式の場合、ここでデータ列配列を出力します。
    if (opt.m_IsBinaryFormat)
    {
        ROutBinaryDataStreams(os, dataStreams, true);
    }

    //-----------------------------------------------------------------------------
    // ファイルを閉じます。
    ofs.close();

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザを実行します。
    status = Do3dOptimizer(filePath, fileType, opt);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 3D テクスチャーコンバーターを実行します。
//!
//! @param[in] texCvtrPath 3D テクスチャーコンバーターのパスです。
//! @param[in] cvtrStr 3D テクスチャーコンバーターのオプション文字列です。
//! @param[in] texFolderPath テクスチャー出力フォルダのパスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus Do3dTextureConverter(
    const std::string& texCvtrPath,
    const std::string& cvtrStr,
    const std::string& texFolderPath
)
{
    //-----------------------------------------------------------------------------
    // 3D テクスチャーコンバーターの exe ファイルが存在するかチェックします。
    if (!RFileExists(texCvtrPath))
    {
        return RStatus(RStatus::FAILURE, "3D texture converter cannot be found: " + texCvtrPath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // オプションファイルを作成します。
    //cerr << "tex cvtrOss:" << endl << cvtrStr << endl;
    const std::string cvtrOptFilePath = texFolderPath +
        "Nintendo_3dTextureConverter_" +
        RGetNumberString(static_cast<int>(GetCurrentProcessId())) + ".txt";
    std::ofstream ofs(cvtrOptFilePath, ios::binary);
    if (!ofs)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + cvtrOptFilePath); // RShowError
    }
    ofs.write(cvtrStr.c_str(), cvtrStr.size());
    ofs.close();

    //-----------------------------------------------------------------------------
    // 3D テクスチャーコンバーターを起動します。
    std::string cmd = "\"" + RGetWindowsFilePath(texCvtrPath) + "\"";
    cmd += " -s";
    //cmd += " --final_folder=\"" + texFolderPath + "\"";
    //cmd += " --tile-mode linear";
    cmd += " --job-list=\"" + cvtrOptFilePath + "\"";
    //cerr << "texcvtr: " << cmd << endl;

    std::string outMsg;
    std::string errMsg;
    const int exitCode = RExecProcessWithPipe(nullptr, &outMsg, &errMsg, cmd.c_str(), SW_HIDE);
    //cerr << "texcvtr msg:" << endl << errMsg << endl;
    if (!outMsg.empty())
    {
        //cerr << outMsg << endl;
    }
    RStatus status;
    if (exitCode != 0)
    {
        status = RStatus(RStatus::FAILURE, "3D texture converter failed\n" + errMsg); // RShowError
    }

    //-----------------------------------------------------------------------------
    // オプションファイルを削除します。
    ::DeleteFileA(cvtrOptFilePath.c_str());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 出力するテクスチャのフォーマットを取得します。
//!
//! @param[in] ctexFormat ctex ファイルのフォーマットです。
//! @param[in] texLinearFlag テクスチャのリニア変換フラグです。
//!
//! @brief 出力するテクスチャのフォーマットを返します。
//-----------------------------------------------------------------------------
std::string GetFtxTextureFormat(
    const std::string& ctexFormat,
    const int texLinearFlag
)
{
    const bool isSrgb = ((texLinearFlag & C2NnOption::LinearFlag_Rgb) == C2NnOption::LinearFlag_Rgb);
    return
        (ctexFormat == "L4"     ) ? "unorm_8"   : // or unorm_bc4
        (ctexFormat == "L8"     ) ? "unorm_8"   :
        (ctexFormat == "A4"     ) ? "unorm_8"   : // or unorm_bc4
        (ctexFormat == "A8"     ) ? "unorm_8"   :
        (ctexFormat == "La4"    ) ? "unorm_8_8" : // unorm_4_4 は Cafe 以外で使用できない
        (ctexFormat == "La8"    ) ? "unorm_8_8" :
        (ctexFormat == "Hilo8"  ) ? "snorm_bc5" : // or snorm_8_8
        (ctexFormat == "Rgb565" ) ? "unorm_5_6_5" :
        (ctexFormat == "Rgb8"   ) ? ((isSrgb) ? "srgb_8_8_8_8" : "unorm_8_8_8_8") :
        (ctexFormat == "Rgb5_a1") ? "unorm_5_5_5_1"                               :
        (ctexFormat == "Rgba4"  ) ? "unorm_4_4_4_4"                               :
        (ctexFormat == "Rgba8"  ) ? ((isSrgb) ? "srgb_8_8_8_8" : "unorm_8_8_8_8") :
        (ctexFormat == "Etc1"   ) ? ((isSrgb) ? "srgb_bc1"     : "unorm_bc1"    ) :
        (ctexFormat == "Etc1_a4") ? ((isSrgb) ? "srgb_bc3"     : "unorm_bc3"    ) :
        ((isSrgb) ? "srgb_8_8_8_8" : "unorm_8_8_8_8");
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ中間ファイル群を出力します。
//!
//! @param[in,out] pOutputFtxPaths 出力済みの ftx ファイルのパス配列へのポインタです。
//! @param[in] texInfos テクスチャ情報配列です。
//! @param[in] texFolderPath テクスチャ出力フォルダのパスです。
//! @param[in] opt 変換オプションです。
//! @param[in] pInputFile 入力ファイルへのポインタです。
//!                       テクスチャ要素の存在するファイルが入力ファイルと同じなら
//!                       入力ファイルを解析します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputTextureFiles(
    RStringArray* pOutputFtxPaths,
    const CTexInfoArray& texInfos,
    const std::string& texFolderPath,
    const C2NnOption& opt,
    const CIntermediateFile* pInputFile
)
{
    //-----------------------------------------------------------------------------
    // テクスチャ出力フォルダを作成します。
    RStatus status;
    status = CreateFolderIfNotExist(RGetFullFilePath(texFolderPath + "../", true));
    RCheckStatus(status);
    status = CreateFolderIfNotExist(texFolderPath);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // ctex ファイルを DDS ファイルに変換します。
    RStringArray ddsPaths;
    std::ostringstream cvtrOss;
    for (size_t texInfoIdx = 0; texInfoIdx < texInfos.size(); ++texInfoIdx)
    {
        //-----------------------------------------------------------------------------
        // すでに出力済みの ftx ファイルであればスキップします。
        const CTexInfo& texInfo = texInfos[texInfoIdx];
        const std::string ftxPath = texFolderPath + texInfo.GetOutFileName();
        if (RFindValueInArray(*pOutputFtxPaths, ftxPath) != -1)
        {
            continue;
        }
        pOutputFtxPaths->push_back(ftxPath);

        //-----------------------------------------------------------------------------
        // テクスチャ出力フォルダにテクスチャと同名の DDS ファイルが存在する場合は
        // DDS ファイル名の末尾に数値を追加します。
        std::string ddsName = texInfo.m_OutName;
        int id = 1;
        while (RFileExists(texFolderPath + ddsName + ".dds"))
        {
            ddsName = texInfo.m_TexName + RGetNumberString(id++ , "_%d");
        }
        const std::string ddsPath = texFolderPath + ddsName + ".dds";

        //-----------------------------------------------------------------------------
        // DDS ファイルに変換します。
        ddsPaths.push_back(ddsPath);
        std::string ctexFormat;
        int mipLevel = 1;
        if (pInputFile != nullptr && pInputFile->m_Path == texInfo.m_CtexPath)
        {
            status = CreateImageFileFromCtex(&ctexFormat, &mipLevel, opt.m_pTextureEncoder,
                ddsPath, *pInputFile, texInfo.m_TexName);
        }
        else
        {
            //cerr << "read ctex: " << texInfo.m_CtexPath << endl;
            status;
            CIntermediateFile ctexFile(&status, texInfo.m_CtexPath);
            if (status)
            {
                status = CreateImageFileFromCtex(&ctexFormat, &mipLevel, opt.m_pTextureEncoder,
                    ddsPath, ctexFile, texInfo.m_TexName);
            }
        }
        if (!status)
        {
            break;
        }

        //-----------------------------------------------------------------------------
        // 3D テクスチャーコンバーターのオプションを追加します。
        const std::string ftxFormat = GetFtxTextureFormat(ctexFormat, opt.m_TexLinearFlag);
        std::string cvtrOpt = "\"" + ddsPath + "\" -o=\"" + ftxPath + "\"";
        if (!opt.m_ProjectRootPath.empty())
        {
            cvtrOpt += " --project-root=\"" + opt.m_ProjectRootPath + "\"";
        }
        if (opt.m_DisablesFileInfo)
        {
            cvtrOpt += " --disable-file-info";
        }

        if (!texInfo.m_Hint.empty())
        {
            std::string hint = texInfo.m_Hint;
            if (hint == CTexInfo::HintAuto)
            {
                hint = (ctexFormat == "Hilo8") ? RImage::HINT_NORMAL : RImage::HINT_ALBEDO;
            }
            cvtrOpt += " --hint=\"" + hint + "\"";
        }

        const bool isLinear = (ctexFormat != "Hilo8" && opt.m_TexLinearFlag != C2NnOption::LinearFlag_None);
        if (isLinear)
        {
            cvtrOpt += " --linear=\"" + GetOptStringFromLinearFlag(opt.m_TexLinearFlag) + "\"";
        }

        cvtrOpt += " --format=\"" + ftxFormat + "\"";

        if (mipLevel >= 2 && (opt.m_ForcesMaxMip || isLinear))
        {
            // ctex ファイルがミップマップを使用する場合に、
            // ctex ファイルのミップマップ画像を無視して NintendoSDK での最大レベルまで強制的に作成します。
            cvtrOpt += " --mip-level=\"" + RGetNumberString(RImage::MipCountMax) + "\"";
            if (!opt.m_MipGenFilter.empty())
            {
                cvtrOpt += " --mip-gen-filter=\"" + opt.m_MipGenFilter + "\"";
            }
        }

        if (ctexFormat == "A4" || ctexFormat == "A8")
        {
            cvtrOpt += " --comp-sel=\"000r\"";
        }
        cvtrOpt += " --weighted-compress=\"" + std::string(RBoolStr(false)) + "\"";

        if (!opt.m_CommentText.empty())
        {
            cvtrOpt += " --comment=\"" + opt.m_CommentText + "\"";
        }

        cvtrOss << cvtrOpt << R_ENDL;
        //cerr << "texcvtr: " << cvtrOpt << endl;
    }

    //-----------------------------------------------------------------------------
    // 3D テクスチャーコンバーターで DDS ファイルを ftx ファイルに変換します。
    const std::string cvtrStr = cvtrOss.str();
    if (status && !cvtrStr.empty())
    {
        const std::string texCvtrPath = opt.m_3dToolsPath + "3dTextureConverter.exe";
        status = Do3dTextureConverter(texCvtrPath, cvtrStr, texFolderPath);
    }

    //-----------------------------------------------------------------------------
    // DDS ファイルを削除します。
    for (size_t ddsIdx = 0; ddsIdx < ddsPaths.size(); ++ddsIdx)
    {
        const std::string& ddsPath = ddsPaths[ddsIdx];
        if (RFileExists(ddsPath))
        {
            ::DeleteFileA(ddsPath.c_str());
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 配列要素から名前で要素を検索します。
//!
//! @param[in] arrayElem 配列要素です。
//! @param[in] name 検索する名前です。空文字なら名前をチェックしません。
//! @param[in] groupName 検索する要素のグループ名です。空文字ならグループ名をチェックしません。
//! @param[in] filePath 配列要素が含まれるファイルのパスです。
//!
//! @return 要素を返します。見つからなければ nullptr を返します。
//-----------------------------------------------------------------------------
const RXMLElement* FindElementByName(
    const RXMLElement* arrayElem,
    const std::string& name,
    const std::string& groupName,
    const std::string& filePath
)
{
    if (arrayElem != nullptr)
    {
        for (size_t valueIdx = 0; valueIdx < arrayElem->nodes.size(); ++valueIdx)
        {
            //-----------------------------------------------------------------------------
            // グループ名をチェックします。
            const RXMLElement* valueElem = &arrayElem->nodes[valueIdx];
            if (!groupName.empty())
            {
                if (groupName == "SkeletalAnimation")
                {
                    if (valueElem->name != "SkeletalAnimationData")
                    {
                        continue;
                    }
                }
                else if (groupName == "MaterialAnimation")
                {
                    if (valueElem->name != "MaterialAnimationData")
                    {
                        continue;
                    }
                }
                else
                {
                    const std::string elemGroupName = valueElem->GetAttribute("TargetAnimationGroupName", "", false);
                    if (elemGroupName != groupName)
                    {
                        continue;
                    }
                }
            }

            //-----------------------------------------------------------------------------
            // 名前をチェックします。
            std::string elemName = valueElem->GetAttribute("Name");
            if (elemName.empty())
            {
                elemName = RGetNoExtensionFilePath(RGetFileNameFromFilePath(filePath));
            }
            if (name.empty() || elemName == name)
            {
                return valueElem;
            }
        }
    }
    return nullptr;
}

//-----------------------------------------------------------------------------
//! @brief fmd ファイルに変換します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in,out] pOutputFtxPaths 出力済みの ftx ファイルのパス配列へのポインタです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] fmdPath fmd ファイルのパスです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertToFmd(
    CExpOpt* pExpOpt,
    RStringArray* pOutputFtxPaths,
    const CIntermediateFile& inputFile,
    const std::string& fmdPath,
    const C2NnOption& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // モデル要素を取得します。
    const RXMLElement* modelsElem = inputFile.m_GraphicsElem->FindElement("Models", false);
    const RXMLElement* modelElem = FindElementByName(modelsElem, opt.m_ElementName, "", inputFile.m_Path);
    if (modelElem == nullptr)
    {
        return RStatus(RStatus::FAILURE, "Model element cannot be found: " + opt.m_ElementName); // RShowError
    }

    //-----------------------------------------------------------------------------
    // モデルのエディットデータを解析します。
    const RXMLElement* editDataElem = modelElem->FindElement("EditData", false);
    if (editDataElem != nullptr)
    {
        const RXMLElement* expOptElem = editDataElem->FindElement("ModelDccToolExportOption", false);
        if (expOptElem != nullptr)
        {
            pExpOpt->m_StartFrame = atol(expOptElem->GetAttribute("ExportStartFrame").c_str());
            pExpOpt->m_Magnify = atof(expOptElem->GetAttribute("Magnify").c_str());
            if (pExpOpt->m_Magnify <= 0.0) // NintendoSDK では 0 より大きい値である必要があるので修正
            {
                pExpOpt->m_Magnify = 1.0;
            }
            // expOptElem->GetAttribute("AdjustSkinning");
            // expOptElem->GetAttribute("MeshVisibilityMode");
        }
    }

    //-----------------------------------------------------------------------------
    // マテリアルを解析します。
    const std::string srcRootFolder = RGetFolderFromFilePath(inputFile.m_Path) + "/";
    CModel cmodel;
    const RXMLElement* fileMatsElem = inputFile.m_GraphicsElem->FindElement("Materials", false);
    const RXMLElement* matsElem = modelElem->FindElement("Materials", false);
    if (matsElem != nullptr)
    {
        for (size_t matIdx = 0; matIdx < matsElem->nodes.size(); ++matIdx)
        {
            const RXMLElement* matElem = GetRealMaterialElem(fileMatsElem, &matsElem->nodes[matIdx]);
            if (matElem == nullptr)
            {
                return RStatus(RStatus::FAILURE, "Material cannot be found: " + matsElem->nodes[matIdx].text); // RShowError
            }
            status = ParseMaterial(&cmodel.m_Materials, &cmodel.m_TexInfos,
                matElem, inputFile, srcRootFolder, pExpOpt->m_OutFtxFlag);
            RCheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // ノードビジビリティを解析します。
    CNodeVisArray nodeViss;
    const RXMLElement* nodeVissElem = modelElem->FindElement("MeshNodeVisibilities", false);
    if (nodeVissElem != nullptr)
    {
        for (size_t nodeVisIdx = 0; nodeVisIdx < nodeVissElem->nodes.size(); ++nodeVisIdx)
        {
            ParseNodeVis(nodeViss, &nodeVissElem->nodes[nodeVisIdx]);
        }
    }

    //-----------------------------------------------------------------------------
    // スケルトンを解析します。
    const RXMLElement* skeletonElem = modelElem->FindElement("Skeleton", false);
    status = ParseSkeleton(&cmodel, skeletonElem, modelElem, nodeViss, opt.m_ConversionMagnify);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // メッシュを解析します。
    CMeshArray meshes;
    const RXMLElement* meshesElem = modelElem->FindElement("Meshes", false);
    if (meshesElem != nullptr)
    {
        for (size_t meshIdx = 0; meshIdx < meshesElem->nodes.size(); ++meshIdx)
        {
            ParseMesh(meshes, &meshesElem->nodes[meshIdx]);
        }
    }

    //-----------------------------------------------------------------------------
    // シェイプを解析します。
    const RXMLElement* shapesElem = modelElem->FindElement("Shapes", false);
    if (shapesElem != nullptr)
    {
        for (size_t shapeIdx = 0; shapeIdx < shapesElem->nodes.size(); ++shapeIdx)
        {
            const int meshIdx = FindMeshByShape(meshes, static_cast<int>(shapeIdx));
            if (meshIdx != -1)
            {
                status = ParseShape(&cmodel, &shapesElem->nodes[shapeIdx], meshes[meshIdx],
                    nodeViss, opt.m_ConversionMagnify, opt.m_VtxColLinearFlag);
                RCheckStatus(status);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // スキニングに使用するボーンの行列パレットインデックスを設定します。
    SetBoneSkinningMtxIdx(&cmodel);

    //-----------------------------------------------------------------------------
    // モデル用ユーザーデータ配列を取得します。
    ParseUserDatas(&cmodel.m_UserDatas, modelElem->FindElement("UserData", false));

    //-----------------------------------------------------------------------------
    // 入力フォルダと Textures フォルダに存在する全 ctex ファイルの情報を取得します。
    if (opt.m_ConvertsAllTex)
    {
        AddTexInfosFromFolder(&cmodel.m_TexInfos, srcRootFolder);
        AddTexInfosFromFolder(&cmodel.m_TexInfos, srcRootFolder + Nw4cTexFolderName + "/");
    }

    //-----------------------------------------------------------------------------
    // テクスチャ中間ファイル群を出力します。
    if (pExpOpt->m_OutFtxFlag && !cmodel.m_TexInfos.empty())
    {
        const std::string texFolderPath = RGetFolderFromFilePath(fmdPath) + "/" +
            RExpOpt::TexFolderName + "/";
        status = OutputTextureFiles(pOutputFtxPaths, cmodel.m_TexInfos, texFolderPath, opt, &inputFile);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // fmd ファイルを出力します。
    return OutputOneFile(&cmodel, fmdPath, RExpOpt::FMD, *pExpOpt, opt);
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルに変換します。
//!
//! @param[in,out] pOutputFtxPaths 出力済みの ftx ファイルのパス配列へのポインタです。
//! @param[in] ctexPath ctex ファイルのパスです。
//! @param[in] ftxPath ftx ファイルのパスです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertToFtx(
    RStringArray* pOutputFtxPaths,
    const std::string& ctexPath,
    const std::string& ftxPath,
    const C2NnOption& opt
)
{
    //-----------------------------------------------------------------------------
    // テクスチャ情報配列を作成します。
    const std::string texName = (!opt.m_ElementName.empty()) ?
        opt.m_ElementName : RGetNoExtensionFilePath(RGetFileNameFromFilePath(ctexPath));
    const std::string outName = RGetNoExtensionFilePath(RGetFileNameFromFilePath(ftxPath));
    CTexInfoArray texInfos;
    texInfos.push_back(CTexInfo(texName, ctexPath, CTexInfo::HintAuto, outName, opt.m_IsBinaryFormat));

    //-----------------------------------------------------------------------------
    // テクスチャ中間ファイルを出力します。
    const std::string texFolderPath = RGetFolderFromFilePath(ftxPath) + "/";
    return OutputTextureFiles(pOutputFtxPaths, texInfos, texFolderPath, opt, nullptr);
}

//-----------------------------------------------------------------------------
//! @brief アニメーション要素の属性とユーザーデータを解析します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in,out] pUserDatas ユーザーデータ配列へのポインタです。
//! @param[in] animElem アニメーション要素です。nullptr なら何もしません。
//-----------------------------------------------------------------------------
void ParseAnimElem(
    CExpOpt* pExpOpt,
    RUserDataArray* pUserDatas,
    const RXMLElement* animElem
)
{
    if (animElem != nullptr)
    {
        //-----------------------------------------------------------------------------
        // 属性を取得します。
        pExpOpt->m_OutFrameCount = atol(animElem->GetAttribute("FrameSize").c_str());
        pExpOpt->m_LoopAnim = (animElem->GetAttribute("LoopMode") == "Loop"); // or OneTime
        pExpOpt->m_StartFrame = 0;
        pExpOpt->m_EndFrame = pExpOpt->m_OutFrameCount;

        //-----------------------------------------------------------------------------
        // アニメーション用ユーザーデータ配列を取得します。
        ParseUserDatas(pUserDatas, animElem->FindElement("UserData", false));
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブデータを自動作成する際のフレームの精度を取得します。
//-----------------------------------------------------------------------------
int GetFramePrecision(const std::string& framePrecision)
{
    return
        (framePrecision == "_1_2" ) ?  2 :
        (framePrecision == "_1_5" ) ?  5 :
        (framePrecision == "_1_10") ? 10 :
        1; // _1_1
}

//-----------------------------------------------------------------------------
//! @brief エディットデータから 3DEditor のバインドターゲット名を取得します。
//!
//! @param[in] animElem アニメーション要素です。
//!
//! @brief バインドターゲット名を返します。データがなければ空文字を返します。
//-----------------------------------------------------------------------------
std::string Get3DEditorBindTargetName(const RXMLElement* animElem)
{
    const RXMLElement* editDataElem = (animElem != nullptr) ?
        animElem->FindElement("EditData", false) : nullptr;
    if (editDataElem != nullptr)
    {
        for (size_t dataIdx = 0; dataIdx < editDataElem->nodes.size(); ++dataIdx)
        {
            const RXMLElement* metaDataElem = &editDataElem->nodes[dataIdx];
            const RXMLElement* keyElem = metaDataElem->FindElement("Key", false);
            if (keyElem != nullptr && keyElem->text == "3DEditor_BindTargetName")
            {
                const RXMLElement* valueElem = metaDataElem->FindElement("Values/StringSet", false);
                if (valueElem != nullptr)
                {
                    return valueElem->text;
                }
            }
        }
    }
    return "";
}

//-----------------------------------------------------------------------------
//! @brief スケルタルアニメーションのエディットデータを解析します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] animElem アニメーション要素です。nullptr なら何もしません。
//-----------------------------------------------------------------------------
void ParseSkeletalAnimEditData(CExpOpt* pExpOpt, const RXMLElement* animElem)
{
    const RXMLElement* editDataElem = (animElem != nullptr) ?
        animElem->FindElement("EditData", false) : nullptr;
    if (editDataElem != nullptr)
    {
        const RXMLElement* metaDataElem = editDataElem->FindElement("GenericMetaData", false);
        if (metaDataElem != nullptr)
        {
            const RXMLElement* expOptElem = metaDataElem->FindElement("SkeletalAnimationDccToolExportOption", false);
            if (expOptElem != nullptr)
            {
                pExpOpt->m_StartFrame = atol(expOptElem->GetAttribute("StartFrame").c_str());
                pExpOpt->m_EndFrame = atol(expOptElem->GetAttribute("EndFrame").c_str());
                pExpOpt->m_Magnify = atof(expOptElem->GetAttribute("Magnify").c_str());
                const RXMLElement* bakeOptElem = expOptElem->FindElement("SkeletalAnimationBakeOption", false);
                if (bakeOptElem != nullptr)
                {
                    pExpOpt->m_BakeAllAnim = (bakeOptElem->GetAttribute("IsBakeAllEnabled") == "true");
                    pExpOpt->m_FramePrecision = GetFramePrecision(bakeOptElem->GetAttribute("FramePrecision"));
                    pExpOpt->m_TolS = static_cast<float>(atof(bakeOptElem->GetAttribute("ScaleTolerance").c_str()));
                    pExpOpt->m_TolR = static_cast<float>(atof(bakeOptElem->GetAttribute("RotateTolerance").c_str()) * R_M_RAD_TO_DEG);
                    pExpOpt->m_TolT = static_cast<float>(atof(bakeOptElem->GetAttribute("TranslateTolerance").c_str()));
                }
            }
        }
    }
    pExpOpt->UpdateFrameCount();
}

//-----------------------------------------------------------------------------
//! @brief マテリアルアニメーションのエディットデータを解析します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] animElem アニメーション要素です。nullptr なら何もしません。
//-----------------------------------------------------------------------------
void ParseMaterialAnimEditData(CExpOpt* pExpOpt, const RXMLElement* animElem)
{
    const RXMLElement* editDataElem = (animElem != nullptr) ?
        animElem->FindElement("EditData", false) : nullptr;
    if (editDataElem != nullptr)
    {
        const RXMLElement* metaDataElem = editDataElem->FindElement("GenericMetaData", false);
        if (metaDataElem != nullptr)
        {
            const RXMLElement* expOptElem = metaDataElem->FindElement("MaterialAnimationDccToolExportOption", false);
            if (expOptElem != nullptr)
            {
                pExpOpt->m_StartFrame = atol(expOptElem->GetAttribute("StartFrame").c_str());
                pExpOpt->m_EndFrame = atol(expOptElem->GetAttribute("EndFrame").c_str());
                pExpOpt->m_Magnify = atof(expOptElem->GetAttribute("Magnify").c_str());
                const RXMLElement* bakeOptElem = expOptElem->FindElement("MaterialAnimationBakeOption", false);
                if (bakeOptElem != nullptr)
                {
                    pExpOpt->m_BakeAllAnim = (bakeOptElem->GetAttribute("IsBakeAllEnabled") == "true");
                    pExpOpt->m_FramePrecision = GetFramePrecision(bakeOptElem->GetAttribute("FramePrecision"));
                    pExpOpt->m_TolTexS = static_cast<float>(atof(bakeOptElem->GetAttribute("TextureScaleTolerance").c_str()));
                    pExpOpt->m_TolTexR = static_cast<float>(atof(bakeOptElem->GetAttribute("TextureRotateTolerance").c_str()) * R_M_RAD_TO_DEG);
                    pExpOpt->m_TolTexT = static_cast<float>(atof(bakeOptElem->GetAttribute("TextureTranslateTolerance").c_str()));
                    pExpOpt->m_TolC    = static_cast<float>(atof(bakeOptElem->GetAttribute("ColorTolerance").c_str()));
                }
            }
        }
    }
    pExpOpt->UpdateFrameCount();
}

//-----------------------------------------------------------------------------
//! @brief サブフレーム数を取得します。
//-----------------------------------------------------------------------------
int GetSubFrameCount(const int outFrameCount, const CExpOpt& expOpt)
{
    // 最終フレームより後は対象外なので (m_FramePrecision - 1) を引きます。
    return (outFrameCount + 1) * expOpt.m_FramePrecision - (expOpt.m_FramePrecision - 1);
}

//-----------------------------------------------------------------------------
//! @brief 整数サブフレームから浮動小数点数フレームを取得します。
//-----------------------------------------------------------------------------
float GetFloatFrameFromSubFrame(const int subFrame, const void* pParam)
{
    const CExpOpt& expOpt = *reinterpret_cast<const CExpOpt*>(pParam);
    return static_cast<float>(subFrame) / expOpt.m_FramePrecision;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブの範囲外のフレームの評価モードを取得します。
//-----------------------------------------------------------------------------
RAnimCurve::Wrap GetAnimCurveWrap(const std::string& repeatMethod)
{
    return
        (repeatMethod == "Repeat"        ) ? RAnimCurve::REPEAT          :
        (repeatMethod == "RelativeRepeat") ? RAnimCurve::RELATIVE_REPEAT :
        (repeatMethod == "Mirror"        ) ? RAnimCurve::MIRROR          :
        RAnimCurve::CLAMP; // None
}

//-----------------------------------------------------------------------------
//! @brief アニメーションキーの値が一定なら true を返します。
//!
//! @param[in] curve アニメーションカーブです。
//-----------------------------------------------------------------------------
bool IsAllKeyValueSame(const RAnimCurve& curve)
{
    const int keyCount = static_cast<int>(curve.m_Keys.size());
    if (keyCount >= 2)
    {
        const float firstValue = curve.m_Keys[0].m_Value;
        for (int keyIdx = 1; keyIdx < keyCount; ++keyIdx)
        {
            if (!RIsSame(curve.m_Keys[keyIdx].m_Value, firstValue))
            {
                return false;
            }
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションキーのスロープがすべて 0 なら true を返します。
//!
//! @param[in] curve アニメーションカーブです。
//-----------------------------------------------------------------------------
bool IsAllSlopeZero(const RAnimCurve& curve)
{
    const int keyCount = static_cast<int>(curve.m_Keys.size());
    for (int keyIdx = 0; keyIdx < keyCount; ++keyIdx)
    {
        const RAnimKey& key = curve.m_Keys[keyIdx];
        if (keyIdx >= 1           && !RIsSame(key.m_InSlope, 0.0f))
        {
            return false;
        }
        if (keyIdx < keyCount - 1 && !RIsSame(key.m_OtSlope, 0.0f))
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 範囲外のフレームの評価モードを考慮してアニメーションカーブを評価します。
//!        RAnimCurve::Evaluate は範囲外のフレームの評価モードがクランプ以外の場合に
//!        非対応なのでこの関数で調整したフレームを評価します。
//!
//! @param[in] curve アニメーションカーブです。
//! @param[in] frame 評価するフレームです。
//-----------------------------------------------------------------------------
float EvaluateAnimCurve(const RAnimCurve& curve, const float frame)
{
    //-----------------------------------------------------------------------------
    // 範囲内の場合
    const RAnimKeyArray& keys = curve.m_Keys;
    const int keyCount = static_cast<int>(keys.size());
    if (keyCount == 0)
    {
        return (!curve.m_FullValues.empty()) ? curve.m_FullValues[0] : 0.0f;
    }
    const float frameS = keys[0           ].m_Frame;
    const float frameE = keys[keyCount - 1].m_Frame;
    if (frameS <= frame && frame <= frameE)
    {
        return curve.Evaluate(frame);
    }

    //-----------------------------------------------------------------------------
    // 範囲外でクランプの場合
    const float valueS = keys[0           ].m_Value;
    const float valueE = keys[keyCount - 1].m_Value;
    if (frame < frameS && curve.m_PreWrap == RAnimCurve::CLAMP)
    {
        return valueS;
    }
    if (frame > frameE && curve.m_PostWrap == RAnimCurve::CLAMP)
    {
        return valueE;
    }

    //-----------------------------------------------------------------------------
    // 範囲外でリピート／ミラーの場合
    const RAnimCurve::Wrap wrap = (frame < frameS) ? curve.m_PreWrap : curve.m_PostWrap;
    const float duration = (frameS < frameE) ? frameE - frameS : 1.0f;
    const int repeatIdx = static_cast<int>(std::floor((frame - frameS) / duration));
    float adjFrame = frame - duration * repeatIdx;
    if (wrap == RAnimCurve::MIRROR && (repeatIdx & 1))
    {
        adjFrame = frameE - (adjFrame - frameS);
    }
    const float ofs = (wrap == RAnimCurve::RELATIVE_REPEAT) ?
        (valueE - valueS) * repeatIdx : 0.0f;
    return curve.Evaluate(adjFrame) + ofs;
}

//-----------------------------------------------------------------------------
//! @brief アニメーションセグメントを解析します。
//!
//! @param[in,out] pCurve アニメーションカーブへのポインタです。
//! @param[in] curveElem アニメーションセグメント要素です。
//! @param[in] magnify 値に掛ける倍率です。角度アニメーションの場合、この値は無視されます。
//-----------------------------------------------------------------------------
void ParseAnimSegment(
    RAnimCurve* pCurve,
    const RXMLElement* segmentElem,
    const float magnify
)
{
    //-----------------------------------------------------------------------------
    // 接線タイプを取得します。
    RAnimKey::TangentType tanType;
    if (segmentElem->name.find("Hermite") == 0)
    {
        tanType = RAnimKey::HERMITE;
    }
    else if (segmentElem->name.find("Linear") == 0)
    {
        tanType = RAnimKey::LINEAR;
    }
    else
    {
        tanType = RAnimKey::STEP;
    }

    //-----------------------------------------------------------------------------
    // 各キーを解析します。
    const float valueScale = (pCurve->m_AngleFlag) ?
        static_cast<float>(R_M_RAD_TO_DEG) : magnify;
    const RXMLElement* keysElem = segmentElem->FindElement("Keys");
    for (size_t keyIdx = 0; keyIdx < keysElem->nodes.size(); ++keyIdx)
    {
        //-----------------------------------------------------------------------------
        // キーのフレーム／値／スロープを取得します。
        const RXMLElement* keyElem = &keysElem->nodes[keyIdx];
        const float frame = static_cast<float>(atof(keyElem->GetAttribute("Frame").c_str()));
        const float value = static_cast<float>(atof(keyElem->GetAttribute("Value").c_str())) * valueScale;
        float inSlope = 0.0f;
        float otSlope = 0.0f;
        if (tanType == RAnimKey::HERMITE)
        {
            inSlope = static_cast<float>(atof(keyElem->GetAttribute("InSlope" ).c_str())) * valueScale;
            otSlope = static_cast<float>(atof(keyElem->GetAttribute("OutSlope").c_str())) * valueScale;
        }

        //-----------------------------------------------------------------------------
        // 複数のセグメントに分かれている場合、
        // 現在のセグメントの最初のキーのフレームと値が
        // 1 つ前のセグメントの最後のキーと同じならキーを追加しません。
        if (keyIdx == 0 && !pCurve->m_Keys.empty())
        {
            RAnimKey& prevKey = pCurve->m_Keys.back();
            if (prevKey.m_Frame == frame &&
                prevKey.m_Value == value)
            {
                prevKey.m_Type = tanType;
                prevKey.m_OtSlope = otSlope;
                continue;
            }
        }

        //-----------------------------------------------------------------------------
        // リニア補間のセグメントとエルミート補間のセグメントが混在している場合のために
        // リニア補間の場合のスロープを計算して設定します。
        if (tanType == RAnimKey::LINEAR && keyIdx > 0)
        {
            RAnimKey& prevKey = pCurve->m_Keys.back();
            const float frameD = frame - prevKey.m_Frame;
            if (frameD != 0.0f)
            {
                prevKey.m_OtSlope = inSlope = (value - prevKey.m_Value) / frameD;
            }
        }

        //-----------------------------------------------------------------------------
        // キーを追加します。
        pCurve->m_Keys.push_back(RAnimKey(frame, value, tanType, inSlope, otSlope));
    }
}

//-----------------------------------------------------------------------------
//! @brief アニメーションカーブを解析します。
//!
//! @param[in,out] pCurve アニメーションカーブへのポインタです。
//! @param[in] curveElem アニメーションカーブ要素です。
//! @param[in] magnify 値に掛ける倍率です。角度アニメーションの場合、この値は無視されます。
//-----------------------------------------------------------------------------
void ParseAnimCurve(
    RAnimCurve* pCurve,
    const RXMLElement* curveElem,
    const float magnify
)
{
    //-----------------------------------------------------------------------------
    // アニメーションカーブの属性を取得します。
    pCurve->m_PreWrap  = GetAnimCurveWrap(curveElem->GetAttribute("PreRepeatMethod" ));
    pCurve->m_PostWrap = GetAnimCurveWrap(curveElem->GetAttribute("PostRepeatMethod"));

    //-----------------------------------------------------------------------------
    // アニメーションセグメントを解析します。
    const RXMLElement* segmentsElem = curveElem->FindElement("Segments");
    for (size_t segmentIdx = 0; segmentIdx < segmentsElem->nodes.size(); ++segmentIdx)
    {
        ParseAnimSegment(pCurve, &segmentsElem->nodes[segmentIdx], magnify);
    }

    //-----------------------------------------------------------------------------
    // 一定フラグを更新します。
    pCurve->m_ConstantFlag = (pCurve->m_Keys.size() < 2);
    if (!pCurve->m_ConstantFlag && IsAllKeyValueSame(*pCurve))
    {
        if (pCurve->GetTangentType() == RAnimKey::HERMITE)
        {
            pCurve->m_ConstantFlag = IsAllSlopeZero(*pCurve);
        }
        else
        {
            pCurve->m_ConstantFlag = true;
        }
    }

    //-----------------------------------------------------------------------------
    // アニメーションの初期値（0 フレームの値）を設定します。
    if (!pCurve->m_Keys.empty())
    {
        const float baseValue = EvaluateAnimCurve(*pCurve, 0.0f);
        if (pCurve->m_FullValues.empty())
        {
            pCurve->m_FullValues.push_back(baseValue);
        }
        else
        {
            pCurve->m_FullValues[0] = baseValue;
        }
        pCurve->m_UseFlag = true;
    }
}

//-----------------------------------------------------------------------------
//! @brief ベクトルアニメーションを解析します。
//!
//! @param[in,out] curves XYZ 成分に対応したアニメーションカーブの配列です。
//! @param[in,out] pBinarizes アニメーションのバイナリ出力フラグへのポインタです。
//! @param[in] vectorAnimElem ベクトルアニメーション要素です。
//! @param[in] magnify 値に掛ける倍率です。角度アニメーションの場合、この値は無視されます。
//-----------------------------------------------------------------------------
void ParseVectorAnim(
    RAnimCurve* curves,
    bool* pBinarizes,
    const RXMLElement* vectorAnimElem,
    const float magnify
)
{
    if (vectorAnimElem != nullptr)
    {
        for (size_t curveIdx = 0; curveIdx < vectorAnimElem->nodes.size(); ++curveIdx)
        {
            const RXMLElement* curveElem = &vectorAnimElem->nodes[curveIdx];
            const int xyzIdx = curveElem->name[0] - 'X';
            if (0 <= xyzIdx && xyzIdx < R_XYZ_COUNT)
            {
                ParseAnimCurve(&curves[xyzIdx], curveElem, magnify);
            }
        }
        if (pBinarizes != nullptr)
        {
            *pBinarizes = true;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief RGBA カラーアニメーションを解析します。
//!
//! @param[in,out] rgbCurves RGB 成分に対応したアニメーションカーブの配列です。
//! @param[in,out] pAlphaCurve A 成分に対応したアニメーションカーブへのポインタです。
//!                            nullptr なら A 成分のアニメーションを取得しません。
//! @param[in] rgbaColorAnimElem RGBA カラーアニメーション要素です。
//-----------------------------------------------------------------------------
void ParseRgbaColorAnim(
    RAnimCurve* rgbCurves,
    RAnimCurve* pAlphaCurve,
    const RXMLElement* rgbaColorAnimElem
)
{
    if (rgbaColorAnimElem != nullptr)
    {
        for (size_t compIdx = 0; compIdx < rgbaColorAnimElem->nodes.size(); ++compIdx)
        {
            const RXMLElement* colorAnimElem = &rgbaColorAnimElem->nodes[compIdx];
            const char compName = colorAnimElem->name[0];
            const int rgbaIdx =
                (compName == 'R') ? 0 :
                (compName == 'G') ? 1 :
                (compName == 'B') ? 2 :
                (compName == 'A') ? 3 :
                -1;
            if (0 <= rgbaIdx && rgbaIdx <= 2)
            {
                ParseAnimCurve(&rgbCurves[rgbaIdx], colorAnimElem, 1.0f);
            }
            else if (rgbaIdx == 3 && pAlphaCurve != nullptr)
            {
                ParseAnimCurve(pAlphaCurve, colorAnimElem, 1.0f);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ブールアニメーションカーブを解析します。
//!
//! @param[in,out] pCurve アニメーションカーブへのポインタです。
//! @param[in] curveElem アニメーションカーブ要素です。
//-----------------------------------------------------------------------------
void ParseBoolAnimCurve(RAnimCurve* pCurve, const RXMLElement* curveElem)
{
    //-----------------------------------------------------------------------------
    // アニメーションカーブの属性を取得します。
    pCurve->m_PreWrap  = GetAnimCurveWrap(curveElem->GetAttribute("PreRepeatMethod" ));
    pCurve->m_PostWrap = GetAnimCurveWrap(curveElem->GetAttribute("PostRepeatMethod"));
    const bool defaultValue = (curveElem->GetAttribute("DefaultValue") == "true");
    pCurve->m_FullValues.push_back((defaultValue) ? 1.0f : 0.0f);

    //-----------------------------------------------------------------------------
    // キーを取得します。
    const RXMLElement* keysElem = curveElem->FindElement("Keys");
    for (size_t keyIdx = 0; keyIdx < keysElem->nodes.size(); ++keyIdx)
    {
        const RXMLElement* keyElem = &keysElem->nodes[keyIdx];
        const float frame = static_cast<float>(atof(keyElem->GetAttribute("Frame").c_str()));
        const float value = (keyElem->GetAttribute("Value") == "true") ? 1.0f : 0.0f;
        pCurve->m_Keys.push_back(RAnimKey(frame, value, RAnimKey::STEP));
    }

    //-----------------------------------------------------------------------------
    // 一定フラグを更新します。
    pCurve->m_ConstantFlag = (pCurve->m_Keys.size() < 2);
}

//-----------------------------------------------------------------------------
//! @brief ボーンアニメーションのボーン名を取得します。
//!
//! @param[in] memberAnimElem メンバーアニメーションデータ要素です。
//!
//! @return ボーン名を返します。アニメーションパスが不正なら空文字を返します。
//-----------------------------------------------------------------------------
std::string GetBoneAnimBoneName(const RXMLElement* memberAnimElem)
{
    std::string boneName;
    const std::string animPath = memberAnimElem->FindElement("Path")->text;
    const std::string bonesTop = "Bones[";
    const size_t boneIdxsTop = animPath.find(bonesTop);
    if (boneIdxsTop != std::string::npos)
    {
        boneName = GetDoubleQuatedStr(animPath.substr(boneIdxsTop + bonesTop.size()));
        boneName = RAdjustElementNameString(boneName);
    }
    return boneName;
}

//-----------------------------------------------------------------------------
//! @brief ボーンアニメーションからボーン配列を取得します。
//!
//! @param[in,out] pBones ボーン配列へのポインタです。
//! @param[in] memberAnimsElem メンバーアニメーションデータ配列要素です。
//-----------------------------------------------------------------------------
void GetBonesFromBoneAnim(CBoneArray* pBones, const RXMLElement* memberAnimsElem)
{
    for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
    {
        //-----------------------------------------------------------------------------
        // アニメーションパスからボーン名を取得します。
        const RXMLElement* memberAnimElem = &memberAnimsElem->nodes[memberIdx];
        const std::string boneName = GetBoneAnimBoneName(memberAnimElem);
        if (boneName.empty())
        {
            continue;
        }

        //-----------------------------------------------------------------------------
        // ボーンを追加します。
        pBones->push_back(CBone());
        CBone& bone = pBones->back();
        bone.m_Name = boneName;
    }
}

//-----------------------------------------------------------------------------
//! @brief ボーンアニメーションを解析します。
//!
//! @param[in,out] pBones ボーン配列へのポインタです。
//! @param[in] memberAnimElem メンバーアニメーションデータ要素です。
//! @param[in] conversionMagnify 移動値に掛ける倍率です。
//-----------------------------------------------------------------------------
void ParseBoneAnim(
    CBoneArray* pBones,
    const RXMLElement* memberAnimElem,
    const float conversionMagnify
)
{
    //-----------------------------------------------------------------------------
    // アニメーションパスからボーン名を取得します。
    const std::string boneName = GetBoneAnimBoneName(memberAnimElem);
    if (boneName.empty())
    {
        return;
    }
    const int boneIdx = FindObjectByName(*pBones, boneName);
    if (boneIdx == -1)
    {
        return;
    }
    CBone& bone = (*pBones)[boneIdx];

    //-----------------------------------------------------------------------------
    // トランスフォームアニメーションを解析します。
    const RXMLElement* xformElem = memberAnimElem->FindElement("TransformAnimation");

    const RXMLElement* scaleElem = xformElem->FindElement("ScaleVector3Animation", false);
    ParseVectorAnim(&bone.m_Anims[RBone::SCALE_X], &bone.m_BinarizesScale, scaleElem, 1.0f);
    const RXMLElement* rotateElem = xformElem->FindElement("RotateVector3Animation", false);
    ParseVectorAnim(&bone.m_Anims[RBone::ROTATE_X], &bone.m_BinarizesRotate, rotateElem, 1.0f);
    const RXMLElement* translateElem = xformElem->FindElement("TranslateVector3Animation", false);
    ParseVectorAnim(&bone.m_Anims[RBone::TRANSLATE_X], &bone.m_BinarizesTranslate, translateElem, conversionMagnify);
}

//-----------------------------------------------------------------------------
//! @brief fsk ファイルに変換します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] fskPath fsk ファイルのパスです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertToFsk(
    CExpOpt* pExpOpt,
    const CIntermediateFile& inputFile,
    const std::string& fskPath,
    const C2NnOption& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // スケルタルアニメーション要素を取得します。
    // cskla ファイルはアニメーションがなければ <Animations> 要素が存在しませんが、
    // fsk ファイルは <skeletal_anim> 要素が常に存在する必要があります。
    const RXMLElement* animsElem = inputFile.m_GraphicsElem->FindElement("Animations", false);
    const RXMLElement* animElem = FindElementByName(animsElem, opt.m_ElementName, "SkeletalAnimation", inputFile.m_Path);
    if (animElem == nullptr)
    {
        if (!opt.m_ElementName.empty())
        {
            return RStatus(RStatus::FAILURE, "Skeletal animation element cannot be found: " + opt.m_ElementName); // RShowError
        }
    }
    CModel cmodel;
    ParseAnimElem(pExpOpt, &cmodel.m_AnimUserDatas, animElem);
    ParseSkeletalAnimEditData(pExpOpt, animElem);

    //-----------------------------------------------------------------------------
    // スケルトンを解析します。
    // fsk ファイルでは <skeletal_anim> 要素の子に <bone_anim_array> 要素が必要なので、
    // アニメーション要素がない場合はダミーのボーン 1 つのみの <bone_anim_array> 要素を出力します。
    // fsk ファイルに <bone_anim_array> 要素がないと 3DEditor で読み込みエラーとなります。
    CNodeVisArray nodeViss;
    const RXMLElement* memberAnimsElem = (animElem != nullptr) ?
        animElem->FindElement("MemberAnimationDataSet") : nullptr;
    const RXMLElement* skeletonElem = (animElem != nullptr) ?
        animElem->FindElement("Skeleton", false) : nullptr;
    if (skeletonElem == nullptr && memberAnimsElem != nullptr && !memberAnimsElem->nodes.empty())
    {
        // 古い中間ファイルで <SkeletalAnimationData> の子に <Skeleton> がない場合は、
        // <MemberAnimationDataSet> からボーン配列を取得します。
        GetBonesFromBoneAnim(&cmodel.m_Bones, memberAnimsElem);
    }
    else
    {
        status = ParseSkeleton(&cmodel, skeletonElem, nullptr, nodeViss, opt.m_ConversionMagnify);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // ボーンのスケルタルアニメーションを初期化します。
    for (size_t boneIdx = 0; boneIdx < cmodel.m_Bones.size(); ++boneIdx)
    {
        cmodel.m_Bones[boneIdx].InitSkeletalAnim(*pExpOpt);
    }

    //-----------------------------------------------------------------------------
    // メンバーアニメーションデータを解析します。
    if (memberAnimsElem != nullptr)
    {
        for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
        {
            ParseBoneAnim(&cmodel.m_Bones, &memberAnimsElem->nodes[memberIdx], opt.m_ConversionMagnify);
        }
    }

    //-----------------------------------------------------------------------------
    // fsk ファイルを出力します。
    return OutputOneFile(&cmodel, fskPath, RExpOpt::FSK, *pExpOpt, opt);
}

//-----------------------------------------------------------------------------
//! @brief モデルアニメーションを解析します。
//!
//! @param[in,out] pBones ボーン配列へのポインタです。
//! @param[in] memberAnimElem メンバーアニメーションデータ要素です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseModelAnim(CBoneArray* pBones, const RXMLElement* memberAnimElem)
{
    //-----------------------------------------------------------------------------
    // アニメーションパスからボーン名を取得します。
    const std::string animPath = memberAnimElem->FindElement("Path")->text;
        // Meshes[0].IsVisible                      (Bind By Index)
        // MeshNodeVisibilities["pCone1"].IsVisible (Bind By Name )
    const std::string nodeVissTop = "MeshNodeVisibilities";
    const size_t nodeVisIdxs = animPath.find(nodeVissTop);
    if (nodeVisIdxs == std::string::npos)
    {
        // Bind By Index の場合はエラー（3DEditor も Bind By Index には非対応）
        return RStatus(RStatus::FAILURE, "Visibility animation binded by the index is not supported"); // RShowError
    }
    std::string boneName = GetDoubleQuatedStr(animPath.substr(nodeVisIdxs + nodeVissTop.size()));
    boneName = RAdjustElementNameString(boneName);

    //-----------------------------------------------------------------------------
    // ボーンを追加します。
    pBones->push_back(CBone());
    CBone& bone = pBones->back();
    bone.m_Name = boneName;
    bone.m_RigidBody = true; // リジッドボディで使用されるケースが多いと仮定

    //-----------------------------------------------------------------------------
    // ブールアニメーションを解析します。
    const RXMLElement* curveElem = memberAnimElem->FindElement("BoolAnimation/BoolCurve");
    ParseBoolAnimCurve(&bone.m_VisAnim, curveElem);

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief fvb ファイルに変換します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] fvbPath fvb ファイルのパスです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertToFvb(
    CExpOpt* pExpOpt,
    const CIntermediateFile& inputFile,
    const std::string& fvbPath,
    const C2NnOption& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // ビジビリティアニメーション要素を取得します。
    // cmdla ファイルはアニメーションがなければ <Animations> 要素が存在しませんが、
    // fvb ファイルは <bone_visibility_anim> 要素が常に存在する必要があります。
    const RXMLElement* animsElem = inputFile.m_GraphicsElem->FindElement("Animations", false);
    const RXMLElement* animElem = FindElementByName(animsElem, opt.m_ElementName, "VisibilityAnimation", inputFile.m_Path);
    if (animElem == nullptr)
    {
        if (!opt.m_ElementName.empty())
        {
            return RStatus(RStatus::FAILURE, "Visibility animation element cannot be found: " + opt.m_ElementName); // RShowError
        }
    }
    CModel cmodel;
    ParseAnimElem(pExpOpt, &cmodel.m_AnimUserDatas, animElem);
    pExpOpt->UpdateFrameCount();

    //-----------------------------------------------------------------------------
    // メンバーアニメーションデータを解析します。
    if (animElem != nullptr)
    {
        const RXMLElement* memberAnimsElem = animElem->FindElement("MemberAnimationDataSet");
        for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
        {
            status = ParseModelAnim(&cmodel.m_Bones, &memberAnimsElem->nodes[memberIdx]);
            RCheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // fvb ファイルを出力します。
    return OutputOneFile(&cmodel, fvbPath, RExpOpt::FVB, *pExpOpt, opt);
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのカラーアニメーションを解析します。
//!
//! @param[in,out] pMaterials マテリアル配列へのポインタです。
//! @param[in] memberAnimElem メンバーアニメーションデータ要素です。
//! @param[in] expOpt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseMatColorAnim(
    CMaterialArray* pMaterials,
    const RXMLElement* memberAnimElem,
    const CExpOpt& expOpt
)
{
    //-----------------------------------------------------------------------------
    // アニメーションパスからマテリアル名を取得します。
    const std::string animPath = memberAnimElem->FindElement("Path")->text;
        // Materials["lambert2"].MaterialColor.Diffuse
    const std::string colorTop = ".MaterialColor";
    const size_t colIdx = animPath.find(colorTop);
    if (colIdx == std::string::npos) // カラーアニメーションでない場合
    {
        return RStatus::SUCCESS;
    }
    std::string matName = GetDoubleQuatedStr(animPath.substr(0, colIdx));
    matName = RAdjustElementNameString(matName);

    //-----------------------------------------------------------------------------
    // アニメーション対象のカラーを取得します。
    const std::string paramName = RGetExtensionFromFilePath(animPath, false);
    int paramIdx = -1;
    if (paramName == "Diffuse")
    {
        paramIdx = RMaterial::DIFFUSE_R;
    }
    else if (paramName == "Ambient")
    {
        paramIdx = RMaterial::AMBIENT_R;
    }
    else if (paramName == "Emission")
    {
        paramIdx = RMaterial::EMISSION_R;
    }
    else if (paramName == "Specular0")
    {
        paramIdx = RMaterial::SPECULAR_R;
    }
    else if (paramName == "Specular1")
    {
        paramIdx = CMaterial::SPECULAR1_R;
    }
    else if (paramName.find("Constant") == 0)
    {
        const int constIdx = paramName[paramName.size() - 1] - '0';
        if (0 <= constIdx && constIdx < CMaterial::CONSTANT_COUNT)
        {
            paramIdx = CMaterial::CONSTANT0_R + constIdx * R_RGB_COUNT * 2;
        }
    }

    if (paramIdx == -1)
    {
        return RStatus::SUCCESS;
    }

    //-----------------------------------------------------------------------------
    // 新規マテリアルなら追加します。
    int matIdx = FindObjectByName(*pMaterials, matName);
    if (matIdx == -1)
    {
        matIdx = static_cast<int>(pMaterials->size());
        pMaterials->push_back(CMaterial());
        CMaterial& newMat = pMaterials->back();
        newMat.m_Name = matName;
        newMat.InitColorAnim(expOpt);
    }
    CMaterial& mat = (*pMaterials)[matIdx];

    //-----------------------------------------------------------------------------
    // RGBA カラーアニメーションを解析します。
    const RXMLElement* rgbaColorAnimElem = memberAnimElem->FindElement("RgbaColorAnimation");
    RAnimCurve* pAlphaCurve = nullptr;
    if      (paramIdx == RMaterial::DIFFUSE_R  ) pAlphaCurve = &mat.m_Anims[RMaterial::OPACITY_R];
    else if (paramIdx >= CMaterial::CONSTANT0_R) pAlphaCurve = &mat.m_Anims[paramIdx + 3];
        // NW4C の Diffuse の A 成分アニメーションは
        // NintendoSDK の opacity の RGB 成分アニメーションとして出力します。
    ParseRgbaColorAnim(&mat.m_Anims[paramIdx], pAlphaCurve, rgbaColorAnimElem);
    if (paramIdx == RMaterial::DIFFUSE_R)
    {
        mat.m_Anims[RMaterial::OPACITY_G] =
        mat.m_Anims[RMaterial::OPACITY_B] =
            mat.m_Anims[RMaterial::OPACITY_R];
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief fcl ファイルに変換します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] fclPath fcl ファイルのパスです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertToFcl(
    CExpOpt* pExpOpt,
    const CIntermediateFile& inputFile,
    const std::string& fclPath,
    const C2NnOption& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // マテリアルアニメーション要素を取得します。
    // cmcla ファイルはアニメーションがなければ <Animations> 要素が存在しませんが、
    // fcl ファイルは <shader_param_anim> 要素が常に存在する必要があります。
    const RXMLElement* animsElem = inputFile.m_GraphicsElem->FindElement("Animations", false);
    const RXMLElement* animElem = FindElementByName(animsElem, opt.m_ElementName, "MaterialAnimation", inputFile.m_Path);
    if (animElem == nullptr)
    {
        if (!opt.m_ElementName.empty())
        {
            return RStatus(RStatus::FAILURE, "Material animation element cannot be found: " + opt.m_ElementName); // RShowError
        }
    }
    CModel cmodel;
    ParseAnimElem(pExpOpt, &cmodel.m_AnimUserDatas, animElem);
    ParseMaterialAnimEditData(pExpOpt, animElem);

    //-----------------------------------------------------------------------------
    // メンバーアニメーションデータを解析します。
    if (animElem != nullptr)
    {
        const RXMLElement* memberAnimsElem = animElem->FindElement("MemberAnimationDataSet");
        for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
        {
            status = ParseMatColorAnim(&cmodel.m_Materials, &memberAnimsElem->nodes[memberIdx], *pExpOpt);
            RCheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // fcl ファイルを出力します。
    return OutputOneFile(&cmodel, fclPath, RExpOpt::FCL, *pExpOpt, opt);
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのテクスチャ SRT アニメーションを解析します。
//!
//! @param[in,out] pMaterials マテリアル配列へのポインタです。
//! @param[in] memberAnimElem メンバーアニメーションデータ要素です。
//! @param[in] expOpt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseMatTexSrtAnim(
    CMaterialArray* pMaterials,
    const RXMLElement* memberAnimElem,
    const CExpOpt& expOpt
)
{
    //-----------------------------------------------------------------------------
    // アニメーションパスからマテリアル名を取得します。
    const std::string animPath = memberAnimElem->FindElement("Path")->text;
        // Materials["lambert2"].TextureCoordinators[0].Scale
    const std::string coordinatorsTop = ".TextureCoordinators";
    const size_t coordinatorsIdx = animPath.find(coordinatorsTop);
    if (coordinatorsIdx == std::string::npos) // テクスチャ SRT アニメーションでない場合
    {
        return RStatus::SUCCESS;
    }
    std::string matName = GetDoubleQuatedStr(animPath.substr(0, coordinatorsIdx));
    matName = RAdjustElementNameString(matName);

    //-----------------------------------------------------------------------------
    // アニメーション対象のサンプラインデックスとパラメータインデックスを取得します。
    const int samperIdx = atol(GetBracketedStr(animPath.substr(coordinatorsIdx + coordinatorsTop.size())).c_str());
    const std::string paramName = RGetExtensionFromFilePath(animPath, false);
    int paramIdx;
    if (paramName == "Scale"    )
    {
        paramIdx = ROriginalTexsrt::SCALE_X;
    }
    else if (paramName == "Rotate")
    {
        paramIdx = ROriginalTexsrt::ROTATE;
    }
    else if (paramName == "Translate")
    {
        paramIdx = ROriginalTexsrt::TRANSLATE_X;
    }
    else
    {
        return RStatus::SUCCESS;
    }

    //-----------------------------------------------------------------------------
    // 新規マテリアルなら追加します。
    int matIdx = FindObjectByName(*pMaterials, matName);
    if (matIdx == -1)
    {
        matIdx = static_cast<int>(pMaterials->size());
        pMaterials->push_back(CMaterial());
        CMaterial& newMat = pMaterials->back();
        newMat.m_Name = matName;
        newMat.InitColorAnim(expOpt);
    }
    CMaterial& mat = (*pMaterials)[matIdx];

    //-----------------------------------------------------------------------------
    // 新規テクスチャ SRT アニメーションなら追加します。
    const std::string hint = "albedo" + RGetNumberString(samperIdx); // 現在は albedoN で固定
    int texSrtAnimIdx = FindTexSrtAnimByHint(mat.m_TexSrtAnims, hint);
    if (texSrtAnimIdx == -1)
    {
        texSrtAnimIdx = static_cast<int>(mat.m_TexSrtAnims.size());
        mat.m_TexSrtAnims.push_back(CTexSrtAnim(hint, "maya", expOpt)); // 現在は maya で固定
    }
    CTexSrtAnim& texSrtAnim = mat.m_TexSrtAnims[texSrtAnimIdx];

    //-----------------------------------------------------------------------------
    // アニメーションカーブを解析します。
    if (paramIdx == ROriginalTexsrt::ROTATE)
    {
        const RXMLElement* curveElem = memberAnimElem->FindElement("FloatAnimation/SegmentsFloatCurve");
        if (curveElem != nullptr)
        {
            ParseAnimCurve(&texSrtAnim.m_Anims[paramIdx], curveElem, 1.0f);
        }
    }
    else
    {
        const RXMLElement* vectorAnimElem = memberAnimElem->FindElement("Vector2Animation");
        ParseVectorAnim(&texSrtAnim.m_Anims[paramIdx], nullptr, vectorAnimElem, 1.0f);
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief fts ファイルに変換します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] ftsPath fts ファイルのパスです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertToFts(
    CExpOpt* pExpOpt,
    const CIntermediateFile& inputFile,
    const std::string& ftsPath,
    const C2NnOption& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // マテリアルアニメーション要素を取得します。
    // cmtsa ファイルはアニメーションがなければ <Animations> 要素が存在しませんが、
    // fts ファイルは <shader_param_anim> 要素が常に存在する必要があります。
    const RXMLElement* animsElem = inputFile.m_GraphicsElem->FindElement("Animations", false);
    const RXMLElement* animElem = FindElementByName(animsElem, opt.m_ElementName, "MaterialAnimation", inputFile.m_Path);
    if (animElem == nullptr)
    {
        if (!opt.m_ElementName.empty())
        {
            return RStatus(RStatus::FAILURE, "Material animation element cannot be found: " + opt.m_ElementName); // RShowError
        }
    }
    CModel cmodel;
    ParseAnimElem(pExpOpt, &cmodel.m_AnimUserDatas, animElem);
    ParseMaterialAnimEditData(pExpOpt, animElem);

    //-----------------------------------------------------------------------------
    // メンバーアニメーションデータを解析します。
    if (animElem != nullptr)
    {
        const RXMLElement* memberAnimsElem = animElem->FindElement("MemberAnimationDataSet");
        for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
        {
            status = ParseMatTexSrtAnim(&cmodel.m_Materials, &memberAnimsElem->nodes[memberIdx], *pExpOpt);
            RCheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // fts ファイルを出力します。
    return OutputOneFile(&cmodel, ftsPath, RExpOpt::FTS, *pExpOpt, opt);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャパターン要素を解析します。
//!
//! @param[in,out] pTexInfos テクスチャ情報配列へのポインタです。
//! @param[in] patternElem テクスチャパターン要素です。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] srcRootFolder NW4C 中間ファイルが存在するフォルダのパスです。
//! @param[in] convertsTex ctex ファイルを ftx ファイルに変換するなら true です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseTexturePattern(
    CTexInfoArray* pTexInfos,
    const RXMLElement* patternElem,
    const CIntermediateFile& inputFile,
    const std::string& srcRootFolder,
    const bool convertsTex
)
{
    //-----------------------------------------------------------------------------
    // テクスチャ名と ctex ファイルのパスを取得します。
    const std::string refPath = RDecodeXmlString(patternElem->FindElement("TextureReference")->text);
    std::string ctexPath;
    const std::string texName = GetTexNameFromRefPath(&ctexPath, refPath, inputFile, srcRootFolder);
    //cerr << "tex pat: " << texName << ": " << ctexPath << R_ENDL;
    const bool ctexExists = RFileExists(ctexPath);
    if (!ctexExists && convertsTex)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + ctexPath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // テクスチャ情報を追加します。
    const std::string imgHint = "albedo"; // 現在は albedo で固定
    pTexInfos->push_back(CTexInfo(texName, ctexPath, imgHint, "", true));

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルのテクスチャパターンアニメーションを解析します。
//!
//! @param[in,out] pMaterials マテリアル配列へのポインタです。
//! @param[in] memberAnimElem メンバーアニメーションデータ要素です。
//! @param[in] expOpt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseMatTexPatAnim(
    CMaterialArray* pMaterials,
    const RXMLElement* memberAnimElem,
    const CExpOpt& expOpt
)
{
    //-----------------------------------------------------------------------------
    // アニメーションパスからマテリアル名を取得します。
    const std::string animPath = memberAnimElem->FindElement("Path")->text;
        // Materials["lambert2"].TextureMappers[0].Texture
    const std::string mappersTop = ".TextureMappers";
    const size_t mappersIdx = animPath.find(mappersTop);
    if (mappersIdx == std::string::npos || // テクスチャパターンアニメーションでない場合
        RGetExtensionFromFilePath(animPath, false) != "Texture")
    {
        return RStatus::SUCCESS;
    }
    std::string matName = GetDoubleQuatedStr(animPath.substr(0, mappersIdx));
    matName = RAdjustElementNameString(matName);

    //-----------------------------------------------------------------------------
    // アニメーション対象のサンプラインデックスを取得します。
    const int samperIdx = atol(GetBracketedStr(animPath.substr(mappersIdx + mappersTop.size())).c_str());

    //-----------------------------------------------------------------------------
    // 新規マテリアルなら追加します。
    int matIdx = FindObjectByName(*pMaterials, matName);
    if (matIdx == -1)
    {
        matIdx = static_cast<int>(pMaterials->size());
        pMaterials->push_back(CMaterial());
        CMaterial& newMat = pMaterials->back();
        newMat.m_Name = matName;
        newMat.InitColorAnim(expOpt);
    }
    CMaterial& mat = (*pMaterials)[matIdx];

    //-----------------------------------------------------------------------------
    // テクスチャパターンアニメーションを追加します。
    const std::string samplerName = "_a" + RGetNumberString(samperIdx); // 現在は _aN で固定
    const std::string hint = "albedo" + RGetNumberString(samperIdx); // 現在は albedoN で固定
    mat.m_TexPatAnims.push_back(CTexPatAnim(samplerName, hint, expOpt));
    CTexPatAnim& texPatAnim = mat.m_TexPatAnims.back();

    //-----------------------------------------------------------------------------
    // アニメーションカーブを解析します。
    const RXMLElement* curveElem = memberAnimElem->FindElement("IntAnimation/SegmentsFloatCurve");
    if (curveElem != nullptr)
    {
        ParseAnimCurve(&texPatAnim.m_Anim, curveElem, 1.0f);
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ftp ファイルに変換します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in,out] pOutputFtxPaths 出力済みの ftx ファイルのパス配列へのポインタです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] ftpPath ftp ファイルのパスです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertToFtp(
    CExpOpt* pExpOpt,
    RStringArray* pOutputFtxPaths,
    const CIntermediateFile& inputFile,
    const std::string& ftpPath,
    const C2NnOption& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // マテリアルアニメーション要素を取得します。
    // cmtpa ファイルはアニメーションがなければ <Animations> 要素が存在しませんが、
    // ftp ファイルは <shader_param_anim> 要素が常に存在する必要があります。
    const RXMLElement* animsElem = inputFile.m_GraphicsElem->FindElement("Animations", false);
    const RXMLElement* animElem = FindElementByName(animsElem, opt.m_ElementName, "MaterialAnimation", inputFile.m_Path);
    if (animElem == nullptr)
    {
        if (!opt.m_ElementName.empty())
        {
            return RStatus(RStatus::FAILURE, "Material animation element cannot be found: " + opt.m_ElementName); // RShowError
        }
    }
    CModel cmodel;
    ParseAnimElem(pExpOpt, &cmodel.m_AnimUserDatas, animElem);
    ParseMaterialAnimEditData(pExpOpt, animElem);

    //-----------------------------------------------------------------------------
    // テクスチャパターン要素を解析します。
    const std::string srcRootFolder = RGetFolderFromFilePath(inputFile.m_Path) + "/";
    const RXMLElement* patternsElem = (animElem != nullptr) ?
        animElem->FindElement("TexturePatterns", false) : nullptr;
    if (patternsElem != nullptr)
    {
        for (size_t patternIdx = 0; patternIdx < patternsElem->nodes.size(); ++patternIdx)
        {
            status = ParseTexturePattern(&cmodel.m_TexInfos,
                &patternsElem->nodes[patternIdx], inputFile, srcRootFolder, pExpOpt->m_OutFtxFlag);
            RCheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // メンバーアニメーションデータを解析します。
    if (animElem != nullptr)
    {
        const RXMLElement* memberAnimsElem = animElem->FindElement("MemberAnimationDataSet");
        for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
        {
            status = ParseMatTexPatAnim(&cmodel.m_Materials, &memberAnimsElem->nodes[memberIdx], *pExpOpt);
            RCheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャ中間ファイル群を出力します。
    if (pExpOpt->m_OutFtxFlag && !cmodel.m_TexInfos.empty())
    {
        const std::string texFolderPath = RGetFolderFromFilePath(ftpPath) + "/" +
            RExpOpt::TexFolderName + "/";
        status = OutputTextureFiles(pOutputFtxPaths, cmodel.m_TexInfos, texFolderPath, opt, &inputFile);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // ftp ファイルを出力します。
    return OutputOneFile(&cmodel, ftpPath, RExpOpt::FTP, *pExpOpt, opt);
}

//-----------------------------------------------------------------------------
//! @brief fma ファイルに変換します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in,out] pOutputFtxPaths 出力済みの ftx ファイルのパス配列へのポインタです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] fmaPath fma ファイルのパスです。
//! @param[in] fileType 中間ファイルタイプです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertToFma(
    CExpOpt* pExpOpt,
    RStringArray* pOutputFtxPaths,
    const CIntermediateFile& inputFile,
    const std::string& fmaPath,
    const RExpOpt::FileType fileType,
    const C2NnOption& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // マテリアルアニメーション要素を取得します。
    // cmtpa ファイルはアニメーションがなければ <Animations> 要素が存在しませんが、
    // fma ファイルは <material_anim> 要素が常に存在する必要があります。
    const RXMLElement* animsElem = inputFile.m_GraphicsElem->FindElement("Animations", false);
    const RXMLElement* animElem = FindElementByName(animsElem, opt.m_ElementName, "MaterialAnimation", inputFile.m_Path);
    if (animElem == nullptr)
    {
        if (!opt.m_ElementName.empty())
        {
            return RStatus(RStatus::FAILURE, "Material animation element cannot be found: " + opt.m_ElementName); // RShowError
        }
    }
    CModel cmodel;
    ParseAnimElem(pExpOpt, &cmodel.m_AnimUserDatas, animElem);
    ParseMaterialAnimEditData(pExpOpt, animElem);

    //-----------------------------------------------------------------------------
    // テクスチャパターン要素を解析します。
    const bool getsTexPatAnim = (pExpOpt->UsesSingleFma() || fileType == RExpOpt::FTP);
    if (getsTexPatAnim)
    {
        const std::string srcRootFolder = RGetFolderFromFilePath(inputFile.m_Path) + "/";
        const RXMLElement* patternsElem = (animElem != nullptr) ?
            animElem->FindElement("TexturePatterns", false) : nullptr;
        if (patternsElem != nullptr)
        {
            for (size_t patternIdx = 0; patternIdx < patternsElem->nodes.size(); ++patternIdx)
            {
                status = ParseTexturePattern(&cmodel.m_TexInfos,
                    &patternsElem->nodes[patternIdx], inputFile, srcRootFolder, pExpOpt->m_OutFtxFlag);
                RCheckStatus(status);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // メンバーアニメーションデータを解析します。
    if (animElem != nullptr)
    {
        const RXMLElement* memberAnimsElem = animElem->FindElement("MemberAnimationDataSet");
        for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
        {
            if (pExpOpt->UsesSingleFma() || fileType == RExpOpt::FCL)
            {
                status = ParseMatColorAnim(&cmodel.m_Materials, &memberAnimsElem->nodes[memberIdx], *pExpOpt);
                RCheckStatus(status);
            }
            if (pExpOpt->UsesSingleFma() || fileType == RExpOpt::FTS)
            {
                status = ParseMatTexSrtAnim(&cmodel.m_Materials, &memberAnimsElem->nodes[memberIdx], *pExpOpt);
                RCheckStatus(status);
            }
            if (getsTexPatAnim)
            {
                status = ParseMatTexPatAnim(&cmodel.m_Materials, &memberAnimsElem->nodes[memberIdx], *pExpOpt);
                RCheckStatus(status);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャ中間ファイル群を出力します。
    if (getsTexPatAnim)
    {
        if (pExpOpt->m_OutFtxFlag && !cmodel.m_TexInfos.empty())
        {
            const std::string texFolderPath = RGetFolderFromFilePath(fmaPath) + "/" +
                RExpOpt::TexFolderName + "/";
            status = OutputTextureFiles(pOutputFtxPaths, cmodel.m_TexInfos, texFolderPath, opt, &inputFile);
            RCheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // fma ファイルを出力します。
    return OutputOneFile(&cmodel, fmaPath, fileType, *pExpOpt, opt);
}

//-----------------------------------------------------------------------------
//! @brief 環境オブジェクトアニメーションを解析します。
//!
//! @param[in,out] pEnvObj 環境オブジェクトへのポインタです。
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] animElem アニメーション要素です。
//-----------------------------------------------------------------------------
void ParseEnvObjAnim(
    REnvObj* pEnvObj,
    CExpOpt* pExpOpt,
    const RXMLElement* animElem
)
{
    //-----------------------------------------------------------------------------
    // アニメーション要素の属性を取得します。
    pEnvObj->m_FrameCount = atol(animElem->GetAttribute("FrameSize").c_str());
    pEnvObj->m_LoopAnim = (animElem->GetAttribute("LoopMode") == "Loop"); // or OneTime

    if (pEnvObj->m_FrameCount > pExpOpt->m_EndFrame)
    {
        pExpOpt->m_EndFrame = pEnvObj->m_FrameCount;
    }

    //-----------------------------------------------------------------------------
    // アニメーション用ユーザーデータ配列を取得します。
    ParseUserDatas(&pEnvObj->m_UserDatas, animElem->FindElement("UserData", false));
}

//-----------------------------------------------------------------------------
//! @brief カメラの捻りを計算します。
//!
//! @param[in] pos カメラの位置です。
//! @param[in] aim 注視点の位置です。
//! @param[in] up 上方向ベクトルです。
//! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
//!
//! @return カメラの捻りを返します。
//-----------------------------------------------------------------------------
float GetCameraTwist(
    const RVec3& pos,
    const RVec3& aim,
    const RVec3& up,
    const bool snapToZero
)
{
    // 回転行列の成分を計算します。
    const RVec3 vz = (pos - aim).Normal();
    const RVec3 vx = (up.Normal() ^ vz).Normal();
    const RVec3 vy = (vz ^ vx).Normal();

    // 回転順序が ZXY の場合の回転角度を求めます。
    const float cosEpsilon = 0.00001f;
    const float sxr = -vz.y;
    const float cxr = static_cast<float>(sqrt(RMax(1.0f - sxr * sxr, 0.0f)));
    float twist = (cxr < cosEpsilon) ?
        atan2(-vy.x, vx.x) : atan2(vx.y, vy.y);
    twist *= static_cast<float>(R_M_RAD_TO_DEG);
    return (snapToZero) ? RSnapToZero(twist) : twist;
}

//-----------------------------------------------------------------------------
//! @brief カメラを解析します。
//!
//! @param[in,out] pCams カメラ配列へのポインタです。
//! @param[in] camElem カメラ要素です。
//! @param[in] conversionMagnify 移動値に掛ける倍率です。
//-----------------------------------------------------------------------------
void ParseCamera(
    CCameraArray* pCams,
    const RXMLElement* camElem,
    const float conversionMagnify
)
{
    //-----------------------------------------------------------------------------
    // カメラを追加します。
    const std::string name = RAdjustElementNameString(camElem->GetAttribute("Name"));
    const RXMLElement* rotateElem = camElem->FindElement("RotateViewUpdater", false);
    const RXMLElement* aimElem    = camElem->FindElement("AimTargetViewUpdater", false);
    const RXMLElement* lookAtElem = camElem->FindElement("LookAtTargetViewUpdater", false);
    const RCamera::RotateMode rotateMode = (rotateElem != nullptr) ?
        RCamera::ROTATE_EULER_ZXY : RCamera::ROTATE_AIM;
    const RXMLElement* projElem = camElem->FindElement("PerspectiveProjectionUpdater", false);
    const RCamera::ProjectionMode projMode = (projElem != nullptr) ?
        RCamera::PERSP : RCamera::ORTHO;
    if (projMode == RCamera::ORTHO)
    {
        projElem = camElem->FindElement("OrthoProjectionUpdater");
    }
    pCams->push_back(CCamera(name, rotateMode, projMode));
    CCamera& cam = pCams->back();

    //-----------------------------------------------------------------------------
    // 属性を取得します。
    ParseTransform(&cam, camElem->FindElement("Transform"), conversionMagnify);
    if (rotateElem != nullptr)
    {
        GetXyzAttr(cam.m_Rotate, rotateElem->FindElement("ViewRotate")); // radian
        cam.m_Rotate *= static_cast<float>(R_M_RAD_TO_DEG);
    }
    else if (aimElem != nullptr)
    {
        GetXyzAttr(cam.m_Aim, aimElem->FindElement("TargetPosition"));
        cam.m_Aim *= conversionMagnify;
        cam.m_Twist = static_cast<float>(atof(aimElem->GetAttribute("Twist").c_str()) * R_M_RAD_TO_DEG);
    }
    else if (lookAtElem != nullptr)
    {
        GetXyzAttr(cam.m_Aim, lookAtElem->FindElement("TargetPosition"));
        cam.m_Aim *= conversionMagnify;
        RVec3 up;
        GetXyzAttr(up, lookAtElem->FindElement("UpwardVector"));
        cam.m_Twist = GetCameraTwist(cam.m_Translate, cam.m_Aim, up, true);
    }

    cam.m_Aspect   = static_cast<float>(atof(projElem->GetAttribute("AspectRatio").c_str()));
    cam.m_NearClip = static_cast<float>(atof(projElem->GetAttribute("Near").c_str())) * conversionMagnify;
    cam.m_FarClip  = static_cast<float>(atof(projElem->GetAttribute("Far").c_str())) * conversionMagnify;
    if (projMode == RCamera::ORTHO)
    {
        cam.m_OrthoHeight = static_cast<float>(atof(projElem->GetAttribute("Height").c_str())) * conversionMagnify;
    }
    else
    {
        cam.m_PerspFovy = static_cast<float>(atof(projElem->GetAttribute("Fovy").c_str()) * R_M_RAD_TO_DEG);
    }

    //-----------------------------------------------------------------------------
    // アニメーションを初期化します。
    cam.InitAnim();

    //-----------------------------------------------------------------------------
    // カメラ用ユーザーデータ配列を取得します。
    ParseUserDatas(&cam.m_UserDatas, camElem->FindElement("UserData", false));
}

//-----------------------------------------------------------------------------
//! @brief カメラのメンバーアニメーションデータを解析します。
//!
//! @param[in,out] pCam カメラへのポインタです。
//! @param[in] memberAnimElem メンバーアニメーションデータ要素です。
//! @param[in] conversionMagnify 移動値に掛ける倍率です。
//-----------------------------------------------------------------------------
void ParseCameraMemberAnim(
    CCamera* pCam,
    const RXMLElement* memberAnimElem,
    const float conversionMagnify
)
{
    const std::string animPath = memberAnimElem->FindElement("Path")->text;
    if (animPath == "Transform")
    {
        const RXMLElement* xformElem = memberAnimElem->FindElement("TransformAnimation");
        const RXMLElement* translateElem = xformElem->FindElement("TranslateVector3Animation", false);
        ParseVectorAnim(&pCam->m_Anims[RCamera::POSITION_X], nullptr, translateElem, conversionMagnify);
    }
    else if (animPath == "ViewUpdater.TargetPosition")
    {
        const RXMLElement* vec3AnimElem = memberAnimElem->FindElement("Vector3Animation", false);
        ParseVectorAnim(&pCam->m_Anims[RCamera::AIM_X], nullptr, vec3AnimElem, conversionMagnify);
    }
    else if (animPath == "ViewUpdater.ViewRotate")
    {
        const RXMLElement* vec3AnimElem = memberAnimElem->FindElement("Vector3Animation", false);
        ParseVectorAnim(&pCam->m_Anims[RCamera::ROTATE_X], nullptr, vec3AnimElem, 1.0f);
    }
    else if (animPath == "ViewUpdater.UpwardVector")
    {
        const RXMLElement* vec3AnimElem = memberAnimElem->FindElement("Vector3Animation", false);
        ParseVectorAnim(&pCam->m_UpAnims[0], nullptr, vec3AnimElem, 1.0f);
    }
    else
    {
        const int paramIdx =
            (animPath == "ViewUpdater.Twist"            ) ? RCamera::TWIST        :
            (animPath == "ProjectionUpdater.AspectRatio") ? RCamera::ASPECT       :
            (animPath == "ProjectionUpdater.Near"       ) ? RCamera::NEAR_CLIP    :
            (animPath == "ProjectionUpdater.Far"        ) ? RCamera::FAR_CLIP     :
            (animPath == "ProjectionUpdater.Height"     ) ? RCamera::ORTHO_HEIGHT :
            (animPath == "ProjectionUpdater.Fovy"       ) ? RCamera::PERSP_FOVY   :
            -1;
        if (paramIdx >= 0)
        {
            const RXMLElement* floatCurveElem = memberAnimElem->FindElement("FloatAnimation/SegmentsFloatCurve");
            const float scale = RCamera::IsTranslate(paramIdx) ? conversionMagnify : 1.0f;
            ParseAnimCurve(&pCam->m_Anims[paramIdx], floatCurveElem, scale);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief カメラアニメーションを解析します。
//!
//! @param[in,out] pCams カメラ配列へのポインタです。
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] animElem カメラアニメーション要素です。
//! @param[in] opt 変換オプションです。
//-----------------------------------------------------------------------------
void ParseCameraAnim(
    CCameraArray* pCams,
    CExpOpt* pExpOpt,
    const RXMLElement* animElem,
    const C2NnOption& opt
)
{
    //-----------------------------------------------------------------------------
    // アニメーション名をチェックします。
    const std::string animName = animElem->GetAttribute("Name");
    if (!opt.m_ElementName.empty() && opt.m_ElementName != animName)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // バインドターゲットのカメラ名を取得します。
    std::string camName = Get3DEditorBindTargetName(animElem);
    if (camName.empty())
    {
        camName = animName;
        const size_t extIdx = camName.find(".CameraAnimation");
        if (extIdx != std::string::npos)
        {
            camName = camName.substr(0, extIdx);
        }
    }

    //-----------------------------------------------------------------------------
    // カメラ配列を検索します。
    int camIdx = FindObjectByName(*pCams, camName);
    if (camIdx == -1)
    {
        return;
        // 見つからなかった場合は新規カメラを追加します。
        // ↑3DEditor ではカメラアニメーションのみの中間ファイルを作成できないので保留
        //camIdx = static_cast<int>(pCams->size());
        //const RCamera::RotateMode rotateMode = (animElem->GetAttribute("ViewMode") == "Rotate") ?
        //  RCamera::ROTATE_EULER_ZXY : RCamera::ROTATE_AIM;
        //const RCamera::ProjectionMode projMode = (animElem->GetAttribute("ProjectionMode") == "Perspective") ?
        //  RCamera::PERSP : RCamera::ORTHO;
        //pCams->push_back(CCamera(camName, rotateMode, projMode));
        //CCamera& newCam = pCams->back();
        //newCam.InitAnim();
    }
    CCamera& cam = (*pCams)[camIdx];

    //-----------------------------------------------------------------------------
    // すでにアニメーションを取得済みならスキップします。
    if (cam.m_IsAnimGot)
    {
        return;
    }
    cam.m_IsAnimGot = true;

    //-----------------------------------------------------------------------------
    // カメラアニメーション要素を解析します。
    ParseEnvObjAnim(&cam, pExpOpt, animElem);

    //-----------------------------------------------------------------------------
    // メンバーアニメーションデータを解析します。
    const RXMLElement* memberAnimsElem = animElem->FindElement("MemberAnimationDataSet");
    for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
    {
        ParseCameraMemberAnim(&cam, &memberAnimsElem->nodes[memberIdx], opt.m_ConversionMagnify);
    }

    //-----------------------------------------------------------------------------
    // 捻りのアニメーションカーブを作成します。
    RAnimCurve& twistCurve = cam.m_Anims[RCamera::TWIST];
    if (cam.m_RotateMode == RCamera::ROTATE_AIM &&
        !twistCurve.m_UseFlag &&
        (cam.HasPositionAnim() || cam.HasAimAnim() || cam.HasUpAnim()))
    {
        const RAnimCurve* posCurves = &cam.m_Anims[RCamera::POSITION_X];
        const RAnimCurve* aimCurves = &cam.m_Anims[RCamera::AIM_X];
        const RAnimCurve* upCurves  = &cam.m_UpAnims[0];
        twistCurve.m_FullValues.clear();
        const int subFrameCount = GetSubFrameCount(cam.m_FrameCount, *pExpOpt);
        for (int frameIdx = 0; frameIdx < subFrameCount; ++frameIdx)
        {
            const float frame = GetFloatFrameFromSubFrame(frameIdx, pExpOpt);
            const RVec3 pos(
                EvaluateAnimCurve(posCurves[0], frame),
                EvaluateAnimCurve(posCurves[1], frame),
                EvaluateAnimCurve(posCurves[2], frame));
            const RVec3 aim(
                EvaluateAnimCurve(aimCurves[0], frame),
                EvaluateAnimCurve(aimCurves[1], frame),
                EvaluateAnimCurve(aimCurves[2], frame));
            const RVec3 up(
                EvaluateAnimCurve(upCurves[0], frame),
                EvaluateAnimCurve(upCurves[1], frame),
                EvaluateAnimCurve(upCurves[2], frame));
            const float twist = GetCameraTwist(pos, aim, up, true);
            twistCurve.m_FullValues.push_back(twist);
        }
        twistCurve.m_Tolerance = pExpOpt->m_TolR;
        twistCurve.UpdateConstantFlag();
        if (!twistCurve.m_ConstantFlag)
        {
            twistCurve.MakeKeys(GetFloatFrameFromSubFrame, pExpOpt, true);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ライトのタイプを取得します。
//-----------------------------------------------------------------------------
RLight::Type GetLightType(const std::string& kind)
{
    return
        (kind == "DirectionalLight") ? RLight::DIRECTIONAL :
        (kind == "PointLight"      ) ? RLight::POINT       :
        (kind == "SpotLight"       ) ? RLight::SPOT        :
        RLight::AMBIENT; // Unused
}

//-----------------------------------------------------------------------------
//! @brief ライトを解析します。
//!
//! @param[in,out] pLgts ライト配列へのポインタです。
//! @param[in] lgtElem ライト要素です。
//! @param[in] conversionMagnify 移動値に掛ける倍率です。
//-----------------------------------------------------------------------------
void ParseLight(
    CLightArray* pLgts,
    const RXMLElement* lgtElem,
    const float conversionMagnify
)
{
    //-----------------------------------------------------------------------------
    // ライトを追加します。
    const std::string name = RAdjustElementNameString(lgtElem->GetAttribute("Name"));
    const RLight::Type type = (lgtElem->name == "AmbientLightCtr") ?
        RLight::AMBIENT : GetLightType(lgtElem->GetAttribute("LightKind"));
    pLgts->push_back(CLight(name, type));
    CLight& lgt = pLgts->back();

    //-----------------------------------------------------------------------------
    // 属性を取得します。
    lgt.m_Enable = (lgtElem->GetAttribute("IsBranchVisible") == "true");
    ParseTransform(&lgt, lgtElem->FindElement("Transform"), conversionMagnify);
    if (lgt.m_Type != RLight::AMBIENT)
    {
        GetXyzAttr(lgt.m_Direction, lgtElem->FindElement("Direction"));
    }
    lgt.m_Aim = (lgt.m_Type == RLight::SPOT) ?
        lgt.m_Translate + lgt.m_Direction : RVec3::kZero;

    if (lgt.m_Type != RLight::AMBIENT)
    {
        lgt.m_UsesDistAttn = (lgtElem->GetAttribute("IsDistanceAttenuationEnabled") == "true");
        lgt.m_DistAttnStart = static_cast<float>(
            atof(lgtElem->GetAttribute("DistanceAttenuationStart").c_str())) * conversionMagnify;
        lgt.m_DistAttnEnd   = static_cast<float>(
            atof(lgtElem->GetAttribute("DistanceAttenuationEnd"  ).c_str())) * conversionMagnify;
    }
    if (lgt.m_Type == RLight::SPOT)
    {
        lgt.m_AngleAttnStart = 20.0f;
        lgt.m_AngleAttnEnd   = 20.0f;
    }

    GetRgbAttr(lgt.m_Color0, lgtElem->FindElement(
        (lgt.m_Type == RLight::AMBIENT) ? "Ambient" : "Diffuse"));

    //-----------------------------------------------------------------------------
    // アニメーションを初期化します。
    lgt.InitAnim();

    //-----------------------------------------------------------------------------
    // ライト用ユーザーデータ配列を取得します。
    ParseUserDatas(&lgt.m_UserDatas, lgtElem->FindElement("UserData", false));
}

//-----------------------------------------------------------------------------
//! @brief ライトのメンバーアニメーションデータを解析します。
//!
//! @param[in,out] pLgt ライトへのポインタです。
//! @param[in] memberAnimElem メンバーアニメーションデータ要素です。
//! @param[in] conversionMagnify 移動値に掛ける倍率です。
//-----------------------------------------------------------------------------
void ParseLightMemberAnim(
    CLight* pLgt,
    const RXMLElement* memberAnimElem,
    const float conversionMagnify
)
{
    const std::string animPath = memberAnimElem->FindElement("Path")->text;
    if (animPath == "IsLightEnabled")
    {
        const RXMLElement* boolCurveElem = memberAnimElem->FindElement("BoolAnimation/BoolCurve");
        ParseBoolAnimCurve(&pLgt->m_Anims[RLight::ENABLE], boolCurveElem);
    }
    else if (animPath == "Transform")
    {
        const RXMLElement* xformElem = memberAnimElem->FindElement("TransformAnimation");
        const RXMLElement* translateElem = xformElem->FindElement("TranslateVector3Animation", false);
        ParseVectorAnim(&pLgt->m_Anims[RLight::POSITION_X], nullptr, translateElem, conversionMagnify);
    }
    else if (animPath == "Direction")
    {
        const RXMLElement* vec3AnimElem = memberAnimElem->FindElement("Vector3Animation", false);
        ParseVectorAnim(&pLgt->m_Anims[RLight::DIRECTION_X], nullptr, vec3AnimElem, 1.0f);
    }
    else if (animPath.find("DistanceAttenuation") == 0)
    {
        const int paramIdx = (animPath == "DistanceAttenuationStart") ?
            RLight::DIST_ATTN_START : RLight::DIST_ATTN_END;
        const RXMLElement* floatCurveElem = memberAnimElem->FindElement("FloatAnimation/SegmentsFloatCurve");
        ParseAnimCurve(&pLgt->m_Anims[paramIdx], floatCurveElem, conversionMagnify);
    }
    else if (
        (animPath == "Ambient" && pLgt->m_Type == RLight::AMBIENT) ||
        (animPath == "Diffuse" && pLgt->m_Type != RLight::AMBIENT))
    {
        const RXMLElement* rgbaColorAnimElem = memberAnimElem->FindElement("RgbaColorAnimation");
        ParseRgbaColorAnim(&pLgt->m_Anims[RLight::COLOR0_R], nullptr, rgbaColorAnimElem);
    }
}

//-----------------------------------------------------------------------------
//! @brief ライトアニメーションを解析します。
//!
//! @param[in,out] pLgts ライト配列へのポインタです。
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] animElem ライトアニメーション要素です。
//! @param[in] opt 変換オプションです。
//-----------------------------------------------------------------------------
void ParseLightAnim(
    CLightArray* pLgts,
    CExpOpt* pExpOpt,
    const RXMLElement* animElem,
    const C2NnOption& opt
)
{
    //-----------------------------------------------------------------------------
    // アニメーション名をチェックします。
    const std::string animName = animElem->GetAttribute("Name");
    if (!opt.m_ElementName.empty() && opt.m_ElementName != animName)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // バインドターゲットのライト名を取得します。
    std::string lgtName = Get3DEditorBindTargetName(animElem);
    if (lgtName.empty())
    {
        lgtName = animName;
        const size_t extIdx = lgtName.find(".LightAnimation");
        if (extIdx != std::string::npos)
        {
            lgtName = lgtName.substr(0, extIdx);
        }
    }

    //-----------------------------------------------------------------------------
    // ライト配列を検索します。
    int lgtIdx = FindObjectByName(*pLgts, lgtName);
    if (lgtIdx == -1)
    {
        return;
        // 見つからなかった場合は新規ライトを追加します。
        // ↑3DEditor ではライトアニメーションのみの中間ファイルを作成できないので保留
        //lgtIdx = static_cast<int>(pLgts->size());
        //const RLight::Type lgtType = GetLightType(animElem->GetAttribute("LightKind"));
        //pLgts->push_back(CLight(lgtName, lgtType));
        //CLight& newLgt = pLgts->back();
        //newLgt.InitAnim();
    }
    CLight& lgt = (*pLgts)[lgtIdx];

    //-----------------------------------------------------------------------------
    // すでにアニメーションを取得済みならスキップします。
    if (lgt.m_IsAnimGot)
    {
        return;
    }
    lgt.m_IsAnimGot = true;

    //-----------------------------------------------------------------------------
    // ライトアニメーション要素を解析します。
    ParseEnvObjAnim(&lgt, pExpOpt, animElem);

    //-----------------------------------------------------------------------------
    // メンバーアニメーションデータを解析します。
    const RXMLElement* memberAnimsElem = animElem->FindElement("MemberAnimationDataSet");
    for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
    {
        ParseLightMemberAnim(&lgt, &memberAnimsElem->nodes[memberIdx], opt.m_ConversionMagnify);
    }

    //-----------------------------------------------------------------------------
    // 目標の位置のアニメーションカーブを作成します。
    if (lgt.m_Type == RLight::SPOT &&
        (lgt.HasPositionAnim() || lgt.HasDirectionAnim()))
    {
        const RAnimCurve* posCurves = &lgt.m_Anims[RLight::POSITION_X ];
        const RAnimCurve* dirCurves = &lgt.m_Anims[RLight::DIRECTION_X];
        RAnimCurve* aimCurves = &lgt.m_Anims[RLight::AIM_X];
        aimCurves[0].m_FullValues.clear();
        aimCurves[1].m_FullValues.clear();
        aimCurves[2].m_FullValues.clear();
        const int subFrameCount = GetSubFrameCount(lgt.m_FrameCount, *pExpOpt);
        for (int frameIdx = 0; frameIdx < subFrameCount; ++frameIdx)
        {
            const float frame = GetFloatFrameFromSubFrame(frameIdx, pExpOpt);
            const RVec3 pos(
                EvaluateAnimCurve(posCurves[0], frame),
                EvaluateAnimCurve(posCurves[1], frame),
                EvaluateAnimCurve(posCurves[2], frame));
            const RVec3 dir(
                EvaluateAnimCurve(dirCurves[0], frame),
                EvaluateAnimCurve(dirCurves[1], frame),
                EvaluateAnimCurve(dirCurves[2], frame));
            const RVec3 aim = pos + dir;
            aimCurves[0].m_FullValues.push_back(aim.x);
            aimCurves[1].m_FullValues.push_back(aim.y);
            aimCurves[2].m_FullValues.push_back(aim.z);
        }
        for (int xyzIdx = 0; xyzIdx < R_XYZ_COUNT; ++xyzIdx)
        {
            RAnimCurve& curve = aimCurves[xyzIdx];
            curve.m_Tolerance = pExpOpt->m_TolT;
            curve.UpdateConstantFlag();
            if (!curve.m_ConstantFlag)
            {
                curve.MakeKeys(GetFloatFrameFromSubFrame, pExpOpt);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief フォグを解析します。
//!
//! @param[in,out] pFogs フォグ配列へのポインタです。
//! @param[in] fogElem フォグ要素です。
//! @param[in] conversionMagnify 移動値に掛ける倍率です。
//-----------------------------------------------------------------------------
void ParseFog(
    CFogArray* pFogs,
    const RXMLElement* fogElem,
    const float conversionMagnify
)
{
    //-----------------------------------------------------------------------------
    // フォグを追加します。
    const std::string name = RAdjustElementNameString(fogElem->GetAttribute("Name"));
    pFogs->push_back(CFog(name));
    CFog& fog = pFogs->back();

    //-----------------------------------------------------------------------------
    // 属性を取得します。
    const RXMLElement* updaterElem = fogElem->FindElement("LinearFogUpdaterCtr", false);
    if (updaterElem == nullptr) updaterElem = fogElem->FindElement("ExponentFogUpdaterCtr", false);
    if (updaterElem == nullptr) updaterElem = fogElem->FindElement("ExponentSquareFogUpdaterCtr", false);
    if (updaterElem != nullptr)
    {
        fog.m_DistAttnStart = static_cast<float>(atof(updaterElem->GetAttribute("MinFogDepth").c_str())) * conversionMagnify;
        fog.m_DistAttnEnd   = static_cast<float>(atof(updaterElem->GetAttribute("MaxFogDepth").c_str())) * conversionMagnify;
    }
    GetRgbAttr(fog.m_Color, fogElem->FindElement("Color"));

    //-----------------------------------------------------------------------------
    // アニメーションを初期化します。
    fog.InitAnim();

    //-----------------------------------------------------------------------------
    // フォグ用ユーザーデータ配列を取得します。
    ParseUserDatas(&fog.m_UserDatas, fogElem->FindElement("UserData", false));
}

//-----------------------------------------------------------------------------
//! @brief フォグのメンバーアニメーションデータを解析します。
//!
//! @param[in,out] pFog フォグへのポインタです。
//! @param[in] memberAnimElem メンバーアニメーションデータ要素です。
//-----------------------------------------------------------------------------
void ParseFogMemberAnim(CFog* pFog, const RXMLElement* memberAnimElem)
{
    const std::string animPath = memberAnimElem->FindElement("Path")->text;
    if (animPath == "Color")
    {
        const RXMLElement* rgbaColorAnimElem = memberAnimElem->FindElement("RgbaColorAnimation");
        ParseRgbaColorAnim(&pFog->m_Anims[RFog::COLOR_R], nullptr, rgbaColorAnimElem);
    }
}

//-----------------------------------------------------------------------------
//! @brief フォグアニメーションを解析します。
//!
//! @param[in,out] pFogs フォグ配列へのポインタです。
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] animElem ライトアニメーション要素です。
//! @param[in] opt 変換オプションです。
//-----------------------------------------------------------------------------
void ParseFogAnim(
    CFogArray* pFogs,
    CExpOpt* pExpOpt,
    const RXMLElement* animElem,
    const C2NnOption& opt
)
{
    //-----------------------------------------------------------------------------
    // アニメーション名をチェックします。
    const std::string animName = animElem->GetAttribute("Name");
    if (!opt.m_ElementName.empty() && opt.m_ElementName != animName)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // バインドターゲットのフォグ名を取得します。
    std::string fogName = Get3DEditorBindTargetName(animElem);
    if (fogName.empty())
    {
        fogName = animName;
        const size_t extIdx = fogName.find(".FogAnimation");
        if (extIdx != std::string::npos)
        {
            fogName = fogName.substr(0, extIdx);
        }
    }

    //-----------------------------------------------------------------------------
    // フォグ配列を検索します。
    int fogIdx = FindObjectByName(*pFogs, fogName);
    if (fogIdx == -1)
    {
        return;
        // 見つからなかった場合は新規フォグを追加します。
        // ↑3DEditor ではフォグアニメーションのみの中間ファイルを作成できないので保留
        //fogIdx = static_cast<int>(pFogs->size());
        //pFogs->push_back(CFog(fogName));
        //CFog& newFog = pFogs->back();
        //newFog.InitAnim();
    }
    CFog& fog = (*pFogs)[fogIdx];

    //-----------------------------------------------------------------------------
    // すでにアニメーションを取得済みならスキップします。
    if (fog.m_IsAnimGot)
    {
        return;
    }
    fog.m_IsAnimGot = true;

    //-----------------------------------------------------------------------------
    // フォグアニメーション要素を解析します。
    ParseEnvObjAnim(&fog, pExpOpt, animElem);

    //-----------------------------------------------------------------------------
    // メンバーアニメーションデータを解析します。
    const RXMLElement* memberAnimsElem = animElem->FindElement("MemberAnimationDataSet");
    for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
    {
        ParseFogMemberAnim(&fog, &memberAnimsElem->nodes[memberIdx]);
    }
}

//-----------------------------------------------------------------------------
//! @brief fsn ファイルに変換します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] fsnPath fsn ファイルのパスです。
//! @param[in] opt 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertToFsn(
    CExpOpt* pExpOpt,
    const CIntermediateFile& inputFile,
    const std::string& fsnPath,
    const C2NnOption& opt
)
{
    RStatus status;
    CModel cmodel;
    const RXMLElement* animsElem = inputFile.m_GraphicsElem->FindElement("Animations", false);

    //-----------------------------------------------------------------------------
    // カメラを解析します。
    CCameraArray& cams = cmodel.m_Cameras;
    const RXMLElement* camsElem = inputFile.m_GraphicsElem->FindElement("Cameras", false);
    if (camsElem != nullptr)
    {
        for (size_t camIdx = 0; camIdx < camsElem->nodes.size(); ++camIdx)
        {
            ParseCamera(&cams, &camsElem->nodes[camIdx], opt.m_ConversionMagnify);
        }
    }

    //-----------------------------------------------------------------------------
    // カメラアニメーションを解析します。
    if (animsElem != nullptr)
    {
        for (size_t animIdx = 0; animIdx < animsElem->nodes.size(); ++animIdx)
        {
            const RXMLElement* animElem = &animsElem->nodes[animIdx];
            if (animElem->name == "CameraAnimationData")
            {
                ParseCameraAnim(&cams, pExpOpt, animElem, opt);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // ライトを解析します。
    CLightArray& lgts = cmodel.m_Lights;
    const RXMLElement* lgtsElem = inputFile.m_GraphicsElem->FindElement("Lights", false);
    if (lgtsElem != nullptr)
    {
        for (size_t lgtIdx = 0; lgtIdx < lgtsElem->nodes.size(); ++lgtIdx)
        {
            ParseLight(&lgts, &lgtsElem->nodes[lgtIdx], opt.m_ConversionMagnify);
        }
    }

    //-----------------------------------------------------------------------------
    // ライトアニメーションを解析します。
    if (animsElem != nullptr)
    {
        for (size_t animIdx = 0; animIdx < animsElem->nodes.size(); ++animIdx)
        {
            const RXMLElement* animElem = &animsElem->nodes[animIdx];
            if (animElem->name == "LightAnimationData")
            {
                ParseLightAnim(&lgts, pExpOpt, animElem, opt);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // フォグを解析します。
    CFogArray& fogs = cmodel.m_Fogs;
    const RXMLElement* fogsElem = inputFile.m_GraphicsElem->FindElement("Fogs", false);
    if (fogsElem != nullptr)
    {
        for (size_t fogIdx = 0; fogIdx < fogsElem->nodes.size(); ++fogIdx)
        {
            ParseFog(&fogs, &fogsElem->nodes[fogIdx], opt.m_ConversionMagnify);
        }
    }

    //-----------------------------------------------------------------------------
    // フォグアニメーションを解析します。
    if (animsElem != nullptr)
    {
        for (size_t animIdx = 0; animIdx < animsElem->nodes.size(); ++animIdx)
        {
            const RXMLElement* animElem = &animsElem->nodes[animIdx];
            if (animElem->name == "AnimationData" &&
                animElem->GetAttribute("TargetAnimationGroupName", "", false) == "FogAnimation")
            {
                ParseFogAnim(&fogs, pExpOpt, animElem, opt);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // fsn ファイルを出力します。
    return OutputOneFile(&cmodel, fsnPath, RExpOpt::FSN, *pExpOpt, opt);
}

//-----------------------------------------------------------------------------
//! @brief 環境オブジェクトアニメーションが存在するなら true を返します。
//!
//! @param[in] animsElem アニメーション配列要素です。
//! @param[in] opt 変換オプションです。
//!
//! @return 環境オブジェクトアニメーションが存在するなら true を返します。
//-----------------------------------------------------------------------------
bool EnvObjAnimExists(const RXMLElement* animsElem, const C2NnOption& opt)
{
    if (animsElem != nullptr)
    {
        for (size_t animIdx = 0; animIdx < animsElem->nodes.size(); ++animIdx)
        {
            const RXMLElement* animElem = &animsElem->nodes[animIdx];
            const bool isFogAnim = (
                animElem->name == "AnimationData" &&
                animElem->GetAttribute("TargetAnimationGroupName", "", false) == "FogAnimation");
            if (animElem->name == "CameraAnimationData" ||
                animElem->name == "LightAnimationData"  ||
                isFogAnim)
            {
                const std::string animName = animElem->GetAttribute("Name");
                if (opt.m_ElementName.empty() || opt.m_ElementName == animName)
                {
                    return true;
                }
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 入力ファイルに存在する要素からファイル出力フラグを設定します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] opt 変換オプションです。
//! @param[in] inputFile 入力ファイルです。
//-----------------------------------------------------------------------------
RStatus SetOutFileFlagForElement(
    CExpOpt* pExpOpt,
    const C2NnOption& opt,
    const CIntermediateFile& inputFile
)
{
    //-----------------------------------------------------------------------------
    // マテリアルアニメーション以外の要素の存在をチェックします。
    const bool isCres = (RGetExtensionFromFilePath(inputFile.m_Path) == "cres");
    const RXMLElement* animsElem = inputFile.m_GraphicsElem->FindElement("Animations", false);
    if (isCres)
    {
        //-----------------------------------------------------------------------------
        // モデル要素の存在をチェックします。
        const RXMLElement* modelsElem = inputFile.m_GraphicsElem->FindElement("Models", false);
        const RXMLElement* modelElem = FindElementByName(modelsElem, opt.m_ElementName, "", inputFile.m_Path);
        pExpOpt->m_OutFileFlag[RExpOpt::FMD] = (modelElem != nullptr);

        //-----------------------------------------------------------------------------
        // スケルタルアニメーション要素の存在をチェックします。
        const RXMLElement* skeletalAnimElem = FindElementByName(animsElem, opt.m_ElementName, "SkeletalAnimation", inputFile.m_Path);
        pExpOpt->m_OutFileFlag[RExpOpt::FSK] = (skeletalAnimElem != nullptr);

        //-----------------------------------------------------------------------------
        // ビジビリティアニメーション要素の存在をチェックします。
        const RXMLElement* visAnimElem = FindElementByName(animsElem, opt.m_ElementName, "VisibilityAnimation", inputFile.m_Path);
        pExpOpt->m_OutFileFlag[RExpOpt::FVB] = (visAnimElem != nullptr);

        //-----------------------------------------------------------------------------
        // カメラ／ライト／フォグ要素とそれらのアニメーション要素の存在をチェックします。
        const RXMLElement* camsElem = inputFile.m_GraphicsElem->FindElement("Cameras", false);
        const RXMLElement* lgtsElem = inputFile.m_GraphicsElem->FindElement("Lights", false);
        const RXMLElement* fogsElem = inputFile.m_GraphicsElem->FindElement("Fogs", false);
        if (
            (camsElem != nullptr && !camsElem->nodes.empty()) ||
            (lgtsElem != nullptr && !lgtsElem->nodes.empty()) ||
            (fogsElem != nullptr && !fogsElem->nodes.empty()) ||
            EnvObjAnimExists(animsElem, opt)
        )
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FSN] = true;
        }
    }

    //-----------------------------------------------------------------------------
    // マテリアルアニメーション要素の存在をチェックします。
    const RXMLElement* matAnimElem = FindElementByName(animsElem, opt.m_ElementName, "MaterialAnimation", inputFile.m_Path);
    if (matAnimElem != nullptr)
    {
        const RXMLElement* memberAnimsElem = matAnimElem->FindElement("MemberAnimationDataSet");
        for (size_t memberIdx = 0; memberIdx < memberAnimsElem->nodes.size(); ++memberIdx)
        {
            const RXMLElement* memberAnimElem = &memberAnimsElem->nodes[memberIdx];
            const std::string animPath = memberAnimElem->FindElement("Path")->text;
            const std::string animParam = RGetExtensionFromFilePath(animPath, false);
            if (animPath.find(".MaterialColor") != std::string::npos)
            {
                pExpOpt->m_OutFileFlag[RExpOpt::FCL] = true;
            }
            else if (animPath.find(".TextureCoordinators") != std::string::npos)
            {
                pExpOpt->m_OutFileFlag[RExpOpt::FTS] = true;
            }
            else if (animPath.find(".TextureMappers") != std::string::npos &&
                animParam == "Texture")
            {
                pExpOpt->m_OutFileFlag[RExpOpt::FTP] = true;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // いずれかのファイル出力フラグが ON になっているかチェックします。
    bool outputsSome = false;
    for (int fileType = 0; fileType < RExpOpt::FILE_TYPE_COUNT; ++fileType)
    {
        if (pExpOpt->m_OutFileFlag[fileType])
        {
            outputsSome = true;
            break;
        }
    }
    if (!outputsSome)
    {
        const std::string nameStr = (opt.m_ElementName.empty()) ? "" : " (" + opt.m_ElementName + ")";
        if (isCres)
        {
            return RStatus(RStatus::FAILURE, "Model or animation element cannot be found: " + inputFile.m_Path + nameStr); // RShowError
        }
        else
        {
            return RStatus(RStatus::FAILURE, "Material animation element cannot be found: " + inputFile.m_Path + nameStr); // RShowError
        }
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief コンバータ用エクスポートオプションを設定します。
//!
//! @param[in,out] pExpOpt エクスポートオプションへのポインタです。
//! @param[in] opt 変換オプションです。
//! @param[in] inputFile 入力ファイルです。
//-----------------------------------------------------------------------------
RStatus SetExportOption(
    CExpOpt* pExpOpt,
    const C2NnOption& opt,
    const CIntermediateFile& inputFile
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 入力ファイルパス
    pExpOpt->m_SrcPath = opt.m_InputPath;

    //-----------------------------------------------------------------------------
    // 出力ファイル名と出力フォルダ
    pExpOpt->m_OutFileName = RGetFileNameFromFilePath(opt.m_NoExtOutputPath);
    pExpOpt->m_OutFolderPath = RGetFolderFromFilePath(opt.m_NoExtOutputPath);

    //-----------------------------------------------------------------------------
    // 編集用コメント文
    pExpOpt->m_CommentText = opt.m_CommentText;

    //-----------------------------------------------------------------------------
    // ファイル出力フラグ
    pExpOpt->m_IsBinaryFormat = opt.m_IsBinaryFormat;
    pExpOpt->ClearOutFileFlag();
    pExpOpt->m_OutFtxFlag = opt.m_ConvertsTex;
    const std::string inExt = RGetExtensionFromFilePath(opt.m_InputPath);
    const std::string outExt = (opt.m_OutputPath != opt.m_NoExtOutputPath) ?
        RGetExtensionFromFilePath(opt.m_OutputPath) : "";
    const bool usesFma = ((outExt.empty() && !opt.m_UsesFclFtsFtp) ||
        outExt.find("fma") == 0);
    pExpOpt->m_UsesFclFtsFtp = !usesFma;
    pExpOpt->m_SeparatesFma = opt.m_SeparatesFma;

    bool isOutExtValid = true;
    if (inExt == "cmdl")
    {
        pExpOpt->m_OutFileFlag[RExpOpt::FMD] = true;
        isOutExtValid = (outExt.find("fmd") == 0);
    }
    else if (inExt == "cskla")
    {
        pExpOpt->m_OutFileFlag[RExpOpt::FSK] = true;
        isOutExtValid = (outExt.find("fsk") == 0);
    }
    else if (inExt == "cmdla")
    {
        pExpOpt->m_OutFileFlag[RExpOpt::FVB] = true;
        isOutExtValid = (outExt.find("fvb") == 0);
    }
    else if (inExt == "cmcla")
    {
        pExpOpt->m_OutFileFlag[RExpOpt::FCL] = true;
        isOutExtValid = (usesFma || outExt.find("fcl") == 0);
    }
    else if (inExt == "cmtsa")
    {
        pExpOpt->m_OutFileFlag[RExpOpt::FTS] = true;
        isOutExtValid = (usesFma || outExt.find("fts") == 0);
    }
    else if (inExt == "cmtpa")
    {
        pExpOpt->m_OutFileFlag[RExpOpt::FTP] = true;
        isOutExtValid = (usesFma || outExt.find("ftp") == 0);
    }
    else if (inExt == "ccam" || inExt == "clgt" || inExt == "cfog" || inExt == "cenv")
    {
        pExpOpt->m_OutFileFlag[RExpOpt::FSN] = true;
        isOutExtValid = (outExt.find("fsn") == 0);
    }
    else if (inExt == "cmata")
    {
        if (outExt.find("fcl") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FCL] = true;
        }
        else if (outExt.find("fts") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FTS] = true;
        }
        else if (outExt.find("ftp") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FTP] = true;
        }
        else if (usesFma || outExt.empty())
        {
            status = SetOutFileFlagForElement(pExpOpt, opt, inputFile);
            RCheckStatus(status);
        }
        else
        {
            isOutExtValid = false;
        }
    }
    else if (inExt == "cres")
    {
        if (outExt.find("fmd") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FMD] = true;
        }
        else if (outExt.find("fsk") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FSK] = true;
        }
        else if (outExt.find("fvb") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FVB] = true;
        }
        else if (outExt.find("fcl") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FCL] = true;
        }
        else if (outExt.find("fts") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FTS] = true;
        }
        else if (outExt.find("ftp") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FTP] = true;
        }
        else if (outExt.find("fsn") == 0)
        {
            pExpOpt->m_OutFileFlag[RExpOpt::FSN] = true;
        }
        else if (usesFma || outExt.empty())
        {
            status = SetOutFileFlagForElement(pExpOpt, opt, inputFile);
            RCheckStatus(status);
        }
        else
        {
            isOutExtValid = false;
        }
    }

    if (!outExt.empty() && !isOutExtValid)
    {
        return RStatus(RStatus::FAILURE, "Output file extension is wrong for " + inExt + " file: " + opt.m_OutputPath); // RShowError
    }

    return RStatus::SUCCESS;
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief テクスチャのリニア変換情報を表示します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] opt 変換オプションです。
//-----------------------------------------------------------------------------
void DisplayTextureLinearInfo(std::ostream& os, const C2NnOption& opt)
{
    if (opt.m_TexLinearFlag != C2NnOption::LinearFlag_None)
    {
        os << "LinearT: " << GetOptStringFromLinearFlag(opt.m_TexLinearFlag) << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief グラフィックスツールフォルダのパス（末尾はスラッシュ）を取得します。
//!
//! @param[in] converterFolderPath C2Nn コンバータの存在するフォルダのパス（末尾はスラッシュ）です。
//!
//! @return グラフィックスツールフォルダのパスを返します。
//-----------------------------------------------------------------------------
std::string GetGraphicsToolsPath(const std::string& converterFolderPath)
{
    std::string path = converterFolderPath + "../GraphicsTools/";
    if (!RFileExists(path + "TextureConverterEncoder.dll"))
    {
        path = RGetEnvVar("NINTENDO_SDK_ROOT");
        if (!path.empty())
        {
            path = RGetFilePathWithEndingSlash(RGetUnixFilePath(path)) +
                "Tools/Graphics/GraphicsTools/";
        }
        else
        {
            path = "./";
        }
    }
    return path;
}

//-----------------------------------------------------------------------------
//! @brief 3D ツールフォルダのパス（末尾はスラッシュ）を取得します。
//!
//! @param[in] converterFolderPath C2Nn コンバータの存在するフォルダのパス（末尾はスラッシュ）です。
//!
//! @return 3D ツールフォルダのパスを返します。
//-----------------------------------------------------------------------------
std::string Get3dToolsPath(const std::string& converterFolderPath)
{
    std::string path = converterFolderPath;
    if (!RFileExists(path + "3dIntermediateFileOptimizer.exe")) // コンバータと同じフォルダに存在すれば優先して使用
    {
        path = RGetEnvVar("NINTENDO_SDK_ROOT");
        if (!path.empty())
        {
            path = RGetFilePathWithEndingSlash(RGetUnixFilePath(path)) +
                "Tools/Graphics/3dTools/";
        }
        else
        {
            path = "./";
        }
    }
    return path;
}

//=============================================================================
// オプション引数情報配列です。
//=============================================================================
const OptionArgInfo GlobalOptionArgInfos[] =
{
    { ' ', "version"  , false },
    { 'h', "help"     , false },
    { 's', "silent"   , false },
    { ' ', "args-file", true  },
    { ' ', "job-list" , true  },
    { ' ', nullptr, false }, // 配列の最後を表します。
};

const OptionArgInfo JobOptionArgInfos[] =
{
    { ' ', "project-root"          , true  },
    { ' ', "disable-file-info"     , false },
    { 'o', "output"                , true  },
    { 'n', "name"                  , true  },
    { 'g', "magnify"               , true  },
    { 'd', "disable-texture"       , false },
    { 'a', "all-textures"          , false },
    { 'x', "max-mipmap"            , false },
    { ' ', "mip-gen-filter"        , true  },
    { 'l', "linear-texture"        , true  },
    { 'c', "linear-vertex-color"   , true  },
    { ' ', "use-fcl-fts-ftp"       , false },
    { ' ', "separate-material-anim", false },
    { ' ', "comment"               , true  },
    { ' ', "arrange-format"        , false }, // 非公開オプションです。
    { ' ', nullptr, false }, // 配列の最後を表します。
};

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 引数がグローバルオプションなら true を返します。
//-----------------------------------------------------------------------------
bool C2NnConverter::IsGlobalOptionArg(const wchar_t* arg) const
{
    const std::string argA = RGetShiftJisFromUnicode(arg);
    bool isLongName;
    int suffixNum;
    return (GetOptionArgInfo(&isLongName, &suffixNum, GlobalOptionArgInfos, argA) != nullptr);
}

//-----------------------------------------------------------------------------
//! @brief 指定した引数の次の引数がグローバルオプションの値なら true を返します。
//-----------------------------------------------------------------------------
bool C2NnConverter::IsNextArgGlobalOptionValue(const wchar_t* arg) const
{
    const std::string argA = RGetShiftJisFromUnicode(arg);
    return IsNextArgOptionValue(GlobalOptionArgInfos, argA);
}

//-----------------------------------------------------------------------------
//! @brief 指定した引数の次の引数がジョブオプションの値なら true を返します。
//-----------------------------------------------------------------------------
bool C2NnConverter::IsNextArgJobOptionValue(const wchar_t* arg) const
{
    const std::string argA = RGetShiftJisFromUnicode(arg);
    return IsNextArgOptionValue(JobOptionArgInfos, argA);
}

//-----------------------------------------------------------------------------
//! @brief C2Nn コンバータにグローバルオプションを設定します。
//-----------------------------------------------------------------------------
bool C2NnConverter::SetOption(const wchar_t* options[])
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // グローバルオプションを初期化します。
    InitializeGlobalOption();

    //-----------------------------------------------------------------------------
    // グローバルオプションのユニコード文字列配列を解析します。
    RStringArray tokens;
    const int tokenCount = RGetStringArray(tokens, options);
    for (int tokenIdx = 0; tokenIdx < tokenCount; ++tokenIdx)
    {
        const std::string& curArg = tokens[tokenIdx];
        const OptionArgInfo* pInfo;
        int suffixNum;
        std::string value;
        status = ParseOptionArg(&pInfo, &suffixNum, &value, &tokenIdx, tokens,
            GlobalOptionArgInfos);
        if (!status)
        {
            break;
        }

        const std::string longName = pInfo->longName;
        if (longName == "silent")
        {
            m_IsSilent = true;
        }
        else
        {
            status = RStatus(RStatus::FAILURE, "Invalid global option: " + curArg); // RShowError
            break;
        }
    }

    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief 変換オプションにジョブオプションを設定します。
//-----------------------------------------------------------------------------
RStatus C2NnOption::SetJobOption(const wchar_t* options[])
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // ジョブオプションのユニコード文字列配列を解析します。
    RStringArray tokens;
    const int tokenCount = RGetStringArray(tokens, options);
    for (int tokenIdx = 0; tokenIdx < tokenCount; ++tokenIdx)
    {
        const std::string& curArg = tokens[tokenIdx];
        const OptionArgInfo* pInfo;
        int suffixNum;
        std::string value;
        status = ParseOptionArg(&pInfo, &suffixNum, &value, &tokenIdx, tokens,
            JobOptionArgInfos);
        RCheckStatus(status);

        const std::string longName = pInfo->longName;
        bool isValidValue = true;
        if (longName == "project-root")
        {
            m_ProjectRootPath = RGetFilePathWithEndingSlash(RGetFullFilePath(value, true));
        }
        else if (longName == "disable-file-info")
        {
            m_DisablesFileInfo = true;
        }
        else if (longName == "output")
        {
            m_OutputPath = RGetFullFilePath(value, true);
        }
        else if (longName == "name")
        {
            m_ElementName = value;
        }
        else if (longName == "magnify")
        {
            m_ConversionMagnify = static_cast<float>(atof(value.c_str()));
            isValidValue = (m_ConversionMagnify > 0.0f);
        }
        else if (longName == "disable-texture")
        {
            m_ConvertsTex = false;
        }
        else if (longName == "all-textures")
        {
            m_ConvertsAllTex = true;
        }
        else if (longName == "max-mipmap")
        {
            m_ForcesMaxMip = true;
        }
        else if (longName == "mip-gen-filter")
        {
            m_MipGenFilter = value;
            isValidValue = (value == "point" || value == "linear" || value == "cubic");
        }
        else if (longName == "linear-texture")
        {
            m_TexLinearFlag = GetLinearFlagFromOptString(value);
            isValidValue = (m_TexLinearFlag != C2NnOption::LinearFlag_Invalid);
        }
        else if (longName == "linear-vertex-color")
        {
            m_VtxColLinearFlag = GetLinearFlagFromOptString(value);
            isValidValue = (m_VtxColLinearFlag != C2NnOption::LinearFlag_Invalid);
        }
        else if (longName == "use-fcl-fts-ftp")
        {
            m_UsesFclFtsFtp = true;
        }
        else if (longName == "separate-material-anim")
        {
            m_SeparatesFma = true;
        }
        else if (longName == "comment")
        {
            m_CommentText = value;
        }
        else if (longName == "arrange-format")
        {
            m_ArrangesFormat = true;
        }
        else
        {
            status = RStatus(RStatus::FAILURE, "Invalid file option: " + curArg); // RShowError
        }

        if (status && !isValidValue)
        {
            status = RStatus(RStatus::FAILURE, "--" + longName + " is wrong: " + value); // RShowError
        }
        RCheckStatus(status);
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 変換オプションに入力ファイルパスを設定します。
//-----------------------------------------------------------------------------
RStatus C2NnOption::SetInputPath(const wchar_t* path)
{
    //-----------------------------------------------------------------------------
    // 引数のパス配列をフルパスに変換します。
    m_InputPath = RGetFullFilePath(RGetShiftJisFromUnicode(path), true);
    const std::string inExt = RGetExtensionFromFilePath(m_InputPath);
    if (!IsSupportedNw4cExtension(inExt))
    {
        return RStatus(RStatus::FAILURE, "Input file type is not supported: " + m_InputPath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 出力ファイルパスが指定されていなければ、
    // 入力ファイルと同じフォルダに拡張子を変えて出力します。
    if (m_OutputPath.empty())
    {
        m_OutputPath = RGetNoExtensionFilePath(m_InputPath);
    }

    //-----------------------------------------------------------------------------
    // 出力ファイルパスの末尾が / なら出力フォルダとみなします。
    if (!m_OutputPath.empty() && m_OutputPath[m_OutputPath.size() - 1] == '/')
    {
        m_OutputPath += RGetNoExtensionFilePath(RGetFileNameFromFilePath(m_InputPath));
    }

    //-----------------------------------------------------------------------------
    // 拡張子を含まない出力ファイルパスを取得します。
    const std::string outExt = RGetExtensionFromFilePath(m_OutputPath);
    m_NoExtOutputPath = m_OutputPath;
    m_IsBinaryFormat = true;
    if (IsNnExtension(outExt))
    {
        m_NoExtOutputPath = RGetNoExtensionFilePath(m_NoExtOutputPath);
        m_IsBinaryFormat = (outExt[outExt.size() - 1] == 'b');
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 変換オプションの入力ファイルの情報を表示します。
//-----------------------------------------------------------------------------
void C2NnOption::DisplayFileInfo(std::ostream& os) const
{
    os << "Input  : " << m_InputPath << endl;
    if (!m_ElementName.empty())
    {
        os << "Name   : " << m_ElementName << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief 入力ファイルを変換して NintendoSDK 中間ファイルを出力します。
//-----------------------------------------------------------------------------
RStatus C2NnConverter::ConvertSub(const wchar_t* path, const wchar_t* options[])
{
    //-----------------------------------------------------------------------------
    // メモリをクリアし、ジョブオプションを初期化します。
    Clear();
    ClearError();

    //-----------------------------------------------------------------------------
    // ジョブオプションを設定します。
    RStatus status = m_Option.SetJobOption(options);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 入力ファイルパスを設定します。
    status = m_Option.SetInputPath(path);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 入力ファイルの情報を表示します。
    if (!m_Option.m_IsSilent)
    {
        m_Option.DisplayFileInfo(cout);
    }

    //-----------------------------------------------------------------------------
    // ctex ファイルを 1 つだけ変換する場合の処理です。
    static const char* DisplayOutputTop = "Output : ";
    const std::string inExt = RGetExtensionFromFilePath(m_Option.m_InputPath);
    if (inExt == "ctex")
    {
        const std::string outExt = (m_Option.m_OutputPath != m_Option.m_NoExtOutputPath) ?
            RGetExtensionFromFilePath(m_Option.m_OutputPath) : "";
        if (!outExt.empty() && outExt.find("ftx") != 0)
        {
            return RStatus(RStatus::FAILURE, "Output file extension is wrong for " + inExt + " file: " + m_Option.m_OutputPath); // RShowError
        }
        const std::string outputPath = m_Option.m_NoExtOutputPath + "." +
            (m_Option.m_IsBinaryFormat ? "ftxb" : "ftxa");
        if (!m_Option.m_IsSilent)
        {
            cout << DisplayOutputTop << outputPath << endl;
            DisplayTextureLinearInfo(cout, m_Option);
        }
        return ConvertToFtx(&m_OutputFtxPaths, m_Option.m_InputPath, outputPath, m_Option);
    }

    //-----------------------------------------------------------------------------
    // 入力ファイルをリードします。
    CIntermediateFile inputFile(&status, m_Option.m_InputPath);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // コンバータ用エクスポートオプションを設定します。
    CExpOpt expOpt;
    status = SetExportOption(&expOpt, m_Option, inputFile);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 各ファイルタイプに変換します。
    bool isSingleFmaOutput = false;
    for (int fileTypeIdx = 0; fileTypeIdx < RExpOpt::FILE_TYPE_COUNT; ++fileTypeIdx)
    {
        const RExpOpt::FileType fileType = static_cast<RExpOpt::FileType>(fileTypeIdx);
        if (expOpt.m_OutFileFlag[fileType])
        {
            if (expOpt.UsesSingleFma() && expOpt.IsFclFtsFtp(fileType))
            {
                if (isSingleFmaOutput)
                {
                    continue;
                }
                isSingleFmaOutput = true;
            }
            const std::string outputPath = m_Option.m_NoExtOutputPath +
                expOpt.GetSuffix(fileType) + "." + expOpt.GetExtension(fileType);
            if (!m_Option.m_IsSilent)
            {
                cout << DisplayOutputTop << outputPath << endl;
                if (m_Option.m_ConversionMagnify != 1.0f)
                {
                    if (fileType == RExpOpt::FMD ||
                        fileType == RExpOpt::FSK ||
                        fileType == RExpOpt::FSN)
                    {
                        cout << "Magnify: " << m_Option.m_ConversionMagnify << endl;
                    }
                }
                if (fileType == RExpOpt::FMD ||
                    fileType == RExpOpt::FTP)
                {
                    DisplayTextureLinearInfo(cout, m_Option);
                }
                if (fileType == RExpOpt::FMD && m_Option.m_VtxColLinearFlag != C2NnOption::LinearFlag_None)
                {
                    cout << "LinearC: " << GetOptStringFromLinearFlag(m_Option.m_VtxColLinearFlag) << endl;
                }
            }

            if (fileType == RExpOpt::FMD)
            {
                status = ConvertToFmd(&expOpt, &m_OutputFtxPaths, inputFile, outputPath, m_Option);
            }
            else if (fileType == RExpOpt::FSK)
            {
                status = ConvertToFsk(&expOpt, inputFile, outputPath, m_Option);
            }
            else if (fileType == RExpOpt::FVB)
            {
                status = ConvertToFvb(&expOpt, inputFile, outputPath, m_Option);
            }
            else if (!expOpt.m_UsesFclFtsFtp && expOpt.IsFclFtsFtp(fileType))
            {
                status = ConvertToFma(&expOpt, &m_OutputFtxPaths, inputFile, outputPath, fileType, m_Option);
            }
            else if (fileType == RExpOpt::FCL)
            {
                status = ConvertToFcl(&expOpt, inputFile, outputPath, m_Option);
            }
            else if (fileType == RExpOpt::FTS)
            {
                status = ConvertToFts(&expOpt, inputFile, outputPath, m_Option);
            }
            else if (fileType == RExpOpt::FTP)
            {
                status = ConvertToFtp(&expOpt, &m_OutputFtxPaths, inputFile, outputPath, m_Option);
            }
            else if (fileType == RExpOpt::FSN)
            {
                status = ConvertToFsn(&expOpt, inputFile, outputPath, m_Option);
            }
            RCheckStatus(status);
        }
    }

    return RStatus::SUCCESS;
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief 入力ファイルを変換して NintendoSDK 中間ファイルを出力します。
//-----------------------------------------------------------------------------
bool C2NnConverter::Convert(const wchar_t* path, const wchar_t* options[])
{
    RStatus status = ConvertSub(path, options);
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief 出力したテクスチャファイルの情報を表示します。
//-----------------------------------------------------------------------------
void C2NnConverter::DisplayTextureInfo(std::ostream& os) const
{
    const int ftxCount = static_cast<int>(m_OutputFtxPaths.size());
    if (!m_Option.m_IsSilent && ftxCount != 0)
    {
        std::string msg = "Texture: " + RGetNumberString(ftxCount) + " ( ";
        for (int ftxIdx = 0; ftxIdx < ftxCount; ++ftxIdx)
        {
            msg += RGetNoExtensionFilePath(RGetFileNameFromFilePath(m_OutputFtxPaths[ftxIdx]));
            if (ftxIdx < ftxCount - 1)
            {
                msg += ", ";
            }
        }
        msg += " )";
        os << msg << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
//! @brief C2Nn コンバータのバージョンを返します。
//-----------------------------------------------------------------------------
int C2NnConverter::GetConverterVersion() const
{
    return (ConverterVersionMajor  << 24) |
           (ConverterVersionMinor  << 16) |
           (ConverterVersionMicro  <<  8) |
           (ConverterVersionBugfix      );
}

//-----------------------------------------------------------------------------
//! @brief C2Nn コンバータの使用方法を表示します。
//-----------------------------------------------------------------------------
void C2NnConverter::ShowUsage(
    std::ostream& os,
    const wchar_t* converterName,
    const wchar_t* cmdName,
    const bool isVersionOnly
) const
{
    os << RGetShiftJisFromUnicode(converterName) << " "
       << ConverterVersionMajor << "."
       << ConverterVersionMinor << "."
       << ConverterVersionMicro << "."
       << ConverterVersionBugfix << endl;
    if (isVersionOnly)
    {
        os << endl;
        os << "Copyright (C)Nintendo All rights reserved." << endl;
        os << endl;
        os << "These coded instructions, statements, and computer programs contain" << endl;
        os << "proprietary information of Nintendo of America Inc. and/or Nintendo" << endl;
        os << "Company Ltd., and are protected by Federal copyright law.  They may" << endl;
        os << "not be disclosed to third parties or copied or duplicated in any form," << endl;
        os << "in whole or in part, without the prior written consent of Nintendo." << endl;
        return;
    }
    os << "Usage: " << RGetShiftJisFromUnicode(cmdName) << " [INPUT_FILE] [Options]" << endl;
    os << endl;
    os << "Options:" << endl;
    os << "  --version                       Display the version information and exit." << endl;
    os << "  -h --help                       Display this help and exit." << endl;
    os << "  -s --silent                     Do not output progress messages." << endl;
    os << "  --args-file=ARGS_FILE           Import arguments from a file." << endl;
    os << "  --job-list=JOB_LIST_FILE        Job list file." << endl;
    os << "  --project-root=ROOT_PATH        Specify the project root path." << endl;
    os << "  --disable-file-info             Disable outputting <file_info>." << endl;
    os << "  -o --output=OUTPUT_FILE         Output file." << endl;
    os << "  -n --name=SOURCE_NAME           Model (animation) name to convert." << endl;
    os << "  -g --magnify=MAGNIFY            Magnification ratio." << endl;
    os << "  -d --disable-texture            Disable converting used textures" << endl;
    os << "                                  to ftx files." << endl;
    os << "  -a --all-textures               Convert all ctex files in an input folder" << endl;
    os << "                                  and 'Textures' folder." << endl;
    os << "  -x --max-mipmap                 Force creating all mipmap levels." << endl;
    os << "  --mip-gen-filter=FILTER         Mipmap generation filter." << endl;
    os << "  -l --linear-texture=LINEAR      Linear conversion flags of textures." << endl;
    os << "  -c --linear-vertex-color=LINEAR Linear conversion flags of vertex colors." << endl;
    os << "  --use-fcl-fts-ftp               Create files with different extensions" << endl;
    os << "                                  for each type of material animation." << endl;
    os << "  --separate-material-anim        Create separate fma files for each type" << endl;
    os << "                                  of material animation." << endl;
    os << "  --comment=COMMENT               Specify the comment text." << endl;
    os << endl;
    os << "Supported Input Files:" << endl;
    os << "  cmdl / ctex / cskla / cmdla / cmata / cmcla / cmtsa / cmtpa" << endl;
    os << "  ccam / clgt / cfog / cenv / cres" << endl;
    os << endl;
}

//-----------------------------------------------------------------------------
//! @brief C2Nn コンバータのコンストラクタです。
//-----------------------------------------------------------------------------
C2NnConverter::C2NnConverter(HINSTANCE hinstDLL)
: m_hDLL(hinstDLL)
{
    //cerr << "C2NnConverter()" << endl;

    //-----------------------------------------------------------------------------
    // 各ツールのフォルダを取得します。
    const std::string converterPath = RGetFullFilePath(GetModuleFilePath(hinstDLL), true);
    const std::string converterFolderPath = RGetFolderFromFilePath(converterPath) + "/";

    m_GraphicsToolsPath = GetGraphicsToolsPath(converterFolderPath);
    //cerr << "gfx tools: " << m_GraphicsToolsPath << endl;
    m_3dToolsPath = Get3dToolsPath(converterFolderPath);
    //cerr << "3d tools : " << m_3dToolsPath << endl;

    const std::string textureEncoderPath = m_GraphicsToolsPath + "TextureConverterEncoder.dll";
    m_TextureEncoder.SetDllPath(textureEncoderPath.c_str());
}

//-----------------------------------------------------------------------------
//! @brief C2Nn コンバータのデストラクタです。
//-----------------------------------------------------------------------------
C2NnConverter::~C2NnConverter()
{
    //cerr << "~C2NnConverter()" << endl;
}

//=============================================================================
// c2nn ネームスペースを終了します。
//=============================================================================
} // c2nn
} // g3dTool
} // nn

