﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// variables
// Process ParseArgument MainProc

//=============================================================================
// include
//=============================================================================
#include "C2NnLib.h"

using namespace std;
using namespace nn::gfx::tool::dcc;
using namespace nn::g3dTool::c2nn;

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
// constants
//=============================================================================
const char* ConverterName = "3dCmdlConverter"; //!< コンバータ名です。

//=============================================================================
//! @brief ジョブ引数の構造体です。
//=============================================================================
struct JobArgument
{
    RStringArray options; //!< オプション配列です。
    std::string filePath; //!< 入力ファイルパスです。
};

//=============================================================================
//! @brief コンバータ引数の構造体です。
//=============================================================================
struct ConverterArgument
{
    RStringArray globalOptions; //!< グローバルオプション配列です。
    std::vector<JobArgument> jobArgs; //!< ジョブ引数配列です。
};

//-----------------------------------------------------------------------------
//! @brief 構造化例外の変換関数です。
//-----------------------------------------------------------------------------
void SeTranslatorFunction(unsigned int /*code*/, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; // 標準 C++ の例外を発生させます。
}

//-----------------------------------------------------------------------------
//! @brief エラーを表示します。
//-----------------------------------------------------------------------------
void RShowError(const char* message)
{
    cerr << ConverterName << ": Error: " << message << endl;
}

//-----------------------------------------------------------------------------
//! @brief 変換処理を実行します。
//!
//! @param[in,out] pConverter コンバータへのポインタです。
//! @param[in] convArg コンバータ引数です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool Process(C2NnConverter* pConverter, const ConverterArgument& convArg)
{
    //-----------------------------------------------------------------------------
    // グローバルオプションを設定します。
    if (!pConverter->SetOption(RUnicodeCArray(convArg.globalOptions).GetCArray()))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // ジョブについてループします。
    bool isSucceeded = true;
    for (size_t jobIdx = 0; jobIdx < convArg.jobArgs.size(); ++jobIdx)
    {
        const JobArgument& jobArg = convArg.jobArgs[jobIdx];
        if (!pConverter->Convert(
            RGetUnicodeFromShiftJis(jobArg.filePath).c_str(),
            RUnicodeCArray(jobArg.options).GetCArray()))
        {
            isSucceeded = false;
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // 出力したテクスチャファイルの情報を表示します。
    if (isSucceeded)
    {
        pConverter->DisplayTextureInfo(cout);
    }

    //-----------------------------------------------------------------------------
    // コンバータのメモリをクリアします。
    pConverter->Clear();

    return isSucceeded;
}

//-----------------------------------------------------------------------------
//! @brief 引数ファイルが指定されていればファイル内の引数を展開します。
//!
//! @param[out] pDstArgs 引数配列を格納します。
//! @param[in] srcArgs 展開前の引数配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ExpandArguments(RStringArray* pDstArgs, const RStringArray& srcArgs)
{
    RStatus status;
    RStringArray& dstArgs = *pDstArgs;
    dstArgs.clear();
    for (int argIdx = 0; argIdx < static_cast<int>(srcArgs.size()); ++argIdx)
    {
        const std::string& arg = srcArgs[argIdx];
        if (arg == "--args-file" || arg.find("--args-file=") == 0)
        {
            std::string argsFilePath;
            status = GetOptionArgValue(&argsFilePath, &argIdx, srcArgs);
            if (status)
            {
                status = GetArgTokensFromFile(&dstArgs, argsFilePath);
            }
        }
        else
        {
            dstArgs.push_back(arg);
        }
        RCheckStatus(status);
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ジョブリストファイルの 1 行を解析します。
//!
//! @param[in,out] pConvArg コンバータ引数へのポインタです。
//! @param[in] converter コンバータです。
//! @param[in] commonJobArg 共通ジョブ引数です。
//! @param[in] lineStr 1 行の文字列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseJobListLine(
    ConverterArgument* pConvArg,
    const C2NnConverter& converter,
    const JobArgument& commonJobArg,
    const std::string& lineStr
)
{
    RStringArray srcArgs;
    GetArgTokensFromLine(&srcArgs, lineStr);
    RStringArray args;
    RStatus status = ExpandArguments(&args, srcArgs);
    RCheckStatus(status);

    JobArgument jobArg = commonJobArg;
    for (int argIdx = 0; argIdx < static_cast<int>(args.size()); ++argIdx)
    {
        const std::string& arg = args[argIdx];
        const std::wstring argW = RGetUnicodeFromShiftJis(arg);
        if (arg.find('-') == 0)
        {
            if (converter.IsGlobalOptionArg(argW.c_str()))
            {
                return RStatus(RStatus::FAILURE, "Wrong option is specified in the job list file: " + arg); // RShowError
            }
            else
            {
                jobArg.options.push_back(arg);
                if (converter.IsNextArgJobOptionValue(argW.c_str()))
                {
                    ++argIdx;
                    if (argIdx < static_cast<int>(args.size()))
                    {
                        jobArg.options.push_back(args[argIdx]);
                    }
                }
            }
        }
        else
        {
            const std::string inputPath = RDequoteString(arg);
            if (!jobArg.filePath.empty())
            {
                return RStatus(RStatus::FAILURE, "Multi input files are specified: " + inputPath); // RShowError
            }
            jobArg.filePath = RGetFullFilePath(inputPath, true);
        }
    }

    if (jobArg.filePath.empty())
    {
        return RStatus(RStatus::FAILURE, "No input file: " + lineStr); // RShowError
    }
    pConvArg->jobArgs.push_back(jobArg);
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ジョブリストファイルを解析します。
//!
//! @param[in,out] pConvArg コンバータ引数へのポインタです。
//! @param[in] converter コンバータです。
//! @param[in] commonJobArg 共通ジョブ引数です。
//! @param[in] jobListFilePath ジョブリストファイルのパスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseJobListFile(
    ConverterArgument* pConvArg,
    const C2NnConverter& converter,
    const JobArgument& commonJobArg,
    const std::string& jobListFilePath
)
{
    RStatus status;

    ifstream ifs(jobListFilePath.c_str());
    if (!ifs)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + jobListFilePath); // RShowError
    }

    std::string lineStr;
    while (!ifs.eof())
    {
        getline(ifs, lineStr);
        lineStr = RTrimString(lineStr);
        if (!lineStr.empty())
        {
            if (lineStr.find('#') != 0)
            {
                status = ParseJobListLine(pConvArg, converter, commonJobArg, lineStr);
                RCheckStatus(status);
            }
        }
    }
    ifs.close();
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 引数を解析します。
//!
//! @param[out] pConvArg コンバータ引数へのポインタです。
//! @param[in] converter コンバータです。
//! @param[in] argc 引数の数です。
//! @param[in] argv 引数の配列です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ParseArgument(
    ConverterArgument* pConvArg,
    const C2NnConverter& converter,
    int argc,
    char* argv[]
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // コマンド名を取得します。
    const std::wstring converterName = RGetUnicodeFromShiftJis(ConverterName);
    const std::wstring cmdName = RGetUnicodeFromShiftJis(RGetFileNameFromFilePath(argv[0]));

    //-----------------------------------------------------------------------------
    // 引数ファイルで指定された引数を展開します。
    RStringArray srcArgs;
    for (int argIdx = 1; argIdx < argc; ++argIdx)
    {
        srcArgs.push_back(std::string(argv[argIdx]));
    }
    RStringArray args;
    status = ExpandArguments(&args, srcArgs);
    if (!status)
    {
        RShowError(status.GetMessage().c_str());
        return false;
    }

    //-----------------------------------------------------------------------------
    // 引数を解析します。
    JobArgument commonJobArg;
    RStringArray jobListFilePaths;
    for (int argIdx = 0; argIdx < static_cast<int>(args.size()); ++argIdx)
    {
        const std::string& arg = args[argIdx];
        const std::wstring argW = RGetUnicodeFromShiftJis(arg);
        if (arg == "--version")
        {
            converter.ShowUsage(cout, converterName.c_str(), cmdName.c_str(), true);
            exit(0);
        }
        else if (arg == "-h" || arg == "--help")
        {
            converter.ShowUsage(cout, converterName.c_str(), cmdName.c_str(), false);
            exit(0);
        }
        else if (arg == "--job-list" || arg.find("--job-list=") == 0)
        {
            std::string jobListFilePath;
            status = GetOptionArgValue(&jobListFilePath, &argIdx, args);
            if (status)
            {
                jobListFilePaths.push_back(jobListFilePath);
            }
        }
        else if (arg.find('-') == 0)
        {
            if (converter.IsGlobalOptionArg(argW.c_str()))
            {
                pConvArg->globalOptions.push_back(arg);
                if (converter.IsNextArgGlobalOptionValue(argW.c_str()))
                {
                    ++argIdx;
                    if (argIdx < static_cast<int>(args.size()))
                    {
                        pConvArg->globalOptions.push_back(args[argIdx]);
                    }
                    else
                    {
                        status = RStatus(RStatus::FAILURE, "Option has no value: " + arg); // RShowError
                    }
                }
            }
            else
            {
                commonJobArg.options.push_back(arg);
                if (converter.IsNextArgJobOptionValue(argW.c_str()))
                {
                    ++argIdx;
                    if (argIdx < static_cast<int>(args.size()))
                    {
                        commonJobArg.options.push_back(args[argIdx]);
                    }
                    else
                    {
                        status = RStatus(RStatus::FAILURE, "Option has no value: " + arg); // RShowError
                    }
                }
            }
        }
        else
        {
            const std::string inputPath = RDequoteString(arg);
            if (!commonJobArg.filePath.empty())
            {
                RShowError(("Multi input files are specified: " + inputPath).c_str());
                return false;
            }
            commonJobArg.filePath = RGetFullFilePath(inputPath, true);
        }

        if (!status)
        {
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // ジョブリストファイル群を解析します。
    if (status)
    {
        if (!jobListFilePaths.empty())
        {
            for (size_t jobIdx = 0; jobIdx < jobListFilePaths.size(); ++jobIdx)
            {
                status = ParseJobListFile(pConvArg, converter, commonJobArg, jobListFilePaths[jobIdx]);
                if (!status)
                {
                    break;
                }
            }
            if (status && pConvArg->jobArgs.empty())
            {
                status = RStatus(RStatus::FAILURE, "Job list is empty"); // RShowError
            }
        }
        else if (!commonJobArg.filePath.empty())
        {
            pConvArg->jobArgs.push_back(commonJobArg);
        }
    }

    if (!status)
    {
        RShowError(status.GetMessage().c_str());
        return false;
    }

    if (pConvArg->jobArgs.empty())
    {
        converter.ShowUsage(cout, converterName.c_str(), cmdName.c_str(), false);
        return false;
    }

    return true;
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief メイン関数です。
//-----------------------------------------------------------------------------
int main(int argc, char* argv[]) // MainProc
{
    //-----------------------------------------------------------------------------
    // 構造化例外の変換関数を設定します。
    _set_se_translator(SeTranslatorFunction); // このスレッドでのみ有効です。

    //-----------------------------------------------------------------------------
    // parse argument
    C2NnConverter converter(nullptr);
    ConverterArgument convArg;
    if (!ParseArgument(&convArg, converter, argc, argv))
    {
        return 1;
    }

    //-----------------------------------------------------------------------------
    // process
    if (!Process(&converter, convArg))
    {
        const std::string msg = converter.GetErrorShiftJisString();
        if (!msg.empty())
        {
            RShowError(msg.c_str());
            //system("pause"); // デバッガでエラー発生時にコンソールを閉じないようにします。
        }
        return 1;
    }
    return 0;
}

