﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinUserData.h>

namespace nn {
namespace g3dTool {

namespace {
uint32_t tbl_user_data_type_table[] =
{
    nn::gfx::ResUserData::Type_Float,
    nn::gfx::ResUserData::Type_Int,
    nn::gfx::ResUserData::Type_Stream,
    nn::gfx::ResUserData::Type_String,
};
}

void BinUserData::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_user_data& elem)
{
    pCtx->SetStr(elem.name.value.c_str());
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    // user_string, user_wstring の文字列を登録
    if( m_pElem->user_string )
    {
        for ( auto strIter = m_pElem->user_string->string_array.cbegin(); strIter != m_pElem->user_string->string_array.cend(); ++strIter )
        {
            pCtx->SetStr( strIter->stream.c_str() );
        }
    }
    if( m_pElem->user_wstring )
    {
        for ( auto strIter = m_pElem->user_wstring->wstring_array.cbegin(); strIter != m_pElem->user_wstring->wstring_array.cend(); ++strIter )
        {
            std::string str = nw::g3d::tool::util::ToUTF8( strIter->stream.c_str() );
            pCtx->SetStr( str.c_str() );
        }
    }
}

void BinUserData::CalculateSize()
{
    size_t dataCount = 0;

    if (m_pElem->type == nw::g3d::tool::g3dif::StreamTypeInt)
    {
        dataCount = m_pElem->user_int->count.value;
        m_Chunk[ChunkType_Data].size = dataCount * sizeof(uint32_t);
    }
    else if (m_pElem->type == nw::g3d::tool::g3dif::StreamTypeFloat)
    {
        dataCount = m_pElem->user_float->count.value;
        m_Chunk[ChunkType_Data].size = dataCount * sizeof(uint32_t);
    }
    else if (m_pElem->type == nw::g3d::tool::g3dif::StreamTypeString)
    {
        dataCount = m_pElem->user_string->count.value;
        m_Chunk[ChunkType_StrPtrArray].size = sizeof( nn::util::BinPtrToString ) * m_pElem->user_string->string_array.size();
    }
    else if (m_pElem->type == nw::g3d::tool::g3dif::StreamTypeWString)
    {
        dataCount = m_pElem->user_wstring->count.value;
        m_Chunk[ChunkType_StrPtrArray].size = sizeof( nn::util::BinPtrToString ) * m_pElem->user_wstring->wstring_array.size();
    }
    else if (m_pElem->type == nw::g3d::tool::g3dif::StreamTypeByte)
    {
        dataCount = nw::g3d::tool::util::Align(m_pElem->user_stream->stream.count) >> 2;

        // サイズ用の領域確保
        dataCount += 1;

        m_Chunk[ChunkType_Data].size = dataCount * sizeof(uint32_t);
    }

    m_DataCount = static_cast<int>(dataCount);

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinUserData::Convert( std::shared_ptr<Context> pCtx )
{
    nn::gfx::ResUserDataData& userdata = *GetPtr<nn::gfx::ResUserDataData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );


    pCtx->LinkStr( &userdata.pName, nn::util::string_view( m_pElem->name.value.c_str() ) );

    userdata.count = static_cast<uint16_t>(m_DataCount);
    userdata.type = static_cast<uint8_t>( tbl_user_data_type_table[m_pElem->type] );
    if( m_pElem->type == nw::g3d::tool::g3dif::StreamTypeWString )
    {
        userdata.type = nn::gfx::ResUserData::Type_String;
    }

    if (userdata.count != 0)
    {
        if (userdata.type == nn::gfx::ResUserData::Type_Int)
        {
            void* pDst = GetPtr<void>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Data].offset);
            memcpy(pDst, m_pElem->user_int->rawdata.get(), sizeof(int32_t) * userdata.count);
            pCtx->LinkPtr( &userdata.pData, pDst );
        }
        else if (userdata.type == nn::gfx::ResUserData::Type_Float)
        {
            void* pDst = GetPtr<void>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Data].offset);
            memcpy(pDst, m_pElem->user_float->rawdata.get(), sizeof(float) * userdata.count);
            pCtx->LinkPtr( &userdata.pData, pDst );
        }
        else if (userdata.type == nn::gfx::ResUserData::Type_String)
        {
            nn::util::BinPtrToString* pStrPtrArray = GetPtr< nn::util::BinPtrToString >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_StrPtrArray].offset);
            if( m_pElem->user_string )
            {
                for (auto iter = m_pElem->user_string->string_array.cbegin(); iter != m_pElem->user_string->string_array.cend(); ++iter, ++pStrPtrArray)
                {
                    pCtx->LinkStr( pStrPtrArray, iter->stream.c_str() );
                }
            }
            if( m_pElem->user_wstring )
            {
                for (auto iter = m_pElem->user_wstring->wstring_array.cbegin(); iter != m_pElem->user_wstring->wstring_array.cend(); ++iter, ++pStrPtrArray)
                {
                    std::string str = nw::g3d::tool::util::ToUTF8( iter->stream.c_str() );
                    pCtx->LinkStr( pStrPtrArray, str.c_str() );
                }
            }
            pCtx->LinkPtr( &userdata.pData,
                GetPtr< nn::util::BinPtrToString >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_StrPtrArray].offset) );
        }
        else if (userdata.type == nn::gfx::ResUserData::Type_Stream)
        {
            userdata.count = static_cast<uint32_t>(m_pElem->user_stream->size.value);
            void* pDst = GetPtr< void >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Data].offset);
            memcpy(pDst, m_pElem->user_stream->stream.rawdata.get(), m_pElem->user_stream->stream.count);
            pCtx->LinkPtr( &userdata.pData, pDst );
        }
    }
}

}
}
