﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinSkeleton.h>
#include <algorithm>
#include <util/UtilMath.h>
#include <g3dif/Skeleton.h>
#include <nn/nn_Macro.h>
#include <util/UtilError.h>

namespace nn {
namespace g3dTool {

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    bone, billboard, binary,
    nn::g3d::ResBone::Flag_BillboardNone,
    nn::g3d::ResBone::Flag_BillboardWorldViewVector,
    nn::g3d::ResBone::Flag_BillboardWorldViewPoint,
    nn::g3d::ResBone::Flag_BillboardScreenViewVector,
    nn::g3d::ResBone::Flag_BillboardScreenViewPoint,
    nn::g3d::ResBone::Flag_BillboardYaxisViewVector,
    nn::g3d::ResBone::Flag_BillboardYaxisViewPoint
);

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    skeleton_info, scale_mode, binary,
    nn::g3d::ResSkeleton::ScaleMode_Std,
    nn::g3d::ResSkeleton::ScaleMode_Maya,
    nn::g3d::ResSkeleton::ScaleMode_Softimage
    );

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    skeleton_info, rotate_mode, binary,
    nn::g3d::ResSkeleton::RotateMode_EulerXyz,
    nn::g3d::ResSkeleton::RotateMode_Quat
    );

//--------------------------------------------------------------------------------------------------

void BinBone::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_bone& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_DicUserData.Build(pCtx, elem.user_data_array.size());
    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr(elem.name.value.c_str());

    // user_data
    int index = 0;
    for (auto iter = m_pElem->user_data_array.cbegin();
        iter != m_pElem->user_data_array.cend(); ++iter, ++index)
    {
        m_DicUserData.SetName(index, iter->name.value);
    }
}

void BinBone::CalculateSize()
{
    // 構造体のサイズは親のサイズに含まれているのでここでのサイズ計算は行わない。

    m_Chunk[ChunkType_UserDataData].size = sizeof( nn::gfx::ResUserDataData ) * m_UserDataArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinBone::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_UserDataData ].offset;
    for( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinBone::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResBoneData& bone = *GetPtr<nn::g3d::ResBoneData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    pCtx->LinkStr( &bone.pName, nn::util::string_view( m_pElem->name.value.c_str() ) );

    bone.index = static_cast<uint16_t>( m_Index );
    bone.smoothMtxIndex = static_cast<int16_t>(m_pElem->matrix_index->data.get()[0]);
    bone.rigidMtxIndex = static_cast<int16_t>(m_pElem->matrix_index->data.get()[1]);

    bone.parentIndex = static_cast<uint16_t>( (m_pElem->parent_index == -1) ? std::numeric_limits<uint16_t>::max() : m_pElem->parent_index );

    nn::Bit32 flag = 0;
    if (m_pElem->visibility.value)
    {
        flag |= nn::g3d::ResBone::Flag_Visibility;
    }
    if (m_pElem->billboard.value != nw::g3d::tool::g3dif::elem_bone::none)
    {
        flag |= ToEnum_binary(m_pElem->billboard.value);
    }
    if (m_pElem->scale_compensate.value)
    {
        flag |= nn::g3d::ResBone::Flag_SegmentScaleCompensate;
    }

    flag |= ConvertSRT(bone);
    flag |= m_MirroringState;
    bone.flag = flag;

    bone.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    pCtx->LinkPtr( &bone.pUserDataArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_UserDataData].offset) );
    m_DicUserData.ConvertData(pCtx, bone.pUserDataDic, m_UserDataArray);
}

nn::Bit32 BinBone::ConvertSRT(nn::g3d::ResBoneData &bone)
{
    nn::Bit32 flag = 0;

    // Scale : 0 は最小スケールに置き換える
    nn::util::Float3& scale = bone.scale;
    scale.x = nw::g3d::tool::g3dif::ScaleAdjuster::ZeroToMin(m_pElem->scale->x);
    scale.y = nw::g3d::tool::g3dif::ScaleAdjuster::ZeroToMin(m_pElem->scale->y);
    scale.z = nw::g3d::tool::g3dif::ScaleAdjuster::ZeroToMin(m_pElem->scale->z);
    if (scale.x == scale.y && scale.y == scale.z)
    {
        flag |= nn::g3d::ResBone::Flag_ScaleUniform;
    }
    if (scale.x * scale.y * scale.z == 1.0f)
    {
        flag |= nn::g3d::ResBone::Flag_ScaleVolumeOne;
    }

    // Rotate
    nn::util::Float4& rotate = bone.rotate.quat;
    rotate.x = m_pElem->rotate->x;
    rotate.y = m_pElem->rotate->y;
    rotate.z = m_pElem->rotate->z;
    rotate.w = m_pElem->rotate->w;
    if (m_RotateMode == nw::g3d::tool::g3dif::elem_skeleton_info::euler_xyz)
    {
        flag |= nn::g3d::ResBone::RotateMode_EulerXyz;
        rotate.x = nw::g3d::tool::util::DegToRad(rotate.x);
        rotate.y = nw::g3d::tool::util::DegToRad(rotate.y);
        rotate.z = nw::g3d::tool::util::DegToRad(rotate.z);
        rotate.w = 1.0f;
    }
    else if (m_RotateMode == nw::g3d::tool::g3dif::elem_skeleton_info::quaternion)
    {
        flag |= nn::g3d::ResBone::RotateMode_Quat;
    }

    if (rotate.x== 0.0f && rotate.y == 0.0f && rotate.z == 0.0f)
    {
        flag |= nn::g3d::ResBone::Flag_RotateZero;
    }

    // Translate
    nn::util::Float3& translate = bone.translate;
    translate.x = m_pElem->translate->x;
    translate.y = m_pElem->translate->y;
    translate.z = m_pElem->translate->z;
    if (translate.x == 0.0f && translate.y == 0.0f && translate.z == 0.0f)
    {
        flag |= nn::g3d::ResBone::Flag_TranslateZero;
    }

    if (m_RotateMode != nw::g3d::tool::g3dif::elem_skeleton_info::quaternion && m_RotateMode != nw::g3d::tool::g3dif::elem_skeleton_info::euler_xyz)
    {
        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_InvalidValue", "rotate mode");
    }

    return flag;
}

//--------------------------------------------------------------------------------------------------

void BinSkeleton::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_skeleton& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_DicBone.Build(pCtx, elem.bone_array.size());
    m_BoneArray.resize(elem.bone_array.size());
    SetParentBlockArray(m_BoneArray, this);
    BuildArray(pCtx, m_BoneArray, elem.bone_array);

    // bone
    int boneIndex = 0;
    m_NumSmoothBone = 0;
    m_NumRigidBone = 0;

    const nw::g3d::tool::g3dif::elem_skeleton_info& info = elem.skeleton_info;
    nw::g3d::tool::g3dif::elem_skeleton_info::enum_rotate_mode rotateMode = info.rotate_mode.value;
    for (auto iter = m_pElem->bone_array.cbegin();
        iter != m_pElem->bone_array.cend(); ++iter, ++boneIndex)
    {
        m_DicBone.SetName(boneIndex, iter->name.value);

        BinBone& bone = m_BoneArray[boneIndex];
        bone.SetIndex(boneIndex);
        bone.SetRotateMode(rotateMode);

        if (iter->matrix_index->data.get()[0] >= 0)
        {
            ++m_NumSmoothBone;
        }

        if (iter->matrix_index->data.get()[1] >= 0)
        {
            ++m_NumRigidBone;
        }
    }

    // 中間ファイルにミラーリング情報をバイナリに含める指定がされている場合のみ呼ぶ。
    if (info.motion_mirroring_enable.value == true)
    {
        BuildMirroringInformation();
        m_IsMirroringEnabled = true;
    }
}

void BinSkeleton::CalculateInvMatrix(nn::util::Matrix4x3f* pOut, const nw::g3d::tool::g3dif::elem_bone& elemBone)
{
    // 自身が逆行列を持っていない場合
    if ((elemBone.matrix_index.value.data.get()[0] == -1) && (elemBone.matrix_index.value.data.get()[0] == -1))
    {
        if (elemBone.parent_index == -1)
        {
            // 親がいない場合は単位行列を返す
            nn::util::Matrix4x3f identity;
            *pOut = identity.Identity();
            return;
        }
        else
        {
            // 親がいる場合は親の逆行列を問い合わせて、自身の RT を元に自身の逆行列を計算する
            nw::g3d::tool::g3dif::elem_bone elemParentBone = m_pElem->bone_array[elemBone.parent_index];
            CalculateInvMatrix(pOut, elemParentBone);

            // 親に対する自身の逆行列を求める
            nn::util::Matrix4x3f tmpInvMtx;
            tmpInvMtx = tmpInvMtx.Identity();

            nn::util::Float3 rotateRad;
            nn::util::Vector3fType euler;
            rotateRad.x = nw::g3d::tool::util::DegToRad(elemBone.rotate.value.x);
            rotateRad.y = nw::g3d::tool::util::DegToRad(elemBone.rotate.value.y);
            rotateRad.z = nw::g3d::tool::util::DegToRad(elemBone.rotate.value.z);
            VectorLoad(&euler, rotateRad);
            MatrixSetRotateXyz(&tmpInvMtx, euler);
            nn::util::Vector3fType translate;
            VectorLoad(&translate, elemBone.translate.value.a);
            tmpInvMtx.SetAxisW(translate);
            tmpInvMtx.Inverse();

            nn::util::Matrix4x3f invParentMtx = *pOut;
            *pOut = invParentMtx * tmpInvMtx;
        }
    }
    else
    {
        nw::g3d::tool::util::Mtx34_t nwInvModelMatrix = elemBone.inv_model_matrix.Get();
        nn::util::Matrix4x3f invModelMatrix(
            nwInvModelMatrix.m00, nwInvModelMatrix.m10, nwInvModelMatrix.m20,
            nwInvModelMatrix.m01, nwInvModelMatrix.m11, nwInvModelMatrix.m21,
            nwInvModelMatrix.m02, nwInvModelMatrix.m12, nwInvModelMatrix.m22,
            nwInvModelMatrix.m03, nwInvModelMatrix.m13, nwInvModelMatrix.m23);
        *pOut = invModelMatrix;
    }
}

// value が 0±tolerance かどうかを返す
bool EqualZero(float value, float tolerance)
{
    return ( fabsf(value) < tolerance );
}

// ミラーリングテーブルの作成と、各ボーンの MirroringState の決定、スケルトンの MirroringMode の決定を行う。
void BinSkeleton::BuildMirroringInformation()
{
    // オイラー角のみミラーリングに対応
    if (m_pElem->skeleton_info.rotate_mode.value != nw::g3d::tool::g3dif::elem_skeleton_info::euler_xyz)
    {
        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_XML_INVALID_USERDATA, "Identifier_InvalidRotateModeMirroring");
    }

    // スケルトンが収まる体積の立方根の 1% を、ミラー対象ボーンの位置判定の許容誤差としておく(要望に応じて緩めてもよい)。
    float maxX = std::numeric_limits<float>::min();
    float maxY = std::numeric_limits<float>::min();
    float maxZ = std::numeric_limits<float>::min();
    float minX = std::numeric_limits<float>::max();
    float minY = std::numeric_limits<float>::max();
    float minZ = std::numeric_limits<float>::max();
    for (int boneIndex = 0; boneIndex < static_cast<int>(m_pElem->bone_array.size()); ++boneIndex)
    {
        nw::g3d::tool::g3dif::elem_bone elemBone = m_pElem->bone_array[boneIndex];
        nn::util::Matrix4x3f invMtx, searchInvMtx;
        CalculateInvMatrix(&invMtx, elemBone);
        invMtx.Inverse();
        maxX = std::max(maxX, invMtx.GetAxisW().GetX());
        maxY = std::max(maxY, invMtx.GetAxisW().GetY());
        maxZ = std::max(maxZ, invMtx.GetAxisW().GetZ());
        minX = std::min(minX, invMtx.GetAxisW().GetX());
        minY = std::min(minY, invMtx.GetAxisW().GetY());
        minZ = std::min(minZ, invMtx.GetAxisW().GetZ());
    }
    float translateTolerance = cbrtf((maxX - minX) * (maxY - minY) * (maxZ - minZ)) * 0.01f;
    if (translateTolerance < 0.01f)
    {
        translateTolerance = 0.01f;
    }
    float rotateTolerance = 0.1f;

    // Center ボーンのミラー方向が変更されたかどうかのフラグ。一度変更されるとその他の方向に変更した場合エラーになる。
    bool isCenterRotated = false;
    m_MirroringBoneTable.resize(m_pElem->bone_array.size());
    for (int boneIndex = 0; boneIndex < static_cast<int>(m_pElem->bone_array.size()); ++boneIndex)
    {
        nw::g3d::tool::g3dif::elem_bone elemBone = m_pElem->bone_array[boneIndex];
        BinBone& binBone = m_BoneArray[boneIndex];
        int parentIndex = (elemBone.parent_index < 0) ? 0 : elemBone.parent_index;
        nw::g3d::tool::g3dif::elem_bone elemParentBone = m_pElem->bone_array[parentIndex];
        BinBone& binParentBone = m_BoneArray[parentIndex];

        int mirroringBoneIndex = -1;
        switch (elemBone.side.value)
        {
        case nw::g3d::tool::g3dif::elem_bone::side_none:
            {
                // none はデフォルト値に設定
                binBone.SetMirroringState(nn::g3d::ResBone::MirroringState_CenterPreRotate);
                //PRINT_LOG("None : %hs", elemBone.name.value.c_str());
            }
            break;
        case nw::g3d::tool::g3dif::elem_bone::side_center:
            {
                mirroringBoneIndex = boneIndex;

                // 親のボーンステートで場合分け
                switch (binParentBone.GetMirroringState())
                {
                case nn::g3d::ResBone::MirroringState_CenterPreRotate:
                    {
                        // PreRotate の検出。Y と Z に回転が入っていない場合は軸は変わらない。
                        if (EqualZero(elemBone.rotate.value.y, rotateTolerance) && EqualZero(elemBone.rotate.value.z, rotateTolerance))
                        {
                            binBone.SetMirroringState(nn::g3d::ResBone::MirroringState_CenterPreRotate);
                            //PRINT_LOG("CenterPreRotate : %hs", elemBone.name.value.c_str());
                            break;
                        }
                        else
                        {
                            float mod90x = fmod(fabsf(elemBone.rotate.value.x), 90.f);
                            float mod180x = fmod(fabsf(elemBone.rotate.value.x), 180.f);
                            float mod90y = fmod(fabsf(elemBone.rotate.value.y), 90.f);
                            float mod180y = fmod(fabsf(elemBone.rotate.value.y), 180.f);
                            float mod90z = fmod(fabsf(elemBone.rotate.value.z), 90.f);
                            float mod180z = fmod(fabsf(elemBone.rotate.value.z), 180.f);
                            // Y と Z の回転は 90 の倍数である必要がある
                            if ((EqualZero(mod90y, rotateTolerance) || EqualZero(90.f - mod90y, rotateTolerance)) && (EqualZero(mod90z, rotateTolerance) || EqualZero(90.f - mod90z, rotateTolerance)))
                            {
                                // Y と Z の回転が 0 と 180 で構成される場合は X 方向のミラー。
                                if ((EqualZero(mod180y, rotateTolerance) || EqualZero(180.f - mod180y, rotateTolerance)) && (EqualZero(mod180z, rotateTolerance) || EqualZero(180.f - mod180z, rotateTolerance)))
                                {
                                    binBone.SetMirroringState(nn::g3d::ResBone::MirroringState_CenterPreRotate);
                                    //PRINT_LOG("CenterPreRotate : %hs", elemBone.name.value.c_str());
                                    break;
                                }

                                // X が 0 か 180 の倍数の場合は Y 方向のミラー。
                                if (EqualZero(mod180x, rotateTolerance) || EqualZero(180.f - mod180x, rotateTolerance))
                                {
                                    if (isCenterRotated == true && m_MirroringMode != nn::g3d::ResSkeleton::MirroringMode_XY)
                                    {
                                        // 既に別のミラー方向に変更されている場合はエラー
                                        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SKELETON_INVALID_BONE, "Identifier_InvalidRotateValueMirroring", elemBone.name.value.c_str());
                                    }
                                    m_MirroringMode = nn::g3d::ResSkeleton::MirroringMode_XY;
                                    binBone.SetMirroringState(nn::g3d::ResBone::MirroringState_CenterRotate);
                                    isCenterRotated = true;
                                    //PRINT_LOG("CenterRotateY : %hs", elemBone.name.value.c_str());
                                    break;
                                }
                                // X が 90 の倍数(0,180の倍数以外)の場合は Z 方向のミラー。
                                else if (EqualZero(mod90x, rotateTolerance) || EqualZero(90.f - mod90x, rotateTolerance))
                                {
                                    if (isCenterRotated == true && m_MirroringMode != nn::g3d::ResSkeleton::MirroringMode_XZ)
                                    {
                                        // 既に別のミラー方向に変更されている場合はエラー
                                        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SKELETON_INVALID_BONE, "Identifier_InvalidRotateValueMirroring", elemBone.name.value.c_str());
                                    }
                                    m_MirroringMode = nn::g3d::ResSkeleton::MirroringMode_XZ;
                                    binBone.SetMirroringState(nn::g3d::ResBone::MirroringState_CenterRotate);
                                    isCenterRotated = true;
                                    //PRINT_LOG("CenterRotateZ : %hs", elemBone.name.value.c_str());
                                    break;
                                }
                            }
                        }
                        // 上記のどの条件にも合致しない場合はエラー
                        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SKELETON_INVALID_BONE, "Identifier_InvalidRotateValueMirroring", elemBone.name.value.c_str());
                    }
                    break;
                case nn::g3d::ResBone::MirroringState_CenterRotate:
                case nn::g3d::ResBone::MirroringState_CenterPostRotate:
                    {
                        // ミラー方向が変わる回転が入っていないことを確認
                        if(m_MirroringMode == nn::g3d::ResSkeleton::MirroringMode_XY &&
                            EqualZero(elemBone.rotate.value.x, rotateTolerance) && EqualZero(elemBone.rotate.value.z, rotateTolerance))
                        {
                            binBone.SetMirroringState(nn::g3d::ResBone::MirroringState_CenterPostRotate);
                            //PRINT_LOG("CenterPostRotate : %hs", elemBone.name.value.c_str());
                        }
                        else if (m_MirroringMode == nn::g3d::ResSkeleton::MirroringMode_XZ &&
                            EqualZero(elemBone.rotate.value.x, rotateTolerance) && EqualZero(elemBone.rotate.value.y, rotateTolerance))
                        {
                            binBone.SetMirroringState(nn::g3d::ResBone::MirroringState_CenterPostRotate);
                            //PRINT_LOG("CenterPostRotate : %hs", elemBone.name.value.c_str());
                        }
                        else
                        {
                            // MirrroingMode_X の場合、もしくはミラー方向が変わる回転が入っていた場合はエラー
                            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SKELETON_INVALID_BONE, "Identifier_InvalidRotateValueMirroring", elemBone.name.value.c_str());
                        }
                    }
                    break;
                default:
                    break;
                }
            }
            break;
        case nw::g3d::tool::g3dif::elem_bone::side_left:
        case nw::g3d::tool::g3dif::elem_bone::side_right:
            {
                // 親が none の場合はエラー
                if (elemParentBone.side.value == nw::g3d::tool::g3dif::elem_bone::side_none)
                {
                    THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SKELETON_INVALID_BONE, "Identifier_InvalidSideNone", elemBone.name.value.c_str(), elemParentBone.name.value.c_str());
                }

                // 親が Center の場合は SideRoot を探す
                nn::g3d::ResBone::MirroringState searchMirroringState = nn::g3d::ResBone::MirroringState_Side;
                if (binParentBone.GetMirroringState() == nn::g3d::ResBone::MirroringState_CenterPreRotate ||
                    binParentBone.GetMirroringState() == nn::g3d::ResBone::MirroringState_CenterRotate ||
                    binParentBone.GetMirroringState() == nn::g3d::ResBone::MirroringState_CenterPostRotate)
                {
                    searchMirroringState = nn::g3d::ResBone::MirroringState_SideRoot;
                    // side はミラー方向が変更された後のボーンから生える必要があるので、この時点でミラー方向を確定する
                    isCenterRotated = true;
                }

                int oldCorrespondingCharCountMax = 0;
                int newCorrespondingCharCountMax = 0;
                for (int searchBoneIndex = 0; searchBoneIndex < static_cast<int>(m_pElem->bone_array.size()); ++searchBoneIndex)
                {
                    // SideRoot の検索の場合、親が一致するボーンのみを採用
                    nw::g3d::tool::g3dif::elem_bone elemSearchBone = m_pElem->bone_array[searchBoneIndex];
                    if (searchMirroringState == nn::g3d::ResBone::MirroringState_SideRoot && parentIndex != elemSearchBone.parent_index)
                    {
                        continue;
                    }
                    // 自身と反対の side のボーンを探す
                    nw::g3d::tool::g3dif::elem_bone::enum_side correspondedSide = (elemBone.side.value == nw::g3d::tool::g3dif::elem_bone::side_left) ? nw::g3d::tool::g3dif::elem_bone::side_right : nw::g3d::tool::g3dif::elem_bone::side_left;
                    if (elemSearchBone.side.value == correspondedSide)
                    {
                        // グローバルなボーン座標を求め、x が正負反転、yz が等しい事を確認
                        nn::util::Matrix4x3f invMtx, searchInvMtx;
                        CalculateInvMatrix(&invMtx, elemBone);
                        CalculateInvMatrix(&searchInvMtx, elemSearchBone);
                        invMtx.Inverse();
                        searchInvMtx.Inverse();
                        if (EqualZero(invMtx.GetAxisW().GetX() + searchInvMtx.GetAxisW().GetX(), translateTolerance) &&
                            EqualZero(invMtx.GetAxisW().GetY() - searchInvMtx.GetAxisW().GetY(), translateTolerance) &&
                            EqualZero(invMtx.GetAxisW().GetZ() - searchInvMtx.GetAxisW().GetZ(), translateTolerance))
                        {
                            // 複数のミラー対応先のボーンが見つかった場合、ボーン名の一致文字数が多い方を採用する。
                            std::string boneName = elemBone.name.value;
                            newCorrespondingCharCountMax = 0;
                            for (int offset = 0; offset < static_cast<int>(boneName.length()); offset++)
                            {
                                for (int size = 1; size < static_cast<int>(boneName.length()) - offset + 1; size++)
                                {
                                    if (elemSearchBone.name.value.find(boneName.substr(offset, size)) != -1)
                                    {
                                        newCorrespondingCharCountMax = std::max(newCorrespondingCharCountMax, size);
                                    }
                                }
                            }
                            if (newCorrespondingCharCountMax > oldCorrespondingCharCountMax)
                            {
                                mirroringBoneIndex = searchBoneIndex;
                                oldCorrespondingCharCountMax = newCorrespondingCharCountMax;
                            }
                            else if (newCorrespondingCharCountMax == oldCorrespondingCharCountMax)
                            {
                                // 一致文字数が同じ場合はボーン名の文字数の差が少ない方を採用する
                                if(fabs(boneName.length() - elemSearchBone.name.value.length()) < fabs(boneName.length() - m_pElem->bone_array[mirroringBoneIndex].name.value.length()))
                                {
                                    mirroringBoneIndex = searchBoneIndex;
                                }
                            }
                        }
                    }
                }

                if (mirroringBoneIndex != -1)
                {
                    binBone.SetMirroringState(searchMirroringState);
                    //PRINT_LOG("    Side : %hs - %hs", elemBone.name.value.c_str(), m_pElem->bone_array[mirroringBoneIndex].name.value.c_str());
                }
                else
                {
                    THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SKELETON_INVALID_BONE, "Identifier_NoMirroredBone", elemBone.name.value.c_str());
                }
            }
            break;
        default:
            break;
        }
        m_MirroringBoneTable[boneIndex] = static_cast<int16_t>(mirroringBoneIndex);
    }

    // ミラー方向が Y,Z の場合、SideRoot がミラー方向変更後のボーンから派生しているかを確認
    if (m_MirroringMode == nn::g3d::ResSkeleton::MirroringMode_XY || m_MirroringMode == nn::g3d::ResSkeleton::MirroringMode_XZ)
    {
        for (int boneIndex = 0; boneIndex < static_cast<int>(m_pElem->bone_array.size()); ++boneIndex)
        {
            nw::g3d::tool::g3dif::elem_bone elemBone = m_pElem->bone_array[boneIndex];
            if (m_BoneArray[boneIndex].GetMirroringState() == nn::g3d::ResBone::MirroringState_SideRoot &&
                m_BoneArray[elemBone.parent_index].GetMirroringState() == nn::g3d::ResBone::MirroringState_CenterPreRotate)
            {
                THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SKELETON_INVALID_BONE, "Identifier_InvalidMirroringDirection", m_pElem->bone_array[boneIndex].name.value.c_str());
            }
        }
    }
}

void BinSkeleton::CalculateSize()
{
    m_Chunk[ChunkType_Skelton].size = sizeof(nn::g3d::ResSkeletonData);
    m_Chunk[ChunkType_Bone].size = sizeof(nn::g3d::ResBoneData) * m_BoneArray.size();

    m_Chunk[ChunkType_MtxBoneTbl].size = nw::g3d::tool::util::Align(sizeof(uint16_t) * m_pElem->matrixPalette.size());
    m_Chunk[ChunkType_InvModelMtx].size = sizeof(nw::g3d::tool::util::Mtx34_t) * m_NumSmoothBone;

    if (m_IsMirroringEnabled == true)
    {
        m_Chunk[ChunkType_MirroringBoneTbl].size = nw::g3d::tool::util::Align(sizeof(int16_t) * m_BoneArray.size());
    }
    else
    {
        m_Chunk[ChunkType_MirroringBoneTbl].size = 0;
    }
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinSkeleton::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Bone].offset;
    for (auto iter = m_BoneArray.begin(); iter != m_BoneArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResBoneData);
    }
}

void BinSkeleton::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResSkeletonData& skeleton = *GetPtr<nn::g3d::ResSkeletonData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    skeleton.blockHeader.signature.SetPacked( nn::g3d::ResSkeleton::Signature );
    skeleton.pUserPtr.Set( nullptr );
    pCtx->AddBinBlockHeader( &skeleton.blockHeader );

    nn::Bit16 flag = 0;
    if (m_pElem->skeleton_info.scale_enable.value)
    {
        flag |= ToEnum_binary(m_pElem->skeleton_info.scale_mode.value);
    }
    else
    {
        flag |= nn::g3d::ResSkeleton::ScaleMode_None;
    }

    flag |= ToEnum_binary(m_pElem->skeleton_info.rotate_mode.value);
    flag |= m_MirroringMode;
    skeleton.flag = flag;

    ConvertBone(skeleton, pCtx);
}

void BinSkeleton::ConvertBone(nn::g3d::ResSkeletonData &skeleton, std::shared_ptr<Context> pCtx )
{
    skeleton.boneCount = static_cast<uint16_t>(m_BoneArray.size());
    nn::g3d::ResBoneData* pBone = GetPtr<nn::g3d::ResBoneData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Bone].offset);
    pCtx->LinkPtr( &skeleton.pBoneArray, pBone );
    m_DicBone.ConvertArrayData(pCtx, skeleton.pBoneDic, pBone, skeleton.boneCount);

    skeleton.smoothMtxCount = static_cast<uint16_t>(m_NumSmoothBone);
    skeleton.rigidMtxCount = static_cast<uint16_t>(m_NumRigidBone);

    uint16_t* pTable = GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_MtxBoneTbl].offset);
    pCtx->LinkPtr( &skeleton.pMtxToBoneTable, pTable );

    size_t paddingSize = m_Chunk[ChunkType_MtxBoneTbl].size - sizeof(uint16_t) * m_pElem->matrixPalette.size();

    for (auto iter = m_pElem->matrixPalette.cbegin(); iter != m_pElem->matrixPalette.cend(); ++iter, ++pTable)
    {
        *pTable = static_cast<uint16_t>(iter->boneIndex);
    }

    std::memset(pTable, NULL, paddingSize);

    nw::g3d::tool::util::Mtx34_t* pInvModelMatrix = GetPtr<nw::g3d::tool::util::Mtx34_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_InvModelMtx].offset);
    for (int idxMatrix = 0; idxMatrix < m_NumSmoothBone; ++idxMatrix, ++pInvModelMatrix)
    {
        const nw::g3d::tool::g3dif::elem_bone& bone = m_pElem->bone_array[m_pElem->matrixPalette[idxMatrix].boneIndex];
        for (int mIndex = 0; mIndex < NN_ARRAY_SIZE(nw::g3d::tool::util::Mtx34_t::a); mIndex++)
        {
            pInvModelMatrix->a[mIndex] = bone.inv_model_matrix.Get().a[mIndex];
        }
    }
    pCtx->LinkPtr( &skeleton.pInvModelMatrixArray, GetPtr<nw::g3d::tool::util::Mtx34_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_InvModelMtx].offset) );

    // モーションミラーリング用の情報
    if (m_IsMirroringEnabled == true)
    {
        int16_t* pMirroringBoneTable = GetPtr<int16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_MirroringBoneTbl].offset);
        pCtx->LinkPtr(&skeleton.pMirroringBoneTable, pMirroringBoneTable);

        paddingSize = m_Chunk[ChunkType_MirroringBoneTbl].size - sizeof(int16_t) * m_BoneArray.size();

        for (int boneIndex = 0; boneIndex < static_cast<int>(m_BoneArray.size()); ++boneIndex, ++pMirroringBoneTable)
        {
            *pMirroringBoneTable = m_MirroringBoneTable[boneIndex];
        }
        std::memset(pMirroringBoneTable, NULL, paddingSize);
    }
    else
    {
        pCtx->LinkPtr(&skeleton.pMirroringBoneTable, NULL);
    }
}

}
}
