﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinSkeletalAnim.h>
#include <algorithm>
#include <util/UtilMath.h>

namespace nn {
namespace g3dTool {

extern uint32_t ToEnum_binary(nw::g3d::tool::g3dif::elem_skeleton_info::enum_scale_mode mode);
extern uint32_t ToEnum_binary(nw::g3d::tool::g3dif::elem_skeleton_info::enum_rotate_mode mode);

void BinBoneAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_bone_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    // バイナリ化するターゲットのみ収集する。
    m_pBoneAnimTargetElemArray.reserve(elem.bone_anim_target_array.size());
    for (auto el_bone_anim_target = elem.bone_anim_target_array.cbegin();
        el_bone_anim_target != elem.bone_anim_target_array.cend(); ++el_bone_anim_target)
    {
        if (el_bone_anim_target->target.value <= nw::g3d::tool::g3dif::elem_bone_anim_target::scale_z)
        {
            if (elem.binarize_scale.value)
            {
                m_pBoneAnimTargetElemArray.push_back(&(*el_bone_anim_target));
            }
        }
        else if (el_bone_anim_target->target.value <= nw::g3d::tool::g3dif::elem_bone_anim_target::rotate_w)
        {
            if (elem.binarize_rotate.value)
            {
                m_pBoneAnimTargetElemArray.push_back(&(*el_bone_anim_target));
            }
        }
        else if (el_bone_anim_target->target.value <= nw::g3d::tool::g3dif::elem_bone_anim_target::translate_z)
        {
            if (elem.binarize_translate.value)
            {
                m_pBoneAnimTargetElemArray.push_back(&(*el_bone_anim_target));
            }
        }
    }

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    curveArray.reserve(m_pBoneAnimTargetElemArray.size());
    for (auto iter = m_pBoneAnimTargetElemArray.cbegin();
        iter != m_pBoneAnimTargetElemArray.cend(); ++iter)
    {
        if ((*iter)->curve)
        {
            curveArray.push_back(&(*iter)->curve.Get());
        }
    }
    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    static const ptrdiff_t tblOffset[] = { 1, 2, 3, 8, 9, 10, 11, 4, 5, 6 }; // BoneAnimResult 参照。
    int curveIndex = 0;
    for (auto iter = m_pBoneAnimTargetElemArray.cbegin();
        iter != m_pBoneAnimTargetElemArray.cend(); ++iter)
    {
        if ((*iter)->curve)
        {
            BinAnimCurve& curve = m_CurveArray[curveIndex++];
            curve.SetTargetOffset(sizeof(float) * tblOffset[(*iter)->target.value]);
            curve.SetType(BinAnimCurve::FLOAT);

            if (nw::g3d::tool::g3dif::elem_bone_anim_target::rotate_x <= (*iter)->target.value &&
                (*iter)->target.value <= nw::g3d::tool::g3dif::elem_bone_anim_target::rotate_w)
            {
                // rotate の場合は無条件で Degree であると仮定する。
                // Quaternion の場合は SetRotateMode で false に設定する。
                curve.SetDegreeValue(true);
            }
        }
    }

    // 文字列の登録。
    pCtx->SetStr( elem.bone_name.value.c_str() );
}

void BinBoneAnim::CalculateSize()
{
    int numBaseValue = 0;
    if (m_pElem->binarize_scale.value)
    {
        numBaseValue += 3;
    }
    if (m_pElem->binarize_rotate.value)
    {
        numBaseValue += 4;
    }
    if (m_pElem->binarize_translate.value)
    {
        numBaseValue += 3;
    }
    m_Chunk[ChunkType_BaseValue].size = sizeof(float) * numBaseValue;
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinBoneAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset( pCtx );

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }
}

void BinBoneAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResBoneAnimData& boneAnim = *GetPtr<nn::g3d::ResBoneAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    pCtx->LinkStr( &boneAnim.pName, nn::util::string_view( m_pElem->bone_name.value.c_str() ) );

    uint32_t flag = 0;
    uint32_t curveFlag = 0;
    float* pBaseValue = GetPtr<float>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BaseValue].offset);
    pCtx->LinkPtr( &boneAnim.pBaseValueArray, pBaseValue );
    boneAnim.beginBaseTranslate = 0;

    bool hasScaleCurve = false;
    bool hasRotateCurve = false;
    bool hasTranslateCurve = false;
    float scale[3] = { 1.0f, 1.0f, 1.0f };
    float rotate[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
    float traslate[3] = { 0.0f, 0.0f, 0.0f };
    for (auto iter = m_pBoneAnimTargetElemArray.cbegin();
        iter != m_pBoneAnimTargetElemArray.cend(); ++iter)
    {
        float baseValue = (*iter)->base_value.value;

        if ((*iter)->target.value <= nw::g3d::tool::g3dif::elem_bone_anim_target::scale_z)
        {
            scale[(*iter)->target.value] = baseValue;
            if ((*iter)->curve)
            {
                hasScaleCurve = true;
            }
        }
        else if ((*iter)->target.value <= nw::g3d::tool::g3dif::elem_bone_anim_target::rotate_w)
        {
            if (m_RotateMode == nw::g3d::tool::g3dif::elem_skeleton_info::euler_xyz)
            {
                baseValue = nn::util::DegreeToRadian(baseValue);
                // euler の場合は w が常に 1.0f
                rotate[3] = 1.0f;
            }
            rotate[(*iter)->target.value - nw::g3d::tool::g3dif::elem_bone_anim_target::rotate_x] = baseValue;
            if ((*iter)->curve)
            {
                hasRotateCurve = true;
            }
        }
        else if ((*iter)->target.value <= nw::g3d::tool::g3dif::elem_bone_anim_target::translate_z)
        {
            traslate[(*iter)->target.value - nw::g3d::tool::g3dif::elem_bone_anim_target::translate_x] = baseValue;
            if ((*iter)->curve)
            {
                hasTranslateCurve = true;
            }
        }

        if ((*iter)->curve)
        {
            curveFlag |= nn::g3d::ResBoneAnim::Flag_CurveScaleX << ((*iter)->target.value - nw::g3d::tool::g3dif::elem_bone_anim_target::scale_x);
        }
    }

    if (m_pElem->binarize_scale.value)
    {
        memcpy(pBaseValue, scale, sizeof(float) * 3);
        pBaseValue += 3;

        if (!hasScaleCurve)
        {
            if (scale[0] == scale[1] && scale[1] == scale[2])
            {
                flag |= nn::g3d::ResBoneAnim::Flag_ScaleUniform;
            }
            if (scale[0] * scale[1] * scale[2] == 1.0f)
            {
                flag |= nn::g3d::ResBoneAnim::Flag_ScaleVolumeOne;
            }
        }

        flag |= nn::g3d::ResBoneAnim::Flag_BaseScale;
        flag |= curveFlag & nn::g3d::ResBoneAnim::Mask_CurveScale;
        boneAnim.beginBaseTranslate += 3;
    }

    if (m_pElem->binarize_rotate.value)
    {
        memcpy(pBaseValue, rotate, sizeof(float) * 4);
        pBaseValue += 4;

        // カーブが存在するものはフラグを立てません。
        if (!hasRotateCurve)
        {
            if (rotate[0] == 0.0f && rotate[1] == 0.0f && rotate[2] == 0.0f)
            {
                flag |= nn::g3d::ResBoneAnim::Flag_RotateZero;
            }
        }

        flag |= nn::g3d::ResBoneAnim::Flag_BaseRotate;
        flag |= curveFlag & nn::g3d::ResBoneAnim::Mask_CurveRotate;
        boneAnim.beginBaseTranslate += 4;
    }

    if (m_pElem->binarize_translate.value)
    {
        memcpy(pBaseValue, traslate, sizeof(float) * 3);
        pBaseValue += 3;

        // カーブが存在するものはフラグを立てません。
        if (!hasTranslateCurve)
        {
            if (traslate[0] == 0.0f && traslate[1] == 0.0f && traslate[2] == 0.0f)
            {
                flag |= nn::g3d::ResBoneAnim::Flag_TranslateZero;
            }
        }

        flag |= nn::g3d::ResBoneAnim::Flag_BaseTranslate;
        flag |= curveFlag & nn::g3d::ResBoneAnim::Mask_CurveTranslate;
    }

    if (m_pElem->scale_compensate.value)
    {
        flag |= nn::g3d::ResBoneAnim::Flag_SegmentScaleCompensate;
    }
    boneAnim.flag = flag;

    boneAnim.beginRotate = static_cast<uint8_t>(std::count_if(
        m_pBoneAnimTargetElemArray.cbegin(), m_pBoneAnimTargetElemArray.cend(),
        [](const nw::g3d::tool::g3dif::elem_bone_anim_target* el_target) -> bool
    {
        return el_target->target.value < nw::g3d::tool::g3dif::elem_bone_anim_target::rotate_x ? true : false;
    }));
    boneAnim.beginTranslate = static_cast<uint8_t>(std::count_if(
        m_pBoneAnimTargetElemArray.cbegin(), m_pBoneAnimTargetElemArray.cend(),
        [](const nw::g3d::tool::g3dif::elem_bone_anim_target* el_target) -> bool
    {
        return el_target->target.value < nw::g3d::tool::g3dif::elem_bone_anim_target::translate_x ? true : false;
    }));
    boneAnim.curveCount = static_cast<uint8_t>(m_CurveArray.size());
    boneAnim.beginCurve = m_BeginCurve;

    if (boneAnim.curveCount != 0)
    {
        pCtx->LinkPtr(&boneAnim.pCurveArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset));
    }
    else
    {
        boneAnim.pCurveArray.Set(nullptr);
    }
}

void BinSkeletalAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_skeletal_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_BoneAnimArray.resize(elem.bone_anim_array.size());
    SetParentBlockArray(m_BoneAnimArray, this);
    BuildArray(pCtx, m_BoneAnimArray, elem.bone_anim_array);

    int32_t numCurve = 0;
    for (auto iter = m_BoneAnimArray.begin(); iter != m_BoneAnimArray.end(); ++ iter)
    {
        iter->SetRotateMode(elem.skeletal_anim_info.rotate_mode.value);
        iter->SetBeginCurve(numCurve);
        numCurve += iter->GetCurveCount();
    }
    m_CurveCount = numCurve;

    m_DicUserData.Build(pCtx, elem.user_data_array.size());
    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr( elem.path.c_str() );
    pCtx->SetStr( elem.name.c_str() );

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinSkeletalAnim::CalculateSize()
{
    auto numAnim =  m_pElem->bone_anim_array.size();

    // ResSkeletalAnimData は親から割り当てる。
    m_Chunk[ChunkType_BindIndex].size = nw::g3d::tool::util::Align(sizeof(uint16_t) * numAnim, ALIGNMENT_DEFAULT);
    m_Chunk[ChunkType_BoneAnim].size = sizeof(nn::g3d::ResBoneAnimData) * numAnim;
    m_Chunk[ChunkType_UserDataData].size = sizeof( nn::gfx::ResUserDataData ) * m_UserDataArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinSkeletalAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_BoneAnim].offset;
    for (auto iter = m_BoneAnimArray.begin(); iter != m_BoneAnimArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResBoneAnimData);
    }

    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_UserDataData ].offset;
    for( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinSkeletalAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResSkeletalAnimData& skeletalAnim = *GetPtr<nn::g3d::ResSkeletalAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    skeletalAnim.blockHeader.signature.SetPacked( nn::g3d::ResSkeletalAnim::Signature );
    pCtx->AddBinBlockHeader( &skeletalAnim.blockHeader );
    pCtx->LinkStr( &skeletalAnim.pName, m_pElem->name.c_str() );
    pCtx->LinkStr( &skeletalAnim.pPath, m_pElem->path.c_str() );

    const nw::g3d::tool::g3dif::elem_skeletal_anim_info& info = m_pElem->skeletal_anim_info;
    uint16_t flag = 0;
    if (info.loop.value)
    {
        flag |= nn::g3d::ResSkeletalAnim::Flag_PlayPolicyLoop;
    }
    flag |= ToEnum_binary(info.scale_mode.value);
    flag |= ToEnum_binary(info.rotate_mode.value);
    skeletalAnim.flag = flag;
    skeletalAnim.boneAnimCount = static_cast<uint16_t>(m_BoneAnimArray.size());
    skeletalAnim.frameCount = m_pElem->skeletal_anim_info.frame_count.value;
    skeletalAnim.curveCount = m_CurveCount;

    // Adjust で代入する
    skeletalAnim.bakedSize = 0;

    skeletalAnim.pBindSkeleton.Set( nullptr );
    uint16_t* pBindIndexArray = GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BindIndex].offset);
    pCtx->LinkPtr( &skeletalAnim.pBindIndexArray, pBindIndexArray );
    int numAnim = static_cast<int>(m_pElem->bone_anim_array.size());
    for (int idxAnim = 0; idxAnim < numAnim; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = nn::g3d::AnimFlag::AnimFlag_NotBound;
    }
    if (numAnim & 0x1)
    {
        // 奇数個の場合は最後にパディングを埋める。
        pBindIndexArray[numAnim] = 0;
    }

    pCtx->LinkPtr( &skeletalAnim.pBoneAnimArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BoneAnim].offset));

    // UserData
    skeletalAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    pCtx->LinkPtr( &skeletalAnim.pUserDataArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_UserDataData].offset) );
    m_DicUserData.ConvertData(pCtx, skeletalAnim.pUserDataDic, m_UserDataArray);
}

void BinSkeletalAnim::Adjust( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResSkeletalAnim* skeletalAnim = GetPtr<nn::g3d::ResSkeletalAnim>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    nn::g3d::ResSkeletalAnimData& skeletalAnimData = skeletalAnim->ToData();
    skeletalAnimData.pBoneAnimArray.Relocate(pCtx->GetBasePtr());

    size_t size = 0;
    for (int idxAnim = 0, numAnim = skeletalAnim->GetBoneAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        nn::g3d::ResBoneAnim* pBoneAnim = skeletalAnim->GetBoneAnim(idxAnim);
        nn::g3d::ResBoneAnimData& boneAnimData = pBoneAnim->ToData();
        boneAnimData.pCurveArray.Relocate(pCtx->GetBasePtr());
        for (int idxCurve = 0, numCurve = pBoneAnim->GetCurveCount(); idxCurve < numCurve; ++idxCurve)
        {
            nn::g3d::ResAnimCurve* curve = pBoneAnim->GetCurve(idxCurve);
            size += curve->CalculateBakedFloatSize();
        }
        boneAnimData.pCurveArray.Unrelocate(pCtx->GetBasePtr());
    }
    skeletalAnimData.bakedSize = static_cast<uint32_t>(size);
    skeletalAnimData.pBoneAnimArray.Unrelocate(pCtx->GetBasePtr());
}

}
}
