﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <unordered_map>

#include <g3dif/Model.h>
#include <BinShape.h>
#include <BinVertex.h>
#include <BinSkeleton.h>
#include <ShapeUtil.h>
#include <GfxHelper.h>
#include <util/UtilError.h>

namespace nn {
namespace g3dTool {

void BinMesh::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_mesh& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;
    m_QuantizeType = elem.quantize_type.value;
    m_IndicesCount = elem.count.value;

    // 隣接頂点を検索する
    if (m_IsBinarizeAdjacency)
    {
        if (elem.mode.value != nw::g3d::tool::g3dif::elem_mesh::triangles)
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_InvalidMeshMode");
        }

        // TriangleList の場合インデクス数が２倍
        m_IndicesCount *= 2;

        if (m_QuantizeType == nw::g3d::tool::g3dif::elem_mesh::uint16)
        {
            if (m_IndicesCount >= 0x1u << (sizeof(uint16_t) * 8))
            {
                m_QuantizeType = nw::g3d::tool::g3dif::elem_mesh::uint32;
            }
        }
        // 正確には 32 ビット最大と比較していないが、
        // どうせ 32 ビットは使えないのは問題ない。
        else if (m_IndicesCount >= 0x1u << (sizeof(uint32_t) * 8 - 1))
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SHAPE_TOO_MANY_VERTICES, "Identifier_TooManyVertices");
        }

        int faceCount = elem.count.value / 3;
        m_AdjIndices.reset(malloc(sizeof(int) * m_IndicesCount), free);

        if (m_QuantizeType == nw::g3d::tool::g3dif::elem_mesh::uint16)
        {
            ComputeAdjacency<int, uint32_t>(m_AdjIndices.get(), elem.stream.rawdata.get(), faceCount, m_VertexCount);
        }
        else
        {
            ComputeAdjacency<int, uint64_t>(m_AdjIndices.get(), elem.stream.rawdata.get(), faceCount, m_VertexCount);
        }
    }
}

void BinMesh::CalculateSize()
{
    // ResMeshData は親の ResShapeData から割り当てている。

    // submesh
    m_Chunk[ChunkType_Submesh].size = sizeof(nn::g3d::ResSubMeshData) * m_pElem->submesh_array.size();

    // gfx::Buffer
    m_Chunk[ ChunkType_GfxBuff ].size = GetGfxMaxImplDataSize<nn::gfx::BufferImplData>();

    // gfx::BufferInfoData
    m_Chunk[ ChunkType_GfxBuffInfoData ].size = sizeof( nn::gfx::BufferInfoData );

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));

    // インデックスバッファの実データ。
    if (m_QuantizeType == nw::g3d::tool::g3dif::elem_mesh::uint16)
    {
        SetBlockSize(Context::MemBlockType_IdxStream, nw::g3d::tool::util::Align(sizeof(uint16_t) * m_IndicesCount));
    }
    else
    {
        SetBlockSize(Context::MemBlockType_IdxStream, sizeof(uint32_t) * m_IndicesCount);
    }
}

void BinMesh::Convert( std::shared_ptr<Context> pCtx )
{
    // ResMeshData のヒープは親から与えられている。
    nn::g3d::ResMeshData& mesh = *GetPtr<nn::g3d::ResMeshData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    if (m_IsBinarizeAdjacency)
    {
        mesh.primType = GetGfxPrimitiveTopologyAdjacency( m_pElem->mode.value );
    }
    else
    {
        mesh.primType = GetGfxPrimitiveTopology( m_pElem->mode.value );
    }

    mesh.format = GetGfxIdxFmt( m_QuantizeType );

    mesh.count = m_IndicesCount;
    mesh.subMeshCount = static_cast<uint16_t>(m_pElem->submesh_array.size());
    mesh.offset = static_cast<uint32_t>(m_pElem->offset);

    size_t size;
    size_t offsetContainMax;
    if (m_QuantizeType == nw::g3d::tool::g3dif::elem_mesh::uint16)
    {
        size = sizeof(uint16_t) * m_IndicesCount;
        offsetContainMax = sizeof(uint16_t);
    }
    else
    {
        size = sizeof(uint32_t) * m_IndicesCount;
        offsetContainMax = sizeof(uint32_t);
    }

    // SubMesh
    mesh.subMeshCount = static_cast<uint16_t>(m_pElem->submesh_array.size());
    nn::g3d::ResSubMeshData* pSubMesh = GetPtr<nn::g3d::ResSubMeshData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Submesh].offset);
    pCtx->LinkPtr( &mesh.pSubMeshArray, pSubMesh );

    for (auto iter = m_pElem->submesh_array.cbegin(); iter != m_pElem->submesh_array.cend();
        ++iter, ++pSubMesh)
    {
        if (iter->count.value <= 0)
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SHAPE_INVALID_SUBMESH_COUNT, "Identifier_NoVertexIndivces");
        }
        pSubMesh->offset = static_cast<uint32_t>(iter->offset.value * offsetContainMax);
        pSubMesh->count = iter->count.value;

        if (m_IsBinarizeAdjacency)
        {
            pSubMesh->offset = pSubMesh->offset * 2;
            pSubMesh->count = iter->count.value * 2;
        }
    }

    pCtx->LinkPtr( &mesh.pIndexBuffer, GetPtr<void>(pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_GfxBuff ].offset ) );
    pCtx->LinkPtr( &mesh.pMemoryPool, pCtx->GetMemBlockPtr( Context::MemBlockType_GfxMemPool ) );

    nn::gfx::BufferInfoData* pBufferInfoData = GetPtr<nn::gfx::BufferInfoData>(pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_GfxBuffInfoData ].offset);
    nn::gfx::BufferInfo* bufferInfo = nn::gfx::DataToAccessor( pBufferInfoData );
    bufferInfo->SetDefault();
    pBufferInfoData->size = static_cast<int32_t>( size );
    pCtx->LinkPtr( &mesh.pIndexBufferInfo, pBufferInfoData );

    void* pData = GetPtr(pCtx, Context::MemBlockType_IdxStream, 0);

    util::BytePtr	idxStreamChunkPtr( pCtx->GetMemBlockPtr( Context::MemBlockType_IdxStream ) );
    mesh.memoryPoolOffset = static_cast<uint32_t>( idxStreamChunkPtr.Distance( GetPtr(pCtx, Context::MemBlockType_IdxStream, 0) ) );

    const uint32_t* pSrc = m_IsBinarizeAdjacency ? static_cast<const uint32_t*>(m_AdjIndices.get())
        : static_cast<const uint32_t*>(m_pElem->stream.rawdata.get());
    if (m_QuantizeType == nw::g3d::tool::g3dif::elem_mesh::uint16)
    {
        uint16_t* pDst = static_cast<uint16_t*>(pData);
        for (uint32_t i = 0; i < m_IndicesCount; ++i)
        {
            pDst[i] = static_cast<uint16_t>(pSrc[i]);
        }

        // count が奇数の場合は最後を０で埋める。
        if (m_IndicesCount % 2)
        {
            pDst[m_IndicesCount] = 0x0;
        }
    }
    else
    {
        std::memcpy(pData, pSrc, size);
    }
}

//--------------------------------------------------------------------------------------------------

void BinShape::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shape& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;
    std::vector<const nw::g3d::tool::g3dif::elem_mesh*> mesh_array;
    mesh_array.reserve(elem.mesh_array.size());
    for (auto iter = elem.mesh_array.cbegin();
        iter != elem.mesh_array.cend();
        ++iter)
    {
        mesh_array.emplace_back(&(*iter));
    }

    m_Meshes.resize(mesh_array.size());
    for (auto mesh = m_Meshes.begin();
        mesh != m_Meshes.end();
        ++mesh)
    {
        if (elem.shape_info.mesh_adjacency)
        {
            mesh->SetVertexCount(elem.shape_info.vertex_count);
            mesh->BinarizeAdjacency();
        }
    }

    SetParentBlockArray(m_Meshes, this);
    BuildArray(pCtx, m_Meshes, mesh_array);

    m_KeyShapeDic.Build(pCtx, elem.key_shape_array.size());

    int keyShapeIndex = 0;
    for (auto iter = elem.key_shape_array.cbegin();
        iter != elem.key_shape_array.cend();
        ++iter, ++keyShapeIndex)
    {
        m_KeyShapeDic.SetName(keyShapeIndex, iter->name.value);
    }

    // 文字列の登録。
    pCtx->SetStr(elem.name.value.c_str());

    for (auto iter = elem.key_shape_array.cbegin();
        iter != elem.key_shape_array.cend();
        ++iter)
    {
        pCtx->SetStr(iter->name.value.c_str());
    }
}

void BinShape::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    // Mesh にオフセットをセットする。
    ptrdiff_t ofsMesh = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Mesh].offset;
    for (auto iter = m_Meshes.begin(); iter != m_Meshes.end(); ++iter)
    {
        iter->SetStructOffset(ofsMesh);
        ofsMesh += sizeof(nn::g3d::ResMeshData);
    }
}

void BinShape::CalculateSize()
{
    // 配列化のために Shape から Mesh にヒープを与える。
    m_Chunk[ChunkType_Mesh].size = sizeof(nn::g3d::ResMeshData) * m_Meshes.size();

    const std::vector<nw::g3d::tool::util::RefBoneWeight>& boneIndices = m_Vertex->GetRefBoneIndices();
    m_Chunk[ChunkType_SkinBoneIdx].size = nw::g3d::tool::util::Align(boneIndices.size() * sizeof(uint16_t));

    m_Chunk[ChunkType_KeyShape].size =
        sizeof(static_cast<nn::g3d::ResKeyShapeData*>(nullptr)->targetAttribIndices) * m_pElem->key_shape_array.size();

    auto& meshArray = m_pElem->mesh_array;
    for( size_t idxMesh = 0; idxMesh < meshArray.size(); ++idxMesh )
    {
        int submeshCount = static_cast<int>( meshArray[idxMesh].submesh_array.size() );
        m_Chunk[ChunkType_SubmeshBounding].size += ( submeshCount + 1 ) * sizeof( nn::g3d::Bounding );
    }

    m_Chunk[ChunkType_BoundingSpherePerLod].size = meshArray.size() * sizeof( float );

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinShape::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResShapeData& shape = *GetPtr<nn::g3d::ResShapeData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    shape.blockHeader.signature.SetPacked( nn::g3d::ResShape::Signature );
    pCtx->LinkStr( &shape.pName, nn::util::string_view( m_pElem->name.value.c_str() ) );

    shape.pUserPtr.Set(nullptr);

    nn::Bit8 flag = 0;

    if (m_HasOwnership)
    {
        flag |= nn::g3d::ResShape::Flag_OwnVertex;
    }

    shape.flag = flag;

    shape.vtxSkinCount = static_cast<uint8_t>(m_pElem->shape_info.vertex_skinning_count.value);
    shape.index = static_cast<uint16_t>(this->m_Index);
    shape.materialIndex = static_cast<uint16_t>(m_pElem->shape_info.mat_index);
    shape.vertexIndex = static_cast<uint16_t>(m_pElem->shape_info.vertex_index.value);
    pCtx->LinkPtr( &shape.pVertex, m_Vertex->GetPtr( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) ) );
    shape.boneIndex = static_cast<uint16_t>(m_pElem->shape_info.bone_index);

    // Mesh
    shape.meshCount = static_cast<uint8_t>(m_Meshes.size());
    pCtx->LinkPtr( &shape.pMeshArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Mesh].offset ) );

    // shape が参照する vertex_attrbute の "blendindex"とmax("blendweight") のリストを取得(model::ResolveBoneIndex で計算済み)
    const std::vector<nw::g3d::tool::util::RefBoneWeight>& refBoneIndices = m_Vertex->GetRefBoneIndices();
    int boneCount = static_cast<int>(refBoneIndices.size());
    shape.skinBoneIndexCount = static_cast<uint16_t>(refBoneIndices.size());

    uint16_t* boneIndices = GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_SkinBoneIdx].offset);
    pCtx->LinkPtr( &shape.pSkinBoneIndexArray, boneIndices );

    // シェイプから参照する全ボーンのリストを格納する。
    const std::vector<nw::g3d::tool::g3dif::elem_skeleton::MatrixPalette>& matrixPalette = m_Skeleton->GetMatrixPalette();
    if (boneCount != 0)
    {
        std::vector<nw::g3d::tool::util::RefBoneWeight> copyBoneIndices;
        copyBoneIndices.resize(boneCount);
        for (int i = 0; i < boneCount; ++i)
        {
            copyBoneIndices[i].index = matrixPalette[refBoneIndices[i].index].boneIndex;
        }
        std::sort(copyBoneIndices.begin(), copyBoneIndices.end(), LessIndex());
        for (int i = 0; i < boneCount; ++i)
        {
            boneIndices[i] = static_cast<uint16_t>( copyBoneIndices[i].index );
        }
    }

    const std::vector<nw::g3d::tool::util::Vec3_t>& positions = m_Vertex->GetPositions();

    if (positions.size() == 0)
    {
        DumpLogRecursively();
        THROW_TRANSLATED_ERROR_INTERNAL(ERRCODE_INTERNAL, "Identifier_InternalError");
    }

    // スムーススキニングの場合、stream の頂点位置はモデル座標系で入っているため、影響を受ける各ボーンのローカル座標に変換しておく
    // (リジッドスキニングの場合は blendindex が指すボーンローカル座標系で入っている)
    const std::vector<nw::g3d::tool::util::Vec3_t>* pLocalPositions = &positions;
    std::vector<nw::g3d::tool::util::Vec3_t> localPositions;
    if (m_pElem->shape_info.vertex_skinning_count.value > 1)
    {
        auto& boneLists = m_Vertex->GetBoneLists();
        if (boneLists.empty() && positions.size() > 0)
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SHAPE_INVALID_VERTEX, "Identifier_NoMatrixPalette", m_pElem->name.value.c_str());
        }
        localPositions.reserve(positions.size() * 4);
        for (int idxPosition = 0; idxPosition < nw::g3d::tool::NumericCast<int>(positions.size()); ++idxPosition)
        {
            const auto& pos = positions[idxPosition];
            auto& boneList = boneLists[idxPosition];
            for (int idxBone = 0; idxBone < nw::g3d::tool::NumericCast<int>(boneList.size()); ++idxBone)
            {
                int boneIndex = matrixPalette[boneList[idxBone]].boneIndex;
                const auto& invMtx = m_Skeleton->GetInvMatrix(boneIndex);
                nw::g3d::tool::util::Vec3_t out;
                out.x = invMtx.m00 * pos.x + invMtx.m01 * pos.y + invMtx.m02 * pos.z + invMtx.m03;
                out.y = invMtx.m10 * pos.x + invMtx.m11 * pos.y + invMtx.m12 * pos.z + invMtx.m13;
                out.z = invMtx.m20 * pos.x + invMtx.m21 * pos.y + invMtx.m22 * pos.z + invMtx.m23;
                localPositions.emplace_back(out);
            }
        }
        pLocalPositions = &localPositions;
    }

    // シェイプとサブメッシュのバウンディングを計算
    {
        nn::g3d::Bounding* pSubmeshBoundings = GetPtr<nn::g3d::Bounding>( pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_SubmeshBounding].offset);
        float* pBoundingRadiusArray = GetPtr<float>( pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BoundingSpherePerLod].offset );
        pCtx->LinkPtr( &shape.pSubMeshBoundingArray, pSubmeshBoundings );
        pCtx->LinkPtr( &shape.pRadiusArray, pBoundingRadiusArray );
        std::vector<int> lodOffsets;
        lodOffsets.push_back( 0 );
        if( m_Vertex->GetLodOffsets() != nullptr )
        {
            for( size_t idxLodOffsets = 0; idxLodOffsets < m_Vertex->GetLodOffsets()->size(); ++idxLodOffsets )
            {
                int lodOffset = ( *m_Vertex->GetLodOffsets() )[idxLodOffsets];
                lodOffsets.push_back( lodOffset );
            }
        }
        CalculateBounding( pSubmeshBoundings, pBoundingRadiusArray, *m_pElem, pLocalPositions, &lodOffsets );
    }

    // フラグ計算
    if( HasBoundingPerLod( GetPtr<nn::g3d::Bounding>( pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_SubmeshBounding].offset), *m_pElem ) )
    {
        shape.flag |= nn::g3d::ResShape::Flag_SubMeshBoundaryConsistent;
    }

    // key_shape
    shape.keyShapeCount = static_cast<uint8_t>(m_pElem->key_shape_array.size());

    if (m_pElem->key_shape_array.size() > 0)
    {
        shape.targetAttrCount = static_cast<uint8_t>(m_pElem->key_shape_array[0].target_attrib_array.size());

        uint8_t* pKeyShape = GetPtr<uint8_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_KeyShape].offset);
        pCtx->LinkPtr( &shape.pKeyShapeArray, pKeyShape );

        nn::util::ResDicData* pKeyShapeDic = m_KeyShapeDic.GetPtr<nn::util::ResDicData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
        pCtx->LinkPtr( &shape.pKeyShapeDic, pKeyShapeDic );
        int keyShapeIndex = 0;
        for (auto iter = m_pElem->key_shape_array.cbegin();
            iter != m_pElem->key_shape_array.cend();
            ++iter, ++keyShapeIndex)
        {
            ptrdiff_t offsetToNextKeyShapeElement = sizeof(static_cast<nn::g3d::ResKeyShapeData*>(nullptr)->targetAttribIndices);
            std::fill(pKeyShape, pKeyShape + offsetToNextKeyShapeElement, 0);
            static const struct KeyAttribSet
            {
                const char* name;
                int offset;
                int count;
            } s_KeyAttribTable[] =
            {
                { "position", nn::g3d::ResKeyShape::KeyAttr_PositionOffset, nn::g3d::ResKeyShape::KeyAttr_PositionCount },
                { "normal", nn::g3d::ResKeyShape::KeyAttr_NormalOffset, nn::g3d::ResKeyShape::KeyAttr_NormalCount },
                { "tangent", nn::g3d::ResKeyShape::KeyAttr_TangentOffset, nn::g3d::ResKeyShape::KeyAttr_TangentCount },
                { "binormal", nn::g3d::ResKeyShape::KeyAttr_BinormalOffset, nn::g3d::ResKeyShape::KeyAttr_BinormalCount },
                { "color", nn::g3d::ResKeyShape::KeyAttr_ColorOffset, nn::g3d::ResKeyShape::KeyAttr_ColorCount }
            };
            for (auto& target : iter->target_attrib_array)
            {
                auto found = std::find_if(s_KeyAttribTable, s_KeyAttribTable + ARRAYSIZE(s_KeyAttribTable), [&](const KeyAttribSet& keyAttrib)
                    { return target.attrib_hint.compare(0, strlen(keyAttrib.name), keyAttrib.name) == 0; });
                if (found == s_KeyAttribTable + ARRAYSIZE(s_KeyAttribTable))
                {
                    THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SHAPE_INVALID_KEY_SHAPE, "Identifier_InvalidKeyShapeAttribute", target.attrib_hint);
                }
                else
                {
                    auto dest = std::find(pKeyShape + found->offset, pKeyShape + found->offset + found->count, 0);
                    if (dest == pKeyShape + found->offset + found->count)
                    {
                        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SHAPE_INVALID_KEY_SHAPE, "Identifier_TooManyKeyShapeAttribute", target.attrib_hint);
                    }
                    else
                    {
                        *dest = nw::g3d::tool::NumericCast<uint8_t>(target.attrib_index + 1);
                    }
                }
            }
            pKeyShape += offsetToNextKeyShapeElement;
        }
    }
    else
    {
        shape.targetAttrCount = 0;
        shape.pKeyShapeDic.Set( nullptr );
    }
}

}
}
