﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinFile.h>
#include <GfxHelper.h>
#include <util/UtilErrorCode.h>
#include <util/UtilError.h>

namespace nn {
namespace g3dTool {


namespace {

//! @brief エンディアンを判定するための BOM です。
const uint16_t BYTE_ORDER_MARK = 0xFEFF;

} // anonymous namespace

void BinExternalFile::Build( std::shared_ptr<Context> pCtx, const BinExternalFile::ExternalFileElem& elem )
{
    m_pElem = &elem;
    pCtx->blocks.push_back(this);

    pCtx->SetStr(m_pElem->name.c_str());
}

void BinExternalFile::CalculateSize()
{
    // ExternalFile のサイズは nn::g3d::ResFileData に含まれるためにここでのサイズ計算には含めません。

    // 外部ファイルのデータサイズを計算。
    size_t size = nw::g3d::tool::util::Align(m_pElem->size, m_pElem->alignment);

    SetBlockSize(Context::MemBlockType_ExternalFile, size, m_pElem->alignment);
}

void BinExternalFile::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResExternalFileData& externalFile =
        *GetPtr<nn::g3d::ResExternalFileData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    void* pData = GetPtr( pCtx, Context::MemBlockType_ExternalFile, 0 );
    pCtx->LinkPtr( &externalFile.pData, pData );

    memcpy(pData, m_pElem->data.get(), m_pElem->size);
    size_t padding = nw::g3d::tool::util::Align(m_pElem->size, m_pElem->alignment) - m_pElem->size;
    memset(nw::g3d::tool::util::AddOffset(pData, m_pElem->size), 0, padding);

    externalFile.size = static_cast<uint32_t>(m_pElem->size);
}

void BinFile::Build( std::shared_ptr<Context> pCtx )
{
    pCtx->blocks.push_back(this);

    // 出力ファイル名を文字列プールに追加する。
    pCtx->SetStr(this->m_Name.c_str());

    std::vector<const nw::g3d::tool::g3dif::elem_model*> modelArray;
    std::vector<const nw::g3d::tool::g3dif::elem_skeletal_anim*> skeletalAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_tex_pattern_anim*> texPatternAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_shader_param_anim*> shaderParamAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_shader_param_anim*> colorAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_shader_param_anim*> texSrtAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_mat_visibility_anim*> matVisibilityAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_bone_visibility_anim*> boneVisibilityAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_scene_anim*> sceneAnimArray;
    std::vector<const nw::g3d::tool::g3dif::elem_shape_anim*> shapeAnimArray;
    std::vector<const BinExternalFile::ExternalFileElem*> externalFileArray;
    std::vector<const nw::g3d::tool::g3dif::elem_nn_material_anim*> matAnimArray;

    auto numModel = m_fmdArray.size();
    modelArray.reserve(numModel);

    auto numSkel = m_fskArray.size();
    skeletalAnimArray.reserve(numSkel);

    auto numTexPattern = m_ftpArray.size();
    texPatternAnimArray.reserve(numTexPattern);

    auto numShaderParam = m_fspArray.size();
    shaderParamAnimArray.reserve(numShaderParam);

    auto numColor = m_fclArray.size();
    colorAnimArray.reserve(numColor);

    auto numTexSRT = m_ftsArray.size();
    texSrtAnimArray.reserve(numTexSRT);

    auto numBoneVisibility = m_fvbArray.size();
    boneVisibilityAnimArray.reserve(numBoneVisibility);

    auto numSceneAnim = m_fsnArray.size();
    sceneAnimArray.reserve(numSceneAnim);

    auto numShapeAnim = m_fshArray.size();
    shapeAnimArray.reserve(numShapeAnim);

    auto numMatVisibility = m_fvmArray.size();
    matVisibilityAnimArray.reserve(numMatVisibility);

    auto numExternalFile = m_externalFileArray.size();
    externalFileArray.reserve(numExternalFile);

    auto numMatAnim = m_fmaArray.size();
    matAnimArray.reserve(numMatAnim);

    // バイナリ化対象のファイルを収集する。
    for (auto iter = m_fmdArray.cbegin(); iter != m_fmdArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_model& fmd = **iter;
        modelArray.push_back(&fmd);
    }
    for (auto iter = m_fskArray.cbegin(); iter != m_fskArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_skeletal_anim& fsk = **iter;
        skeletalAnimArray.push_back(&fsk);
    }
    for (auto iter = m_ftpArray.cbegin(); iter != m_ftpArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_tex_pattern_anim& ftp = **iter;
        texPatternAnimArray.push_back(&ftp);
    }

    for (auto iter = m_fspArray.cbegin(); iter != m_fspArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_shader_param_anim& fsp = **iter;
        shaderParamAnimArray.push_back(&fsp);
    }

    for (auto iter = m_fclArray.cbegin(); iter != m_fclArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_shader_param_anim& fcl = **iter;
        colorAnimArray.push_back(&fcl);
    }

    for (auto iter = m_ftsArray.cbegin(); iter != m_ftsArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_shader_param_anim& fts = **iter;
        texSrtAnimArray.push_back(&fts);
    }

    for (auto iter = m_fvmArray.cbegin(); iter != m_fvmArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_mat_visibility_anim& fvm = **iter;
        matVisibilityAnimArray.push_back(&fvm);
    }
    for (auto iter = m_fvbArray.cbegin(); iter != m_fvbArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_bone_visibility_anim& fvb = **iter;
        boneVisibilityAnimArray.push_back(&fvb);
    }

    for (auto iter = m_fsnArray.cbegin(); iter != m_fsnArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_scene_anim& fsn = **iter;
        sceneAnimArray.push_back(&fsn);
    }

    for (auto iter = m_fshArray.cbegin(); iter != m_fshArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_shape_anim& fsh = **iter;
        shapeAnimArray.push_back(&fsh);
    }

    for (auto iter = m_externalFileArray.cbegin(); iter != m_externalFileArray.cend(); ++iter)
    {
        const BinExternalFile::ExternalFileElem& exFile = *iter;
        externalFileArray.push_back(&exFile);

        // 外部ファイルのアライメントを収集します。
        m_ExternalFileAlignment = std::max(m_ExternalFileAlignment, static_cast<int>(exFile.alignment));
    }

    for (auto iter = m_fmaArray.cbegin(); iter != m_fmaArray.cend(); ++iter)
    {
        const nw::g3d::tool::g3dif::elem_nn_material_anim& fma = **iter;
        matAnimArray.push_back(&fma);
    }

    // 可変長要素の辞書をコンバート対象に追加する。
    m_DicModel.Build( pCtx, modelArray.size());
    m_DicSkeletalAnim.Build(pCtx, skeletalAnimArray.size());
    if( matAnimArray.size() )
    {
        m_DicMaterialAnim.Build(pCtx, matAnimArray.size());
    }
    else
    {
        size_t dicElemCount =
            texPatternAnimArray.size() + shaderParamAnimArray.size() + colorAnimArray.size() +
            texSrtAnimArray.size() + matVisibilityAnimArray.size();
        m_DicMaterialAnim.Build(pCtx, dicElemCount);
    }
    m_DicBoneVisibilityAnim.Build(pCtx, boneVisibilityAnimArray.size());
    m_DicSceneAnim.Build(pCtx, sceneAnimArray.size());
    m_DicShapeAnim.Build(pCtx, shapeAnimArray.size());
    m_DicExternalFile.Build(pCtx, externalFileArray.size());

    m_ModelArray.resize(modelArray.size());

    m_SkeletalAnimArray.resize(skeletalAnimArray.size());
    m_TexPatternAnimArray.resize(texPatternAnimArray.size());
    m_ShaderParamAnimArray.resize(shaderParamAnimArray.size());
    m_ColorAnimArray.resize(colorAnimArray.size());
    m_TexSrtAnimArray.resize(texSrtAnimArray.size());
    m_MatVisibilityAnimArray.resize(matVisibilityAnimArray.size());
    m_BoneVisibilityAnimArray.resize(boneVisibilityAnimArray.size());
    m_SceneAnimArray.resize(sceneAnimArray.size());
    m_ShapeAnimArray.resize(shapeAnimArray.size());
    m_ExternalFileArray.resize(externalFileArray.size());
    m_MaterialAnimArray.resize(matAnimArray.size());

    SetParentBlockArray(m_ModelArray, this);
    SetParentBlockArray(m_SkeletalAnimArray, this);
    SetParentBlockArray(m_TexPatternAnimArray, this);
    SetParentBlockArray(m_ShaderParamAnimArray, this);
    SetParentBlockArray(m_ColorAnimArray, this);
    SetParentBlockArray(m_TexSrtAnimArray, this);
    SetParentBlockArray(m_MatVisibilityAnimArray, this);
    SetParentBlockArray(m_BoneVisibilityAnimArray, this);
    SetParentBlockArray(m_SceneAnimArray, this);
    SetParentBlockArray(m_ShapeAnimArray, this);
    SetParentBlockArray(m_ExternalFileArray, this);
    SetParentBlockArray(m_MaterialAnimArray, this);

    BuildArray(pCtx, m_ModelArray, modelArray);
    BuildArray(pCtx, m_SkeletalAnimArray, skeletalAnimArray);
    BuildArray(pCtx, m_TexPatternAnimArray, texPatternAnimArray);
    BuildArray(pCtx, m_ShaderParamAnimArray, shaderParamAnimArray);
    BuildArray(pCtx, m_ColorAnimArray, colorAnimArray);
    BuildArray(pCtx, m_TexSrtAnimArray, texSrtAnimArray);
    BuildArray(pCtx, m_MatVisibilityAnimArray, matVisibilityAnimArray);
    BuildArray(pCtx, m_BoneVisibilityAnimArray, boneVisibilityAnimArray);
    BuildArray(pCtx, m_SceneAnimArray, sceneAnimArray);
    BuildArray(pCtx, m_ShapeAnimArray, shapeAnimArray);
    BuildArray(pCtx, m_ExternalFileArray, externalFileArray);
    BuildArray(pCtx, m_MaterialAnimArray, matAnimArray);

    // 文字列の登録
    int modelIndex = 0;
    for (auto iter = modelArray.cbegin(); iter != modelArray.cend(); ++iter, ++modelIndex)
    {
        m_DicModel.SetName(modelIndex, (*iter)->name);
    }
    int skeletalAnimIndex = 0;
    for (auto iter = skeletalAnimArray.cbegin(); iter != skeletalAnimArray.cend(); ++iter, ++skeletalAnimIndex)
    {
        m_DicSkeletalAnim.SetName(skeletalAnimIndex, (*iter)->name);
    }
    int materialAnimIndex = 0;
    if( m_MaterialAnimArray.size() )
    {
        for (auto iter = matAnimArray.cbegin(); iter != matAnimArray.cend(); ++iter, ++materialAnimIndex)
        {
            m_DicMaterialAnim.SetName(materialAnimIndex, (*iter)->name);
        }
    }
    else
    {
        for (auto iter = texPatternAnimArray.cbegin(); iter != texPatternAnimArray.cend(); ++iter, ++materialAnimIndex)
        {
            m_DicMaterialAnim.SetName(materialAnimIndex, (*iter)->name);
        }

        for (auto iter = shaderParamAnimArray.cbegin(); iter != shaderParamAnimArray.cend(); ++iter, ++materialAnimIndex)
        {
            m_DicMaterialAnim.SetName(materialAnimIndex, (*iter)->name);
        }

        for (auto iter = colorAnimArray.cbegin(); iter != colorAnimArray.cend(); ++iter, ++materialAnimIndex)
        {
            m_DicMaterialAnim.SetName(materialAnimIndex, (*iter)->name);
        }

        for (auto iter = texSrtAnimArray.cbegin(); iter != texSrtAnimArray.cend(); ++iter, ++materialAnimIndex)
        {
            m_DicMaterialAnim.SetName(materialAnimIndex, (*iter)->name);
        }

        for (auto iter = matVisibilityAnimArray.cbegin(); iter != matVisibilityAnimArray.cend(); ++iter, ++materialAnimIndex)
        {
            m_DicMaterialAnim.SetName(materialAnimIndex, (*iter)->name);
        }
    }

    int boneVisibilityAnimIndex = 0;
    for (auto iter = boneVisibilityAnimArray.cbegin(); iter != boneVisibilityAnimArray.cend(); ++iter, ++boneVisibilityAnimIndex)
    {
        m_DicBoneVisibilityAnim.SetName(boneVisibilityAnimIndex, (*iter)->name);
    }

    int sceneAnimIndex = 0;
    for (auto iter = sceneAnimArray.cbegin(); iter != sceneAnimArray.cend(); ++iter, ++sceneAnimIndex)
    {
        m_DicSceneAnim.SetName(sceneAnimIndex, (*iter)->name);
    }

    int shapeAnimIndex = 0;
    for (auto iter = shapeAnimArray.cbegin(); iter != shapeAnimArray.cend(); ++iter, ++shapeAnimIndex)
    {
        m_DicShapeAnim.SetName(shapeAnimIndex, (*iter)->name);
    }

    int externalFileIndex = 0;
    for (auto iter = externalFileArray.cbegin(); iter != externalFileArray.cend(); ++iter, ++externalFileIndex)
    {
        m_DicExternalFile.SetName(externalFileIndex, (*iter)->name);
    }
}

void BinFile::CalculateSize()
{
    m_Chunk[ChunkType_File].size = sizeof(nn::g3d::ResFileData);
    m_Chunk[ChunkType_ModelArray].size = sizeof(nn::g3d::ResModelData) * m_ModelArray.size();
    m_Chunk[ChunkType_SkeltalAnimArray].size = sizeof(nn::g3d::ResSkeletalAnimData) * m_SkeletalAnimArray.size();

    if(m_MaterialAnimArray.size())
    {
        m_Chunk[ChunkType_MaterialAnimArray].size = sizeof(nn::g3d::ResMaterialAnimData) * m_MaterialAnimArray.size();
    }
    else
    {
        m_Chunk[ChunkType_MaterialAnimArray].size =
            sizeof(nn::g3d::ResMaterialAnimData) *
            ( m_TexPatternAnimArray.size() + m_ShaderParamAnimArray.size() +
            m_ColorAnimArray.size() + m_TexSrtAnimArray.size() + m_MatVisibilityAnimArray.size() );
    }
    m_Chunk[ChunkType_BoneVisibilityAnimArray].size = sizeof(nn::g3d::ResBoneVisibilityAnimData) * m_BoneVisibilityAnimArray.size();
    m_Chunk[ChunkType_SceneAnimArray].size = sizeof(nn::g3d::ResSceneAnimData) * m_SceneAnimArray.size();
    m_Chunk[ChunkType_ShapeAnimArray].size = sizeof(nn::g3d::ResShapeAnimData) * m_ShapeAnimArray.size();
    m_Chunk[ChunkType_GfxMemoryPoolInfoData].size = sizeof( nn::gfx::MemoryPoolInfoData );
    m_Chunk[ChunkType_ExternalFile].size = sizeof(nn::g3d::ResExternalFileData) * m_ExternalFileArray.size();

    // CalcChunk で m_Chunk の各オフセットが計算される
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinFile::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    // 自身の各 Context::MemBlockType の開始 offset を求める
    BinaryBlock::CalculateOffset(pCtx);

    // 子にヒープへのオフセットを与えます。
    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_ModelArray ].offset;
    for (auto iter = m_ModelArray.begin(); iter != m_ModelArray.end(); ++iter)
    {
        iter->SetStructOffset( offset );
        offset += sizeof( nn::g3d::ResModelData );
    }

    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_SkeltalAnimArray ].offset;
    for (auto iter = m_SkeletalAnimArray.begin(); iter != m_SkeletalAnimArray.end(); ++iter)
    {
        iter->SetStructOffset( offset );
        offset += sizeof( nn::g3d::ResSkeletalAnimData );
    }

    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_MaterialAnimArray ].offset;
    if( m_MaterialAnimArray.size() )
    {
        for (auto iter = m_MaterialAnimArray.begin(); iter != m_MaterialAnimArray.end(); ++iter)
        {
            iter->SetStructOffset( offset );
            offset += sizeof( nn::g3d::ResMaterialAnimData );
        }
    }
    else
    {
        for (auto iter = m_TexPatternAnimArray.begin(); iter != m_TexPatternAnimArray.end(); ++iter)
        {
            iter->SetStructOffset( offset );
            offset += sizeof( nn::g3d::ResMaterialAnimData );
        }
        for (auto iter = m_ShaderParamAnimArray.begin(); iter != m_ShaderParamAnimArray.end(); ++iter)
        {
            iter->SetStructOffset( offset );
            offset += sizeof( nn::g3d::ResMaterialAnimData );
        }
        for (auto iter = m_ColorAnimArray.begin(); iter != m_ColorAnimArray.end(); ++iter)
        {
            iter->SetStructOffset( offset );
            offset += sizeof( nn::g3d::ResMaterialAnimData );
        }
        for (auto iter = m_TexSrtAnimArray.begin(); iter != m_TexSrtAnimArray.end(); ++iter)
        {
            iter->SetStructOffset( offset );
            offset += sizeof( nn::g3d::ResMaterialAnimData );
        }
        for (auto iter = m_MatVisibilityAnimArray.begin(); iter != m_MatVisibilityAnimArray.end(); ++iter)
        {
            iter->SetStructOffset( offset );
            offset += sizeof( nn::g3d::ResMaterialAnimData );
        }
    }

    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_BoneVisibilityAnimArray ].offset;
    for (auto iter = m_BoneVisibilityAnimArray.begin(); iter != m_BoneVisibilityAnimArray.end(); ++iter)
    {
        iter->SetStructOffset( offset );
        offset += sizeof( nn::g3d::ResBoneVisibilityAnimData );
    }

    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_SceneAnimArray ].offset;
    for (auto iter = m_SceneAnimArray.begin(); iter != m_SceneAnimArray.end(); ++iter)
    {
        iter->SetStructOffset( offset );
        offset += sizeof( nn::g3d::ResSceneAnimData );
    }

    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_ShapeAnimArray ].offset;
    for (auto iter = m_ShapeAnimArray.begin(); iter != m_ShapeAnimArray.end(); ++iter)
    {
        iter->SetStructOffset( offset );
        offset += sizeof( nn::g3d::ResShapeAnimData );
    }

    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_ExternalFile].offset;
    for (auto iter = m_ExternalFileArray.begin(); iter != m_ExternalFileArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResExternalFileData);
    }
}

void BinFile::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResFileData& file = *GetPtr<nn::g3d::ResFileData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    nn::util::BinaryFileHeader& header = file.fileHeader;
    header.signature.SetPacked( nn::g3d::ResFile::Signature );
    header.version.SetPacked( NN_UTIL_CREATE_BINVERSION( NN_G3D_MODEL_BINARY_VERSION_MAJOR, NN_G3D_MODEL_BINARY_VERSION_MINOR, NN_G3D_MODEL_BINARY_VERSION_MICRO ) );
    header.SetByteOrderMark( nn::util::ByteOrderMark_Normal );	// 0xFEFF

    pCtx->LinkStr( &file.pName, nn::util::string_view( this->m_Name.c_str() ) );
    file.sizeStringPool = static_cast<uint32_t>( pCtx->GetStrPoolMemBlock()->GetSize() );
    pCtx->LinkStr( &file.pStringPool, nn::util::string_view( "" ) );	// 文字列プールの先頭は空文字
    file.pUserPtr.Set(nullptr);

    // Model
    file.modelCount = static_cast<uint16_t>(m_ModelArray.size());
    void* pModelArray = this->GetPtr<void>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_ModelArray ].offset );
    if( file.modelCount != 0 )
    {
        pCtx->LinkPtr( &file.pModelArray, pModelArray );
    }
    m_DicModel.ConvertData(pCtx, file.pModelDic, m_ModelArray);

    // SkeletalAnim
    file.skeletalAnimCount = static_cast<uint16_t>(m_SkeletalAnimArray.size());
    if( file.skeletalAnimCount != 0 )
    {
        pCtx->LinkPtr( &file.pSkeletalAnimArray,
            GetPtr< nn::g3d::ResSkeletalAnimData >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_SkeltalAnimArray ].offset ) );
    }
    m_DicSkeletalAnim.ConvertData(pCtx, file.pSkeletalAnimDic, m_SkeletalAnimArray);

    if( m_MaterialAnimArray.size() )
    {
        file.materialAnimCount = static_cast<uint16_t>(m_MaterialAnimArray.size());
    }
    else
    {
        file.materialAnimCount += static_cast<uint16_t>(m_TexPatternAnimArray.size());	// TexPatternAnim
        file.materialAnimCount += static_cast<uint16_t>(m_ShaderParamAnimArray.size());	// ShaderParamAnim
        file.materialAnimCount += static_cast<uint16_t>(m_ColorAnimArray.size());			// ColorAnim
        file.materialAnimCount += static_cast<uint16_t>(m_TexSrtAnimArray.size());		// TexSrtAnim
        file.materialAnimCount += static_cast<uint16_t>(m_MatVisibilityAnimArray.size());	// MatVisAnim
    }

    if( file.materialAnimCount != 0 )
    {
        pCtx->LinkPtr( &file.pMaterialAnimArray,
            GetPtr< nn::g3d::ResMaterialAnimData>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_MaterialAnimArray ].offset ) );
    }
    m_DicMaterialAnim.ConvertData(pCtx, file.pMaterialAnimDic);

    // BoneVisibilityAnim
    file.boneVisAnimCount = static_cast<uint16_t>(m_BoneVisibilityAnimArray.size());
    if( file.boneVisAnimCount != 0 )
    {
        pCtx->LinkPtr( &file.pBoneVisAnimArray,
            GetPtr< nn::g3d::ResBoneVisibilityAnimData >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_BoneVisibilityAnimArray ].offset ) );
    }
    m_DicBoneVisibilityAnim.ConvertData(pCtx, file.pBoneVisAnimDic, m_BoneVisibilityAnimArray);

    // SceneAnim
    file.sceneAnimCount = static_cast<uint16_t>(m_SceneAnimArray.size());
    if( file.sceneAnimCount != 0 )
    {
        pCtx->LinkPtr( &file.pSceneAnimArray,
            GetPtr< nn::g3d::ResSceneAnimData >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_SceneAnimArray ].offset ) );
    }
    m_DicSceneAnim.ConvertData(pCtx, file.pSceneAnimDic, m_SceneAnimArray);

    // ShapeAnim
    file.shapeAnimCount = static_cast<uint16_t>(m_ShapeAnimArray.size());
    if( file.shapeAnimCount != 0 )
    {
        pCtx->LinkPtr( &file.pShapeAnimArray,
            GetPtr< nn::g3d::ResShapeAnimData >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_ShapeAnimArray ].offset ) );
    }
    m_DicShapeAnim.ConvertData(pCtx, file.pShapeAnimDic, m_ShapeAnimArray);

    // ExternalFile
    file.externalFileCount = static_cast<uint16_t>(m_ExternalFileArray.size());
    if( file.externalFileCount != 0 )
    {
        pCtx->LinkPtr( &file.pExternalFileArray,
            GetPtr< nn::g3d::ResExternalFileData >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_ExternalFile ].offset ) );
    }
    m_DicExternalFile.ConvertData(pCtx, file.pExternalFileDic, m_ExternalFileArray);

    // MemoryPool
    if( pCtx->GetMemBlockSize( Context::MemBlockType_GfxMemPool ) != 0 )
    {
        pCtx->LinkPtr( &file.pBufferMemoryPool, pCtx->GetMemBlockPtr( Context::MemBlockType_GfxMemPool ) );
    }

    // MemoryPoolInfo
    nn::gfx::MemoryPoolInfoData* pMemPoolInfoData =
        GetPtr<nn::gfx::MemoryPoolInfoData>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_GfxMemoryPoolInfoData ].offset );
    pMemPoolInfoData->memoryPoolProperty =
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached;
    pMemPoolInfoData->memorySize  = static_cast<uint32_t>( pCtx->GetMemBlockSize( Context::MemBlockType_IdxStream ) );
    pMemPoolInfoData->memorySize += static_cast<uint32_t>( pCtx->GetMemBlockSize( Context::MemBlockType_VtxStream ) );

    // メモリプールのアライメントが適切か確認する
    if( !nn::util::is_aligned( pMemPoolInfoData->memorySize, ALIGNMENT_MEMORY_POOL_SIZE ) )
    {
        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_INTERNAL, "Identifier_InvalidSizeAlignment", "MemoryPoolSize", pMemPoolInfoData->memorySize, ALIGNMENT_MEMORY_POOL_SIZE);
    }

    if( pCtx->GetMemBlockSize( Context::MemBlockType_VtxStream ) != 0 )
    {
        pCtx->LinkPtr( reinterpret_cast< nn::util::BinPtr* >( &pMemPoolInfoData->pMemory ), pCtx->GetMemBlockPtr( Context::MemBlockType_IdxStream ) );
        pCtx->LinkPtr( &file.pBufferMemoryPoolInfo, GetPtr( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_GfxMemoryPoolInfoData ].offset ) );
    }
}

void BinFile::Adjust( std::shared_ptr<Context> pCtx )
{
    NN_UNUSED( pCtx );
}

}
}
