﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/g3d_ResShape.h>
#include <g3dif/Vertex.h>
#include <bindefs.h>
#include <BinaryBlock.h>
#include <BinDictionary.h>

namespace nn {
namespace g3dTool {


//--------------------------------------------------------------------------------------------------

class BinVertex;

//! @brief 頂点属性データのバイナリ化を行うクラスです。
class BinVtxAttrib : public BinaryBlock
{
public:
    BinVtxAttrib()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_BufferIndex(-1)
        , m_VtxOffset(0)
        , m_DataOffset(-1)
        , m_VtxStride(0)
        , m_VtxMax(0)
        , m_Padding(0)
    {
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_vtx_attrib& elem);
    virtual void CalculateSize();
    virtual void Convert( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("VertexAttribute : %hs", m_pElem->name.value.c_str());
    }

    void SetBufferIndex(int index) { m_BufferIndex = index; }
    void SetVtxOffset(size_t offset) { m_VtxOffset = offset; }
    void SetPaddingRight(size_t padding) { m_Padding = padding; }

    void SetDataOffset(ptrdiff_t offset) { m_DataOffset = offset; }
    void SetVtxStride(size_t stride) { m_VtxStride = stride; }
    void SetVtxMax(int max) { m_VtxMax = max; }

    template<typename T>
    void Quantize(void* pDst, const T* pSrc, int numVertex, int size);

    template<typename TSrc>
    static bool QuantizeToUnorm4_4(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToUnorm8(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToUnorm10_10_10_2_Type3(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToUnorm10_10_10_2_Type4(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToSnorm10_10_10_2_Type3(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToSnorm10_10_10_2_Type4(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToUnorm16(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToUInt8(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToUInt10_10_10_2_Type3(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToUInt10_10_10_2_Type4(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToSInt10_10_10_2_Type3(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToSInt10_10_10_2_Type4(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToUInt16(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToUInt32(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToSnorm8(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToSnorm16(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToSInt8(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToSInt16(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToSInt32(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToFloat16(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToFloat10_11_11(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TSrc>
    static bool QuantizeToFloat32(void* /*pDst*/, TSrc /*src*/)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }

    template<>
    static bool QuantizeToUnorm4_4(void* pDst, const float* src);

    template<>
    static bool QuantizeToUnorm8(void* pDst, float src);

    template<>
    static bool QuantizeToUnorm10_10_10_2_Type3(void* pDst, const float* src);

    template<>
    static bool QuantizeToUnorm10_10_10_2_Type4(void* pDst, const float* src);

    template<>
    static bool QuantizeToSnorm10_10_10_2_Type3(void* pDst, const float* src);

    template<>
    static bool QuantizeToSnorm10_10_10_2_Type4(void* pDst, const float* src);

    template<>
    static bool QuantizeToUnorm16(void* pDst, float src);

    template<>
    static bool QuantizeToUInt8(void* pDst, int src);

    template<>
    static bool QuantizeToUInt10_10_10_2_Type3(void* pDst, const int* src);

    template<>
    static bool QuantizeToUInt10_10_10_2_Type4(void* pDst, const int* src);

    template<>
    static bool QuantizeToSInt10_10_10_2_Type3(void* pDst, const int* src);

    template<>
    static bool QuantizeToSInt10_10_10_2_Type4(void* pDst, const int* src);

    template<>
    static bool QuantizeToUInt16(void* pDst, int src);

    template<>
    static bool QuantizeToUInt32(void* pDst, int src);

    template<>
    static bool QuantizeToSnorm8(void* pDst, float src);

    template<>
    static bool QuantizeToSnorm16(void* pDst, float src);

    template<>
    static bool QuantizeToSInt8(void* pDst, int src);

    template<>
    static bool QuantizeToSInt16(void* pDst, int src);

    template<>
    static bool QuantizeToSInt32(void* pDst, int src);

    template<>
    static bool QuantizeToFloat10_11_11(void* pDst, const float* src);

    template<>
    static bool QuantizeToFloat16<float>(void* pDst, float src);

    template<>
    static bool QuantizeToFloat32<float>(void* pDst, float src);


    template<typename TDst>
    static bool DequantizeFromUnorm8(TDst* pDst, uint8_t src)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TDst>
    static bool DequantizeFromUnorm16(TDst* pDst, uint16_t src)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<typename TDst>
    static bool DequantizeFromSnorm10_10_10_2(TDst* dst0, TDst* dst1, TDst* dst2, TDst* dst3, uint32_t src)
    {
        PRINT_SYSTEM_LOG("Unsupported quantize type.");
        return false;
    }
    template<>
    static bool DequantizeFromUnorm8(float* pDst, uint8_t src);

    template<>
    static bool DequantizeFromUnorm16(float* pDst, uint16_t src);

    template<>
    static bool DequantizeFromSnorm10_10_10_2(float* dst0, float* dst1, float* dst2, float* dst3, uint32_t src);

private:
    const nw::g3d::tool::g3dif::elem_vtx_attrib* m_pElem;

    int m_BufferIndex;
    size_t m_VtxOffset;

    ptrdiff_t m_DataOffset;
    size_t m_VtxStride;
    size_t m_VtxMax;
    size_t m_Padding;
    void* m_pDst;

    friend class BinVertex;
};

//--------------------------------------------------------------------------------------------------

//! @brief 頂点データを定義するクラスです。
class BinVertex : public BinaryBlock
{
public:
    BinVertex()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_VtxAttribArray()
        , m_VtxBufferArray()
        , m_AttribTable()
        , m_HasOwner(false)
        , m_SkinningCount(0)
    {
        for( int chunkIdx = 0; chunkIdx < ChunkType_Count; chunkIdx++ )
        {
            m_Chunk[ chunkIdx ].alignment = 0;
            m_Chunk[ chunkIdx ].offset = 0;
            m_Chunk[ chunkIdx ].size = 0;
        }
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_vertex& elem);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );
    virtual void Adjust( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("Vertex(index) : %d", m_pElem->index);
    }

    void SplitInput(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_vertex& elem);

    void ConvertVtxBuffer(nn::g3d::ResVertexData &shape, std::shared_ptr<Context> pCtx);

    bool HasOwner() const { return this->m_HasOwner; }
    void SetOwner(bool hasOwner) { this->m_HasOwner = hasOwner; }
    void SetSkinningCount(int skinningCount) { m_SkinningCount = skinningCount; }

    const std::vector<nw::g3d::tool::util::Vec3_t>& GetPositions() const
    {
        return m_pElem->positions;
    }

    const std::vector<nw::g3d::tool::util::RefBoneWeight>& GetRefBoneIndices() const
    {
        return m_pElem->boneIndices;
    }

    const std::vector<std::vector<int>>& GetBoneLists() const
    {
        return m_pElem->boneLists;
    }

    const std::vector<int>* GetLodOffsets() const
    {
        if( m_pElem->lod_offset )
        {
            return &m_pElem->lod_offset->lodOffset;
        }
        return nullptr;
    }

private:
    const nw::g3d::tool::g3dif::elem_vertex* m_pElem;

    std::vector<BinVtxAttrib> m_VtxAttribArray;
    std::vector<std::pair<size_t, int>> m_VtxBufferArray;
    std::vector<int> m_AttribTable;

    BinDictionary m_DicVtxAttrib;
    bool m_HasOwner;
    int m_SkinningCount;

    enum ChunkType
    {
        ChunkType_VtxAttr,
        ChunkType_GfxBuffArray,
        ChunkType_GfxBuffInfoDataArray,
        ChunkType_GfxVtxBuffStateInfoDataArray,
        ChunkType_GfxBuffPtrArray,
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

}
}
