﻿namespace G3dCore.Storages
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Data;
    using G3dCore.Converters;
    using G3dCore.Entities;
    using G3dCore.Utilities;
    using G3dCore.ViewModels;
    using Opal.Storages;

    /// <summary>
    /// G3d用ストレージクラスです。
    /// </summary>
    public sealed class G3dStorage : Storage
    {
        private readonly ObservableCollection<G3dFileViewModel> fileViewModels = new ObservableCollection<G3dFileViewModel>();

        /// <summary>
        /// ファイルビューモデルを取得します。
        /// </summary>
        public ICollectionView FileViewModels
        {
            get
            {
                return CollectionViewSource.GetDefaultView(this.fileViewModels);
            }
        }

        /// <summary>
        /// データを格納するストレージの状態をクリアします。
        /// </summary>
        public override void Clear()
        {
        }

        /// <summary>
        /// 初期化処理を行います。
        /// </summary>
        public void Initialize()
        {
            ShaderConverterManager.Initialize();
            ShaderConverterManager.InitializeGL();
        }

        /// <summary>
        /// 中間ファイルを読み込みます。
        /// </summary>
        /// <param name="path">中間ファイルのパスです。</param>
        public void LoadFile(string path)
        {
            foreach (G3dFileViewModel file in this.fileViewModels)
            {
                // 既に開かれているファイルは二重には開かない。
                if (path == file.FilePath)
                {
                    file.OpenDocument();
                    return;
                }
            }

            G3dFile g3dFile = G3dIfUtility.ReadFile(path);
            g3dFile.Reset();
            this.AddG3dFile(g3dFile);
        }

        /// <summary>
        /// 中間ファイルを追加します。
        /// </summary>
        /// <param name="g3dFile">追加する中間ファイルです。</param>
        public void AddG3dFile(G3dFile g3dFile)
        {
            Debug.Assert(g3dFile != null);

            G3dFileViewModel fileViewModel = new G3dFileViewModel(g3dFile);
            this.AddFileViewModel(fileViewModel);
            fileViewModel.OpenDocument();
        }

        /// <summary>
        /// 中間ファイルファイルビューモデルを追加します。
        /// </summary>
        /// <param name="fileViewModel">追加する中間ファイルビューモデルです。</param>
        public void AddFileViewModel(G3dFileViewModel fileViewModel)
        {
            Debug.Assert(fileViewModel != null);

            this.fileViewModels.Add(fileViewModel);
        }

        /// <summary>
        /// 中間ファイルファイルビューモデルを削除します。
        /// </summary>
        /// <param name="fileViewModel">削除する中間ファイルビューモデルです。</param>
        public void RemoveFileViewModel(G3dFileViewModel fileViewModel)
        {
            Debug.Assert(fileViewModel != null);

#if false
            string extension = Path.GetExtension(fileViewModel.FilePath);
            if (!string.IsNullOrEmpty(extension))
            {
                extension = extension.Substring(1);
            }
#endif

            this.fileViewModels.Remove(fileViewModel);

#if false
            FileCloseArg arg = new FileCloseArg(fileViewModel.FilePath, null);
            arg.Instance = fileViewModel.File;

            // ファイルクローズフックの前処理
            {
                foreach (var plugin in this.pluginFileManager.GetAlwaysUseFileCloseHookPlugins())
                {
                    plugin.PreCloseFiles(arg);
                }

                foreach (var plugin in this.pluginFileManager.GetFileCloseHookPlugins(extension))
                {
                    plugin.PreCloseFiles(arg);
                }
            }

            this.fileViewModels.Remove(fileViewModel);

            // ファイルクローズフックの後処理
            {
                foreach (var plugin in this.pluginFileManager.GetAlwaysUseFileCloseHookPlugins())
                {
                    plugin.PostCloseFiles(arg);
                }

                foreach (var plugin in this.pluginFileManager.GetFileCloseHookPlugins(extension))
                {
                    plugin.PostCloseFiles(arg);
                }
            }

            // ファイルクローズ後の処理
            {
                foreach (var plugin in this.pluginFileManager.GetOnFileClosedPlugins())
                {
                    plugin.OnFileClosed(fileViewModel.File);
                }
            }
#endif
        }

        public bool ConfirmAndCloseDocumentsAll()
        {
            for (int idx = this.fileViewModels.Count - 1; idx >= 0; --idx)
            {
                G3dFileViewModel file = this.fileViewModels[idx];

                if (!file.ConfirmAndCloseDocument())
                {
                    return false;
                }
            }

            return true;
        }
    }
}
