﻿namespace G3dCore.Entities
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using nw.g3d.nw4f_3dif;
    using Opal.Security.Cryptography;

    /// <summary>
    /// シェーダ設定定義のエンティティクラスです。
    /// </summary>
    public class Shader : ObservableEntity<shaderType>
    {
        private readonly ObservableCollection<Macro> macros = new ObservableCollection<Macro>();
        private readonly ObservableCollection<Variation> variations = new ObservableCollection<Variation>();
        private int index = 0;
        private string name = string.Empty;
        private bool materialShader = false;
        private VertexShader vertexShader = new VertexShader();
        private GeometryShader geometryShader = new GeometryShader();
        private FragmentShader fragmentShader = new FragmentShader();
        private ComputeShader computeShader = new ComputeShader();
        private bool isVariationsUpdated = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Shader()
        {
            this.macros.CollectionChanged += this.OnCollectionChanged<Macro>;
            this.variations.CollectionChanged += this.OnCollectionChanged<Variation>;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="data">設定するデータです。</param>
        public Shader(shaderType data)
            : this()
        {
            this.index = data.index;
            this.name = data.name ?? string.Empty;
            this.materialShader = data.material_shader;

            if (data.vertex_shader != null)
            {
                this.vertexShader = new VertexShader(data.vertex_shader);
            }

            if (data.geometry_shader != null)
            {
                this.geometryShader = new GeometryShader(data.geometry_shader);
            }

            if (data.fragment_shader != null)
            {
                this.fragmentShader = new FragmentShader(data.fragment_shader);
            }

            if (data.compute_shader != null)
            {
                this.computeShader = new ComputeShader(data.compute_shader);
            }

            if (data.macro_array != null)
            {
                this.AddMacros(data.macro_array.macro);
            }

            if (data.variation_array != null)
            {
                this.AddVariations(data.variation_array.variation);
            }
        }

        /// <summary>
        /// インデックスを取得設定します。
        /// </summary>
        public int Index
        {
            get
            {
                return this.index;
            }

            set
            {
                this.SetProperty(ref this.index, value);
            }
        }

        /// <summary>
        /// 名前を取得設定します。
        /// </summary>
        public string Name
        {
            get
            {
                return this.name;
            }

            set
            {
                this.SetProperty(ref this.name, value ?? string.Empty, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// マテリアル用のシェーダ設定かどうかを取得設定します。
        /// </summary>
        public bool MaterialShader
        {
            get
            {
                return this.materialShader;
            }

            set
            {
                this.SetProperty(ref this.materialShader, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 頂点シェーダ設定を取得します。
        /// </summary>
        public VertexShader VertexShader
        {
            get
            {
                return this.vertexShader;
            }
        }

        /// <summary>
        /// ジオメトリシェーダ設定を取得します。
        /// </summary>
        public GeometryShader GeometryShader
        {
            get
            {
                return this.geometryShader;
            }
        }

        /// <summary>
        /// フラグメントシェーダ設定を取得します。
        /// </summary>
        public FragmentShader FragmentShader
        {
            get
            {
                return this.fragmentShader;
            }
        }

        /// <summary>
        /// 演算シェーダ設定を取得します。
        /// </summary>
        public ComputeShader ComputeShader
        {
            get
            {
                return this.computeShader;
            }
        }

        /// <summary>
        /// マクロを取得します。
        /// </summary>
        public ObservableCollection<Macro> Macros
        {
            get
            {
                return this.macros;
            }
        }

        /// <summary>
        /// バリエーションを取得します。
        /// </summary>
        public ObservableCollection<Variation> Variations
        {
            get
            {
                return this.variations;
            }
        }

        /// <summary>
        /// バリエーション情報が更新されているかを取得設定します。
        /// </summary>
        public bool IsVariationsUpdated
        {
            get
            {
                return this.isVariationsUpdated;
            }

            set
            {
                this.isVariationsUpdated = value;
            }
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクトを返します。</returns>
        public override IEntity Clone()
        {
            var instance = new Shader();
            instance.index = this.index;
            instance.name = this.name;
            instance.materialShader = this.materialShader;

            if (this.vertexShader != null)
            {
                instance.vertexShader = this.vertexShader.Clone() as VertexShader;
            }

            if (this.geometryShader != null)
            {
                instance.geometryShader = this.geometryShader.Clone() as GeometryShader;
            }

            if (this.fragmentShader != null)
            {
                instance.fragmentShader = this.fragmentShader.Clone() as FragmentShader;
            }

            if (this.computeShader != null)
            {
                instance.computeShader = this.computeShader.Clone() as ComputeShader;
            }

            foreach (var macro in this.macros)
            {
                instance.macros.Add(macro.Clone() as Macro);
            }

            foreach (var variation in this.variations)
            {
                instance.variations.Add(variation.Clone() as Variation);
            }

            return instance;
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shaderType CreateWriteData()
        {
            shaderType instance = new shaderType();
            instance.index = this.index;
            instance.name = this.name;
            instance.material_shader = this.materialShader;

            if (this.vertexShader != null)
            {
                instance.vertex_shader = this.vertexShader.CreateWriteData();
            }

            if (this.geometryShader != null)
            {
                instance.geometry_shader = this.geometryShader.CreateWriteData();
            }

            if (this.fragmentShader != null)
            {
                instance.fragment_shader = this.fragmentShader.CreateWriteData();
            }

            if (this.computeShader != null)
            {
                instance.compute_shader = this.computeShader.CreateWriteData();
            }

            instance.macro_array = this.CreateMacroArray();
            instance.variation_array = this.CreateVariationArray();

            return instance;
        }

        /// <summary>
        /// エンティティのCRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRCの値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            // Index プロパティは、出力時に計算しなおすので、CRCの計算対象外
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.name)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.materialShader)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.vertexShader.GetHashCode())));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.geometryShader.GetHashCode())));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.fragmentShader.GetHashCode())));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.computeShader.GetHashCode())));

            foreach (var macro in this.macros)
            {
                buffers.AddRange(BitConverter.GetBytes(macro.HashValue));
            }

            foreach (var variation in this.variations)
            {
                buffers.AddRange(BitConverter.GetBytes(variation.HashValue));
            }

            var hashValue = crc.ComputeHashUInt32(buffers.ToArray());
            return hashValue;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var macro in this.macros)
            {
                macro.Reset();
            }

            foreach (var variation in this.variations)
            {
                variation.Reset();
            }

            if (this.vertexShader != null)
            {
                this.vertexShader.Reset();
            }

            if (this.geometryShader != null)
            {
                this.geometryShader.Reset();
            }

            if (this.fragmentShader != null)
            {
                this.fragmentShader.Reset();
            }

            if (this.computeShader != null)
            {
                this.computeShader.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var macro in this.macros)
            {
                macro.Refresh();
            }

            foreach (var variation in this.variations)
            {
                variation.Refresh();
            }

            if (this.vertexShader != null)
            {
                this.vertexShader.Refresh();
            }

            if (this.geometryShader != null)
            {
                this.geometryShader.Refresh();
            }

            if (this.fragmentShader != null)
            {
                this.fragmentShader.Refresh();
            }

            if (this.computeShader != null)
            {
                this.computeShader.Refresh();
            }
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var macro in this.macros)
            {
                macro.AutoCalc = this.AutoCalc;
            }

            foreach (var variation in this.variations)
            {
                variation.AutoCalc = this.AutoCalc;
            }

            if (this.vertexShader != null)
            {
                this.vertexShader.AutoCalc = this.AutoCalc;
            }

            if (this.geometryShader != null)
            {
                this.geometryShader.AutoCalc = this.AutoCalc;
            }

            if (this.fragmentShader != null)
            {
                this.fragmentShader.AutoCalc = this.AutoCalc;
            }

            if (this.computeShader != null)
            {
                this.computeShader.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// マクロ配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected macro_arrayType CreateMacroArray()
        {
            if (this.macros.FirstOrDefault() == null)
            {
                return null;
            }

            var macroArray = new macro_arrayType();
            List<macroType> macroTypes = new List<macroType>();

            int index = 0;
            foreach (var macro in this.macros)
            {
                macro.Index = index;
                ++index;
                macroTypes.Add(macro.CreateWriteData());
            }

            macroArray.macro = macroTypes.ToArray();
            macroArray.length = macroTypes.Count;
            return macroArray;
        }

        /// <summary>
        /// バリエーション配列のデータを作成します。
        /// </summary>
        /// <returns>データのインスタンスを返します。</returns>
        protected variation_arrayType CreateVariationArray()
        {
            if (this.variations.FirstOrDefault() == null)
            {
                return null;
            }

            var variationArray = new variation_arrayType();
            List<variationType> variationTypes = new List<variationType>();

            int index = 0;
            foreach (var variation in this.variations)
            {
                if (!variation.IsOutputable)
                {
                    continue;
                }

                variation.Index = index;
                ++index;
                variationTypes.Add(variation.CreateWriteData());
            }

            if (variationTypes.FirstOrDefault() == null)
            {
                return null;
            }

            variationArray.variation = variationTypes.ToArray();
            variationArray.length = variationTypes.Count;
            return variationArray;
        }

        /// <summary>
        /// マクロを追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddMacros(macroType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                Macro macro = new Macro(value);
                this.macros.Add(macro);
            }
        }

        /// <summary>
        /// バリエーションを追加します。
        /// </summary>
        /// <param name="data">追加するデータです。</param>
        protected void AddVariations(variationType[] data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var value in data)
            {
                Variation variation = new Variation(value);
                this.variations.Add(variation);
            }
        }
    }
}
