﻿namespace G3dCore.Entities
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Linq.Expressions;
    using System.Runtime.CompilerServices;
    using System.Text;
    using System.Threading;
    using System.Threading.Tasks;
    using System.Xml.Serialization;
    using G3dCore.Entities.Generic;
    using Opal.ComponentModel;

    /// <summary>
    /// データ操作用の基底クラスです。ジェネリック版
    /// </summary>
    /// <typeparam name="TEntity">エンティティタイプのテンプレートの型です。</typeparam>
    public abstract class Entity<TEntity> : ObservableObject, IEntity<TEntity> where TEntity : class
    {
        [NonSerialized]
        private bool dirtyFlag = false;

        [NonSerialized]
        private uint initValue = 0;

        [NonSerialized]
        private uint changeValue = 0;

        [NonSerialized]
        private bool autoCalc = false;

        /// <summary>
        /// エンティティの値が変更されているかどうかを取得します。
        /// </summary>
        [XmlIgnore]
        public bool IsDirty
        {
            get
            {
                return this.dirtyFlag;
            }

            private set
            {
                this.SetProperty(ref this.dirtyFlag, value);
            }
        }

        /// <summary>
        /// エンティティがもつ値が変更された場合に自動でCRCを計算するフラグを取得設定します。
        /// </summary>
        [XmlIgnore]
        public bool AutoCalc
        {
            get
            {
                return this.autoCalc;
            }

            set
            {
                this.SetProperty(ref this.autoCalc, value, () => this.SetAutoCalcFlagInternal());
            }
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        [XmlIgnore]
        public uint HashValue
        {
            get
            {
                return this.changeValue;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。
        /// </summary>
        public void Reset()
        {
            this.ResetInternal();
            this.initValue = this.changeValue = this.CreateCRCInternal();
            this.IsDirty = false;
            this.AutoCalc = true;
        }

        /// <summary>
        /// エンティティの状態を更新します。
        /// </summary>
        public void Refresh()
        {
            this.RefreshInternal();
            this.CalcCRC();
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        object IEntity.CreateWriteData()
        {
            return this.CreateWriteData();
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクトを返します。</returns>
        object ICloneable.Clone()
        {
            return this.Clone();
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクトを返します。</returns>
        public abstract IEntity Clone();

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public abstract TEntity CreateWriteData();

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// 継承先で階層構造になっているものにリセット処理を伝搬するために使用します。
        /// </summary>
        protected virtual void ResetInternal()
        {
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// 継承先で階層構造になっているものに更新処理を伝搬するために使用します。
        /// </summary>
        protected virtual void RefreshInternal()
        {
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// 継承先で階層構造になっているものに更新処理を伝搬するために使用します。
        /// </summary>
        protected virtual void SetAutoCalcFlagInternal()
        {
        }

        /// <summary>
        /// CRCを計算します。
        /// </summary>
        protected void CalcCRC()
        {
            if (this.autoCalc)
            {
                this.changeValue = this.CreateCRCInternal();
                this.IsDirty = !(this.initValue == this.changeValue);
            }
        }

        /// <summary>
        /// エンティティのCRC を作成します。（内部処理用）
        /// 継承先で実装します。
        /// </summary>
        /// <returns>CRCの値を返します。</returns>
        protected abstract uint CreateCRCInternal();
    }
}
