﻿namespace G3dCore.Converters
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Entities;
    using G3dCore.IO;
    using G3dCore.Modules.ErrorView;
    using G3dCore.Resources;
    using G3dCore.Utilities;
    using Opal.Logs;

    /// <summary>
    /// シェーダコンバータの結果を格納するクラスです。
    /// </summary>
    public class ShaderConverterResult
    {
        private string stdOutLogFilePath = string.Empty;
        private string stdErrLogFilePath = string.Empty;
        private byte[] data = null;
        private string outputPath = string.Empty;
        private ErrorViewLog errorViewLog = null;

        /// <summary>
        /// コンバート結果のデータを取得します。
        /// </summary>
        public byte[] Data
        {
            get
            {
                return this.data;
            }

            internal set
            {
                this.data = value;
            }
        }

        /// <summary>
        /// ファイル出力先を取得します。
        /// </summary>
        public string OutputPath
        {
            get
            {
                return this.outputPath;
            }

            internal set
            {
                this.outputPath = value;
            }
        }

        /// <summary>
        /// エラーがあるかを取得します。
        /// </summary>
        public bool HasError
        {
            get
            {
                return this.errorViewLog != null;
            }
        }

        /// <summary>
        /// ログの内容を出力するかを取得設定します。
        /// </summary>
        public bool OutputLog
        {
            get;
            set;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderConverterResult()
        {
            this.OutputLog = true;
        }

        /// <summary>
        /// コンバート結果の中間ファイルを取得します。
        /// </summary>
        /// <returns>中間ファイルを返します。</returns>
        public G3dFile GetG3dFile()
        {
            if (this.HasError)
            {
                return null;
            }

            G3dFile file = G3dIfUtility.ReadFile(this.OutputPath);
            if (file == null)
            {
                this.WriteError(ShaderConverterMessage.ErrorCannotReadFsd);
            }

            return file;
        }

        /// <summary>
        /// コンバート結果のシェーダ定義を取得します。
        /// </summary>
        /// <returns>シェーダ定義を返します。</returns>
        public ShaderDefinition GetShaderDefinition()
        {
            G3dFile file = this.GetG3dFile();
            if (file != null)
            {
                ShaderDefinition shaderDefinition = file.GetRootEntity<ShaderDefinition>();
                return shaderDefinition;
            }

            return null;
        }

        /// <summary>
        /// エラービューログを生成します。
        /// </summary>
        /// <returns>生成したエラービューログです。</returns>
        public ErrorViewLog CreateErrorViewLog()
        {
            ErrorViewLog warningLog = null;

            if (this.OutputLog)
            {
                StringBuilder logs;
                if (this.GetLogFileText(out logs))
                {
                    if (this.HasError)
                    {
                        this.errorViewLog.Details = logs.ToString();
                    }
                    else
                    {
                        // シェーダコンバータの標準エラーログがある場合には警告として表示する。
                        warningLog = new ErrorViewLog()
                        {
                            Category = ErrorViewLog.CategoryType.Warning,
                            Summary = ShaderConverterMessage.HasWarning,
                            Details = logs.ToString()
                        };
                    }
                }
            }

            return this.HasError ? this.errorViewLog : warningLog;
        }

        internal void OpenLogFile()
        {
            string path = TemporaryFileUtility.GetTemporaryDir();
            string name = TemporaryFileUtility.MakeTemporaryName();

            this.stdOutLogFilePath =
                Path.Combine(path, string.Format("ShaderConverter_OutLog.{0}.txt", name));
            this.stdErrLogFilePath =
                Path.Combine(path, string.Format("ShaderConverter_ErrLog.{0}.txt", name));

            ShaderConverterManager.OpenLogFileStream(stdOutLogFilePath, stdErrLogFilePath);
        }

        internal void CloseLogFile()
        {
            ShaderConverterManager.CloseLogFileStream();
        }

        internal void WriteError(string message)
        {
            this.errorViewLog = new ErrorViewLog()
            {
                Category = ErrorViewLog.CategoryType.Error,
                Summary = message
            };
        }

        private bool GetLogFileText(out StringBuilder logs)
        {
            logs = null;
            string stdOutLog = GetLogFileString(this.stdOutLogFilePath);
            string stdErrLog = GetLogFileString(this.stdErrLogFilePath);

            if (!string.IsNullOrEmpty(stdErrLog))
            {
                logs = new StringBuilder();
                logs.AppendLine(string.Empty);
                logs.AppendLine("From ShaderConverter -------------------------------");
                logs.AppendLine(stdOutLog);
                logs.AppendLine(stdErrLog);

                return true;
            }

            return false;
        }

        private static string GetLogFileString(string path)
        {
            string logFileStr = string.Empty;

            try
            {
                using (StreamReader reader = new StreamReader(path))
                {
                    logFileStr = reader.ReadToEnd();
                }

                File.Delete(path);
            }
            catch (Exception e)
            {
                string errMsg = string.Format("GetLogFileString Exception {0}", e.Message);
                Debug.WriteLine(errMsg);
            }

            return logFileStr;
        }
    }
}
