﻿namespace G3dCore.Converters
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.InteropServices;
    using G3dCore.Resources;
    using G3dCore.Utilities;
    using System.Reflection;

    /// <summary>
    /// シェーダコンバータマネージャクラスです。
    /// </summary>
    public class ShaderConverterManager
    {
        private static IntPtr dllHandle = IntPtr.Zero;
        private static bool glInitialized = false;

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void InitDelegate();
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void ShutdownDelegate();
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void InitGLContextDelegate();
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void ShutdownGLContextDelegate();
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ClearDelegate();
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate UInt32 GetCvtrVersionDelegate();
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool SetOptionsDelegate(
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool AddFileDelegate(
            IntPtr pData,
            int dataSize,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] paths,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool MakeDelegate();
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate int CalcDefinitionSizeDelegate(
            [MarshalAs(UnmanagedType.LPWStr)] string fullpath
        );
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate int CalcArchiveSizeDelegate(
        );
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool WriteDelegate(
            IntPtr ppData,
            int pDataSize
        );
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ConvertDelegate(
            IntPtr ppData,
            int pDataSize
        );
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool EndianSwapDelegate(
            IntPtr ppData,
            int pDataSize
        );
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate IntPtr OpenLogFileDelegate(
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] outArgv,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] errArgv
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void CloseLogFileDelegate(
        );

        private static InitDelegate Init;
        private static ShutdownDelegate Shutdown;
        private static InitGLContextDelegate InitGLContext;
        private static ShutdownGLContextDelegate ShutdownGLContext;
        private static ClearDelegate Clear;
        private static GetCvtrVersionDelegate GetCvtrVersion;
        private static SetOptionsDelegate SetOptions;
        private static AddFileDelegate AddFile;
        private static MakeDelegate Make;
        private static CalcDefinitionSizeDelegate CalcDefinitionSize;
        private static CalcArchiveSizeDelegate CalcArchiveSize;
        private static WriteDelegate Write;
        private static ConvertDelegate Convert;
        private static EndianSwapDelegate EndianSwap;
        private static OpenLogFileDelegate OpenLogFile;
        private static CloseLogFileDelegate CloseLogFile;

        private static string DllFilepath
        {
            get
            {
                string assemblyDir = System.IO.Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
                string g3dToolsDir = System.IO.Path.Combine(assemblyDir, "../3dTools");
                if (!System.IO.Directory.Exists(g3dToolsDir))
                {
                    // デバッグ実行時
                    g3dToolsDir = System.IO.Path.Combine(
                        assemblyDir,
                        "../../../../../../../../../Tools/Graphics/3dTools");
                }

                return System.IO.Path.Combine(
                    g3dToolsDir,
                    Environment.Is64BitProcess ? "3dShaderConverter.dll" : "x86/3dShaderConverter.dll");
            }
        }

        public static void Initialize()
        {
            string currentDir = Environment.CurrentDirectory;

            try
            {
                Directory.SetCurrentDirectory(Path.GetDirectoryName(DllFilepath));
                dllHandle = Kernel32.LoadLibrary(DllFilepath);

                SetExportFunction();
                Init();
            }
            catch (Exception e)
            {
                LogUtility.WriteError(e.ToString());
            }
            finally
            {
                Environment.CurrentDirectory = currentDir;
            }
        }

        public static void InitializeGL()
        {
            try
            {
                // GL関連の初期化は、こちらで行う。
                InitGLContext();
                glInitialized = true;
            }
            catch (Exception e)
            {
                LogUtility.WriteError(e.ToString());
            }
        }

        public static void FinalizeGL()
        {
            ShutdownGLContext();
        }

        public static void Destroy()
        {
            Clear = null;

            if (dllHandle != IntPtr.Zero)
            {
                Shutdown();

                Kernel32.FreeLibrary(dllHandle);
                dllHandle = IntPtr.Zero;
            }
        }

        ////---------------------------------------------------------------------

        /// <summary>
        /// シェーダコンバータにエラーログをオープンします。
        /// </summary>
        public static void OpenLogFileStream(string stdOutLogFilePath, string stdErrLogFilePath)
        {
            try
            {
                string mode = "w+";
                OpenLogFile(
                    new string[] { stdOutLogFilePath, mode },
                    new string[] { stdErrLogFilePath, mode });
            }
            catch (Exception e)
            {
                LogUtility.WriteError("[ShaderConverter] SetLogFileStream() failed: " + e.Message);
            }
        }

        /// <summary>
        /// シェーダコンバータのエラーログをクローズします。
        /// </summary>
        public static void CloseLogFileStream()
        {
            try
            {
                CloseLogFile();
            }
            catch (Exception e)
            {
                LogUtility.WriteError("[ShaderConverter] CloseLogFileStream() failed: " + e.Message);

                string errMsg = string.Format("ResetBinaryConverterLogFileStream Exception {0}", e.Message);
            }
        }

        /// <summary>
        /// シェーダ設定データからシェーダ定義データを生成します。
        /// </summary>
        public static ShaderConverterResult ConvertToShaderDefinition(
            byte[] shaderConfigData,
            string shaderConfigFilePath,
            string shaderDefinitionFilePath,
            string[] additionalOptions
        )
        {
            WriteDebug("ConvertToShaderDefinition() started.");

            var result = new ShaderConverterResult();
            result.OutputPath = shaderDefinitionFilePath;
            result.OpenLogFile();

            try
            {
                if (Clear == null)
                {
                    Initialize();

                    if (!glInitialized)
                    {
                        InitializeGL();
                    }
                }

                if (!Clear())
                {
                    WriteDebug("Clear failed.");
                    result.WriteError(ShaderConverterMessage.ErrorUnknown);
                    return result;
                }

                string[] options = new string[] { null };
                if (additionalOptions != null)
                {
                    options = Enumerable.Concat(additionalOptions, options).ToArray();
                }

                if (!SetOptions(options))
                {
                    WriteDebug("SetOptions failed.");
                    result.WriteError(ShaderConverterMessage.ErrorSetOptions);
                    return result;
                }

                unsafe
                {
                    fixed (byte* pData = shaderConfigData)
                    {
                        if (!AddFile(
                                (IntPtr)pData,
                                Marshal.SizeOf(shaderConfigData[0]) * shaderConfigData.Length,
                                new string[] {
                                    shaderConfigFilePath,
                                    Path.GetFileNameWithoutExtension(shaderConfigFilePath),
                                    Path.GetExtension(shaderConfigFilePath) },
                                new string[] { null })
                            )
                        {
                            WriteDebug("AddFile failed.");
                            result.WriteError(ShaderConverterMessage.ErrorSetOptions);
                            return result;
                        }

                        if (!Make())
                        {
                            WriteDebug("Make failed.");
                            result.WriteError(ShaderConverterMessage.ErrorMake);
                            return result;
                        }

                        int size = CalcDefinitionSize(shaderDefinitionFilePath);
                        if (size == 0)
                        {
                            WriteDebug("CalcDefinitionSize failed.");
                            result.WriteError(ShaderConverterMessage.ErrorCalcDefinitionSize);
                            return result;
                        }

                        byte[] output = new byte[size / Marshal.SizeOf(typeof(byte))];
                        fixed (byte* pOutput = output)
                        {
                            if (!Write((IntPtr)pOutput, size))
                            {
                                WriteDebug("Write failed.");
                                result.WriteError(ShaderConverterMessage.ErrorSaveFile);
                                return result;
                            }

                            result.Data = output;

                            if (File.Exists(shaderDefinitionFilePath))
                            {
                                File.Delete(shaderDefinitionFilePath);
                            }

                            // バイナリ形式でファイルに書き出し。
                            using (BinaryWriter w = new BinaryWriter(File.OpenWrite(shaderDefinitionFilePath)))
                            {
                                w.Write(output, 0, size);
                            }
                        }
                    }
                }
            }
            catch (Exception e)
            {
                WriteDebug("Exception: " + e.ToString());
                result.WriteError(ShaderConverterMessage.ErrorUnknown);
                result.Data = null;
                return result;
            }
            finally
            {
                result.CloseLogFile();
            }

            WriteDebug("ConvertToShaderDefinition() finished successfully.");
            return result;
        }


        ////---------------------------------------------------------------------

        private static void SetExportFunction()
        {
            Init = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterInitialize"), typeof(InitDelegate)) as InitDelegate;
            Shutdown = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterShutdown"), typeof(ShutdownDelegate)) as ShutdownDelegate;
            InitGLContext = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterInitializeGlContext"), typeof(InitGLContextDelegate)) as InitGLContextDelegate;
            ShutdownGLContext = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterShutdownGlContext"), typeof(ShutdownGLContextDelegate)) as ShutdownGLContextDelegate;
            Clear = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterClear"), typeof(ClearDelegate)) as ClearDelegate;
            GetCvtrVersion = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterGetConverterVersion"), typeof(GetCvtrVersionDelegate)) as GetCvtrVersionDelegate;
            SetOptions = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterSetOptions"), typeof(SetOptionsDelegate)) as SetOptionsDelegate;
            AddFile = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterAddFile"), typeof(AddFileDelegate)) as AddFileDelegate;
            CalcArchiveSize = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterCalculateArchiveSize"), typeof(CalcArchiveSizeDelegate)) as CalcArchiveSizeDelegate;
            Convert = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterConvert"), typeof(ConvertDelegate)) as ConvertDelegate;
            EndianSwap = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterSwapEndian"), typeof(EndianSwapDelegate)) as EndianSwapDelegate;
            Make = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterMake"), typeof(MakeDelegate)) as MakeDelegate;
            CalcDefinitionSize = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterCalculateDefinitionSize"), typeof(CalcDefinitionSizeDelegate)) as CalcDefinitionSizeDelegate;
            Write = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterWrite"), typeof(WriteDelegate)) as WriteDelegate;

            // ログファイル出力用
            OpenLogFile = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterOpenLogFile"), typeof(OpenLogFileDelegate)) as OpenLogFileDelegate;
            CloseLogFile = Marshal.GetDelegateForFunctionPointer(Kernel32.GetProcAddress(dllHandle, "nng3dToolShaderConverterCloseLogFile"), typeof(CloseLogFileDelegate)) as CloseLogFileDelegate;
        }

        [Conditional("DEBUG")]
        private static void WriteDebug(string message)
        {
            Debug.WriteLine("[ShaderConverter] " + message);
        }
    }
}
