﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Entities;
    using G3dCore.Resources;

    /// <summary>
    /// シェーダ設定定義を追加や編集するウィザードのビューモデルです。
    /// </summary>
    public class WizardShaderViewModel : WizardViewModel
    {
        private readonly ShaderSettingViewModel shaderSettingViewModel;
        private readonly MacroSettingViewModel macroSettingViewModel;
        private readonly ShaderConverterViewModel shaderConverterViewModel;
        private readonly string[] includeFullPaths;
        private readonly string filePath;

        private ShaderConfigViewModel shaderConfigViewModel;
        private ShaderViewModel shaderViewModel;

        /// <summary>
        /// 対象の ShaderConfigViewModel を取得します。
        /// </summary>
        public ShaderConfigViewModel ShaderConfigViewModel
        {
            get
            {
                return this.shaderConfigViewModel;
            }
        }

        /// <summary>
        /// 対象の ShaderViewModel を取得します。
        /// </summary>
        public ShaderViewModel ShaderViewModel
        {
            get
            {
                return this.shaderViewModel;
            }
        }

        /// <summary>
        /// インクルードパスのフルパスを取得します。
        /// </summary>
        public string[] IncludeFullPaths
        {
            get
            {
                return this.includeFullPaths;
            }
        }

        /// <summary>
        /// シェーダ設定ファイルパスを取得します。
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.filePath;
            }
        }

        /// <summary>
        /// シェーダ設定ファイルディレクトリを取得します。
        /// </summary>
        public string FileDir
        {
            get
            {
                return Path.GetDirectoryName(this.FilePath);
            }
        }

        /// <summary>
        /// シェーダ名を取得します。
        /// </summary>
        public string ShaderName
        {
            get;
            private set;
        }

        /// <summary>
        /// マテリアルシェーダかどうかを取得します。
        /// </summary>
        public bool IsMaterialShader
        {
            get;
            private set;
        }

        /// <summary>
        /// 頂点シェーダパスを取得します。
        /// </summary>
        public string VertexShaderPath
        {
            get;
            private set;
        }

        /// <summary>
        /// ジオメトリシェーダパスを取得します。
        /// </summary>
        public string GeometryShaderPath
        {
            get;
            private set;
        }

        /// <summary>
        /// フラグメントシェーダパスを取得します。
        /// </summary>
        public string FragmentShaderPath
        {
            get;
            private set;
        }

        /// <summary>
        /// 演算シェーダパスを取得します。
        /// </summary>
        public string ComputeShaderPath
        {
            get;
            private set;
        }

        /// <summary>
        /// マクロビューモデルを取得します。
        /// </summary>
        public MacroViewModel[] MacroViewModels
        {
            get;
            private set;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="shaderConfigViewModel">対象のシェーダ設定ビューモデルです。</param>
        /// <param name="shaderViewModel">対象のシェーダビューモデルです。
        /// null を指定した場合には追加モードになります。</param>
        public WizardShaderViewModel(ShaderConfigViewModel shaderConfigViewModel, ShaderViewModel shaderViewModel)
        {
            if (shaderConfigViewModel == null) throw new ArgumentNullException();

            this.shaderConfigViewModel = shaderConfigViewModel;
            this.shaderViewModel = shaderViewModel;

            {
                List<string> includeFullPathsList = new List<string>();
                foreach (IncludePath includePath in this.shaderConfigViewModel.ShaderConfigData.IncludePaths)
                {
                    string includeFullPath =
                        Path.GetFullPath(Path.Combine(
                            Path.GetDirectoryName(this.shaderConfigViewModel.FileViewModel.FilePath),
                            includePath.Path));
                    includeFullPathsList.Add(includeFullPath);
                }

                this.includeFullPaths = includeFullPathsList.ToArray();
                this.filePath = shaderConfigViewModel.FileViewModel.FilePath;
            }

            this.shaderSettingViewModel = new ShaderSettingViewModel(this, this.shaderViewModel == null);
            this.macroSettingViewModel = new MacroSettingViewModel(this);
            this.shaderConverterViewModel = new ShaderConverterViewModel(this, false);

            this.PageViewModels = new WizardPageViewModel[]
            {
                this.shaderSettingViewModel,
                this.macroSettingViewModel,
                this.shaderConverterViewModel
            };

            if (this.shaderViewModel != null)
            {
                this.SetEditData();
                this.Title = Labels.EditShader;
            }
            else
            {
                this.Title = Labels.AddShader;
            }
        }

        /// <summary>
        /// ページを移動した際の処理です。
        /// </summary>
        /// <param name="pageNo">ページ番号です。</param>
        /// <param name="goNext">次へを押された場合には true になります。</param>
        public override void ProcPage(int pageNo, bool goNext)
        {
            if (this.PageViewModels[pageNo] is MacroSettingViewModel && goNext)
            {
                this.ShaderName = this.shaderSettingViewModel.Name;
                this.IsMaterialShader = this.shaderSettingViewModel.IsMaterialShader;
                this.VertexShaderPath = this.shaderSettingViewModel.VertexShaderSourceSettingViewModel.ShaderSource;
                this.GeometryShaderPath = this.shaderSettingViewModel.GeometryShaderSourceSettingViewModel.ShaderSource;
                this.FragmentShaderPath = this.shaderSettingViewModel.FragmentShaderSourceSettingViewModel.ShaderSource;
                this.ComputeShaderPath = this.shaderSettingViewModel.ComputeShaderSourceSettingViewModel.ShaderSource;

                this.MacroViewModels = this.macroSettingViewModel.GetMacroViewModels();

                this.shaderConverterViewModel.Reset();
            }
        }

        /// <summary>
        /// ShaderViewModel を生成します。
        /// </summary>
        /// <returns>生成した ShaderViewModel を返します。</returns>
        public ShaderViewModel CreateShaderViewModel()
        {
            ShaderViewModel viewModel = null;

            if (this.shaderConverterViewModel.IsConvertCheckOK)
            {
                viewModel = new ShaderViewModel(this.shaderConverterViewModel.GetConvertedShader(), this.shaderConfigViewModel);
                viewModel.AddVariationsFromShaderDefinition(this.shaderConverterViewModel.GetConvertedShaderDefinition());
                viewModel.UpdateAndAddVariations();
            }

            return viewModel;
        }

        private void SetEditData()
        {
            this.shaderSettingViewModel.Name = this.shaderViewModel.ShaderData.Name;
            this.shaderSettingViewModel.IsMaterialShader = this.shaderViewModel.ShaderData.MaterialShader;

            if (this.shaderViewModel.ShaderData.VertexShader != null)
            {
                this.shaderSettingViewModel.VertexShaderSourceSettingViewModel.ShaderSource =
                    this.shaderViewModel.ShaderData.VertexShader.Path;
            }

            if (this.shaderViewModel.ShaderData.GeometryShader != null)
            {
                this.shaderSettingViewModel.GeometryShaderSourceSettingViewModel.ShaderSource =
                    this.shaderViewModel.ShaderData.GeometryShader.Path;
            }

            if (this.shaderViewModel.ShaderData.FragmentShader != null)
            {
                this.shaderSettingViewModel.FragmentShaderSourceSettingViewModel.ShaderSource =
                    this.shaderViewModel.ShaderData.FragmentShader.Path;
            }

            if (this.shaderViewModel.ShaderData.ComputeShader != null)
            {
                this.shaderSettingViewModel.ComputeShaderSourceSettingViewModel.ShaderSource =
                    this.shaderViewModel.ShaderData.ComputeShader.Path;
            }

            this.macroSettingViewModel.MacroViewModels.Clear();

            foreach (Macro macro in this.shaderViewModel.ShaderData.Macros)
            {
                var macroViewModel = new MacroViewModel()
                {
                    Name = macro.Name,
                    Value = macro.Value
                };

                this.macroSettingViewModel.MacroViewModels.Add(macroViewModel);
            }
        }
    }
}
