﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows;
    using Opal.ViewModels;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands;

    /// <summary>
    /// 編集可能なリストボックスアイテムのビューモデルです。
    /// </summary>
    public class EditableListBoxItemViewModel : ViewModel
    {
        private object tag = null;
        private string label;
        private string editText;
        private bool hasError;
        private bool isEditable;
        private bool isSelected;
        private bool isReadyTextBox = false;
        private Visibility selectedItemVisibility = Visibility.Collapsed;

        /// <summary>
        /// アイテム編集前デリゲートです。
        /// </summary>
        /// <param name="item">対象アイテムです。</param>
        /// <returns>編集テキストを返します。</returns>
        public delegate string BeforeEditDelegate(EditableListBoxItemViewModel item);

        /// <summary>
        /// アイテム編集後デリゲートです。
        /// </summary>
        /// <param name="item">対象アイテムです。</param>
        /// <param name="editText">編集テキストです。</param>
        public delegate void AfterEditDelegate(EditableListBoxItemViewModel item, string editText);

        /// <summary>
        /// タグを取得設定します。
        /// </summary>
        public Object Tag
        {
            get
            {
                return this.tag;
            }

            set
            {
                if (value != this.tag)
                {
                    this.tag = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// ラベルを取得設定します。
        /// </summary>
        public string Label
        {
            get
            {
                return this.label;
            }

            set
            {
                if (value != this.label)
                {
                    this.label = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// 編集テキストを取得します。
        /// </summary>
        public string EditText
        {
            get
            {
                return this.editText;
            }

            protected set
            {
                if (value != this.editText)
                {
                    this.editText = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// エラーがあるかを取得設定します。
        /// </summary>
        public bool HasError
        {
            get
            {
                return this.hasError;
            }

            set
            {
                if (value != this.hasError)
                {
                    this.hasError = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// 選択されているかを取得設定します。
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (value != this.isSelected)
                {
                    this.isSelected = value;
                    this.RaisePropertyChanged();

                    if (!this.isSelected)
                    {
                        this.isReadyTextBox = false;

                        if (this.selectedItemVisibility == Visibility.Visible)
                        {
                            if (this.AfterEdit != null)
                            {
                                this.AfterEdit.Invoke(this, this.EditText);
                            }
                            else
                            {
                                this.Label = this.EditText;
                            }

                            this.selectedItemVisibility = Visibility.Collapsed;
                            this.RaisePropertyChanged(() => this.TextBlockVisibility);
                            this.RaisePropertyChanged(() => this.TextBoxVisibility);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// TextBlock のビジビリティを取得します。
        /// </summary>
        public Visibility TextBlockVisibility
        {
            get
            {
                return this.selectedItemVisibility == Visibility.Visible ? Visibility.Collapsed : Visibility.Visible;
            }
        }

        /// <summary>
        /// TextBox のビジビリティを取得します。
        /// </summary>
        public Visibility TextBoxVisibility
        {
            get
            {
                return this.selectedItemVisibility;
            }
        }

        /// <summary>
        /// マウスボタンが押されているかを設定取得します。
        /// </summary>
        public bool IsMouseDownProcessed
        {
            get;
            set;
        }

        /// <summary>
        /// アイテム編集前デリゲートを取得設定します。
        /// </summary>
        public BeforeEditDelegate BeforeEdit
        {
            get;
            set;
        }

        /// <summary>
        /// アイテム編集後デリゲートを取得設定します。
        /// </summary>
        public AfterEditDelegate AfterEdit
        {
            get;
            set;
        }

        /// <summary>
        /// マウス Up コマンドを取得します。
        /// </summary>
        public DelegateCommand MouseUpCommand
        {
            get
            {
                return new DelegateCommand(this.MouseUpExecute);
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="isEditable">編集可能かを設定します。</param>
        public EditableListBoxItemViewModel(bool isEditable)
        {
            this.isEditable = isEditable;
        }

        private void MouseUpExecute()
        {
            this.IsMouseDownProcessed = true;

            if (this.isEditable && this.IsSelected)
            {
                if (this.isReadyTextBox && this.selectedItemVisibility == Visibility.Collapsed)
                {
                    if (this.BeforeEdit != null)
                    {
                        this.EditText = this.BeforeEdit.Invoke(this);
                    }
                    else
                    {
                        this.EditText = this.Label;
                    }

                    this.selectedItemVisibility = Visibility.Visible;
                    this.RaisePropertyChanged(() => this.TextBlockVisibility);
                    this.RaisePropertyChanged(() => this.TextBoxVisibility);
                }
                else
                {
                    this.isReadyTextBox = true;
                }
            }
        }
    }
}
