﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using ShaderAssistAddons.Modules.ShaderConfig.Actions;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands.Generic;
    using G3dCore.Entities;
    using G3dCore.Extensions;
    using G3dCore.Messaging;
    using G3dCore.Modules.ErrorView;
    using G3dCore.ViewModels;
    using Opal.Operations;

    /// <summary>
    /// シェーダ設定ビューモデルクラスです。
    /// </summary>
    public class ShaderConfigViewModel : G3dDocumentViewModel
    {
        private readonly ObservableCollection<ShaderViewModel> shaderViewModels = new ObservableCollection<ShaderViewModel>();
        private readonly ShaderSourceListViewModel shaderSourceListViewModel = new ShaderSourceListViewModel();
        private readonly Messenger wizardMessenger = new Messenger();
        private WeakReference<ShaderConfig> shaderConfigData;
        private ShaderViewModel selectedShaderViewModel = null;
        private bool isControlEnabled = true;

        /// <summary>
        /// シェーダ設定定義ビューモデルを取得します。
        /// </summary>
        public ObservableCollection<ShaderViewModel> ShaderViewModels
        {
            get
            {
                return this.shaderViewModels;
            }
        }

        /// <summary>
        /// シェーダソース一覧ビューモデルを取得します。
        /// </summary>
        public ShaderSourceListViewModel ShaderSourceListViewModel
        {
            get
            {
                return this.shaderSourceListViewModel;
            }
        }

        /// <summary>
        /// シェーダ設定のデータを取得します。
        /// </summary>
        public ShaderConfig ShaderConfigData
        {
            get
            {
                ShaderConfig data;
                this.shaderConfigData.TryGetTarget(out data);
                Debug.Assert(data != null);
                return data;
            }
        }

        /// <summary>
        /// 選択シェーダ設定定義ビューモデルを取得します。
        /// </summary>
        public ShaderViewModel SelectedShaderViewModel
        {
            get
            {
                return this.selectedShaderViewModel;
            }

            protected set
            {
                if (value != this.selectedShaderViewModel)
                {
                    this.selectedShaderViewModel = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// ファイル名を取得します。
        /// </summary>
        public string FileName
        {
            get
            {
                return Path.GetFileName(this.File.FilePath);
            }
        }

        /// <summary>
        /// コントロールが有効かどうかを取得設定します。
        /// </summary>
        public bool IsControlEnabled
        {
            get
            {
                return this.isControlEnabled;
            }

            set
            {
                this.SetProperty(ref this.isControlEnabled, value);
            }
        }

#region DelegateCommands

        /// <summary>
        /// シェーダノードの選択変化時コマンドを取得します。
        /// </summary>
        public DelegateCommand<object> SelectedItemChangedCommand
        {
            get
            {
                return new DelegateCommand<object>(this.SelectedItemChangedExecute);
            }
        }

        /// <summary>
        /// fsc 設定実行時コマンドを取得します。
        /// </summary>
        public DelegateCommand ShaderConfigSettingCommand
        {
            get
            {
                return new DelegateCommand(this.ShaderConfigSettingExecute);
            }
        }

        /// <summary>
        /// バリエーション再抽出実行時コマンドを取得します。
        /// </summary>
        public DelegateCommand RenewalVariationCommand
        {
            get
            {
                return new DelegateCommand(this.RenewalVariationExecute);
            }
        }

        /// <summary>
        /// シェーダ設定の追加実行時コマンドを取得します。
        /// </summary>
        public DelegateCommand AddShaderCommand
        {
            get
            {
                return new DelegateCommand(this.AddShaderExecute, this.CanAddShader);
            }
        }

        /// <summary>
        /// シェーダ設定の削除実行時コマンドを取得します。
        /// </summary>
        public DelegateCommand RemoveShaderCommand
        {
            get
            {
                return new DelegateCommand(this.RemoveShaderExecute, this.CanRemoveShader);
            }
        }

        /// <summary>
        /// シェーダ設定の編集実行時コマンドを取得します。
        /// </summary>
        public DelegateCommand EditShaderCommand
        {
            get
            {
                return new DelegateCommand(this.EditShaderExecute, this.CanEditShader);
            }
        }

#endregion

        /// <summary>
        /// ウィザード表示メッセンジャーを取得します。
        /// </summary>
        public Messenger WizardMessenger
        {
            get
            {
                return this.wizardMessenger;
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderConfigViewModel()
        {
        }

        /// <summary>
        /// ViewModelの初期化を行います。
        /// </summary>
        protected override async void SetupInternal()
        {
            this.IsControlEnabled = false;

            var shaderConfig = this.File.GetRootEntity<ShaderConfig>();

            this.shaderConfigData = new WeakReference<ShaderConfig>(shaderConfig);

            this.shaderViewModels.Clear();

            using (var logger = new ErrorViewLogger())
            {
                foreach (var shader in shaderConfig.Shaders)
                {
                    var viewModel = new ShaderViewModel(shader, this);

                    ErrorViewLog log = null;
                    await Task.Run(() =>
                        {
                            log = viewModel.ExtractVariationFromShaderDefinition();
                        });

                    if (log != null)
                    {
                        logger.Add(log);
                    }

                    this.shaderViewModels.Add(viewModel);
                }

                this.File.Reset();

                foreach (ShaderViewModel shaderViewModel in this.shaderViewModels)
                {
                    shaderViewModel.UpdateAndAddVariations();
                }
            }

            this.IsControlEnabled = true;
        }

        /// <summary>
        /// オブジェクト破棄の内部処理です。継承した先で固有の処理を実装します。
        /// </summary>
        protected override void DisposeInternal()
        {
        }

        private void ShaderConfigSettingExecute()
        {
            var wizardViewModel = new WizardShaderConfigViewModel(this);
            var input = new WizardAction.InputArg()
            {
                WizardViewModel = wizardViewModel
            };

            this.WizardMessenger.Raise(
                new Message(input),
                m =>
                {
                    var output = (WizardAction.OutputArg)m.Response;
                    if (output.Result == true)
                    {
                        var data = this.ShaderConfigData;

                        OperationSet operations = new OperationSet();
                        operations.Description = "シェーダ設定の編集";

                        operations.Add(
                            new PropertyEditOperation<G3dFileViewModel, string>(this.FileViewModel, "FilePath", wizardViewModel.FilePath));
                        operations.Add(
                            data.ShaderConfigInfo.CreateEditCodePageOperation(wizardViewModel.CodePage));

                        operations.Add(
                            data.CreateRelpaceIncludePathsOperation(wizardViewModel.GetIncludePaths(wizardViewModel.FileDir)));
                        operations.Add(
                            data.CreateRelpaceForceIncludeFilesOperation(wizardViewModel.GetForceIncludeFiles()));

                        operations.Add(wizardViewModel.CreateAdjustShaderPathOperation(data.Shaders));

                        this.ExecuteOperation(operations);
                    }
                });
        }

        private void RenewalVariationExecute()
        {
            string selectedShaderName = null;

            if (this.SelectedShaderViewModel != null)
            {
                selectedShaderName = this.SelectedShaderViewModel.Name;
            }

            this.SetupInternal();

            if (selectedShaderName != null)
            {
                this.SelectedItemChangedExecute(
                    this.shaderViewModels.FirstOrDefault(x => x.Name == selectedShaderName));
            }
        }

        private void AddShaderExecute()
        {
            var wizardViewModel = new WizardShaderViewModel(this, null);
            var input = new WizardAction.InputArg()
            {
                WizardViewModel = wizardViewModel
            };

            this.WizardMessenger.Raise(
                new Message(input),
                m =>
                {
                    var output = (WizardAction.OutputArg)m.Response;
                    if (output.Result == true)
                    {
                        var data = this.ShaderConfigData;

                        ShaderViewModel newShaderViewModel = wizardViewModel.CreateShaderViewModel();
                        if (newShaderViewModel != null)
                        {
                            OperationSet operations = new OperationSet();
                            operations.Description = "シェーダの追加";

                            operations.Add(
                                data.CreateAddShadersOperation(newShaderViewModel.ShaderData));

                            operations.Add(
                                new ListItemAddOperation<ShaderViewModel>(this.shaderViewModels, newShaderViewModel));

                            this.ExecuteOperation(operations);
                        }
                    }
                });
        }

        private bool CanAddShader()
        {
            return true;
        }

        private void RemoveShaderExecute()
        {
            var shaderConfig = this.ShaderConfigData;
            Shader shader = this.selectedShaderViewModel.ShaderData;

            OperationSet operations = new OperationSet();
            operations.Description = "シェーダの追加";

            operations.Add(
                shaderConfig.CreateRemoveShadersOperation(shader));

            operations.Add(
                new ListItemRemoveOperation<ShaderViewModel>(this.shaderViewModels, this.selectedShaderViewModel));
            operations.Add(
                new PropertyEditOperation<ShaderConfigViewModel, ShaderViewModel>(this, "SelectedShaderViewModel", null));

            this.ExecuteOperation(operations);
        }

        private bool CanRemoveShader()
        {
            return this.selectedShaderViewModel != null;
        }

        private void EditShaderExecute()
        {
            var wizardViewModel = new WizardShaderViewModel(this, this.selectedShaderViewModel);
            var input = new WizardAction.InputArg()
            {
                WizardViewModel = wizardViewModel
            };

            this.WizardMessenger.Raise(
                new Message(input),
                m =>
                {
                    var output = (WizardAction.OutputArg)m.Response;
                    if (output.Result == true)
                    {
                        Shader data = this.selectedShaderViewModel.ShaderData;

                        OperationSet operations = new OperationSet();
                        operations.Description = "シェーダの編集";

                        operations.Add(
                            data.CreateEditOperation(
                                wizardViewModel.ShaderName,
                                wizardViewModel.IsMaterialShader,
                                wizardViewModel.VertexShaderPath,
                                wizardViewModel.GeometryShaderPath,
                                wizardViewModel.FragmentShaderPath,
                                wizardViewModel.ComputeShaderPath));

                        operations.Add(
                            data.CreateClearMacroOperation());
                        foreach (MacroViewModel macroViewModel in wizardViewModel.MacroViewModels)
                        {
                            operations.Add(
                                data.CreateAddMacroOperation(macroViewModel.Name, macroViewModel.Value));
                        }

                        this.ExecuteOperation(operations);
                    }
                });
        }

        private bool CanEditShader()
        {
            return this.selectedShaderViewModel != null;
        }

        private void SelectedItemChangedExecute(object selectedItem)
        {
            if (selectedItem is ShaderViewModel)
            {
                this.SelectedShaderViewModel = (ShaderViewModel)selectedItem;
                this.shaderSourceListViewModel.SetCurrentShaderViewModel(this.SelectedShaderViewModel, this);
            }
            else
            {
                this.SelectedShaderViewModel = null;
                this.shaderSourceListViewModel.SetCurrentShaderViewModel(null, null);
            }
        }

        private void ExecuteOperation(OperationSet operations)
        {
            if (operations.Count > 0)
            {
                this.FileViewModel.ExecuteOperation(operations);
            }
        }
    }
}
