﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;

namespace Viewer
{
    /// <summary>
    /// メッセージキュー
    /// </summary>
    public sealed class MessageQueue
    {
        private readonly object syncObject_ = new object();

        private readonly Queue<BaseMessage>	_queue = new Queue<BaseMessage>();
        private ProcessSameMessage _process = new ProcessSameMessage();

        private bool canPush = true;

        public bool IsExist
        {
            get{
                lock (syncObject_)
                {
                    return _queue.Count > 0;
                }
            }
        }

        /// <summary>ProcessSameMessageクラスの設定</summary>
        public ProcessSameMessage ProcessClass
        {
            set
            {
                _process = value;
            }
        }

        public void PushMessage(BaseMessage message)
        {
            if (canPush)
            {
                lock(syncObject_)
                {
                    _process.Enqueue(_queue, message);
                }
            }
        }

        public void PushMessages(IEnumerable<BaseMessage> messages)
        {
            if (canPush)
            {
                lock(syncObject_)
                {
                    foreach(BaseMessage message in messages)
                    {
                        _process.Enqueue(_queue, message);
                    }
                }
            }
        }

        public BaseMessage PopMessage()
        {
            if (_queue.Count > 0)
            {
                lock(syncObject_)
                {
                    if (_queue.Count > 0)
                    {
                        return _queue.Dequeue();
                    }
                }
            }

            return null;
        }

        public BaseMessage[] PopMessages()
        {
            if (_queue.Count > 0)
            {
                lock(syncObject_)
                {
                    BaseMessage[] messages = _queue.ToArray();

                    _queue.Clear();

                    return messages;
                }
            }

            return new BaseMessage[0];
        }

        public void ClearMessages()
        {
            if (_queue.Count > 0)
            {
                lock(syncObject_)
                {
                    _queue.Clear();
                }
            }
        }

        public void StartCanPush()
        {
            lock(syncObject_)
            {
                canPush = true;
            }
        }

        public void StartCanPush(BaseMessage message)
        {
            lock(syncObject_)
            {
                canPush = true;

                _process.Enqueue(_queue, message);
            }
        }

        public void StartCanPush(IEnumerable<BaseMessage> messages)
        {
            lock(syncObject_)
            {
                canPush = true;

                foreach(BaseMessage message in messages)
                {
                	_process.Enqueue(_queue, message);
                }
            }
        }

        public void StopCanPush()
        {
            lock(syncObject_)
            {
                canPush = false;
            }
        }
    }
    /// <summary>
    /// メッセージキューに同じメッセージがある場合に処理を行うクラス
    /// </summary>
    public class ProcessSameMessage
    {
        /// <summary>処理を行う</summary>
        public virtual void Enqueue(Queue<BaseMessage> queue, BaseMessage message)
        {
            // 圧縮不可の場合は、追加して抜ける
            if (!message.IsCompressible)
            {
                queue.Enqueue(message);
                return;
            }

            BaseMessage[] messages = queue.ToArray();

            int count = messages.GetLength(0);
            if (count == 0)
            {
                // メッセージが入っていない場合はreturn。
                queue.Enqueue(message);
                return;
            }

            queue.Clear();

            // 消滅する場合は末尾を消去
            if (message.Anihhilate(messages[messages.Length - 1]))
            {
                for (int i = 0; i < messages.Length - 1; i++)
                {
                    queue.Enqueue(messages[i]);
                }
                return;
            }

            // 同一メッセージの消去
            foreach (BaseMessage Qmessage in messages)
            {
                //　対象メッセージでなければ、キューに追加する。
                if (message.GetType() != Qmessage.GetType())
                {
                    queue.Enqueue(Qmessage);
                }
                else
                {
                    // 同じメッセージでも、編集ターゲットが違う場合などがあるので、
                    // メッセージごとの判定メソッドを呼び出して判定する。
                    if (!message.IsSameTarget(Qmessage))
                    {
                        queue.Enqueue(Qmessage);
                    }
                    else
                    {
                        // debug
                        App.DebugConsole.WriteLine("ProcessSameMessage : ({0})", Qmessage.ToString());
                    }
                }
            }

            queue.Enqueue(message);
        }
    }
}
