﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Windows.Forms;

namespace App.Utility
{
    /// <summary>
    /// マウスドラッグクラス。
    /// </summary>
    public sealed class MouseDrag
    {
        // ターゲット
        private readonly Control _target;
        // ボタンフラグ
        private MouseButtons _button = MouseButtons.None;
        // 開始点
        private Point _startPt = Point.Empty;
        // 中継点
        private Point _relayPt = Point.Empty;
        // 現在点
        private Point _currentPt = Point.Empty;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MouseDrag(Control target)
        {
            _target = target;
        }

        /// <summary>
        /// ボタン。
        /// </summary>
        public MouseButtons Button
        {
            get { return _button; }
        }

        /// <summary>
        /// 開始点。
        /// </summary>
        public Point StartPt
        {
            get { return _startPt; }
            set { _startPt = value; }
        }

        /// <summary>
        /// 中継点。
        /// </summary>
        public Point RelayPt
        {
            get { return _relayPt; }
            set { _relayPt = value; }
        }

        /// <summary>
        /// 現在点。
        /// </summary>
        public Point CurrentPt
        {
            get { return _currentPt; }
            set { _currentPt = value; }
        }

        /// <summary>
        /// ドラッグ領域。
        /// </summary>
        public Rectangle DragBounds
        {
            get
            {
                return new Rectangle(
                    Math.Min(_startPt.X, _currentPt.X),
                    Math.Min(_startPt.Y, _currentPt.Y),
                    Math.Abs(_currentPt.X - _startPt.X),
                    Math.Abs(_currentPt.Y - _startPt.Y)
                );
            }
        }

        /// <summary>
        /// 増分領域。
        /// </summary>
        public Rectangle DeltaBounds
        {
            get
            {
                return new Rectangle(
                    Math.Min(_relayPt.X, _currentPt.X),
                    Math.Min(_relayPt.Y, _currentPt.Y),
                    Math.Abs(_currentPt.X - _relayPt.X),
                    Math.Abs(_currentPt.Y - _relayPt.Y)
                );
            }
        }

        /// <summary>
        /// ドラッグサイズ。
        /// </summary>
        public Size DragSize
        {
            get
            {
                return new Size(
                    _currentPt.X - _startPt.X,
                    _currentPt.Y - _startPt.Y
                );
            }
        }

        /// <summary>
        /// 増分サイズ。
        /// </summary>
        public Size DeltaSize
        {
            get
            {
                return new Size(
                    _currentPt.X - _relayPt.X,
                    _currentPt.Y - _relayPt.Y
                );
            }
        }

        /// <summary>
        /// 開始。
        /// </summary>
        public void Start(Point point, MouseButtons button)
        {
            _button    = button;
            _startPt   = point;
            _relayPt   = point;
            _currentPt = point;

            // キャプチャ設定
            if (!_target.Capture)
            {
                _target.Capture = true;
            }
        }

        /// <summary>
        /// ドラッグ。
        /// </summary>
        public void Drag(Point point)
        {
            _relayPt   = _currentPt;
            _currentPt = point;
        }

        /// <summary>
        /// 終了。
        /// </summary>
        public void End()
        {
            _button    = MouseButtons.None;
            _startPt   = Point.Empty;
            _relayPt   = Point.Empty;
            _currentPt = Point.Empty;

            // キャプチャ解除
            if (_target.Capture)
            {
                _target.Capture = false;
            }
        }

        /// <summary>
        /// ドラッグ中かどうか。
        /// </summary>
        public bool IsDragging()
        {
            return _button != MouseButtons.None;
        }

        /// <summary>
        /// ドラッグ中かどうか。
        /// </summary>
        public bool IsDragging(MouseButtons button)
        {
            return _button == button;
        }
    }
}
