﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using App.Controls;

namespace App.Utility
{
    /// <summary>
    /// Clipboard での ExternalException 発生時に、再試行確認ダイアログの表示を追加したユーティリティクラス。
    /// </summary>
    static class ClipboardUtility
    {
        public static IDataObject GetDataObject()
        {
            bool retry = false;
            do
            {
                try
                {
                    // 他アプリがクリップボードにアクセスしている場合はクリップボードから取得に失敗し、ExternalException が発生する。
                    // Clipboard.GetDataObject() では 100 ミリ秒毎に 10 回の取得を試みる。
                    return Clipboard.GetDataObject();
                }
                catch (System.Runtime.InteropServices.ExternalException)
                {
                    // 他アプリがクリップボードにアクセスしているため取得できなかった。
                    // 再試行を確認する。
                    retry = UIMessageBox.YesNo(App.res.Strings.Clipboard_DataFetchFailure);
                }
            }
            while (retry);

            return null;
        }

        public static void SetDataObject(object data)
        {
            // 引数 copy への設定値は Clipboard.SetDataObject() のデフォルト値。
            SetDataObject(data, false);
        }

        public static void SetDataObject(object data, bool copy)
        {
            // 引数 retryTimes、retryDelay への設定値は Clipboard.SetDataObject() のデフォルト値。
            SetDataObject(data, copy, 10, 100);
        }

        public static void SetDataObject(object data, bool copy, int retryTimes, int retryDelay)
        {
            bool retry = false;
            do
            {
                try
                {
                    // 他アプリがクリップボードにアクセスしている場合はクリップボードへの設定に失敗するので、retryDelay ミリ秒毎に retryTimes 回試みる。
                    // それでも失敗した場合は ExternalException が発生する。
                    Clipboard.SetDataObject(data, copy, retryTimes, retryDelay);
                }
                catch (System.Runtime.InteropServices.ExternalException)
                {
                    // 他アプリがクリップボードにアクセスしているため設定できなかった。
                    // 再試行を確認する。
                    retry = UIMessageBox.YesNo(App.res.Strings.Clipboard_DataStoreFailure);
                }
            }
            while (retry);
        }

        public static void SetText(string text)
        {
            // 引数 format への設定値は Clipboard.SetText() のデフォルト値。
            SetText(text, TextDataFormat.UnicodeText);
        }

        public static void SetText(string text, TextDataFormat format)
        {
            // Clipboard.SetText() の再試行回数と間隔はアンドキュメントなので SetDataObject() を使う。
            switch (format)
            {
                case TextDataFormat.Text:
                    SetDataObject(new DataObject(DataFormats.Text, text));
                    break;
                case TextDataFormat.UnicodeText:
                    SetDataObject(new DataObject(DataFormats.UnicodeText, text));
                    break;
                case TextDataFormat.Rtf:
                    SetDataObject(new DataObject(DataFormats.Rtf, text));
                    break;
                case TextDataFormat.Html:
                    SetDataObject(new DataObject(DataFormats.Html, text));
                    break;
                case TextDataFormat.CommaSeparatedValue:
                    SetDataObject(new DataObject(DataFormats.CommaSeparatedValue, text));
                    break;
                default:
                    {
                        // ここに到達することはない。
                        System.Diagnostics.Debug.Assert(false);

                        // Release 版では Assert を通過するので、念のため処理を入れておく。
                        bool retry = false;
                        do
                        {
                            try
                            {
                                // 他アプリがクリップボードにアクセスしている場合はクリップボードへの設定に失敗し、ExternalException が発生する。
                                // Clipboard.SetText() の再試行回数と間隔はアンドキュメント。
                                Clipboard.SetText(text, format);
                            }
                            catch (System.Runtime.InteropServices.ExternalException)
                            {
                                // 他アプリがクリップボードにアクセスしているため設定できなかった。
                                // 再試行を確認する。
                                retry = UIMessageBox.YesNo(App.res.Strings.Clipboard_DataStoreFailure);
                            }
                        }
                        while (retry);

                        break;
                    }
            }
        }
    }
}
