﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Data;
using App.res;
using ConfigCommon;

namespace App.PropertyEdit
{
    /// <summary>
    /// シェーダー定義プロパティパネルクラス。
    /// </summary>
    public sealed class ShaderDefinitionPropertyPanel : ObjectPropertyPanel
    {
        public enum SearchDir
        {
            Prior,
            Next
        };

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ShaderDefinitionPropertyPanel()
        {
            SelectedTabPageIndex = ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelSubPageTabIndex;
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_shader_definition_property_window;
            }
        }

        private int selectedTabPageIndex;
        public int SelectedTabPageIndex
        {
            get {
                return selectedTabPageIndex;
            }
            set
            {
                selectedTabPageIndex = value;
            }
        }

        private PropertyCategoryNode ctgProgram    = null;
        private PropertyCategoryNode ctgSourceCode = null;

        private void InitializeCategoryView()
        {
            // カテゴリ登録
                ctgProgram = RegisterCategory(1, Strings.ObjectPropertyPanel_ShaderDefinition_ShadingModel, PropertyPageID.ShaderDefinitionShadingModel, ShaderDefinitionShadingModelPage.CreateInstance);
                ctgSourceCode = RegisterCategory(1, Strings.ObjectPropertyPanel_ShaderDefinition_SourceCode, PropertyPageID.ShaderDefinitionSourceCode, ShaderDefinitionSourceCodePage.CreateInstance,
                    updateModifiedFunc: () => ShaderDefinitionSourceCodePage.IsModified(ActiveTarget));
            var ctgRoot = RegisterCategory(0, Strings.ObjectPropertyPanel_ShaderDefinition_Root, PropertyPageID.ShaderDefinitionRoot, ShaderDefinitionRootPage.CreateInstance,
                    updateModifiedFunc: () => ShaderDefinitionRootPage.IsModified(ActiveTarget));
            var ctgFileInfo = RegisterCategory(1, Strings.ObjectPropertyPanel_FileInformation, PropertyPageID.FileInformation, FileInformationPage.CreateInstance,
                    updateModifiedFunc: () => FileInformationPage.IsModified(ActiveTarget));

            CategoryView.Nodes.Add(ctgRoot);
            {
                ctgRoot.Nodes.Add(ctgProgram);
                ctgRoot.Nodes.Add(ctgSourceCode);
                ctgRoot.Nodes.Add(ctgFileInfo);
            }
            CategoryView.ExpandAll();
        }

        protected override void InitializeFormInternal()
        {
            InitializeCategoryView();
            UpdateCategoryView();
            base.InitializeFormInternal();
            SetCategoryMinimumSize();

            Action SelectedChangedHandler = () =>
            {
                if (Targets != null && Targets.Active != null && Targets.Active.ObjectID == GuiObjectID.ShaderDefinition)
                {
                    UpdateCategoryView();
                }
            };
            DocumentAddOrRemovedEventHandler AddedOrRemoved = (o, a, r, s, reload) =>
                {
                    if (Targets != null && Targets.Active != null &&
                    Targets.Active.ObjectID == GuiObjectID.ShaderDefinition &&
                    Owner != null && Owner.Visible)
                    {
                        UpdateCategoryView();
                    }
                };

            App.AppContext.DocumentAddedOrRemoved += AddedOrRemoved;
            Owner.TargetChanged += SelectedChangedHandler;
            // 削除（Disposedで行う）
            Disposed += delegate
            {
                App.AppContext.DocumentAddedOrRemoved -= AddedOrRemoved;
            };
        }

        private readonly Dictionary<KeyValuePair<string, int>, PropertyCategoryNode> programSubPages_ = new Dictionary<KeyValuePair<string, int>, PropertyCategoryNode>();
        private readonly Dictionary<KeyValuePair<string, int>, PropertyCategoryNode> sourceCodeSubPages_ = new Dictionary<KeyValuePair<string, int>, PropertyCategoryNode>();

        private void UpdateCategoryView()
        {
            CategoryView.SuspendLayout();
            {
                // プログラム
                UpdateSubPage(
                    ActiveTarget.Definitions.Select(x => x.Name),
                    ctgProgram,
                    programSubPages_,
                    PropertyPageID.ShaderDefinitionShadingModelSub,
                    ShaderDefinitionShadingModelSubPage.CreateInstance
                );

                // ソースコード
                UpdateSubPage(
                    ActiveTarget.Data.shader_src_array.shader_src.Select(x => x.path),
                    ctgSourceCode,
                    sourceCodeSubPages_,
                    PropertyPageID.ShaderDefinitionSourceCodeSub,
                    ShaderDefinitionSourceCodeSubPage.CreateInstance
                );
            }
            CategoryView.ResumeLayout();

            SetCategoryMinimumSize();
        }

        private void UpdateSubPage(
            IEnumerable<string>							targetNames,
            PropertyCategoryNode						targetPage,
            Dictionary<KeyValuePair<string, int>, PropertyCategoryNode>	subPages,
            PropertyPageID								propertyPageID,
            CreatePropertyPageHandler					createPageHandler
        )
        {
            bool isExpand = targetPage.IsExpanded || targetPage.Nodes.Count == 0;
            string selectedName = null;
            var selectedNode = CategoryView.SelectedNode;

            if (CategoryView.SelectedNode != null && CategoryView.SelectedNode.Parent == targetPage)
            {
                Debug.Assert(CategoryView.SelectedNode is PropertyCategoryNode);

                var node = CategoryView.SelectedNode as PropertyCategoryNode;
                if (node.PropertyPage.PageID == propertyPageID)
                {
                    selectedName = CategoryView.SelectedNode.Text;
                    selectedNode = CategoryView.TopNode;
                }
            }

            List<PropertyCategoryNode> nodes = new List<PropertyCategoryNode>();
            if (Targets != null && Targets.Active != null && Targets.Active.ObjectID == GuiObjectID.ShaderDefinition)
            {
                var targetWithIndies = new List<Tuple<string, int>>();
                {
                    var index = 0;
                    foreach (var name in targetNames)
                    {
                        targetWithIndies.Add(new Tuple<string, int>(name, index));

                        ++ index;
                    }
                }

                foreach(var targetWithIndex in targetWithIndies.OrderBy(x => x.Item1))
                {
                    var name = targetWithIndex.Item1;
                    var index = targetWithIndex.Item2;

                    var key = new KeyValuePair<string, int>(name, index);

                    if (!subPages.ContainsKey(key))
                    {
                        subPages[key] = RegisterCategory(2, name, propertyPageID, createPageHandler, index);
                    }

                    if (propertyPageID == PropertyPageID.ShaderDefinitionSourceCodeSub)
                    {
                        subPages[key].UpdateModified = () => ShaderDefinitionSourceCodeSubPage.IsModified(ActiveTarget, key.Key, key.Value);
                    }
                    else if (propertyPageID == PropertyPageID.ShaderDefinitionShadingModelSub)
                    {
                        subPages[key].UpdateModified = () => ActiveTarget != null && ActiveTarget.ModifiedShadingModels.Contains(key.Key);
                    }


                    nodes.Add(subPages[key]);
                    if (name == selectedName)
                    {
                        selectedNode = subPages[key];
                    }
                }
            }

            if (selectedNode == null)
            {
                CategoryView.SelectedNode = CategoryView.TopNode;
                selectedNode = CategoryView.TopNode;
            }

            targetPage.Nodes.Clear();
            targetPage.Nodes.AddRange(nodes.ToArray());

            if (selectedNode != CategoryView.SelectedNode)
            {
                CategoryView.SelectedNode = selectedNode;
            }

            if (isExpand)
            {
                targetPage.ExpandAll();
            }
        }

        /// <summary>
        /// アクティブターゲット。
        /// </summary>
        private ShaderDefinition ActiveTarget
        {
            get { return (ShaderDefinition)base.Targets.Active; }
        }

        public int SourceCodeNodeCount { get { return sourceCodeSubPages_.Count(); } }

        // get戻り値： -1 でソースコード以外が選択されている
        public int CurrentSourceCodeNodeIndex
        {
            get
            {
                var selectedNode = CategoryView.SelectedNode;

                if (sourceCodeSubPages_.Any(x => x.Value == selectedNode))
                {
                    var node = sourceCodeSubPages_.FirstOrDefault(x => x.Value == selectedNode);
                    return node.Key.Value;
                }
                else
                {
                    // ソースコード以外が選択されている
                    return -1;
                }
            }

            set
            {
                if (sourceCodeSubPages_.Any(x => x.Key.Value == value))
                {
                    var node = sourceCodeSubPages_.FirstOrDefault(x => x.Key.Value == value);
                    CategoryView.SelectedNode = node.Value;
                }
            }
        }

        public int MakeNextSourceCodeNodeIndex(int currentNodeIndex, SearchDir dir)
        {
            var index = 0;

            foreach (var page in sourceCodeSubPages_)
            {
                if (page.Key.Value == currentNodeIndex)
                {
                    break;
                }
                else
                {
                    ++ index;
                }
            }

            index += (dir == ShaderDefinitionPropertyPanel.SearchDir.Prior ? -1 : +1);
            if (index == SourceCodeNodeCount)
            {
                index = 0;
            }

            if (index == -1)
            {
                index = SourceCodeNodeCount - 1;
            }

            return sourceCodeSubPages_.ToArray()[index].Key.Value;
        }

        public PropertyCategoryNode GetPropertyCategoryNodeFromNodeIndex(int index)
        {
            foreach (var page in sourceCodeSubPages_)
            {
                if (page.Key.Value == index)
                {
                    // この時点で作られていない場合があるので
                    CreateCategoryNode(page.Value);

                    return page.Value;
                }
            }

            return null;
        }
    }
}
