﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace App.PropertyEdit
{
    using System.Diagnostics;

    using App.Controls;

    [ToolboxBitmap(typeof(UserControl))]
    [ToolboxItem(true)]
    public partial class KeyMoveScaleUtility : UIUserControl
    {
        protected enum ScaleTarget
        {
            Value,
            Frame,
        }

        protected enum ScaleAxis
        {
            Min,
            Center,
            Max,
            User,
        }

        public KeyMoveScaleUtility()
        {
            InitializeComponent();

            rbaTarget.SetRadioButton(rbtTargetValue, ScaleTarget.Value);
            rbaTarget.SetRadioButton(rbtTargetFrame, ScaleTarget.Frame);
            rbaTarget.SelectionChanged += (sender, args) =>
                {
                    scaleTargetMode = (ScaleTarget)rbaTarget.SelectedRadioButtonData;
                    this.fibScaleAxis.Value = scaleTargetMode == ScaleTarget.Value ? scaleValueAxis : scaleFrameAxis;
                };
            rbaTarget.SelectedRadioButtonData = scaleTargetMode;

            rbaAxis.SetRadioButton(rbtAxisMin, ScaleAxis.Min);
            rbaAxis.SetRadioButton(rbtAxisCenter, ScaleAxis.Center);
            rbaAxis.SetRadioButton(rbtAxisMax, ScaleAxis.Max);
            rbaAxis.SetRadioButton(rbtAxisUser, ScaleAxis.User);
            rbaAxis.SelectionChanged += (sender, args) =>
                {
                    scaleAxisMode = (ScaleAxis)rbaAxis.SelectedRadioButtonData;
                    fibScaleAxis.Enabled = (scaleAxisMode == ScaleAxis.User);
                };
            rbaAxis.SelectedRadioButtonData = scaleAxisMode;

            this.tsmFrameScale_0_25.Click += (sender, args) => fibScale.Value = 0.25f;
            this.tsmFrameScale_0_50.Click += (sender, args) => fibScale.Value = 0.50f;
            this.tsmFrameScale_0_75.Click += (sender, args) => fibScale.Value = 0.75f;
            this.tsmFrameScale_1_00.Click += (sender, args) => fibScale.Value = 1.00f;
            this.tsmFrameScale_1_25.Click += (sender, args) => fibScale.Value = 1.25f;
            this.tsmFrameScale_1_50.Click += (sender, args) => fibScale.Value = 1.50f;
            this.tsmFrameScale_2_00.Click += (sender, args) => fibScale.Value = 2.00f;

            this.Load += (sender, args) =>
                {
                    this.fibMove.Value = moveValue;
                    this.fibScaleAxis.Value = scaleTargetMode == ScaleTarget.Value ?  scaleValueAxis : scaleFrameAxis;
                    this.fibScale.Value = scaleValue;
                    this.Visible = isVisible;
                };
            this.Disposed += (sender, args) => this.SetValues();

            this.fibMove.ValueChanged += (sender, args) => moveValue = this.fibMove.Value;
            this.fibScale.ValueChanged += (sender, args) => scaleValue = this.fibScale.Value;
            this.fibScaleAxis.ValueChanged += (sender, args) =>
                {
                    if (scaleTargetMode == ScaleTarget.Value)
                    {
                        scaleValueAxis = this.fibScaleAxis.Value;
                    }
                    else
                    {
                        scaleFrameAxis = this.fibScaleAxis.Value;
                    }
                };

            //カーブビューのコンテキストメニューを表示させないための空メニュー
            this.ContextMenuStrip = new ContextMenuStrip();
        }

        private void SetValues()
        {
            scaleTargetMode = (ScaleTarget)rbaTarget.SelectedRadioButtonData;
            scaleAxisMode = (ScaleAxis)rbaAxis.SelectedRadioButtonData;
            moveValue = this.fibMove.Value;
            scaleValue = this.fibScale.Value;
            if (scaleTargetMode == ScaleTarget.Value)
            {
                scaleValueAxis = this.fibScaleAxis.Value;
            }
            else
            {
                scaleFrameAxis = this.fibScaleAxis.Value;
            }
            isVisible = this.Visible;
        }

        private static float moveValue = 1f;
        private static float scaleValue = 2f;
        private static float scaleValueAxis = 0f;
        private static float scaleFrameAxis = 0f;
        private static ScaleTarget scaleTargetMode = ScaleTarget.Value;
        private static ScaleAxis scaleAxisMode = ScaleAxis.Min;

        private static bool isVisible = false;

        private CurveEditorPanel CurveEditorPanel
        {
            get
            {
                var curveView = this.Parent as CurveView;
                return curveView == null ? null : curveView.ParentPanel;
            }
        }

        public static bool IsVisible
        {
            get
            {
                return isVisible;
            }
            set
            {
                isVisible = value;
            }
        }

        public void Display(bool show)
        {
            isVisible = show;
            this.Visible = isVisible;
        }

        private void MoveKeys(float frame, float value)
        {
            if (CurveEditorPanel != null)
            {
                CurveEditorPanel.MoveSelectedKeysByUtil(frame, value);
            }
        }

        private void BtnRight_Click(object sender, EventArgs e)
        {
            MoveKeys(this.fibMove.Value, 0.0f);
        }

        private void BtnLeft_Click(object sender, EventArgs e)
        {
            MoveKeys(-this.fibMove.Value, 0.0f);
        }

        private void BtnDown_Click(object sender, EventArgs e)
        {
            MoveKeys(0.0f, -this.fibMove.Value);
        }

        private void BtnUp_Click(object sender, EventArgs e)
        {
            MoveKeys(0.0f, this.fibMove.Value);
        }

        private void BtnScaleTime_Click(object sender, EventArgs e)
        {
            if (CurveEditorPanel == null) { return; }
            var axis = 0.0f;
            var keyFrames = this.CurveEditorPanel.MovableCurves.SelectMany(x => x.KeyFrames);
            if (!keyFrames.Any())
            {
                return;
            }
            var selectedKeys = keyFrames.Where(x => x.Selected);
            if (!selectedKeys.Any())
            {
                return;
            }
            if (scaleTargetMode == ScaleTarget.Value)
            {
                switch (scaleAxisMode)
                {
                    case ScaleAxis.Min:
                        axis = selectedKeys.Select(x => x.Value).Min();
                        break;
                    case ScaleAxis.Max:
                        axis = selectedKeys.Select(x => x.Value).Max();
                        break;
                    case ScaleAxis.Center:
                        axis = 0.5f
                               * (selectedKeys.Select(x => x.Value).Min()
                                  + selectedKeys.Select(x => x.Value).Max());
                        break;
                    case ScaleAxis.User:
                        axis = this.fibScaleAxis.Value;
                        break;
                }
                this.CurveEditorPanel.ScaleSelectedKeysValueByUtil(axis, this.fibScale.Value);
            }
            else
            {
                switch (scaleAxisMode)
                {
                    case ScaleAxis.Min:
                        axis = selectedKeys.Select(x => x.Frame).Min();
                        break;
                    case ScaleAxis.Max:
                        axis = selectedKeys.Select(x => x.Frame).Max();
                        break;
                    case ScaleAxis.Center:
                        axis = 0.5f
                               * (selectedKeys.Select(x => x.Frame).Min()
                                  + selectedKeys.Select(x => x.Frame).Max());
                        break;
                    case ScaleAxis.User:
                        axis = this.fibScaleAxis.Value;
                        break;
                }
                this.CurveEditorPanel.ScaleSelectedKeysFrameByUtil(axis, this.fibScale.Value);
            }
        }

        public void SetScalingControlState(bool enable)
        {
            ugbMove.Enabled = enable;
            ugbScale.Enabled = enable;
        }
    }
}
