﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
//#define USE_SNAP_IN_SCALE // スケール中のスナップの有効無効(有効の時、横方向のマイナススケールにバグあり)

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;

using App.ConfigData;
using App.Data;

using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class CurveViewState
    {
        internal class KeyFrameHolder : DragableInterface
        {
            public KeyFrame			KeyFrame{		get; private set; }
            private KeyFrame			BeginKeyFrame{	get; set; }
            public KeyFrame			EndKeyFrame{	get; private set; }
            public List<KeyFrame> BeginKeyFrames { get; set; }
            public IAnimationCurve	Curve{			get; private set; }

            // Insert 用
            private curve_wrapType PreWrap { get; set; }

            // Insert 用
            private curve_wrapType PostWrap { get; set; }

            private bool				EnableFrameEdit{	get; set;	}
            public bool				EnableValueEdit{ private get; set;	}
            public bool AutoSlopeSpline { private set; get; }

            private readonly CurveViewState	state_;
            private readonly NearSelectType	nearSelect_;
            private readonly EditModeType	editMode_;

            private readonly double	beginOutHandleOffsetX_;
            private readonly double	beginOutHandleOffsetY_;
            private readonly double	beginInHandleOffsetX_;
            private readonly double	beginInHandleOffsetY_;

            private readonly double	beginOutHandleRadian_;
            private readonly double	beginInHandleRadian_;

            private readonly double	beginKeyScreenX_;
            private readonly double	beginKeyScreenY_;

            // ドラッグ中のカーブ編集のインデックス
            public int parentIndex;
            public int curveIndex;
            public int curveComponentIndex;
            public int curveIndexInBinary;
            public bool sendCurve;
            public GuiObject target;

            public enum NearSelectType
            {
                Key,
                InHandle,
                OutHandle,
                //
                NoNearSelect
            }

            public KeyFrameHolder(
                KeyFrame keyFrame,
                CurveViewState state,
                IAnimationCurve curve,
                NearSelectType nearSelect,
                EditModeType editMode)
            {
                // テスト用
                AutoSlopeSpline = false;
                if (editMode != EditModeType.Scale && editMode != EditModeType.Insert)
                {
                    if (nearSelect == NearSelectType.NoNearSelect)
                    {
                        if (keyFrame.Selected)
                        {
                            AutoSlopeSpline = true;
                        }
                    }
                    else if (nearSelect == NearSelectType.Key)
                    {
                        AutoSlopeSpline = true;
                    }
                }

                EnableFrameEdit	= true;
                EnableValueEdit	= true;
                //
                Curve = curve;
                state_ = state;
                nearSelect_ = nearSelect;
                editMode_ = editMode;
                KeyFrame = keyFrame;

                // インデックスの作成
                target = state.view_.TargetGroup.Active;
                AnimTarget animTarget = curve.GetAnimTarget(target);
                if (animTarget != null)
                {
                    PreWrap = animTarget.pre_wrap;
                    PostWrap = animTarget.post_wrap;
                }

                switch (editMode)
                {
                    case EditModeType.Add:
                        {
                            BeginKeyFrames = curve.KeyFrames.ToList();
                            Debug.Assert(Sorted());
                            keyFrame.Index = curve.KeyFrames.Count;
                            curve.KeyFrames.Add(keyFrame);
                            //UpdateSlope_ForAdd();
                            break;
                        }
                    case EditModeType.Insert:
                        {
                            BeginKeyFrames = curve.KeyFrames.ToList();
                            Debug.Assert(Sorted());
                            keyFrame.Index = curve.KeyFrames.Count;
                            curve.KeyFrames.Add(keyFrame);
                            UpdateSlope_ForInsert();
                            break;
                        }
#if USE_SNAP_IN_SCALE
                    case EditModeType.Scale:
                        {
                            // Scaleの場合はスロープの調整のために元の値をすべてクローンしておく
                            BeginKeyFrames = new List<KeyFrame>();
                            foreach(var key in Curve.KeyFrames)
                            {
                                BeginKeyFrames.Add(key.Clone());
                            }
                            break;
                        }
#endif
                }
                BeginKeyFrame = keyFrame.Clone();

                // 追加時はインデックスがずれる可能性があるので後で行う必要がある
                if (editMode != EditModeType.Add || editMode != EditModeType.Insert)
                {
                    sendCurve = CurveEditorPanel.CurveIndex(target, curve, out parentIndex, out curveIndex, out curveComponentIndex, out curveIndexInBinary, out animTarget);
                }

                state.MakeCurveViewScreenPos(
                    BeginKeyFrame.Frame,
                    BeginKeyFrame.Value,
                    out beginKeyScreenX_,
                    out beginKeyScreenY_
                );

                state_.MakeCurveViewScreenHandleOffset(
                    KeyFrame.OutSlope,
                    CurveEditorConst.HandleLineLengthScale,
                    out beginOutHandleOffsetX_,
                    out beginOutHandleOffsetY_
                );

                state_.MakeCurveViewScreenHandleOffset(
                    KeyFrame.InSlope,
                    CurveEditorConst.HandleLineLengthScale,
                    out beginInHandleOffsetX_,
                    out beginInHandleOffsetY_
                );

                beginOutHandleRadian_	= Math.Atan2(beginOutHandleOffsetY_, beginOutHandleOffsetX_);
                beginInHandleRadian_	= Math.Atan2(beginInHandleOffsetY_ , beginInHandleOffsetX_ );
            }

            public void SetCurveIndex()
            {
                AnimTarget animTarget;
                sendCurve = CurveEditorPanel.CurveIndex(target, Curve, out parentIndex, out curveIndex, out curveComponentIndex, out curveIndexInBinary, out animTarget);
            }

            public void Dragging(DragDetailBase detail, int moveDiffX, int moveDiffY, int moveClipDiffX, int moveClipDiffY, InputControlBase inputControl)
            {
                switch(editMode_)
                {
                    case EditModeType.Scale:	Dragging_Scale (detail, moveDiffX, moveDiffY, moveClipDiffX, moveClipDiffY, inputControl);	break;
                    case EditModeType.Add: Dragging_Add(detail, moveDiffX, moveDiffY, moveClipDiffX, moveClipDiffY, inputControl); break;
                    case EditModeType.Insert: Dragging_Insert(detail, moveDiffX, moveDiffY, moveClipDiffX, moveClipDiffY, inputControl); break;
                    default:					Dragging_Normal(detail, moveDiffX, moveDiffY, moveClipDiffX, moveClipDiffY, inputControl);	break;
                }
            }

            private void Dragging_Add(DragDetailBase detail, int moveDiffX, int moveDiffY, int moveClipDiffX, int moveClipDiffY, InputControlBase inputControl)
            {
                // フレーム更新
                if (EnableFrameEdit)
                {
                    if (state_.view_.IsSnapToKey)
                    {
                        KeyFrame.Frame = SnapToKeyFrame(moveClipDiffX, inputControl);
                    }
                    else
                    {
                        KeyFrame.Frame = (float)state_.SnapFrame(BeginKeyFrame.Frame - moveClipDiffX * state_.ScaleX);
                    }

                    KeyFrame.Frame = (float)state_.ClampFrame(KeyFrame.Frame);
                }

                // 値更新
                if (EnableValueEdit)
                {
                    if (state_.view_.IsSnapToKey)
                    {
                        KeyFrame.Value = SnapToKeyValue(moveClipDiffY, inputControl);
                    }
                    else
                    {
                        KeyFrame.Value = (float)state_.SnapValue(BeginKeyFrame.Value + moveClipDiffY * state_.ScaleY);
                    }
                    KeyFrame.Value = Curve.SnapValue(KeyFrame.Value, 0);

                    if (state_.IsClampValue)
                    {
                        KeyFrame.Value = Curve.ClampValue(KeyFrame.Value);
                    }
                }

                // 傾き更新
                //UpdateSlope_ForAdd();
            }

            private void Dragging_Insert(DragDetailBase detail, int moveDiffX, int moveDiffY, int moveClipDiffX, int moveClipDiffY, InputControlBase inputControl)
            {
                // フレーム更新
                if (state_.view_.IsSnapToKey)
                {
                    KeyFrame.Frame = SnapToKeyFrame(moveClipDiffX, inputControl);
                }
                else
                {
                    KeyFrame.Frame = (float)state_.SnapFrame(BeginKeyFrame.Frame - moveClipDiffX * state_.ScaleX);
                }
                KeyFrame.Frame = (float)state_.ClampFrame(KeyFrame.Frame);

                // 値更新
                KeyFrame.Value = AnimationCurveExtensions.GetValue(BeginKeyFrames, Curve.CurveInterpolationType, KeyFrame.Frame, PreWrap, PostWrap);
                KeyFrame.Value = (float)state_.SnapValue(KeyFrame.Value);
                KeyFrame.Value = Curve.SnapValue(KeyFrame.Value, 0);

                if (state_.IsClampValue)
                {
                    KeyFrame.Value = Curve.ClampValue(KeyFrame.Value);
                }

                // 傾き更新
                UpdateSlope_ForInsert();
            }

            private void UpdateSlope_ForInsert()
            {
                float inSlope;
                float outSlope;
                AnimationCurveExtensions.MakeHermiteSmoothSlope(BeginKeyFrames, Curve.CurveInterpolationType, KeyFrame.Frame, PreWrap, PostWrap, out inSlope, out outSlope);
                KeyFrame.InSlope = inSlope;
                KeyFrame.OutSlope = outSlope;
            }

            // デバッグ用
            private bool Sorted()
            {
                for (int i = 0; i < BeginKeyFrames.Count - 1; i++)
                {
                    if (BeginKeyFrames[i].Frame > BeginKeyFrames[i + 1].Frame)
                    {
                        return false;
                    }
                }

                return true;
            }

            /// <summary>
            /// 追加する点の傾きを更新
            /// </summary>
            private void UpdateSlope_ForAdd()
            {
                int i;
                for (i = 0; i < BeginKeyFrames.Count; i++)
                {
                    if (KeyFrame.Frame < BeginKeyFrames[i].Frame)
                    {
                        break;
                    }
                }

                KeyFrame previous;
                KeyFrame next;
                if (i == 0)
                {
                    previous = KeyFrame;
                }
                else
                {
                    previous = BeginKeyFrames[i - 1];
                }

                if (i == BeginKeyFrames.Count)
                {
                    next = KeyFrame;
                }
                else
                {
                    next = BeginKeyFrames[i];
                }

                if (next.Frame - previous.Frame < 0.0001)
                {
                    KeyFrame.InSlope = KeyFrame.OutSlope = 0;
                }
                else
                {
                    KeyFrame.InSlope = KeyFrame.OutSlope = (next.Value - previous.Value) / (next.Frame - previous.Frame);
                }
            }

            private void Dragging_Normal(DragDetailBase detail, int moveDiffX, int moveDiffY, int moveClipDiffX, int moveClipDiffY, InputControlBase inputControl)
            {
                // 近いキー選択時は選択状態を無視する
                bool selected			= KeyFrame.Selected;
                bool inHandleSelected	= KeyFrame.InHandleSelected;
                bool outHandleSelected	= KeyFrame.OutHandleSelected;
                {
                    if (nearSelect_ != NearSelectType.NoNearSelect)
                    {
                        KeyFrame.Selected			= false;
                        KeyFrame.InHandleSelected	= false;
                        KeyFrame.OutHandleSelected	= false;
                    }

                    if (KeyFrame.Selected || (nearSelect_ == NearSelectType.Key))
                    {
                        if (EnableFrameEdit)
                        {
                            if (state_.view_.IsSnapToKey)
                            {
                                KeyFrame.Frame = SnapToKeyFrame(moveClipDiffX, inputControl);
                            }
                            else
                            {
                                KeyFrame.Frame = (float)state_.SnapFrame(BeginKeyFrame.Frame - moveClipDiffX * state_.ScaleX);
                            }
                            KeyFrame.Frame = (float)state_.ClampFrame(KeyFrame.Frame);
                        }

                        if (EnableValueEdit)
                        {
                            if (state_.view_.IsSnapToKey)
                            {
                                KeyFrame.Value = SnapToKeyValue(moveClipDiffY, inputControl);
                            }
                            else
                            {
                                KeyFrame.Value = (float)state_.SnapValue(BeginKeyFrame.Value + moveClipDiffY * state_.ScaleY);
                            }
                            KeyFrame.Value = Curve.SnapValue(KeyFrame.Value, 0);
                            if (state_.IsClampValue)
                            {
                                KeyFrame.Value = Curve.ClampValue(KeyFrame.Value);
                            }
                        }
                    }
                    else
                    {
                        if (KeyFrame.InHandleSelected && KeyFrame.OutHandleSelected && (nearSelect_ == NearSelectType.NoNearSelect))
                        {
                            // なにもしない
                            ;
                        }
                        else
                        if (KeyFrame.OutHandleSelected || (nearSelect_ == NearSelectType.OutHandle))
                        {
                            double nextX	= beginOutHandleOffsetX_ - moveClipDiffX;
                            double nextY	= beginOutHandleOffsetY_ - moveClipDiffY;

                            KeyFrame.OutSlope			= (float)state_.MakeSlope(nextX, nextY);

                            if (KeyFrame.IsUnionSlope)
                            {
                                double diffRadian = beginOutHandleRadian_ - Math.Atan2(nextY, nextX);
                                double moveRadian = beginInHandleRadian_ - diffRadian;

                                double inNextX =  Math.Cos(moveRadian);
                                double inNextY = -Math.Sin(moveRadian) * -1;		// -1は天地逆

                                KeyFrame.InSlope = (float)state_.MakeSlope(inNextX, inNextY);
                            }
                        }
                        else
                        if (KeyFrame.InHandleSelected || (nearSelect_ == NearSelectType.InHandle))
                        {
                            double nextX	= beginInHandleOffsetX_ + moveClipDiffX;
                            double nextY	= beginInHandleOffsetY_ + moveClipDiffY;

                            KeyFrame.InSlope			= (float)state_.MakeSlope(nextX, nextY);

                            if (KeyFrame.IsUnionSlope)
                            {
                                double diffRadian = beginInHandleRadian_ - Math.Atan2(nextY, nextX);
                                double moveRadian = beginOutHandleRadian_ - diffRadian;

                                double outNextX =  Math.Cos(moveRadian);
                                double outNextY = -Math.Sin(moveRadian) * -1;		// -1は天地逆

                                KeyFrame.OutSlope = (float)state_.MakeSlope(outNextX, outNextY);
                            }
                        }
                    }
                }

                // もとに戻す
                KeyFrame.Selected			= selected;
                KeyFrame.InHandleSelected	= inHandleSelected;
                KeyFrame.OutHandleSelected	= outHandleSelected;
            }

            private float SnapToKeyValue(int moveClipDiffY, InputControlBase inputControl)
            {
                var value = (float)(BeginKeyFrame.Value + moveClipDiffY * state_.ScaleY);
                if (!state_.view_.IsSnapToKey || inputControl.DragMove.IsClipY)
                {
                    return value;
                }

                // 全てのカーブのキーからスナップ対象を探す(ドラッグ中のキーは除く)
                var values =
                    new HashSet<float>(
                        state_.view_.VisibledCurves.SelectMany(x => x.KeyFrames)
                            .Except(inputControl.DraggingKeys)
                            .Select(x => x.Value - value));
                if (!values.Any())
                {
                    return value;
                }

                var mindif = values.First();

                foreach (var v in values)
                {
                    if (Math.Abs(v) < Math.Abs(mindif))
                    {
                        mindif = v;
                    }
                }

                if ((Math.Abs(mindif) / state_.ScaleY) < ApplicationConfig.Setting.CurveEditor.KeySnapFactor)
                {
                    var keysnapValue = value + mindif;
                    state_.view_.SnappedValues.Add(keysnapValue);
                    return keysnapValue;
                }
                return value;
            }

            private float SnapToKeyFrame(int moveClipDiffX, InputControlBase inputControl)
            {
                var frame = (float)(BeginKeyFrame.Frame - moveClipDiffX * state_.ScaleX);
                if (!state_.view_.IsSnapToKey || inputControl.DragMove.IsClipX)
                {
                    return frame;
                }

                //ドラッグ中のカーブ以外のカーブのキーからスナップ対象を探す
                var frames =
                    new HashSet<float>(
                        state_.view_.VisibledCurves.Where(x => x != Curve)
                            .SelectMany(x => x.KeyFrames)
                            .Except(inputControl.DraggingKeys)
                            .Select(x => x.Frame));
                // 現在のカーブのキーフレームと同じ値は除外(ドラッグ中のキーは除く)
                frames.ExceptWith(
                    inputControl.DraggingCurvesKeys.Except(inputControl.DraggingKeys).Select(x => x.Frame));

                if (!frames.Any())
                {
                    return frame;
                }
                var mindif = frames.First();

                foreach (var f in frames)
                {
                    if (Math.Abs(f - frame) < Math.Abs(mindif))
                    {
                        mindif = f - frame;
                    }
                }
                if ((Math.Abs(mindif) / state_.ScaleX) < ApplicationConfig.Setting.CurveEditor.KeySnapFactor)
                {
                    var keysnapFrame = frame + mindif;
                    state_.view_.SnappedFrames.Add(keysnapFrame);
                    return keysnapFrame;
                }
                return frame;
            }

            private void Dragging_Scale(DragDetailBase detail, int moveDiffX, int moveDiffY, int moveClipDiffX, int moveClipDiffY, InputControlBase inputControl)
            {
                // なるべくMAYAっぽい動作に

                var vectX = (beginKeyScreenX_ - inputControl.DownArgs.X) * +CurveEditorConst.ScaleEditSensitivity;
                var vectY = (beginKeyScreenY_ - inputControl.DownArgs.Y) * -CurveEditorConst.ScaleEditSensitivity;

                if (EnableFrameEdit && (inputControl.MoveDir == MoveDirType.Horizontality))
                {
#if USE_SNAP_IN_SCALE

                    KeyFrame.Frame = (float)state_.SnapFrame(BeginKeyFrame.Frame - moveClipDiffX * state_.ScaleX * vectX);
                    KeyFrame.Frame = (float)state_.ClampFrame(KeyFrame.Frame);
                    // スナップ対応版（バグあり）
                    if (BeginKeyFrames != null && Curve.KeyFrames != null)
                    {
                        Debug.Assert(BeginKeyFrames.Count == Curve.KeyFrames.Count);
                        bool doSwapSlope = false;

                        if (KeyFrame.Index > 0) // inSlope
                        {
                            var newDif = KeyFrame.Frame - Curve.KeyFrames[KeyFrame.Index - 1].Frame;
                            var oldDif = BeginKeyFrames[KeyFrame.Index].Frame - BeginKeyFrames[KeyFrame.Index - 1].Frame;
                            doSwapSlope = (newDif < 0.0f);
                            if (Math.Abs(newDif) > 0.0001f)
                            {
                                KeyFrame.InSlope = BeginKeyFrame.InSlope * oldDif / newDif;
                            }

                            // Debug.Print 使うとボタンが反応しなくなることがある。
                            Debug.Print("{0}:{1}", newDif, oldDif);
                        }

                        if (KeyFrame.Index < BeginKeyFrames.Count - 1) // outSlope
                        {
                            var newDif = Curve.KeyFrames[KeyFrame.Index + 1].Frame - KeyFrame.Frame;
                            var oldDif = BeginKeyFrames[KeyFrame.Index + 1].Frame - BeginKeyFrames[KeyFrame.Index].Frame;
                            if (Math.Abs(newDif) > 0.0001f)
                            {
                                KeyFrame.OutSlope = BeginKeyFrame.OutSlope * oldDif / newDif;
                            }
                        }

                        if (doSwapSlope)
                        {
                            var tmpSlope = KeyFrame.InSlope;
                            KeyFrame.InSlope = KeyFrame.OutSlope;
                            KeyFrame.OutSlope = tmpSlope;
                        }

                    }
#else
                    // スナップ未対応
                    KeyFrame.Frame = (float)(BeginKeyFrame.Frame - moveClipDiffX * state_.ScaleX * vectX);
                    var scale = (float)(moveClipDiffX * -CurveEditorConst.ScaleEditSensitivity) + 1.0f;
                    if (Math.Abs(scale) > 0.00001f)
                    {
                        scale = 1.0f / scale;
                        if (scale > 0.0f)
                        {
                            KeyFrame.InSlope = BeginKeyFrame.InSlope * scale;
                            KeyFrame.OutSlope = BeginKeyFrame.OutSlope * scale;
                        }
                        else
                        {
                            KeyFrame.OutSlope = BeginKeyFrame.InSlope * scale;
                            KeyFrame.InSlope = BeginKeyFrame.OutSlope * scale;
                        }
                    }
#endif
                }

                if (EnableValueEdit && (inputControl.MoveDir == MoveDirType.Verticality))
                {
#if USE_SNAP_IN_SCALE
                    KeyFrame.Value = (float)state_.SnapValue(BeginKeyFrame.Value + moveClipDiffY * state_.ScaleY * vectY);
                    KeyFrame.Value = Curve.SnapValue(KeyFrame.Value, 0);

                    if (state_.IsClampValue)
                    {
                        KeyFrame.Value = (float)Curve.ClampValue(KeyFrame.Value);
                    }

                    if (BeginKeyFrames != null && Curve.KeyFrames != null)
                    {
                        Debug.Assert(BeginKeyFrames.Count == Curve.KeyFrames.Count);
                        if (KeyFrame.Index > 0) // inSlope
                        {
                            var newDif = Curve.KeyFrames[KeyFrame.Index].Value - Curve.KeyFrames[KeyFrame.Index - 1].Value;
                            var oldDif = BeginKeyFrames[KeyFrame.Index].Value - BeginKeyFrames[KeyFrame.Index - 1].Value;
                            if (Math.Abs(oldDif) > 0.0001f)
                            {
                                KeyFrame.InSlope = BeginKeyFrame.InSlope * newDif / oldDif;
                            }

                            // Debug.Print 使うとボタンが反応しなくなることがある。
                            Debug.Print("{0}:{1}",newDif, oldDif);
                        }

                        if (KeyFrame.Index < BeginKeyFrames.Count - 1) // outSlope
                        {
                            var newDif = Curve.KeyFrames[KeyFrame.Index+1].Value - Curve.KeyFrames[KeyFrame.Index].Value;
                            var oldDif = BeginKeyFrames[KeyFrame.Index+1].Value - BeginKeyFrames[KeyFrame.Index].Value;
                            if (Math.Abs(oldDif) > 0.0001f)
                            {
                                KeyFrame.OutSlope = BeginKeyFrame.OutSlope * newDif / oldDif;
                            }
                        }
                    }
#else
                    //Debug.WriteLine("{0}: KeyFrame = {1}/{2}", KeyFrame.Frame, KeyFrame.Index, BeginKeyFrames == null ? -1 : BeginKeyFrames.Count);
                    KeyFrame.Value = Curve.SnapValue((float)(BeginKeyFrame.Value + moveClipDiffY * state_.ScaleY * vectY), 0);
                    var scale = (float)(moveClipDiffY * CurveEditorConst.ScaleEditSensitivity + 1.0f);
                    KeyFrame.InSlope = BeginKeyFrame.InSlope * scale;
                    KeyFrame.OutSlope = BeginKeyFrame.OutSlope * scale;
#endif
                }
            }

            public void DragEnd(DragDetailBase detail, InputControlBase inputControl)
            {
                /*
                // 結果を保存する
                EndKeyFrame = KeyFrame.Clone();

                // 元に戻す
                KeyFrame.Set(BeginKeyFrame);

                switch (editMode_)
                {
                    case EditModeType.Add:
                    case EditModeType.Insert:
                        Curve.KeyFrames.Clear();
                        Curve.KeyFrames.AddRange(BeginKeyFrames);
                        EndKeyFrame.Index = Curve.KeyFrames.Count;
                        break;
                }*/
            }

            public void DragCancel(DragDetailBase detail, InputControlBase inputControl)
            {
                /*
                // 元に戻す
                KeyFrame.Set(BeginKeyFrame);
                switch (editMode_)
                {
                    case EditModeType.Add:
                    case EditModeType.Insert:
                        Curve.KeyFrames.Clear();
                        Curve.KeyFrames.AddRange(BeginKeyFrames);
                        break;
                }*/
            }
        }
    }
}
