﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.Properties;
using App.Utility;
using ConfigCommon;

namespace App.PropertyEdit
{
    public partial class CurveViewPainter
    {
        private static readonly Bitmap visibleKeyOnImage_	= Resources.AnimationEditor_VisibleKey_On;
        private static readonly Bitmap visibleKeyOffImage_	= Resources.AnimationEditor_VisibleKey_Off;
        private static readonly Dictionary<bool, Bitmap> visibleKeysImage_ = new Dictionary<bool, Bitmap>
        {
            {true,  visibleKeyOnImage_},
            {false, visibleKeyOffImage_},
        };

        private static readonly int	visibleKeyImageWidth_	= visibleKeyOnImage_.Width;
        private static readonly int	visibleKeyImageHeight_	= visibleKeyOnImage_.Height;

        public static int VisibleKeyImageWidth{		get{ return visibleKeyImageWidth_;  } }
        public static int VisibleKeyImageHeight{	get{ return visibleKeyImageHeight_; } }

        // 矢印先から左上までのオフセット
        private static int VisibleKeyLeftOffset{	get{ return -VisibleKeyImageWidth / 2; } }
        private static int VisibleKeyTopOffset{		get{ return -VisibleKeyImageHeight;    } }

        // ビジブルカーブ編集用
        private bool DrawHRulerVisibleBar(PaintEventArgs e, GuiObject target)
        {
            if ((target.ObjectID != GuiObjectID.BoneVisibilityAnimation) &&
                (target.ObjectID != GuiObjectID.MaterialVisibilityAnimation) &&
                (target.ObjectID != GuiObjectID.MaterialAnimation) &&
                (view_.SelectedVisibleCurves.Any(x => !x.IsEditable)))
            {
                return false;
            }

            var curves = view_.SelectedVisibleCurves;

            if (curves.Any(x => x.KeyFrames.Any()))
            {
                DrawVisibleBar(e, curves, target);

                using (var hqg = new HighQualityGraphics(e.Graphics, isInterpolationMode:false, isSmoothingMode:false))
                {
                    DrawVisibleKeyCursors(e, curves);
                }

                return true;
            }
            if (curves.Any())
            {
                DrawEmptyBar(e);
                return true;
            }

            return false;
        }

        private void DrawVisibleBar(PaintEventArgs e, List<IAnimationCurve> curves, GuiObject target)
        {
            var rect = new Rectangle(CurveBgLeft, HRulerBgTop - CurveEditorConst.ColorBarSize, CurveBgWidth, CurveEditorConst.ColorBarSize);

            if (curves.Count >= 2)
            {
                // 複数あれば単色のグレー
                e.Graphics.FillRectangle(Brushes.Gray, rect);
            }
            else
            {
                var curve = curves.First();
                var animTarget = curve.GetAnimTarget(target);
                if (curve.KeyFrames.Count >= 2 && curve.KeyFrames[curve.KeyFrames.Count - 1].Frame - curve.KeyFrames[0].Frame < State.ScaleX &&
                    (animTarget.pre_wrap != nw.g3d.nw4f_3dif.curve_wrapType.clamp || animTarget.post_wrap != nw.g3d.nw4f_3dif.curve_wrapType.clamp))
                {
                    // 幅が狭いときはグレー
                    e.Graphics.FillRectangle(Brushes.Gray, rect);
                }
                else
                {
                    // 下地
                    e.Graphics.FillRectangle(Brushes.Black, rect);

                    foreach (var i in MakeEditBarIntervals(curves.First(), target).Where(x => x.Value == 1.0f))
                    {
                        e.Graphics.SafeFillRectangle(
                            Brushes.White,
                            i.LeftX, HRulerBgTop - CurveEditorConst.ColorBarSize, i.Width, CurveEditorConst.ColorBarSize);
                    }
                }
            }
        }

        private void DrawVisibleKeyCursors(PaintEventArgs e, List<IAnimationCurve> curves)
        {
            if (curves.Count == 1)
            {
                var curve = curves.First();

                foreach(var keyFrame in curve.KeyFrames)
                {
                    var frameScreenX = (float)MakeCurveViewScreenPosX(keyFrame.Frame);
                    DrawVisibleKeyCursor(e.Graphics, frameScreenX, HRulerBgTop, keyFrame.Value == 1.0f);
                }
            }
        }

        // x, y は矢印の先の座標を指定します。
        private void DrawVisibleKeyCursor(Graphics g, float x, float y, bool isVisible)
        {
            x += VisibleKeyLeftOffset;
            y += VisibleKeyTopOffset - CurveEditorConst.ColorBarSize;

            g.DrawImageSafe(visibleKeysImage_[isVisible], x, y);
        }
    }
}
