﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.Properties;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    using System.Drawing.Drawing2D;

    using frameFlag = KeyValuePair<float, bool>;
    public partial class CurveViewPainter
    {
        public static Bitmap[] _BmpHandles = {
            Resources.AnimationEditor_Handle_Normal,
            Resources.AnimationEditor_Handle_NormalMouseOver,
            Resources.AnimationEditor_Handle_Selected,
            Resources.AnimationEditor_Handle_SelectedMouseOver
        };

        public static Bitmap[] _BmpDisabledHandles = {
            Resources.AnimationEditor_Handle_Gray,
            Resources.AnimationEditor_Handle_Gray,
            Resources.AnimationEditor_Handle_Gray,
            Resources.AnimationEditor_Handle_Gray
        };

        public Bitmap[] BmpHandles = {};
        public Bitmap[] BmpDisabledHandles = {};


        public void InitBitmaps()
        {
            // 高速化のためにPARGB形式に変換しておく
            var bmpList = new List<Bitmap>();
            foreach (var bmp in _BmpHandles)
            {
                var dst = new Bitmap(bmp.Width, bmp.Height, PixelFormat.Format32bppPArgb);
                using (var g = Graphics.FromImage(dst)) { g.DrawImage(bmp, 0, 0); }
                bmpList.Add(dst);
            }
            BmpHandles = bmpList.ToArray();

            bmpList.Clear();
            foreach (var bmp in _BmpDisabledHandles)
            {
                var dst = new Bitmap(bmp.Width, bmp.Height, PixelFormat.Format32bppPArgb);
                using (var g = Graphics.FromImage(dst)) { g.DrawImage(bmp, 0, 0); }
                bmpList.Add(dst);
            }
            BmpDisabledHandles= bmpList.ToArray();
        }

        private Bitmap GetHandleImage(IAnimationCurve curve, HandlePosition key)
        {
            Bitmap[] table = curve.IsEditable ? BmpHandles : BmpDisabledHandles;
            int index = key.Selected ?
                key.AreaSelected ? 3 : 2 :
                key.AreaSelected ? 1 : 0;
            return table[index];
        }

        /// <summary>
        /// キーを描画するためのブラシとペンを取得します。
        /// </summary>
        /// <param name="curve">カーブ</param>
        /// <param name="key">キー</param>
        /// <param name="curvePen">カーブと同じ色のペン</param>
        /// <param name="keyBrush">キーを描画するためのブラシ</param>
        /// <param name="keyPen">キーを描画するためのペン</param>
        private void GetKeyBrushAndPen(IAnimationCurve curve, KeyFrame key, Pen curvePen, out Brush keyBrush, out Pen keyPen)
        {
            if (curve.IsEditable)
            {
                if (key.Selected)
                {
                    keyPen = CurveEditorConst.CurveSelectedEnabledKeyPen;
                    keyBrush = CurveEditorConst.CurveSelectedEnabledKeyBrush;
                }
                else
                {
                    keyPen = curvePen;
                    keyBrush = CurveEditorConst.CurveEnabledKeyBrush;
                }
            }
            else
            {
                if (key.Selected)
                {
                    keyPen = CurveEditorConst.CurveSelectedDisabledKeyPen;
                    keyBrush = CurveEditorConst.CurveSelectedDisabledKeyBrush;
                }
                else
                {
                    keyPen = CurveEditorConst.CurveDisabledKeyPen;
                    keyBrush = CurveEditorConst.CurveDisabledKeyBrush;
                }
            }
        }

        private const double KeyFrameImageRadius	= 8.0;
        private const double HandleImageRadius		= 8.0;
        private const float PixelMax = 1 << 16;// 描画範囲 1 << 16 なら OK のソースは見つからず、DrawLine で 1 << 16 + 1 << 15 を使うと変になることは確認

        public class OldDrawCurvePosition
        {
            public IAnimationCurve	Curve{ get; set; }
            public List<PointF> 	Positions{ get; set; }
        }

        public bool		IsVisibleFrameValue{	get { return view_.IsVisibleFrameValue;		} }
        public bool		IsVisibleSlope{			get { return view_.IsVisibleSlope;			} }
        public bool		IsVisibleMinMax{		get { return view_.IsVisibleMinMax;			} }
        public bool		IsVisibleCurrentValue{	get { return view_.IsVisibleCurrentValue;	} }
        public bool IsVisibleCurveName { get { return view_.IsVisibleCurveName; } }

        private bool	IsVisiblePrePostWrapCurve{	get { return true;	} }						// TODO:今のところ固定でプライベート

        public List<OldDrawCurvePosition> OldDrawCurvePositions { get; private set; }

        private static readonly Font FullPathFont	= new Font("Tahoma", 8.25F * 1.15f, FontStyle.Regular, GraphicsUnit.Point, 0);

        private static readonly float[] DashPattern = { 4.0f, 2.0f };

        private void DrawCurves(PaintEventArgs e, GuiObject target)
        {
            Debug.Assert(target != null);

            OldDrawCurvePositions = new List<OldDrawCurvePosition>();

            var drawInfo = new AnimationCurveDrawInfo(this, e)
            {
                IsVisibleFrameValue		= IsVisibleFrameValue,
                IsVisibleSlope			= IsVisibleSlope,
                IsVisibleMinMax			= IsVisibleMinMax,
                IsVisibleCurrentValue	= IsVisibleCurrentValue,
                IsVisibleCurveName = IsVisibleCurveName,
            };

            double dummy;
            double frameEndX;
            double frameBeginX;

            MakeCurveViewScreenPos(0.0f,              0.0f, out frameBeginX, out dummy);
            MakeCurveViewScreenPos(FrameCount, 0.0f, out frameEndX,   out dummy);

            // カーブ一覧表示
            {
                var xPos = CurveBgLeft + 5;
                var yPos = CurveBgTop  + 1;

                foreach(var x in SelectedCurves.Where(x => x.KeyFrames.Any()))
                {
                    // 箱
                    if (drawInfo.IsVisibleCurveName || drawInfo.IsVisibleCurrentValue)
                    {
                        var rect = new Rectangle(xPos, yPos + 3, 11, 11);
                        using (var brush = new SolidBrush(x.CurveColor))
                        {
                            e.Graphics.FillRectangle(brush, rect);
                        }

                        e.Graphics.DrawRectangle(Pens.Black, rect);
                    }

                    // 文字色黒、アンダーラインにカーブ色
                    {
                        var line = drawInfo.IsVisibleCurveName? x.FullPath: "";

                        if (drawInfo.IsVisibleCurrentValue)
                        {
                            var animTarget = x.GetAnimTarget(view_.TargetGroup.Active);
                            if (line.Length > 0)
                            {
                                line += " : ";
                            }
                            line += x.GetValue(
                                State.Pausing ?
                                    (float)State.PauseFrame :
                                    (float)view_.State.CurrentFrame,
                                animTarget.pre_wrap,
                                animTarget.post_wrap
                            ).ToString();
                        }

                        using (var pen = new Pen(x.CurveColor))
                        {
                            var lineWidth = e.Graphics.MeasureString(line, FullPathFont, 1000);
                            e.Graphics.DrawLine(pen, xPos + 15, yPos + 15, xPos + 15 + lineWidth.Width, yPos + 15);
                        }

                        e.Graphics.DrawString(
                            line,
                            FullPathFont,
                            Brushes.LightGray,
                            xPos + 15,
                            yPos
                        );
                    }

                    yPos += 19;
                }
            }

            // カーブそのもの
            VisibledCurves.ForEach(
                curve =>
                {
                    var oldDrawCurvePosition = new OldDrawCurvePosition
                    {
                        Curve = curve,
                    };

                    drawInfo.IsSelected = SelectedCurves.IndexOf(curve) != -1;

                    if (curve.KeyFrames.Count == 1)
                    {
                        // コンスタントカーブを描画
                        using (var pen = new Pen(curve.IsEditable ? curve.CurveColor : Color.Gray))
                        {
                            pen.DashStyle = DashStyle.Custom;
                            pen.DashPattern = DashPattern;
                            DrawLineKeyCoord(drawInfo.EventArgs.Graphics, pen, 0, curve.KeyFrames[0].Value, FrameCount, curve.KeyFrames[0].Value);
                        }
                    }

                    // 点1、コントロールポイント1、コントロールポイント2、点2の4要素が並ぶ配列
                    oldDrawCurvePosition.Positions = DrawCurve(curve, drawInfo);

                    OldDrawCurvePositions.Add(oldDrawCurvePosition);

                    // 前後ラップカーブを描画する
                    if (IsVisiblePrePostWrapCurve)
                    {
                        var animTarget = curve.GetAnimTarget(target);
                        Debug.Assert(animTarget != null);

                        if (oldDrawCurvePosition.Positions.Any())
                        {
                            using (var pen = new Pen(curve.IsEditable ? curve.CurveColor : Color.Gray))
                            {
                                pen.DashStyle = DashStyle.Custom;
                                pen.DashPattern = DashPattern;
                                using (var hqg = new HighQualityGraphics(drawInfo.EventArgs.Graphics, isInterpolationMode: false, isSmoothingMode: false))
                                {
                                    DrawPostWrapCurve(pen, oldDrawCurvePosition.Positions, animTarget.post_wrap, drawInfo, (float)frameBeginX, (float)frameEndX);
                                    DrawPreWrapCurve(pen, oldDrawCurvePosition.Positions, animTarget.pre_wrap, drawInfo, (float)frameBeginX, (float)frameEndX);
                                }
                            }
                        }
                    }
                }
            );
        }

        private class HandlePosition
        {
            public PointF		Center{			get; set; }
            public PointF		Handle{			get; set; }
            public bool			Selected{		get; set; }
            public bool			AreaSelected{	get; set; }
            public float		Slope{			get; set; }
            public HandleType	Type{			get; set; }
        }

        private enum HandleType
        {
            Unite,
            In,
            Out,
        }

        private delegate void MakeCurveViewScreenPosFast(float srcX, float srcY, out double dstX, out double dstY);

        private List<PointF> DrawCurve(IAnimationCurve curve, AnimationCurveDrawInfo info)
        {
            List<KeyFrame>	keyFrames	= curve.KeyFrames;
            var curveInterpolationType = curve.CurveInterpolationType;

            var positions			= new List<PointF>();
            var selectedPositions	= new List<List<PointF>>();
            var unselectedPositions	= new List<List<PointF>>();
            var handlePositions		= new List<HandlePosition>();

            // MakeCurveViewScreenPos の展開を行う
            //(CurveBgWidth  * 0.5) - (State.ScrollX * State.ScaleX) + (srcX / State.ScaleX);
            //(CurveBgHeight * 0.5) - (State.ScrollY * State.ScaleY) - (srcY / State.ScaleY);
            var curveBgWidthDiv2 = CurveBgWidth * 0.5;
            var stateScrollXMulScaleX = State.ScrollX * State.ScaleX;
            var curveBgWidthDiv2SubScrollXMulScaleX = curveBgWidthDiv2 - stateScrollXMulScaleX;
            var stateScaleXInv = 1.0 / State.ScaleX;
            var curveBgHeightDiv2 = CurveBgHeight * 0.5;
            var stateScrollYMulScaleY = State.ScrollY * State.ScaleY;
            var curveBgHeightDiv2ScrollYMulScaleY = curveBgHeightDiv2 - stateScrollYMulScaleY;
            var stateScaleYInv = 1.0 / State.ScaleY;

            {

                if (keyFrames.Any())
                {
//					var curveLeft  = (float)info.Painter.GetXInCurve(CurveBgLeft);
//					var curveRight = (float)info.Painter.GetXInCurve(CurveBgRight);

                    double xl, yl;
                    double xr, yr;

                    for (var i = 0; i != keyFrames.Count - 1; ++i)
                    {
                        var left			= keyFrames[i + 0];
                        var right			= keyFrames[i + 1];
                        var isKeySelected	= left.Selected || right.Selected;

                        var switchPositions = new List<PointF>();
                        if (isKeySelected)
                        {
                            selectedPositions.Add(switchPositions);
                        }
                        else
                        {
                            unselectedPositions.Add(switchPositions);
                        }

                        // カーブ
                        {
                            //MakeCurveViewScreenPos(left.Frame,  left.Value,  out xl, out yl);
                            //MakeCurveViewScreenPos(right.Frame, right.Value, out xr, out yr);
                            xl = curveBgWidthDiv2SubScrollXMulScaleX + stateScaleXInv * left.Frame;
                            yl = curveBgHeightDiv2ScrollYMulScaleY   - stateScaleYInv * left.Value;
                            xr = curveBgWidthDiv2SubScrollXMulScaleX + stateScaleXInv * right.Frame;
                            yr = curveBgHeightDiv2ScrollYMulScaleY - stateScaleYInv * right.Value;

                            if (IsVisiblePrePostWrapCurve || IsInDrawArea(xl, yl, xr, yr))
                            {
                                switch (curveInterpolationType)
                                {
                                    case InterpolationType.Hermite:
                                    {
                                        // drawbezier
                                        {
                                            var dFrame = right.Frame - left.Frame;
                                            var dFrameDiv3 = dFrame / 3.0f;

                                            float cx0 = left.Frame + dFrameDiv3;
                                            float cy0 = left.Value + dFrameDiv3 * left.OutSlope;
                                            float cx1 = right.Frame - dFrameDiv3;
                                            float cy1 = right.Value - dFrameDiv3 * right.InSlope;


                                            //MakeCurveViewScreenPos(left.Frame, left.Value, out x0, out y0);
                                            //MakeCurveViewScreenPos(cx0, cy0, out x1, out y1);
                                            //MakeCurveViewScreenPos(cx1, cy1, out x2, out y2);
                                            //MakeCurveViewScreenPos(right.Frame, right.Value, out x3, out y3);
                                            double x1 = curveBgWidthDiv2SubScrollXMulScaleX + stateScaleXInv * cx0;
                                            double y1 = curveBgHeightDiv2ScrollYMulScaleY - stateScaleYInv * cy0;
                                            double x2 = curveBgWidthDiv2SubScrollXMulScaleX + stateScaleXInv * cx1;
                                            double y2 = curveBgHeightDiv2ScrollYMulScaleY - stateScaleYInv * cy1;


                                            var p0 = new PointF((float)xl, (float)yl);
                                            var p1 = new PointF((float)x1, (float)y1);
                                            var p2 = new PointF((float)x2, (float)y2);
                                            var p3 = new PointF((float)xr, (float)yr);

                                            positions.Add(p0);
                                            positions.Add(p1);
                                            positions.Add(p2);
                                            positions.Add(p3);
                                            switchPositions.Add(p0);
                                            switchPositions.Add(p1);
                                            switchPositions.Add(p2);
                                            switchPositions.Add(p3);
                                        }

                                        break;
                                    }

                                    case InterpolationType.Linear:
                                    {
                                        var p0 = new PointF((float)xl, (float)yl);
                                        var p1 = new PointF((float)xr, (float)yr);
                                        var c0 = new PointF((p0.X + p1.X) * 0.5f, (p0.Y + p1.Y) * 0.5f);

                                        positions.Add(p0);
                                        positions.Add(c0);
                                        positions.Add(c0);
                                        positions.Add(p1);

                                        switchPositions.Add(p0);
                                        switchPositions.Add(c0);
                                        switchPositions.Add(c0);
                                        switchPositions.Add(p1);

                                        break;
                                    }

                                    case InterpolationType.Step:
                                    {
                                        var p0 = new PointF((float)xl, (float)yl);
                                        var p1 = new PointF((float)xr, (float)yl);
                                        var p2 = new PointF((float)xr, (float)yr);
                                        var c0 = new PointF((p0.X + p1.X) * 0.5f, (p0.Y + p1.Y) * 0.5f);
                                        var c1 = new PointF((p1.X + p2.X) * 0.5f, (p1.Y + p2.Y) * 0.5f);

                                        positions.Add(p0);
                                        positions.Add(c0);
                                        positions.Add(c0);
                                        positions.Add(p1);

                                        positions.Add(p1);
                                        positions.Add(c1);
                                        positions.Add(c1);
                                        positions.Add(p2);

                                        switchPositions.Add(p0);
                                        switchPositions.Add(c0);
                                        switchPositions.Add(c0);
                                        switchPositions.Add(p1);

                                        switchPositions.Add(p1);
                                        switchPositions.Add(c1);
                                        switchPositions.Add(c1);
                                        switchPositions.Add(p2);

                                        break;
                                    }

                                    default:
                                    {
                                        Debug.Assert(false);
                                        break;
                                    }
                                }
                            }
                        }
                    }

                    // ハンドル
                    if (curveInterpolationType == InterpolationType.Hermite)
                    {
                        foreach (var key in keyFrames)
                        {
                            if (key.AnySelected)
                            {
                                {
                                    //MakeCurveViewScreenPos(key.Frame, key.Value, out xl, out yl);
                                    xl = curveBgWidthDiv2SubScrollXMulScaleX + stateScaleXInv * key.Frame;
                                    yl = curveBgHeightDiv2ScrollYMulScaleY - stateScaleYInv * key.Value;

                                    double handleX, handleY;
                                    MakeCurveViewScreenHandleOffset(key.OutSlope, CurveEditorConst.HandleLineLengthScale, out handleX, out handleY);

                                    handlePositions.Add(
                                        new HandlePosition
                                        {
                                            Center = new PointF((float)xl, (float)yl),
                                            Handle = new PointF((float)(xl + handleX), (float)(yl + handleY)),
                                            Selected = key.OutHandleSelected,
                                            AreaSelected = key.OutHandleAreaSelected,
                                            Slope = key.OutSlope,
                                            Type = key.IsUnionSlope ? HandleType.Unite : HandleType.Out,
                                        }
                                    );
                                }

                                {
                                    double handleX, handleY;
                                    MakeCurveViewScreenHandleOffset(key.InSlope, CurveEditorConst.HandleLineLengthScale, out handleX, out handleY);

                                    handlePositions.Add(
                                        new HandlePosition
                                        {
                                            Center = new PointF((float)xl, (float)yl),
                                            Handle = new PointF((float)(xl - handleX), (float)(yl - handleY)),
                                            Selected = key.InHandleSelected,
                                            AreaSelected = key.InHandleAreaSelected,
                                            Slope = key.InSlope,
                                            Type = key.IsUnionSlope ? HandleType.Unite : HandleType.In,
                                        }
                                    );
                                }
                            }
                        }
                    }
                }
            }

            // 描画する
            {
                var g = info.EventArgs.Graphics;

                using (var hqg = new HighQualityGraphics(g, isInterpolationMode: false, isSmoothingMode:false))
                {
                    // カーブ
                    {
                        var curveColorFlag = curve.IsEditable
                            || curve.NonEditableKind == AnimationDocument.NonEditableKind.ShapeAnimation_NotEditable
                            || curve.NonEditableKind == AnimationDocument.NonEditableKind.SkeletalAnimation_NotEditable;
                        // 非選択キーのカーブ部分
                        using (var curvePen = new Pen(curveColorFlag ? curve.CurveColor: Color.Gray))
                        {
                            foreach(var posList in unselectedPositions.Where(c => c.Count >= 4))
                            {
                                for (var i = 0; i < posList.Count; i+=4)
                                {
                                    g.SafeDrawBezier(curvePen,
                                        posList[i], posList[i+1], posList[i+2], posList[i+3]);
                                }
                            }
                        }

                        // 選択キーのカーブ部分
                        using (var curvePen = new Pen(curveColorFlag ? Color.White : Color.Gray))
                        {
                            foreach(var posList in selectedPositions.Where(c => c.Count >= 2))
                            {
                                for (var i = 0; i < posList.Count; i += 4)
                                {
                                    g.SafeDrawBezier(curvePen,
                                        posList[i], posList[i + 1], posList[i + 2], posList[i + 3]);
                                }
                            }
                        }
                    }

                    // ハンドル
                    {
                        foreach (var handlePosition in handlePositions)
                        {
                            Pen pen = CurveEditorConst.CurveHandlePen;
                            if (handlePosition.Selected)
                            {
                                pen = CurveEditorConst.CurveSelectedKeyHandlePen;
                            }
                            else
                            {
                                switch (handlePosition.Type)
                                {
                                    case HandleType.In:
                                        pen = CurveEditorConst.CurveInHandlePen;
                                        break;
                                    default:
                                        pen = CurveEditorConst.CurveOutHandlePen;
                                        break;
                                }
                            }
#if false
                            g.DrawLine(
                                pen,
                                handlePosition.Center,
                                handlePosition.Handle
                            );
#else
                            SafeDrawLine(
                                g,
                                pen,
                                handlePosition.Center,
                                handlePosition.Handle
                            );
#endif
                            var handleImage = GetHandleImage(curve, handlePosition);
                            var handleImageOffset = handleImage.Width * 0.5;

                            g.DrawImageSafe(
                                handleImage,
                                (float)(handlePosition.Handle.X - handleImageOffset),
                                (float)(handlePosition.Handle.Y - handleImageOffset)
                            );

                            if (info.IsVisibleSlope)
                            {
                                DrawString(g, string.Format("Slope: {0:f4}", handlePosition.Slope), handlePosition.Handle.X, handlePosition.Handle.Y, handlePosition.Selected);
                            }
                        }
                    }

                    // キーフレームの描画
                    {
                        var curveColorFlag = curve.IsEditable
                            || curve.NonEditableKind == AnimationDocument.NonEditableKind.ShapeAnimation_NotEditable
                            || curve.NonEditableKind == AnimationDocument.NonEditableKind.SkeletalAnimation_NotEditable;

                        using (var curvePen = new Pen(curveColorFlag ? curve.CurveColor: Color.Gray, 1.6f))
                        {
                            KeyFrame prevKey = null;

                            foreach (var key in keyFrames)
                            {
                                //MakeCurveViewScreenPos(key.Frame, key.Value, out x, out y);
                                double x = curveBgWidthDiv2SubScrollXMulScaleX + stateScaleXInv * key.Frame;
                                double y = curveBgHeightDiv2ScrollYMulScaleY - stateScaleYInv * key.Value;

                                // キーのピクセルサイズを設定
                                const float normalKeySize = 8.0f;
                                const float selectedKeySize = 12.0f;
                                float keySize = key.AreaSelected ? selectedKeySize : normalKeySize;

                                Brush keyBrush;
                                Pen keyPen;

                                GetKeyBrushAndPen(curve, key, curvePen, out keyBrush, out keyPen);

                                // 円を描画
                                {
                                    var ellipseSize = keySize * 0.75;
                                    var ellipseOffset = ellipseSize * 0.5;

                                    g.FillEllipse(
                                        keyBrush,
                                        (float)(x - ellipseOffset),
                                        (float)(y - ellipseOffset),
                                        (float)ellipseSize,
                                        (float)ellipseSize);
                                }

                                // 円の輪郭を描画
                                {
                                    var ellipseSize = keySize * 0.75;
                                    var ellipseOffset = ellipseSize * 0.5;

                                    SmoothingMode oldSmoothingMode = g.SmoothingMode;
                                    PixelOffsetMode oldPixelOffsetMode = g.PixelOffsetMode;
                                    g.SmoothingMode = SmoothingMode.AntiAlias;
                                    g.PixelOffsetMode = PixelOffsetMode.HighQuality;

                                    g.DrawEllipse(
                                        keyPen,
                                        (float)(x - ellipseOffset),
                                        (float)(y - ellipseOffset),
                                        (float)ellipseSize,
                                        (float)ellipseSize);

                                    g.SmoothingMode = oldSmoothingMode;
                                    g.PixelOffsetMode = oldPixelOffsetMode;
                                }

                                // 重複するキーに白丸をつけて目立たせる
                                if (prevKey != null)
                                {
                                    SmoothingMode oldSmoothingMode = g.SmoothingMode;

                                    const float overlapDistance = 2.0f;  // キーの重複を判定するディスプレイ上の距離
                                    const float ellipseSize = 1.4f;      // 丸の大きさをキーイメージの何倍にするか

                                    // フレームと値のディスプレイスケールが近いキーに丸をつける
                                    if (Math.Abs(prevKey.Frame - key.Frame) / State.ScaleX <= overlapDistance &&
                                        Math.Abs(prevKey.Value - key.Value) / State.ScaleY <= overlapDistance)
                                    {
                                        var ellipseWidth = keySize * ellipseSize;
                                        var ellipseOffset = ellipseWidth * 0.5;

                                        SmoothingMode prevSmoothingMode = g.SmoothingMode;
                                        PixelOffsetMode prevPixelOffsetMode = g.PixelOffsetMode;
                                        g.SmoothingMode = SmoothingMode.AntiAlias;
                                        g.PixelOffsetMode = PixelOffsetMode.HighQuality;

                                        g.DrawEllipse(Pens.White, (float)(x - ellipseOffset), (float)(y - ellipseOffset), (float)ellipseWidth, (float)ellipseWidth);

                                        g.SmoothingMode = prevSmoothingMode;
                                        g.PixelOffsetMode = prevPixelOffsetMode;
                                    }
                                }

                                prevKey = key;

                                // キー、バリューの数値
                                if (info.IsVisibleFrameValue)
                                {
                                    if (key.Selected || key.AreaSelected)
                                    {
                                        string kv = null;
                                        {
                                            switch(curve.CurvePrimitiveType)
                                            {
                                                case PrimitiveTypeKind.Float:
                                                    kv = string.Format("({0:f2}, {1:f2})", key.Frame,       key.Value);
                                                    break;
                                                case PrimitiveTypeKind.Int:
                                                case PrimitiveTypeKind.Uint:
                                                    kv = string.Format("({0:f2}, {1})",    key.Frame, (int)(key.Value));
                                                    break;
                                                case PrimitiveTypeKind.Bool:
                                                    kv = string.Format("({0:f2}, {1})",    key.Frame, key.Value >= 1.0f);
                                                    break;
                                            }
                                        }
                                        Debug.Assert(kv != null);

                                        DrawString(g, kv, x, y, key.Selected);
                                    }
                                }
                            }
                        }
                    }

                    // 最小値最大値
                    if (info.IsVisibleMinMax)
                    {
                        if (keyFrames.Any() && (curveInterpolationType == InterpolationType.Hermite))
                        {
                            for (int i = 0;i < keyFrames.Count - 1;++ i)
                            {
                                var left			= keyFrames[i + 0];
                                var right			= keyFrames[i + 1];
                                var isKeySelected	= left.Selected || right.Selected;

                                if (isKeySelected || left.OutHandleSelected || right.InHandleSelected)
                                {
                                    KeyValuePair<float, float> min, max;
                                    MathUtility.HermiteMinMax(left, right, out min, out max);

                                    {
                                        double x0, y0;

                                        float value = min.Value;

                                        MakeCurveViewScreenPos(min.Key, value, out x0, out y0);

                                        g.SafeDrawLine(CurveEditorConst.CurveMinMaxPen, (float)x0, (float)y0, (float)x0, (float)y0 - 25);

                                        string kv = string.Format("Min({0:f2}, {1:f2})", min.Key, value);
                                        DrawString(g, kv, x0, y0 - 25, false);
                                    }

                                    {
                                        double x0, y0;

                                        float value = max.Value;

                                        MakeCurveViewScreenPos(max.Key, value, out x0, out y0);

                                        g.SafeDrawLine(CurveEditorConst.CurveMinMaxPen, (float)x0, (float)y0, (float)x0, (float)y0 + 25);

                                        string kv = string.Format("Max({0:f2}, {1:f2})", max.Key, value);
                                        DrawString(g, kv, x0, y0 + 25 - 16, false);
                                    }
                                }
                            }
                        }
                    }
                }
            }

            return positions;
        }

        private void SafeDrawLine(Graphics g, Pen pen, PointF p0, PointF p1)
        {
            g.SafeDrawLine(pen, p0.X, p0.Y, p1.X, p1.Y);
        }

        // キーの座標系で直線を描画
        private void DrawLineKeyCoord(Graphics g, Pen pen, float x0, float y0, float x1, float y1)
        {
            double a0, b0, a1, b1;
            MakeCurveViewScreenPos(x0, y0, out a0, out b0);
            MakeCurveViewScreenPos(x1, y1, out a1, out b1);
            g.SafeDrawLine(pen, a0, b0, a1, b1);
        }

        const int MaxWrapPointNum = 10000;
        private void DrawPostWrapCurve(Pen pen, List<PointF> positions, curve_wrapType wrapType, AnimationCurveDrawInfo info, float beginEdgeX, float endEdgeX)
        {
            var frameEndX   = (float)CurveBgRight;

            if (positions[positions.Count - 1].X > endEdgeX)
            {
                // 描画しない
                //return;
            }

            // x座標が小さい順に作る
            var wrapPositions = new List<PointF>();
            {
                var width = (double)(positions.Last().X - positions.First().X);
                var skipRepeat = 1L;
                // 同じフレームに複数キーが打たれている場合。
                if (Math.Abs(width) < 0.00001f)
                {
                    width = 1.0;
                }

                if (width > 0.0 && width < 0.5)
                {
                    skipRepeat = (long)Math.Floor(1.0/width);
                }
                //DebugConsole.WriteLine("Wrap Width: {0}", width * skipRepeat);

                switch (wrapType)
                {
                    case curve_wrapType.clamp:
                    {
                        var p0 = positions.Last();
                        var p1 = new PointF(endEdgeX, p0.Y);

                        if (p0.X <= endEdgeX)
                        {
                            p0.X = Math.Max(p0.X, beginEdgeX);
                            p1.X = Math.Min(p1.X, endEdgeX);
                            var c0 = new PointF((p0.X + p1.X) * 0.5f, (p0.Y + p1.Y) * 0.5f);

                            wrapPositions.Add(p0);
                            wrapPositions.Add(c0);
                            wrapPositions.Add(c0);
                            wrapPositions.Add(p1);
                        }

                        break;
                    }

                    case curve_wrapType.mirror:
                    {
                        // 同じフレームに複数キーが打たれている場合。
                        if (width == 0)
                        {
                            break;
                        }

                        bool isFlip = true;

                        var x = width;

                        for(var isLoop = true;isLoop;isFlip = !isFlip)
                        {
                            {
                                if (isFlip)
                                {
                                    var b = positions.Last().X + positions.First().X;

                                    // Bezierなので4点ずつ処理
                                    var posReverse = positions.Reverse<PointF>().ToArray();
                                    for (var i = 0; i < posReverse.Count(); i += 4)
                                    {
                                        var outLeft  = false;
                                        var outRight = false;
                                        var inArea = false;

                                        foreach (var j in new[]{0,3})
                                        {
                                            var pos = posReverse[i + j];
                                            var putX = (float)((b - pos.X) + x);
                                            if (putX <= beginEdgeX) outLeft = true;
                                            else if (putX >= endEdgeX || putX >= frameEndX) outRight = true;
                                            else { inArea = true; }
                                        }

                                        if (inArea || (outLeft && outRight))
                                        {
                                            for (var j = 0; j < 4; j++)
                                            {
                                                var pos = posReverse[i + j];
                                                var putX = (float)((b - pos.X) + x);
                                                wrapPositions.Add(new PointF { X = putX, Y = pos.Y });
                                            }
                                        }
                                        else if (outRight)
                                        {
                                            isLoop = false;
                                            break;
                                        }
                                    }
                                }
                                else
                                {
                                    // Bezierなので4点ずつ処理
                                    for (var i = 0; i < positions.Count(); i += 4)
                                    {
                                        var outLeft = false;
                                        var outRight = false;
                                        var inArea = false;

                                        foreach (var j in new[] { 0, 3 })
                                        {
                                            var pos = positions[i + j];
                                            var putX = (float)(pos.X + x);
                                            if (putX <= beginEdgeX) outLeft = true;
                                            else if (putX >= endEdgeX || putX >= frameEndX) outRight = true;
                                            else { inArea = true; }
                                        }

                                        if (inArea || (outLeft && outRight))
                                        {
                                            for (var j = 0; j < 4; j++)
                                            {
                                                var pos = positions[i + j];
                                                var putX = (float)(pos.X + x);
                                                wrapPositions.Add(new PointF { X = putX, Y = pos.Y });
                                            }
                                        }
                                        else if (outRight)
                                        {
                                            isLoop = false;
                                            break;
                                        }
                                    }
                                }

                                if (wrapPositions.Count > MaxWrapPointNum)
                                {
                                    isLoop = false;
                                }
                            }
                            x += width * skipRepeat;
                        }

                        break;
                    }

                    case curve_wrapType.repeat:
                    {
                        // 同じフレームに複数キーが打たれている場合。
                        if (width == 0)
                        {
                            break;
                        }

                        var x = width;

                        for(var isLoop = true;isLoop;)
                        {
                            {
                                for (var i = 0; i < positions.Count(); i += 4)
                                {
                                    var outLeft = false;
                                    var outRight = false;
                                    var inArea = false;

                                    foreach (var j in new[] { 0, 3 })
                                    {
                                        var pos = positions[i + j];
                                        var putX = (float)(pos.X + x);
                                        if (putX <= beginEdgeX) outLeft = true;
                                        else if (putX >= endEdgeX || putX >= frameEndX) outRight = true;
                                        else { inArea = true; }
                                    }

                                    if (inArea || (outLeft && outRight))
                                    {
                                        for (var j = 0; j < 4; j++)
                                        {
                                            var pos = positions[i + j];
                                            var putX = (float)(pos.X + x);
                                            wrapPositions.Add(new PointF { X = putX, Y = pos.Y });
                                        }
                                    }
                                    else if(outRight)
                                    {
                                        isLoop = false;
                                        break;
                                    }
                                }

                                if (wrapPositions.Count > MaxWrapPointNum)
                                {
                                    isLoop = false;
                                }
                            }

                            x += width * skipRepeat;
                        }

                        break;
                    }

                    case curve_wrapType.relative_repeat:
                    {
                        // 同じフレームに複数キーが打たれている場合。
                        if (width == 0)
                        {
                            break;
                        }

                        var x = width;

                        var offsetY = positions[positions.Count - 1].Y - positions[0].Y;
                        var y = offsetY;
                        for (var isLoop = true; isLoop; )
                        {
                            {
                                for (var i = 0; i < positions.Count(); i += 4)
                                {
                                    var outLeft = false;
                                    var outRight = false;
                                    var inArea = false;

                                    foreach (var j in new[] { 0, 3 })
                                    {
                                        var pos = positions[i + j];
                                        var putX = (float)(pos.X + x);
                                        if (putX <= beginEdgeX) outLeft = true;
                                        else if (putX >= endEdgeX || putX >= frameEndX) outRight = true;
                                        else { inArea = true; }
                                    }

                                    if (inArea || (outLeft && outRight))
                                    {
                                        for (var j = 0; j < 4; j++)
                                        {
                                            var pos = positions[i + j];
                                            var putX = (float)(pos.X + x);
                                            var putY = pos.Y + y;
                                            wrapPositions.Add(new PointF { X = putX, Y = putY });
                                        }
                                    }
                                    else if (outRight)
                                    {
                                        isLoop = false;
                                        break;
                                    }
                                }

                                if (wrapPositions.Count > MaxWrapPointNum)
                                {
                                    isLoop = false;
                                }
                            }
                            // 和の誤差が蓄積するのを防ぎたければ double を使うか掛け算を使うように変更
                            x += width * skipRepeat;
                            y += offsetY * skipRepeat;
                        }

                        break;
                    }
                }
            }

            if (wrapPositions.Count >= 2)
            {
                var g = info.EventArgs.Graphics;
                var orgClip = g.Clip;

                // 大きい値の入ったRegionでIntersectすると、不正な値になるケースが有る模様
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1930#8
                //var clip = new Region(new RectangleF(beginEdgeX, CurveBgTop, endEdgeX - beginEdgeX, CurveBgHeight));
                var left = MathUtility.ClampedValue(beginEdgeX, CurveBgLeft, CurveBgRight);
                var right = MathUtility.ClampedValue(endEdgeX, CurveBgLeft, CurveBgRight);
                var clip = new Region(new RectangleF(left, CurveBgTop, right - left, CurveBgHeight));
                clip.Intersect(g.Clip);
                g.Clip = clip;

                for (var i = 0; i < wrapPositions.Count; i += 4)
                {
                    g.SafeDrawBezier(pen,
                        wrapPositions[i], wrapPositions[i + 1], wrapPositions[i + 2], wrapPositions[i + 3]);
                }
                g.Clip = orgClip;
            }
        }

        private void DrawPreWrapCurve(Pen pen, List<PointF> positions, curve_wrapType wrapType, AnimationCurveDrawInfo info, float beginEdgeX, float endEdgeX)
        {
            var frameBeginX = (float)CurveBgLeft;

            if (positions[0].X < beginEdgeX)
            {
                // 描画しない
                //return;
            }

            // x座標が小さい順に作る
            var wrapPositions = new List<PointF>();
            {
                var width = (double)(positions.Last().X - positions.First().X);
                var skipRepeat = 1L;
                // 同じフレームに複数キーが打たれている場合。
                if (Math.Abs(width) < 0.00001f)
                {
                    width = 1.0;
                }

                if (width > 0.0 && width < 0.5)
                {
                    skipRepeat = (long)Math.Floor(1.0 / width);
                }


                switch (wrapType)
                {
                    case curve_wrapType.clamp:
                    {
                        var p0 = positions.First();
                        var p1 = new PointF(beginEdgeX, p0.Y);

                        if (p0.X >= beginEdgeX)
                        {
                            p0.X = Math.Min(p0.X, endEdgeX);
                            p1.X = Math.Max(p1.X, beginEdgeX);
                            var c0 = new PointF((p0.X + p1.X) * 0.5f, (p0.Y + p1.Y) * 0.5f);

                            wrapPositions.Add(p0);
                            wrapPositions.Add(c0);
                            wrapPositions.Add(c0);
                            wrapPositions.Add(p1);
                        }

                        break;
                    }

                    case curve_wrapType.mirror:
                    {
                        bool isFlip = true;

                        var x = width;

                        // 同じフレームに複数キーが打たれている場合。
                        if (width == 0)
                        {
                            break;
                        }

                        // 点が極端に多くなる場合は描画しない
                        if (10000 * width < endEdgeX - beginEdgeX)
                        {
                            //break;
                        }

                        for(var isLoop = true;isLoop;isFlip = !isFlip)
                        {
                            if (isFlip)
                            {
                                var b = positions.Last().X + positions.First().X;
                                // Bezierなので4点ずつ処理
                                for (var i = 0; i < positions.Count(); i += 4)
                                {
                                    var outLeft = false;
                                    var outRight = false;
                                    var inArea = false;

                                    foreach (var j in new[] { 0, 3 })
                                    {
                                        var pos = positions[i + j];
                                        var putX = (float)((b - pos.X) - x);
                                        if (putX <= beginEdgeX || putX <= frameBeginX) outLeft = true;
                                        else if (putX >= endEdgeX) outRight = true;
                                        else { inArea = true; }
                                    }

                                    if (inArea || (outLeft && outRight))
                                    {
                                        for (var j = 0; j < 4; j++)
                                        {
                                            var pos = positions[i + j];
                                            var putX = (float)((b - pos.X) - x);
                                            wrapPositions.Add(new PointF { X = putX, Y = pos.Y });
                                        }
                                    }
                                    else if (outLeft)
                                    {
                                        isLoop = false;
                                        break;
                                    }
                                }
                            }
                            else
                            {
                                var posReverse = positions.Reverse<PointF>().ToArray();
                                for (var i = 0; i < posReverse.Count(); i += 4)
                                {
                                    var outLeft = false;
                                    var outRight = false;
                                    var inArea = false;

                                    foreach (var j in new[] { 0, 3 })
                                    {
                                        var pos = posReverse[i + j];
                                        var putX = (float)(pos.X - x);
                                        if (putX <= beginEdgeX || putX <= frameBeginX) outLeft = true;
                                        else if (putX >= endEdgeX) outRight = true;
                                        else { inArea = true; }
                                    }

                                    if (inArea || (outLeft && outRight))
                                    {
                                        for (var j = 0; j < 4; j++)
                                        {
                                            var pos = posReverse[i + j];
                                            var putX = (float)(pos.X - x);
                                            wrapPositions.Add(new PointF { X = putX, Y = pos.Y });
                                        }
                                    }
                                    else if (outLeft)
                                    {
                                        isLoop = false;
                                        break;
                                    }
                                }
                            }

                            if (wrapPositions.Count > MaxWrapPointNum)
                            {
                                isLoop = false;
                            }
                            x += width * skipRepeat;
                        }

                        // 逆順に積んだのでひっくり返す
                        wrapPositions.Reverse(0, wrapPositions.Count);
                        break;
                    }

                    case curve_wrapType.repeat:
                    {
                        var x = width;

                        // 同じフレームに複数キーが打たれている場合。
                        if (width == 0)
                        {
                            break;
                        }

                        // 点が極端に多くなる場合は描画しない
                        if (10000 * width < endEdgeX - beginEdgeX)
                        {
                            //break;
                        }

                        for (var isLoop = true; isLoop; )
                        {
                            var posReverse = positions.Reverse<PointF>().ToArray();
                            for (var i = 0; i < posReverse.Count(); i += 4)
                            {
                                var outLeft = false;
                                var outRight = false;
                                var inArea = false;

                                foreach (var j in new[] { 0, 3 })
                                {
                                    var pos = posReverse[i + j];
                                    var putX = (float)(pos.X - x);
                                    if (putX <= beginEdgeX || putX <= frameBeginX) outLeft = true;
                                    else if (putX >= endEdgeX) outRight = true;
                                    else { inArea = true; }
                                }

                                if (inArea || (outLeft && outRight))
                                {
                                    for (var j = 0; j < 4; j++)
                                    {
                                        var pos = posReverse[i + j];
                                        var putX = (float)(pos.X - x);
                                        wrapPositions.Add(new PointF { X = putX, Y = pos.Y });
                                    }
                                }
                                else if (outLeft)
                                {
                                    isLoop = false;
                                    break;
                                }
                            }

                            if (wrapPositions.Count > MaxWrapPointNum)
                            {
                                isLoop = false;
                            }
                            x += width * skipRepeat;
                        }

                        // 逆順に積んだのでひっくり返す
                        wrapPositions.Reverse(0, wrapPositions.Count);
                        break;
                    }

                    case curve_wrapType.relative_repeat:
                    {
                        var x = width;

                        // 同じフレームに複数キーが打たれている場合。
                        if (width == 0)
                        {
                            break;
                        }

                        // 点が極端に多くなる場合は描画しない
                        if (10000 * width < endEdgeX - beginEdgeX)
                        {
                            //break;
                        }

                        var offsetY = positions[0].Y - positions[positions.Count - 1].Y;
                        var y = offsetY;
                        for (var isLoop = true; isLoop; )
                        {
                            var posReverse = positions.Reverse<PointF>().ToArray();
                            for (var i = 0; i < posReverse.Count(); i += 4)
                            {
                                var outLeft = false;
                                var outRight = false;
                                var inArea = false;

                                foreach (var j in new[] { 0, 3 })
                                {
                                    var pos = posReverse[i + j];
                                    var putX = (float)(pos.X - x);
                                    if (putX <= beginEdgeX || putX <= frameBeginX) outLeft = true;
                                    else if (putX >= endEdgeX) outRight = true;
                                    else { inArea = true; }
                                }

                                if (inArea || (outLeft && outRight))
                                {
                                    for (var j = 0; j < 4; j++)
                                    {
                                        var pos = posReverse[i + j];
                                        var putX = (float)(pos.X - x);
                                        var putY = pos.Y + y;
                                        wrapPositions.Add(new PointF { X = putX, Y = putY });
                                    }
                                }
                                else if (outLeft)
                                {
                                    isLoop = false;
                                    break;
                                }
                            }

                            if (wrapPositions.Count > MaxWrapPointNum)
                            {
                                isLoop = false;
                            }
                            // 和の誤差が蓄積するのを防ぎたければ double を使うか掛け算を使うように変更
                            x += width * skipRepeat;
                            y += offsetY * skipRepeat;
                        }

                        // 逆順に積んだのでひっくり返す
                        wrapPositions.Reverse(0, wrapPositions.Count);
                        break;
                    }
                }
            }


            if (wrapPositions.Count >= 2)
            {
                var g = info.EventArgs.Graphics;
                var orgClip = g.Clip;
                // 大きい値の入ったRegionでIntersectすると、不正な値になるケースが有る模様
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1930#8
                //var clip = new Region(new RectangleF(beginEdgeX, CurveBgTop, endEdgeX - beginEdgeX, CurveBgHeight));
                var left = MathUtility.ClampedValue(beginEdgeX, CurveBgLeft, CurveBgRight);
                var right = MathUtility.ClampedValue(endEdgeX, CurveBgLeft, CurveBgRight);
                var clip = new Region(new RectangleF(left, CurveBgTop, right - left, CurveBgHeight));
                clip.Intersect(g.Clip);
                g.Clip = clip;
                for (var i = 0; i < wrapPositions.Count; i += 4)
                {
                    g.SafeDrawBezier(pen,
                        wrapPositions[i], wrapPositions[i + 1], wrapPositions[i + 2], wrapPositions[i + 3]);
                }
                g.Clip = orgClip;
            }
        }

        // フレームカウント内にフィットする
        private List<PointF> FitWrapPositions(List<PointF> srcPositions, float beginEdgeX, float endEdgeX, bool isPre)
        {
            if (srcPositions.Any())
            {
                if (srcPositions.Count >= 2)
                {
                    // 補完する

                    if (isPre)
                    {
                        if (srcPositions[0].X < beginEdgeX)
                        {
                            var first = srcPositions[0];
                            var next  = srcPositions[1];

                            if (first.X == next.X)
                            {
                                // 何もしない
                            }
                            else
                            {
                                var vectY  = first.Y - next.Y;
                                var scaleX = (beginEdgeX - next.X) / (first.X - next.X);
                                var posY   = vectY * scaleX + next.Y;

                                srcPositions[0] = new PointF(beginEdgeX, posY);
                            }
                        }
                    }
                    else
                    {
                        if (srcPositions[srcPositions.Count - 1].X > endEdgeX)
                        {
                            var last = srcPositions[srcPositions.Count - 1];
                            var next = srcPositions[srcPositions.Count - 2];

                            if (last.X == next.X)
                            {
                                // 何もしない
                            }
                            else
                            {
                                var vectY  = last.Y - next.Y;
                                var scaleX = (endEdgeX - next.X) / (last.X - next.X);
                                var posY   = vectY * scaleX + next.Y;

                                srcPositions[srcPositions.Count - 1] = new PointF(endEdgeX, posY);
                            }
                        }
                    }
                }
            }

            return srcPositions;
        }

        public void MakeCurveViewScreenPos(float srcX, float srcY, out double dstX, out double dstY)
        {
            dstX = MakeCurveViewScreenPosX(srcX);
            dstY = MakeCurveViewScreenPosY(srcY);
        }

        public double MakeCurveViewScreenPosX(float srcX)
        {
            return (CurveBgWidth  * 0.5) - (State.ScrollX * State.ScaleX) + (srcX / State.ScaleX);
        }

        public double MakeCurveViewScreenPosY(float srcY)
        {
            return (CurveBgHeight * 0.5) - (State.ScrollY * State.ScaleY) - (srcY / State.ScaleY);
        }

        public void MakeCurveViewScreenHandleOffset(double slope, double handleLineLength, out double handleX, out double handleY)
        {
            double r = Math.Atan(slope * State.ScaleX / State.ScaleY);

            handleX = handleLineLength *  Math.Cos(r);
            handleY = handleLineLength * -Math.Sin(r);
        }

        public double MakeSlope(double x, double y)
        {
            // 天地逆にする
            y = -y;

            double r = Math.Atan2(y, x);
            double s = Math.Tan(r);

            return s * State.ScaleY / State.ScaleX;
        }

        public bool IsInDrawArea(double xl, double yl, double xr, double yr)
        {
            return
                ! (((xl < CurveBgLeft                ) && (xr < CurveBgLeft               )) ||
                   ((xl > CurveBgLeft + CurveBgWidth ) && (xr > CurveBgLeft + CurveBgWidth)));
        }

        public void DrawString(Graphics g, string s, double x, double y, bool selected)
        {
            // 大雑把にクリップ
            if (g.ClipBounds.Left - 1000 < x && x < g.ClipBounds.Right + 1000 && g.ClipBounds.Top - 1000 < y && y < g.ClipBounds.Bottom + 1000)
            {
                g.DrawString(s, TheApp.GuiFontBold, CurveEditorConst.CurveFontDarkBrush, (float)x + 1.0f, (float)y);
                g.DrawString(s, TheApp.GuiFontBold, CurveEditorConst.CurveFontDarkBrush, (float)x - 1.0f, (float)y);
                g.DrawString(s, TheApp.GuiFontBold, CurveEditorConst.CurveFontDarkBrush, (float)x, (float)y - 1.0f);
                g.DrawString(s, TheApp.GuiFontBold, CurveEditorConst.CurveFontDarkBrush, (float)x, (float)y + 1.0f);

                g.DrawString(
                    s,
                    TheApp.GuiFontBold,
                    selected ?
                        CurveEditorConst.CurveSelectedKeyFontBrush :
                        CurveEditorConst.CurveFontLightBrush,
                    (float)x,
                    (float)y
                );
            }
        }
    }
}
