﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using Microsoft.WindowsAPICodePack.Dialogs;
using App.Data;
using Microsoft.Scripting.Utils;
using nw.g3d.iflib.nw3de;

namespace App.Controls
{
    using System.IO;

    using App.ConfigData;
    using App.Properties;
    using App.Utility;


    using nw.g3d.iflib;
    using nw.g3d.nw4f_3dif;

    using TeamConfig;


    public partial class MaterialParentDialog : OkCancelDialog
    {
        private Material ParentMaterial;
        private Material ChildMaterial;
        public MaterialInfo ParentMaterialInfo { get; set; }
        public Dictionary<string, List<MaterialInfo>> TeamSettingMaterials = null;

        private static string parentMaterialsFolder = string.Empty;
        private static string teamSettingNameValue = string.Empty;
        private static ListType selectedListType = ListType.TeamSetting;

        private enum ListType
        {
            TeamSetting,
            Loaded,
            UserDefined
        }

        public class MaterialInfo
        {
            public string FullPath { get; private set; }
            public string RelativePath { get; private set; }
            public string MaterialName { get; private set; }
            public bool IsLoaded { get; private set; }
            public commentType ModelComment { get; private set; }
            public commentType MaterialComment { get; private set; }
            public MaterialInfo()
            {
                FullPath = null;
                RelativePath = null;
                MaterialName = null;
                ModelComment = null;
                MaterialComment = null;
            }

            public MaterialInfo(Material material, bool isLoaded)
                : this(
                material?.OwnerDocument?.FilePath,
                null,
                material?.Name,
                (material?.OwnerDocument is Model) ? (material?.OwnerDocument as Model).Data?.comment :
                (material?.OwnerDocument is SeparateMaterial) ? (material?.OwnerDocument as SeparateMaterial).Data?.comment : null,
                material?.Data?.comment,
                isLoaded)
            {
            }

            public MaterialInfo(string fullPath, string relativePath, string materialName, commentType modelComment, commentType materialComment, bool isLoaded)
                : this()
            {
                FullPath = fullPath;
                RelativePath = relativePath;
                MaterialName = materialName;
                ModelComment = modelComment;
                MaterialComment = materialComment;
                IsLoaded = isLoaded;
            }


            public bool HasMaterial()
            {
                return !string.IsNullOrEmpty(FullPath) && !string.IsNullOrEmpty(MaterialName);
            }

            public bool IsSeparateMaterial
            {
                get
                {
                    return
                        string.Equals(System.IO.Path.GetExtension(FullPath), G3dPath.MaterialBinaryExtension, StringComparison.OrdinalIgnoreCase) ||
                        string.Equals(System.IO.Path.GetExtension(FullPath), G3dPath.MaterialTextExtension, StringComparison.OrdinalIgnoreCase);
                }
            }

            protected bool Equals(MaterialInfo other)
            {
                return string.Equals(FullPath, other.FullPath) && string.Equals(MaterialName, other.MaterialName) && IsLoaded == other.IsLoaded && Equals(ModelComment, other.ModelComment) && Equals(MaterialComment, other.MaterialComment);
            }

            public override bool Equals(object obj)
            {
                if (ReferenceEquals(null, obj)) return false;
                if (ReferenceEquals(this, obj)) return true;
                if (obj.GetType() != this.GetType()) return false;
                return Equals((MaterialInfo) obj);
            }

            public override int GetHashCode()
            {
                unchecked
                {
                    var hashCode = (FullPath != null ? FullPath.GetHashCode() : 0);
                    hashCode = (hashCode * 397) ^ (MaterialName != null ? MaterialName.GetHashCode() : 0);
                    hashCode = (hashCode * 397) ^ IsLoaded.GetHashCode();
                    hashCode = (hashCode * 397) ^ (ModelComment != null ? ModelComment.GetHashCode() : 0);
                    hashCode = (hashCode * 397) ^ (MaterialComment != null ? MaterialComment.GetHashCode() : 0);
                    return hashCode;
                }
            }
        }


        public MaterialParentDialog(Material childMaterial, Material parentMaterial = null)
        {
            InitializeComponent();

            ChildMaterial = childMaterial;
            ParentMaterial = parentMaterial;

            if (ParentMaterial != null)
            {
                ParentMaterialInfo = new MaterialInfo(ParentMaterial, true);
            }

            tbxParentFolder.Text = parentMaterialsFolder ?? string.Empty;

            if (selectedListType == ListType.TeamSetting && !ApplicationConfig.FileIo.ParentMaterialPaths.Any())
            {
                selectedListType = ListType.Loaded;
            }

            switch (selectedListType)
            {
                case ListType.Loaded:
                    rbtLoaded.Checked = true;
                    break;
                case ListType.UserDefined:
                    rbtUserDefine.Checked = true;
                    break;
                case ListType.TeamSetting:
                default:
                    rbtTeamSetting.Checked = true;
                    break;
            }

            UpdateSelectControls();
            UpdateListview();
        }

        public static void ListUpMatchedFilesAndFolders(string path, out List<string> files, out List<string> folders)
        {
            // path を正規化する。
            // Path.GetFullPath() では存在しないパスの正規化はできないので PathCanonicalize() を使う。
            // 存在しないパスの場合 Win32.Constants.MAX_PATH に収まっているとは限らないので、長い方で確保する。
            var pathBuilder = new StringBuilder(Math.Max(path.Length, Win32.Constants.MAX_PATH));
            if (Win32.NativeMethods.PathCanonicalize(pathBuilder, path))
            {
                path = pathBuilder.ToString().Replace(System.IO.Path.AltDirectorySeparatorChar, System.IO.Path.DirectorySeparatorChar);
            }
            // パスルート
            var root = Path.GetPathRoot(path);
            // ルート以外のパスをフォルダごとに分割
            var underroot = path.Substring(root.Length);
            var splited = underroot.Split(new[] { '\\' }, StringSplitOptions.RemoveEmptyEntries);
            // 分割したパスをワイルドカードを含むフォルダ、ファイルが末尾になるように連結
            // AAA\BBB*\CCC*\*.fmdb のばあい、「AAA\BBB*」「CCC*」「*.fmdb」
            var searchs = new List<string>();
            var joined = string.Empty;
            foreach (var entry in splited)
            {
                joined = Path.Combine(joined, entry);
                if (splited.Last() == entry || entry.Contains('*') || entry.Contains('?'))
                {
                    searchs.Add(joined);
                    joined = string.Empty;
                }
            }

            // 検索対象がフォルダだけかどうかを末尾のパス区切り文字の有無で判定
            var isSearchFolder = path.Last() == '\\';

            // ルートから検索していく
            folders = new List<string> {root};
            files = new List<string>();

            for (var i = 0; i < searchs.Count; i++)
            {
                var search = searchs[i];
                var isLast = i == (searchs.Count - 1);
                var newFolders = new List<string>();
                foreach (var candidate in folders)
                {
                    try
                    {
                        var fileSystemEntries = Directory.EnumerateFileSystemEntries(candidate, search);
                        foreach (var entry in fileSystemEntries)
                        {
                            if (Directory.Exists(entry))
                            {
                                newFolders.Add(entry);
                            }
                            else if (isLast && !isSearchFolder && (entry.EndsWith(".fmdb") || entry.EndsWith(".fmda") || entry.EndsWith(G3dPath.MaterialBinaryExtension) || entry.EndsWith(G3dPath.MaterialTextExtension)) && File.Exists(entry))
                            {
                                files.Add(entry);
                            }
                        }
                    }
                    catch (DirectoryNotFoundException e)
                    {
                        DebugConsole.WriteLine(@"ParentMaterialPath not found: " + e.Message);
                    }
                    catch (Exception e)
                    {
                        DebugConsole.WriteLine(@"ParentMaterialPath exception: " + e.Message);
                    }
                }

                folders = newFolders;
            }
        }


        private void UpdateListview()
        {
            lvwParentMaterials.BeginUpdate();
            lvwParentMaterials.Items.Clear();
            var childMaterialOwner = ChildMaterial.OwnerDocument;
            var childShaderAssign = ChildMaterial.Data.shader_assign;
            var loadedModelNames = DocumentManager.Models.Select(x => x.Name).ToArray();
            var items = new List<ListViewItem>();


            using (var cursor = new WaitCursor())
            using (var watch = new DebugStopWatch("MaterialParentDialog UpdateListview"))
            {
                if (selectedListType == ListType.Loaded)
                {
                    foreach (var model in DocumentManager.Models.Where(x => !string.IsNullOrEmpty(x.FilePath)))
                    {
                        var modelName = GuiObjectNameWithComment(model);
                        var modelFolder = model != childMaterialOwner ? model.FileLocation : string.Empty;
                        var materials = GetParentableMaterials(model, ChildMaterial);
                        items.AddRange(
                            materials.Select(
                                material =>
                                    new ListViewItem(new[] {GuiObjectNameWithComment(material), modelName, modelFolder})
                                    {
                                        Tag = new MaterialInfo(material, true)
                                    }));
                    }
                    foreach (var material in DocumentManager.SeparateMaterials.Where(x => !string.IsNullOrEmpty(x.FilePath)))
                    {
                        var materialName = GuiObjectNameWithComment(material);
                        var materialFolder = material != childMaterialOwner ? material.FileLocation : string.Empty;
                        var materials = GetParentableMaterials(material, ChildMaterial);
                        items.AddRange(
                            materials.Select(
                                x =>
                                    new ListViewItem(new[] { GuiObjectNameWithComment(x), materialName, materialFolder })
                                    {
                                        Tag = new MaterialInfo(x, true)
                                    }));
                    }
                }
                else if (selectedListType == ListType.UserDefined)
                {
                    var path = parentMaterialsFolder;
                    if (!string.IsNullOrEmpty(path) && Path.IsPathRooted(path))
                    {
                        IEnumerable<string> files;
                        try
                        {
                            files = Directory.EnumerateFiles(path, "*.fmdb").Concat(Directory.EnumerateFiles(path, "*.fmda")).Concat(Directory.EnumerateFiles(path, "*.fmtb"))
                                .Where(
                                    x =>
                                        DocumentManager.Models.All(model => !model.Name.Equals(x, StringComparison.OrdinalIgnoreCase))
                                        || DocumentManager.Models.Any(model => x.Equals(model.FilePath, StringComparison.OrdinalIgnoreCase)));
                        }
                        catch (Exception e)
                        {
                            UIMessageBox.Error(
                                string.Format(
                                    Resources.MaterialParentDialog_UpdateListview_SearchParentMaterialException,
                                    path,
                                    e.Message));
                            return;
                        }
                        foreach (var filePath in files)
                        {
                            var model = GetModelFromFile(filePath);
                            if (model?.material_array?.material?.Length > 0)
                            {
                                var modelName = GetNameWithComment(Path.GetFileNameWithoutExtension(filePath), model.comment);
                                // パスの一部をもとの指定で置き換え(環境変数など)
                                var modelFolder = Path.GetDirectoryName(filePath);

                                var isChildMaterialOwner = childMaterialOwner.FilePath.Equals(filePath, StringComparison.OrdinalIgnoreCase);
                                var viewItems =
                                    model.material_array.material
                                        .Where(x => IsSameShaderAssigned(x.shader_assign, childShaderAssign))
                                        .Where(x => !isChildMaterialOwner || ChildMaterial.Name != x.name)
                                        .Where(x => ChildMaterial.ParentMaterials.All(y => !(y.Name.Equals(x.name) && y.OwnerDocument.FilePath.Equals(filePath))))
                                        .Select(
                                            x => new ListViewItem(new[] { GetNameWithComment(x.name, x.comment), modelName, modelFolder })
                                            {
                                                Tag = new MaterialInfo(filePath, null, x.name, model.comment, x.comment, false)
                                            });
                                items.AddRange(viewItems);
                                continue;
                            }

                            var material = GetMaterialFromFile(filePath);
                            if (material != null)
                            {
                                // fmt ではファイル名がマテリアル名。
                                var materialName = Path.GetFileNameWithoutExtension(filePath);
                                // パスの一部をもとの指定で置き換え(環境変数など)
                                var materialFolder = Path.GetDirectoryName(filePath);

                                var isChildMaterialOwner = childMaterialOwner.FilePath.Equals(filePath, StringComparison.OrdinalIgnoreCase);
                                var viewItems =
                                    Enumerable.Repeat(material, 1)
                                        .Where(x => IsSameShaderAssigned(x.shader_assign, childShaderAssign))
                                        .Where(x => !isChildMaterialOwner || ChildMaterial.Name != materialName)
                                        .Where(x => ChildMaterial.ParentMaterials.All(y => !(y.Name.Equals(materialName) && y.OwnerDocument.FilePath.Equals(filePath))))
                                        .Select(
                                            x => new ListViewItem(new[] { GetNameWithComment(materialName, material.comment), string.Empty, materialFolder})
                                            {
                                                Tag = new MaterialInfo(filePath, null, materialName, null, x.comment, false)
                                            });
                                items.AddRange(viewItems);
                                continue;
                            }
                        }
                    }
                }
                else //(selectedListType == ListType.TeamSetting)
                {
                    if (TeamSettingMaterials == null)
                    {
                        UpdateTeamConfigParentMaterialInfo();
                    }
                    IEnumerable<ListViewItem> viewItems = null;
                    IEnumerable<MaterialInfo> materialInfos = null;
                    if (string.IsNullOrEmpty(teamSettingNameValue))
                    {
                        materialInfos = TeamSettingMaterials.SelectMany(x => x.Value).Distinct();
                    }
                    else
                    {
                        if (TeamSettingMaterials.ContainsKey(teamSettingNameValue))
                        {
                            materialInfos = TeamSettingMaterials[teamSettingNameValue];
                        }
                    }
                    if (materialInfos != null)
                    {
                        viewItems = materialInfos.Select(x => new ListViewItem(new[]
                        {
                            GetNameWithComment(x.MaterialName, x.MaterialComment),
                            x.IsSeparateMaterial ? string.Empty : GetNameWithComment(Path.GetFileNameWithoutExtension(x.FullPath), x.ModelComment),
                            Path.GetDirectoryName(x.FullPath)
                        })
                        {
                            Tag = x
                        });
                        items.AddRange(viewItems);
                    }
                }

                var index = items.FindIndex(x => (x.Tag as MaterialInfo)?.Equals(ParentMaterialInfo) == true);
                lvwParentMaterials.Items.AddRange(items.ToArray());
                foreach (ColumnHeader column in lvwParentMaterials.Columns)
                {
                    column.Width = -2;
                }
                lvwParentMaterials.EndUpdate();
                btnOK.Enabled = false;
                if (index >= 0)
                {
                    lvwParentMaterials.SetSelectedItem(index);
                }
            }
        }

        private static modelType GetModelFromFile(string filePath)
        {
            return GetModelOrMaterialFromFile(filePath) as modelType;
        }

        private static materialType GetMaterialFromFile(string filePath)
        {
            return GetModelOrMaterialFromFile(filePath) as materialType;
        }

        private static object GetModelOrMaterialFromFile(string filePath)
        {
            nw4f_3difType nwif = null;
            var isIfUpdated = false;
            if (G3dPath.IsStreamBinaryPath(filePath))
            {
                var fileImage = File.ReadAllBytes(filePath);
                var binaryOffset = 0;
                nwif = IfBinaryReadUtility.Read(fileImage, DocumentManager.XsdBasePath, out binaryOffset, out isIfUpdated);
            }
            else
            {
                nwif = IfReadUtility.Read(filePath, DocumentManager.XsdBasePath, out isIfUpdated);
            }

            return nwif?.Item;
        }

        private bool UpdateTeamConfigParentMaterialInfo()
        {
            var childMaterialOwner = ChildMaterial.OwnerDocument;
            TeamSettingMaterials = new Dictionary<string, List<MaterialInfo>>();
            TeamSettingMaterials[string.Empty] = new List<MaterialInfo>();
            var parentMaterialPaths = ApplicationConfig.FileIo.ParentMaterialPaths;
            var lastTeamSettingName = teamSettingNameValue;
            cmbTeamSettingNames.Items.Clear();

            foreach (var parentMaterialPathInfo in parentMaterialPaths)
            {
                var parentMaterialPath = parentMaterialPathInfo.path.Replace('/', '\\');
                if (string.IsNullOrEmpty(parentMaterialPath) || !Path.IsPathRooted(parentMaterialPath))
                {
                    continue;
                }
                parentMaterialPath = PathUtility.GetFullPath(parentMaterialPath);
                List<string> matchedFiles;
                List<string> matchedFolders;
                ListUpMatchedFilesAndFolders(parentMaterialPath, out matchedFiles, out matchedFolders);

                // フルパスと親マテリアルパスからの相対
                var pathInfos = new List<Tuple<string, string>>();

                pathInfos.AddRange(matchedFiles.Select(x => new Tuple<string, string>(x, Path.GetFileName(x))));

                var option = parentMaterialPathInfo.Recursive ? SearchOption.AllDirectories : SearchOption.TopDirectoryOnly;
                try
                {
                    if (matchedFolders.Any())
                    {
                        foreach (var matchedFolder in matchedFolders)
                        {
                            var folderFiles = Directory.EnumerateFiles(matchedFolder, "*.fmdb", option)
                                .Concat(Directory.EnumerateFiles(matchedFolder, "*.fmda", option))
                                .Concat(Directory.EnumerateFiles(matchedFolder, "*.fmtb", option));
                            pathInfos.AddRange(
                                folderFiles.Select(file => new Tuple<string, string>(file, Path.GetFileName(file))));
                    }
                }
                }
                catch (Exception e)
                {
                    UIMessageBox.Error(
                        string.Format(
                            Resources.MaterialParentDialog_UpdateListview_SearchParentMaterialException,
                            parentMaterialPathInfo.pathXml,
                            e.Message));
                    continue;
                }

                pathInfos = pathInfos.Where(x => DocumentManager.Models.All(model => !model.Name.Equals(Path.GetFileNameWithoutExtension(x.Item1), StringComparison.OrdinalIgnoreCase))
                                     || DocumentManager.Models.Any(model => model.FilePath.Equals(x.Item1, StringComparison.OrdinalIgnoreCase))).ToList();
                if (!pathInfos.Any())
                {
                    continue;
                }

                if (!PrePostIO.ExecutePreOpen(pathInfos.Select(x => x.Item1).ToArray()))
                {
                    return false;
                }

                var regex = WildCardToRegex(parentMaterialPath);

                foreach (var pathInfo in pathInfos)
                {
                    var filePath = pathInfo.Item1;

                    var model = GetModelFromFile(filePath);
                    if (model?.material_array?.material?.Length > 0)
                    {
                        var name = parentMaterialPathInfo.Name ?? string.Empty;
                        var match = regex.Match(filePath);
                        for (var i = 1; i < match.Groups.Count; i++)
                        {
                            var matchGroup = match.Groups[i];
                            if (matchGroup.Captures.Count == 0)
                            {
                                continue;
                            }
                            name = name.Replace($"\\{i}", matchGroup.Value);
                        }

                        List<MaterialInfo> materialInfos;
                        if (TeamSettingMaterials.ContainsKey(name))
                        {
                            materialInfos = TeamSettingMaterials[name];
                        }
                        else
                        {
                            materialInfos = new List<MaterialInfo>();
                            TeamSettingMaterials.Add(name, materialInfos);
                        }

                        var isChildMaterialOwner = childMaterialOwner.FilePath.Equals(filePath, StringComparison.OrdinalIgnoreCase);
                        materialInfos.AddRange(model.material_array.material
                            .Where(x => IsSameShaderAssigned(x.shader_assign, ChildMaterial.Data.shader_assign))
                            .Where(x => !isChildMaterialOwner || ChildMaterial.Name != x.name)
                            .Where(x => ChildMaterial.ParentMaterials.All(y => !(y.Name.Equals(x.name) && y.OwnerDocument.FilePath.Equals(filePath))))
                            .Select(x => new MaterialInfo(pathInfo.Item1, pathInfo.Item2, x.name, model.comment, x.comment, false)));

                        continue;
                    }

                    var material = GetMaterialFromFile(filePath);
                    if (material != null)
                    {
                        var name = parentMaterialPathInfo.Name ?? string.Empty;
                        var match = regex.Match(filePath);
                        for (var i = 1; i < match.Groups.Count; i++)
                        {
                            var matchGroup = match.Groups[i];
                            if (matchGroup.Captures.Count == 0)
                            {
                                continue;
                            }
                            name = name.Replace($"\\{i}", matchGroup.Value);
                        }

                        List<MaterialInfo> materialInfos;
                        if (TeamSettingMaterials.ContainsKey(name))
                        {
                            materialInfos = TeamSettingMaterials[name];
                        }
                        else
                        {
                            materialInfos = new List<MaterialInfo>();
                            TeamSettingMaterials.Add(name, materialInfos);
                        }

                        // fmt ではファイル名がマテリアル名。
                        var materialName = System.IO.Path.GetFileNameWithoutExtension(filePath);

                        var isChildMaterialOwner = childMaterialOwner.FilePath.Equals(filePath, StringComparison.OrdinalIgnoreCase);
                        materialInfos.AddRange(Enumerable.Repeat(material, 1)
                            .Where(x => IsSameShaderAssigned(x.shader_assign, ChildMaterial.Data.shader_assign))
                            .Where(x => !isChildMaterialOwner || ChildMaterial.Name != materialName)
                            .Where(x => ChildMaterial.ParentMaterials.All(y => !(y.Name.Equals(materialName) && y.OwnerDocument.FilePath.Equals(filePath))))
                            .Select(x => new MaterialInfo(pathInfo.Item1, pathInfo.Item2, materialName, null, x.comment, false)));

                        continue;
                    }
                }
            }

            TeamSettingMaterials = TeamSettingMaterials.OrderBy(x => x.Key).ToDictionary(x => x.Key, x => x.Value);
            cmbTeamSettingNames.DisplayMember = "Display";
            cmbTeamSettingNames.ValueMember= "Value";
            cmbTeamSettingNames.DataSource = TeamSettingMaterials.Select(x => new ComboBoxItem(x.Key)).ToList();
            teamSettingNameValue = lastTeamSettingName;
            if (!TeamSettingMaterials.ContainsKey(teamSettingNameValue))
            {
                teamSettingNameValue = string.Empty;
            }
            cmbTeamSettingNames.Value = teamSettingNameValue;
            return true;
        }

        internal class ComboBoxItem
        {
            public string Value { get; set; }
            public string Display { get; set; }
            public ComboBoxItem(string value)
            {
                Value = value;
                Display = string.IsNullOrEmpty(value) ? Resources.MaterialParentDialog_ComboBoxItem_DisplayAll : value;
            }
        }


        private static Regex WildCardToRegex(string path)
        {
            var sb = new StringBuilder();
            var inCapture = false;
            foreach (var c in path)//.TrimEnd('\\'))
            {
                switch (c)
                {
                    case '?':
                        sb.Append(inCapture ? "." : "(.");
                        inCapture = true;
                        break;
                    case '*':
                        sb.Append(inCapture ? ".*" : "(.*");
                        inCapture = true;
                        break;
                    default:
                        sb.Append((inCapture ? "?)" : "") + Regex.Escape(c.ToString()));
                        inCapture = false;
                        break;
                }
            }
            if (inCapture)
            {
                sb.Append(")");
            }
            if (path.EndsWith("\\"))
            {
                sb.Append(".*");
            }
            return new Regex(sb.ToString(), RegexOptions.IgnoreCase);
        }

        private static bool IsSameShaderAssigned(shader_assignType assignA, shader_assignType assignB)
        {
            var archiveNameA = assignA?.shader_archive;
            var archiveNameB = assignB?.shader_archive;
            var shaderNameA = assignA?.shading_model;
            var shaderNameB = assignB?.shading_model;
            if (archiveNameA == null || archiveNameB == null || shaderNameA == null || shaderNameB == null)
            {
                return false;
            }
            return archiveNameA.Equals(archiveNameB, StringComparison.Ordinal) && shaderNameA.Equals(shaderNameB, StringComparison.Ordinal);
        }

        private static bool IsSameShaderAssigned(Material mtlA, Material mtlB)
        {
            var shaderFileNameA = mtlA?.MaterialShaderAssign?.ShaderDefinitionFileName;
            var shaderFileNameB = mtlB?.MaterialShaderAssign?.ShaderDefinitionFileName;
            var shaderNameA = mtlA?.MaterialShaderAssign?.ShaderName;
            var shaderNameB = mtlB?.MaterialShaderAssign?.ShaderName;
            if (shaderFileNameA == null || shaderFileNameB == null || shaderNameA == null || shaderNameB == null)
            {
                return false;
            }
            return shaderFileNameA.Equals(shaderFileNameB, StringComparison.Ordinal) && shaderNameA.Equals(shaderNameB, StringComparison.Ordinal);
        }

        private static string GuiObjectNameWithComment(GuiObject guiObject)
        {
            return GetNameWithComment(guiObject.Name, guiObject.GetComment());
        }

        private static string GetNameWithComment(string name, commentType comment)
        {
            return !string.IsNullOrEmpty(comment?.label) ? $"{comment.label} ({name})" :
                        (!string.IsNullOrEmpty(comment?.text) ? $"{comment.text} ({name})" : name);
        }

        private IEnumerable<Material> GetParentableMaterials(Model model, Material childMaterial)
        {
            var materials = model.Materials.Where(x => x == ParentMaterial || (x != childMaterial && !childMaterial.ParentMaterials.Contains(x) && !childMaterial.ChildMaterials.Contains(x) && x.CheckParentMaterialsReference(new List<Material> { childMaterial }) && childMaterial.CheckParentMaterialsReference(new List<Material> { x }) && IsSameShaderAssigned(childMaterial, x)));
            return materials;
        }

        private IEnumerable<Material> GetParentableMaterials(SeparateMaterial material, Material childMaterial)
        {
            var materials = material.Materials.Where(x => x == ParentMaterial || (x != childMaterial && !childMaterial.ParentMaterials.Contains(x) && !childMaterial.ChildMaterials.Contains(x) && x.CheckParentMaterialsReference(new List<Material> { childMaterial }) && childMaterial.CheckParentMaterialsReference(new List<Material> { x }) && IsSameShaderAssigned(childMaterial, x)));
            return materials;
        }

        private void lvwParentMaterials_SelectionChanged(object sender, EventArgs e)
        {
            ParentMaterialInfo = lvwParentMaterials.SelectedItemData as MaterialInfo;
            btnOK.Enabled = ParentMaterialInfo != null && ParentMaterialInfo.HasMaterial();
        }

        private void btnSelectParentFolder_Click(object sender, EventArgs e)
        {
            //parentMaterialsFolder
            using (var dialog = new CommonOpenFileDialog())
            {
                dialog.IsFolderPicker = true;
                dialog.Title = Resources.MaterialParentDialog_btnSelectParentFolder_Click_SelectParentMaterialFolder;

                if (Directory.Exists(parentMaterialsFolder))
                {
                    dialog.InitialDirectory = Path.GetFullPath(parentMaterialsFolder);
                }
                else if (Directory.Exists(ChildMaterial.OwnerDocument.fileLocation))
                {
                    dialog.InitialDirectory = Path.GetFullPath(ChildMaterial.OwnerDocument.fileLocation);
                }
                var mainFrame = TheApp.MainFrame.IsDisposed ? null : TheApp.MainFrame;
                if (dialog.ShowDialog(this.Handle) == CommonFileDialogResult.Ok)
                {
                    parentMaterialsFolder = dialog.FileName;
                    tbxParentFolder.Text = parentMaterialsFolder;
                }
                UpdateListview();
            }
        }

        private void lvwParentMaterials_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            DialogResult = DialogResult.OK;
            Close();
        }

        private void EnterFolderPath()
        {
            if (!string.IsNullOrEmpty(tbxParentFolder.Text))
            {
                var path = tbxParentFolder.Text;
                path = path.Contains("%") ? Environment.ExpandEnvironmentVariables(path) : path;

                if (Directory.Exists(path))
                {
                    tbxParentFolder.Text = path;
                    parentMaterialsFolder = path;
                    UpdateListview();
                    return;
                }
            }
            tbxParentFolder.Text = parentMaterialsFolder;
        }

        private void tbxParentFolder_Leave(object sender, EventArgs e)
        {
            EnterFolderPath();
        }

        private void tbxParentFolder_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                EnterFolderPath();
            }
        }

        private void rbt_RadioChecked(object sender, EventArgs e)
        {
            if (sender == rbtLoaded)
            {
                selectedListType = ListType.Loaded;
            }
            else if (sender == rbtUserDefine)
            {
                selectedListType = ListType.UserDefined;
            }
            else
            {
                selectedListType = ListType.TeamSetting;
            }
            UpdateSelectControls();
            UpdateListview();
        }

        private void cmbTeamSettingNames_SelectedIndexChanged(object sender, EventArgs e)
        {
            teamSettingNameValue = cmbTeamSettingNames.SelectedValue.ToString();
            UpdateSelectControls();
            UpdateListview();
        }

        private void UpdateSelectControls()
        {
            cmbTeamSettingNames.Enabled = rbtTeamSetting.Checked;
            tbxParentFolder.Enabled = rbtUserDefine.Checked;
            btnSelectParentFolder.Enabled = rbtUserDefine.Checked;
        }
    }
}
